// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_TextLiteral.cc,v 1.3 1999/07/23 21:08:14 dmartin Exp $
// 
//---------------------------------------------------------------------------
#include <ctype.h>
#include "IIR_TextLiteral.hh"
#include "IIR_AccessTypeDefinition.hh"
#include "IIR_ArrayTypeDefinition.hh"
#include "StandardPackage.hh"
#include "IIR_Name.hh"
#include "resolution_func.hh"
#include "symbol_table.hh"

IIRScram_TextLiteral::IIRScram_TextLiteral(){
}


IIRScram_TextLiteral::~IIRScram_TextLiteral(){
}


void 
IIRScram_TextLiteral::_publish_vhdl(ostream &_vhdl_out) {
  _print( _vhdl_out );
}


void 
IIRScram_TextLiteral::_publish_cc() {
  _print_cc( _cc_out );
}


ostream &
IIRScram_TextLiteral::_print_cc( ostream &os ) {
  int i;
  
  for( i = 0; i < get_text_length(); i++ ){
    // Check for two consecutive double quotes.
    if ((i > 0) && (i < (get_text_length() - 1))) {
      if ((operator[](i) == '\"') && (operator[](i + 1) == '\"')) {
        os << "\\\"";
        i++;
      } else if (operator[](i) == '\\') {
	// If the character is a backslash output two backslashes
	os << operator[](i) << operator[](i);
      } else if ((operator[](i) == '%') && (operator[](i + 1) == '%')) {
	os << operator[](i);
	i++;
      } else {
        os << operator[](i);
      }
    }
    else {
      // If the string's delimiters are percentages instead of quotes
      if (operator[](i) == '%') {
	os << '\"';
      } else {
	os << operator[](i);
      }
    }
  }
  return os;
}


ostream &
IIRScram_TextLiteral::_print( ostream &os ){
  os.write( get_text(), get_text_length() );

  return os;
}


int 
IIRScram_TextLiteral::_cmp(IIR_TextLiteral *a, IIR_TextLiteral *b) {
  int size_a, size_b;
  size_a = a->get_text_length();
  size_b = b->get_text_length();

  if ( size_a != size_b ){
    if ( size_a < size_b ){
      return -1;
    }
    else{
      return 1;
    }
  }

  IIR_Char *text_a = a->get_text();
  IIR_Char *text_b = b->get_text();
  
  if( text_a == text_b ){
    return 0;
  }
  else{
    return memcmp( text_a, text_b, size_a );
  }
}


int 
IIRScram_TextLiteral::_cmp(IIR_TextLiteral *a, char *b) {
  int size_a = a->get_text_length();
  int size_b = strlen(b);
  int bound = size_a;
  int retval = 0;

  if ( size_a != size_b ){
    if (size_a < size_b) {
      retval = -1;
      bound = size_a;
    }
    else {
      retval = 1;
      bound = size_b;
    }
  }

  IIR_Char *text_a = a->get_text();

  for(int i = 0; i < bound; i++) {
    if (text_a[i] != b[i]) {
      return text_a[i] - b[i];
    }
  }

  return retval;
}


int 
IIRScram_TextLiteral::_cmp(IIR_TextLiteral *a, IIR_Name *b){
  int retval;

  if ( b->get_kind() != IIR_SIMPLE_NAME ){
    retval = -1;
  }
  else if ( b->get_prefix()->_is_text_literal() == false ){
    retval = -1;
  }
  else {
    retval = _cmp( a, (IIR_TextLiteral *)b->get_prefix() );
  }

  return retval;
}


int 
IIRScram_TextLiteral::_ncmp(IIR_TextLiteral *a, char *b, int length) {
  int size_a = a->get_text_length();
  int size_b = strlen(b);
  int retval = 0;

  if ( size_a != size_b && ( size_a < length || size_b < length )) {
    if (size_a < size_b) {
      retval = -1;
      length = size_a;
    }
    else {
      retval = -1;
      length = size_b;
    }
  }

  IIR_Char *text_a = a->get_text();

  for (int i = 0; i < length; i++) {
    if (text_a[i] != b[i]) {
      return text_a[i] - b[i];
    }
  }
  return retval;
}


void 
IIRScram_TextLiteral::_to_lower(){
  char *my_string = get_text();
  int my_len = get_text_length();
  
  char *tmp_buf = new char[my_len];

  int i;
  for( i = 0; i < my_len; i++ ){
    tmp_buf[i] = (char)tolower( my_string[i] );
  }

  _set_text( tmp_buf, my_len );
  
  delete [] tmp_buf;
}


void 
IIRScram_TextLiteral::_set_text( IIR_Char *new_text, IIR_Int32 new_length ){
  ASSERT( new_length > 0 );
  ASSERT( new_text != NULL);

  text = string_table.hash_look( new_text, new_length );
}


set<IIR_Declaration> *
IIRScram_TextLiteral::_symbol_lookup(){
  set<IIR_Declaration> *retval;

  retval = new set<IIR_Declaration>(*_get_symbol_table()->find_set( (IIR_TextLiteral *)this ));

  return retval;
}


set<IIR_Declaration> *
IIRScram_TextLiteral::_symbol_lookup(IIR_Boolean(IIR::*constraint_function)()){
  return IIR::_symbol_lookup( constraint_function );
}


set<IIR_TypeDefinition> *
IIRScram_TextLiteral::_get_rval_set(IIR_Boolean (IIR::*constraint_function)() ) {
  set<IIR_TypeDefinition> *retval = NULL;

  // Get the set of declarations that correspond to our name, and then
  // get all of their RVALS.
  set<IIR_Declaration> *decl_set = _symbol_lookup();

  retval = decl_set_to_typedef_set( decl_set, constraint_function );
  delete decl_set;

  return retval;
}


set<IIR_TypeDefinition> *
IIRScram_TextLiteral::_get_rval_set_for_string(){
  set<IIR_TypeDefinition> *retval = NULL;

  int my_length = get_text_length();
  // A string literal can be a constant initializer for an array of
  // enumeration literals.  We'll look up each of the elements, and if
  // they are enumeration literals, we'll get the enumeration type they
  // belong to, and it it to the return set.  Also, we store the quotes in
  // the string..  So we need to start at 1, and end at length - 1.
  char delimiter;

  int start = 0;
  int i;
  for( i = 0 ; i < my_length; i++ ){
    if ( operator[](i) == '"' ){
      delimiter = '"';
      break;
    }
    else if (  operator[](i) == '%' ){
      delimiter = '%';
      break;
    }
  }
  
  start = i + 1;
  ASSERT( start < my_length );

  if ( operator[]( start ) == delimiter ){
    // Then this is an empty string.
    retval = new set<IIR_TypeDefinition>( StandardPackage::savant_null_type_definition );
  }
  else{
    set<IIR_ArrayTypeDefinition> *one_d_array_types;
    one_d_array_types = _get_symbol_table()->get_in_scope_array_types();

    retval = new set<IIR_TypeDefinition>( );

    IIR_TypeDefinition *current = one_d_array_types->get_element();
    while( current != NULL ){
      retval->add( current );
      current = one_d_array_types->get_next_element();
    }

    retval->reduce_set( &IIR::_is_character_type );

    if ( retval->num_elements() == 0 ){
      delete retval;
      retval = NULL;
    }

  }
  return retval;
}

