
// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_BlockConfiguration.cc,v 1.4 1999/07/23 21:07:54 dmartin Exp $
// 
//---------------------------------------------------------------------------
#include "IIRScram_BlockConfiguration.hh"
#include "IIR_ArchitectureDeclaration.hh"
#include "IIR_ComponentConfiguration.hh"
#include "IIR_ConfigurationDeclaration.hh"
#include "IIR_ConcurrentGenerateIfStatement.hh"
#include "IIR_ConcurrentGenerateForStatement.hh"
#include "IIR_IndexedName.hh"
#include "IIR_IntegerSubtypeDefinition.hh"
#include "IIR_Label.hh"
#include "IIR_SliceName.hh"
#include "IIR_Statement.hh"
#include "StandardPackage.hh"
#include "error_func.hh"
#include "library_manager.hh"
#include "resolution_func.hh"
#include "symbol_table.hh"
#include <strstream.h>

extern bool parse_error;
extern bool debug_symbol_table;

IIRScram_BlockConfiguration::~IIRScram_BlockConfiguration(){}

void 
IIRScram_BlockConfiguration::_publish_vhdl(ostream &_vhdl_out) {
  _vhdl_out << " for ";
  get_block_specification()->_publish_vhdl(_vhdl_out);
  _vhdl_out << "\n";
  use_clause_list._publish_vhdl(_vhdl_out);
  configuration_item_list._publish_vhdl(_vhdl_out);
  _vhdl_out << " end for;\n";
}

void
IIRScram_BlockConfiguration::_publish_cc_headers() {
  _cc_out << "#include \"";
  get_block_specification()->_publish_cc_binding_name();
  _cc_out << "_elab.hh\"" << endl;
  use_clause_list._publish_cc_headers();
  configuration_item_list._publish_cc_headers();
}


void
IIRScram_BlockConfiguration::_publish_cc_object_pointers_init() {
  _cc_out << "{" << endl;
  get_block_specification()->_publish_cc_binding_name();
  _cc_out << "_elab* enclosingArch = configuredEntity;" << endl;  
  get_block_specification()->_publish_cc_binding_name();
  _cc_out << "_elab* enclosingArch1 = enclosingArch;" << endl;  
  _cc_out << "{" << endl;
  configuration_item_list._publish_cc_configure_block();
  _cc_out << "};" << endl;
  _cc_out << "};" << endl;
}

void
IIRScram_BlockConfiguration::_publish_cc_configure_block(IIR_Boolean _is_within_generate_for) {
  IIR_ConcurrentGenerateIfStatement  *tempGenerateIfStmt;
  IIR_ConcurrentGenerateForStatement *tempGenerateForStmt;
  IIR_SliceName                      *tempSlice;
  IIR_Label                          *mangled_label = NULL;

  if((get_block_specification()->_is_block_statement() == TRUE) ||
     (get_block_specification()->_is_concurrent_generate_statement() == TRUE)) {
    
    if (get_block_specification()->get_kind() == IIR_LABEL) {
      if (((IIR_Label *) get_block_specification())->get_statement()->get_kind() == IIR_CONCURRENT_GENERATE_IF_STATEMENT) {
	tempGenerateIfStmt = (IIR_ConcurrentGenerateIfStatement *) ((IIR_Label *) get_block_specification())->get_statement();
	ASSERT ( tempGenerateIfStmt->get_kind() == IIR_CONCURRENT_GENERATE_IF_STATEMENT );
	tempGenerateIfStmt->_publish_cc_generate_condition();
      }
      if (((IIR_Label *) get_block_specification())->get_statement()->get_kind() == IIR_CONCURRENT_GENERATE_FOR_STATEMENT) {
	tempGenerateForStmt = (IIR_ConcurrentGenerateForStatement *) ((IIR_Label *)get_block_specification())->get_statement();
	ASSERT(tempGenerateForStmt->get_kind() == IIR_CONCURRENT_GENERATE_FOR_STATEMENT);
	tempGenerateForStmt->_publish_cc_generate_for_loop_with_zero("generateConstant");
      }
    }
    else if (get_block_specification()->get_kind() == IIR_SLICE_NAME) {
      tempSlice = (IIR_SliceName *) get_block_specification();
      ASSERT ( tempSlice->get_kind() == IIR_SLICE_NAME );
      tempSlice->_publish_cc_slice_name_as_for_loop();
    }
    else if (get_block_specification()->get_kind() == IIR_INDEXED_NAME) {
      ((IIR_IndexedName *) get_block_specification())->_publish_cc_index_name_as_for_loop();
    }
    else {
      // The case we have hit on so far is a slice name
      // Don't know if anything else is going to pop up. So will put a
      // check for some extraneous case that we have not hit upon so far.
      
      cerr << "Warning : " << get_block_specification()->get_kind_text()
	   << " Not yet handled in block configurations" << endl;
    }
      
    _cc_out << "{" << endl;
    
    get_block_specification()->_publish_cc_binding_name();
    _cc_out << "_elab* enclosingArch = enclosingArch1->";

    if (get_block_specification()->get_kind() == IIR_LABEL) {
      mangled_label = ((IIR_Label*)get_block_specification())->get_statement()->_get_label();
    }
    else if (get_block_specification()->get_kind() == IIR_SLICE_NAME) {
      mangled_label = (IIR_Label *) ((IIR_SliceName *) get_block_specification())->get_prefix();
      mangled_label = mangled_label->get_statement()->_get_label();
    }
    else {
      cerr << "Warning : " << get_block_specification()->get_kind_text()
	   << " Not yet handled in block configurations" << endl;
    }

    ASSERT ( mangled_label != NULL );
    ASSERT ( mangled_label->get_kind() == IIR_LABEL );
    
    mangled_label->_publish_cc_elaborate();
    _cc_out << "_elab_obj;" << endl;
    get_block_specification()->_publish_cc_binding_name();
    _cc_out << "_elab* enclosingArch1 = enclosingArch;" << endl;    
  }


  if (get_block_specification()->get_kind() == IIR_SLICE_NAME) {
    configuration_item_list._publish_cc_configure_block(TRUE);
  }
  else if (get_block_specification()->get_kind() == IIR_LABEL && ((IIR_Label *)get_block_specification())->get_statement()->get_kind() == IIR_CONCURRENT_GENERATE_FOR_STATEMENT){
    configuration_item_list._publish_cc_configure_block(TRUE);
  }
  else {
    configuration_item_list._publish_cc_configure_block(FALSE);
  }
  
  if((get_block_specification()->_is_block_statement() == TRUE) ||
     (get_block_specification()->_is_concurrent_generate_statement() == TRUE)) {
    _cc_out << "};" << endl;
  }
}

void 
IIRScram_BlockConfiguration::_type_check_configuration_item( IIR_List &statement_list, 
							     IIR *enclosing_region ){
  _get_symbol_table()->open_scope( this );
  _type_check_block_specification( statement_list, enclosing_region );
  ASSERT( get_block_specification()->_is_resolved() == TRUE );

  if( debug_symbol_table == true ){
    cerr << "Type checking block configuration - making |"
	 << *get_block_specification() << "| visible" << endl;
  }
  get_block_specification()->_make_interface_visible( _get_symbol_table() );

  configuration_item_list._type_check_configuration_item( *get_block_specification()->_get_statement_list(),
							  this );
  _get_symbol_table()->close_scope( this );
}

void 
IIRScram_BlockConfiguration::_type_check_block_specification( IIR_List &statement_list,
							      IIR *enclosing_region ){
  // This method implements the type checking described in 1.3.1 of the '93 LRM.
  ASSERT( enclosing_region != NULL );

  // What type the specification is allowed to be depends largely on region 
  // it is contained in.
  switch( enclosing_region->get_kind() ){
  case IIR_CONFIGURATION_DECLARATION:{
    _resolve_specification_inside_configuration_declaration( (IIR_ConfigurationDeclaration *)enclosing_region );
    
    break;
  }
  case IIR_COMPONENT_CONFIGURATION:{
    _resolve_specification_inside_component_configuration( (IIR_ComponentConfiguration *)enclosing_region );
    break;
  }
  case IIR_BLOCK_CONFIGURATION:{
    _resolve_specification_inside_block_configuration( statement_list,
						       (IIR_BlockConfiguration *)enclosing_region);
    break;
  }
  default:{
    ostrstream err;
    err << "Internal error in IIRScram_BlockConfiguration::_type_check_block_specification -"
	<< " don't know what to do with enclosing region type " 
	<< enclosing_region->get_kind_text() << ends;
    report_error( this, err );
    return;
  }
  }
  
  ASSERT( get_block_specification()->_is_resolved() == TRUE );

}

void 
IIRScram_BlockConfiguration::_resolve_specification_inside_configuration_declaration( IIR_ConfigurationDeclaration *enclosing_declaration ){
  // This method implements pg. 13 line 362 of the 93 LRM.

  ASSERT( enclosing_declaration != NULL );
  // The block specification must be an architecture name, and that
  // architecture name must denote a design entity body whose
  // interface is defined by the entity declaration denoted by the
  // entity name of the enclosing configuration declaration.
  IIR_EntityDeclaration *configuration_entity = enclosing_declaration->get_entity();
  ASSERT( configuration_entity != NULL );

  IIR *original_specification = get_block_specification();
  IIR_Declaration *specification_declaration = NULL;
  if( original_specification->get_kind() != IIR_SIMPLE_NAME &&
      original_specification->get_kind() != IIR_SELECTED_NAME ){
    ostrstream err;
    err << "Syntax error at |" << *original_specification << "| -"
	<< " expecting a simple or selected name." << ends;
    report_error( original_specification, err );
    return;
  }

  ASSERT( original_specification->_is_iir_name() == TRUE );
  specification_declaration = 
    _get_library_manager()->_lookup_architecture( TRUE,
						  configuration_entity,
						  (IIR_Name *)original_specification );

  if( specification_declaration == NULL ){
    ASSERT( parse_error == TRUE );
    return;
  }

  set_block_specification( original_specification->_decl_to_decl( specification_declaration ) );
}

void 
IIRScram_BlockConfiguration::_resolve_specification_inside_component_configuration( IIR_ComponentConfiguration *component_configuration ){
  // This method implements pg. 13 line 367 of the 93 LRM - If a block
  // configuration appears immediately within a component configuration,
  // then the corresponding components must be fully bound (see section
  // 5.2.1.1), the block specification of that block configuration denote
  // the same architecture body as that to which the corresponding
  // components are bound.

  ASSERT( component_configuration != NULL );
  ASSERT( component_configuration->get_entity_aspect() != NULL );
  ASSERT( get_block_specification() != NULL );
  ostrstream err;
  err << "Block configurations within component configurations aren't handled yet." << ends;
  report_error( this, err );
  abort();
}

void 
IIRScram_BlockConfiguration::_resolve_specification_inside_block_configuration( IIR_List &,
										IIR_BlockConfiguration * ){
  IIR_IndexedName *original_indexed_name = NULL;
  IIR_Name *to_lookup = NULL;
  if( get_block_specification()->get_kind() == IIR_INDEXED_NAME ){
    original_indexed_name = (IIR_IndexedName *)get_block_specification();
    IIR *prefix = original_indexed_name->get_prefix();
    ASSERT( prefix->_is_iir_name() == TRUE );
    to_lookup = (IIR_Name *)prefix;
  }
  else{
    ASSERT( get_block_specification()->_is_iir_name() == TRUE );
    to_lookup = (IIR_Name *)get_block_specification();
  }
  
  IIR_Label *my_label = to_lookup->_lookup_label( TRUE );
  IIR_Statement *current_statement = my_label->get_statement();
  if( current_statement->get_kind() != IIR_BLOCK_STATEMENT &&
      current_statement->get_kind() != IIR_CONCURRENT_GENERATE_FOR_STATEMENT &&
      current_statement->get_kind() != IIR_CONCURRENT_GENERATE_IF_STATEMENT ){
    ostrstream err;
    err << "A block configuration that appears within another block configuration must "
	<< "have a block specification that specifies either a block statement label, "
	<< "or a generate statement label. |" << *my_label << "| specifies neither."
	<< ends;
    report_error( this, err );
    return;
  }
      
  if( original_indexed_name != NULL ){
    if( my_label->get_statement()->get_kind() != IIR_CONCURRENT_GENERATE_FOR_STATEMENT ){
      ostrstream err;
      err << "Index specifications may only be applied to generate statements"
	  << " using the \"for\" syntax." << ends;
      report_error( this, err );
    }
    IIR_SliceName *new_spec = new IIR_SliceName();
    copy_location( this, new_spec );
    new_spec->set_prefix( my_label );
    IIR *suffix = original_indexed_name->get_suffix(); 

    suffix = suffix->_semantic_transform( StandardPackage::savant_universal_integer );
    suffix->_type_check( StandardPackage::savant_universal_integer );
    suffix = suffix->_rval_to_decl( StandardPackage::savant_universal_integer );

    new_spec->set_suffix( suffix );

    set_block_specification( new_spec );
  }
  else{
    set_block_specification( my_label );
  }
}

ostream &
IIRScram_BlockConfiguration::_print( ostream &os ){
  if( get_block_specification() != NULL ){
    os << *get_block_specification();
  }
  else{
    os << "<ANONYMOUS>";
  }

  return os;
}
