/************************************************************************/
/*  Editor, File actions.						*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>

#   include	<Xm/Xm.h>
#   include	<Xm/MainW.h>
#   include	<Xm/Form.h>
#   include	<Xm/RowColumn.h>
#   include	<Xm/ScrolledW.h>
#   include	<Xm/ScrollBar.h>
#   include	<Xm/ToggleB.h>

#   include	<Xm/Protocols.h>
#   include	<Xm/MwmUtil.h>

#   include	<appSystem.h>
#   include	"appFrame.h"

#   include	<debugon.h>

/************************************************************************/
/*									*/
/*  Callback for the file menu.						*/
/*									*/
/************************************************************************/

void appDocFileSave(	Widget		option,
			XtPointer	voided,
			XtPointer	call_data	 )
    {
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;

    if  ( ! ea->eaSaveDocument )
	{ XDEB(ea->eaSaveDocument); return;	}

    if  ( ! ed->edFilename		||
	  ed->edFileReadOnly		||
	  ed->edFormat < 0		)
	{ appDocFileSaveAs( option, voided, call_data ); return; }

    if  ( (*ea->eaSaveDocument)( ea, ed, ed->edFormat, ed->edFilename ) )
	{
	appReportSaveFailure( ea, option, ed->edTopWidget, ed->edFilename );
	}
    else{ appDocumentChanged( ea, ed, 0 ); }

    return;
    }

/************************************************************************/
/*									*/
/*  'Close' callback for the file menu and for the window manager menu	*/
/*									*/
/*  1)  As 'mwm' sometimes sends a 'Save yourself' when 'Close' is	*/
/*	selected from the window manager menu, deactivate the protocol.	*/
/*	B.T.W. This does not help.					*/
/*									*/
/*									*/
/************************************************************************/

void appDocFileClose(	Widget		option,
			XtPointer	voided,
			XtPointer	call_data	 )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    if  ( option == ed->edTopWidget )
	{ option= (Widget)0;	}

#   if 0
    /*  1  */
    if  ( ea->eaSaveAtom != None )
	{ XmDeactivateWMProtocol( ed->edTopWidget, ea->eaSaveAtom ); }
#   endif

    if  ( ed->edHasBeenChanged )
	{ appRunReallyCloseDialog( option, ea, ed, call_data ); return; }

    appCloseDocument( ea, ed );
    }

static void appDocFileSaveYourself(	Widget		option,
					XtPointer	voided,
					XtPointer	call_data	 )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    char		filename[1000+1];
    char		savename[1000+1];
    char *		argv[5];
    long		now;

    if  ( option == ed->edTopWidget )
	{ option= (Widget)0;	}

    if  ( ed->edFilename )
	{
	if  ( appAbsoluteName( filename, 1000, ed->edFilename ) < 0 )
	    { SDEB(ed->edFilename); return;	}
	}
    else{ filename[0]= '\0';	}

    if  ( ed->edHasBeenChanged || ! ed->edFilename )
	{
	char		home[200+1];
	int		len;

	if  ( ! ea->eaSaveDocument		||
	      ed->edFormat < 0			)
	    { XLDEB(ea->eaSaveDocument,ed->edFormat); return;	}

	len= appHomeDirectory( home, 200 );
	if  ( len < 0 )
	    { LDEB(len); return;	}

	sprintf( savename, "%s/.%s", home, ea->eaApplicationName );
	if  ( appTestDirectory( savename )	&&
	      appMakeDirectory( savename )	)
	    { SDEB(savename); return;	}

	now= appGetTimestamp();

	if  ( ea->eaFileExtensions[ed->edFormat].afeExtension )
	    {
	    sprintf( savename, "%s/.%s/%08lx.%s",
			home, ea->eaApplicationName, now,
			ea->eaFileExtensions[ed->edFormat].afeExtension );
	    }
	else{
	    sprintf( savename, "%s/.%s/%08lx",
					home, ea->eaApplicationName, now );
	    }

	if  ( (*ea->eaSaveDocument)( ea, ed, ed->edFormat, savename ) )
	    {
	    appReportSaveFailure( ea, option, ed->edTopWidget, savename );
	    }
	else{
	    argv[0]= ea->eaArgv[0];
	    argv[1]= "++Restore";
	    argv[2]= savename;
	    argv[3]= ed->edTitle;

	    if  ( ed->edFilename )
		{
		argv[4]= filename;

		XSetCommand( ea->eaDisplay, XtWindow( ed->edTopWidget ),
								argv, 5 );
		}
	    else{
		XSetCommand( ea->eaDisplay, XtWindow( ed->edTopWidget ),
								argv, 4 );
		}
	    }
	}
    else{
	argv[0]= ea->eaArgv[0];
	argv[1]= filename;

	XSetCommand( ea->eaDisplay, XtWindow( ed->edTopWidget ), argv, 2 );
	}

    return;
    }

void appDocFileNew(	Widget		option,
			XtPointer	voided,
			XtPointer	call_data	 )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    appAppFileNew( option, (void *)ea, call_data );
    }

void appDocFileQuit(	Widget		option,
			XtPointer	voided,
			XtPointer	call_data	 )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    appQuitApplication( option, ed->edTopWidget, ea, call_data );

    return;
    }

void appDocFileMini(	Widget		option,
			XtPointer	voided,
			XtPointer	call_data	 )
    {
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;
    Window			win= XtWindow( ed->edTopWidget );

    XtVaSetValues( ed->edTopWidget, XmNinitialState, IconicState, NULL );
    XIconifyWindow( ea->eaDisplay, win, ea->eaScreen );

    return;
    }

void appDocFileHide(	Widget		option,
			XtPointer	voided,
			XtPointer	call_data	 )
    {
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;

    if  ( ed->edIsVisible )
	{
	XtUnmapWidget( ed->edTopWidget );

	ed->edIsVisible= 0;

	appDocVisible( ea, ed, ed->edIsVisible );
	}

    return;
    }

void appReportSaveFailure(	EditApplication *	ea,
				Widget			option,
				Widget			relative,
				const char *		filename )
    {
    AppFileMessageResources *	afmr= &(ea->eaFileMessageResources);

    if  ( appTestFileWritable( filename ) )
	{
	appQuestionRunSubjectErrorDialog( ea, relative, option,
					filename, afmr->afmrFileNotWritable );
	}
    else{
	appQuestionRunSubjectErrorDialog( ea, relative, option,
					filename, afmr->afmrFileNotWritable );
	}
    }

/************************************************************************/
/*  The user selected the 'about' option.				*/
/************************************************************************/

void appDocAbout(	Widget		option,
			XtPointer	voided,
			XtPointer	call_data	 )
    {
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;
    Window			win= XtWindow( ea->eaTopWidget );

    ea->eaMainVisibleAsAbout= 1;

    XtVaSetValues( ea->eaTopWidget, XmNinitialState, NormalState, NULL );
    XMapRaised( ea->eaDisplay, win );

    return;
    }

/************************************************************************/
/*  Print a document.							*/
/************************************************************************/

void appDocFilePrint(	Widget		printOption,
			XtPointer	voided,
			XtPointer	call_data	 )
    {
    EditDocument *	ed= (EditDocument *)voided;
    EditApplication *	ea= ed->edApplication;

    if  ( ! ea->eaPrintDocument )
	{ XDEB(ea->eaPrintDocument); return;	}

    appRunPrintDialog( ea, ed, printOption, "print" );
    }

int appPrintDocument(	EditApplication *	ea,
			EditDocument *		ed,
			int			printer )
    {
    FILE *		f;
    char		scratchName[L_tmpnam+1];
    char		scratchCommand[L_tmpnam+200];
    PrintDestination *	pd= ea->eaPrintDestinations+ printer;

    switch( pd->pdPrintKind )
	{
	case APPprinterPIPE:
	    f= popen( pd->pdCommand, "w" );

	    if  ( ! f )
		{ SXDEB(pd->pdCommand,f); return -1;	}

	    (*ea->eaPrintDocument)( f, ed );

	    pclose( f );
	    return 0;
	case APPprinterTMPFILE:
	    tmpnam( scratchName );
	    f= fopen( scratchName, "w" );
	    if  ( ! f )
		{ SXDEB(scratchName,f); return -1;	}

	    (*ea->eaPrintDocument)( f, ed );

	    fclose( f );

	    sprintf( scratchCommand, pd->pdCommand, scratchName );
	    system( scratchCommand );
	    return 0;
	case APPprinterTOFILE:
	default:
	    LDEB(pd->pdPrintKind);
	    return -1;
	}
    }

static char *	appDocBuildFaxCommand(		const char *	faxCommand,
						int		commandLen,
						const char *	fileName,
						const char *	faxNumber,
						char *		title,
						int		fileCount,
						int		faxCount,
						int		titleCount )
    {
    char *		command;
    const char *	s;
    char *		to;

    int			fileLen;
    int			faxLen;
    int			titleLen;

    if  ( ! title )
	{ title= "";	}

    fileLen= strlen( fileName );
    faxLen= strlen( faxNumber );
    titleLen= strlen( title );

    to= command= (char *)malloc(
			commandLen+ fileCount* fileLen+ faxCount* faxLen+ 1 );
    if  ( ! command )
	{ XDEB(command); return (char *)0;	}

    s= faxCommand;
    while( *s )
	{
	if  ( s[0] == '%' && s[1] == 'f' )
	    {
	    strcpy( to, fileName ); to += fileLen;
	    s += 2; continue;
	    }

	if  ( s[0] == '%' && s[1] == 'n' )
	    {
	    strcpy( to, faxNumber ); to += faxLen;
	    s += 2; continue;
	    }

	if  ( s[0] == '%' && s[1] == 't' )
	    {
	    strcpy( to, title ); to += titleLen;
	    s += 2; continue;
	    }

	*(to++)= *(s++);
	}
    *to= '\0';

    return command;
    }

int appFaxDocument(	EditApplication *	ea,
			EditDocument *		ed,
			const char *		faxNumber )
    {
    int		fileCount= 0;
    int		faxCount= 0;
    int		titleCount= 0;

    int		commandLen;

    char	scratchName[L_tmpnam+1];

    char *	s;
    char *	command;

    FILE *	f;

    if  ( ! ea->eaFaxCommand )
	{ XDEB(ea->eaFaxCommand); return -1;	}

    s= ea->eaFaxCommand;
    while( *s )
	{
	if  ( s[0] == '%' && s[1] == 'f' )
	    { fileCount++; s += 2; continue;	}

	if  ( s[0] == '%' && s[1] == 'n' )
	    { faxCount++; s += 2; continue;	}

	if  ( s[0] == '%' && s[1] == 't' )
	    { titleCount++; s += 2; continue;	}

	s++;
	}

    commandLen= s- ea->eaFaxCommand;

    if  ( faxCount == 0 )
	{ LDEB(faxCount); return -1;	}

    if  ( fileCount > 0 )
	{
	tmpnam( scratchName );
	command= appDocBuildFaxCommand( ea->eaFaxCommand, commandLen,
					scratchName, faxNumber, ed->edTitle,
					fileCount, faxCount, titleCount );

	if  ( ! command )
	    { XDEB(command); return -1;	}

	f= fopen( scratchName, "w" );
	if  ( ! f )
	    { SXDEB(scratchName,f); free( command ); return -1;	}

	(*ea->eaPrintDocument)( f, ed );

	fclose( f );

	system( command );
	}
    else{
	scratchName[0]= '\0';

	command= appDocBuildFaxCommand( ea->eaFaxCommand, commandLen,
					scratchName, faxNumber, ed->edTitle,
					fileCount, faxCount, titleCount );

	f= popen( command, "w" );
	if  ( ! f )
	    { SXDEB(command,f); free( command ); return -1;	}

	(*ea->eaPrintDocument)( f, ed );

	pclose( f );
	}

    free( command );

    return 0;
    }

/************************************************************************/
/*  Callback for the item in the files menu.				*/
/************************************************************************/
void appMakeDocVisible(	EditApplication *	ea,
			EditDocument *		ed )
    {
    Window			win= XtWindow( ed->edTopWidget );

    if  ( ! ed->edIsVisible )
	{
	XtVaSetValues( ed->edTopWidget, XmNinitialState, NormalState, NULL );
	XtMapWidget( ed->edTopWidget );

	ed->edIsVisible= 1;

	appDocVisible( ea, ed, ed->edIsVisible );
	}

    XMapRaised( ea->eaDisplay, win );
    }

void appDocToFront(	Widget		option,
			XtPointer	voided,
			XtPointer	call_data	 )
    {
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;

    Boolean			state;

    appMakeDocVisible( ea, ed );

    state= XmToggleButtonGetState( option );
    XmToggleButtonSetState( option, ! state, False );

    return;
    }

/************************************************************************/
/*  Callbacks for the edit menu.					*/
/************************************************************************/
void appDocEditCopy(	Widget		option,
			XtPointer	voided,
			XtPointer	voidpbcs	 )
    {
    EditDocument *			ed= (EditDocument *)voided;
    EditApplication *			ea= ed->edApplication;
    XmPushButtonCallbackStruct *	pbcs=
					(XmPushButtonCallbackStruct *)voidpbcs;

    if  ( ! ea->eaDocCopy )
	{ SXDEB(ed->edFilename,ea->eaDocCopy); return;	}

    ea->eaDocCopy( ed, pbcs->event ); return;
    }

void appDocEditCut(	Widget		option,
			XtPointer	voided,
			XtPointer	voidpbcs	 )
    {
    EditDocument *			ed= (EditDocument *)voided;
    EditApplication *			ea= ed->edApplication;
    XmPushButtonCallbackStruct *	pbcs=
					(XmPushButtonCallbackStruct *)voidpbcs;

    if  ( ! ea->eaDocCut )
	{ SXDEB(ed->edFilename,ea->eaDocCut); return;	}

    ea->eaDocCut( ed, pbcs->event ); return;
    }

void appDocEditPaste(	Widget		option,
			XtPointer	voided,
			XtPointer	voidpbcs	 )
    {
    EditDocument *			ed= (EditDocument *)voided;
    XmPushButtonCallbackStruct *	pbcs=
					(XmPushButtonCallbackStruct *)voidpbcs;

    appDocAskForPaste( ed, "PRIMARY", pbcs->event->xbutton.time );
    }

void appDocEditSelAll(	Widget		option,
			XtPointer	voided,
			XtPointer	voidpbcs	 )
    {
    EditDocument *			ed= (EditDocument *)voided;
    EditApplication *			ea= ed->edApplication;
    XmPushButtonCallbackStruct *	pbcs=
					(XmPushButtonCallbackStruct *)voidpbcs;

    if  ( ! ea->eaDocSelAll )
	{ SXDEB(ed->edFilename,ea->eaDocSelAll); return;	}

    ea->eaDocSelAll( ed, pbcs->event ); return;
    }

/************************************************************************/
/*  Set the menu of a document window.					*/
/************************************************************************/

static void appDocMakeMenu(	EditDocument *		ed,
				EditApplication *	ea,
				Widget			mainWindow )
    {
    ed->edMenuBar= XmCreateMenuBar( mainWindow, WIDGET_NAME, NULL, 0 );

    ed->edFileMenu= appMakeMenu( &(ed->edFileMenuButton),
			ea, ed->edMenuBar, *(ea->eaDocFileMenuText), 0,
			ea->eaDocFileMenuItems, ea->eaDocFileMenuItemCount,
			(void *)ed );

    ed->edEditMenu= appMakeMenu( &(ed->edEditMenuButton),
			ea, ed->edMenuBar, *(ea->eaDocEditMenuText), 0,
			ea->eaDocEditMenuItems, ea->eaDocEditMenuItemCount,
			(void *)ed );

    if  ( ea->eaMakePrivateDocumentMenus )
	{ (*ea->eaMakePrivateDocumentMenus)( ea, ed, ed->edMenuBar ); }

    ed->edWindowMenu= appMakeMenu( &(ed->edWindowMenuButton),
		    ea, ed->edMenuBar,
		    *(ea->eaDocWindowMenuText), 0,
		    ea->eaDocWindowMenuItems, ea->eaDocWindowMenuItemCount,
		    (void *)ed );

    if  ( ea->eaDocHelpMenuItems )
	{
	ed->edHelpMenu= appMakeMenu( &(ed->edHelpMenuButton),
		    ea, ed->edMenuBar,
		    *(ea->eaDocHelpMenuText), 1,
		    ea->eaDocHelpMenuItems, ea->eaDocHelpMenuItemCount,
		    (void *)ed );
	}

    XtManageChild( ed->edMenuBar );
    }

/************************************************************************/
/*									*/
/*  Close a a document.							*/
/*									*/
/************************************************************************/
void appCloseDocument(	EditApplication *	ea,
			EditDocument *		ed	)
    {
    appRemoveDocument( ea, ed );

    XtDestroyWidget( ed->edTopWidget );

    return;
    }

/************************************************************************/
/*									*/
/*  Callbacks for the scrollbars, and the size admin  of a document.	*/
/*									*/
/************************************************************************/
void appDocVerticalScrollbarCallback(	Widget		w,
					XtPointer	voided,
					XtPointer	voidscbs )
    {
    XmScrollBarCallbackStruct *	scbs= (XmScrollBarCallbackStruct *)voidscbs;
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;

    Display *			display= XtDisplay( ed->edDocumentWidget );
    Window			win= XtWindow( ed->edDocumentWidget );
    GC				gc= ed->edGc;

    AppDrawingData *		add= &(ed->edDrawingData);
    DocumentRectangle *		drBack= &(add->addBackRect);
    DocumentRectangle		drClip;

    int				ox;
    int				oy;

    int				high;
    int				wide;
    int				d;

    high= ed->edVisibleRect.drY1- ed->edVisibleRect.drY0+ 1;
    wide= ed->edVisibleRect.drX1- ed->edVisibleRect.drX0+ 1;
    d= scbs->value- ed->edVisibleRect.drY0;

    ed->edVisibleRect.drY0 += d;
    ed->edVisibleRect.drY1 += d;

    XSetClipMask( display, gc, None );

    if  ( d == 0 )
	{ return;	}

    drClip= *drBack;

    if  ( d > 0 )
	{
	if  ( high > d )
	    {
	    XCopyArea( display, win, win, gc, 0, d, wide, high- d, 0, 0 );

	    drClip.drY0= ed->edVisibleRect.drY1- d;
	    drClip.drY1= ed->edVisibleRect.drY1;
	    }
	else{
	    drClip.drY0= ed->edVisibleRect.drY0;
	    drClip.drY1= ed->edVisibleRect.drY1;
	    }
	}
    else{
	if  ( high+ d > 0 )
	    {
	    XCopyArea( display, win, win, gc, 0, 0, wide, high+ d, 0, -d );

	    drClip.drY0= ed->edVisibleRect.drY0;
	    drClip.drY1= ed->edVisibleRect.drY0- d;
	    }
	else{
	    drClip.drY0= ed->edVisibleRect.drY0;
	    drClip.drY1= ed->edVisibleRect.drY1;
	    }
	}

    ox= ed->edVisibleRect.drX0;
    oy= ed->edVisibleRect.drY0;

    (*ea->eaDrawRectangle)( ed->edDocumentWidget, ed, &drClip,
							ox, oy, win, gc );

    if  ( ed->edLeftRuler && ea->eaScrollVerticalRuler )
	{
	(*ea->eaScrollVerticalRuler)
				( ed->edLeftRuler, ed->edLeftRulerWidget, d );
	}

    return;
    }


void appDocHorizontalScrollbarCallback(	Widget		w,
					XtPointer	voided,
					XtPointer	voidscbs )
    {
    XmScrollBarCallbackStruct *	scbs= (XmScrollBarCallbackStruct *)voidscbs;
    EditDocument *		ed= (EditDocument *)voided;
    EditApplication *		ea= ed->edApplication;

    Display *			display= XtDisplay( ed->edDocumentWidget );
    Window			win= XtWindow( ed->edDocumentWidget );
    GC				gc= ed->edGc;

    DocumentRectangle		drClip;

    int				ox;
    int				oy;

    int				high;
    int				wide;
    int				d;

    high= ed->edVisibleRect.drY1- ed->edVisibleRect.drY0+ 1;
    wide= ed->edVisibleRect.drX1- ed->edVisibleRect.drX0+ 1;
    d= scbs->value- ed->edVisibleRect.drX0;

    ed->edVisibleRect.drX0 += d;
    ed->edVisibleRect.drX1 += d;

    XSetClipMask( display, gc, None );

    if  ( d == 0 )
	{ return;	}

    drClip= ed->edVisibleRect;

    if  ( d > 0 )
	{
	if  ( wide > d )
	    {
	    XCopyArea( display, win, win, gc, d, 0, wide- d, high, 0, 0 );

	    drClip.drX0= ed->edVisibleRect.drX1- d;
	    drClip.drX1= ed->edVisibleRect.drX1;
	    }
	else{
	    drClip.drX0= ed->edVisibleRect.drX0;
	    drClip.drX1= ed->edVisibleRect.drX1;
	    }
	}
    else{
	if  ( wide+ d > 0 )
	    {
	    XCopyArea( display, win, win, gc, 0, 0, wide+ d, high, -d, 0 );

	    drClip.drX0= ed->edVisibleRect.drX0;
	    drClip.drX1= ed->edVisibleRect.drX0- d;
	    }
	else{
	    drClip.drX0= ed->edVisibleRect.drX0;
	    drClip.drX1= ed->edVisibleRect.drX1;
	    }
	}

    ox= ed->edVisibleRect.drX0;
    oy= ed->edVisibleRect.drY0;

    (*ea->eaDrawRectangle)( ed->edDocumentWidget, ed, &drClip,
							ox, oy, win, gc );

    if  ( ed->edTopRuler && ea->eaScrollHorizontalRuler )
	{
	(*ea->eaScrollHorizontalRuler)
				( ed->edTopRuler, ed->edTopRulerWidget, d );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Prevent the Shell that contains a document from being resized	*/
/*  beyond normal limits.						*/
/*									*/
/************************************************************************/
void appSetShellConstraints(	EditDocument *		ed )
    {
    Dimension		width;
    Dimension		height;

    Arg			al[20];
    int			ac= 0;

    AppDrawingData *	add= &(ed->edDrawingData);

    XtVaGetValues( ed->edTopWidget,
			XmNwidth,	&width,
			XmNheight,	&height,
			NULL );

    if  ( ed->edShellExtraWidth >= 0 )
	{
	XtWidgetGeometry	preferred;

	int			maxWidth;

	maxWidth= add->addBackRect.drX1- add->addBackRect.drX0;
	maxWidth += ed->edShellExtraWidth;

	XtQueryGeometry( ed->edMenuBar, (XtWidgetGeometry *)0, &preferred );
	if  ( maxWidth < preferred.width+ preferred.border_width )
	    { maxWidth=  preferred.width+ preferred.border_width;	}

	XtSetArg( al[ac],	XmNmaxWidth,	maxWidth ); ac++;

	if  ( width > maxWidth )
	    {
	    XtSetArg( al[ac],	XmNwidth,	maxWidth ); ac++;

	    ed->edVisibleRect.drX1 += maxWidth- width;
	    }
	}

    if  ( ed->edShellExtraHeight >= 0 )
	{
	int	maxHeight;

	maxHeight= add->addBackRect.drY1- add->addBackRect.drY0;

	if  ( maxHeight < add->addPaperRect.drY1- add->addPaperRect.drY0 )
	    { maxHeight=  add->addPaperRect.drY1- add->addPaperRect.drY0; }

	maxHeight += ed->edShellExtraHeight;

	XtSetArg( al[ac],	XmNmaxHeight,	maxHeight ); ac++;

	if  ( height > maxHeight )
	    {
	    XtSetArg( al[ac],	XmNheight,	maxHeight ); ac++;

	    ed->edVisibleRect.drY1 += maxHeight- height;
	    }
	}

    if  ( ac > 0 )
	{ XtSetValues( ed->edTopWidget, al, ac );	}

    return;
    }

static void appFileAdaptHorizontalRulerRange(	EditDocument *	ed,
						int		width )
    {
    EditApplication *		ea= ed->edApplication;

    if  ( ed->edTopRuler && ea->eaSetHorizontalRulerRange )
	{
	Dimension	leftRulerWidth= 0;

	if  ( ed->edLeftRulerWidget )
	    {
	    XtVaGetValues( ed->edLeftRulerWidget,
				XmNwidth,		&leftRulerWidth,
				NULL );
	    }

	(*ea->eaSetHorizontalRulerRange)( ed->edTopRuler,
					ed->edTopRulerWidget,
					ed->edVisibleRect.drX0,
					ed->edVisibleRect.drX1,
					width+ leftRulerWidth );
	}

    return;
    }

static void appFileAdaptVerticalRulerRange(	EditDocument *	ed,
						int		height )
    {
    EditApplication *		ea= ed->edApplication;

    if  ( ed->edLeftRuler && ea->eaSetVerticalRulerRange )
	{
	(*ea->eaSetVerticalRulerRange)( ed->edLeftRuler,
					ed->edLeftRulerWidget,
					ed->edVisibleRect.drY0,
					ed->edVisibleRect.drY1,
					height );
	}

    return;
    }

static void appAdaptToDocumentSize(	EditDocument *	ed,
					int		height,
					int		width )
    {
    Widget		w= ed->edDocumentWidget;
    AppDrawingData *	add= &(ed->edDrawingData);
    int			changed= 0;
    int			d;

    if  ( ed->edVisibleRect.drY1 != ed->edVisibleRect.drY0+ height- 1 )
	{
	ed->edVisibleRect.drY1= ed->edVisibleRect.drY0+ height- 1;
	changed= 1;

	if  ( ed->edVisibleRect.drY1 > add->addBackRect.drY1 )
	    {
	    d= ed->edVisibleRect.drY1- add->addBackRect.drY1;

	    if  ( d > ed->edVisibleRect.drY0 )
		{ d= ed->edVisibleRect.drY0;	}

	    ed->edVisibleRect.drY0 -= d;
	    ed->edVisibleRect.drY1 -= d;
	    }

	appFileAdaptVerticalRulerRange( ed, height );

	XClearArea( XtDisplay( w ), XtWindow( w ), 0, 0, 0, 0, True );
	}

    if  ( ed->edVisibleRect.drX1 != ed->edVisibleRect.drX0+ width+ 1 )
	{
	ed->edVisibleRect.drX1= ed->edVisibleRect.drX0+ width+ 1;
	changed= 1;

	if  ( ed->edVisibleRect.drX1 > add->addBackRect.drX1 )
	    {
	    d= ed->edVisibleRect.drX1- add->addBackRect.drX1;

	    if  ( d > ed->edVisibleRect.drX0 )
		{ d= ed->edVisibleRect.drX0;	}

	    ed->edVisibleRect.drX0 -= d;
	    ed->edVisibleRect.drX1 -= d;
	    }

	appFileAdaptHorizontalRulerRange( ed, width );

	XClearArea( XtDisplay( w ), XtWindow( w ), 0, 0, 0, 0, True );
	}

    if  ( changed )
	{ appDocSetScrollbarValues( ed );	}

    return;
    }

static void appDocTopConfigure(	Widget			w,
				void *			voided,
				XEvent *		event,
				Boolean *		pRefused )
    {
    EditDocument *	ed= (EditDocument *)voided;

    if  ( event->type == MapNotify && ed->edShellExtraWidth < 0 )
	{
	Dimension	shellWidth;
	Dimension	shellHeight;

	Dimension	docWidth;
	Dimension	docHeight;

	XtVaGetValues( ed->edTopWidget,
			    XmNwidth,		&shellWidth,
			    XmNheight,		&shellHeight,
			    NULL );

	XtVaGetValues( ed->edDocumentWidget,
			    XmNwidth,		&docWidth,
			    XmNheight,		&docHeight,
			    NULL );

	ed->edShellExtraWidth= shellWidth- docWidth;
	ed->edShellExtraHeight= shellHeight- docHeight;

	appSetShellConstraints( ed );

	appAdaptToDocumentSize( ed, docHeight, docWidth );

	ed->edMapped= 1;
	}

    if  ( event->type == ConfigureNotify )
	{
	XConfigureEvent *	cevent= (XConfigureEvent *)event;

	if  ( ! ed->edMapped && cevent->y < 0 )
	    {
	    XtVaSetValues( ed->edTopWidget,
				XmNy,		0,
				NULL );
	    }
	}

    *pRefused= 1;
    return;
    }

static void appDocConfigure(	Widget			w,
				void *			voided,
				XEvent *		event,
				Boolean *		pRefused )
    {
    XConfigureEvent *	cevent= &(event->xconfigure);
    EditDocument *	ed= (EditDocument *)voided;

    if  ( event->type != ConfigureNotify )
	{ return;	}

    appAdaptToDocumentSize( ed, cevent->height, cevent->width );

    *pRefused= 1;
    return;
    }

/************************************************************************/
/*									*/
/*  Make a document window.						*/
/*									*/
/*  1)  Allocate a document.						*/
/*  3)  Make a shell to contain everything.				*/
/*  4)  Make the window pane.						*/
/*									*/
/************************************************************************/

static int appDocMakeMainWindow(	EditDocument *		ed,
					const char *		title,
					EditApplication *	ea )
    {
    Arg			al[20];
    int			ac= 0;
    Pixmap		pixmap= (Pixmap)0;

    const char *	baseName;

    int			len;
    static char *	fullTitle;
    static char *	fullIconName;

    int			maxWidth= ( 5* ea->eaScreenPixelsWide )/ 6;
    int			maxHeight= ( 5* ea->eaScreenPixelsHigh )/ 6;

    baseName= strrchr( title, '/' );
    if  ( baseName )
	{ baseName++;	}
    else{ baseName= title;	}

    if  ( ea->eaDocumentIcon					&&
	  appGetImagePixmap( ea, ea->eaDocumentIcon, &pixmap )	)
	{ SDEB(ea->eaDocumentIcon); return -1;	}

    len= strlen( ea->eaApplicationName )+ 2+ strlen( title )+ 1;
    fullTitle= (char *)realloc( fullTitle, len );
    if  ( ! fullTitle )
	{ LXDEB(len,fullTitle); return -1;	}

    len= strlen( ea->eaApplicationName )+ 2+ strlen( baseName )+ 1;
    fullIconName= (char *)realloc( fullIconName, len );
    if  ( ! fullIconName )
	{ LXDEB(len,fullIconName); return -1;	}

    sprintf( fullTitle, "%s: %s", ea->eaApplicationName, title );
    sprintf( fullIconName, "%s: %s", ea->eaApplicationName, baseName );

    XtSetArg( al[ac], XmNtitle,			fullTitle ); ac++;
    XtSetArg( al[ac], XmNiconName,		fullIconName ); ac++;
    XtSetArg( al[ac], XmNuserData,		(void *)ed ); ac++;
    XtSetArg( al[ac], XmNdeleteResponse,	XmDO_NOTHING ); ac++;
    XtSetArg( al[ac], XmNallowShellResize,	True ); ac++;
    XtSetArg( al[ac], XmNmappedWhenManaged,	False ); ac++;
    XtSetArg( al[ac], XmNinput,			True ); ac++;
    XtSetArg( al[ac], XmNmaxWidth,		maxWidth ); ac++;
    XtSetArg( al[ac], XmNmaxHeight,		maxHeight ); ac++;

    if  ( pixmap )
	{ XtSetArg( al[ac], XmNiconPixmap,	pixmap ); ac++; }

#   if 1
    {
    int		argc= 1;
    char *	argv[1];

    argv[0]= ea->eaApplicationName;

    XtSetArg( al[ac], XmNargc,		argc ); ac++;
    XtSetArg( al[ac], XmNargv,		argv ); ac++;
    }
#   endif

    ed->edTopWidget= XtAppCreateShell( ea->eaApplicationName, 
					    ea->eaDocumentWidgetName,
					    applicationShellWidgetClass,
					    ea->eaDisplay, al, ac );

    if  ( ea->eaCloseAtom != None )
	{
	XmAddWMProtocolCallback( ed->edTopWidget, ea->eaCloseAtom,
					appDocFileClose, (XtPointer)ed );
	}

    if  ( ea->eaSaveAtom != None )
	{
	XmAddWMProtocolCallback( ed->edTopWidget, ea->eaSaveAtom,
				    appDocFileSaveYourself, (XtPointer)ed );
	}

    ac= 0;
    XtSetArg( al[ac], XmNuserData, (void *)ed ); ac++;
    ed->edMainWindow= XmCreateMainWindow( ed->edTopWidget, WIDGET_NAME, al, ac );

    appDocMakeMenu( ed, ea, ed->edMainWindow );

    ac= 0;
    ed->edMainForm= XmCreateForm( ed->edMainWindow, WIDGET_NAME, al, ac );

    return 0;
    }

int appSetDocumentFilename(	EditDocument *		ed,
				char *			filename )
    {
    char *			s;

    s= strdup( filename );
    if  ( ! s )
	{ XDEB(s); return -1;	}

    if  ( ed->edFilename )
	{ free( ed->edFilename );	}
    ed->edFilename= s;

    return 0;
    }

int appSetDocumentTitle(	EditDocument *		ed,
				char *			title )
    {
    EditApplication *		ea= ed->edApplication;

    char *			s;
    char *			baseName;

    int				len;
    static char *		fullTitle;
    static char *		fullIconName;

    s= strdup( title );
    if  ( ! s )
	{ XDEB(s); return -1;	}

    if  ( ed->edTitle )
	{ free( ed->edTitle );	}
    ed->edTitle= s;

    baseName= strrchr( ed->edTitle, '/' );
    if  ( baseName )
	{ baseName++;			}
    else{ baseName= ed->edTitle;	}

    len= strlen( ea->eaApplicationName )+ 2+ strlen( title )+ 1;
    fullTitle= (char *)realloc( fullTitle, len );
    if  ( ! fullTitle )
	{ LXDEB(len,fullTitle); return -1;	}

    len= strlen( ea->eaApplicationName )+ 2+ strlen( baseName )+ 1;
    fullIconName= (char *)realloc( fullIconName, len );
    if  ( ! fullIconName )
	{ LXDEB(len,fullIconName); return -1;	}

    sprintf( fullTitle, "%s: %s", ea->eaApplicationName, title );
    sprintf( fullIconName, "%s: %s", ea->eaApplicationName, baseName );

    XtVaSetValues( ed->edTopWidget,
			XmNtitle,		fullTitle,
			XmNiconName,		fullIconName,
			NULL );

    appRenameDocumentOptions( ed->edApplication, ed, ed->edTitle );

    return 0;
    }


static int appDocMakeToolbar(	EditDocument *		ed,
				EditApplication *	ea )
    {
#   if 0
    Arg			al[20];
    int			ac= 0;

    Display *		display= ea->eaDisplay;
    int			screen= ea->eaScreen;

    int			verPixPerCM;

    verPixPerCM= ( 10* DisplayHeight( display, screen ) )/
					DisplayHeightMM( display, screen );
    XtSetArg( al[ac], XmNleftAttachment,	XmATTACH_FORM ); ac++;
    XtSetArg( al[ac], XmNtopAttachment,		XmATTACH_FORM ); ac++;
    XtSetArg( al[ac], XmNrightAttachment,	XmATTACH_FORM ); ac++;
    XtSetArg( al[ac], XmNheight,		verPixPerCM ); ac++;

    ed->edToolbar= XmCreateForm( ed->edMainForm, WIDGET_NAME, al, ac );

#   endif
    return 0;
    }

static int appDocMakeScrolledWindow(	EditDocument *		ed,
					EditApplication *	ea )
    {
    Arg			al[20];
    int			ac= 0;
    
    ac= 0;
    XtSetArg( al[ac], XmNleftAttachment,	XmATTACH_FORM ); ac++;
    if  ( ed->edToolbar )
	{
	XtSetArg( al[ac], XmNtopAttachment,	XmATTACH_WIDGET ); ac++;
	XtSetArg( al[ac], XmNtopWidget,		ed->edToolbar ); ac++;
	}
    else{
	XtSetArg( al[ac], XmNtopAttachment,	XmATTACH_FORM ); ac++;
	}
    XtSetArg( al[ac], XmNrightAttachment,	XmATTACH_FORM ); ac++;
    XtSetArg( al[ac], XmNbottomAttachment,	XmATTACH_FORM ); ac++;
    XtSetArg( al[ac], XmNscrollingPolicy,	XmAPPLICATION_DEFINED ); ac++;
    XtSetArg( al[ac], XmNscrollBarDisplayPolicy, XmSTATIC ); ac++;
    XtSetArg( al[ac], XmNhighlightThickness,	0 ); ac++;

    ed->edScrolledWindow= XmCreateScrolledWindow( ed->edMainForm, WIDGET_NAME, al, ac );

    ac= 0;
    XtSetArg( al[ac], XmNorientation,		XmVERTICAL ); ac++;
    XtSetArg( al[ac], XmNprocessingDirection,	XmMAX_ON_BOTTOM ); ac++;
    ed->edVerticalScrollbar=
		    XmCreateScrollBar( ed->edScrolledWindow, WIDGET_NAME, al, ac );

    XtAddCallback( ed->edVerticalScrollbar, XmNvalueChangedCallback,
			    ea->eaDocVerticalScrollbarCallback, (XtPointer)ed );
    XtAddCallback( ed->edVerticalScrollbar, XmNdragCallback,
			    ea->eaDocVerticalScrollbarCallback, (XtPointer)ed );

    ac= 0;
    XtSetArg( al[ac], XmNorientation,		XmHORIZONTAL ); ac++;
    XtSetArg( al[ac], XmNprocessingDirection,	XmMAX_ON_RIGHT ); ac++;
    ed->edHorizontalScrollbar=
		    XmCreateScrollBar( ed->edScrolledWindow, WIDGET_NAME, al, ac );

    XtAddCallback( ed->edHorizontalScrollbar, XmNvalueChangedCallback,
			    appDocHorizontalScrollbarCallback, (XtPointer)ed );
    XtAddCallback( ed->edHorizontalScrollbar, XmNdragCallback,
			    appDocHorizontalScrollbarCallback, (XtPointer)ed );

    return 0;
    }

/************************************************************************/
/*  Free all resources allocated for a document.			*/
/************************************************************************/

void appFreeDocument(		EditApplication *	ea,
				EditDocument *		ed )
    {
    if  ( ed->edFilename )
	{ free( ed->edFilename );	}
    if  ( ed->edTitle )
	{ free( ed->edTitle );	}

    appCleanDrawingData( &(ed->edDrawingData) );
    appCleanColors( ea->eaDisplay, &(ed->edColors) );

    free( ed );

    return;
    }

int appMakeDocumentWindow(	EditDocument **		pEd,
				EditApplication *	ea,
				int			readonly,
				const char *		title,
				const char *		filename )
    {
    EditDocument *	ed;

    /*  1  */
    ed= (EditDocument *)malloc( sizeof(EditDocument) );
    if  ( ! ed )
	{ XDEB(ed); return -1;	}

    ed->edApplication= ea;

    ed->edFilename= (char *)0;
    ed->edTitle= (char *)0;
    ed->edFormat= -1;
    ed->edFileReadOnly= 0;

    ed->edTopWidget= (Widget)0;
    ed->edMenuBar= (Widget)0;
    ed->edMainWindow= (Widget)0;
	ed->edFileMenu= (Widget)0;
	ed->edFileMenuButton= (Widget)0;
	ed->edEditMenu= (Widget)0;
	ed->edEditMenuButton= (Widget)0;
	ed->edWindowMenu= (Widget)0;
	ed->edWindowMenuButton= (Widget)0;
	ed->edHelpMenu= (Widget)0;
	ed->edHelpMenuButton= (Widget)0;

    ed->edMainForm= (Widget)0;
    ed->edToolbar= (Widget)0;
    ed->edScrolledWindow= (Widget)0;
	ed->edVerticalScrollbar= (Widget)0;
	ed->edHorizontalScrollbar= (Widget)0;
	ed->edDocumentWidget= (Widget)0;

	ed->edLeftRulerWidget= (Widget)0;
	ed->edTopRulerWidget= (Widget)0;
	ed->edRightRulerWidget= (Widget)0;
	ed->edBottomRulerWidget= (Widget)0;

	ed->edLeftRuler= (void *)0;
	ed->edTopRuler= (void *)0;
	ed->edRightRuler= (void *)0;
	ed->edBottomRuler= (void *)0;

    ed->edGc= (GC)0;
    ed->edInputContext= (XIC)0;

    ed->edLeftRulerWidth= 0;
    ed->edTopMargin= 0;
    ed->edRightRulerWidth= 0;
    ed->edBottomMargin= 0;

    ed->edHasBeenChanged= 0;
    ed->edIsReadonly= readonly;
    ed->edIsVisible= 0;

    ed->edShellExtraWidth= -1;
    ed->edShellExtraHeight= -1;

    appInitColors( &(ed->edColors) );
    appInitDrawingData( &(ed->edDrawingData) );

    ed->edMapped= 0;

    if  ( filename )
	{ ed->edFilename= strdup( filename );	}
    else{ ed->edFilename= (char *)0;		}

    if  ( title )
	{ ed->edTitle= strdup( title );	}
    else{ ed->edTitle= (char *)0;	}

    if  ( ea->eaSetPrivateData && (*ea->eaSetPrivateData)( ea, ed ) )
	{ appFreeDocument( ea, ed ); return -1; }

    /*  3  */
    if  ( appDocMakeMainWindow( ed, title, ea )			||
	  appDocMakeToolbar( ed, ea )					||
	  appDocMakeScrolledWindow( ed, ea )				||
	  (*ea->eaMakeDocumentWidget)( ea, ed, ed->edScrolledWindow )	)
	{ appFreeDocument( ea, ed ); return -1; }

    XtAddEventHandler( ed->edDocumentWidget, StructureNotifyMask, False,
					appDocConfigure, (void *)ed );
    XtAddEventHandler( ed->edTopWidget, StructureNotifyMask, False,
					appDocTopConfigure, (void *)ed );

    if  ( ea->eaDocSelectionTypeCount > 0 )
	{
	XtAddEventHandler( ed->edDocumentWidget,
			PropertyChangeMask, True, appDocSelectionInput, ed );
	}

    XtManageChild( ed->edVerticalScrollbar );
    XtManageChild( ed->edHorizontalScrollbar );

    XtManageChild( ed->edDocumentWidget );
    XtManageChild( ed->edScrolledWindow );

    if  ( ed->edToolbar )
	{ XtManageChild( ed->edToolbar ); }

    XtManageChild( ed->edMainForm );
    XtManageChild( ed->edMainWindow );

    *pEd= ed; return 0;
    }

void appDocSetScrollbarValues(	EditDocument *	ed )
    {
    AppDrawingData *	add= &(ed->edDrawingData);

    int			sliderSize;
    int			maximum;
    int			minimum;
    int			value;

    sliderSize= ed->edVisibleRect.drY1- ed->edVisibleRect.drY0+ 1;
    minimum= add->addBackRect.drY0;
    value= ed->edVisibleRect.drY0;

    if  ( sliderSize >= add->addBackRect.drY1+ 1 )
	{ maximum= ed->edVisibleRect.drY1+ 1;	}
    else{ maximum= add->addBackRect.drY1+ 1;	}

    if  ( value+ sliderSize > maximum )
	{ value= maximum- sliderSize; }

    XtVaSetValues( ed->edVerticalScrollbar,
			XmNminimum,		minimum,
			XmNmaximum,		maximum,
			XmNvalue,		value,
			XmNsliderSize,		sliderSize,
			XmNpageIncrement,	( 9* sliderSize+ 9 )/10,
			XmNincrement,		( sliderSize+ 19 )/ 20,
			NULL );

    sliderSize= ed->edVisibleRect.drX1- ed->edVisibleRect.drX0+ 1;
    minimum= add->addBackRect.drX0;
    value= ed->edVisibleRect.drX0;

    if  ( sliderSize >= add->addBackRect.drX1+ 1 )
	{ maximum= ed->edVisibleRect.drX1+ 1;	}
    else{ maximum= add->addBackRect.drX1+ 1;	}

    XtVaSetValues( ed->edHorizontalScrollbar,
			XmNminimum,	minimum,
			XmNmaximum,	maximum,
			XmNvalue,	value,
			XmNsliderSize,	sliderSize,
			NULL );

    return;
    }

int appSetupDocument(	EditApplication *	ea,
			EditDocument *		ed	)
    {
    AppDrawingData *	add= &(ed->edDrawingData);

    add->addPhysicalFontList.apflAfmDirectory= ea->eaAfmDirectory;

    if  ( (*ea->eaSetupDocument)( ea, ed ) )
	{ SDEB(ed->edFilename); return -1; }

    appDocSetScrollbarValues( ed );

    XtRealizeWidget( ed->edTopWidget );

    if  ( ea->eaFinishDocumentSetup && (*ea->eaFinishDocumentSetup)( ea, ed ) )
	{ SDEB(ed->edFilename); return -1; }

    XtMapWidget( ed->edTopWidget );

    ed->edIsVisible= 1;
    ed->edHasBeenChanged= 0;

    if  ( ! ea->eaCurrentDocMenuPixel )
	{
	Display *	display= ea->eaDisplay;
	int		screen= ea->eaScreen;
	Colormap	colorMap= DefaultColormap( display, screen );
	const char *	focusColor= "CadetBlue";

	XColor		color;
	XColor		ignored;

	XtVaGetValues( ed->edMenuBar,
			    XmNbackground,	&ea->eaOtherDocMenuPixel,
			    NULL );

	/* mwm:		CadetBlue:	r=96,g=160,b=160 */
	/* fvwm:	#c06077		r=192,g=96,b=119 */

	if  ( ! XAllocNamedColor( display, colorMap, focusColor,
						    &color, &ignored ) )
	    {
	    SDEB(focusColor);
	    ea->eaCurrentDocMenuPixel= ea->eaOtherDocMenuPixel;
	    }
	else{ ea->eaCurrentDocMenuPixel= color.pixel; }
	}

    if  ( ea->eaDocumentCursor )
	{
	XDefineCursor( ea->eaDisplay, XtWindow( ed->edDocumentWidget ),
						    ea->eaDocumentCursor );
	}

    appDocVisible( ea, ed, ed->edIsVisible );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Open a a document.							*/
/*									*/
/*  1)  Allocate a document.						*/
/*  3)  Make a shell to contain everything.				*/
/*  4)  Make the window pane.						*/
/*									*/
/************************************************************************/
EditDocument * appOpenDocument(	EditApplication *	ea,
				Widget			relative,
				Widget			option,
				int			readonly,
				const char *		filename	)
    {
    EditDocument *		ed;
    int				fileReadonly= 0;
    AppFileMessageResources *	afmr= &(ea->eaFileMessageResources);

    if  ( readonly )
	{ fileReadonly= 1;	}
    else{
	if  ( appTestFileReadable( filename ) )
	    {
	    int	resp;

	    resp= appQuestionRunSubjectOkCancelDialog( ea,
					relative, option,
					filename, afmr->afmrFileReadOnly,
					(char *)0, (char *)0 );

	    if  ( resp != AQDrespOK )
		{ return (EditDocument *)0;	}

	    fileReadonly= 1;
	    }
	}

    /*  1,3,4  */
    if  ( appMakeDocumentWindow( &ed, ea, readonly, filename, filename ) )
	{ SDEB(filename); return (EditDocument *)0;	}

    ed->edFileReadOnly= fileReadonly;

    if  ( (*ea->eaOpenDocument)( ea, ed, relative, option, filename ) )
	{ return (EditDocument *)0; }

    if  ( appSetupDocument( ea, ed ) )
	{ SDEB(filename); return (EditDocument *)0; }

    appSetDocument( ea, ed, filename );

    return ed;
    }

int appNewDocument(	EditApplication *	ea,
			const char *		filename )
    {
    const int			readonly= 0;
    const char *		title= filename;
    char			scratch[100];

    EditDocument *		ed;
    AppFileMessageResources *	afmr= &(ea->eaFileMessageResources);

    /*  1,3,4  */
    if  ( ! filename )
	{
	static int		count= 1;

	sprintf( scratch, afmr->afmrNamelessTitleFormat, count++ );

	title= scratch;
	}
    else{ scratch[0]= '\0';	}

    if  ( appMakeDocumentWindow( &ed, ea, readonly, title, filename ) )
	{ SDEB(scratch); return -1;	}

    if  ( (*ea->eaNewDocument)( ea, ed, filename ) )
	{ SDEB(title); return -1; }

    if  ( appSetupDocument( ea, ed ) )
	{ SDEB(title); return -1; }

    appSetDocument( ea, ed, title );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Calculate magnification factors.					*/
/*									*/
/************************************************************************/
void appGetFactors(	EditApplication *	ea,
			double *		pHorPixPerMM,
			double *		pVerPixPerMM,
			double *		pXfac,
			double *		pYfac )
    {
    Display *		display= ea->eaDisplay;
    int			screen= ea->eaScreen;

    double		horPixPerMM;
    double		verPixPerMM;
    double		xfac;
    double		yfac;

    horPixPerMM= ( (double)ea->eaScreenPixelsWide )/
					DisplayWidthMM( display, screen );
    verPixPerMM= ( (double)ea->eaScreenPixelsHigh )/
					DisplayHeightMM( display, screen );

    xfac=  ( 25.4/ ( 20.0* 72.0 ) )* horPixPerMM;
    yfac=  ( 25.4/ ( 20.0* 72.0 ) )* verPixPerMM;

    *pHorPixPerMM= horPixPerMM;
    *pVerPixPerMM= verPixPerMM;

    *pXfac= ea->eaMagnification* xfac;
    *pYfac= ea->eaMagnification* yfac;

    ea->eaMagnifiedPixelsPerTwip= ea->eaMagnification* xfac;

    return;
    }


/************************************************************************/
/*									*/
/*  Finish the setup of the document after it is realised.		*/
/*									*/
/************************************************************************/

static XIMStyle	appXimPreferences[]=
    {
    XIMPreeditCallbacks,
    XIMPreeditPosition,
    XIMPreeditArea,
    XIMPreeditNothing,
    XIMStatusCallbacks,
    XIMStatusArea,
    XIMStatusNothing,
    };

int appFinishDocumentSetup(	EditApplication *	ea,
				EditDocument *		ed )
    {
    ed->edGc= XCreateGC( ea->eaDisplay, XtWindow( ed->edDocumentWidget ),
								    0, NULL );

    if  ( ea->eaInputMethod )
	{
	XIMStyles *	availableStyles;
	XIMStyle	supportedStyles;
	XIMStyle	likedStyle;
	int		style;

	XGetIMValues( ea->eaInputMethod,
		XNQueryInputStyle, &availableStyles,
		NULL );

	supportedStyles=	XIMPreeditArea		|
				XIMPreeditNothing	|
				XIMPreeditNone		|
				XIMStatusArea		|
				XIMStatusNothing	|
				XIMStatusNone		;

	likedStyle= 0;
	for ( style= 0; style < availableStyles->count_styles; style++ )
	    {
	    XIMStyle	foundStyle= availableStyles->supported_styles[style];
	    unsigned	j;

	    if  ( ( supportedStyles & foundStyle ) != foundStyle )
		{ continue;	}

	    if  ( ! likedStyle )
		{ likedStyle= foundStyle; continue;	}

	    for ( j= 0; j < sizeof(appXimPreferences)/sizeof(XIMStyle); j++ )
		{
		if  (   ( foundStyle & appXimPreferences[j] )	&&
		      ! ( likedStyle & appXimPreferences[j] )	)
		    { likedStyle= foundStyle; break;	}
		}
	    }

	if  ( likedStyle )
	    {
	    ed->edInputContext= XCreateIC( ea->eaInputMethod,
		XNClientWindow,		XtWindow( ed->edDocumentWidget ),
		XNInputStyle,		likedStyle,
		/*
		XNPeeditAttributes,	attributeList,
		XNStatusAttributes,	attributeList,
		*/
		NULL );
	    }
	else{ ed->edInputContext= (XIC)0;	}
	}
    else{ ed->edInputContext= (XIC)0;	}

    return 0;
    }
