/************************************************************************/
/*									*/
/*  Geometry calculations.						*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<limits.h>
#   define	y0	math_y0
#   define	y1	math_y1
#   include	<math.h>
#   undef	y0
#   undef	y1

#   include	"tedApp.h"
#   include	"docPs.h"

#   include	<debugon.h>

/************************************************************************/
/*									*/
/*  Forward declaration.						*/
/*									*/
/************************************************************************/

static int tedLayoutItemAndParents(	BufferItem *		bi,
					BufferDocument *	bd,
					AppDrawingData *	add,
					int			y0,
					int *			pY,
					int *			pLastY,
					int			andParents );

/************************************************************************/
/*									*/
/*  Calculate border width.						*/
/*									*/
/************************************************************************/

int tedBorderThick(		int *				pWide,
				const BorderProperties *	bp,
				const AppDrawingData *		add )
    {
    int			wide= bp->bpWidthTwips+ bp->bpSpacingTwips;
    int			thick= bp->bpWidthTwips;

    if  ( ! bp->bpIsSet )
	{ thick= 0;	}

    if  ( thick > 0 )
	{
	thick *= add->addMagnifiedPixelsPerTwip;
	wide  *= add->addMagnifiedPixelsPerTwip;

	if  ( thick == 0 )
	    { thick= 1;	}

	*pWide= wide;
	}
    else{ *pWide= 0;	}

    return thick;
    }

/************************************************************************/
/*									*/
/*  Make sure all fonts in a paragraph are known and open.		*/
/*									*/
/*  NOTE that inserted objects have a font, this is for two reasons:	*/
/*  *)  It determines the descender height below the object.		*/
/*  *)  Saving the object, and editing in its immediate neighbourhood	*/
/*	becomes MUCH easier.						*/
/*									*/
/************************************************************************/
static int tedOpenParaFonts(	BufferItem *		bi,
				BufferDocument *	bd,
				AppDrawingData *	add )
    {
    int			part;
    TextParticule *	tp;

    tp= bi->biParaParticules;
    for ( part= 0; part < bi->biParaParticuleCount; tp++, part++ )
	{
	if  ( tp->tpPhysicalFont < 0 )
	    {
	    tp->tpPhysicalFont= appOpenDocumentFont( add,
				    &(bd->bdFontList), tp->tpTextAttribute );

	    if  ( tp->tpPhysicalFont < 0	)
		{
		DEB(DEBFUN( "\"%.*s\"\n",
			(int)tp->tpStrlen, bi->biParaString+ tp->tpStroff ));
		LLDEB(part,tp->tpPhysicalFont);
		return -1;
		}
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Calculate the size of the objects for a paragraph.			*/
/*									*/
/************************************************************************/
static int tedSizeParaObjects(	BufferItem *		bi,
				BufferDocument *	bd,
				double			f )
    {
    int			part;
    TextParticule *	tp;

    tp= bi->biParaParticules;
    for ( part= 0; part < bi->biParaParticuleCount; tp++, part++ )
	{
	if  ( tp->tpKind != DOCkindOBJECT )
	    { continue;	}

	if  ( tp->tpPhysicalFont < 0 )
	    {
	    InsertedObject *	io= bi->biParaObjects+ tp->tpObjectNumber;

	    io->ioPixelsWide= TWIPStoPIXELS( f,
				( io->ioScaleX* io->ioTwipsWide )/ 100 );
	    io->ioPixelsHigh= TWIPStoPIXELS( f,
				(  io->ioScaleY* io->ioTwipsHigh )/ 100 );
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Calculate the width of a string.					*/
/*									*/
/************************************************************************/
static int tedTextWidth(	AppPhysicalFont *		apf,
				const unsigned char *		s,
				int				len )
    {
    if  ( apf->apfFontSet )
	{ return XmbTextEscapement( apf->apfFontSet, (const char *)s, len ); }
    else{ return XTextWidth( apf->apfFontStruct, (const char *)s, len ); }
    }

static int tedVisibleTextWidth(	AppPhysicalFont *		apf,
				const unsigned char *		s,
				int				len )
    {
    while( len > 0 && s[len- 1] == ' ' )
	{ len--;	}

    if  ( len == 0 )
	{ return 0;	}

    if  ( apf->apfFontSet )
	{ return XmbTextEscapement( apf->apfFontSet, (const char *)s, len ); }
    else{ return XTextWidth( apf->apfFontStruct, (const char *)s, len ); }
    }

/************************************************************************/
/*									*/
/*  Place as many particules on a line as possible.			*/
/*									*/
/*  1)  Accept at least one particule.					*/
/*  9)  Predictable undefied behaviour.					*/
/*									*/
/************************************************************************/
static int tedLayoutParticules(	const BufferItem *		bi,
				AppDrawingData *		add,
				const FormattingFrame *		ff,
				TextParticule *			tp,
				ParticuleData *			pd,
				int				particuleCount,
				int				x0 )
    {
    int				accepted;
    int				width;

    int				len= 0;
    int				xFrom= x0;
    const unsigned char *	from= bi->biParaString+ tp->tpStroff;

    int				physicalFont= tp->tpPhysicalFont;
    AppPhysicalFontList *	apfl= &(add->addPhysicalFontList);
    AppPhysicalFont *		apf= apfl->apflFonts+ physicalFont;

    accepted= 0;
    while( accepted < particuleCount )
	{
	switch( tp->tpKind )
	    {
	    int		x1;

	    case DOCkindTEXT:
		while( accepted < particuleCount		&&
		       tp->tpPhysicalFont == physicalFont	&&
		       tp->tpKind == DOCkindTEXT		)
		    {
		    len += tp->tpStrlen;
		    width= tedTextWidth( apf, from, len );

		    pd->pdVisiblePixels= tedVisibleTextWidth(
						apf,
						bi->biParaString+ tp->tpStroff,
						tp->tpStrlen );
		    pd->pdWhiteUnits= xFrom+ width- x0- pd->pdVisiblePixels;

		    tp->tpX0= x0;
		    tp->tpPixelsWide= xFrom+ width- x0;
		    x0= xFrom+ width;

		    accepted++; tp++;
		    pd++; break;
		    }

		break;
	    case DOCkindTAB:
		x1= TWIPStoPIXELS( add->addMagnifiedPixelsPerTwip,
							pd->pdTabPosition );

		pd->pdVisiblePixels= 0;
		pd->pdWhiteUnits= x1- x0;
		tp->tpObjectNumber= pd->pdTabNumber;

		tp->tpX0= x0;
		tp->tpPixelsWide= x1- x0;
		x0= x1;

		accepted++; tp++;
		pd++;
		break;

	    case DOCkindOBJECT:
		width= bi->biParaObjects[tp->tpObjectNumber].ioPixelsWide;

		pd->pdVisiblePixels= width;
		pd->pdWhiteUnits= 0;

		tp->tpX0= x0;
		tp->tpPixelsWide= width;
		x0 += width;

		accepted++; tp++;
		pd++;
		break;

	    case DOCkindFIELDSTART:
	    case DOCkindFIELDEND:
	    case DOCkindBKMKSTART:
	    case DOCkindBKMKEND:
	    case DOCkindXE:
	    case DOCkindTC:
		if  ( pd )
		    { pd->pdVisiblePixels= 0; pd->pdWhiteUnits= 0;	}

		tp->tpX0= x0;
		tp->tpPixelsWide= 0;
		accepted++; tp++;
		if  ( pd )
		    { pd++;	}
		break;

	    default:
		LDEB(tp->tpKind); return -1;
	    }

	if  ( accepted >= particuleCount )
	    { return accepted;	}

	physicalFont= tp->tpPhysicalFont;
	apf= apfl->apflFonts+ physicalFont;
	from= bi->biParaString+ tp->tpStroff;
	xFrom= x0;
	len= 0;
	}

    return accepted;
    }

/************************************************************************/
/*									*/
/*  Determine the height of a line.					*/
/*									*/
/************************************************************************/

static void tedLineHeight(	const BufferItem *	bi,
				const TextParticule *	tp,
				const AppDrawingData *	add,
				int			particuleCount,
				int *			pAscent,
				int *			pDescent )
    {
    const AppPhysicalFontList *	apfl= &(add->addPhysicalFontList);
    int				physicalFont= -1;
    AppPhysicalFont *		apf;
    int				ascent= 0;
    int				descent= 0;
    int				objectHeight= 0;

    int				leading;

    InsertedObject *		io;

    if  ( particuleCount <= 0 )
	{ LDEB(particuleCount); return;	}

    while( particuleCount > 0 )
	{
	switch( tp->tpKind )
	    {
	    case DOCkindTEXT:
	    case DOCkindTAB:
		physicalFont= tp->tpPhysicalFont;
		apf= apfl->apflFonts+ physicalFont;

		if  ( ascent < apf->apfAscentPixels )
		    { ascent=  apf->apfAscentPixels;	}
		if  ( descent < apf->apfDescentPixels )
		    { descent=  apf->apfDescentPixels; }
		break;

	    case DOCkindOBJECT:
		physicalFont= tp->tpPhysicalFont;
		apf= apfl->apflFonts+ physicalFont;

		if  ( ascent < apf->apfAscentPixels )
		    { ascent=  apf->apfAscentPixels;	}
		if  ( descent < apf->apfDescentPixels )
		    { descent=  apf->apfDescentPixels; }

		io= bi->biParaObjects+ tp->tpObjectNumber;

		if  ( objectHeight < io->ioPixelsHigh )
		    { objectHeight=  io->ioPixelsHigh;	}
		break;

	    case DOCkindFIELDSTART:
	    case DOCkindFIELDEND:
	    case DOCkindBKMKSTART:
	    case DOCkindBKMKEND:
	    case DOCkindXE:
	    case DOCkindTC:
		break;

	    default:
		LDEB(tp->tpKind); break;
	    }

	tp++; particuleCount--;
	}

    leading= ( ascent+ descent )/ LINEDISTFAC;

    if  ( ascent < objectHeight )
	{ ascent=  objectHeight; }

    *pAscent= ascent;
    *pDescent= descent;
    }

/************************************************************************/
/*									*/
/*  Justify the particules in a line of text.				*/
/*									*/
/*  1)  Start justification after the last tab of the line. Justifying	*/
/*	after anything else than a left tab is ridiculous. Simply	*/
/*	refuse to.							*/
/*  2)  Ignore organisational particules such as the delimitation of	*/
/*	links and bookmarks at the end of the line.			*/
/*									*/
/************************************************************************/

static void tedJustifyLine(	const BufferItem *	bi,
				TextParticule *		tp,
				ParticuleData *		pd,
				int			accepted,
				int			x1TestLines )
    {
    TextParticule *	tpp;
    ParticuleData *	pdd;

    int			i;
    int			extra;
    int			totalWeight;
    int			step;
    int			left;

    /*  1  */
    left= 0;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	if  ( tpp->tpKind == DOCkindTAB )
	    { left= i+ 1;	}
	}

    if  ( left > 0 )
	{
	if  ( pd[left-1].pdTabKind != DOCtkLEFT )
	    { LDEB(pd[left-1].pdTabKind); return;	}

	tp += left; pd += left; accepted -= left;
	}

    /*  2  */
    while( accepted > 0 && tp[accepted- 1].tpStrlen == 0 )
	{ accepted--;	}

    if  ( accepted < 2 )
	{ LDEB(accepted); return;	}

    extra= x1TestLines- tp[accepted- 1].tpX0- pd[accepted- 1].pdVisiblePixels;

    totalWeight= 0;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	pdd->pdWhiteUnits= 0;
	pdd->pdCorrectBy= 0;

	if  ( bi->biParaString[tpp->tpStroff+tpp->tpStrlen-1] == ' ' )
	    {
	    pdd->pdWhiteUnits=
		sqrt( (double)pdd[0].pdWidth+ (double)pdd[1].pdVisibleWidth );

	    totalWeight += pdd->pdWhiteUnits;
	    }
	}

    left= extra;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	if  ( pdd->pdWhiteUnits > 0 )
	    {
	    step= ( extra* pdd->pdWhiteUnits )/ totalWeight;
	    if  ( step > left )
		{ step= left;	}

	    pdd->pdCorrectBy += step;

	    left -= step;
	    }
	}

    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	if  ( pdd->pdWhiteUnits > 0 )
	    {
	    step= 1;
	    if  ( step > left )
		{ step= left;	}

	    pdd->pdCorrectBy += step;

	    left -= step;
	    }
	}

    step= 0;
    tpp= tp; pdd= pd;
    for ( i= 0; i < accepted- 1; tpp++, pdd++, i++ )
	{
	tpp->tpX0 += step;
	if  ( pdd->pdWhiteUnits > 0 )
	    {
	    tpp->tpPixelsWide += pdd->pdCorrectBy;
	    step += pdd->pdCorrectBy;
	    }
	}

    tpp->tpX0 += step;

    return;
    }

/************************************************************************/
/*									*/
/*  Recalculate the geometry from the start of the selection to the	*/
/*  end of the line.							*/
/*									*/
/*  'part'	is known to be the first particule of the line.		*/
/*									*/
/************************************************************************/

static void tedSqueezeParticules(	TextParticule *		tp,
					ParticuleData *		pd,
					int			accepted,
					int			x0,
					int			x1 )
    {
    TextParticule *	tpp= tp+ accepted- 1;
    ParticuleData *	pdd= pd+ accepted- 1;
    int			i;
    int			whitePixels= 0;
    int			shortage= tpp->tpX0+ pdd->pdVisiblePixels- x1;
    int			xx0;
    int			done;

    for ( i= 0; i < accepted- 1; i++ )
	{
	if  ( pd[i].pdWhiteUnits > 0 )
	    { whitePixels += pd[i].pdWhiteUnits; }

	pd[i].pdCorrectBy= 0;
	}

    if  ( whitePixels < shortage )
	{
	/*
	LLDEB(whitePixels,shortage);
	LLLDEB(x0,x1,tpp->tpX0+ pdd->pdVisiblePixels);
	LLDEB(whitePixels,tpp->tpX0+ pdd->pdVisiblePixels-x1);

	for ( i= 0; i < accepted; i++ )
	    { docListParticule( 0, "SH", part+ i, bi, tp+ i ); }
	*/
	}
    else{
	done= 0;

	for ( i= 0; i < accepted- 1; i++ )
	    {
	    int		step;

	    if  ( pd[i].pdWhiteUnits > 0 )
		{
		step= ( pd[i].pdWhiteUnits* shortage )/ whitePixels;

		if  ( step > 0 && done < shortage )
		    {
		    if  ( step > shortage- done )
			{ step= shortage- done;	}

		    pd[i].pdCorrectBy += step;
		    done += step;
		    }
		}
	    }

	for ( i= 0; i < accepted- 1 && done < shortage; i++ )
	    {
	    if  ( pd[i].pdWhiteUnits > pd[i].pdCorrectBy )
		{ pd[i].pdCorrectBy++; done++; }
	    }

	if  ( done != shortage )
	    { LLDEB(done,shortage);	}

	xx0= x0; whitePixels= 0;
	for ( i= 0; i < accepted- 1; i++ )
	    {
	    tp[i].tpX0 -= whitePixels;

	    if  ( pd[i].pdCorrectBy > 0 )
		{
		tp[i].tpPixelsWide -= pd[i].pdCorrectBy;
		whitePixels += pd[i].pdCorrectBy;
		}
	    }

	tp[i].tpX0 -= whitePixels;
	tp[i].tpPixelsWide= x1- tp[i].tpX0;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Layout one line on screen, reconciling PostScript and X11 geometry.	*/
/*									*/
/************************************************************************/

static int tedLayoutLine(	const BufferItem *	bi,
				AppDrawingData *	add,
				const FormattingFrame *	ff,
				TextParticule *		tp,
				ParticuleData *		pd,
				int			particuleCount,
				int			x0 )
    {
    int			past= 0;
    int			part= 0;
    int			x1;

    int			accepted;

    while( past < particuleCount )
	{
	int		squeeze= 0;
	int		includeTab= 0;
	int		j;

	TextParticule *	tpp;
	ParticuleData *	pdd;

	j= 0;
	while( past < particuleCount && tp[j].tpKind != DOCkindTAB )
	    { j++; past++;	}

	if  ( past < particuleCount )
	    { includeTab= 1; past++;	}

	accepted= tedLayoutParticules( bi, add, ff, tp, pd, past- part, x0 );

	if  ( accepted != past- part )
	    { LLDEB(accepted,past- part); return -1;	}

	tpp= tp+ accepted- 1;
	pdd= pd+ accepted- 1;

	if  ( includeTab )
	    {
	    x1= TWIPStoPIXELS( add->addMagnifiedPixelsPerTwip,
						    pdd->pdX0+ pdd->pdWidth );

	    if  ( tpp->tpX0+ pdd->pdVisiblePixels > x1	)
		{
		/*
		LLDEB(tpp->tpX0+ tpp->tpPixelsWide,x1);
		LLDEB(tpp->tpX0+ pdd->pdVisiblePixels,x1);
		docListParticule( 0, "WIDE", part, bi, tpp );
		*/

		squeeze= 1;
		}
	    else{
		if  ( tpp->tpX0+ tpp->tpPixelsWide != x1 )
		    {
		    if  ( tpp->tpX0 > x1 )
			{
			LLDEB(tpp->tpX0+ tpp->tpPixelsWide,x1);
			tpp->tpX0= x1;
			tpp->tpPixelsWide= 0;
			}
		    else{
			tpp->tpPixelsWide= x1- tpp->tpX0;
			}
		    }
		}
	    }
	else{
	    x1= ff->ffX1TextLines;

	    if  ( tpp->tpX0+ pdd->pdVisiblePixels > x1	)
		{
		/*
		LLDEB(tpp->tpX0+ tpp->tpPixelsWide,x1);
		LLDEB(tpp->tpX0+ pdd->pdVisiblePixels,x1);
		docListParticule( 0, "WIDE", part, bi, tpp );
		*/

		squeeze= 1;
		}
	    }

	if  ( squeeze )
	    { tedSqueezeParticules( tp, pd, accepted, x0, x1 ); }

	part += accepted; tp += accepted; pd += accepted;
	x0= x1; past= part;
	}

    return 0;
    }

static int tedLineBox(	const BufferItem *		bi,
			const BufferDocument *		bd,
			AppDrawingData *		add,
			const int			part,
			int				paraAscent,
			int				paraDescent,
			const FormattingFrame *		ffPixels,
			const FormattingFrame *		ffTwips,
			int				y,
			int *				pY,
			TextLine *			tl )
    {
    TextParticule *		tp;

    int				accepted;
    int				len;
    int				x1;

    int				ascent;
    int				descent;
    int				lineDist;

    int				xShift;
    int				x0Pixels;
    int				x0Twips;

    LineBox			lb;
    const DocumentGeometry *	dg= &(bd->bdGeometry);
    int				atTop= 0;
    ParticuleData *		pd;

    tp= bi->biParaParticules+ part;

    if  ( part == 0 )
	{
	x0Twips= ffTwips->ffX0FirstLine;
	x0Pixels= ffPixels->ffX0FirstLine;
	}
    else{
	x0Twips= ffTwips->ffX0TextLines;
	x0Pixels= ffPixels->ffX0TextLines;
	}

    if  ( docPsClaimParticuleData( bi, &pd ) )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    accepted= docPsLineBox( &lb, bi, part,
		    atTop, paraAscent, paraDescent, dg,
		    bd->bdProperties.dpTabIntervalTwips, &bd->bdFontList,
		    &(add->addPhysicalFontList),
		    tp, pd, x0Twips, ffTwips );

    if  ( accepted < 1 )
	{ LLDEB(accepted,bi->biParaParticuleCount); return -1;	}

    if  ( tedLayoutLine( bi, add, ffPixels, tp, pd, accepted, x0Pixels ) )
	{ LDEB(1); return -1;	}

    switch( bi->biParaAlignment )
	{
	case DOCiaLEFT:
	    xShift= 0; break;
	case DOCiaRIGHT:
	    x1= tp[accepted-1].tpX0+ pd[accepted-1].pdVisiblePixels;
	    xShift= ffPixels->ffX1TextLines- x1;
	    break;
	case DOCiaCENTERED:
	    x1= tp[accepted-1].tpX0+ pd[accepted-1].pdVisiblePixels;
	    xShift= ( ffPixels->ffX1TextLines- x1 )/ 2;
	    break;
	case DOCiaJUSTIFIED:
	    xShift= 0;
	    if  ( part+ accepted < bi->biParaParticuleCount )
		{
		tedJustifyLine( bi, tp, pd, accepted,
						ffPixels->ffX1TextLines );
		}
	    break;
	default:
	    LDEB(bi->biParaAlignment); xShift= 0; break;
	}

    if  ( xShift > 0 )
	{
	int	i;

	for ( i= 0; i < accepted; i++ )
	    { tp[i].tpX0 += xShift; }
	}

    len= tp[accepted-1].tpStroff+ tp[accepted-1].tpStrlen- tp->tpStroff;

    tedLineHeight( bi, bi->biParaParticules+ part, add, accepted,
							&ascent, &descent );

    lineDist= TWIPStoPIXELS(  add->addMagnifiedPixelsPerTwip,
							lb.lbLineDistance );

    tl->tlStroff= tp->tpStroff;
    tl->tlFirstParticule= part;

    tl->tlStrlen= len;
    tl->tlParticuleCount= accepted;
    tl->tlY= y+ ascent;
    tl->tlY0= y;
    tl->tlY1= y+ lineDist- 1;

    *pY= y+ lineDist;

    return accepted;
    }

/************************************************************************/
/*									*/
/*  Layout successive lines of a paragraph.				*/
/*									*/
/************************************************************************/

static int tedLayoutLines(	BufferItem *		bi,
				const BufferDocument *	bd,
				AppDrawingData *	add,
				int			part,
				int			line,
				int			paraAscent,
				int			paraDescent,
				const FormattingFrame *	ffPixels,
				const FormattingFrame *	ffTwips,
				int			y,
				int *			pY	)
    {
    TextParticule *	tp= bi->biParaParticules+ part;

    while( part < bi->biParaParticuleCount )
	{
	int		accepted;
	int		yBelow;

	const int	stroff= tp->tpStroff;

	TextLine *	tl;

	tl= docInsertTextLine( bi, line, stroff, part );
	if  ( ! tl )
	    { LDEB(bi->biParaLineCount); return -1;	}

	accepted= tedLineBox( bi, bd, add, part,
		paraAscent, paraDescent, ffPixels, ffTwips, y, &yBelow, tl );

	if  ( accepted < 1 )
	    { LDEB(accepted); return -1;	}

	part += accepted; tp += accepted; line++;
	y= yBelow;
	}

    *pY= y; return part;
    }

/************************************************************************/
/*									*/
/*  Derive the frame for a paragraph from the page rectangle and the	*/
/*  paragraph properties.						*/
/*									*/
/*  For paragraphs inside a table cell, geometry is derived from the	*/
/*  table column.							*/
/*									*/
/************************************************************************/

void tedParagraphFrame(		FormattingFrame *	ff,
				const AppDrawingData *	add,
				int			bottom,
				const BufferItem *	bi )
    {
    ff->ffX0Geometry= add->addDocRect.drX0;
    ff->ffX1Geometry= add->addDocRect.drX1;

    ff->ffY1= -1;

    if  ( bottom > 0 )
	{ ff->ffY1= bottom;	}

    if  ( bi->biParaInTable )
	{
	int			col;
	BufferItem *		cell= bi->biParent;
	BufferItem *		row= cell->biParent;
	CellProperties *	cp;

	cp= &(cell->biCellProperties);
	ff->ffX1Geometry= add->addDocRect.drX0+ cp->cpRightBoundaryPixels;

	col= cell->biNumberInParent;

	ff->ffX1Geometry -= row->biRowHalfGapWidthPixels;

	if  ( col == 0 )
	    { ff->ffX0Geometry += row->biRowLeftIndentPixels; }
	else{
	    BufferItem *		prevCell;

	    prevCell= row->biGroupChildren[col- 1];

	    cp= &(prevCell->biCellProperties);
	    ff->ffX0Geometry += cp->cpRightBoundaryPixels;
	    }

	ff->ffX0Geometry += row->biRowHalfGapWidthPixels;
	}

    ff->ffX0TextLines= ff->ffX0Geometry+ bi->biParaLeftIndentPixels;
    ff->ffX1TextLines= ff->ffX1Geometry- bi->biParaRightIndentPixels;

    ff->ffX0FirstLine= ff->ffX0TextLines+ bi->biParaFirstIndentPixels;

    return;
    }

/************************************************************************/
/*									*/
/*  Move successor items, because an item above them changed height.	*/
/*									*/
/************************************************************************/

static void tedPlaceItems(		BufferItem *		biParent,
					int			from,
					int			y,
					int *			pY )
    {
    int		i;
    TextLine *	tl;

    for ( i= from; i < biParent->biGroupChildCount; i++ )
	{
	BufferItem *	child= biParent->biGroupChildren[i];
	int		high;
	int		by;

	int		j;

	high= child->biY1- child->biY0+ 1;

	by= y- child->biY0;
	child->biY0= y;

	switch( child->biLevel )
	    {
	    case DOClevDOC:
	    case DOClevSECT:
	    case DOClevCELL:
	    rowAsGroup:
		tedPlaceItems( child, 0, y, &y );
		child->biY1= y- 1;
		break;

	    case DOClevROW:
		if  ( ! child->biRowHasTableParagraphs )
		    { goto rowAsGroup;	}

		for ( j= 0; j < child->biGroupChildCount; j++ )
		    {
		    BufferItem *	cellBi= child->biGroupChildren[j];
		    int			yy;

		    cellBi->biY0 += by;
		    cellBi->biY1 += by;

		    if  ( cellBi->biGroupChildCount > 0 )
			{
			tedPlaceItems( cellBi, 0,
				cellBi->biGroupChildren[0]->biY0+ by, &yy );
			}
		    }

		y= child->biY0+ high;
		child->biY1= y- 1;
		break;

	    case DOClevPARA:
		y += high;
		child->biY1= y- 1;

		tl= child->biParaLines;
		for ( j= 0; j < child->biParaLineCount; tl++, j++ )
		    {
		    tl->tlY += by;
		    tl->tlY0 += by;
		    tl->tlY1 += by;
		    }
		break;
	    default:
		LDEB(biParent->biLevel); break;
	    }
	}

    *pY= y; return;
    }

/************************************************************************/
/*									*/
/*  Adjust the geometry of a parent item to changes in a child.		*/
/*									*/
/************************************************************************/
static int tedAdjustParentGeometry(	BufferItem *		bi,
					AppDrawingData *	add,
					int			by,
					int *			pLastY )
    {
    int			lastY= *pLastY;

    int			y= bi->biY1+ 1;

    BufferItem *	biParent;

    biParent= bi->biParent;

    while( biParent )
	{
	int		changed= 0;

	switch( biParent->biLevel )
	    {
	    case DOClevDOC:
	    case DOClevSECT:
	    case DOClevCELL:
	    parentRowAsGroup:
		if  ( bi->biNumberInParent == 0			&&
		      bi->biY0 != biParent->biY0		)
		    {
		    LLDEB(biParent->biY0,bi->biY0);
		    biParent->biY0= bi->biY0; changed= 1;
		    }

		if  ( bi->biNumberInParent < biParent->biGroupChildCount-1 )
		    {
		    tedPlaceItems( biParent, bi->biNumberInParent+ 1, y, &y );
		    }

		if  ( biParent->biY1 != y- 1 )
		    {
		    if  ( lastY < biParent->biY1 )
			{ lastY=  biParent->biY1;	}

		    biParent->biY1= y- 1;
		    changed= 1;
		    }

		break;

	    case DOClevROW:
		if  ( ! biParent->biRowHasTableParagraphs )
		    { goto parentRowAsGroup;	}

		y= biParent->biY0;

		if  ( biParent->biRowHeightTwips != 0 )
		    { y += biParent->biRowHeightPixels;	}

		if  ( biParent->biRowHeightTwips >= 0 )
		    {
		    int		i;

		    for ( i= 0; i < biParent->biGroupChildCount; i++ )
			{
			if  ( y < biParent->biGroupChildren[i]->biY1+ 1 )
			    { y=  biParent->biGroupChildren[i]->biY1+ 1; }
			}
		    }

		if  ( biParent->biY1 != y- 1 )
		    {
		    if  ( lastY < biParent->biY1 )
			{ lastY=  biParent->biY1;	}

		    biParent->biY1= y- 1;
		    changed= 1;
		    }

		break;

	    default:
		LDEB(biParent->biLevel); return -1;
	    }

	if  ( ! changed )
	    { break;	}

	if  ( lastY < biParent->biY1 )
	    { lastY=  biParent->biY1;	}

	y= biParent->biY1+ 1;

	bi= biParent; biParent= bi->biParent;
	}

    if  ( ! biParent )
	{
	int	margin= add->addBackRect.drY1- add->addDocRect.drY1;

	if  ( lastY < add->addBackRect.drY1 )
	    { lastY=  add->addBackRect.drY1;	}

	add->addDocRect.drY1= y;
	add->addBackRect.drY1= y+ margin;

	if  ( lastY < add->addBackRect.drY1 )
	    { lastY=  add->addBackRect.drY1;	}
	}

    *pLastY= lastY; return 0;
    }

/************************************************************************/
/*									*/
/*  Recalculate the geometry from the start of the selection to the	*/
/*  end of the paragraph.						*/
/*									*/
/************************************************************************/

int tedRecalculateParaLayout(	BufferItem *			bi,
				BufferDocument *		bd,
				AppDrawingData *		add,
				int				line,
				int				yShift,
				int				stroffShift,
				int				upto,
				int				refY1,
				DocumentRectangle *		drChanged )
    {
    const DocumentGeometry *	dg= &(bd->bdGeometry);

    TextLine *			tl= bi->biParaLines+ line;
    int				part= tl->tlFirstParticule;

    int				accepted;
    int				off;

    int				y;
    int				yBelow;
    int				oldLineY1;

    FormattingFrame		ffPixels;
    FormattingFrame		ffTwips;

    int				paraAscent= 0;
    int				paraDescent= 0;

    int				by;
    int				uptoEnd= 1;

    if  ( tedSizeParaObjects( bi, bd, add->addMagnifiedPixelsPerTwip ) )
	{ LDEB(1); return -1;	}

    if  ( tedOpenParaFonts( bi, bd, add ) )
	{ LDEB(1); return -1;	}

    tedParagraphFrame( &ffPixels, add, -1, bi );
    docParagraphFrame( &ffTwips, dg, INT_MAX, bi );

    if  ( docPsParagraphLineExtents( &paraAscent, &paraDescent,
					&(add->addPhysicalFontList), bi ) )
	{ LDEB(1); return -1;	}

    y= tl->tlY0+ yShift;

    if  ( drChanged->drY0 > y )
	{
	drChanged->drY0=  y;

	if  ( drChanged->drX0 > add->addBackRect.drX0 )
	    { drChanged->drX0=  add->addBackRect.drX0;	}
	if  ( drChanged->drX1 < add->addBackRect.drX1 )
	    { drChanged->drX1=  add->addBackRect.drX1;	}
	}

    oldLineY1= tl->tlY1;

    accepted= tedLineBox( bi, bd, add, part,
					paraAscent, paraDescent,
					&ffPixels, &ffTwips, y, &yBelow, tl );
    if  ( accepted < 0 )
	{ LDEB(accepted); return -1;	}

    off= tl->tlStroff+ tl->tlStrlen;
    yShift= tl->tlY1- oldLineY1;
    line++; tl++; part += accepted; y= yBelow;

    if  ( line >= bi->biParaLineCount )
	{
	if  ( tedLayoutLines( bi, bd, add, part, line,
				    paraAscent, paraDescent,
				    &ffPixels, &ffTwips, y, &yBelow ) < 0 )
	    { LDEB(part); return -1;	}

	y= yBelow;
	line= bi->biParaLineCount;
	}
    else{
	while( part < bi->biParaParticuleCount )
	    {
	    if  ( line >= bi->biParaLineCount )
		{
		if  ( tedLayoutLines( bi, bd, add, part, line,
				    paraAscent, paraDescent,
				    &ffPixels, &ffTwips, y, &yBelow ) < 0 )
		    { LDEB(part); return -1;	}

		y= yBelow;
		line= bi->biParaLineCount;
		break;
		}

	    if  ( tl->tlStroff + stroffShift == off && off >= upto )
		{
		while( line < bi->biParaLineCount )
		    {
		    tl->tlFirstParticule= part;
		    part += tl->tlParticuleCount;
		    tl->tlY0 += yShift; tl->tlY += yShift; tl->tlY1 += yShift;
		    tl->tlStroff += stroffShift;
		    tl++; line++;
		    }

		y= yBelow;
		yBelow= bi->biY1+ 1+ yShift;
		uptoEnd= 0;

		break;
		}

	    oldLineY1= tl->tlY1;
	    accepted= tedLineBox( bi, bd, add, part,
					paraAscent, paraDescent,
					&ffPixels, &ffTwips, y, &yBelow, tl );
	    if  ( accepted < 0 )
		{ LDEB(accepted); return -1;	}

	    off= tl->tlStroff+ tl->tlStrlen;
	    yShift= tl->tlY1- oldLineY1;
	    line++; tl++; part += accepted; y= yBelow;
	    }

	if  ( line < bi->biParaLineCount )
	    { docDeleteLines( bi, line, bi->biParaLineCount- line ); }
	}

    if  ( yShift > 0 )
	{ y= yBelow;	}
    if  ( yShift < 0 )
	{ y= bi->biY1;	}

    if  ( uptoEnd )
	{ yBelow += bi->biParaSpaceAfterPixels; }

    by= ( yBelow- 1 )- refY1;

    bi->biY1= yBelow- 1;

    if  ( drChanged->drY1 < y || by != 0 )
	{
	if  ( drChanged->drY1 < y )
	    { drChanged->drY1=  y;	}

	if  ( drChanged->drX0 > add->addBackRect.drX0 )
	    { drChanged->drX0=  add->addBackRect.drX0;	}
	if  ( drChanged->drX1 < add->addBackRect.drX1 )
	    { drChanged->drX1=  add->addBackRect.drX1;	}
	}

    if  ( by != 0 && tedAdjustParentGeometry( bi, add, by, &drChanged->drY1 ) )
	{ LDEB(by); return -1;	}

    return part;
    }

/************************************************************************/
/*									*/
/*  Calculate the layout of a paragraph.				*/
/*									*/
/*  1)  Make sure all fonts in the paragraph are known and open.	*/
/*	Calculate the size of all objects. The latter MUST be done	*/
/*	first.								*/
/*									*/
/************************************************************************/

static int tedLayoutParaContents(	BufferItem *		bi,
					BufferDocument *	bd,
					AppDrawingData *	add,
					int			y,
					int *			pY	)
    {
    const DocumentGeometry *	dg= &(bd->bdGeometry);

    FormattingFrame		ffPixels;
    FormattingFrame		ffTwips;
    int				wide;

    int				paraAscent= 0;
    int				paraDescent= 0;

    tedParagraphFrame( &ffPixels, add, -1, bi );
    docParagraphFrame( &ffTwips, dg, INT_MAX, bi );

    if  ( docPsParagraphLineExtents( &paraAscent, &paraDescent,
					&(add->addPhysicalFontList), bi ) )
	{ LDEB(1); return -1;	}

    bi->biParaLineCount= 0;

    if  ( ! bi->biParaInTable && bi->biParaStartsOnNewPage )
	{
	y += add->addBackRect.drY1- add->addDocRect.drY1;
	y += add->addDocRect.drY0;
	}

    y += bi->biParaSpaceBeforePixels;

    if  ( bi->biParaTopBorder.bpIsSet )
	{
	tedBorderThick( &wide, &(bi->biParaTopBorder), add );
	y += wide;
	}
    else{
	if  ( bi->biParaBoxBorder.bpIsSet )
	    {
	    BufferItem *	prevBi= (BufferItem *)0;

	    if  ( bi->biNumberInParent > 0 )
		{
		prevBi= bi->biParent->biGroupChildren[
					    bi->biNumberInParent- 1];
		}
	    if  ( ! prevBi				||
		  ! prevBi->biParaBoxBorder.bpIsSet	)
		{
		tedBorderThick( &wide, &(bi->biParaBoxBorder), add );
		y += wide;
		}
	    }
	}

    if  ( tedLayoutLines( bi, bd, add, 0, 0,
				    paraAscent, paraDescent,
				    &ffPixels, &ffTwips, y, &y ) < 0 )
	{ LDEB(y); return -1;	}

    if  ( bi->biParaBottomBorder.bpIsSet )
	{
	tedBorderThick( &wide, &(bi->biParaBottomBorder), add );
	y += wide;
	}
    else{
	if  ( bi->biParaBoxBorder.bpIsSet )
	    {
	    BufferItem *	nextBi= (BufferItem *)0;

	    if  ( bi->biNumberInParent <
				bi->biParent->biGroupChildCount- 1 )
		{
		nextBi= bi->biParent->biGroupChildren[
					    bi->biNumberInParent+ 1];
		}
	    if  ( ! nextBi				||
		  ! nextBi->biParaBoxBorder.bpIsSet	)
		{
		tedBorderThick( &wide, &(bi->biParaBoxBorder), add );
		y += wide;
		}
	    }
	}

    y += bi->biParaSpaceAfterPixels;

    *pY= y; return 0;
    }

/************************************************************************/
/*									*/
/*  Calculate pixel values from the twip based geometry information.	*/
/*									*/
/************************************************************************/

void tedParaScreenGeometry(	BufferItem *		bi,
				const FormattingFrame *	ff,
				double			f )
    {
    int			tab;
    TabStop *		ts;

    bi->biParaLeftIndentPixels= TWIPStoPIXELS( f, bi->biParaLeftIndentTwips );
    bi->biParaFirstIndentPixels= TWIPStoPIXELS( f, bi->biParaFirstIndentTwips );
    bi->biParaRightIndentPixels= TWIPStoPIXELS( f, bi->biParaRightIndentTwips );

    bi->biParaSpaceBeforePixels= TWIPStoPIXELS( f, bi->biParaSpaceBeforeTwips );
    bi->biParaSpaceAfterPixels= TWIPStoPIXELS( f, bi->biParaSpaceAfterTwips );
    bi->biParaLineSpacingPixels= TWIPStoPIXELS( f, bi->biParaLineSpacingTwips );

    ts= bi->biParaTabStops;
    for ( tab= 0; tab < bi->biParaTabCount; ts++, tab++ )
	{ ts->tsPixels= ff->ffX0Geometry+ TWIPStoPIXELS( f, ts->tsTwips ); }

    return;
    }

static int tedLayoutParaItem(	BufferItem *		bi,
				BufferDocument *	bd,
				AppDrawingData *	add,
				int			y,
				int *			pY	)
    {
    FormattingFrame	ff;

    tedParagraphFrame( &ff, add, -1, bi );

    /*  1  */
    if  ( tedSizeParaObjects( bi, bd, add->addMagnifiedPixelsPerTwip ) )
	{ LDEB(1); return -1;	}

    if  ( tedOpenParaFonts( bi, bd, add ) )
	{ LDEB(1); return -1;	}

    tedParaScreenGeometry( bi, &ff, add->addMagnifiedPixelsPerTwip );

    if  ( tedLayoutParaContents( bi, bd, add, y, pY ) < 0 )
	{ LDEB(y); return -1;	}

    return 0;
    }

static int tedLayoutRowItem(	BufferItem *		row,
				BufferDocument *	bd,
				AppDrawingData *	add,
				int			y0,
				int *			pY	)
    {
    int			i;
    int			n;
    CellProperties *	cp;

    int			y= y0;

    double		xfac= add->addMagnifiedPixelsPerTwip;

    row->biRowHalfGapWidthPixels= TWIPStoPIXELS( xfac,
						row->biRowHalfGapWidthTwips );
    row->biRowLeftIndentPixels= TWIPStoPIXELS( xfac,
						row->biRowLeftIndentTwips );

    row->biRowHeightPixels= TWIPStoPIXELS( xfac, row->biRowHeightTwips );
    if  ( row->biRowHeightPixels < 0 )
	{ row->biRowHeightPixels= -row->biRowHeightPixels;	}

    cp= row->biRowCells;
    for ( i= 0; i < row->biRowCellCount; cp++, i++ )
	{
	cp->cpRightBoundaryPixels=
			    TWIPStoPIXELS( xfac, cp->cpRightBoundaryTwips );
	}

    n= row->biGroupChildCount;
    if  ( n > row->biRowCellCount )
	{ n=  row->biRowCellCount;	}

    cp= row->biRowCells;
    for ( i= 0; i < n; cp++, i++ )
	{ row->biGroupChildren[i]->biCellProperties= *cp; }

    while( i < row->biGroupChildCount )
	{
	docInitCellProperties(
			&(row->biGroupChildren[i]->biCellProperties) );
	i++;
	}

    for ( i= 0; i < row->biGroupChildCount; i++ )
	{
	int		yy;
	int		lastY;
	BufferItem *	cell= row->biGroupChildren[i];

	if  ( tedLayoutItemAndParents( cell, bd, add, y0, &yy, &lastY, 0 ) )
	    { LLDEB(i,y); return -1;	}

	if  ( y < yy )
	    { y=  yy;	}
	}

    if  ( row->biRowHeightTwips > 0 )
	{
	if  ( y < y0+ row->biRowHeightPixels )
	    { y=  y0+ row->biRowHeightPixels;	}
	}

    if  ( row->biRowHeightTwips < 0 )
	{ y= y0+ row->biRowHeightPixels;	}

    *pY= y; return 0;
    }

static int tedLayoutItemAndParents(	BufferItem *		bi,
					BufferDocument *	bd,
					AppDrawingData *	add,
					int			y0,
					int *			pY,
					int *			pLastY,
					int			andParents )
    {
    int			i;
    int			y;
    int			y1;
    int			by;
    int			lastY= *pLastY;

    y1= bi->biY1;

    bi->biY0= y= y0;

    if  ( lastY < bi->biY1 )
	{ lastY=  bi->biY1;	}

    switch( bi->biLevel )
	{
	case DOClevDOC:
	case DOClevSECT:
	case DOClevCELL:
	rowAsGroup:
	    for ( i= 0; i < bi->biGroupChildCount; i++ )
		{
		if  ( tedLayoutItemAndParents( bi->biGroupChildren[i],
						bd, add, y0, &y, &lastY, 0 ) )
		    { LLDEB(i,y); return -1;	}

		y0= y;
		}
	    break;
	case DOClevROW:
	    if  ( ! bi->biRowHasTableParagraphs )
		{ goto rowAsGroup;	}

	    if  ( tedLayoutRowItem( bi, bd, add, y0, &y ) )
		{ LDEB(y); return -1;	}
	    break;

	case DOClevPARA:
	    if  ( tedLayoutParaItem( bi, bd, add, y0, &y ) )
		{ LDEB(y); return -1;	}
	    break;
	default:
	    LDEB(bi->biLevel); return -1;
	}

    y1= y- 1;
    by= y1- bi->biY1;

    bi->biY1= y1;

    if  ( lastY < bi->biY1 )
	{ lastY=  bi->biY1;	}

    if  ( andParents						&&
	  tedAdjustParentGeometry( bi, add, by, &lastY )	)
	{ LDEB(by); return -1;	}

    *pLastY= lastY; *pY= y; return 0;
    }

int tedLayoutItem(	BufferItem *		bi,
			BufferDocument *	bd,
			AppDrawingData *	add,
			int			y0,
			int *			pY,
			int *			pLastY )
    {
    return tedLayoutItemAndParents( bi, bd, add, y0, pY, pLastY, 1 );
    }

int tedLayoutHeaderItem(	BufferItem *		bi,
				BufferDocument *	bd,
				AppDrawingData *	add,
				int			y0,
				int *			pY,
				int *			pLastY )
    {
    return tedLayoutItemAndParents( bi, bd, add, y0, pY, pLastY, 0 );
    }

/************************************************************************/
/*									*/
/*  Find the object in the document for an x,y position.		*/
/*									*/
/************************************************************************/
static BufferItem * tedFindItem(	BufferItem *		bi,
					const AppDrawingData *	add,
					int			x,
					int			y	)
    {
    int			i;
    BufferItem *	child;
    BufferItem *	found;

    if  ( y < bi->biY0 )
	{ /* LLDEB(y,bi->biY0); */ return (BufferItem *)0;	}
    if  ( y > bi->biY1 )
	{ /* LLDEB(y,bi->biY1); */ return (BufferItem *)0;	}

    switch( bi->biLevel )
	{
	case DOClevDOC:
	case DOClevSECT:
	case DOClevCELL:
	rowAsGroup:
	    for ( i= 0; i < bi->biGroupChildCount; i++ )
		{
		child= bi->biGroupChildren[i];

		if  ( child->biY0 <= y && y <= child->biY1 )
		    {
		    found= tedFindItem( child, add, x, y );
		    return found;
		    }
		}
	    LDEB(y); LLDEB(bi->biY0,bi->biY1);
	    for ( i= 0; i < bi->biGroupChildCount; i++ )
		{
		child= bi->biGroupChildren[i];
		LLDEB(child->biY0,child->biY1);
		}
	    break;

	case DOClevROW:
	    if  ( ! bi->biRowHasTableParagraphs )
		{ goto rowAsGroup;	}

	    for ( i= 0; i < bi->biGroupChildCount; i++ )
		{
		BufferItem *	para;

		child= bi->biGroupChildren[i];

		if  ( x < add->addDocRect.drX0+
				    child->biCellRightBoundaryPixels	||
		      i == bi->biGroupChildCount- 1			)
		    {
		    para= tedFindItem( child, add, x, y );

		    if  ( para )
			{ return para;	}

		    if  ( y > child->biY1		&&
			  child->biGroupChildCount > 0	)
			{
			found= child->biGroupChildren[
						child->biGroupChildCount-1];
			return found;
			}
		    }
		}
	    LDEB(y); return (BufferItem *)0;
	    break;

	case DOClevPARA:
	    return bi;
	default:
	    LDEB(bi->biLevel); return (BufferItem *)0;
	}

    LLDEB(y,bi->biLevel); return (BufferItem *)0;
    }

static int tedFindLine(	const BufferItem *	bi,
			int			x,
			int			y	)
    {
    int			i;
    TextLine *		tl;

    if  ( y < bi->biY0 )
	{ LLDEB(y,bi->biY0); return -1;	}

    if  ( y >= bi->biY1 )
	{ /* LLDEB(y,bi->biY1); */ return -1; }

    if  ( bi->biLevel != DOClevPARA )
	{ LLDEB(bi->biLevel,DOClevPARA); return -1;	}

    tl= bi->biParaLines;
    if  ( bi->biParaLineCount > 0	&&
	  y < tl->tlY1			)
	{ return 0;	}

    for ( i= 0; i < bi->biParaLineCount; tl++, i++ )
	{
	if  ( /* tl->tlY0 <= y && */ y < tl->tlY1 )
	    { return i;	}
	}

    tl= bi->biParaLines+ bi->biParaLineCount- 1;
    if  ( bi->biParaLineCount > 0	&&
	  y > tl->tlY0			)
	{ return bi->biParaLineCount- 1; }

    /* LDEB(y); */ return -1;
    }

int tedFindParticule(	TextLine *	tl,
			TextParticule *	tp,
			int		x,
			int		y	)
    {
    int				i;

    tp += tl->tlFirstParticule;

    if  ( x < tp->tpX0 )
	{ return tl->tlFirstParticule;	}

    for ( i= 0; i < tl->tlParticuleCount; tp++, i++ )
	{
	if  ( tp->tpX0 <= x &&  x < tp->tpX0+ tp->tpPixelsWide )
	    { return tl->tlFirstParticule+ i;	}
	}

    return tl->tlFirstParticule+ tl->tlParticuleCount- 1;
    }

/************************************************************************/
/*  Return the X position for an I bar.					*/
/************************************************************************/
int tedCalculateX(	const BufferItem *	bi,
			const TextParticule *	tp,
			const AppDrawingData *	add,
			int			stroff	)
    {
    const AppPhysicalFontList *	apfl= &(add->addPhysicalFontList);
    AppPhysicalFont *		apf;
    const unsigned char *	s;
    int				len;
    int				x;

    switch( tp->tpKind )
	{
	case DOCkindOBJECT:
	case DOCkindTAB:
	    len= stroff- tp->tpStroff;

	    if  ( len == 0 )
		{ x= tp->tpX0;	}
	    else{
		if  ( len == 1 )
		    { x= tp->tpX0+ tp->tpPixelsWide;	}
		else{
		    LLDEB(stroff,tp->tpStroff);
		    docListParticule( 0, "???", tp-bi->biParaParticules,
								    bi, tp );
		    docListItem( 0, bi );
		    x= tp->tpX0;
		    }
		}
	    return x;
	case DOCkindTEXT:
	    if  ( tp->tpPhysicalFont < 0 )
		{ LDEB(tp->tpPhysicalFont); return tp->tpX0;	}

	    s= bi->biParaString+ tp->tpStroff;
	    apf= apfl->apflFonts+ tp->tpPhysicalFont;
	    len= stroff- tp->tpStroff;

	    if  ( len == tp->tpStrlen )
		{ x= tp->tpX0+ tp->tpPixelsWide;		}
	    else{ x= tp->tpX0+ tedTextWidth( apf, s, len );	}
	    return x;
	case DOCkindFIELDSTART:
	case DOCkindFIELDEND:
	case DOCkindBKMKSTART:
	case DOCkindBKMKEND:
	case DOCkindXE:
	case DOCkindTC:
	    x= tp->tpX0;
	    return x;
	default:
	    LDEB(tp->tpKind);
	    x= tp->tpX0;
	    return x;
	}
    }

/************************************************************************/
/*									*/
/*  Return the offset in the paragraph string of a mouse event		*/
/*									*/
/*  1)  For Empty particules report the position of the beginning of	*/
/*	the particule.							*/
/*	For clicks in the left margin, report the position of the	*/
/*	particule.							*/
/*  2)  Note the <=, also consider the position after the last		*/
/*	character.							*/
/*									*/
/************************************************************************/
int tedFindStringOffset(	const TextParticule *	tp,
				const unsigned char *	paraString,
				const AppDrawingData *	add,
				int *			pBarX,
				int			x,
				int			y	)
    {
    const AppPhysicalFontList *	apfl= &(add->addPhysicalFontList);
    int				len;
    int				stroff= tp->tpStroff+ 1;
    const unsigned char *	s= paraString+ tp->tpStroff;
    AppPhysicalFont *		apf;
    int				before= tp->tpX0;
    int				past= tp->tpX0;

    /*  1  */
    if  ( x < before || tp->tpStrlen == 0 )
	{ *pBarX= tp->tpX0; return tp->tpStroff; }

    if  ( tp->tpPhysicalFont < 0 )
	{ LDEB(tp->tpPhysicalFont); return tp->tpStroff;	}

    apf= apfl->apflFonts+ tp->tpPhysicalFont;

    /*  2  */
    for ( len= 1; len <= tp->tpStrlen; stroff++, len++ )
	{
	int	between;

	if  ( len == tp->tpStrlen )
	    { past= tp->tpX0+ tp->tpPixelsWide;			}
	else{ past= tp->tpX0+ tedTextWidth( apf, s, len );	}

	between= ( before+ past )/ 2;

	if  ( before <= x && x < between )
	    { *pBarX= before; return stroff- 1;	}
	if  ( between <= x && x < past )
	    { *pBarX= past; return stroff;		}

	before= past;
	}

    *pBarX= tp->tpX0+ tp->tpPixelsWide; return tp->tpStroff+ tp->tpStrlen;
    }

/************************************************************************/
/*									*/
/*  Translate a Window coordinate to a position in a text buffer.	*/
/*									*/
/************************************************************************/

int tedFindPosition(	BufferDocument *	bd,
			const AppDrawingData *	add,
			int			x,
			int			y,
			TextParticule **	pTp,
			TextLine **		pTl,
			BufferPosition *	bp	)
    {
    const BufferItem *	bi;

    int			part;
    int			stroff;
    int			barX;
    int			line;

    bi= tedFindItem( &(bd->bdItem), add, x, y );
    if  ( ! bi )
	{
	if  ( y < bd->bdItem.biY0 )
	    {
	    if  ( docFirstPosition( &(bd->bdItem), bp ) )
		{ LLDEB(bd->bdItem.biY0,y); return -1;	}

	    bi= bp->bpBi;
	    *pTl= bi->biParaLines+ bp->bpLine;
	    *pTp= bi->biParaParticules+ bp->bpParticule;

	    return 0;
	    }

	if  ( y > bd->bdItem.biY1 )
	    {
	    if  ( docLastPosition( &(bd->bdItem), bp ) )
		{ LLDEB(bd->bdItem.biY0,y); return -1;	}

	    bi= bp->bpBi;
	    *pTl= bi->biParaLines+ bp->bpLine;
	    *pTp= bi->biParaParticules+ bp->bpParticule;

	    return 0;
	    }

	LDEB(y);
	LLDEB(bd->bdItem.biY0,bd->bdItem.biY1);
	return -1;
	}

    if  ( bi->biLevel != DOClevPARA )
	{ LLDEB(bi->biLevel,DOClevPARA); return -1;	}

    if  ( bi->biParaLineCount > 0	&&
	  ! bi->biParaInTable		&&
	  bi->biParaStartsOnNewPage	&&
	  y < bi->biParaLines[0].tlY0	)
	{
	line= 0;
	part= 0;
	stroff= 0;
	barX= bi->biParaParticules[0].tpX0;
	}
    else{
	line= tedFindLine( bi, x, y );
	if  ( line < 0 )
	    {
	    if  ( y >= bi->biY1			&&
		  bi->biParaLineCount > 0	&&
		  bi->biParaParticuleCount > 0	)
		{
		line= bi->biParaLineCount- 1;
		part= bi->biParaParticuleCount- 1;
		stroff= bi->biParaStrlen;
		barX= bi->biParaParticules[part].tpX0+
				    bi->biParaParticules[part].tpPixelsWide;
		}
	    else{
		line= 0; part= 0; stroff= 0;
		if  ( bi->biParaParticuleCount == 0 )
		    { LLDEB(line,bi->biParaParticuleCount); return -1;	}
		barX= bi->biParaParticules[part].tpX0;
		}
	    }
	else{
	    part= tedFindParticule( bi->biParaLines+ line,
						bi->biParaParticules, x, y );
	    if  ( part < 0 )
		{ LLDEB(x,part); return -1;			}

	    stroff= tedFindStringOffset( bi->biParaParticules+ part,
					bi->biParaString, add, &barX, x, y );
	    if  ( stroff < 0 )
		{ LDEB(stroff); return -1;	}
	    }
	}

#   if 0
    {
    TextParticule *	tp= bi->biParaParticules+ part;

    DEBFUN( "(%3d,%3d): %.*s|%.*s\n", x, y,
			stroff- tp->tpStroff, bi->biParaString+ tp->tpStroff,
			tp->tpStroff+ tp->tpStrlen- stroff,
			bi->biParaString+ stroff );
    }
#   endif

    *pTl= bi->biParaLines+ line;
    *pTp= bi->biParaParticules+ part;

    bp->bpBi= (BufferItem *)bi;	/*  Discards const */
    bp->bpLine= line;
    bp->bpParticule= part;
    bp->bpStroff= stroff;

    bp->bpX= barX;
    bp->bpY0= bi->biParaLines[line].tlY0;
    bp->bpY1= bi->biParaLines[line].tlY1;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Calculate the smallest rectangle that contains the selection.	*/
/*									*/
/*  1)  Same paragraph?							*/
/*  1b) For a single position, widen to contain the whole I-Bar.	*/
/*  2)  Same table row?							*/
/*  3)  Expose the whole cell.						*/
/*									*/
/************************************************************************/

#   define	IW	5

void tedSelectionRectangle(	DocumentRectangle *	drClip,
				const AppDrawingData *	add,
				const BufferSelection *	bs )
    {
    FormattingFrame	ff;
    DocumentRectangle	dr;

    drClip->drY0= bs->bsBegin.bpY0;
    drClip->drY1= bs->bsEnd.bpY1;

    /*  1  */
    if  ( bs->bsEnd.bpBi == bs->bsBegin.bpBi )
	{
	if  ( bs->bsEnd.bpLine == bs->bsBegin.bpLine )
	    {
	    /*  1b  */
	    if  ( bs->bsEnd.bpStroff == bs->bsBegin.bpStroff )
		{
		drClip->drX0= bs->bsBegin.bpX- IW;
		drClip->drX1= bs->bsEnd.bpX+ IW;
		}
	    else{
		drClip->drX0= bs->bsBegin.bpX;
		drClip->drX1= bs->bsEnd.bpX;
		}

	    return;
	    }

	if  ( bs->bsBegin.bpBi->biParaInTable )
	    {
	    tedParagraphFrame( &ff, add, -1, bs->bsBegin.bpBi );

	    drClip->drX0= ff.ffX0TextLines;
	    drClip->drX1= ff.ffX1TextLines;

	    return;
	    }

	drClip->drX0= add->addBackRect.drX0;
	drClip->drX1= add->addBackRect.drX1;

	return;
	}

    /*  2  */
    if  ( ! bs->bsBegin.bpBi->biParaInTable		||
	  ! bs->bsEnd.bpBi->biParaInTable		||
	  ! docSelectionInsideRow( bs )			)
	{
	drClip->drX0= add->addBackRect.drX0;
	drClip->drX1= add->addBackRect.drX1;
	}

    /*  3  */
    if  ( bs->bsBegin.bpBi->biParaInTable )
	{
	tedParagraphFrame( &ff, add, -1, bs->bsBegin.bpBi );

	dr.drX0= ff.ffX0TextLines;
	dr.drX1= ff.ffX1TextLines;

	if  ( docSelectionInsideCell( bs ) )
	    {
	    dr.drY0= bs->bsBegin.bpBi->biY0;
	    dr.drY1= bs->bsBegin.bpBi->biY1;
	    }
	else{
	    dr.drY0= bs->bsBegin.bpBi->biParent->biParent->biY0;
	    dr.drY1= bs->bsBegin.bpBi->biParent->biParent->biY1;
	    }

	docUnionRectangle( drClip, drClip, &dr );
	}

    /*  3  */
    if  ( bs->bsEnd.bpBi->biParaInTable )
	{
	tedParagraphFrame( &ff, add, -1, bs->bsEnd.bpBi );

	dr.drX0= ff.ffX0TextLines;
	dr.drX1= ff.ffX1TextLines;

	if  ( docSelectionInsideCell( bs ) )
	    {
	    dr.drY0= bs->bsEnd.bpBi->biY0;
	    dr.drY1= bs->bsEnd.bpBi->biY1;
	    }
	else{
	    dr.drY0= bs->bsBegin.bpBi->biParent->biParent->biY0;
	    dr.drY1= bs->bsBegin.bpBi->biParent->biParent->biY1;
	    }

	docUnionRectangle( drClip, drClip, &dr );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  A selection was found, do bookkeeping.				*/
/*									*/
/************************************************************************/

void tedCalculateSelectedLines(	TedDocument *	td	)
    {
    BufferItem *	bi;
    TextLine *		tl;
    int			line;

    BufferPosition *	bp;

    bp= &(td->tdSelection.bsBegin);
    bi= bp->bpBi;
    tl= bi->biParaLines; line= 0;
    while( tl->tlFirstParticule+ tl->tlParticuleCount < bp->bpParticule )
	{ tl++; line++;	}

    if  ( line < bi->biParaLineCount- 1					&&
	  tl->tlFirstParticule+ tl->tlParticuleCount == bp->bpParticule	)
	{ tl++; line++;	}

    bp->bpLine= line;

    bp= &(td->tdSelection.bsEnd);
    bi= bp->bpBi;
    tl= bi->biParaLines; line= 0;
    while( tl->tlFirstParticule+ tl->tlParticuleCount < bp->bpParticule )
	{ tl++; line++;	}

    if  ( line < bi->biParaLineCount- 1					&&
	  tl->tlFirstParticule+ tl->tlParticuleCount == bp->bpParticule	)
	{ tl++; line++;	}

    bp->bpLine= line;
    }

void tedPositionCoordinates(	BufferPosition *	bp,
				const AppDrawingData *	add )
    {
    BufferItem *	bi;
    TextLine *		tl;
    TextParticule *	tp;

    bi= bp->bpBi;
    tl= bi->biParaLines+ bp->bpLine;
    tp= bi->biParaParticules+ bp->bpParticule;

    bp->bpX= tedCalculateX( bi, tp, add, bp->bpStroff );
    bp->bpY0= tl->tlY0;
    bp->bpY1= tl->tlY1;

#   if 0
    Gives silly high I Bar.
    if  ( bp->bpLine == bi->biParaLineCount- 1 )
	{ bp->bpY1 += bi->biParaSpaceAfterPixels; }
#   endif

    return;
    }

void tedSelectionCoordinates(	BufferSelection *	bs,
				const AppDrawingData *	add )
    {
    tedPositionCoordinates( &(bs->bsBegin), add );
    tedPositionCoordinates( &(bs->bsEnd), add );
    }

/************************************************************************/
/*									*/
/*  Use the new ruler settings, or other geometry properties of the	*/
/*  paragraph.								*/
/*									*/
/************************************************************************/

void tedApplyItemFormat(	BufferItem *			bi,
				BufferDocument *		bd,
				AppDrawingData *		add,
				DocumentRectangle *		drChanged )
    {
    int			y= bi->biY0;

    DocumentRectangle	drLocal;

    drLocal.drX0= add->addBackRect.drX0;
    drLocal.drX1= add->addBackRect.drX1;
    drLocal.drY0= bi->biY0;
    drLocal.drY1= bi->biY1;

    if  ( tedLayoutItem( bi, bd, add, y, &y, &drLocal.drY1 ) )
	{ LDEB(y); return;	}

    docUnionRectangle( drChanged, drChanged, &drLocal );

    drLocal.drX0= add->addBackRect.drX0;
    drLocal.drX1= add->addBackRect.drX1;
    drLocal.drY0= bi->biY0;
    drLocal.drY1= bi->biY1;

    docUnionRectangle( drChanged, drChanged, &drLocal );

    return;
    }

