/************************************************************************/
/*									*/
/*  Print Postscript.							*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>

#   include	"tedApp.h"
#   include	<appImage.h>
#   include	<appWinMeta.h>
#   include	<sioMemory.h>
#   include	<sioHex.h>
#   include	"docPs.h"

#   include	<debugon.h>

#   define	ISO_SUFFIX	"-ISO1"

/************************************************************************/
/*									*/
/*  Forward declaration.						*/
/*									*/
/************************************************************************/

static int psPrintItem(	BufferItem *		bi,
			DocumentFontList *	dfl,
			const BufferDocument *	bd,
			int			y0,
			int *			pY,
			PrintingState *		ps );

/************************************************************************/
/*									*/
/*  Printing of borders.						*/
/*									*/
/************************************************************************/

static void psPrintHorizontalBorder(	const BorderProperties *	bp,
					FILE *				f,
					int				x0,
					int				x1,
					int				y )
    {
    if  ( ! bp->bpIsSet )
	{ return;	}

    fprintf( f, "%d %d %d %d rectfill\n", x0, y, x1- x0, bp->bpWidthTwips );
    return;
    }

static void psPrintVerticalBorder(	const BorderProperties *	bp,
					FILE *				f,
					int				x,
					int				y0,
					int				y1 )
    {
    if  ( ! bp->bpIsSet )
	{ return;	}

    fprintf( f, "%d %d %d %d rectfill\n", x, y0, bp->bpWidthTwips, y1- y0 );
    return;
    }

/************************************************************************/
/*									*/
/*  Print a series of particules with the same attributes.		*/
/*									*/
/************************************************************************/
static int psPrintMetafile(	FILE *			f,
				const ObjectData *	od,
				DocumentFontList *	dfl,
				const char *		afmDirectory,
				int			x0,
				int			baseline,
				int			scaleX,
				int			scaleY,
				int			xExt,
				int			yExt,
				int			twipsWide,
				int			twipsHigh )
    {
    SimpleInputStream *	sisMem;
    SimpleInputStream *	sisMeta;

    MemoryBuffer	mb;

    int			y0;

    mb.mbCapacity= od->odSize;
    mb.mbBytes= od->odBytes;

    sisMem= sioInMemoryOpen( &mb );
    if  ( ! sisMem )
	{ XDEB(sisMem); return -1;	}

    sisMeta= sioInHexOpen( sisMem );
    if  ( ! sisMeta )
	{ XDEB(sisMem); return -1;	}

    y0= baseline- ( ( scaleY/100.0 )* twipsHigh );

    fprintf( f, "gsave %d %d translate\n", x0, y0 );

    if  ( scaleX != 100 || scaleY != 100 )
	{ fprintf( f, "%f %f scale\n", scaleX/100.0, scaleY/100.0 ); }

    if  ( appMetaPlayFilePs( f, sisMeta, dfl,
			    afmDirectory, xExt, yExt, twipsWide, twipsHigh ) )
	{ LDEB(1);	}

    fprintf( f, "grestore\n" );

    sioInClose( sisMeta );
    sioInClose( sisMem );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Write links to be picked up by the distiller.			*/
/*									*/
/************************************************************************/

static void psUriLinkDestination(	PrintingState *	ps )
    {
    appPsPrintString( ps->psFile,
	(const unsigned char *)ps->psLinkFile, ps->psLinkFileSize );

    if  ( ps->psLinkMarkSize > 0 )
	{
	putc( '#', ps->psFile );
	appPsPrintString( ps->psFile,
			    (const unsigned char *)ps->psLinkMark,
			    ps->psLinkMarkSize );
	}
    }

static void psWebLinkDestination(	PrintingState *	ps )
    {
    fprintf( ps->psFile, "  /Action << /Subtype /URI /URI (" );

    psUriLinkDestination( ps );

    fprintf( ps->psFile, ") >>\n" );

    return;
    }

static void psFileLinkDestination(	PrintingState *		ps )
    {
    const unsigned char *	file;
    int				size;

    file= (const unsigned char *)ps->psLinkFile;
    size= ps->psLinkFileSize;

    if  ( size > 5 && ! strncmp( (const char *)file, "file:", 5 ) )
	{ file += 5; size -= 5; }
    else{
	while( size > 0 && isalpha( *file ) )
	    { file++; size--;	}

	if  ( size > 0 && *file == ':' )
	    { psWebLinkDestination( ps ); return; }

	file= (const unsigned char *)ps->psLinkFile;
	size= ps->psLinkFileSize;
	}

    fprintf( ps->psFile, "  /Action /Launch /File (" );

    appPsPrintString( ps->psFile, file, size );

    fprintf( ps->psFile, ")\n" );

    if  ( ps->psLinkMarkSize )
	{
	fprintf( ps->psFile, "  /URI (" );
	psUriLinkDestination( ps );
	fprintf( ps->psFile, ")\n" );
	}

    return;
    }

static void psFlushLink(	PrintingState *		ps,
				const ParticuleData *	pd,
				int			lineTop,
				int			lineBottom )
    {
    if  ( ps->psLinkParticulesDone > 0 )
	{
	fprintf( ps->psFile, "[ /Rect [ %d %d %d %d ]\n",
				    ps->psLinkRectLeft, lineBottom, 
				    pd->pdX0+ pd->pdVisibleWidth,
				    lineTop );

	fprintf( ps->psFile, "  /Border [ 0 0 0 ]\n" );

	if  ( ps->psLinkFileSize == 0 )
	    {
	    fprintf( ps->psFile, "  /Action /Goto\n" );
	    fprintf( ps->psFile, "  /Dest /%.*s\n",
					ps->psLinkMarkSize, ps->psLinkMark );
	    }
	else{
	    psFileLinkDestination( ps );
	    }


	fprintf( ps->psFile, "  /Subtype /Link\n" );
	fprintf( ps->psFile, "/LNK pdfmark\n" );

	ps->psLinkParticulesDone= 0;
	ps->psLinkRectLeft= pd->pdX0;
	}

    return;
    }

static int psPrintParticules(	PrintingState *		ps,
				const BufferItem *	bi,
				DocumentFontList *	dfl,
				const char *		afmDirectory,
				TextParticule *		tp,
				ParticuleData *		pd,
				int			count,
				int			baseLine,
				int			lineTop,
				int			lineBottom )
    {
    int				drawn;
    int				i;
    int				len;
    int				high;
    InsertedObject *		io;

    DocumentField *		df;
    int				fontSizeTwips;
    int				capHeight;
    int				y;

    /*  1  */
    switch( tp->tpKind )
	{
	case DOCkindTAB:
	    drawn= 1;

	    if  ( pd->pdTabNumber >= 0			&&
		  pd->pdTabNumber < bi->biParaTabCount	)
		{
		TabStop *	ts= bi->biParaTabStops+ pd->pdTabNumber;

		switch( ts->tsLeader )
		    {
		    case DOCtlNONE:
			break;

		    case DOCtlDOTS:
			fprintf( ps->psFile,
				"gsave 10 setlinewidth [10 30] 0 setdash " );
			fprintf( ps->psFile, "%d %d moveto ",
							pd->pdX0, baseLine );
			fprintf( ps->psFile, "%d 0 rlineto stroke grestore\n",
							pd->pdWidth );
			break;

		    case DOCtlHYPH:
		    case DOCtlUNDERLINE:
		    case DOCtlTHICK:
		    case DOCtlEQUAL:
		    default:
			LDEB(ts->tsLeader);
		    }
		}

	    return drawn;

	case DOCkindTEXT:
	    break;

	case DOCkindBKMKSTART:
	    df= bi->biParaFieldList.dflFields+ tp->tpObjectNumber;
	    fprintf( ps->psFile, "[ /Dest" );
	    fprintf( ps->psFile, "  /%.*s\n", df->dfInstructions.odSize,
						df->dfInstructions.odBytes );

	    fprintf( ps->psFile, "  /View [ /XYZ null %d null ]\n",
				( ps->psPrinterHighTwips- lineTop )/ 20 );

	    fprintf( ps->psFile, "/DEST pdfmark\n" );
	    drawn= 1;
	    return drawn;

	case DOCkindFIELDSTART:
	    df= bi->biParaFieldList.dflFields+ tp->tpObjectNumber;

	    if  ( df->dfKind == DOCfkHYPERLINK )
		{
		if  ( docGetHyperlink( df,
				&(ps->psLinkFile), &(ps->psLinkFileSize),
				&(ps->psLinkMark), &(ps->psLinkMarkSize) ) )
		    { LDEB(1); return -1;	}

		ps->psInLink= 1;
		ps->psLinkParticulesDone= 0;
		ps->psLinkRectLeft= pd->pdX0;

		appPsSetLinkColor( ps );

		drawn= 1;
		return drawn;
		}

	    drawn= 1;
	    return drawn;

	case DOCkindFIELDEND:
	    if  ( ps->psInLink )
		{
		psFlushLink( ps, pd, lineTop, lineBottom );
		fprintf( ps->psFile, "0 setgray\n" );
		ps->psInLink= 0;
		}
	    drawn= 1;
	    return drawn;

	case DOCkindBKMKEND:
	case DOCkindXE:
	case DOCkindTC:
	    drawn= 1;
	    return drawn;

	case DOCkindOBJECT:
	    io= bi->biParaObjects+ tp->tpObjectNumber;

	    switch( io->ioKind )
		{
		case DOCokPICTWMETAFILE:
		    if  ( psPrintMetafile( ps->psFile, &io->ioObjectData, dfl,
				    afmDirectory, pd->pdX0, baseLine,
				    io->ioScaleX, io->ioScaleY,
				    io->ioXExtent, io->ioYExtent,
				    io->ioTwipsWide, io->ioTwipsHigh ) )
			{ LDEB(1); break;	}

		    ps->psLinkParticulesDone++;
		    return 1;

		case DOCokPICTJPEGBLIP:
		case DOCokPICTPNGBLIP:
		    if  ( io->ioPrivate )
			{
			AppBitmapImage *	abi;
			BitmapDescription *	bd;

			abi= (AppBitmapImage *)io->ioPrivate;
			bd= &abi->abiBitmap;

			if  ( bmPsPrintBitmap( ps->psFile, 1,
				    ( 20.0* io->ioScaleX )/ 100.0,
				    ( -20.0* io->ioScaleY )/ 100.0,
				    pd->pdX0, baseLine, 0, 0,
				    bd->bdPixelsWide, bd->bdPixelsHigh,
				    bd, abi->abiBuffer ) )
			    { LDEB(1); return -1; }

			ps->psLinkParticulesDone++;
			return 1;
			}
		    break;

		case DOCokOLEOBJECT:
		    if  ( io->ioResultKind == DOCokPICTWMETAFILE )
			{
			if  ( psPrintMetafile( ps->psFile,
				    &io->ioResultData, dfl,
				    afmDirectory, pd->pdX0, baseLine,
				    io->ioScaleX, io->ioScaleY,
				    io->ioXExtent, io->ioYExtent,
				    io->ioTwipsWide, io->ioTwipsHigh ) )
			    { LDEB(1); break;	}

			ps->psLinkParticulesDone++;
			return 1;
			}
		    break;
		default:
		    LDEB(io->ioKind); return 0;
		}

	    high= ( io->ioScaleY* io->ioTwipsHigh )/ 100;
	    fprintf( ps->psFile, "%d %d moveto ",
				    pd->pdX0, baseLine- high );
	    fprintf( ps->psFile, "%d %d lineto ",
				    pd->pdX0+ pd->pdWidth, baseLine- high );
	    fprintf( ps->psFile, "%d %d lineto ",
				    pd->pdX0+ pd->pdWidth, baseLine );
	    fprintf( ps->psFile, "%d %d lineto ", pd->pdX0, baseLine );
	    fprintf( ps->psFile, "closepath stroke\n" );

	    ps->psLinkParticulesDone++;
	    return 1;
	default:
	    LDEB(tp->tpKind); return -1;
	}

    for ( drawn= 1; drawn < count; drawn++ )
	{
	if  ( tp[drawn].tpPhysicalFont != tp->tpPhysicalFont	||
	      tp[drawn].tpKind != DOCkindTEXT			)
	    { break;	}

	if  ( bi->biParaAlignment == DOCiaJUSTIFIED )
	    { break;	}
	}

    len= tp[drawn-1].tpStroff+ tp[drawn-1].tpStrlen- tp->tpStroff;

    if  ( len == 0 )
	{ return drawn;	}

    putc( '(', ps->psFile );
    appPsPrintString( ps->psFile, bi->biParaString+ tp->tpStroff, len );
    putc( ')', ps->psFile );

    fontSizeTwips= 10* tp->tpTextAttribute.taFontSizeHalfPoints;
    capHeight= ( fontSizeTwips* pd->pdAfi->afiCapHeight+ 500 )/ 1000;
    if  ( capHeight == 0 )
	{ capHeight= ( fontSizeTwips* pd->pdAfi->afiAscender+ 500 )/ 1000; }
    y= baseLine;

    if  ( tp->tpTextAttribute.taSuperSub == DOCfontSUPERSCRIPT )
	{ y -= ( 4* capHeight )/ 10; }

    if  ( tp->tpTextAttribute.taSuperSub == DOCfontSUBSCRIPT )
	{ y += ( 4* capHeight )/ 10; }

    fprintf( ps->psFile, " %d %d mvs\n", pd->pdX0, y );

    i= 0; while( i < drawn )
	{
	int	x0;
	int	x1;
	int	y0;
	int	h;

	if  ( ! tp[i].tpTextAttribute.taIsUnderlined )
	    { i++; continue;	}

	x1= x0= pd[i].pdX0;
	y0= baseLine- ( fontSizeTwips*
			pd[i].pdAfi->afiUnderlinePosition+ 500 )/ 1000;
	h= ( fontSizeTwips*
			pd[i].pdAfi->afiUnderlineThickness+ 500 )/ 1000;

	while( i < drawn && tp[i].tpTextAttribute.taIsUnderlined )
	    { x1= pd[i].pdX0+ pd[i].pdWidth; i++; }

	fprintf( ps->psFile, "%d %d %d %d rectfill\n", x0, y0, x1- x0, h );
	}

    ps->psLinkParticulesDone += drawn;

    return drawn;
    }

/************************************************************************/
/*									*/
/*  Print one line of output.						*/
/*									*/
/************************************************************************/
static int psPrintTextLine(	PrintingState *		ps,
				DocumentFontList *	dfl,
				const char *		afmDirectory,
				const BufferItem *	bi,
				TextParticule *		tp,
				ParticuleData *		pd,
				int			baseLine,
				int			lineTop,
				int			lineBottom,
				int			particuleCount )
    {
    int				done= 0;

    while( done < particuleCount )
	{
	int		drawn;

	if  ( tp->tpKind == DOCkindTEXT		||
	      tp->tpKind == DOCkindTAB		)
	    {
	    if  ( tp->tpPhysicalFont != ps->psCurrentPhysicalFont )
		{
		appPsSetFont( ps->psFile, tp->tpTextAttribute );
		ps->psCurrentPhysicalFont= tp->tpPhysicalFont;
		}
	    }
	else{ ps->psCurrentPhysicalFont= -1;	}

	drawn= psPrintParticules( ps, bi, dfl, afmDirectory, tp, pd,
			particuleCount- done, baseLine, lineTop, lineBottom );

	if  ( drawn < 1 )
	    { LDEB(drawn); return -1;	}

	done += drawn; tp += drawn; pd += drawn;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Layout and print successive lines of a paragraph.			*/
/*									*/
/************************************************************************/

static int psPrintLines(	PrintingState *		ps,
				const BufferItem *	bi,
				int			paraAscent,
				int			paraDescent,
				int			part,
				DocumentFontList *	dfl,
				AppPhysicalFontList *	apfl,
				const BufferDocument *	bd,
				const FormattingFrame *	ff,
				int			y,
				int *			pY )
    {
    int				done= 0;

    TextParticule *		tp= bi->biParaParticules+ part;
    const ParagraphProperties *	pp= &(bi->biParaProperties);
    const DocumentGeometry *	dg= &(bd->bdGeometry);

    ParticuleData *		pd;

    int				x_x0= ff->ffX0TextLines;

    if  ( pp->ppFirstIndentTwips < 0 )
	{ x_x0 += pp->ppFirstIndentTwips;	}

    if  ( docPsClaimParticuleData( bi, &pd ) )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    while( part < bi->biParaParticuleCount )
	{
	int		accepted;
	int		height;
	int		atTop= y <= dg->dgTopMarginTwips;
	int		x0;

	LineBox		lb;

	docPsInitLineBox( &lb );

	if  ( part == 0 )
	    { x0= ff->ffX0FirstLine;	}
	else{ x0= ff->ffX0TextLines;	}

	if  ( ps->psInLink )
	    { ps->psLinkRectLeft= x0;	}

	accepted= docPsLineBox( &lb, bi, part, 
		    atTop, paraAscent, paraDescent, dg,
		    bd->bdProperties.dpTabIntervalTwips, &bd->bdFontList,
		    apfl, tp, pd, x0, ff );

	if  ( accepted < 1 )
	    { LDEB(accepted); return -1;	}

	height= lb.lbSpaceAboveLine+ lb.lbLineHeight+ lb.lbSpaceBelowLine+
			lb.lbTableMarginAboveLine+ lb.lbTableMarginBelowLine;

	if  ( y+ height > ff->ffY1		&&
	      height < BETWEEN_MARGINS( dg )	)
	    { y= ff->ffY1+ 1; break;	}

	if  ( part == 0 )
	    {
	    y += lb.lbTableMarginAboveLine;
	    y += bi->biParaSpaceBeforeTwips;

	    if  ( lb.lbBorderAboveLine )
		{
		psPrintHorizontalBorder( lb.lbBorderAboveLine, ps->psFile,
						    x0, ff->ffX1TextLines, y );

		y += lb.lbBorderAboveLine->bpWidthTwips;
		y += lb.lbBorderAboveLine->bpSpacingTwips;
		}
	    }

	psPrintTextLine( ps, dfl, apfl->apflAfmDirectory,
				bi, bi->biParaParticules+ part, pd,
				y+ lb.lbLineAscent, y, y+ lb.lbLineHeight,
				accepted );

	if  ( ps->psInLink )
	    { psFlushLink( ps, pd+ accepted- 1, y, y+ lb.lbLineHeight ); }

	y += lb.lbLineDistance;

	if  ( part+ accepted == bi->biParaParticuleCount )
	    {
	    if  ( lb.lbBorderBelowLine )
		{
		psPrintHorizontalBorder( lb.lbBorderBelowLine, ps->psFile,
						    x0, ff->ffX1TextLines, y );

		y += lb.lbBorderBelowLine->bpSpacingTwips;
		y += lb.lbBorderBelowLine->bpWidthTwips;
		}

	    y += bi->biParaSpaceAfterTwips;
	    y += lb.lbTableMarginBelowLine;
	    }

	part += accepted; tp += accepted; done += accepted;
	}

    *pY= y; return done;
    }

/************************************************************************/
/*									*/
/*  Skip to the next page.						*/
/*									*/
/************************************************************************/

static void docPsPrintNextPage( const DocumentGeometry *	dg,
				BufferItem *			rowBi,
				DocumentFontList *		dfl,
				const BufferDocument *		bd,
				int *				pY,
				PrintingState *			ps )
    {
    int		y= dg->dgTopMarginTwips;

    appPsFinishPage( ps->psFile, ps->psCurrentPage++ );

    ps->psCurrentPhysicalFont= -1;

    appPsStartPage( ps, dg );

    if  ( rowBi )
	{
	BufferItem *	sectBi= rowBi->biParent;

	if  ( sectBi->biLevel != DOClevSECT )
	    { LLDEB(sectBi->biLevel,DOClevSECT);	}

	if  ( sectBi->biSectHeader )
	    {
	    if  ( docSubstitutePageNumber( sectBi->biSectHeader,
							ps->psCurrentPage ) )
		{ XDEB(sectBi->biSectHeader); }

	    if  ( psPrintItem( sectBi->biSectHeader, dfl, bd, y, &y, ps ) )
		{ XDEB(sectBi->biSectHeader); }
	    }
	}


    *pY= y; return;
    }

/************************************************************************/
/*									*/
/*  Print a node in the BufferItem hierarchy.				*/
/*									*/
/*  1)  Force a page break by moving the current position off the page.	*/
/*									*/
/************************************************************************/

static int psPrintParaItem(	BufferItem *			bi,
				DocumentFontList *		dfl,
				const BufferDocument *		bd,
				int				y0,
				int *				pY,
				PrintingState *			ps )
    {
    int				y= y0;
    FormattingFrame		ff;
    int				part;

    int				paraAscent= 0;
    int				paraDescent= 0;

    BufferItem *		rowBi= bi->biParent->biParent;

    const DocumentGeometry *	dg= &(bd->bdGeometry);
    AppPhysicalFontList *	apfl;

    apfl= (AppPhysicalFontList *)ps->psPhysicalFontList;

    /*  1  */
    if  ( ! bi->biParaInTable && bi->biParaStartsOnNewPage )
	{ y= bd->bdGeometry.dgPaperHighTwips;	}

    y += bi->biParaSpaceBeforeTwips;

    docParagraphFrame( &ff, dg, -1, bi );
    if  ( docPsParagraphLineExtents( &paraAscent, &paraDescent, apfl, bi ) )
	{ LDEB(1); return -1;	}

    part= 0;
    while( part < bi->biParaParticuleCount )
	{
	int	accepted;

	accepted= psPrintLines( ps, bi, paraAscent, paraDescent,
					part, dfl, apfl, bd, &ff, y, &y );

	if  ( accepted < 0 )
	    { LDEB(accepted); return -1;	}

	part += accepted;

	if  ( part < bi->biParaParticuleCount )
	    { docPsPrintNextPage( dg, rowBi, dfl, bd, &y, ps ); }
	}

    *pY= y; return 0;
    }

typedef struct CellPrintingState
    {
    int			cpsPara;
    int			cpsPart;
    int			cpsY;
    int			cpsX0;
    int			cpsX1;
    int			cpsX11;		/*  Includes right border	*/
    int			cpsParaAscent;
    int			cpsParaDescent;
    FormattingFrame	cpsFormattingFrame;
    } CellPrintingState;

static int psAdvanceCell(	PrintingState *		ps,
				BufferItem *		cellBi,
				DocumentFontList *	dfl,
				const BufferDocument *	bd,
				int			bottom,
				CellPrintingState *	cps )
				
    {
    const DocumentGeometry *	dg= &(bd->bdGeometry);
    AppPhysicalFontList *	apfl;

    apfl= (AppPhysicalFontList *)ps->psPhysicalFontList;

    while( cps->cpsPara < cellBi->biGroupChildCount )
	{
	BufferItem *		paraBi= cellBi->biGroupChildren[cps->cpsPara];
	int			accepted;

	if  ( cps->cpsPart == 0 )
	    {
	    docParagraphFrame( &(cps->cpsFormattingFrame),
						    dg, bottom, paraBi );
	    if  ( docPsParagraphLineExtents( &(cps->cpsParaAscent),
				    &(cps->cpsParaDescent), apfl, paraBi ) )
		{ LDEB(1); return -1;	}
	    }

	accepted= psPrintLines( ps, paraBi,
				cps->cpsParaAscent, cps->cpsParaDescent,
				cps->cpsPart, dfl, apfl,
				bd, &(cps->cpsFormattingFrame),
				cps->cpsY, &(cps->cpsY) );
	if  ( accepted < 0 )
	    { LDEB(accepted); return -1;	}

	cps->cpsPart += accepted;

	if  ( cps->cpsPart < paraBi->biParaParticuleCount )
	    { break;	}

	cps->cpsPart= 0; cps->cpsPara++;
	}

    return 0;
    }

static void psPrintCellSides(	int			x0,
				int			x1,
				int			y0,
				int			y1,
				const CellProperties *	cp,
				const RowProperties *	rp,
				FILE *			f )
    {
    psPrintVerticalBorder( &(cp->cpLeftBorder), f, x0, y0, y1 );
    psPrintVerticalBorder( &(cp->cpRightBorder), f, x1, y0, y1 );

    return;
    }

static int psPrintRowItem(	BufferItem *		rowBi,
				DocumentFontList *	dfl,
				const BufferDocument *	bd,
				int			y0,
				int *			pY,
				PrintingState *		ps )
    {
    int				i;
    int				y;

    int				rowHeight;
    int				tooManyCells= 0;

    const DocumentGeometry *	dg= &(bd->bdGeometry);
    RowProperties *		rp= &(rowBi->biRowProperties);
    CellProperties *		cp;

    BufferItem *		cellBi;

    int				toNextPage;

    static CellPrintingState *	colP;
    CellPrintingState *		cps;

    int				bottom= -1;

    int				x0;
    int				x1;

    int				x= dg->dgLeftMarginTwips;

    if  ( rp->rpCellCount < rowBi->biGroupChildCount )
	{
	LLDEB(rp->rpCellCount,rowBi->biGroupChildCount);
	docListItem( 0, rowBi );
	rowBi->biGroupChildCount= rp->rpCellCount;
	tooManyCells= 1;
	}

    cps= (CellPrintingState *)realloc( colP,
				rp->rpCellCount* sizeof(CellPrintingState) );
    if  ( ! cps )
	{ LXDEB(rp->rpCellCount,cps); return -1;	}
    colP= cps;

    y= y0;
    toNextPage= 0;

    if  ( tooManyCells )
	{
	fprintf( ps->psFile,
			"/DeviceRGB setcolorspace 1.0 0.0 0.0 setcolor\n" );
	}

    if  ( rowBi->biRowHeightTwips < 0 )
	{ bottom= y0- rowBi->biRowHeightTwips;	}

    cp= rp->rpCells;
    x1= x0= x+ rowBi->biRowLeftIndentTwips;
    for ( i= 0; i < rowBi->biGroupChildCount; cp++, i++ )
	{
	x1= x+ cp->cpRightBoundaryTwips;

	cps[i].cpsPara= 0;
	cps[i].cpsPart= 0;

	cps[i].cpsY= y0;
	cps[i].cpsX0= x0;
	cps[i].cpsX1= x1;
	cps[i].cpsX11= x1;

	if  ( cp->cpRightBorder.bpIsSet )
	    { cps[i].cpsX11 +=  cp->cpRightBorder.bpWidthTwips; }

	cellBi= rowBi->biGroupChildren[i];

	psPrintHorizontalBorder( &(cp->cpTopBorder), ps->psFile,
					    cps[i].cpsX0, cps[i].cpsX11, y0 );

	if  ( psAdvanceCell( ps, cellBi, dfl, bd, bottom, cps+ i ) )
	    { LDEB(1); return -1;	}

	if  ( cps[i].cpsY > y )
	    { y= cps[i].cpsY;	}

	if  ( cps[i].cpsPara < cellBi->biGroupChildCount )
	    { toNextPage= 1;	}

	x0= x1;
	}

    rowHeight= y- y0;

    cp= rp->rpCells;
    for ( i= 0; i < rowBi->biGroupChildCount; cp++, i++ )
	{
	psPrintCellSides( cps[i].cpsX0, cps[i].cpsX1,
						y0, y, cp, rp, ps->psFile );
	}

    if  ( rowBi->biRowHeightTwips < 0 && rowHeight >= -rowBi->biRowHeightTwips )
	{ toNextPage= 0;	}

    x0= x+ rowBi->biRowLeftIndentTwips;
    while( toNextPage )
	{
	docPsPrintNextPage( dg, rowBi, dfl, bd, &y0, ps );

	y= y0; toNextPage= 0;

	if  ( rowBi->biRowHeightTwips < 0 )
	    { bottom= y0- rowBi->biRowHeightTwips- rowHeight;	}

	for ( i= 0; i < rowBi->biGroupChildCount; i++ )
	    {
	    cps[i].cpsY= y0;
	    cellBi= rowBi->biGroupChildren[i];

	    if  ( psAdvanceCell( ps, cellBi, dfl, bd, bottom, cps+ i ) )
		{ LDEB(1); return -1;	}

	    if  ( cps[i].cpsY > y )
		{ y= cps[i].cpsY;	}

	    if  ( cps[i].cpsPara < cellBi->biGroupChildCount	&&
		  ( bottom < 0 || y < bottom )			)
		{ toNextPage= 1;	}
	    }

	rowHeight += y- y0;

	if  ( ! toNextPage )
	    { break;	}

	cp= rp->rpCells;
	for ( i= 0; i < rowBi->biGroupChildCount; cp++, i++ )
	    {
	    psPrintCellSides( cps[i].cpsX0, cps[i].cpsX1,
						y0, y, cp, rp, ps->psFile );
	    }
	}

    if  ( rowHeight < rowBi->biRowHeightTwips )
	{ y += rowBi->biRowHeightTwips- rowHeight;	}

    if  ( rowHeight < -rowBi->biRowHeightTwips )
	{ y += -rowBi->biRowHeightTwips- rowHeight;	}

    cp= rp->rpCells;
    for ( i= 0; i < rowBi->biGroupChildCount; cp++, i++ )
	{
	psPrintCellSides( cps[i].cpsX0, cps[i].cpsX1,
						y0, y, cp, rp, ps->psFile );

	psPrintHorizontalBorder( &(cp->cpBottomBorder), ps->psFile,
					    cps[i].cpsX0, cps[i].cpsX11, y );
	}

    if  ( tooManyCells )
	{ fprintf( ps->psFile, "0 setgray\n" );	}

    *pY= y; return 0;
    }

static int psPrintItem(	BufferItem *		bi,
			DocumentFontList *	dfl,
			const BufferDocument *	bd,
			int			y0,
			int *			pY,
			PrintingState *		ps )
    {
    int			i;
    int			y;

    y= y0;

    switch( bi->biLevel )
	{
	case DOClevDOC:
	case DOClevSECT:
	case DOClevCELL:
	rowAsGroup:
	    for ( i= 0; i < bi->biGroupChildCount; i++ )
		{
		if  ( psPrintItem( bi->biGroupChildren[i],
						    dfl, bd, y0, &y, ps ) )
		    { LLDEB(i,y); return -1;	}

		y0= y;
		}
	    break;

	case DOClevROW:
	    if  ( ! bi->biRowHasTableParagraphs )
		{ goto rowAsGroup;	}

	    if  ( psPrintRowItem( bi, dfl, bd, y0, &y, ps ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevPARA:
	    if  ( psPrintParaItem( bi, dfl, bd, y0, &y, ps ) )
		{ LDEB(1); return -1;	}

	    break;

	default:
	    LDEB(bi->biLevel); return -1;
	}


    *pY= y; return 0;
    }

/************************************************************************/
/*									*/
/*  Leave a trace in the PDF document information.			*/
/*									*/
/************************************************************************/

static void psSaveInfo(		const char *		tag,
				const unsigned char *	info,
				FILE *			f )
    {
    if  ( info )
	{
	fprintf( f, "  %s (", tag );
	appPsPrintString( f, info, strlen( (const char *)info ) );
	fprintf( f, ")\n" );
	}
    }

/************************************************************************/
/*									*/
/*  Print a document.							*/
/*									*/
/************************************************************************/
int psPrintDocument(	FILE *			f,
			EditDocument *		ed )
    {
    EditApplication *		ea= ed->edApplication;

    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    DocumentGeometry *		dg= &(bd->bdGeometry);
    AppDrawingData *		add= &(ed->edDrawingData);

    int				y= dg->dgTopMarginTwips;

    PostScriptFont *		fontList= (PostScriptFont *)0;
    int				fontCount= 0;
    int				i;

    PrintingState		ps;

    appPsInitPrintingState( &ps );
    ps.psAfmDirectory= ea->eaAfmDirectory;
    ps.psPrinterWideTwips= ea->eaDefaultDocumentGeometry.dgPaperWideTwips;
    ps.psPrinterHighTwips= ea->eaDefaultDocumentGeometry.dgPaperHighTwips;
    ps.psPhysicalFontList= (void *)&(add->addPhysicalFontList);
    ps.psFile= f;

    if  ( docPsPrintGetItemFonts( &(bd->bdItem),
				(&bd->bdFontList), &(add->addPhysicalFontList),
				&fontList, &fontCount ) )
	{ SDEB(ed->edTitle); return -1;	}

    fprintf( f, "%%!PS-Adobe-2.0\n" );

    if  ( ed->edTitle )
	{ fprintf( f, "%%%%Title: %s\n", ed->edTitle ); }

    appPsBoundingBoxComment( f, dg,
			    ps.psPrinterWideTwips, ps.psPrinterHighTwips,
			    "BoundingBox", "Orientation" );

    fprintf( f, "%%%%Creator: %s\n", ea->eaApplicationName );
    fprintf( f, "%%%%Pages: (atend)\n" );
    fprintf( f, "%%%%PageOrder: Ascend\n" );
    if  ( fontCount > 0 )
	{
	fprintf( f, "%%%%DocumentFonts: %s\n",
					    fontList[0].psfAfi->afiFontName );
	for ( i= 1; i < fontCount; i++ )
	    { fprintf( f, "%%%%+ %s\n", fontList[i].psfAfi->afiFontName ); }
	}
    fprintf( f, "%%%%EndComments\n" );

    fprintf( f, "%%%%BeginProlog\n" );
    fprintf( f, "%%%%EndProlog\n" );
    fprintf( f, "%%%%BeginSetup\n" );

    fprintf( f, "\n/mvs { moveto show } def\n" );

    fprintf( f, "\n/pdfmark where\n" );
    fprintf( f, "  {pop} {userdict /pdfmark /cleartomark load put} ifelse\n" );

    if  ( fontCount > 0 )
	{ fprintf( f, "\n" ); }

    for ( i= 0; i < fontCount; i++ )
	{
	AfmFontInfo *	afi= fontList[i].psfAfi;

	if  ( afi->afiMapToIso )
	    {
	    fprintf( f, "/%s findfont dup length dict begin\n",
					    afi->afiFontName );
	    fprintf( f, "  { 1 index /FID ne {def} {pop pop} ifelse} forall" );
	    fprintf( f, "  /Encoding ISOLatin1Encoding def currentdict\n" );
	    fprintf( f, "end\n" );
	    fprintf( f, "/%s%s exch definefont pop\n",
				afi->afiFontName, ISO_SUFFIX );
	    }

	fprintf( f, "/%s { /%s%s exch selectfont } def\n\n",
			    fontList[i].psfFontId,
			    fontList[i].psfAfi->afiFontName,
			    fontList[i].psfAfi->afiMapToIso?ISO_SUFFIX:"" );
	}

    if  ( docHasDocumentInfo( bd ) )
	{
	const unsigned char *	ted;

	ted= (const unsigned char *)"Ted: http://www.nllgg.nl/Ted";

	fprintf( f, "[\n" );

	psSaveInfo( "/Title",		bd->bdTitle, f );
	psSaveInfo( "/Author",		bd->bdAuthor, f );
	psSaveInfo( "/Subject",		bd->bdSubject, f );
	psSaveInfo( "/Keywords",	bd->bdKeywords, f );
	psSaveInfo( "/Creator",		ted, f );
	psSaveInfo( "/Producer",	ted, f );

	fprintf( f, "/DOCINFO pdfmark\n" );
	}

    fprintf( f, "%%%%EndSetup\n" );

    appPsStartPage( &ps, dg );

    psPrintItem( &(bd->bdItem), &(bd->bdFontList), bd, y, &y, &ps );

    appPsFinishPage( f, ps.psCurrentPage );

    fprintf( f, "%%%%Trailer\n" );
    fprintf( f, "%%%%Pages: %d\n", ps.psCurrentPage+ 1 );
    fprintf( f, "%%%%EOF\n" );

    if  ( fontList )
	{ free( fontList );	}

    return 0;
    }
