/************************************************************************/
/*									*/
/*  Save a BufferDocument into an RTF file.				*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<bitmap.h>

#   include	<appSystem.h>

#   include	"docBuf.h"
#   include	<sioGeneral.h>
#   include	<appImage.h>

#   include	<debugon.h>

#   include	<charnames.h>

#   define	TWIPS_TO_SIZE(x)	(((x)+9)/18)

/************************************************************************/
/*									*/
/*  Information used when writing HTML.					*/
/*									*/
/************************************************************************/
#   define	HTMLmaskFONT		0x01
#   define	HTMLmaskSUPER		0x02
#   define	HTMLmaskSUB		0x04
#   define	HTMLmaskITALIC		0x08
#   define	HTMLmaskBOLD		0x10
#   define	HTMLmaskUNDERLINE	0x20

typedef struct HtmlPushedAttribute
    {
    struct HtmlPushedAttribute *	hpaPrevious;
    unsigned int			hpaChangeMask;
    TextAttribute			hpaAttributeNew;
    } HtmlPushedAttribute;

typedef struct HtmlWritingContext
    {
    const char *	hwcFilename;
    int			hwcBaselength;
    int			hwcRelativeOffset;
    int			hwcUseTableForTabs;
    int			hwcInLink;
    TextAttribute	hwcDefaultAttribute;
    ParagraphProperties	hwcParagraphProperties;
    } HtmlWritingContext;

static void docInitHtmlWritingContext(	HtmlWritingContext *	hwc )
    {
    hwc->hwcFilename= (const char *)0;
    hwc->hwcBaselength= 0;
    hwc->hwcRelativeOffset= 0;
    hwc->hwcUseTableForTabs= 0;
    hwc->hwcInLink= 0;

    docInitTextAttribute( &(hwc->hwcDefaultAttribute) );
    hwc->hwcDefaultAttribute.taFontSizeHalfPoints= 24;
    hwc->hwcDefaultAttribute.taFontNumber= -1;

    docInitParagraphProperties( &(hwc->hwcParagraphProperties) );

    return;
    }

static void docCleanHtmlWritingContext(	HtmlWritingContext *	hwc )
    {
    docCleanParagraphProperties( &(hwc->hwcParagraphProperties) );

    return;
    }

/************************************************************************/
/*									*/
/*  Save a tag with an argument.					*/
/*									*/
/************************************************************************/
static void docHtmlWriteArgTag(	const char *		tag,
				int			arg,
				SimpleOutputStream *	sos )
    {
    char	scratch[20];

    sioOutPutString( " ", sos );
    sioOutPutString( tag, sos );

    sprintf( scratch, "=%d", arg );
    sioOutPutString( scratch, sos );

    return;
    }

static void docHtmlEscapeString(	const unsigned char *	s,
					SimpleOutputStream *	sos )
    {
    while( *s )
	{
	switch( *s )
	    {
	    case '"':
		sioOutPutString( "&quot;", sos );
		break;
	    case '&':
		sioOutPutString( "&amp;", sos );
		break;
	    case '>':
		sioOutPutString( "&gt;", sos );
		break;
	    case '<':
		sioOutPutString( "&lt;", sos );
		break;
	    default:
		sioOutPutCharacter( *s, sos );
		break;
	    }

	s++;
	}

    return;
    }

static int docHtmlFontSize(	int	halfPoints )
    {
    if  ( halfPoints/2 < 6 )	{ return 1;	}
    if  ( halfPoints/2 < 9 )	{ return 2;	}

    if  ( halfPoints/2 > 24 )	{ return 7;	}
    if  ( halfPoints/2 > 19 )	{ return 6;	}
    if  ( halfPoints/2 > 15 )	{ return 5;	}
    if  ( halfPoints/2 > 11 )	{ return 4;	}

    return 3;
    }

static void docHtmlFinishGroup(		SimpleOutputStream *	sos,
					HtmlWritingContext *	hwc )
    {
    if  ( hwc->hwcUseTableForTabs )
	{ sioOutPutString( "</TABLE>\n", sos ); hwc->hwcUseTableForTabs= 0; }
    }

static void docHtmlPopAttributes(	SimpleOutputStream *	sos,
					int *			pCol,
					HtmlPushedAttribute *	hpaOld,
					HtmlPushedAttribute **	pHpaNew )
    {
    HtmlPushedAttribute *	hpaNew;
    int				madeTags= 0;

    if  ( ! hpaOld )
	{ XDEB(hpaOld); *pHpaNew= hpaOld; return;		}

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskUNDERLINE ) )
	{ sioOutPutString( "</U>", sos ); madeTags= 1; *pCol += 4;	}

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskBOLD ) )
	{ sioOutPutString( "</B>", sos ); madeTags= 1; *pCol += 4;	}

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskITALIC ) )
	{ sioOutPutString( "</I>", sos ); madeTags= 1; *pCol += 4;	}

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskSUB ) )
	{ sioOutPutString( "</SMALL></SUB>", sos ); madeTags= 1; *pCol += 14; }

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskSUPER ) )
	{ sioOutPutString( "</SMALL></SUP>", sos ); madeTags= 1; *pCol += 14; }

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskFONT ) )
	{ sioOutPutString( "</FONT>", sos ); madeTags= 1; *pCol += 7;	}

    hpaNew= hpaOld->hpaPrevious;

    free( hpaOld );

    *pHpaNew= hpaNew; return;
    }

/************************************************************************/
/*									*/
/*  Change attributes.							*/
/*									*/
/*  1)  Pop all attributes until nothing is to be turned off compared	*/
/*	to the old attributes. (And the font remains the same)		*/
/*									*/
/************************************************************************/

static int docHtmlChangeAttributes(	SimpleOutputStream *	sos,
					int *			pCol,
					const BufferDocument *	bd,
					HtmlPushedAttribute *	hpaOld,
					HtmlPushedAttribute **	pHpaNew,
					TextAttribute		taDefault,
					TextAttribute		taNew	)
    {
    HtmlPushedAttribute *	hpaNew;

    int				col= *pCol;

    TextAttribute		taOld;

    /*  1  */
    while( hpaOld )
	{
	taOld= hpaOld->hpaAttributeNew;

	if  ( ( ! taOld.taFontIsSlanted			||
		taNew.taFontIsSlanted			)	&&

	      ( ! taOld.taFontIsBold			||
		taNew.taFontIsBold			)	&&

	      ( ! taOld.taIsUnderlined			||
		taNew.taIsUnderlined			)	&&

	      ( taOld.taSuperSub == DOCfontREGULAR	||
		taNew.taSuperSub != DOCfontREGULAR )		&&

	      taOld.taFontNumber ==
				taNew.taFontNumber		&&

	      taOld.taFontSizeHalfPoints ==
				taNew.taFontSizeHalfPoints	)
	    { break;	}

	docHtmlPopAttributes( sos, &col, hpaOld, &hpaOld );
	}

    if  ( hpaOld						&&
	  docEqualFont( &(hpaOld->hpaAttributeNew), &taNew )	)
	{ *pHpaNew= hpaOld; *pCol= col; return 0; }

    hpaNew= (HtmlPushedAttribute *)malloc( sizeof(HtmlPushedAttribute) );
    if  ( ! hpaNew )
	{ XDEB(hpaNew); return -1;	}

    hpaNew->hpaPrevious= hpaOld;
    hpaNew->hpaChangeMask= 0;
    if  ( hpaOld )
	{ taOld= hpaOld->hpaAttributeNew;	}
    else{ taOld= taDefault;			}
    hpaNew->hpaAttributeNew= taNew;

    if  ( taNew.taFontNumber != taOld.taFontNumber			||
	  taNew.taFontSizeHalfPoints != taOld.taFontSizeHalfPoints	)
	{
	const DocumentFontList *	dfl= &bd->bdFontList;
	const DocumentFont *		df= dfl->dflFonts+ taNew.taFontNumber;
	char *				font= (char *)0;

	sioOutPutString( "<FONT", sos );
	col += 5;

	if  ( taNew.taFontNumber != taOld.taFontNumber )
	    {
	    if  ( ! strcmp( df->dfFamilyStyle, "fswiss" ) )
		{ font= "Helvetica,Arial";	}
	    if  ( ! strcmp( df->dfFamilyStyle, "froman" ) )
		{ font= "Times,Times New Roman";	}
	    if  ( ! strcmp( df->dfFamilyStyle, "fmodern" ) )
		{ font= "Courier";	}
	    if  ( ! strcmp( df->dfFamilyStyle, "ftech" ) )
		{ font= "Symbol";	}

	    if  ( font )
		{
		sioOutPutString( " FACE=\"", sos );
		col += 6;
		sioOutPutString( font, sos );
		sioOutPutString( "\"", sos );
		col += strlen( font )+ 2;
		}
	    }

	if  ( taNew.taFontSizeHalfPoints != taOld.taFontSizeHalfPoints )
	    {
	    int		oldSize= docHtmlFontSize( taOld.taFontSizeHalfPoints );
	    int		newSize= docHtmlFontSize( taNew.taFontSizeHalfPoints );

	    if  ( newSize != oldSize )
		{
		docHtmlWriteArgTag( "SIZE", newSize, sos );
		col += 8;
		}
	    }

	hpaNew->hpaChangeMask |= HTMLmaskFONT;
	sioOutPutString( ">", sos );
	col++;
	}

    if  ( taNew.taSuperSub == DOCfontSUPERSCRIPT	&&
	  taOld.taSuperSub == DOCfontREGULAR		)
	{
	hpaNew->hpaChangeMask |= HTMLmaskSUPER;
	sioOutPutString( "<SUP><SMALL>", sos ); col += 5;
	}

    if  ( taNew.taSuperSub == DOCfontSUBSCRIPT	&&
	  taOld.taSuperSub == DOCfontREGULAR	)
	{
	hpaNew->hpaChangeMask |= HTMLmaskSUB;
	sioOutPutString( "<SUB><SMALL>", sos ); col += 5;
	}

    if  ( taNew.taFontIsSlanted && ! taOld.taFontIsSlanted )
	{
	hpaNew->hpaChangeMask |= HTMLmaskITALIC;
	sioOutPutString( "<I>", sos ); col += 3;
	}

    if  ( taNew.taFontIsBold && ! taOld.taFontIsBold )
	{
	hpaNew->hpaChangeMask |= HTMLmaskBOLD;
	sioOutPutString( "<B>", sos ); col += 3;
	}

    if  ( taNew.taIsUnderlined && ! taOld.taIsUnderlined )
	{
	hpaNew->hpaChangeMask |= HTMLmaskUNDERLINE;
	sioOutPutString( "<U>", sos ); col += 3;
	}

    *pCol= col; *pHpaNew= hpaNew; return 0;
    }

static int docHtmlSaveImgTag(	const InsertedObject *		io,
				const BitmapDescription *	bd,
				SimpleOutputStream *		sos,
				const char *			filename )
    {
    int		w;
    int		h;
    int		d;

    sioOutPutString( "<IMG SRC=\"", sos );
    sioOutPutString( filename, sos );
    sioOutPutString( "\"", sos );

    w= TWIPS_TO_SIZE( ( io->ioScaleX* io->ioTwipsWide )/100 );
    h= TWIPS_TO_SIZE( ( io->ioScaleY* io->ioTwipsHigh )/100 );

    d= ( 100* bd->bdPixelsWide- 100* w )/ bd->bdPixelsWide;
    if  ( d < 0 )
	{ d= -d;	}
    if  ( d <= 15 )
	{ w= bd->bdPixelsWide;	}
    docHtmlWriteArgTag( "WIDTH", w, sos );

    d= ( 100* bd->bdPixelsHigh- 100* h )/ bd->bdPixelsHigh;
    if  ( d < 0 )
	{ d= -d;	}
    if  ( d <= 15 )
	{ h= bd->bdPixelsHigh;	}
    docHtmlWriteArgTag( "HEIGHT", h, sos );

    sioOutPutString( " ALT=\"&lt;IMG&gt;\">\n", sos );

    return 0;
    }

static int docHtmlSavePicture(	InsertedObject *	io,
				SimpleOutputStream *	sos,
				int *			pDone,
				HtmlWritingContext *	hwc )
    {
    static char *	scratch;
    char *		fresh;
    AppBitmapImage *	abi;

    if  ( ! hwc->hwcFilename )
	{ return 0;	}

    abi= (AppBitmapImage *)io->ioPrivate;
    if  ( ! abi )
	{ return 0;	}

    if  ( bmCanWriteGifFile( &abi->abiBitmap, 89, 10.0 )	&&
	  bmCanWriteJpegFile( &abi->abiBitmap, 0, 10.0 )	)
	{ return 0;	}

    fresh= (char *)realloc( scratch, hwc->hwcBaselength+ 50 );
    if  ( ! fresh )
	{ XDEB(fresh); return -1;  }
    scratch= fresh;

    strncpy( scratch, hwc->hwcFilename, hwc->hwcBaselength );
    strcpy( scratch+ hwc->hwcBaselength, ".img" );

    if  ( appTestDirectory( scratch )	&&
	  appMakeDirectory( scratch )	)
	{ SDEB(scratch); return -1;	}

    if  ( io->ioBliptag == 0 )
	{ io->ioBliptag= appGetTimestamp();	}

    if  ( ! bmCanWriteGifFile( &abi->abiBitmap, 89, 10.0 ) )
	{
	sprintf( scratch, "%.*s.img/%08x.gif",
			hwc->hwcBaselength, hwc->hwcFilename, io->ioBliptag );

	if  ( bmWriteGifFile( scratch, abi->abiBuffer, &abi->abiBitmap,
								89, 10.0 ) )
	    { SDEB(scratch); return -1;	}

	if  ( docHtmlSaveImgTag( io, &abi->abiBitmap, sos,
					    scratch+ hwc->hwcRelativeOffset ) )
	    { SDEB(scratch); return -1;	}

	*pDone= 1; return 0;
	}

    if  ( ! bmCanWriteJpegFile( &abi->abiBitmap, 89, 10.0 ) )
	{
	sprintf( scratch, "%.*s.img/%08x.jpg",
			hwc->hwcBaselength, hwc->hwcFilename, io->ioBliptag );

	if  ( bmWriteJpegFile( scratch, abi->abiBuffer, &abi->abiBitmap,
								89, 10.0 ) )
	    { SDEB(scratch); return -1;	}

	if  ( docHtmlSaveImgTag( io, &abi->abiBitmap, sos,
					    scratch+ hwc->hwcRelativeOffset ) )
	    { SDEB(scratch); return -1;	}

	*pDone= 1; return 0;
	}

    return 0;
    }

static int docHtmlSaveParticules( SimpleOutputStream *	sos,
				int *			pCol,
				const BufferDocument *	bd,
				const BufferItem *	bi,
				TextAttribute		taDefault,
				int			part,
				int			returnOnTabs,
				HtmlWritingContext *	hwc )
    {
    int				done= 0;

    int				col= *pCol;
    int				afterSpace= 0;

    TextParticule *		tp= bi->biParaParticules+ part;
    int				stroff= tp->tpStroff;
    unsigned char *		s= bi->biParaString+ stroff;

    int				pictureDone;
    InsertedObject *		io;

    HtmlPushedAttribute *	hpa= (HtmlPushedAttribute *)0;

    TextAttribute		taFrom= taDefault;

    while( part < bi->biParaParticuleCount )
	{
	switch( tp->tpKind )
	    {
	    case DOCkindTAB:
		if  ( returnOnTabs )
		    { goto finish;	}
		else{
		    sioOutPutString( " ", sos );
		    col += 1; afterSpace= 1;
		    }
		s++; stroff++;
		break;
	    case DOCkindTEXT:
		if  ( docHtmlChangeAttributes( sos, &col, bd, hpa, &hpa,
					    taDefault, tp->tpTextAttribute ) )
		    { LDEB(1); return -1;	}
		taFrom= tp->tpTextAttribute;

		if  ( afterSpace && col+ tp->tpStrlen > 72 )
		    { sioOutPutString( "\n", sos ); col= 0; }

		while( stroff < tp->tpStroff+ tp->tpStrlen )
		    {
		    switch( *s )
			{
			case '"':
			    sioOutPutString( "&quot;", sos );
			    s++; stroff++; col += 5;
			    break;
			case '&':
			    sioOutPutString( "&amp;", sos );
			    s++; stroff++; col += 4;
			    break;
			case '>':
			    sioOutPutString( "&gt;", sos );
			    s++; stroff++; col += 3;
			    break;
			case '<':
			    sioOutPutString( "&lt;", sos );
			    s++; stroff++; col += 3;
			    break;
			default:
			    sioOutPutCharacter( *s, sos );
			    col++; s++; stroff++;
			    break;
			}
		    }
		afterSpace= 0;
		if  ( tp->tpStrlen > 0 && s[-1] == ' ' )
		    { afterSpace= 1;	}
		break;
	    case DOCkindOBJECT:
		pictureDone= 0;
		io= bi->biParaObjects+ tp->tpObjectNumber;

		if  (   io->ioKind == DOCokPICTWMETAFILE		||
			io->ioKind == DOCokPICTPNGBLIP			||
			io->ioKind == DOCokPICTJPEGBLIP			||
		      ( io->ioKind == DOCokOLEOBJECT 		&&
		        io->ioResultKind == DOCokPICTWMETAFILE	)	)
		    {
		    if  ( docHtmlSavePicture( io, sos, &pictureDone, hwc ) )
			{ XDEB(io);	}
		    }

		if  ( ! pictureDone )
		    { sioOutPutString( " ", sos ); col += 1; }
		else{
		    taFrom= hwc->hwcDefaultAttribute;
		    col= 0;
		    }
		afterSpace= 0; s++; stroff++;
		break;
	    case DOCkindBKMKSTART:
	    case DOCkindBKMKEND:
	    case DOCkindFIELDSTART:
	    case DOCkindFIELDEND:
		goto finish;
		break;
	    case DOCkindXE:
	    case DOCkindTC:
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		break;
	    default:
		LDEB(tp->tpKind);
		s += tp->tpStrlen; stroff += tp->tpStrlen;
		break;
	    }

	done++; part++; tp++;
	}

  finish:
    while( hpa )
	{ docHtmlPopAttributes( sos, &col, hpa, &hpa ); }

    *pCol= col; return done;
    }

static int docHtmlSaveLink(	SimpleOutputStream *	sos,
				int *			pCol,
				const BufferDocument *	bd,
				const BufferItem *	bi,
				TextAttribute		taFrom,
				int			part,
				const char *		fileName,
				int			fileSize,
				const char *		markName,
				int			markSize,
				HtmlWritingContext *	hwc )
    {
    int			col= *pCol;

    int			done;
    int			returnOnTabs= 0;

    sioOutPutString( "<A HREF=\"", sos ); col += 8;

    if  ( fileName )
	{
	while( fileSize > 0 )
	    {
	    sioOutPutCharacter( *fileName, sos );
	    fileName++; fileSize--; col++;
	    }
	}

    if  ( markName && markSize > 0 )
	{
	sioOutPutCharacter( '#', sos );

	while( markSize > 0 )
	    {
	    sioOutPutCharacter( *markName, sos );
	    markName++; markSize--; col++;
	    }
	}

    sioOutPutString( "\">", sos ); col++;

    hwc->hwcInLink++;

    done= docHtmlSaveParticules( sos, &col, bd, bi,
				    taFrom, part, returnOnTabs, hwc );
    
    hwc->hwcInLink--;

    if  ( done < 1 )
	{ LDEB(done); return -1;	}

    sioOutPutString( "</A>", sos ); col += 4;

    *pCol= col; return done;
    }

static int docHtmlSaveBookmark(	SimpleOutputStream *	sos,
				int *			pCol,
				const BufferDocument *	bd,
				const BufferItem *	bi,
				TextAttribute		taFrom,
				int			part,
				const char *		markName,
				int			markSize,
				HtmlWritingContext *	hwc )
    {
    int			col= *pCol;

    int			done;
    int			returnOnTabs= 0;

    sioOutPutString( "<A NAME=\"", sos ); col += 8;

    if  ( markName )
	{
	while( markSize > 0 )
	    {
	    sioOutPutCharacter( *markName, sos );
	    markName++; markSize--; col++;
	    }
	}

    sioOutPutString( "\">", sos ); col++;

    done= docHtmlSaveParticules( sos, &col, bd, bi,
				    taFrom, part, returnOnTabs, hwc );
    
    if  ( done < 0 )
	{ LDEB(done); return -1;	}

    sioOutPutString( "</A>", sos ); col += 4;

    if  ( part+ done < bi->biParaParticuleCount- 1 )
	{
	TextParticule *	tp= bi->biParaParticules+ part+ done;

	if  ( tp->tpKind == DOCkindBKMKEND )
	    { done++;	}
	}

    *pCol= col; return done;
    }

static int docHtmlSaveParaItem(	SimpleOutputStream *	sos,
				const BufferDocument *	bd,
				const BufferItem *	bi,
				HtmlWritingContext *	hwc )
    {
    TextParticule *		tp;
    unsigned char *		s;

    TextAttribute		taDefault;
    TextAttribute		taFrom;

    int				useTABLE= 0;

    int				part= 0;
    int				stroff= 0;
    int				tab= 0;

    int				col= 0;

    int				done;
    HtmlPushedAttribute *	hpa= (HtmlPushedAttribute *)0;

    int				fontHeight= 1000;

    DocumentField *		df;

    /*  1  */
    taDefault= hwc->hwcDefaultAttribute;
    taFrom= hwc->hwcDefaultAttribute;

    if  ( bi->biParaParticuleCount == 0		||
	  bi->biParaStrlen == 0			)
	{
	if  ( hwc->hwcUseTableForTabs )
	    {
	    sioOutPutString( "</TABLE>&nbsp;<BR>\n", sos );
	    hwc->hwcUseTableForTabs= 0;
	    }
	else{ sioOutPutString( "<DIV>&nbsp;</DIV>\n", sos ); }


	return 0;
	}

    part= 0;
    stroff= 0;
    tp= bi->biParaParticules+ part;

    if  ( bi->biParaAlignment == DOCiaLEFT				&&
	  bi->biParaLeftIndentTwips > 0					&&
	  bi->biParaFirstIndentTwips < 0				&&
	  bi->biParaFirstIndentTwips >= -bi->biParaLeftIndentTwips	&&
	  ( tp->tpKind == DOCkindTEXT				    ||
	    ( hwc->hwcUseTableForTabs && tp->tpKind == DOCkindTAB ) )	&&
	  bi->biParaParticuleCount > 1					)
	{
	if  ( bi->biParaAlignment == DOCiaLEFT )
	    {
	    int		i;

	    for ( i= 1; i < bi->biParaParticuleCount; i++ )
		{
		if  ( bi->biParaParticules[i].tpKind == DOCkindTAB )
		    { useTABLE= 1; break;	}
		}
	    }
	}

    tp= bi->biParaParticules+ part;
    s= bi->biParaString+ stroff;
    fontHeight= 10* tp->tpTextAttribute.taFontSizeHalfPoints;

    if  ( ! useTABLE && hwc->hwcUseTableForTabs )
	{
	hwc->hwcUseTableForTabs= useTABLE;
	sioOutPutString( "</TABLE>\n", sos );
	}

    if  ( useTABLE )
	{
	int	left= TWIPS_TO_SIZE( bi->biParaLeftIndentTwips );

	if  ( ! hwc->hwcUseTableForTabs )
	    {
	    sioOutPutString(
		"<TABLE CELLPADDING=0 CELLSPACING=0><TR VALIGN=\"TOP\">",
									sos );
	    col= 36;
	    }
	else{
	    sioOutPutString( "<TR VALIGN=\"TOP\">", sos );
	    col= 20;
	    }

	sioOutPutString( "<TD", sos );
	docHtmlWriteArgTag( "WIDTH", left, sos );

	if  ( ! hwc->hwcUseTableForTabs )
	    {
	    sioOutPutString( ">\n", sos );
	    col= 0;
	    }
	else{
	    sioOutPutString( ">", sos );
	    col += 12;
	    }

	hwc->hwcUseTableForTabs= useTABLE;
	}
    else{
	sioOutPutString( "<DIV", sos );

	switch( bi->biParaAlignment )
	    {
	    case DOCiaLEFT:
		break;
	    case DOCiaRIGHT:
		sioOutPutString( " ALIGN=\"RIGHT\"", sos );
		break;
	    case DOCiaCENTERED:
		sioOutPutString( " ALIGN=\"CENTER\"", sos );
		break;
	    case DOCiaJUSTIFIED:
		sioOutPutString( " ALIGN=\"JUSTIFY\"", sos );
		break;
	    default:
		LDEB(bi->biParaAlignment);
		break;
	    }

	if  ( bi->biParaSpaceBeforeTwips > fontHeight/ 2 )
	    { sioOutPutString( ">&nbsp;<BR>\n", sos );	}
	else{ sioOutPutString( ">\n", sos );		}
	}

    if  ( bi->biParaTopBorder.bpIsSet )
	{ sioOutPutString( "<HR NOSHADE WIDTH=100%>\n", sos ); }

    switch( tp->tpKind )
	{
	case DOCkindTAB:
	    if  ( hwc->hwcUseTableForTabs )
		{ break;	}
	    /*FALLTHROUGH*/
	case DOCkindTEXT:
	    taDefault= tp->tpTextAttribute;
	    taDefault.taFontIsBold= 0;
	    taDefault.taFontIsSlanted= 0;
	    taDefault.taIsUnderlined= 0;

	    if  ( docHtmlChangeAttributes( sos, &col, bd, hpa, &hpa,
				    hwc->hwcDefaultAttribute, taDefault ) )
		{ LDEB(1); return -1;	}

	    taFrom= taDefault;
	    break;
	default:
	    break;
	}

    while( part < bi->biParaParticuleCount )
	{
	fontHeight= 10* tp->tpTextAttribute.taFontSizeHalfPoints;

	switch( tp->tpKind )
	    {
	    case DOCkindTAB:
		if  ( hwc->hwcUseTableForTabs			&&
		      ( tab == 0 || tab < bi->biParaTabCount )	)
		    {
		    if  ( docHtmlChangeAttributes( sos, &col, bd, hpa, &hpa,
				    hwc->hwcDefaultAttribute,
				    hwc->hwcDefaultAttribute ) )
			{ LDEB(1); return -1;	}

		    sioOutPutString( "</TD>", sos ); col += 5;

		    taFrom= hwc->hwcDefaultAttribute;
		    taDefault= hwc->hwcDefaultAttribute;

		    if  ( col > 0 && col+ 5 > 72 )
			{ sioOutPutString( "\n", sos ); col= 0; }
		    sioOutPutString( "<TD>", sos );

		    tab++; col += 4;
		    }
		else{
		    sioOutPutString( " ", sos );
		    done= 1; col += 1;
		    }
		done= 1;
		tab++; s++; stroff++;
		break;
	    case DOCkindTEXT:
	    case DOCkindOBJECT:
		{
		int		returnOnTabs= 0;

		if  ( hwc->hwcUseTableForTabs				&&
		      ( tab == 0 || tab < bi->biParaTabCount )	)
		    { returnOnTabs= 1;	}

		done= docHtmlSaveParticules( sos, &col, bd, bi,
					taDefault, part, returnOnTabs, hwc );
		}
		break;
	    case DOCkindFIELDSTART:
		df= bi->biParaFieldList.dflFields+ tp->tpObjectNumber;

		if  ( df->dfKind == DOCfkHYPERLINK )
		    {
		    const char *	fileName;
		    int			fileSize;
		    const char *	markName;
		    int			markSize;

		    if  ( ! docGetHyperlink( df,
				&fileName, &fileSize, &markName, &markSize ) )
			{
			done= docHtmlSaveLink( sos, &col, bd, bi, taDefault,
						part+ 1, fileName, fileSize,
						markName, markSize, hwc );
			if  ( done < 0 )
			    { LDEB(done); return -1;	}

			done++;
			}
		    else{ done= 1;	}
		    }
		else{ done= 1;	}
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		break;
	    case DOCkindBKMKSTART:
		df= bi->biParaFieldList.dflFields+ tp->tpObjectNumber;

		done= docHtmlSaveBookmark( sos, &col, bd, bi, taDefault,
				    part+ 1, (char *)df->dfInstructions.odBytes,
				    df->dfInstructions.odSize, hwc );
		if  ( done < 0 )
		    { LDEB(done); return -1;	}

		done++;
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		break;
	    default:
		LDEB(tp->tpKind);
	    case DOCkindBKMKEND:
	    case DOCkindFIELDEND:
	    case DOCkindXE:
	    case DOCkindTC:
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		done= 1;
		break;
	    }

	part += done; tp += done;
	}

    while( hpa )
	{ docHtmlPopAttributes( sos, &col, hpa, &hpa ); }

    if  ( bi->biParaBottomBorder.bpIsSet )
	{ sioOutPutString( "<HR NOSHADE WIDTH=100%>\n", sos ); }

    if  ( hwc->hwcUseTableForTabs )
	{ sioOutPutString( "</TD></TR>\n", sos ); }
    else{
	if  ( bi->biParaSpaceAfterTwips > fontHeight/ 2 )
	    { sioOutPutString( "<BR>&nbsp;</DIV>\n", sos );	}
	else{ sioOutPutString( "</DIV>\n", sos );		}
	}

    if  ( docCopyParagraphProperties( &(hwc->hwcParagraphProperties),
						&(bi->biParaProperties) ) )
	{ LDEB(1);	}

    return 0;
    }

static int docHtmlSaveRowItem(	SimpleOutputStream *	sos,
				const BufferDocument *	bd,
				const BufferItem *	row,
				HtmlWritingContext *	hwc )
    {
    int			i;

    if  ( ! hwc->hwcParagraphProperties.ppInTable )
	{
	int			cellPadding;
	int			useBorder= 0;
	CellProperties *	cp;

	cp= row->biRowCells;
	for ( i= 0; i < row->biGroupChildCount; cp++, i++ )
	    {
	    if  ( cp->cpTopBorder.bpIsSet	||
		  cp->cpLeftBorder.bpIsSet	||
		  cp->cpRightBorder.bpIsSet	||
		  cp->cpBottomBorder.bpIsSet	)
		{ useBorder= 1; break;	}
	    }

	sioOutPutString( "<TABLE CELLSPACING=0", sos );

	if  ( useBorder )
	    { sioOutPutString( " BORDER=\"1\"", sos ); }

	cellPadding= TWIPS_TO_SIZE( row->biRowHalfGapWidthTwips )- 4;
	if  ( cellPadding < 1 )
	    { docHtmlWriteArgTag( "CELLPADDING", 0, sos ); }
	if  ( cellPadding > 1 )
	    { docHtmlWriteArgTag( "CELLPADDING", cellPadding, sos ); }

	sioOutPutString( "><TR VALIGN=\"TOP\">", sos );
	hwc->hwcParagraphProperties.ppInTable= 1;
	}
    else{
	sioOutPutString( "<TR VALIGN=\"TOP\">", sos );
	}

    for ( i= 0; i < row->biGroupChildCount; i++ )
	{
	int		j;
	BufferItem *	cellBi= row->biGroupChildren[i];

	FormattingFrame	ff;

	docParagraphFrame( &ff, &(bd->bdGeometry),
					    -1, cellBi->biGroupChildren[0] );

	sioOutPutString( "<TD", sos );
	docHtmlWriteArgTag( "WIDTH",
		    TWIPS_TO_SIZE( ff.ffX1Geometry- ff.ffX0Geometry ), sos );
	sioOutPutString( ">\n", sos );

	for ( j= 0; j < cellBi->biGroupChildCount; j++ )
	    {
	    BufferItem *	para= cellBi->biGroupChildren[j];

	    if  ( docHtmlSaveParaItem( sos, bd, para, hwc ) )
		{ LDEB(1); return -1;	}
	    }

	docHtmlFinishGroup( sos, hwc );

	if  ( i < row->biGroupChildCount- 1 )
	    { sioOutPutString( "</TD>\n", sos );	}
	else{ sioOutPutString( "</TD>", sos );		}
	}

    sioOutPutString( "</TR>\n", sos );

    return 0;
    }

static int docHtmlSaveItem(	SimpleOutputStream *	sos,
				const BufferDocument *  bd,
				const BufferItem *	bi,
				HtmlWritingContext *	hwc )
    {
    int		i;

    switch( bi->biLevel )
	{
	case DOClevSECT:
	case DOClevDOC:
	case DOClevCELL:
	rowAsGroup:
	    for ( i= 0; i < bi->biGroupChildCount; i++ )
		{
		if  ( docHtmlSaveItem( sos, bd,
					    bi->biGroupChildren[i], hwc ) )
		    { LDEB(i); return -1;	}
		}
	    break;

	case DOClevROW:
	    if  ( ! bi->biRowHasTableParagraphs )
		{
		if  ( hwc->hwcParagraphProperties.ppInTable )
		    {
		    sioOutPutString( "</TABLE>\n", sos );
		    hwc->hwcParagraphProperties.ppInTable= 0;
		    }

		goto rowAsGroup;
		}

	    if  ( docHtmlSaveRowItem( sos, bd, bi, hwc ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevPARA:
	    if  ( docHtmlSaveParaItem( sos, bd, bi, hwc ) )
		{ LDEB(1); return -1;	}
	    break;

	default:
	    LDEB(bi->biLevel); return -1;
	}

    return 0;
    }

int docHtmlSaveDocument(	SimpleOutputStream *	sos,
				const BufferDocument *	bd,
				const char *		filename )
    {
    HtmlWritingContext		hwc;

    const BufferItem *		bi= &bd->bdItem;
    const DocumentGeometry *	dg= &(bd->bdGeometry);

#   if 0
    const DocumentFontList *	dfl= &bd->bdFontList;
    const DocumentFont *	df;
    int				i;
#   endif

    docInitHtmlWritingContext( &hwc );
    

    if  ( filename )
	{
	const char *	s;

	hwc.hwcFilename= filename;
	s= strrchr( filename, '.' );
	if  ( s )
	    { hwc.hwcBaselength= s- filename;		}
	else{ hwc.hwcBaselength= strlen( filename );	}

	s= strrchr( filename, '/' );
	if  ( s )
	    { hwc.hwcRelativeOffset= s- filename+ 1;	}
	else{ hwc.hwcRelativeOffset= 0;			}
	}

    sioOutPutString( "<HTML>\n", sos );

#   if 0
    sioOutPutString( "<STYLE>\n", sos );

    df= dfl->dflFonts;
    for ( i= 0; i < dfl->dflCount; df++, i++ )
	{
	char *			font= (char *)0;
	char			scratch[15];

	sprintf( scratch, "%d", i );

	if  ( ! strcmp( df->dfFamilyStyle, "fswiss" ) )
	    { font= "Helvetica,Arial,sans-serif";	}
	if  ( ! strcmp( df->dfFamilyStyle, "froman" ) )
	    { font= "Times,Times New Roman,serif";	}
	if  ( ! strcmp( df->dfFamilyStyle, "fmodern" ) )
	    { font= "Courier,monospace";	}
	if  ( ! strcmp( df->dfFamilyStyle, "ftech" ) )
	    { font= "Symbol";	}

	sioOutPutString( "FONT.f", sos );
	sioOutPutString( scratch, sos );
	sioOutPutString( " {", sos );
	if  ( font )
	    {
	    sioOutPutString( "font-family: ", sos );
	    sioOutPutString( font, sos );
	    }

	sioOutPutString( " }\n", sos );
	}

    sioOutPutString( "</STYLE>\n", sos );
#   endif

    if  ( docHasDocumentInfo( bd ) )
	{
	sioOutPutString( "<HEAD>\n", sos );

	if  ( bd->bdTitle )
	    {
	    sioOutPutString( "<TITLE>", sos );
	    docHtmlEscapeString( bd->bdTitle, sos );
	    sioOutPutString( "</TITLE>\n", sos );
	    }

	if  ( bd->bdSubject )
	    {
	    sioOutPutString( "<META NAME=\"description\" CONTENT=\"", sos );
	    docHtmlEscapeString( bd->bdSubject, sos );
	    sioOutPutString( "\">\n", sos );
	    }

	if  ( bd->bdKeywords )
	    {
	    sioOutPutString( "<META NAME=\"keywords\" CONTENT=\"", sos );
	    docHtmlEscapeString( bd->bdKeywords, sos );
	    sioOutPutString( "\">\n", sos );
	    }

	if  ( bd->bdComment )
	    {
	    sioOutPutString( "<META NAME=\"comment\" CONTENT=\"", sos );
	    docHtmlEscapeString( bd->bdComment, sos );
	    sioOutPutString( "\">\n", sos );
	    }

	if  ( bd->bdAuthor )
	    {
	    sioOutPutString( "<META NAME=\"author\" CONTENT=\"", sos );
	    docHtmlEscapeString( bd->bdAuthor, sos );
	    sioOutPutString( "\">\n", sos );
	    }

	sioOutPutString( "</HEAD>\n", sos );
	}

    sioOutPutString( "<BODY BGCOLOR=\"#ffffff\" TEXT=\"#000000\">\n", sos );

    if  ( dg->dgLeftMarginTwips > 300		||
	  dg->dgRightMarginTwips > 300		||
	  dg->dgTopMarginTwips > 300		||
	  dg->dgBottomMarginTwips > 300		)
	{
	sioOutPutString( "<TABLE>\n", sos );

	if  ( dg->dgTopMarginTwips > 300 )
	    {
	    sioOutPutString( "<TR><TD", sos );

	    docHtmlWriteArgTag( "HEIGHT",
		TWIPS_TO_SIZE( dg->dgTopMarginTwips ), sos );

	    sioOutPutString( ">&nbsp;</TD></TR>", sos );
	    }

	sioOutPutString( "<TR>", sos );

	if  ( dg->dgLeftMarginTwips > 300 )
	    {
	    sioOutPutString( "<TD", sos );

	    docHtmlWriteArgTag( "WIDTH",
		TWIPS_TO_SIZE( dg->dgLeftMarginTwips ), sos );

	    sioOutPutString( ">&nbsp;</TD>", sos );
	    }

	sioOutPutString( "<TD>\n", sos );
	}

    if  ( docHtmlSaveItem( sos, bd, bi, &hwc ) )
	{ LDEB(bi->biLevel); return -1; }

    docHtmlFinishGroup( sos, &hwc );

    if  ( dg->dgLeftMarginTwips > 300		||
	  dg->dgRightMarginTwips > 300		||
	  dg->dgTopMarginTwips > 300		||
	  dg->dgBottomMarginTwips > 300		)
	{
	sioOutPutString( "</TD>", sos );

	if  ( dg->dgRightMarginTwips > 300 )
	    {
	    sioOutPutString( "<TD", sos );

	    docHtmlWriteArgTag( "WIDTH",
		TWIPS_TO_SIZE( dg->dgRightMarginTwips ), sos );

	    sioOutPutString( ">&nbsp;</TD>", sos );
	    }

	sioOutPutString( "</TR>", sos );

	if  ( dg->dgBottomMarginTwips > 300 )
	    {
	    sioOutPutString( "<TR><TD", sos );

	    docHtmlWriteArgTag( "HEIGHT",
		TWIPS_TO_SIZE( dg->dgBottomMarginTwips ), sos );

	    sioOutPutString( ">&nbsp;</TD></TR>", sos );
	    }

	sioOutPutString( "</TABLE>\n", sos );
	}

    sioOutPutString( "</BODY></HTML>\n", sos );

    docCleanHtmlWritingContext( &hwc );

    return 0;
    }
