/**********************************************************************/
/* ESCAPE.C: Escape keys handling functions                           */
/**********************************************************************/

/*
    Copyright (C) 1996, 1997 Free Software Foundation, Inc.
    Ce programme fait partie du package JERED et est soumis, comme le
    reste du package JERED, a la Gnu General Public License version 2
    ou superieure dont voici un extrait et dont vous pouvez lire
    la totalite en consultant le fichier COPYING.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
*/

#include <sys/types.h>
#include <stdio.h>
#include <ctype.h>

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "jered.h"
#include "mappable.h"
#include "messages.h"

        /* just add new functions to the list below */
        /* for now, add only int functions with three parameters: */
        /* a pointer to the current file (FICHIER *) */
        /* a pointer to the editing buffer (char *) */
        /* an integer value as the position in this buffer (int) */
        /* If you add some functions, be sure that this functions accept */
        /* this definition before being added to the following list */
        /* (each of these values may be used or not (e.g. function_help() */
        /* uses none of its parameters) */
static FUNCMAP functable[] =
        {
                { "FUNCTION_GENHELP", &function_genhelp, MSG_F_GENHELP },
                { "FUNCTION_HELP", &function_help, MSG_F_HELP },
                { "FUNCTION_COMMAND", &function_command, MSG_F_COMMAND },
                { "FUNCTION_SHELL", &function_shell, MSG_F_SHELL },
                { "FUNCTION_QUIT", &function_quit, MSG_F_QUIT },
                { "FUNCTION_SAVE", &function_save, MSG_F_SAVE },
                { "FUNCTION_FIND", &function_find, MSG_F_FIND },
                { "FUNCTION_MARK", &function_mark, MSG_F_MARK },
                { "FUNCTION_UNMARK", &function_unmark, MSG_F_UNMARK },
                { "FUNCTION_COPY", &function_copy, MSG_F_COPY },
                { "FUNCTION_REMOVE", &function_remove, MSG_F_REMOVE },
                { "FUNCTION_MOVE", &function_move, MSG_F_MOVE },
                { "FUNCTION_NEXTFILE", &function_nextfile, MSG_F_NEXTFILE },
                { "FUNCTION_INSERTLINE", &function_insertline, MSG_F_INSERTLINE },
                { "FUNCTION_DELETELINE", &function_deleteline, MSG_F_DELETELINE },
                { "FUNCTION_LINEUP", &function_lineup, MSG_F_LINEUP },
                { "FUNCTION_LINEDOWN", &function_linedown, MSG_F_LINEDOWN },
                { "FUNCTION_PAGEUP", &function_pageup, MSG_F_PAGEUP },
                { "FUNCTION_PAGEDOWN", &function_pagedown, MSG_F_PAGEDOWN },
                { "FUNCTION_HOME", &function_home, MSG_F_HOME },
                { "FUNCTION_END", &function_end, MSG_F_END },
                { "FUNCTION_LEFT", &function_left, MSG_F_LEFT },
                { "FUNCTION_RIGHT", &function_right, MSG_F_RIGHT },
                { "FUNCTION_INVERTINSMOD", &function_invertinsmod, MSG_F_INVERTINSMOD },
                { "FUNCTION_CUTLINE", &function_cutline, MSG_F_CUTLINE },
                { "FUNCTION_TAB", &function_tab, MSG_F_TAB },
                { "FUNCTION_BACKTAB", &function_backtab, MSG_F_BACKTAB },
                { "FUNCTION_DELETECHAR", &function_deletechar, MSG_F_DELETECHAR },
                { "FUNCTION_BACKSPACE", &function_backspace, MSG_F_BACKSPACE },
                { "FUNCTION_PARAGRAPHUP", &function_paragraphup, MSG_F_PARAGRAPHUP },
                { "FUNCTION_PARAGRAPHDOWN", &function_paragraphdown, MSG_F_PARAGRAPHDOWN },
                { "FUNCTION_PREVIOUSWORD", &function_previousword, MSG_F_PREVIOUSWORD },
                { "FUNCTION_NEXTWORD", &function_nextword, MSG_F_NEXTWORD },
                { "FUNCTION_UPPERCASE", &function_uppercase, MSG_F_UPPERCASE },
                { "FUNCTION_LOWERCASE", &function_lowercase, MSG_F_LOWERCASE },
                { "FUNCTION_ALIGNBLOCK", &function_alignblock, MSG_F_ALIGNBLOCK },
                { "FUNCTION_WRAPBLOCK", &function_wrapblock, MSG_F_WRAPBLOCK },
                { "FUNCTION_RECORDMACRO", &function_recordmacro, MSG_F_RECORDMACRO },
                { "FUNCTION_PLAYMACRO", &function_playmacro, MSG_F_PLAYMACRO },
                { "FUNCTION_MAN", &function_man, MSG_F_MAN },
                { "FUNCTION_INFO", &function_info, MSG_F_INFO },
                { "FUNCTION_BG", &function_bg, MSG_F_BG },
                { "FUNCTION_REDRAW", &function_redraw, MSG_F_REDRAW },
                { NULL, NULL, 0 }
        };

static KEYMAP esckeytable[KEY_MAX + 2];
static KEYMAP normkeytable[KEY_MAX + 2];

extern int escape_mode;
extern int autoplay;

char *mapmacro(char *macname);
char *maptext(char *text);
void macro_recording(KEYMAP *pk);
int set_macro(KEYMAP *pk);
/**********************************************************************/
int set_macro(KEYMAP *pk)
{
        /* the second condition should never happen */
        if ((macro == RECORDING) || (macro == PLAYING))
        {
                if (fmacro != NULL)
                {
                        fclose(fmacro);
                        fmacro = NULL;
                }
                macro = NOMACRO;
        }
        set_default_macro(pk->macro);

        if (autoplay == OUI)
                function_playmacro(NULL, NULL, 0);

        return(0);
}
/**********************************************************************/
void macro_recording(KEYMAP *pk)
{
        if ((macro == RECORDING) && (fmacro != NULL))
        {
                /* macros are not re-entrant so we record only functions and text */
                if (pk->func != NULL)
                {
                        /* we don't record FUNCTION_PLAYMACRO nor FUNCTION_RECORDMACRO */
                        if ((pk->func->function != &function_recordmacro) && (pk->func->function != &function_playmacro))
                                fprintf(fmacro, "%s\n", pk->func->funcname);
                }
                else if (pk->text != NULL)
                        fprintf(fmacro, "TEXT_%s\n", pk->text);
        }
}
/**********************************************************************/
int handle_normal_key(int key, FICHIER *fichier, char *buffer, int position)
{
        KEYMAP *pk;

        pk = &normkeytable[key];

        macro_recording(pk);

        if (pk->func != NULL)
                return(pk->func->function(fichier, buffer, position));
        else if (pk->text != NULL)
                return(type_text(pk->text, fichier, buffer, position));
        else if (pk->macro != NULL)
        {
                /* macro must not be reentrant !!! */
                if ((macro != PLAYING) && (macro != RECORDING))
                        return(set_macro(pk));
                else
                        return(handle_normal_char(key, fichier, buffer, position));
        }
        else
                return(0);
}
/**********************************************************************/
int handle_escape_key(FICHIER *fichier, char *buffer, int position)
{
        int key;
        int retcode = 0;
        KEYMAP *pk;

        escape_mode = 1;        /* we are in escape mode */

        affiche_help_line();    /* displays the ESC help bar */

        key = fread_key();      /* read a key from keyboard */
        if (key != KEY_ESCAPE)
        {
                if (ismapped(key))  /* is the key mapped to something ? */
                {
                        pk = &esckeytable[key];

                        macro_recording(pk);

                        if (pk->func != NULL)
                        {
                                /* handle the ESC_x key */
                                retcode = pk->func->function(fichier, buffer, position);
                        }
                        else if (pk->text != NULL)
                                retcode = type_text(pk->text, fichier, buffer, position);
                        else if (pk->macro != NULL)
                                retcode = set_macro(pk);
                }
        }
        escape_mode = 0;        /* we return to normal mode */
        return(retcode);        /* nothing to do */
}
/**********************************************************************/
FUNCMAP *findfunction(char *funcname)
{
        FUNCMAP *p;

        for (p = functable; p->funcname != NULL; p++)
        {
                if (! strcasecmp(p->funcname, funcname))
                        return(p);
        }
        return(NULL);
}
/**********************************************************************/
int ismapped(int key)
{
        KEYMAP *pk;

        if (escape_mode)
                pk = &esckeytable[key];
        else
                pk = &normkeytable[key];
        if ((pk->func != NULL) || (pk->text != NULL) || (pk->macro != NULL))
                return(1);
         else
                return(0);
}
/**********************************************************************/
char *get_keymap(int key)
{
        KEYMAP *pk;

        if (key > KEY_MAX)
        {
                key -= (KEY_MAX + 1);
                pk = &esckeytable[key];
        }
        else
                pk = &normkeytable[key];
                
        if (pk->func != NULL)
                return(message[pk->func->funcdesc]);
        else if (pk->text != NULL)
                return(pk->text);
        else if (pk->macro != NULL)
                return(pk->macro);
         else
                return(NULL);
}
/**********************************************************************/
void mapkeyindextofuncmactxt(int kindex, char *funcmactxtname)
{
        KEYMAP *pk;

        if (kindex != -1)
        {
                if (kindex > KEY_MAX)
                {
                        kindex -= (KEY_MAX + 1);
                        pk = &esckeytable[kindex];
                }
                else
                        pk = &normkeytable[kindex];

                /* we ensure nothing is mapped to this key */
                demap_a_key(pk);

                /* we make the new mapping for this key */
                if (! strncasecmp(funcmactxtname, "FUNCTION_", 9))
                        pk->func = findfunction(funcmactxtname);
                else if (! strncasecmp(funcmactxtname, "MACRO_", 6))
                        pk->macro = mapmacro(funcmactxtname + 6);
                else if (! strncasecmp(funcmactxtname, "TEXT_", 5))
                        pk->text = maptext(funcmactxtname + 5);
        }
}
/**********************************************************************/
char *mapmacro(char *macname)
{
        char *buf;

        buf = (char *)malloc(strlen(macname) + 1);
        if (buf != NULL)
                strcpy(buf, macname);
        return(buf);
}
/**********************************************************************/
char *maptext(char *text)
{
        char *buf;

        buf = (char *)malloc(strlen(text) + 1);
        if (buf != NULL)
                strcpy(buf, text);
        return(buf);
}
/**********************************************************************/
void remap_a_key(char *mapping)
{
        char *ptr;      /* pointer to read the key mapping definition */

        ptr = mapping;
        while (ptr && (*ptr != ':'))
                ptr++;
        if (*ptr)
        {
                /* we can eventually remap KEY_: or ESC_KEY_: */
                /* so we must skip the first ':' if two are present */
                if (*(ptr + 1) == ':')
                        ptr++;
                if (! *(ptr + 1))
                        return;
                *ptr++ = '\0';

                mapkeynametofuncmactxt(mapping, ptr);
        }
}
/**********************************************************************/
void mapkeynametofuncmactxt(char *jkeyname, char *funcmactxtname)
{
        int kindex;

        kindex = jkey_index(jkeyname);
        mapkeyindextofuncmactxt(kindex, funcmactxtname);
}
/**********************************************************************/
void build_default_keymap(void)
{
        int i;

        /* initialize key tables */
        for (i = 0; i <= KEY_MAX; i++)
        {
                esckeytable[i].func = NULL;
                esckeytable[i].text = NULL;
                esckeytable[i].macro = NULL;
                normkeytable[i].func = NULL;
                normkeytable[i].text = NULL;
                normkeytable[i].macro = NULL;
        }

        /* default key mappings */
        mapkeynametofuncmactxt("ESC_KEY_a", "FUNCTION_ALIGNBLOCK");
        mapkeynametofuncmactxt("ESC_KEY_h", "FUNCTION_GENHELP");
        mapkeynametofuncmactxt("ESC_KEY_H", "FUNCTION_HELP");
        mapkeynametofuncmactxt("ESC_KEY_l", "FUNCTION_LOWERCASE");
        mapkeynametofuncmactxt("ESC_KEY_r", "FUNCTION_WRAPBLOCK");
        mapkeynametofuncmactxt("ESC_KEY_u", "FUNCTION_UPPERCASE");
        mapkeynametofuncmactxt("ESC_KEY_RIGHT", "FUNCTION_NEXTWORD");
        mapkeynametofuncmactxt("ESC_KEY_LEFT", "FUNCTION_PREVIOUSWORD");
        mapkeynametofuncmactxt("ESC_KEY_DOWN", "FUNCTION_PARAGRAPHDOWN");
        mapkeynametofuncmactxt("ESC_KEY_UP", "FUNCTION_PARAGRAPHUP");
        mapkeynametofuncmactxt("KEY_F1", "FUNCTION_COMMAND");
        mapkeynametofuncmactxt("KEY_F2", "FUNCTION_SHELL");
        mapkeynametofuncmactxt("KEY_F3", "FUNCTION_QUIT");
        mapkeynametofuncmactxt("KEY_F4", "FUNCTION_SAVE");
        mapkeynametofuncmactxt("KEY_F5", "FUNCTION_FIND");
        mapkeynametofuncmactxt("KEY_F6", "FUNCTION_MARK");
        mapkeynametofuncmactxt("KEY_F7", "FUNCTION_UNMARK");
        mapkeynametofuncmactxt("KEY_F8", "FUNCTION_COPY");
        mapkeynametofuncmactxt("KEY_F9", "FUNCTION_MOVE");
        mapkeynametofuncmactxt("KEY_F10", "FUNCTION_REMOVE");
        mapkeynametofuncmactxt("KEY_F11", "FUNCTION_MAN");
        mapkeynametofuncmactxt("KEY_F12", "FUNCTION_INFO");
        mapkeynametofuncmactxt("KEY_CTRL_L", "FUNCTION_REDRAW");
        mapkeynametofuncmactxt("KEY_CTRL_N", "FUNCTION_NEXTFILE");
        mapkeynametofuncmactxt("KEY_CTRL_P", "FUNCTION_PLAYMACRO");
        mapkeynametofuncmactxt("KEY_CTRL_R", "FUNCTION_RECORDMACRO");
        mapkeynametofuncmactxt("KEY_CTRL_T", "FUNCTION_BACKTAB");
        mapkeynametofuncmactxt("KEY_CTRL_W", "FUNCTION_CUTLINE");
        mapkeynametofuncmactxt("KEY_CTRL_Y", "FUNCTION_DELETELINE");
        
        /* the following line doesn't work because of keypad(stdscr, TRUE) */
        /* in 'ecran.c'. In fact KEY_SUSPEND is returned !!! */
        /* mapkeynametofuncmactxt("KEY_CTRL_Z", "FUNCTION_BG"); */
        mapkeynametofuncmactxt("KEY_SUSPEND", "FUNCTION_BG");
        
        mapkeynametofuncmactxt("KEY_RETURN", "FUNCTION_INSERTLINE");
        mapkeynametofuncmactxt("KEY_LINEFEED", "FUNCTION_INSERTLINE");
        mapkeynametofuncmactxt("KEY_UP", "FUNCTION_LINEUP");
        mapkeynametofuncmactxt("KEY_DOWN", "FUNCTION_LINEDOWN");
        mapkeynametofuncmactxt("KEY_RIGHT", "FUNCTION_RIGHT");
        mapkeynametofuncmactxt("KEY_LEFT", "FUNCTION_LEFT");
        mapkeynametofuncmactxt("KEY_PPAGE", "FUNCTION_PAGEUP");
        mapkeynametofuncmactxt("KEY_NPAGE", "FUNCTION_PAGEDOWN");
        mapkeynametofuncmactxt("KEY_HOME", "FUNCTION_HOME");
        mapkeynametofuncmactxt("KEY_END", "FUNCTION_END");
        mapkeynametofuncmactxt("KEY_IC", "FUNCTION_INVERTINSMOD");
        mapkeynametofuncmactxt("KEY_DC", "FUNCTION_DELETECHAR");
        mapkeynametofuncmactxt("KEY_TAB", "FUNCTION_TAB");
        mapkeynametofuncmactxt("KEY_BTAB", "FUNCTION_BACKTAB");
        mapkeynametofuncmactxt("KEY_BACKSPACE", "FUNCTION_BACKSPACE");
}
/**********************************************************************/
void demap_a_key(KEYMAP *pk)
{
        if (pk->text != NULL)
                free(pk->text);
        if (pk->macro != NULL)
                free(pk->macro);
        pk->func = NULL;
        pk->text = NULL;
        pk->macro = NULL;
}
/**********************************************************************/
void demap_keymax(void)
{
        demap_a_key(&normkeytable[KEY_MAX]);
}
/**********************************************************************/
void demap_all_keys(void)
{
        int i;

        for (i = 0; i <= KEY_MAX; i++)
        {
                demap_a_key(&normkeytable[i]);
                demap_a_key(&esckeytable[i]);
        }
}
/**********************************************************************/
int lookup_keymap(char *fname, int esc)
{
        KEYMAP *table;
        KEYMAP *pk;
        int i;

        if (esc)
                table = &esckeytable[0];
        else
                table = &normkeytable[0];
        for (i = 0, pk = table; i <= KEY_MAX; i++, pk++)
        {
                if (pk->func != NULL)
                {
                        if (! strcasecmp(pk->func->funcname, fname))
                                return((int)(pk - table));
                }
        }
        return(NOTFOUND);
}
/**********************************************************************/
