/* vi:set ts=8 sts=0 sw=8:
 * $Id: undo.c,v 1.8 2000/02/18 04:53:08 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "main.h"
#ifdef USE_UNDOREDO
#include <string.h>
#include <gtk/gtk.h>
#include "doc.h"
#include "undo.h"
#include "win.h"
#include "prefs.h"
#include "toolbar.h"

#ifdef GTK_HAVE_FEATURES_1_1_0

static void	disable_before_redo(doc_t *d);
static void	enable_after_redo(doc_t *d);
static void	undo_replace(GtkWidget *text, undo_t *undo);
static void	undo_redo_common(win_t *w, gboolean which);
#ifdef GNP_DEBUG
static void	undo_redo_list_print(doc_t *d, char *label);
#else
#define undo_redo_list_print(d, l)
#endif


/*
 * PUBLIC: undo_list_add
 *
 * adds an undo item to the undolist.  if there exists items on the redo list,
 * removes them first.
 */
void
undo_list_add(doc_t *d, char *text, int start, int end, undo_op_t op)
{
	undo_t *undo;

	undo = g_new(undo_t, 1);
	undo->text = text;
	undo->start = start;
	undo->end = end;
	undo->changed = d->changed;
	undo->op = op;

	/*
	 * if there's anything on the redo list, then we need to wipe out the
	 * redo list because we now have new undo items
	 */
	if (d->redolist) {
		d->redolist = undo_list_delete(d->redolist);
		g_assert(d->redolist == NULL);
		d->redotail = NULL;
#ifdef USE_TOOLBARS
		tb_item_enable(d->w->main_tb, MAIN_TB_STR_REDO, FALSE);
#endif
	}

	/*
	 * add to the undo list.  note that since it's conceivable that the
	 * undo list could get extremely long, appending to the list on each
	 * key typed by the user would result in awful response time.  so what
	 * we do is we keep track of the tail (end) of the list, and always
	 * append using the tail.
	 *
	 * The reason why we even keep the tail around instead of just treating
	 * both undo and redo lists like a stack is because in the case of
	 * Undo/Redo "All", it's a lot faster to append one list to the other
	 * instead of popping off each item and adding them one at a time.
	 * This boneheaded method is what gedit uses (so don't do this).
	 */
	if (!d->undolist) {
#ifdef USE_TOOLBARS
		tb_item_enable(d->w->main_tb, MAIN_TB_STR_UNDO, TRUE);
#endif
		d->undolist = g_list_append(d->undolist, undo);
		d->undotail = d->undolist;
	} else {
		d->undotail = g_list_append(d->undotail, undo);
		d->undotail = g_list_next(d->undotail);
	}
} /* undo_list_add */


/*
 * PUBLIC: undo_list_delete
 *
 * used to remove either the undo list or the redo list.
 */
GList *
undo_list_delete(GList *list)
{
	GList *tmp;
	undo_t *undo;

	while (list) {
		tmp = list;
		list = g_list_remove_link(list, tmp);
		undo = (undo_t *)(tmp->data);
		g_list_free_1(tmp);
		g_free(undo->text);
		g_free(undo);
	}

	return list;
} /* undo_list_delete */


/*
 * PUBLIC: doc_redo_cb
 *
 * menu callback for the 'redo' command
 */
void
redo_cb(GtkWidget *wgt, gpointer cbdata)
{
	undo_redo_common((win_t *)cbdata, 1);
} /* redo_cb */


/*
 * which: 0 = undo, 1 = redo
 */
static void
undo_redo_common(win_t *w, gboolean which)
{
	doc_t *d;
	undo_t *undo;
	GList *tmp;
	GList **srchead, **srctail;
	GList **dsthead, **dsttail;
	int pt;

	d = DOC_CURRENT(w);

	g_assert(which == 0 || which == 1);

	/*
	 * we have two lists, the undo list, and the redo list.  here, figure
	 * out which one we're using, so that we know which list an item moves
	 * from, and which list it moves to.
	 */
	if (which == 0) {
		srchead = &d->undolist;
		srctail = &d->undotail;
		dsthead = &d->redolist;
		dsttail = &d->redotail;
	} else {
		srchead = &d->redolist;
		srctail = &d->redotail;
		dsthead = &d->undolist;
		dsttail = &d->undotail;
	}

	if (!(*srchead) || !GTK_IS_TEXT(d->data))
		return;

	undo_redo_list_print(d, "undo_redo_common before");
	disable_before_redo(d);

	undo = (undo_t *)(*srctail)->data;
	g_assert(undo->op == UndoInsert || undo->op == UndoDelete ||
		 undo->op == UndoReplace);
	gtk_text_set_point(GTK_TEXT(d->data), undo->start);
#ifdef GTK_HAVE_FEATURES_1_1_0
	gtk_editable_set_position(GTK_EDITABLE(d->data), undo->start);
#else
	GTK_EDITABLE(d->data)->current_pos = undo->start;
#endif
	if (which == 0) {	/* UNDO */
		if (undo->op == UndoInsert) {
			gtk_text_forward_delete(GTK_TEXT(d->data),
						undo->end - undo->start);
		} else if (undo->op == UndoReplace) {
			undo_replace(d->data, undo);
		} else {
			gtk_text_insert(GTK_TEXT(d->data), NULL, NULL, NULL,
					undo->text, strlen(undo->text));
			pt = gtk_text_get_point(GTK_TEXT(d->data));
#ifdef GTK_HAVE_FEATURES_1_1_0
			gtk_editable_set_position(GTK_EDITABLE(d->data), pt);
#else
			GTK_EDITABLE(d->data)->current_pos = pt;
#endif
		}
	} else {	/* REDO */
		if (undo->op == UndoInsert) {
			gtk_text_insert(GTK_TEXT(d->data), NULL, NULL, NULL,
					undo->text, strlen(undo->text));
			pt = gtk_text_get_point(GTK_TEXT(d->data));
#ifdef GTK_HAVE_FEATURES_1_1_0
			gtk_editable_set_position(GTK_EDITABLE(d->data), pt);
#else
			GTK_EDITABLE(d->data)->current_pos = pt;
#endif
		} else if (undo->op == UndoReplace) {
			undo_replace(d->data, undo);
		} else {
			gtk_text_forward_delete(GTK_TEXT(d->data),
						undo->end - undo->start);
		}
	}

	/* remove from src list, and place on dst list */
	tmp = *srctail;
	*srctail = g_list_previous(*srctail);
	*srchead = g_list_remove_link(*srchead, tmp);
	if (!(*dsthead)) {
		*dsthead = g_list_concat(*dsthead, tmp);
		*dsttail = *dsthead;
#ifdef USE_TOOLBARS
		tb_item_enable(d->w->main_tb,
			       (which == 0) ?
				       MAIN_TB_STR_REDO : MAIN_TB_STR_UNDO,
			       TRUE);
#endif
	} else {
		*dsttail = g_list_concat(*dsttail, tmp);
		*dsttail = g_list_next(*dsttail);
	}

#ifdef USE_TOOLBARS
	if (!(*srchead)) {
		tb_item_enable(d->w->main_tb,
			       (which == 0) ?
				       MAIN_TB_STR_UNDO : MAIN_TB_STR_REDO,
			       FALSE);
	}
#endif

	/* update the doc info label */
	if ((d->changed != undo->changed) || which == 1) {
		/* redo is always changed */
		d->changed = which ? TRUE : undo->changed;

		if (d->changed_id)
			gtk_signal_disconnect(GTK_OBJECT(d->data),
					      d->changed_id);
		d->changed_id = gtk_signal_connect(
					GTK_OBJECT(d->data), "changed",
					GTK_SIGNAL_FUNC(doc_changed_cb), d);
		doc_info_label_update(d->w);
	}

	enable_after_redo(d);
	undo_redo_list_print(d, "undo_redo_common after");
} /* undo_redo_common */


static void
undo_replace(GtkWidget *text, undo_t *undo)
{
	char *buf;
	int pt;

	buf = gtk_editable_get_chars(GTK_EDITABLE(text),
				     undo->start, undo->end);
	gtk_text_forward_delete(GTK_TEXT(text), undo->end - undo->start);
	gtk_text_insert(GTK_TEXT(text), NULL, NULL, NULL, undo->text,
			strlen(undo->text));
	undo->end = undo->start + strlen(undo->text);
	undo->text = buf;
	pt = gtk_text_get_point(GTK_TEXT(text));
	gtk_editable_set_position(GTK_EDITABLE(text), pt);
} /* undo_replace */


static void
disable_before_redo(doc_t *d)
{
	/*
	 * need to stop the signal, because if we add/delete text, it will
	 * generate the signal again!
	 */
	gtk_signal_disconnect(GTK_OBJECT(d->data), d->ins_txt_id);
	gtk_signal_disconnect(GTK_OBJECT(d->data), d->del_txt_id);
	gtk_text_freeze(GTK_TEXT(d->data));
}


static void
enable_after_redo(doc_t *d)
{
	/* now re-establish the signals */
	d->ins_txt_id = gtk_signal_connect(GTK_OBJECT(d->data), "insert_text",
					GTK_SIGNAL_FUNC(doc_insert_text_cb), d);
	d->del_txt_id = gtk_signal_connect(GTK_OBJECT(d->data), "delete_text",
					GTK_SIGNAL_FUNC(doc_delete_text_cb), d);
	gtk_text_thaw(GTK_TEXT(d->data));
}


/*
 * PUBLIC: undo_cb
 *
 * menu callback for the 'undo' command
 */
void
undo_cb(GtkWidget *wgt, gpointer cbdata)
{
	undo_redo_common((win_t *)cbdata, 0);
} /* undo_cb */


/*
 * PUBLIC: undo_all_cb
 *
 * menu callback for the 'undo all' command
 */
void
undo_all_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	doc_t *d = DOC_CURRENT(w);

	while (d->undolist)
		undo_redo_common(w, 0);
} /* undo_all_cb */


/*
 * PUBLIC: redo_all_cb
 *
 * menu callback for the 'redo all' command
 */
void
redo_all_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	doc_t *d = DOC_CURRENT(w);

	while (d->redolist)
		undo_redo_common(w, 1);
} /* redo_all_cb */


#ifdef GNP_DEBUG
/*
 * PRIVATE: undo_redo_list_print
 *
 * for debugging purposes only: prints the undo and redo lists.
 */
static void
undo_redo_list_print(doc_t *d, char *label)
{
	GList *lp;

	if (dbg_flags & DBG_UNDO) {
		printf("[%s] undolist: ", label);
		for (lp = d->undolist; lp; lp = lp->next)
			printf("%s ", ((undo_t *)(lp->data))->text);
		printf("\n");
		printf("[%s] redolist: ", label);
		for (lp = d->redolist; lp; lp = lp->next)
			printf("%s ", ((undo_t *)(lp->data))->text);
		printf("\n");
	}
} /* undo_redo_list_print */
#endif	/* GNP_DEBUG */

#endif	/* GTK_HAVE_FEATURES_1_1_0 */
#endif	/* USE_UNDOREDO */

/* the end */
