/* vi:set ts=8 sts=0 sw=8:
 * $Id: search.c,v 1.30 2000/02/21 06:52:35 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "main.h"
#ifdef USE_SEARCH
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#ifdef USE_REGEX
#include <sys/types.h>
#include <regex.h>
#endif
#include <gtk/gtk.h>
#include "doc.h"
#include "undo.h"
#include "dialog.h"
#include "misc.h"
#include "search.h"

#include "gnpintl.h"

/*** local function prototypes ***/
static void search_line_by_point(doc_t *d, int point);
static void search_line_execute(GtkWidget *wgt, gpointer cbdata);
static void replace_widgets_create(win_t *w);
static void search_dialog_create(win_t *w, srg_action_t action);
static void search_dialog_destroy(GtkWidget *wgt, gpointer cbdata);
static bool_t search_execute(GtkWidget *wgt, gpointer cbdata);
static bool_t scrollbar_update(GtkText *txt, int tl, int ln);
static void replace_execute(GtkWidget *wgt, gpointer cbdata);
static void search_entries_changed_cb(GtkWidget *entry, gpointer cbdata);
static void search_dialog_cancel(GtkWidget *entry, gpointer cbdata);
static void replace_all_cb(GtkWidget *wgt, gpointer cbdata);


/*** global function definitions ***/
#if !defined(GTK_HAVE_FEATURES_1_1_0) || defined(USE_GNOME)
/*
 * PUBLIC: search_search_cb
 *
 * main callback routine from pulldown menu
 */
void 
search_search_cb(GtkWidget *wgt, gpointer cbdata)
{
	search_cb(NULL, cbdata, Find);
} /* search_search_cb */


/*
 * PUBLIC: goto_line_cb
 *
 * main callback routine from pulldown menu
 */
void 
goto_line_cb(GtkWidget *wgt, gpointer cbdata)
{
	search_cb(NULL, cbdata, GotoLine);
} /* goto_line_cb */
#endif	/* gtk+-1.0.x */


/*
 * PUBLIC: search_replace_cb
 *
 * main callback routine from pulldown menu
 */
void 
search_replace_cb(GtkWidget *wgt, gpointer cbdata)
{
	search_cb(NULL, cbdata, Replace);
} /* search_replace_cb */


/*
 * PUBLIC: search_cb
 *
 * main callback routine from pulldown menu
 */
void 
search_cb(GtkWidget *wgt, gpointer cbdata, guint cbaction)
{
	win_t *w = (win_t *)cbdata;
	srg_action_t action = (srg_action_t)cbaction;
	search_t *sp;

	g_assert(w != NULL);
	sp = w->srg;

	if (sp) {
		if (action == Again && (sp->action == Find ||
					sp->action == Again)) {
			/* do nothing */
		} else if (sp->action == action) {
			(void)misc_show_and_raise(sp->toplev);
		} else if (sp->action != action) {
			gtk_widget_destroy(sp->toplev);
			g_free(sp);
			search_dialog_create(w, action);
		}
	} else
		search_dialog_create(w, action);
} /* search_cb */


/*
 * PUBLIC: search_again_cb
 *
 * menu callback
 */
void 
search_again_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	if (!w->srg)
		return;

	if (w->srg->stxt == NULL || w->srg->stxt[0] == '\0')
		return;

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w->srg->startcur), TRUE);
	search_execute(NULL, w);
} /* search_again_cb */


/*** local function definitions ***/
/*
 * PRIVATE: replace_widgets_create
 *
 * creates the widgets for 'replace text'
 */
static void
replace_widgets_create(win_t *w)
{
	GtkWidget *vbox, *hbox, *tmp;

	/* search & replace information (replace text) */
	tmp = gtk_frame_new(NULL);
	gtk_container_border_width(GTK_CONTAINER(tmp), 6);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(tmp), vbox);

	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(w->srg->toplev)->vbox), tmp,
			   TRUE, TRUE, 0);
	hbox = gtk_hbox_new(FALSE, 1);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
	tmp = gtk_label_new(_("Replace:"));
	gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 2);
	w->srg->rtxt_wgt = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox), w->srg->rtxt_wgt, TRUE, TRUE, 2);
	(void)gtk_signal_connect(GTK_OBJECT(w->srg->rtxt_wgt), "changed",
				 GTK_SIGNAL_FUNC(search_entries_changed_cb), w);
} /* replace_widgets_create */


/*
 * PRIVATE: search_line_execute
 *
 * goes to line by line number
 */
static void
search_line_execute(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	char *buf, *haystack, *needle;
	doc_t *d;
	int numlines, linenum;
	int a, b, len;
	search_t *sp;

	sp = w->srg;
	linenum = atoi(gtk_entry_get_text(GTK_ENTRY(sp->stxt_wgt)));

	GNPDBG_SEARCH(("search_line_execute: linenum = %d\n", linenum));
	if (linenum < 1)
		return;

	numlines = 1;
	d = DOC_CURRENT(w);

	len = gtk_text_get_length(GTK_TEXT(d->data));
	buf = gtk_editable_get_chars(GTK_EDITABLE(d->data), 1, len);

	a = 1;
	b = len;
	haystack = buf;
	do {
		needle = strchr(haystack, '\n');
		if (needle) {
			haystack = needle + 1;
			if (linenum == numlines)
				b = needle - buf + 1;
			numlines++;
			if (linenum == numlines)
				a = needle - buf + 1;
		}
	} while (needle != NULL);

	g_free(buf);
	GNPDBG_SEARCH(("search_line_execute: numlines = %d\n", numlines));
	if (scrollbar_update(GTK_TEXT(d->data), numlines, linenum)) {
		gtk_text_set_point(GTK_TEXT(d->data), a + 1);
		gtk_editable_set_position(GTK_EDITABLE(d->data), a + 1);
		gtk_editable_select_region(GTK_EDITABLE(d->data), a, b);
	}
} /* search_line_execute */


/*
 * PRIVATE: search_line_by_point
 *
 * goes to line by point/index into text.  called only from search_execute(),
 * after text has been found.
 */
static void
search_line_by_point(doc_t *d, int point)
{
	char *buf, *haystack, *needle;
	int numlines, linenum;
	int len;
	bool_t found;

	GNPDBG_SEARCH(("search_line_by_point: point = %d\n", point));
	len = gtk_text_get_length(GTK_TEXT(d->data));
	buf = gtk_editable_get_chars(GTK_EDITABLE(d->data), 1, len);

	found = FALSE;
	linenum = 1;
	numlines = 1;
	haystack = buf;
	do {
		needle = strchr(haystack, '\n');
		if (needle) {
			if (!found && needle - buf > point) {
				linenum = numlines;
				found = TRUE;
			}

			haystack = needle + 1;
			numlines++;
		}
	} while (needle != NULL);

	g_free(buf);
	GNPDBG_SEARCH(("search_line_by_point: numlines = %d, linenum = %d\n",
			numlines, linenum));
	(void)scrollbar_update(GTK_TEXT(d->data), numlines, linenum);
} /* search_line_by_point */


/*
 * PRIVATE: search_dialog_create
 *
 * creates the search dialog box.
 */
static void
search_dialog_create(win_t *w, srg_action_t action)
{
	GtkWidget *tmp, *vbox, *hbox;
	search_t *sp;

	sp = g_new0(search_t, 1);
	w->srg = sp;

	/* top level dialog window */
	sp->toplev = gtk_dialog_new();
	gtk_window_set_policy(GTK_WINDOW(sp->toplev), TRUE, TRUE, TRUE);
	if (action == Find)
		gtk_window_set_title(GTK_WINDOW(sp->toplev), _("Search Text"));
	else if (action == Replace)
		gtk_window_set_title(GTK_WINDOW(sp->toplev),
				     _("Search and Replace Text"));
	else if (action == GotoLine)
		gtk_window_set_title(GTK_WINDOW(sp->toplev), _("Goto Line #"));
	(void)gtk_signal_connect(GTK_OBJECT(sp->toplev), "destroy",
				 GTK_SIGNAL_FUNC(search_dialog_destroy), w);

	vbox = GTK_DIALOG(sp->toplev)->vbox;

	/* the frame that holds everything */
	tmp = gtk_frame_new(NULL);
	gtk_container_border_width(GTK_CONTAINER(tmp), 6);
	gtk_box_pack_start(GTK_BOX(vbox), tmp, TRUE, TRUE, 0);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(tmp), vbox);

	/* create search label and entry */
	hbox = gtk_hbox_new(FALSE, 1);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
	if (action == GotoLine)
		tmp = gtk_label_new(_("Goto Line #:"));
	else
		tmp = gtk_label_new(_("Search:"));
	gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 2);
	sp->stxt_wgt = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox), sp->stxt_wgt, TRUE, TRUE, 2);
	(void)gtk_signal_connect(GTK_OBJECT(sp->stxt_wgt), "changed",
				 GTK_SIGNAL_FUNC(search_entries_changed_cb), w);

	if (action != GotoLine) {
		/* create a couple radio buttons for starting search position */
		sp->startcur = gtk_radio_button_new_with_label(NULL,
				_("Start at cursor position"));
		gtk_box_pack_start(GTK_BOX(vbox), sp->startcur, TRUE, TRUE, 0);
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(sp->startcur),
					    TRUE);
		sp->startbeg = gtk_radio_button_new_with_label(
				gtk_radio_button_group(
					GTK_RADIO_BUTTON(sp->startcur)),
				_("Start at beginning of the document"));
		gtk_box_pack_start(GTK_BOX(vbox), sp->startbeg, TRUE, TRUE, 0);

		/* case sensitive */
		sp->casesens = gtk_check_button_new_with_label(
							_("Case sensitive"));
		gtk_box_pack_start(GTK_BOX(vbox), sp->casesens, TRUE, TRUE, 2);

#ifdef USE_REGEX
		/* regular expression search? */
		sp->regexp = gtk_check_button_new_with_label(
					_("Regular expression search"));
		gtk_box_pack_start(GTK_BOX(vbox), sp->regexp, TRUE, TRUE, 2);
#endif
	}

	/* create replace widgets if needed */
	if (action == Replace)
		replace_widgets_create(w);

	/* lastly, the buttons */
	sp->find = misc_button_new_w_label(
				_("Find"), GNOME_STOCK_PIXMAP_SEARCH,
				(action == GotoLine) ?
					GTK_SIGNAL_FUNC(search_line_execute) :
					GTK_SIGNAL_FUNC(search_execute),
				(gpointer)w,
				GTK_DIALOG(sp->toplev)->action_area,
				PACK_START | PACK_EXPAND | PACK_FILL |
				CANCEL_DEFAULT | GRAB_DEFAULT, 0);
	gtk_widget_set_sensitive(sp->find, FALSE);

	if (action == Replace) {
		sp->repl = misc_button_new_w_label(
				_("Replace"), NULL,
				GTK_SIGNAL_FUNC(replace_execute),
				(gpointer)w,
				GTK_DIALOG(sp->toplev)->action_area,
				PACK_START | PACK_EXPAND | PACK_FILL |
				CANCEL_DEFAULT, 0);
		sp->replall = misc_button_new_w_label(
				_("Replace All"), NULL,
				GTK_SIGNAL_FUNC(replace_all_cb),
				(gpointer)w,
				GTK_DIALOG(sp->toplev)->action_area,
				PACK_START | PACK_EXPAND | PACK_FILL |
				CANCEL_DEFAULT, 0);
	}

	tmp = misc_button_new_w_label(_(BUTTON_CANCEL),
				      GNOME_STOCK_BUTTON_CANCEL,
				      GTK_SIGNAL_FUNC(search_dialog_cancel),
				      (gpointer)w,
				      GTK_DIALOG(sp->toplev)->action_area,
				      PACK_START | PACK_EXPAND | PACK_FILL |
				      CANCEL_DEFAULT, 0);

	gtk_widget_show_all(sp->toplev);
} /* search_dialog_create */


/*
 * PRIVATE: search_dialog_cancel
 *
 * callback for "Cancel" button
 */
static void
search_dialog_cancel(GtkWidget *entry, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_widget_hide(w->srg->toplev);
} /* search_entries_cancel */


/*
 * PRIVATE: search_entries_changed_cb
 *
 * callback for when the entry boxes change in the search dialog.  sets the
 * corresponding sensitivity for the "Find" and/or "Replace" buttons.
 */
static void
search_entries_changed_cb(GtkWidget *entry, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	bool_t nonempty;
	char *txt;

	txt = gtk_entry_get_text(GTK_ENTRY(entry));
	if (!txt)
		return;	/* shouldn't happen */

	nonempty = (strlen(txt) != 0);

	if (entry == w->srg->stxt_wgt)
		gtk_widget_set_sensitive(w->srg->find, (gboolean)nonempty);

	if (entry == w->srg->rtxt_wgt) {
		gtk_widget_set_sensitive(w->srg->repl, (gboolean)nonempty);
		gtk_widget_set_sensitive(w->srg->replall, (gboolean)nonempty);
	}
} /* search_entries_changed_cb */


/*
 * PRIVATE: search_dialog_destroy
 *
 * zap!
 */
static void
search_dialog_destroy(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	if (GTK_WIDGET_VISIBLE(w->srg->toplev))
		gtk_widget_destroy(w->srg->toplev);

	w->srg->toplev = NULL;
	w->srg->stxt_wgt = NULL;
	w->srg->rtxt_wgt = NULL;
	w->srg->startcur = NULL;
	w->srg->startbeg = NULL;
	w->srg->casesens = NULL;
	w->srg->stxt = NULL;
	w->srg->rtxt = NULL;
	w->srg->found = FALSE;
} /* search_dialog_destroy */


/*
 * PRIVATE: search_execute
 *
 * actually search for the text.  returns TRUE if found, FALSE if not.
 */
static bool_t
search_execute(GtkWidget *wgt, gpointer cbdata)
{
	bool_t oldchanged;
	int idx;
	int startpos;		/* starting position/index */
	int slen;		/* search text length */
	int sblen;		/* search buffer length */
	char *searchbuf;	/* text widget represented as one string */
	int matchnum;		/* num chars matched in search */
	char *match;
	doc_t *d;
	win_t *w = (win_t *)cbdata;
	search_t *sp;

	g_assert(w != NULL);
	d = DOC_CURRENT(w);
	g_assert(d != NULL);
	sp = w->srg;
	g_assert(sp != NULL);

	/* get text to search for */
	sp->stxt = gtk_entry_get_text(GTK_ENTRY(sp->stxt_wgt));
	slen = strlen(sp->stxt);
	if (slen < 1)
		return FALSE;	/* don't do anything if empty */

	/* get search start position */
	if (GTK_TOGGLE_BUTTON(sp->startcur)->active) {
		/*
		startpos = gtk_text_get_point(GTK_TEXT(d->data));
		*/
		if (GTK_EDITABLE(d->data)->selection_end_pos)
			startpos = GTK_EDITABLE(d->data)->selection_end_pos;
		else
			startpos = GTK_EDITABLE(d->data)->current_pos;
		GNPDBG_SEARCH(("search_execute: starting at cur pos = %d\n",
				startpos));
	} else {
		startpos = 0;
		GNPDBG_SEARCH(("search_execute: starting at beginning\n"));
	}

	/*
	 * get the text in the text widget...
	 * yikes, this will take a lot of memory.  perhaps in the future, GTK
	 * will have a better text widget.  it's either consume lots of memory,
	 * or have a search that's so slow and pathetic, it's not even useful.
	 * (see gedit for a pathetic search)
	 */
	searchbuf = gtk_editable_get_chars(
					GTK_EDITABLE(d->data),
					startpos,
					gtk_text_get_length(GTK_TEXT(d->data)));

	sblen = strlen(searchbuf);
	if (sblen < slen) {	/* nothing to do if we don't have much text */
		g_free(searchbuf);
		return FALSE;
	}

	/* search for the text */
	matchnum = slen;
	GNPDBG_SEARCH(("search_execute: looking for '%s' at startpos = %d\n",
			sp->stxt, startpos));
#ifdef USE_REGEX
	if (sp->regexp && GTK_TOGGLE_BUTTON(sp->regexp)->active) {
#define SLENGTH 128
		regex_t rege;
		regmatch_t regm;
		int err;

		err = regcomp(&rege, sp->stxt,
			      GTK_TOGGLE_BUTTON(sp->casesens)->active ?
				      REG_NEWLINE :
				      REG_NEWLINE | REG_ICASE);
		if (err) {
			char *msg = g_new(char, SLENGTH + 1);
			(void)regerror(err, &rege, msg, SLENGTH);
			(void)do_dialog_error("Regex error!", msg);
			g_free(msg);
			g_free(searchbuf);
			return FALSE;
		}

		err = regexec(&rege, searchbuf, 1, &regm, 0);
		if (err) {
			if (err == REG_NOMATCH) {
				match = NULL; /* goes to 'no match found' */
			} else {
				char *msg = g_new(char, SLENGTH + 1);
				(void)regerror(err, &rege, msg, SLENGTH);
				(void)do_dialog_error("Search failed!", msg);
				g_free(msg);
				regfree(&rege);
				g_free(searchbuf);
				return FALSE;
			}
		} else {
			match = searchbuf + regm.rm_so;
			matchnum = regm.rm_eo - regm.rm_so;
		}
		regfree(&rege);
#undef SLENGTH
	} else
#endif	/* USE_REGEX */
		match = GTK_TOGGLE_BUTTON(sp->casesens)->active ?
				strstr(searchbuf, sp->stxt)	:
				strcasestr(searchbuf, sp->stxt);

	/* no match found */
	if (!match) {
		int len;
		char *msg;

		sp->found = FALSE;

		len = strlen(sp->stxt) + 37;
		msg = g_new(char, len);

		sprintf(msg, " No more matches for '%s' ", sp->stxt);
		(void)do_dialog_ok( "No more matches", msg);
		g_free(msg);
		g_free(searchbuf);
		return FALSE;
	}

	/* found the text, now go to the location and highlight it */
	sp->found = TRUE;
	idx = (match - searchbuf) + startpos;
	GNPDBG_SEARCH(("search_execute: found match at idx = %d\n", idx));
	gtk_text_set_point(GTK_TEXT(d->data), idx);
	oldchanged = d->changed;
	gtk_text_freeze(GTK_TEXT(d->data));
	gtk_text_insert(GTK_TEXT(d->data), NULL, NULL, NULL, " ", 1);
	gtk_text_backward_delete(GTK_TEXT(d->data), 1);
	d->changed = oldchanged;
	gtk_editable_select_region(GTK_EDITABLE(d->data), idx, idx + matchnum);
#define TEXT_LENGTH(t)              ((t)->text_end - (t)->gap_size) 
	GNPDBG_SEARCH(("search_execute: text len = %d, idx + matchnum = %d\n",
			TEXT_LENGTH(GTK_TEXT(d->data)), idx+matchnum));
#undef TEXT_LENGTH
	gtk_text_set_point(GTK_TEXT(d->data), idx + matchnum);

	search_line_by_point(d, idx + matchnum);
	gtk_text_thaw(GTK_TEXT(d->data));

	g_free(searchbuf);
	return TRUE;
} /* search_execute */


/*
 * PRIVATE: replace_all_cb
 *
 * replaces all text according to settings selected in the search/repl dialog.
 */
static void
replace_all_cb(GtkWidget *wgt, gpointer cbdata)
{
	search_t *sp = NULL;
	bool_t changed = FALSE;
	bool_t first = TRUE;
	win_t *w = (win_t *)cbdata;

	while (search_execute(NULL, cbdata)) {
		replace_execute(NULL, cbdata);

		/*
		 * if we're doing a replace all from the beginning of the
		 * document, we have to set the "start current" button after
		 * searching the first time or else it will start from the
		 * beginning again.
		 */
		if (first) {
			first = FALSE;
			sp = w->srg;
			if (GTK_TOGGLE_BUTTON(sp->startbeg)->active) {
				GTK_TOGGLE_BUTTON(sp->startbeg)->active = FALSE;
				GTK_TOGGLE_BUTTON(sp->startcur)->active = TRUE;
				changed = TRUE;
			}
		}
	}

	if (changed && sp) {
		GTK_TOGGLE_BUTTON(sp->startbeg)->active = TRUE;
		GTK_TOGGLE_BUTTON(sp->startcur)->active = FALSE;
	}
} /* replace_all_cb */


/*
 * PRIVATE: replace_execute
 *
 * actually replaces the text that was found.  returns TRUE if text was
 * replaced, FALSE if not.
 */
static void
replace_execute(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	doc_t *d;
	int curpos;
	search_t *sp;
	char *buf;

	g_assert(w != NULL);
	sp = w->srg;
	g_assert(sp != NULL);

	if (!sp->found)	/* if previous search didn't find anything, then */
		return;	/* there's nothing to be replaced */

	sp->rtxt = gtk_entry_get_text(GTK_ENTRY(sp->rtxt_wgt));
	if (sp->rtxt[0] == '\0')
		sp->rtxt = NULL;

	g_assert(sp->stxt);
	d = DOC_CURRENT(w);
	g_assert(d);

	gtk_text_freeze(GTK_TEXT(d->data));
	curpos = gtk_text_get_point(GTK_TEXT(d->data));
	buf = gtk_editable_get_chars(GTK_EDITABLE(d->data),
				     curpos - strlen(sp->stxt), curpos);
	gtk_text_backward_delete(GTK_TEXT(d->data), strlen(sp->stxt));
	undo_list_add(d, buf, curpos - strlen(sp->stxt), curpos, UndoDelete);

	curpos = gtk_text_get_point(GTK_TEXT(d->data));
	if (sp->rtxt) {
		gtk_text_insert(GTK_TEXT(d->data), NULL, NULL, NULL,
				sp->rtxt, -1);
		undo_list_add(d, (char *)g_strdup(sp->rtxt),
			      curpos, curpos + strlen(sp->rtxt), UndoInsert);
	}
	curpos = gtk_text_get_point(GTK_TEXT(d->data));
	gtk_editable_set_position(GTK_EDITABLE(d->data), curpos);
	gtk_text_thaw(GTK_TEXT(d->data));

	sp->found = FALSE;
	gtk_signal_emit_by_name(GTK_OBJECT(d->data), "changed");
} /* replace_execute */


/*
 * PRIVATE: scrollbar_update
 *
 * scrolls the text widget to the line number specified.
 *
 * tl: total lines
 * ln: line number to scroll/go to
 *
 * returns TRUE if updated, FALSE if not.
 */
static bool_t
scrollbar_update(GtkText *txt, int tl, int ln)
{
	float value;

	GNPDBG_SEARCH(("scrollbar_update: tl=%d, ln=%d\n", tl, ln));
	if (tl < 3)
		return FALSE;

	if (ln > tl) {
		char buf[32];

		g_snprintf(buf, 32, " File only has %d lines! ", tl);
		(void)do_dialog_ok("End of File!", buf);
		return FALSE;
	}

	value = (ln * GTK_ADJUSTMENT(txt->vadj)->upper) /
		tl - GTK_ADJUSTMENT(txt->vadj)->page_increment;

	gtk_adjustment_set_value(GTK_ADJUSTMENT(txt->vadj), value);
	return TRUE;
} /* scrollbar_update */

#endif	/* USE_SEARCH */

/* the end */
