/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996-1998 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: condition.c,v 1.36 1999/09/27 15:33:39 tiggr Exp $  */

#include "trt.h"
#include <tom/tom-r.h>

/* Iff !0, we're raising a nil receiver.  */
static int raising_nil;

void
trt_register (struct trt_condition_info *c)
{
  trt_thread_info *th = &trt_threads[trt_thread_self ()];

  if (th->cond_num == th->cond_cap)
    {
      th->cond_cap = th->cond_cap ? 2 * th->cond_cap : 64;
      th->cond_stack
	= xrealloc (th->cond_stack, th->cond_cap * sizeof (*th->cond_stack));
    }

  th->cond_stack[th->cond_num++] = c;
}

void
trt_unwind (struct trt_condition_info *c)
{
  trt_thread_info *th = &trt_threads[trt_thread_self ()];

  ASSERT (th->cond_num && th->cond_stack[th->cond_num - 1] == c);
  th->cond_num--;
}

static tom_object
signal_raise (tom_object self, int is_raise)
{
  trt_thread_info *th = &trt_threads[trt_thread_self ()];
  struct _es_i_tom_Condition *this;
  tom_object cc;
  int target, target_class;
  struct trt_bind *bi;

  if (c_tom_Runtime_panic_mode)
    fatal ("panic mode %d during a condition %s", c_tom_Runtime_panic_mode,
	   is_raise ? "raise" : "signal");

  this = trt_ext_address (self, _ei_i_tom_Condition);
  this->raised = !!is_raise;
  cc = (void *) this->condition_class;

  for (target = th->cond_num - 1; target >= 0; target--)
    if (th->cond_stack[target]->kind == TRT_CONDITION_KIND_BIND)
      {
	bi = (void *) th->cond_stack[target];

	for (target_class = 0; target_class < bi->num; target_class++)
	  if (TRT_SEND ((byte_imp), cc, SEL (o_isConditionSuper_r),
			bi->handlers[target_class].condition_class))
	    {
	      void *ret;
	      if (bi->tomc_handler)
		ret = bi->tomc_handler (target_class, self);
	      else
		{
		  void *h = bi->handlers[target_class].condition_handler;
		  ret = TRT_SEND (_PI_, h, SEL (r_eval_r), self);
		}

	      if (!is_raise && ret != self)
		{
		  raising_nil = 0;
		  return ret;
		}
	    }
      }

  return 0;
}

void
i_tom_Condition_v_raise (tom_object self, selector cmd)
{
  signal_raise (self, 1);

  if (c_tom_Conditions_unhandled_condition_handler)
    {
      unimplemented ("unhandled raise handler (need invocation from C of "
		     "method with dynamic argument for this to work)");
    }
  else
    {
      tom_object o;
      tom_byte *p;
      tom_int len;

      o = TRT_SEND ((reference_imp), self, SEL (r_description));
      p = TRT_SEND ((pointer_imp), o, SEL (_pi__byteStringContents), &len);

      fatal ("unhandled condition raise of %s", c_string_with_length (p, len));
    }
}

tom_object
i_tom_Condition_r_signal (tom_object self, selector cmd)
{
  return signal_raise (self, 0);
}

void
i_tom_All_v_throw_x (tom_object self, selector cmd, ...)
{
  trt_thread_info *th = &trt_threads[trt_thread_self ()];
  int target, void_throw, i;
  struct trt_catch *ci;
  va_list ap;

  raising_nil = 0;
  if (c_tom_Runtime_panic_mode)
    fatal ("panic mode %d during throw", c_tom_Runtime_panic_mode);

  for (target = th->cond_num - 1; target >= 0; target--)
    if (th->cond_stack[target]->kind == TRT_CONDITION_KIND_CATCH
	&& self == ((struct trt_catch *) th->cond_stack[target])->tag)
      break;

  if (target < 0)
    trt_raise (0, self, cmd, c_tom_Conditions_uncaught_throw, 0);

  for (th->cond_num--; th->cond_num != target;)
    if (th->cond_stack[th->cond_num]->kind != TRT_CONDITION_KIND_UNWIND)
      th->cond_num--;
    else
      ((struct trt_unwind *) th->cond_stack[th->cond_num--])->protect ();

  void_throw = trt_selector_args_match (cmd->in, &_atd_tom_);
  ci = (void *) th->cond_stack[target];
  if (!void_throw)
    {
      if (!trt_selector_args_match (cmd->in, ci->value_desc))
	fatal ("throw value type mismatches catch type");
      va_start (ap, cmd);
    }

  for (i = 0; i < ci->value_desc->num; i++)
    switch (ci->value_desc->args[i])
      {
      case TRT_TE_BOOLEAN:
      case TRT_TE_BYTE:
	*(tom_byte *) ci->values[i] = void_throw ? 0 : VA_ARG_BYTE (ap);
	break;

      case TRT_TE_CHAR:
	*(tom_char *) ci->values[i] = void_throw ? 0 : VA_ARG_CHAR (ap);
	break;

      case TRT_TE_INT:
	*(tom_int *) ci->values[i] = void_throw ? 0 : va_arg (ap, tom_int);
	break;

      case TRT_TE_LONG:
	*(tom_long *) ci->values[i] = void_throw ? 0 : va_arg (ap, tom_long);
	break;

      case TRT_TE_FLOAT:
	*(tom_float *) ci->values[i] = void_throw ? 0 : VA_ARG_FLOAT (ap);
	break;

      case TRT_TE_DOUBLE:
	*(tom_double *) ci->values[i] = (void_throw ? 0
					 : va_arg (ap, tom_double));
	break;

      case TRT_TE_REFERENCE:
      case TRT_TE_POINTER:
      case TRT_TE_SELECTOR:
	*(void **) ci->values[i] = void_throw ? 0 : va_arg (ap, void *);
	break;

      default:
	fatal (__FILE__ "%d: argument description: %d", __LINE__,
	       (ci->value_desc->args[i]));
      }

#if STACK_REF_STRUCT
  if (ci->sp)
    _trt_sp = ci->sp;
#endif

  longjmp (*ci->jmp, 1);
}

void
trt_send_nil (tom_object rcv, selector cmd, ...)
{
  tom_object c;

  if (raising_nil)
    abort ();

  raising_nil = 1;

  c = TRT_SEND ((reference_imp), _mr_c_tom_SelectorCondition,
		SEL (r_for_r_class_r_message_r_selector_s),
		rcv, c_tom_Conditions_nil_receiver,
		byte_string_with_c_string ("nil receiver"), cmd);

  /* This could, of course, simply signal and return the default value for
     the return type expected by the caller.  */
  TRT_SEND ((void_imp), c, SEL (v_raise));
}

int_imp
trt_lookup_nil (selector cmd)
{
  /* This should return the ``nil_method'', but trt_send_nil will do for
     now...  */
  return (int_imp) trt_send_nil;
}
