/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996-1998 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: report.m,v 1.16 1998/01/05 00:58:29 tiggr Exp $  */

#import "ltt.h"
#import <sys/time.h>
#import <sys/resource.h>

/* Make sure the right methods are known.  */
@interface Object (DeclarationOnly)

-(id <TLString>) filename;

-(int) line;

@end

/* The name of current file being parsed, by any parser, and the current
   line.  */
id <TLString> current_filename;
int current_line;

/* The total number of lines processed.  */
int total_num_lines;

/* The number of reported errors and warnings for `this' file.  */
int this_num_errors, this_num_warnings;

/* The number of reported errors and warnings for all previous files.  */
int total_num_errors, total_num_warnings;

/* Verbosity, as indicated on the commandline.  */
int flag_verbose;

/* The last reported context.  */
static id last_context;

/* Stack of contexts of the next error or warning.  If the TOS differs
   from the LAST_CONTEXT, it is output first.  */
static TLVector *report_contexts;

static void
vreport (id <TLString> s, int line, id prefix, va_list ap)
{
  if (current_filename)
    if (line)
      formac (V_stderr_, @"%@:%d: %@", current_filename, line,
	      prefix ? prefix : @"");
    else
      formac (V_stderr_, @"%@: %@", current_filename, prefix ? prefix : @"");
  else
    formac (V_stderr_, @"%@: ", prog, prefix ? prefix : @"");
  vformac (V_stderr_, s, ap);
  formac (V_stderr_, @"\n");
}

static void
report (id <TLString> s, int line, id prefix, ...)
{
  va_list ap;

  va_start (ap, prefix);
  vreport (s, line, prefix, ap);
  va_end (ap);
}

void
push_report_context (id string)
{
  if (!report_contexts)
    {
      report_contexts = [TLVector vector];
      [report_contexts gcLock];
    }

  [report_contexts addElement: string];
}

void
pop_report_context (void)
{
  int n = [report_contexts length];

  if (!n)
    error (@"interal error: report context underflow");
  [report_contexts _removeElementAtIndex: n - 1];
}

static void
do_report_context (int line)
{
  int n = [report_contexts length];
  id report_context = n ? [report_contexts _elementAtIndex: n - 1] : nil;

  if (report_context != last_context)
    {
      last_context = report_context;
      if (report_context)
	report (@"%@:", 0, nil, report_context);
    }
}

void
fatal (int rc, id <TLString> s, ...)
{
  va_list ap;

  va_start (ap, s);
  vcerror (s, ap);
  va_end (ap);

  exit (rc);
}

void
error (id <TLString> s, ...)
{
  va_list ap;

  this_num_errors++;
  va_start (ap, s);
  vcerror (s, ap);
  va_end (ap);
}

void
internal (id <TLString> s, ...)
{
  va_list ap;

  report (@"internal error", current_line, nil);

  this_num_errors++;
  va_start (ap, s);
  vcerror (s, ap);
  va_end (ap);
}

void
cerror (id <TLString> s, ...)
{
  va_list ap;

  va_start (ap, s);
  vcerror (s, ap);
  va_end (ap);
}

void
vcerror (id <TLString> s, va_list ap)
{
  do_report_context (current_line);

  vreport (s, current_line, nil, ap);
}

void
error_for (id o, id <TLString> s, ...)
{
  va_list ap;

  this_num_errors++;
  va_start (ap, s);
  vcerror_for (o, s, ap);
  va_end (ap);
}

void
cerror_for (id o, id <TLString> s, ...)
{
  va_list ap;

  va_start (ap, s);
  vcerror_for (o, s, ap);
  va_end (ap);
}

void
vcerror_for (Object *o, id <TLString> s, va_list ap)
{
  do_report_context ([o line]);

  formac (V_stderr_, @"%@:%d: ", [o filename], [o line]);
  vformac (V_stderr_, s, ap);
  formac (V_stderr_, @"\n");
}

void
warning (id <TLString> s, ...)
{
  va_list ap;

  this_num_warnings++;
  va_start (ap, s);
  vcwarning (s, ap);
  va_end (ap);
}

void
warning_for (id o, id <TLString> s, ...)
{
  va_list ap;

  this_num_warnings++;
  va_start (ap, s);
  vcwarning_for (o, s, ap);
  va_end (ap);
}

void
cwarning (id <TLString> s, ...)
{
  va_list ap;

  va_start (ap, s);
  vcwarning (s, ap);
  va_end (ap);
}

void
cwarning_for (id o, id <TLString> s, ...)
{
  va_list ap;

  va_start (ap, s);
  vcwarning_for (o, s, ap);
  va_end (ap);
}

void
vcwarning (id <TLString> s, va_list ap)
{
  do_report_context (current_line);

  vreport (s, current_line, @"warning: ", ap);
}

void
vcwarning_for (Object *o, id <TLString> s, va_list ap)
{
  do_report_context ([o line]);

  formac (V_stderr_, @"%@:%d: warning: ", [o filename], [o line]);
  vformac (V_stderr_, s, ap);
  formac (V_stderr_, @"\n");
}
