/*
 * Copyright (c) 1999 The University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * Contributed by the Computer Security Research division,
 * INFOSEC Research and Technology Office, NSA.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/* FLASK */

#include "avdeftab.h"

#define AVDEFTAB_HASH(key) (key % AVDEFTAB_SIZE)

#define AVDEFTAB_EQ(key1,key2) (key1 == key2)

int 
avdeftab_insert(avdeftab_t * h, unsigned int key, avdef_datum_t * datum)
{
	int             hvalue;
	avdef_ptr_t     cur, newnode;


	if (!h)
		return -ENOMEM;

	hvalue = AVDEFTAB_HASH(key);
	cur = h->htable[hvalue];
	while (cur != NULL && !AVDEFTAB_EQ(cur->key, key))
		cur = cur->next;

	if (cur != NULL)
		return -EEXIST;

	newnode = (avdef_ptr_t) malloc(sizeof(struct avdef_node_t));
	if (newnode == NULL)
		return -ENOMEM;
	newnode->key = key;
	newnode->datum = *datum;
	newnode->next = h->htable[hvalue];
	h->htable[hvalue] = newnode;

	h->nel++;
	return 0;
}


avdef_datum_t  *
avdeftab_search(avdeftab_t * h, unsigned int key)
{
	int             hvalue;
	avdef_ptr_t     cur;


	if (!h)
		return NULL;

	hvalue = AVDEFTAB_HASH(key);
	cur = h->htable[hvalue];
	while (cur != NULL && !AVDEFTAB_EQ(cur->key, key))
		cur = cur->next;

	if (cur == NULL)
		return NULL;

	return &cur->datum;
}


static int 
perm_destroy(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	FREE(key, strlen(key) + 1);
	FREE(datum, sizeof(perm_datum_t));
	return 0;
}


int 
constraint_expr_destroy(constraint_expr_t * expr)
{
	expr->count--;
	if (expr->count == 0) {
	  ebitmap_destroy(&expr->bitmap);
	  if (expr->left)
	    constraint_expr_destroy(expr->left);
	  if (expr->right)
	    constraint_expr_destroy(expr->right);
	  free(expr);
	}
	return 0;
}


void 
avdeftab_destroy(avdeftab_t * h)
{
	int             i;
	avdef_ptr_t     cur, temp;
	constraint_node_t *constraint, *ctemp;
	


	if (!h)
		return;

	for (i = 0; i < AVDEFTAB_SIZE; i++) {
		cur = h->htable[i];
		while (cur != NULL) {
		  hashtab_map(cur->datum.private.table, perm_destroy, 0);
		  hashtab_destroy(cur->datum.private.table);
		  constraint = cur->datum.constraints;
		  while (constraint) {
		    constraint_expr_destroy(constraint->expr);
		    ctemp = constraint;
		    constraint = constraint->next;
		    free(ctemp);
		  }
		  temp = cur;
		  cur = cur->next;
		  free(temp);
		}
	}
}


int 
avdeftab_map(avdeftab_t * h,
	     int (*apply) (unsigned int key,
			avdef_datum_t * d,
			void *args),
	  void *args)
{
	int             i, ret;
	avdef_ptr_t     cur;


	if (!h)
		return 0;

	for (i = 0; i < AVDEFTAB_SIZE; i++) {
		cur = h->htable[i];
		while (cur != NULL) {
			ret = apply(cur->key, &cur->datum, args);
			if (ret)
				return ret;
			cur = cur->next;
		}
	}
	return 0;
}


int 
avdeftab_init(avdeftab_t * h)
{
	int             i;


	for (i = 0; i < AVDEFTAB_SIZE; i++)
		h->htable[i] = (avdef_ptr_t) NULL;
	h->nel = 0;
	h->ncons = 0;
	return 0;
}

/* FLASK */
