/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include <string.h>
#include <stdio.h>
#include "RCX_Image.h"
#include "RCX_Disasm.h"
#include "RCX_Link.h"
#include "RCX_Cmd.h"
#include "rcxifile.h"

void Write4(ULong d, FILE *fp);
void Write2(UShort d, FILE *fp);
ULong Read4(FILE *fp);
UShort Read2(FILE *fp);
void WriteSymbol(UByte type, UByte index, const char *name, FILE *fp);

static char *NewString(const char *s);

RCX_Image::RCX_Image()
{
	for(int i=0; i<kRCX_MaxVars; i++)
		fVarNames[i] = nil;
}


void RCX_Image::Clear()
{
	fFragments.SetLength(0);

	for(int i=0; i<kRCX_MaxVars; i++)
	{
		delete [] fVarNames[i];
		fVarNames[i] = nil;
	}
}


void RCX_Image::SetFragmentCount(int count)
{
	fFragments.SetLength(count);
}


void RCX_Image::SetFragment(int index, RCX_FragmentType type, UByte number, const UByte *data,
							int length, const char *name)
{
	Fragment *f;
	
	if (index < 0 || index >= (fFragments.GetLength())) return;
	
	f = &fFragments[index];
	
	delete [] f->fData;
	delete [] f->fName;
	
	f->fData = new UByte[length];
	f->fLength = length;
	f->fType = type;
	f->fNumber = number;
	memcpy(f->fData, data, (size_t)length);

	f->fName = NewString(name);
}


void RCX_Image::SetVariable(int index, const char *name)
{
	delete [] fVarNames[index];
	fVarNames[index] = NewString(name);
}


void RCX_Image::Print(RCX_Printer *dst)
{
	Fragment *f;
	char line[256];
	
	for(int i=0; i<kRCX_MaxVars; i++)
	{
		if (fVarNames[i])
		{
			sprintf(line, "*** Var %d = %s\n", i, fVarNames[i]);
			dst->Print(line);
		}
	}
	
	for(int i=0; i<fFragments.GetLength(); i++)
	{
		f = &fFragments[i];
		
		if (f->fType == kRCX_TaskFragment)
			sprintf(line, "\n*** Task %d", f->fNumber);
		else
			sprintf(line, "\n*** Sub %d", f->fNumber);
		dst->Print(line);

		if (f->fName)
			sprintf(line, " = %s\n", f->fName);
		else
			sprintf(line, "\n");
		dst->Print(line);

		gRCX_Disasm.Print(dst, f->fData, f->fLength);
	}
}


RCX_Result RCX_Image::Download(RCX_Link *link, int programNumber)
{
	RCX_Cmd	cmd;
	RCX_Result result = kRCX_OK;
	int i;
		
	// sync with RCX
	result = link->Sync();
	if (RCX_ERROR(result)) return result;

	// stop any running tasks
	result = link->Send(cmd.Set(kRCX_StopAllOp));
	if (RCX_ERROR(result)) return result;
	
	// select program
	if (programNumber)
	{
		
		result = link->Send(cmd.Set(kRCX_SelectProgramOp, (UByte)(programNumber-1)));
		if (result < 0) return result;
	}
	
	// clear existing tasks and/or subs
	result = link->Send(cmd.MakeDeleteTasks());
	if (RCX_ERROR(result)) return result;
	
	result = link->Send(cmd.MakeDeleteSubs());
	if (RCX_ERROR(result)) return result;

	int total = 0;
	for(i=0; i<fFragments.GetLength(); i++)
		total += fFragments[i].fLength;

	for(i=0; i<fFragments.GetLength(); i++)
	{
		Fragment *f = &fFragments[i];
		result = link->SendFragment(f->fType, f->fNumber, f->fData, f->fLength, i==0 ? total : -1);
		if (RCX_ERROR(result)) return result;
	}

	// play sound when done
	link->Send(cmd.MakePlaySound(5));

	return kRCX_OK;
}


RCX_Result RCX_Image::Read(const char *filename)
{
	FILE *fp;
	UShort count;
	UShort symbolCount;
	int i;
	
	Clear();
	
	fp = fopen(filename, "rb");
	if (!fp) return kRCX_FileError;
	
	// check signature
	if (Read4(fp) != kRCXI_Signature) goto ErrorReturn;
	
	// check version
	if (Read2(fp) > kRCXI_CurrentVersion) goto ErrorReturn;
	
	// get counts
	count = Read2(fp);
	symbolCount = Read2(fp);
	
	// skip reserved field
	Read2(fp);
	
	SetFragmentCount(count);
	for(i=0; i<count; i++)
	{
		UByte type;
		UByte number;
		UShort length;
		Fragment *f;
		
		type = (UByte) getc(fp);
		if (type > 2) goto ErrorReturn;
		
		number = (UByte)getc(fp);
		length = Read2(fp);
		
		if (feof(fp) && (length != 0)) goto ErrorReturn;
		
		f = &fFragments[i];
		f->fType = (RCX_FragmentType)type;
		f->fNumber = number;
		f->fLength = length;

		if (length)
		{
			f->fData = new UByte[length];
			if (fread(f->fData, 1, length, fp) != length) goto ErrorReturn;
			fseek(fp, RCXI_PAD_BYTES(length),SEEK_CUR);
		}
		else
			f->fData = nil;
	}

	for(i=0; i<symbolCount; ++i)
	{
		UByte type;
		UByte index;
		UByte length;
		char *string;
		char **namePtr;
		
		// read symbol headers
		type = getc(fp);
		index = getc(fp);		
		length = getc(fp);
		(void)getc(fp);	// reserved
		
		if (length)
		{
			string = new char[length];
			fread(string, length, 1, fp);
			
			namePtr = GetNamePtr(type, index);
			if (namePtr)
			{
				delete [] *namePtr;
				*namePtr = string;
			}
			else
				delete [] string;
		}
	}

	fclose(fp);
	return kRCX_OK;
	
ErrorReturn:
	fclose(fp);
	return kRCX_FormatError;
}


char **RCX_Image::GetNamePtr(UByte type, UByte index)
{
	switch(type)
	{
	case kRCXI_TaskSymbol:
	case kRCXI_SubSymbol:
		for(int i=0; i<fFragments.GetLength(); ++i)
		{
			if (fFragments[i].fType == type &&
				fFragments[i].fNumber == index)
			{
				return &fFragments[i].fName;
			}
		}
		return 0;
	case kRCXI_VarSymbol:
		if (index < kRCX_MaxVars)
			return &fVarNames[index];
		else
			return 0;
	default:
		return 0;
	}
}


RCX_Result RCX_Image::Write(const char *filename)
{
	FILE *fp;
	Fragment *f;
	int pad;
	long zeros = 0;
	int i;
	
	fp = fopen(filename, "wb");
	if (!fp) return kRCX_FileError;
	
	// write header
	Write4(kRCXI_Signature, fp);
	Write2(kRCXI_CurrentVersion, fp);
	Write2((UShort)fFragments.GetLength(), fp);
	Write2((UShort)CountSymbols(), fp);
	Write2(0, fp);
	
	// write fragments
	for(i=0; i<fFragments.GetLength(); i++)
	{
		f = &fFragments[i];
		putc(f->fType, fp);	// this assumes that internal fragment types match the file format
		putc(f->fNumber, fp);
		Write2((UShort)f->fLength, fp);
		if (f->fLength)
			fwrite(f->fData, 1, (size_t)f->fLength, fp);
		pad = RCXI_PAD_BYTES(f->fLength);
		if (pad)
			fwrite(&zeros, (ULong) pad, 1, fp);
	}

	// write code symbols
	for(i=0; i<fFragments.GetLength(); i++)
	{
		f = &fFragments[i];
		if (f->fName)
			WriteSymbol(f->fType, f->fNumber, f->fName, fp); // this assumes that internal fragment types match the file format	
	}
	
	// write var symbols
	for(i=0; i<kRCX_MaxVars; i++)
	{
		if (fVarNames[i])
			WriteSymbol(kRCXI_VarSymbol, i, fVarNames[i], fp);
	}
	
	fclose(fp);
	return kRCX_OK;
}


int RCX_Image::CountSymbols() const
{
	int count = 0;
	int i;
	
	for(i=0; i<fFragments.GetLength(); ++i)
		if (fFragments[i].fName) ++count;
	
	for(i=0; i<kRCX_MaxVars; ++i)
		if (fVarNames[i]) ++count;
	
	return count;
}


RCX_Image::Fragment::Fragment()
{
	fData = nil;
	fLength = 0;
	fName = nil;
}


RCX_Image::Fragment::~Fragment()
{
	delete [] fName;
	delete [] fData;
}


void Write4(ULong d, FILE *fp)
{
	putc((UByte)d, fp);
	putc((UByte)(d>>8), fp);
	putc((UByte)(d>>16), fp);
	putc((UByte)(d>>24), fp);
}


void Write2(UShort d, FILE *fp)
{
	putc(d, fp);
	putc(d>>8, fp);
}


void WriteSymbol(UByte type, UByte index, const char *name, FILE *fp)
{
	int length = name ? strlen(name)+1 : 0;
	putc(type, fp);
	putc(index, fp);
	putc((UByte)length, fp);
	putc(0, fp);

	if (length) fwrite(name, length, 1, fp);
}


ULong Read4(FILE *fp)
{
	ULong d;
	
	d = (UByte)getc(fp);
	d += (UByte)getc(fp) << 8;
	d += (UByte)getc(fp) << 16;
	d += (UByte)getc(fp) << 24;
	
	return d;
}

UShort Read2(FILE *fp)
{
	UShort d;
	
	d = (UByte)getc(fp);
	d += (UByte)getc(fp) << 8;
	
	return d;
}

static char *NewString(const char *s)
{
	if (!s) return nil;
	
	char *newString = new char[strlen(s) + 1];
	strcpy(newString, s);
	return newString;
}
