/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */


#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include "RCX_Image.h"
#include "RCX_Link.h"
#include "RCX_Cmd.h"

#define VERSION "1.0 b4"

void PrintUsage();
int ProcessCommandLine(int argc, char **argv);
void PrintError(RCX_Result error, const char *filename);

const char *gFilename = nil;
Bool gTargetCyberMaster = false;
const char *gSerialPort = nil;
RCX_Link gLink;
Bool gAutoRun = false;

static char *sTestArgs[]={"rcxdl", "-r",  "::test.rcx"};


int main(int argc, char **argv)
{
	RCX_Image image;
	RCX_Result result;
	RCX_Cmd cmd;
	
	if (argc == 0)
	{
		argv = sTestArgs;
		argc = sizeof(sTestArgs) / sizeof(char *);
	}

	if (!ProcessCommandLine(argc, argv))
	{
		PrintUsage();
		return -1;
	}

	result = image.Read(gFilename);
	if (result < 0)
	{
		PrintError(result, gFilename);
		return -1;
	}

	fprintf(stderr, "Downloading...");
	result = gLink.Open(gSerialPort);

	if (result == 0)
	{
		result = image.Download(&gLink);
	}

	if (gAutoRun)
	{
		result = gLink.Send(cmd.Set(kRCX_StartTaskOp, 0));
	}

	gLink.Close();
	
	if (result < 0)
	{
		fprintf(stderr, "error\n");
		PrintError(result, gFilename);
		return -1;
	}
	else
		fprintf(stderr, "complete\n");
	
	return 0;
}


void PrintError(RCX_Result error, const char *filename)
{
	const char *targetName = gTargetCyberMaster ? "CyberMaster" : "RCX";
	if (error >= 0) return;
	
	switch(error)
	{
		case kRCX_OpenSerialError:
			fprintf(stderr, "Could not open serial port\n");
			break;
		case kRCX_IREchoError:
			fprintf(stderr, "Problem talking to IR device\n");
			break;
		case kRCX_ReplyError:
			fprintf(stderr, "No reply from %s\n", targetName);
			break;
		case kRCX_MemFullError:
			fprintf(stderr, "Not enough free memory in %s to download program\n", targetName);
			break; 
		case kRCX_FileError:
			fprintf(stderr, "Error: Could not open \'%s\'\n", filename);
			break;
		case kRCX_FormatError:
			fprintf(stderr, "Error: \'%s\' is not an RCXI file\n", filename);
			break;
		default:
			fprintf(stderr, "Error #%d\n", -error);
			break;
	}
}

void PrintUsage()
{
	fprintf(stderr,"rcxdl version "VERSION" (built "__DATE__"," __TIME__")\n");
	fprintf(stderr,"     written by Dave Baum\n");
	fprintf(stderr,"Usage: rcxdl [-c] [-s serial_port] [-r] filename\n");
	fprintf(stderr,"    -c: download to CyberMaster\n");
	fprintf(stderr,"    -s name: specify serial port\n");
	fprintf(stderr,"    -r: run program after download\n");
}


int ProcessCommandLine(int argc, char **argv)
{
	int v;
	
	for(int i=1; i<argc; i++)
	{
		if (argv[i][0]=='-')
		{
			switch(tolower(argv[i][1]))
			{
				case 's':
					i++;
					if (i==argc || gSerialPort) return 0;
					gSerialPort = argv[i];
					break;
				case 't':
					i++;
					if (i==argc) return 0;
					v = atoi(argv[i]);
					gLink.SetRxTimeout(v);
					break;
				case 'v':
					gLink.SetVerbose(true);
					break;
				case 'c':
					gTargetCyberMaster = true;
					gLink.SetTarget(RCX_Link::kTarget_CyberMaster);
					break;
				case 'r':
					gAutoRun = true;
					break;
				default:
					return 0;
			}
		}
		else if (!gFilename)
			gFilename = argv[i];
		else
			return 0;
	}
	
	return gFilename ? true : false;
}
