(*
 *                     The OCaml-gtk interface
 *
 * Copyright (c) 1997-99    David Monniaux & Pascal Cuoq
 *
 * This file is distributed under the conditions described in
 * the file LICENSE.  
 *)

(* $Id: tut_cat.ml,v 1.13 2000/01/14 09:51:33 sven Exp $ *)

module MsgEnglish = 
  struct
let input = "Input" 
let output = "Output"
let history = "History"
let menu_file = "File"
let menu_quit = "Exit"
let menu_caml = "Caml"
let menu_send = "Send to Caml (C-Return)"
let menu_interrupt = "Interrupt"
  end

module MsgFrancais = 
  struct
let input = "Entres" 
let output = "Rsultats"
let history = "Historique"
let menu_file = "Fichier"
let menu_quit = "Quitter"
let menu_caml = "Caml"
let menu_send = "Envoyer  Caml (C-Return)"
let menu_interrupt = "Interrompre"
  end
    
open GtkObj
open GtkEasy.Layout
open Unix

module Msg = MsgEnglish

(***************************************************************)
(***      interface with the ocaml toplevel                  ***)
(***************************************************************)
let (from_caml,to_caml) as caml_process = Pty.open_process "TERM=dumb ocaml" 
let _ = set_nonblock (descr_of_in_channel from_caml) 

(***************************************************************)
(*** the text widgets used for interaction with the user     ***)
(***************************************************************)    
let output = text_new () 
let history = text_new() 
let input = text_new () 
let _ = input #set_editable true 
(* input is the text widget where the user types caml phrases.  
   output and history, on the other hand, are used to only to display
   text, not to input it *)
                                    

(***************************************************************)
(*** get input from the user and send it to the caml process ***)
(***************************************************************)

(* this function reads what has been typed in the input widget,
   and sends it to the toplevel *)
let input_handler () =
  let t = (input #get_text) in
  let tt = t ^ "\n" in 
  history #insert None tt;
(*    input #set_text ""; better : *)
  input #select_region 0 (-1);
  output_string to_caml tt;
  flush to_caml

(* Now, we connect input_handler to ensure it will be called
   everytime C-Enter is pressed in the input widget *)
let _ = 
(*
  input #connect_key_press 
    ( function input -> function ptr -> 
      let keyval = (Gdk.Event.Extract.key_keyval ptr) in
      let state  = (Gdk.Event.Extract.key_state  ptr) in
*)
  input #connect_key_press 
    ( function input -> function keyval -> function state ->
      if keyval = 65293 (* Return *)
          & (state land 4) <> 0 (* Ctrl *)
      then input_handler () ;
      true ) 

(***************************************************************)
(***    get replies from the caml process and display them   ***)
(***************************************************************)
let rec get_from_caml = 
  let buffer = String.create 200 in
  fun () ->
    try 
      while 
        let r = (Pervasives.input from_caml buffer 0 199) in
        String.set buffer r '\000' ;
        output #insert None buffer ;          
        (r<>0)
      do () done
    with 
      End_of_file -> exit 0 
    | Unix_error(EAGAIN,_,_) 
    | Unix_error(EWOULDBLOCK,_,_) 
    | Sys_error _ -> ()
    | Sys_blocked_io -> ()

(***************************************************************)
(***                  build the interface                    ***)
(***************************************************************)
let cat_quit () =
  let _ = Pty.close_process caml_process (* we ignore the return status *)
  in
  Gtk.main_quit()

(* This little function puts the widget [widget] in a 
   frame, with title [title] *)
let with_title widget title = 
  Frame( title, Gtk.SHADOW_NONE, Widget(widget :> widget) )

open GtkEasy.Menu
let menu = 
  [ Submenu( GtkEasy.Label Msg.menu_file,
	    [ Item( GtkEasy.Label Msg.menu_quit, cat_quit ) ] ) ;
    Submenu( GtkEasy.Label Msg.menu_caml,
            [ Item( GtkEasy.Label Msg.menu_send, input_handler );
              Separator;
              Item( GtkEasy.Label Msg.menu_interrupt,
	           fun () -> output_string to_caml "\003";
	             flush to_caml ) ;
            ]               ) 
 ] 
  
let menu_bar = make_menu_bar menu
	    
let structure = 
  Paned( Horiz,
        Box( Vert,
            [ Widget (menu_bar :> widget) , fixed_1 ;
	      Paned( Vert,
	            with_title history Msg.history,
	            with_title input Msg.input ) , fill_1  ] ),
 	with_title output Msg.output )
  
let window = make_window_from_structure structure "Caml Toplevel";;

let _ = 
  output #realize ;
  history #realize ;
    
  output #set_usize 290 40  ;
  history #set_usize 290 240 ;
  input #set_usize 290 90 ;

  ignore (window #connect_delete_event
    (fun () -> Gtk.main_quit(); false));
  (* The handler installed with the method [#connect_delete_event]
       is called when the user tries to close the window.  *)
  window #show;
    
  ignore (Gtk.timeout_add 100
    (fun _ -> get_from_caml () ; true));

  GtkThr.main ()
  (* And at last enter the event loop. *)



      






