open Gdk
open GtkObj
open GtkEasy
open GtkEasy.Layout
open GtkEasy.Menu
open Gdk.Event
open Gdk.Event.Extract

(* some utility function ... *)
let soi = string_of_int

(* A text widget to make some outputs ... *)

let text =text_new ()
let mesg s = 
(
  text#freeze ();
  text#insert_text (s^"\n") ((String.length s)+1) 0;
  text#thaw ();
)

(* A menu *)

let mesgfunc t = function () -> mesg t
let menu_def =
[ 
  Submenu (GtkEasy.Label "GtkGraphics",
  [
    Item (GtkEasy.Label "About", mesgfunc
    		("This is the GtkGraphics mainwindow,\n"^
		"A Graphics compatible mlgtk module."));
    Item (GtkEasy.Label "Close", Gtk.main_quit)
  ]);
  Submenu (GtkEasy.Label "Misc",
  [
    Item (GtkEasy.Label "Options", mesgfunc "Not yet implemented.");
  ]);
  Item (GtkEasy.Label "Help", mesgfunc "No help available yet.");
]
let menu_bar = make_menu_bar menu_def

(* Drawing area definition ... *)

(* Backing pixmap *)

let pixmap = ref None
exception No_pixmap

let get_pixmap () = match !pixmap with 
	| None -> raise No_pixmap
	| Some p -> p
let free_pixmap () = match !pixmap with
	| None -> ()
	| Some p -> Gdk.pixmap_unref p; pixmap := None; ()
let set_pixmap p = match !pixmap with
	| None -> pixmap := Some p; p
	| Some p' -> Gdk.pixmap_unref p'; pixmap := Some p; p

(* Widget DrawingArea, ou nous allons pouvoir dessiner *)
let drawingarea = drawing_area_new ()

(* L'evenement configure a lieu au debut, et a chaque fois que le widget
   change de taille *)
let drawingarea_configure (x:Gtk.Unsafe.gtkArg list) =
(
	let da = drawingarea#get_gtkobject in
	let w = Gtk.Unsafe.window_of da
	(* the Window in which to copy the pixmap *)
	and black_gc = let st = Gtk.Unsafe.style_of da in st.Gtk.black_gc
	(* The black Graphic Context to fill the empty pixmap *)
	and allocation = Gtk.Unsafe.allocation_of da
	in let width = allocation.Gtk.width
	and height = allocation.Gtk.height
	(* We get the new size of the drawingarea widget after
	   the configure event *)
	in let p = set_pixmap (Gdk.pixmap_new w width height (-1))
	(* We Create a new pixmap of the new size of the drawingarea widget *)
	in let d = Gdk.drawable_from_pixmap p
	(* We get the drawable associated with the pixmap *)
	in
	(
		Gdk.draw_rectangle d black_gc true 0 0 width height;
		(* So we can draw on it ... *)
		mesg "configure ...";
		Gtk.Unsafe.Bool true
		(* We don't want to emmit a destroy event ... *)
	)
)

(* L'evenement expose a lieu apres chaque evenement configure,
   et chaque fois que le widget etait cache et reapparait. *)
let drawingarea_expose (x:Gtk.Unsafe.gtkArg list) =
(
	let da = drawingarea#get_gtkobject in
	let w = Gtk.Unsafe.window_of da
	(* the Window in which to draw the backing pixmap *)
	and fg_gc = let st = Gtk.Unsafe.style_of da
		in st.Gtk.fg_gc.(Gtk.Unsafe.state_of da)
	(* The foreground Graphic Context to draw the backing pixmap *)
	in let d = Gdk.drawable_from_window w
	(* We get the drawable associated with the window *)
	and p = get_pixmap ()
	(* We get the backing pixmap *)
	and (x, y, width, height) =
	( function
		| Gtk.Unsafe.Pointer p ->
			expose_x p, expose_y p, expose_w p, expose_h p
		| _ -> (0, 0, 0, 0)
	) (List.hd x)
	(* We get the the rectangle that got exposed *)
	(* Note : we should put it in a gdkRectangle ... *)
	in
	(
		Gdk.draw_pixmap d fg_gc p x y x y width height;
		(* We draw the backing pixmap on the window *)
		mesg ("expose ..."^(soi x)^","^(soi y)^":"^(soi width)^","^(soi
				height));
		Gtk.Unsafe.Bool false
		(* Why do we send false ? if i understood correctly,
		   this will make gtk to send a destroy event, ... ? *)
	)
)

(* We draw a small square arround the clicked point *)
let draw_point x y =
	let da = drawingarea#get_gtkobject
	in let st = Gtk.Unsafe.style_of da
	in let white = st.Gtk.white_gc
	(* The white Graphic Context ... *)
	and p = get_pixmap ()
	in let d = Gdk.drawable_from_pixmap p
	(* The drawable where to drive ... *)
	and rect = {x=x-5; y=y-5; width=10; height=10}
	(* The rectangle to draw, and later to refresh *)
	in
    (
	  Gdk.draw_rectangle d white true rect.x rect.y rect.width rect.height;
	  (* We draw the rectangle in the backing pixmap *)
	  Gtk.Unsafe.widget_draw da rect
	  (* And ask the widget to refresh the drawen zone. *)
	)

(* We try to get the color of a pixel. *)
(* Note : Not much success here, perhaps we should look directly
   the backing pixmap ... *)
let point_color x y =
	(* This is not very useful, becasue what is contained in pixel is not the color
	    of the pixel but some strange data .... same as pixel value in *)
	let da = drawingarea#get_gtkobject
	in let w = Gtk.Unsafe.window_of da
	in let i = Gdk.image_get w 0 0 (x+1) (y+1)
	(* We get the image associated with the window. *)
	in let pixel = Gdk.image_get_pixel i x y
	(* To get the pixel (x,y) of the screen *)
	(* Note : What is the nature of the pixel returned ?
	   i was said that it is the same as the pixel value in the
	   GdkColor type, but in C they don't have the same type.
	   Also in our case, it reports all the same values ... *)
	in mesg ("Pixel ("^(soi x)^", "^(soi y)^") : "^(soi pixel))

(* Button_press event handler *)
let button_pressed b x y =
(
	mesg ("Bouton "^(soi b)^" pressed at coordinates ("^(soi x)^
		","^(soi y)^") ...");
	match b with
	| 1 -> draw_point x y
	| 2 -> point_color x y
	| _ -> ()
)

(* Button pressed event, we call the button_pressed function here *)
let drawingarea_button_pressed (x:Gtk.Unsafe.gtkArg list) =
(
	let (b, x, y) =
	( function
	  | Gtk.Unsafe.Pointer p -> button_button p, button_x p, button_y p
		| _ -> (-1, 0, 0)
	) (List.hd x)
	(* We get the button that was pressed, as well as
	the coordinates of the click *)
	in button_pressed b x y;
	Gtk.Unsafe.Bool true
)

let _ = drawingarea#set_events [BUTTON_PRESS_MASK; EXPOSURE_MASK]
(* We decide to catch the BUTTON_PRESS and EXPOSURE events *)
let _ = Gtk.Unsafe.signal_connect drawingarea#get_gtkobject
	"button_press_event" drawingarea_button_pressed
let _ = Gtk.Unsafe.signal_connect drawingarea#get_gtkobject
	"configure_event" drawingarea_configure
let _ = Gtk.Unsafe.signal_connect drawingarea#get_gtkobject
	"expose_event" drawingarea_expose
(* And associate our previously defined handlers to our widget *)
let _ = drawingarea#size 300 200
(* We change the size of our widget *)

(* Toplevel stuff ... *)

let structure =
	let me =  Widget (menu_bar:>widget),
		{expand=false; fill=false; padding=1}	
	and da =  Widget (drawingarea:>widget),
		{expand=true; fill=true; padding=1}	
	and t = Widget (text:>widget),
		{expand=false; fill=false; padding=1}
	in Box (Vert, [me; da; t])
(* We put everything in a vertical box *)

let window = make_window_from_structure structure "Toplevel Window" ;;
(* We create the toplevel window ... *)

let _ = window #connect_delete_event( fun () -> Gtk.main_quit(); false )
(* The handler installed with the method [#connect_delete_event]
   is called when the user tries to close the window.  *)

(* The Graphics compatible stuff *)

exception Graphic_failure of string

let open_graph "" = window #show

let close_graph () = Gtk.main_quit ()

let clear_graph () = ()

let size_x () = 0

let size_y () = 0
