/* $Header: /cvs/gnome/gIDE/src/gI_debug.c,v 1.3 1999/12/05 19:10:48 jpr Exp $ */
/* gIDE
 * Copyright (C) 1998-2000 Steffen Kern
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <time.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>
#include "structs.h"
#include "gI_document.h"
#include "gI_file.h"
#include "gI_common.h"
#include "gI_project.h"
#include "gI_compile.h"
#include "gI_debug.h"
#include "gI_menus.h"
#include "gI_window.h"

/* externs */
extern gI_window *main_window;
extern gI_config *cfg;

static glong show_output = 1;	/* set this to 0 and no gdb output */

/* local structs */
typedef struct _item_data item_data;
struct _item_data
{
	gchar *expr;
	gchar *item;
	glong ptr;
};

/* local prototypes */
static void debug_start_gdb( gI_debug_window *dw );
static glong send_gdb_cmd( gI_debug_window *dw, gchar *cmd, glong show );
static void debug_stop_gdb( gI_debug_window *dw );
static gint get_child_pid( gI_debug_window *dw );
static void debug_run( GtkWidget *widget, gI_debug_window *dw );
static gint check_for_data(void);
static void watch_list_select( GtkWidget *widget, gint row, gint column, GdkEventKey *bevent, gI_watch_window *ww );
static void set_expression( GtkWidget *widget, entry_data *ed );
static void debug_brkpnt_win( GtkWidget *widget, gI_debug_window *dw );
static void add_brkpnt( gI_debug_window *dw, gchar *brkpnt );
static void brkpnt_list_select( GtkWidget *widget, gint row, gint column, GdkEventButton *bevent, gI_brkpnt_window *bpw );
static void install_debug_popup( gI_debug_window *dw );
static gint popup_debug_menu( GtkWidget *widget, GdkEvent *ev );
static void debug_kill( GtkWidget *widget, gI_debug_window *dw );
static void debug_step_out( GtkWidget *widget, gI_debug_window *dw );
static void debug_run_to_cursor( GtkWidget *widget, gI_debug_window *dw );
static void debug_backtrace( GtkWidget *widget, gI_debug_window *dw );
static void debug_goto_frame( GtkWidget *widget, gI_debug_window *dw );
static void add_frame( gI_debug_window *dw, gchar *frame );
static gint bt_list_select( GtkWidget *widget, GdkEventButton *event );
static void install_bt_popup( gI_bt_window *bt );
static void debug_bt_view_source( GtkWidget *widget, gI_debug_window *dw );
static void debug_brkpnt_clear( GtkWidget *widget, gI_debug_window *dw );
static void install_bpw_popup( gI_brkpnt_window *bpw );
static gint brkpnt_event_handler( GtkWidget *widget, GdkEventButton *event );
static void debug_bpw_clear_bp( GtkWidget *widget, gpointer data );
static void debug_bpw_view_source( GtkWidget *widget, gpointer data );
static void install_watch_popup( gI_watch_window *ww );
static gint watch_event_handler( GtkWidget *widget, GdkEventButton *event, gI_watch_window *ww );
static void debug_ww_eval_expr( GtkWidget *widget, gpointer data );
static void debug_ww_set_expr( GtkWidget *widget, gpointer data );
static void show_evalexpr( gchar *expr, gchar *data );
static void evalwin_item_bp_handler( GtkWidget *widget, GdkEventButton *event, item_data *id );
static void debug_expressions( GtkWidget *widget, gI_debug_window *dw );
static void debug_exceptions( GtkWidget *widget, gI_debug_window *dw );
static void debug_file( GtkWidget *widget, gpointer data );


/*
 ---------------------------------------------------------------------
     Function: debug_debug()
     Desc: Callback-Function /Debug/Start Debug
 ---------------------------------------------------------------------
*/

/* too long function !!, please simplify it */

void debug_debug( GtkWidget *widget, gpointer data )
{
    gI_document *current;
    gchar filename[STRLEN];
    gchar str[STRLEN];
    gchar *ptr;
    gI_project *project;
    FILE *tmpfile;
    glong i,length;
    glong exec_mod, src_mod;
	gchar cmdstr[MAXLEN];
    static glong recursed = 0;

    current = gI_document_get_current( main_window );

    if( !current )
		return;

	if( !current->window->debug_window )
	{
		current->window->debug_window = gI_debug_window_new( current );
		gtk_widget_show( current->window->debug_window->window );
		debug_start_gdb( current->window->debug_window );
	}
	else
	{
		/* only one debug window */
		return;
	}

    /* special handling for projects,
	project handling has priority, maybe we need a pref. switch to
		decide this in the future... */
	project = gI_project_get_current();
    if( project && project->mtarget && !isempty( project->mtarget ) )
    {
        strcpy( filename, project->mtarget );
        if( !file_exist( filename ) )
        {
	    gchar *mtarget;

	    if( isempty( project->prjroot ) )
	    {
            	g_snprintf( str, STRLEN, _("\n    Can't find executable '%s'!    \n"), filename );
            	gI_error_dialog( str);
            	return;
	    }

	    mtarget = g_strconcat( project->prjroot, "/", filename, NULL );

	    if( !file_exist( mtarget ) )
	    {
            	g_snprintf( str, STRLEN, _("\n    Can't find executable '%s'!    \n"), mtarget );
            	gI_error_dialog( str);
		g_free( mtarget );
            	return;
	   }
	   else
	   {
		strcpy( filename, mtarget );
		g_free( mtarget );
	   }
        } 
    } 
    else
    {
    	if( !current->filename )
    	{
		    /* if theres no filename and the changed flag is not set, your
		       text widget is empty! */
			if( !current->changed )
				return;
	
		    /* create a temp filename, save the current text widget content,
		       try to compile it to a exec. and try to run it */	

		    strcpy( filename, tempnam( cfg->tmpdir, "gide_debug_" ) );  
		    strcat( filename, ".c" ); 
	    	tmpfile = fopen( filename, "w" );
		    if( !tmpfile )
		    {
			gI_error_dialog( _("Unable to Create the Temponary File !") );
	        	return;
		    }	

		    length = gI_text_get_length( current->text );

		    for(i=1;i<=length;i++)
	    	    fputc( gtk_editable_get_chars( GTK_EDITABLE( current->text ), i-1, i )[0], tmpfile );
	
		    fclose( tmpfile );

		    current->filename = g_strdup( filename );

		    compile_compile( widget, (gpointer) 1 );
	
		    debug_debug( widget, data );
	
		    /* let the compiled stuff get started (is 1sec enough??)...*/
		    sleep( 1 );
 
		    g_free( current->filename );
	    	    current->filename = NULL;

		    /* remove the temp. files */
		    /*remove( filename );*/	/* the temp. file does not removed!! */
        
	    	    ptr = strrchr( filename, '.' );
        	    if( ptr )
	   	    {
			*ptr = '\0';
			remove( filename );
	    	    }

		    return;
       	}

        /* check if executable file exists */
        strcpy( filename, current->filename );
        ptr = strrchr( filename, '.' );
        if( ptr )
	{
		    *ptr = '\0';
	    	    if( !file_exist( filename ) )
		    {
			/* if executable does not exist, try to compile the current
		   	   file to a executable and try to run it! */
			if( !recursed )
			{
			    compile_compile( widget, (gpointer) 1 );

			    recursed = 1;

			    debug_debug( widget, data );

			    recursed = 0;

			    return;
			}

			g_snprintf( str, STRLEN, _("\n    Can't find executable '%s'!    \n"), filename );
	    	        gI_error_dialog( str);
	        	return;
        	}	
	    	else
	    	{	
			exec_mod = get_last_mod( filename ); 
		
			strcpy( str, current->filename );
			src_mod = get_last_mod( str );
	        if( src_mod > exec_mod )
            {
                 gI_error_dialog( _("Source File is newer than Executable File!\n(maybe you should recompile it)") );
            }
		}	
	}
    }

	/* process gtk stuff */
	check_for_data();

	/* set prompt */
	send_gdb_cmd( current->window->debug_window, "set prompt " GDB_PROMPT, TRUE );

	/* set confirm off */
	send_gdb_cmd( current->window->debug_window, "set confirm off", TRUE );

	/* set limit to 50 gchars */
	send_gdb_cmd( current->window->debug_window, "set print elements 50", TRUE );

	/* stop on NULL */
	send_gdb_cmd( current->window->debug_window, "set print null-stop", TRUE );

	/* load executable */
	sprintf( cmdstr, "file %s", filename );
	send_gdb_cmd( current->window->debug_window, cmdstr, TRUE );
}


/*
 ---------------------------------------------------------------------
     Function: send_gdb_cmd()
     Desc: Queue/Send Command to GDB 
 ---------------------------------------------------------------------
*/
static glong send_gdb_cmd( gI_debug_window *dw, gchar *cmd, glong show )
{
	GList *queue;
	gI_gdbcmd *gdbcmd;
	gI_gdbcmd *newcmd;

	if( !dw )	/* since we decided that there can be only one debug window.. */
		dw = main_window->debug_window;
	if( !dw )
		return( 0 );

	if( dw->pid == -1 )
		return( 0 );

	if( isempty( cmd ) )
	{
		strcpy( cmd, "\n" );
	}

	/* add new cmd to queue */
	newcmd = gI_gdbcmd_new(); 
	newcmd->cmd = g_strdup( cmd );
	newcmd->show = show;
	dw->cmd_queue = g_list_append( dw->cmd_queue, (gpointer) newcmd );

	/* process command queue */
	queue = dw->cmd_queue;
	while( queue != NULL )
	{
		if( !queue )
		{
			/*queue = queue->next;
			continue;*/
			break;
		}

		gdbcmd = (gI_gdbcmd *) queue->data;
		if( !gdbcmd )
		{
			queue = queue->next;
			continue;
		}

		if( !gdbcmd->cmd )
		{
			queue = queue->next;
			continue;
		}

		if( gdbcmd->status == N_GDBCMD_DONE )
		{
			/* we'll free the complete list, on quit */
			/*g_list_remove( dw->cmd_queue, (gpointer) gdbcmd );
			  g_free( gdbcmd );*/ /* is the memory already freed at this point? */
			  /*gdbcmd = NULL;*/
			queue = queue->next;
			continue;
		}
		else
		{
			while( dw->state != N_GDB_IDLE && dw->state != N_GDB_BREAK )
			{
				gtk_main_iteration_do( TRUE );	
			}

			/* ...or process queue */
			if( gdbcmd->cmd[0] == '\n' )
			{
				write( dw->fh_in, "\n", 1 );
				
				if( gdbcmd->show )
				{	
					gtk_text_insert( GTK_TEXT( dw->text ), NULL, NULL, NULL, "\n", 1 );	
				}
			}
			else
			{	
				if( write( dw->fh_in, gdbcmd->cmd, strlen(gdbcmd->cmd) ) > 0 )
				{
					write( dw->fh_in, "\n", 1 );

					if( gdbcmd->show )
					{
						gtk_text_insert( GTK_TEXT( dw->text ), NULL, NULL, NULL, gdbcmd->cmd, strlen(gdbcmd->cmd) );	
						gtk_text_insert( GTK_TEXT( dw->text ), NULL, NULL, NULL, "\n", 1 );	
					}

					gdbcmd->status = N_GDBCMD_DONE;
				}

				/* set state to running */
				dw->state = N_GDB_RUNNING;

				/* bsd 4.3 function, hope thats works for all */
				usleep(500);
			}
		}
		queue = queue->next;
	} 

	return( 1 );
}


static void add_expression( gI_debug_window *dw, gchar *locale )
{
	gchar *row[3];
	gchar *ptr;
	gchar *dup, *dup2, *dup3 = NULL;
	gchar *sptr;

	dup = g_strdup( locale );
	ptr = strchr( dup, '=' );
	ptr--;
	*ptr = '\0';
	row[0] = dup;

	dup2 = g_strdup( locale );
	ptr = strchr( dup2, '=' );
	ptr += 2;

	sptr = ptr;
	if( *sptr == '(' )
	{
		dup3 = g_strdup( ++sptr );

		sptr = dup3;	
		while( *sptr != ')' )
			sptr++;

		*sptr = '\0';
		ptr = (sptr + 2);

		row[1] = dup3;
	}
	else
	{
		row[1] = NULL;
	}
	row[2] = ptr;

	gtk_clist_append( GTK_CLIST( dw->watch_window->list ), row );  

	g_free( dup );
	g_free( dup2 );
	if( dup3 != NULL )
		g_free( dup3 );
}


/*
 ---------------------------------------------------------------------
     Function: process_debug_output()
     Desc: Process Debugger Output 
 ---------------------------------------------------------------------
*/
static void process_debug_output( gI_debug_window *dw, gchar *outmsg )
{
	gchar *ptr, *retptr;
	gchar *msg;
	gchar *data[6]; 
	gchar *file;
	glong line;
	glong old_state;
	gchar locale[MAXLEN];
	gchar brkpnt[MAXLEN];
	gchar frame[MAXLEN];
	glong end = 0;
	gchar *sptr;
	glong found_empty = -1, rowcnt;
	gchar *type, *expr;
	gchar cmdstr[100];
	gchar *error;

	msg = g_strdup( outmsg );

	/* backup last state */
	old_state = dw->state;

	/* set state */
	gI_debug_window_parse_state( dw, msg );

	if( dw->state == N_GDB_RUNNING )
		gtk_widget_set_sensitive( dw->entry, TRUE );
	else
		gtk_widget_set_sensitive( dw->entry, FALSE );

	/* process the debug stuff (jump to correct lines, etc.) */
	if( (ptr = strstr( msg, "\032\032" )) )
	{
		ptr += 2;

		retptr = strchr( ptr, '\n' );
		if( retptr )
			*retptr = '\0';
		
		SK_GetFields( ptr, data, ':' );

		if( atoi( data[0] ) != 0 )
		{
			file = data[1];
			line = atoi( data[2] );

			if( check_file_open( file ) )
			{	/* if file is already opened, just switch the notebook page */
				goto_file( file );
			}
			else
			{	/* ...open it */
				file_open_by_name( main_window, file ); 
			}
		
			/* scroll to the line */	
			goto_line( line );
		} 

		/* can we be sure that there are not more than 2 fields
		 * to free???? */
		g_free( data[0] ); g_free( data[1] ); g_free( data[2] );

		/* break'd, now we request information about locals
		   and maybe the break- and watchpoints!? */
		send_gdb_cmd( dw, "info locals", TRUE );
		dw->wait_for = N_WAIT_EXPRESSIONS;

		if( dw->brkpnt_window && GTK_WIDGET_VISIBLE( dw->brkpnt_window ) )
		{
			/* update breakpoints */
			send_gdb_cmd( dw, "info breakpoints", TRUE );
			dw->wait_for = N_WAIT_BREAKPOINTS;
		}

		/* should we update the backtrace window here, too? */

		g_free( msg );
		return;
	}

	if( dw->wait_for == N_WAIT_EXPRESSIONS )
	{
		if( dw->watch_window )
		{
			gtk_clist_clear( GTK_CLIST( dw->watch_window->list ) );
		}

		ptr = &msg[0];
		if( !strncmp( ptr, "No locals", 9 ) || !strncmp( ptr, "No symbol table", 15 ) || !strncmp( ptr, "No frame selected", 17 ) || !strncmp( ptr, "The program is not being run.", 29) )
		{
			dw->wait_for = N_WAIT_NOTHING;
			g_free( msg );
			return;	
		}

		/* new watch window */
		if( dw->watch_window == NULL )
			dw->watch_window = gI_watch_window_new();

		if( !strstr( ptr, GDB_PROMPT ) )
		{
			g_warning( "parsing error: endless loop?!" );
			dw->wait_for = N_WAIT_NOTHING;
			g_free( msg );
			return;
		}

		gtk_clist_freeze( GTK_CLIST( dw->watch_window->list ) );		

		while( strncmp( ptr, GDB_PROMPT, strlen(GDB_PROMPT) ) )
		{	/* this may get an endless loop */
			retptr = strchr( ptr, '\n' );
			retptr--;
			strncpy( locale, ptr, (retptr-(ptr-1)) ); 

			locale[(retptr-(ptr-1))] = '\0';		

			if( *ptr == ' ' )
			{
				ptr = retptr + 2;
				continue;
			}

			sptr = strchr( locale, '=' );
			if( !strncmp( sptr, "= {", 3 ) )
			{
				sptr = strstr( locale, "= {" );
				sptr += 2;
				*sptr = '\0';
				strcat( locale, "???" );
			}
			else
			{	
				if( strchr( locale, '{' ) || strchr( locale, '}' ) )
				{
					ptr = retptr + 2;
					continue;
				}
			}

			add_expression( dw, locale );

			ptr = retptr + 2;
		}
		dw->wait_for = N_WAIT_NOTHING;
		g_free( msg );

		/* get first empty types field of the list */
		found_empty = -1;

		for(rowcnt=0;
			rowcnt<GTK_CLIST( dw->watch_window->list )->rows;
			rowcnt++)
		{
			type = NULL;
			gtk_clist_get_text( GTK_CLIST( dw->watch_window->list ), rowcnt, 1, &type );
			if( isempty( type ) )
			{
				found_empty = rowcnt;
				break;
			}
		}

		if( found_empty == -1 )
		{
			gtk_clist_thaw( GTK_CLIST( dw->watch_window->list ) );		
			dw->wait_for = N_WAIT_NOTHING;
			return;
		}

		gtk_clist_get_text( GTK_CLIST( dw->watch_window->list ), found_empty, 0, &expr );

		g_snprintf( cmdstr, sizeof(cmdstr), "whatis %s", expr );

		send_gdb_cmd( dw, cmdstr, TRUE );
		dw->wait_for = N_WAIT_WHATIS;

		return;
	}

	if( dw->wait_for == N_WAIT_BREAKPOINTS )
	{
		if( dw->breakpoints )
			g_list_free( dw->breakpoints );
		dw->breakpoints = NULL;

		if( dw->brkpnt_window )
		{
			gtk_clist_clear( GTK_CLIST( dw->brkpnt_window->list ) );
		}

		if( !strncmp( msg, "No breakpoints or watchpoints", 29 ) )
		{
			dw->wait_for = N_WAIT_NOTHING;
			g_free( msg );
			return;	
		}

		if( !strncmp( msg, GDB_PROMPT, strlen(GDB_PROMPT) ) )
		{
			dw->wait_for = N_WAIT_NOTHING;
			g_free( msg );
			return;	
		}

		if( dw->brkpnt_window == NULL )
			dw->brkpnt_window = gI_brkpnt_window_new();
		
		/* skip first line */
		ptr = strchr( msg, '\n' );
		if( !ptr )
		{
			g_warning( "FATAL PARSING ERROR: (data: :%s:)\nFile: %s, Line: %d\nPlease report to sk@pn.org\n", msg, __FILE__, __LINE__ );
			return;
		}
		ptr++;

		while( strncmp( ptr, GDB_PROMPT, strlen(GDB_PROMPT) ) )
		{	/* this may get an endless loop */
			retptr = strchr( ptr, '\n' );
			if( !retptr )
				return;	

			if( isspace( *++retptr ) )	/* line break */
			{
				retptr--;
				*retptr = ' ';
				retptr = strchr( ptr, '\n' );
			} 
			retptr--;

			/*
			 * maybe it works, if we correct the pointer here, i don't know
			 * why it isnt correct sometimes, but it works with this work-
			 * around!
			 */
			while( *ptr != '\n' )
				ptr--;
			ptr++;

			strncpy( brkpnt, ptr, (retptr-(ptr-1)) ); 
			brkpnt[(retptr-(ptr-1))] = '\0';		

			add_brkpnt( dw, brkpnt );

			ptr = retptr + 2;
		}	
		dw->wait_for = N_WAIT_NOTHING;
		g_free( msg );
		return;
	}

	if( dw->wait_for == N_WAIT_BACKTRACE )
	{
		if( dw->bt_window )
		{
			gtk_clist_clear( GTK_CLIST( dw->bt_window->list ) );
		}

		if( !strncmp( msg, "No stack", 8 ) )
		{
			dw->wait_for = N_WAIT_NOTHING;
			g_free( msg );
			return;	
		}

		if( !dw->bt_window )
			dw->bt_window = gI_bt_window_new(); 

		ptr = &msg[0];

		while( (ptr = strchr( msg, '\n' )) )
			*ptr = ' ';

		ptr = &msg[0];

		ptr = strchr( ptr, '#' );
		ptr++;

		end = 0;	
	
		while( strncmp( ptr, GDB_PROMPT, strlen(GDB_PROMPT) ) )
		{	/* this may get an endless loop */
			retptr = strchr( ptr, '#' );
			if( !retptr && !end )
			{
				end = 1;
				retptr = strchr( ptr, '\0' );
				if( retptr )
					retptr--;	
			}

			if( !retptr )	/* end of backtrace reached */
			{
				dw->wait_for = N_WAIT_NOTHING;
				g_free( msg );
				return;	
			}

			strncpy( frame, ptr, (retptr-ptr) ); 
			frame[(retptr-(ptr))] = '\0';		

			add_frame( dw, frame );	

			ptr = retptr + 1;
		}	

		dw->wait_for = N_WAIT_NOTHING;
		g_free( msg );
		return;
	}

	if( dw->wait_for == N_WAIT_WHATIS )
	{
		/* get first empty types fields of the list */
		for(rowcnt=0;
			rowcnt<GTK_CLIST( dw->watch_window->list )->rows;
			rowcnt++)
		{
			type = NULL;
			gtk_clist_get_text( GTK_CLIST( dw->watch_window->list ), rowcnt, 1, &type );
			if( isempty( type ) )
			{
				found_empty = rowcnt;
				break;
			}
		}

		if( found_empty == -1 )
		{
			gtk_clist_thaw( GTK_CLIST( dw->watch_window->list ) );		
			dw->wait_for = N_WAIT_NOTHING;
			return;
		}	

		if( !strncmp( msg, "No symbol \"", 10 ) )
		{
			gtk_clist_set_text( GTK_CLIST( dw->watch_window->list ), found_empty, 1, "???" );
		}
		else
		{
			ptr = strchr( msg, '=' );
			if( ptr )
			{
				retptr = strchr( ptr, '\n' );
				*retptr = '\0';
				ptr += 2;

				/* strip '(' & ')' */	
				if( *ptr == '(' )
					ptr++;
				if( *--retptr == ')' )
					*retptr = '\0';
	
				gtk_clist_set_text( GTK_CLIST( dw->watch_window->list ), found_empty, 1, ptr );
			}
		}
		g_free( msg );

		found_empty = -1;

		/* get first empty types fields of the list */
		for(rowcnt=0;
			rowcnt<GTK_CLIST( dw->watch_window->list )->rows;
			rowcnt++)
		{
			type = NULL;
			gtk_clist_get_text( GTK_CLIST( dw->watch_window->list ), rowcnt, 1, &type );
			if( isempty( type ) )
			{
				found_empty = rowcnt;
				break;
			}
		}

		if( found_empty == -1 )
		{
			gtk_clist_thaw( GTK_CLIST( dw->watch_window->list ) );		
			dw->wait_for = N_WAIT_NOTHING;
			return;
		}

		gtk_clist_get_text( GTK_CLIST( dw->watch_window->list ), found_empty, 0, &expr );

		g_snprintf( cmdstr, sizeof(cmdstr), "whatis %s", expr );

		send_gdb_cmd( dw, cmdstr, TRUE );
		dw->wait_for = N_WAIT_WHATIS;

		return;
	}

	if( dw->wait_for == N_WAIT_EVALEXPR )
	{
		if( !strncmp( msg, "Cannot access memory at", 23 )
			|| !strncmp( msg, "Attempt to dereference a generic pointer", 40 )
			|| !strncmp( msg, "No symbol \"", 11 ) )
		{
			retptr = strchr( msg, '\n' );
			if( !retptr )
			{
				/* this should not happen! */
			}
			*retptr = '\0';

			error = g_strconcat( "\n    ", msg, "    \n", NULL );
			gI_error_dialog( error);
			g_free( error );

			dw->wait_for = N_WAIT_NOTHING;
			g_free( msg );
			return;		
		}

		ptr = strchr( msg, '=' ); 
		if( !ptr )
		{
			dw->wait_for = N_WAIT_NOTHING;
			g_free( msg );
			return;		
		}
		ptr += 2;

		retptr = strstr( msg, GDB_PROMPT );
		if( !retptr )
		{
			/* we have a problem! */
			dw->wait_for = N_WAIT_NOTHING;
			g_free( msg );
			return;		
		}
		*retptr = '\0';

		if( dw->watch_window->data != NULL )
		{	
			show_evalexpr( dw->watch_window->data, ptr );	

			g_free( dw->watch_window->data );
		}
		else
		{
			g_warning( "problems with data in watch-window!\n" );
		}
	
		dw->wait_for = N_WAIT_NOTHING;
		g_free( msg );
		return;		
	}

	/* needed for c++ support */
	if( dw->wait_for == N_WAIT_EXCEPTIONS )
	{
		/* TODO: implement parsing for exceptions stuff */
		dw->wait_for = N_WAIT_NOTHING;
		g_free( msg );
		return;		
	}	

	g_free( msg );
}


/*
 ---------------------------------------------------------------------
     Function: handle_debug_output()
     Desc: Input-handler function for the gdb output pipe 
 ---------------------------------------------------------------------
*/
static void handle_debug_output( gI_debug_window *dw, gint fh, GdkInputCondition condition )
{
	gchar buf[MAXLEN*9];	/* we need very very big buffer or gdk_events_wait()
						   segsegv's on large gdb/program output */
						/* another reason is that (at the moment), we need all
						   the expressions (or breakpoints) in one buffer */
	gint count;


	count = read( dw->fh_out, buf, sizeof(buf) );

	if( count <= 0 )
	{
		/* stop debugger */
		debug_stop_gdb( dw );

		/* set statusbar */
		gI_window_set_statusbar( gI_document_get_current( main_window ) );

		return;
	}

	buf[count] = '\0';

	if( count == (MAXLEN*9) )
	{
		/* buffer reached maximum size, can this happen?? */
		g_warning( "count reached maximum!" );
	}

	process_debug_output( dw, buf );

	if( !show_output )
		return;

	gtk_text_insert( GTK_TEXT( dw->text ), NULL, NULL, NULL, buf, count );
}


/*
 ---------------------------------------------------------------------
     Function: debug_set_menu()
     Desc: Set state of items in Debug-Menu (sensitive true/false) 
 ---------------------------------------------------------------------
*/
static void debug_set_menu( gI_debug_window *dw, glong debug )
{
	if( debug )
	{
		menus_set_sensitive( "/Debug/Stop  Debug", TRUE );
		menus_set_sensitive( "/Debug/Start Debug", FALSE );
		menus_set_sensitive( "/Debug/Set|Clear Breakpoint", TRUE );
		menus_set_sensitive( "/Debug/Break", TRUE );
		menus_set_sensitive( "/Debug/Step Into", TRUE );
		menus_set_sensitive( "/Debug/Step Over", TRUE );
		menus_set_sensitive( "/Debug/Attach to Process", TRUE );
	}
	else
	{
		menus_set_sensitive( "/Debug/Stop  Debug", FALSE );
		menus_set_sensitive( "/Debug/Start Debug", TRUE );
		menus_set_sensitive( "/Debug/Set|Clear Breakpoint", FALSE );
		menus_set_sensitive( "/Debug/Break", FALSE );
		menus_set_sensitive( "/Debug/Step Into", FALSE );
		menus_set_sensitive( "/Debug/Step Over", FALSE );
		menus_set_sensitive( "/Debug/Attach to Process", FALSE );
	}
}
 

/*
 ---------------------------------------------------------------------
     Function: debug_start_gdb()
     Desc: Start GDB (create socket, system gdbio, etc.) 
 ---------------------------------------------------------------------
*/
static void debug_start_gdb( gI_debug_window *dw )
{
	gchar sockname[90];
	gchar *execstr;
	gchar buf[10];
	struct sockaddr_un sock;
	gint fh;
	gint addrlen;
	/*gint bufsize = 9 * MAXLEN;*/
	gint ret;

	if( !dw )
	{
		return;	
	}

	/* generate socket name */
	g_snprintf( sockname, 90, "%s/gide_deb.%ld.%ld", cfg->tmpdir, time(NULL), (glong) getpid() );

	/* unlink existing files */
	remove( sockname );

	fh = socket( AF_UNIX, SOCK_STREAM, 0 );
	if( fh < 0 )
	{
		perror( "socket" );
		return;

	}


/* someone reported problems with the setsockopt stuff, so we disable it for now */
/*
	/- set socket options: recv buffer size -/
	if( setsockopt( fh, SOL_SOCKET, SO_RCVBUF, &bufsize, 0 ) != 0 )
	{
		perror( "setsockopt" );
		return;	
	} 

	/- set socket options: send buffer size -/
	if( setsockopt( fh, SOL_SOCKET, SO_SNDBUF, &bufsize, 0 ) != 0 )
	{
		perror( "setsockopt" );
		return;	
	} 
*/

	sock.sun_family = AF_UNIX;
	strcpy( sock.sun_path, sockname );

	/* bind name to socket */
	if( bind( fh, (struct sockaddr *)&sock, sizeof( struct sockaddr_un ) ) < 0 )
	{
		perror( "bind" );
		return;
	}

	/* set mode of socket */
	chmod( sockname, 00600 );

	/* queue for incoming connections */
	if( listen( fh, 0 ) < 0 )
	{
		perror( "listen" );
		return;
	}

	/* run gdbio */
	execstr = g_strconcat( "gdbio", " ", cfg->db, " ", sockname, " &", NULL ); 
	ret = system( execstr );
	g_free( execstr );	

	if( ret == 127 || ret == -1 )
	{
		gI_error_dialog( _("Unable to Start 'gdbio' !") );
		remove( sockname );
		return;
	}

	/* accept queued connection */
	addrlen = sizeof( struct sockaddr_un );
	fh = accept( fh, (struct sockaddr *)&sock, &addrlen );
	if( fh < 0 )
	{
		perror( "accept");
		remove( sockname );
		return;
	}

	/* socket is full-duplex, same handle for input and output */
	dw->fh_in = fh; 
	dw->fh_out = fh; 

	/* read pid of gdb */
	read( dw->fh_out, buf, sizeof(buf) );
	dw->pid = atoi( buf );

	/* save name of socket */	
	dw->sockname = g_strdup( sockname );

	/* install input handler */
	dw->input_id = gdk_input_add( dw->fh_out, GDK_INPUT_READ, (GdkInputFunction) handle_debug_output, (gpointer) dw );

	/* set debug menu */
	debug_set_menu( dw, TRUE );

	/* set statusbar */
	gI_window_set_statusbar( gI_document_get_current( main_window ) );
}


/*
 ---------------------------------------------------------------------
     Function: debug_stop_gdb()
     Desc: Stop GDB (close and unlink socket, etc.) 
 ---------------------------------------------------------------------
*/
static void debug_stop_gdb( gI_debug_window *dw )
{
	GList *tmp;

	if( dw->pid == -1 )
		return;

	/* remove input handler */
	gdk_input_remove( dw->input_id );

	/* send 'quit' command */
	/*send_gdb_cmd( dw, "quit" );*/	

	/* close socket */
	close( dw->fh_in );	

	/* process gtk stuff */
	while( gtk_events_pending() )
		gtk_main_iteration();

	/* kill it, if its still there... */
	kill( dw->pid, SIGKILL );

	/* display mesg */
	gI_ok_dialog( _("GDB process terminated!") );

	/* set pid in dw struct */
	dw->pid = -1;

	/* unlink fifos */
	remove( dw->sockname );
	g_free( dw->sockname );
	dw->sockname = NULL;

	/* set debug menu */
	debug_set_menu( dw, FALSE );

	/* we should free the command queue list here */
	tmp = g_list_first( dw->cmd_queue );
	while( tmp )
	{
		gI_gdbcmd *gdbcmd = (gI_gdbcmd *) tmp->data;
		if( gdbcmd )
		{
			g_free( gdbcmd->cmd );
		}	
		tmp = g_list_next( tmp );
	}

	g_list_free( dw->cmd_queue );
	dw->cmd_queue = NULL;
}


static void send_command_cb( GtkWidget *widget, gI_debug_window *dw )
{
	gchar *p;

	p = gtk_entry_get_text( GTK_ENTRY( widget ) );
	if( !p )
		return;

	/* if the state is running, we send the stuff directly to the debugger */
	if( dw->state == N_GDB_RUNNING )
	{
		write( dw->fh_in, p, strlen(p) );
		write( dw->fh_in, "\n", 1 );

		gtk_text_insert( GTK_TEXT( dw->text ), NULL, NULL, NULL, p, strlen(p) );	
		gtk_text_insert( GTK_TEXT( dw->text ), NULL, NULL, NULL, "\n", 1 );	

		gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		return;
	}
	
	send_gdb_cmd( dw, p, TRUE );	
	gtk_entry_set_text( GTK_ENTRY( widget ), "" );
}


static void debug_run_ok( GtkWidget *widget, GtkWidget *entry )
{
	gchar cmdstr[STRLEN];
	gI_document *current;
	gI_debug_window *dw;
	gchar *run_par;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	if( !current->window->debug_window )
		return;

	dw = current->window->debug_window;

	run_par = gtk_entry_get_text( GTK_ENTRY( entry ) );	

	if( !run_par || isempty( run_par ) )
	{
		strcpy( cmdstr, "run" );
	}
	else
	{
		g_snprintf( cmdstr, STRLEN, "run %s", run_par );
	}

	send_gdb_cmd( dw, cmdstr, TRUE );

	gtk_widget_set_sensitive( dw->entry, TRUE );
}


static void debug_run( GtkWidget *widget, gI_debug_window *dw )
{
	entry_dialog( _("Enter Parameters: "), _("Debug - Run Program"), debug_run_ok ); 
}


static void debug_cont( GtkWidget *widget, gI_debug_window *dw )
{
	send_gdb_cmd( dw, "cont", TRUE );
}


static void debug_brkpnt_ok( GtkWidget *widget, GtkWidget *entry )
{
	gchar cmdstr[100];
	gchar *ptr;
	gI_debug_window *dw;
	gI_document *current;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	if( !current->window )
		return;

	dw = current->window->debug_window;
	if( !dw )
		return;

	ptr = gtk_entry_get_text( GTK_ENTRY( entry ) );
	if( !ptr || isempty( ptr ) )
	{
		strcpy( cmdstr, "break" );
	}
	else
	{
		g_snprintf( cmdstr, sizeof(cmdstr), "break %s", ptr );
	}

	send_gdb_cmd( dw, cmdstr, TRUE );
	if( dw->brkpnt_window != NULL && GTK_WIDGET_VISIBLE( dw->brkpnt_window->window ) )
	{ 
		/* update breakpoint window */
		gtk_clist_clear( GTK_CLIST( dw->brkpnt_window->list ) );
		send_gdb_cmd( main_window->debug_window, "info breakpoints", TRUE );
		main_window->debug_window->wait_for = N_WAIT_BREAKPOINTS;
	}
}


static void debug_brkpnt( GtkWidget *widget, gI_debug_window *dw )
{
	entry_dialog( _("Enter Breakpoint: "), _("Debug - Set Breakpoint"), debug_brkpnt_ok );
}


gI_debug_window *gI_debug_window_new( gI_document *document )
{
	gI_debug_window *dw;
	GtkWidget *vbox, *hbox;
	GtkWidget *label;
	
	dw = g_malloc0( sizeof( gI_debug_window ) );
	dw->watch_window = NULL;
	dw->brkpnt_window = NULL;
	dw->sockname = NULL;
	dw->fh_in = -1;
	dw->fh_out = -1;
	dw->pid = -1;
	dw->chpid = -1;
	dw->state = N_GDB_UNDEF;
	dw->cmd_queue = NULL;
	dw->wait_for = -1;
	dw->breakpoints = NULL;

	dw->window = gtk_window_new( GTK_WINDOW_TOPLEVEL );
	gtk_window_set_title( GTK_WINDOW( dw->window ), _("gIDE Debug Output Window") );
	gtk_widget_set_usize( dw->window, 600, 225 );
	gtk_signal_connect( GTK_OBJECT( dw->window ), "destroy",
						GTK_SIGNAL_FUNC( gI_debug_window_destroy ), (gpointer) dw );
	install_debug_popup( dw );

	vbox = gtk_vbox_new( 0, FALSE );
	gtk_container_add( GTK_CONTAINER( dw->window ), vbox );
	gtk_widget_show( vbox );

	dw->text = gtk_text_new( NULL, NULL );	
	gtk_box_pack_start( GTK_BOX( vbox ), dw->text, TRUE, TRUE, 0 );
	gtk_signal_connect_object( GTK_OBJECT( dw->text ), "event",
							   GTK_SIGNAL_FUNC( popup_debug_menu ), GTK_OBJECT( dw->popup_menu ) ); 
	gtk_widget_show( dw->text );
	gtk_widget_grab_focus( dw->text );

	hbox = gtk_hbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 0 );
	gtk_widget_show( hbox );

	label = gtk_label_new( "gdb>" );
	gtk_box_pack_start( GTK_BOX( hbox ), label, FALSE, TRUE, 2 );
	gtk_widget_show( label );
	
	dw->entry = gtk_entry_new();
	gtk_box_pack_start( GTK_BOX( hbox ), dw->entry, TRUE, TRUE, 2 );
	gtk_signal_connect( GTK_OBJECT( dw->entry ), "activate",
						GTK_SIGNAL_FUNC( send_command_cb ), dw ); 
	gtk_widget_show( dw->entry );
	gtk_widget_set_sensitive( dw->entry, FALSE );

	return( dw );			
}


void gI_debug_window_destroy( GtkWidget *widget, gI_debug_window *dw )
{
	/* stop the debugger */
	debug_stop_gdb( dw );

	/* destroy window */
	gtk_widget_destroy( dw->window );
	dw->window = NULL;

	/* free memory */	
	g_free( dw );
	dw = NULL;
	main_window->debug_window = NULL;

	/* set statusbar */
	gI_window_set_statusbar( gI_document_get_current( main_window ) );
}


glong is_break( gI_document *current, gchar *file, glong line )
{
	GList *tmp;
	gI_brkpnt *brkpnt;

	tmp = current->window->debug_window->breakpoints;

	tmp = g_list_first( tmp );

	while( tmp != NULL )
	{
		brkpnt = (gI_brkpnt *) tmp->data;
		if( brkpnt )
		{
			if( !strcmp( brkpnt->file, file ) && brkpnt->line == line )
				return( 1 );
		} 
		tmp = g_list_next( tmp );
	}	

	return( 0 );
} 


void debug_set_breakpoint( GtkWidget *widget, gpointer data )
{
	gchar cmdstr[100];
	gchar *file;
	glong line;
	glong brkno;
	gI_document *current;
	gI_debug_window *dw;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	/* abort, if no debug window */
	if( !current->window->debug_window )
		return;

	/* abort, if no filename is set */
	if( !current->filename )
		return;

	dw = current->window->debug_window;

	/* get filename and line number */
	file = current->filename;
	line = get_line_from_point( gI_text_get_point( current->text ) );

	if( (brkno = is_break( current, file, line )) )
	{	/* remove breakpoint */
		sprintf( cmdstr, "clear %s:%ld", file, line ); 
	}
	else 
	{	/* set breakpoint */
		sprintf( cmdstr, "break %s:%ld", file, line ); 
	}

	/* send command */
	send_gdb_cmd( current->window->debug_window, cmdstr, TRUE );

	if( dw->brkpnt_window != NULL && GTK_WIDGET_VISIBLE( dw->brkpnt_window->window ) )
	{ 
		/* update breakpoint window */
		gtk_clist_clear( GTK_CLIST( dw->brkpnt_window->list ) );
		send_gdb_cmd( main_window->debug_window, "info breakpoints", TRUE );
		main_window->debug_window->wait_for = N_WAIT_BREAKPOINTS;
	}
	else
	{
		/* we need this anyway .. to update the breakpoints GList */
		send_gdb_cmd( main_window->debug_window, "info breakpoints", TRUE );
		main_window->debug_window->wait_for = N_WAIT_BREAKPOINTS;
	}
}


void debug_step_over( GtkWidget *widget, gpointer data )
{
	gI_document *current;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	if( !current->window->debug_window )
		return;

	/* send command */
	send_gdb_cmd( current->window->debug_window, "next", TRUE );
}


void debug_step_into( GtkWidget *widget, gpointer data )
{
	gI_document *current;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	if( !current->window->debug_window )
		return;

	/* send command */
	send_gdb_cmd( current->window->debug_window, "step", TRUE );
}


void debug_break( GtkWidget *widget, gpointer data )
{
	gI_document *current;
	gI_debug_window *dw;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	if( !( dw = current->window->debug_window ) )
		return;

	dw->chpid = get_child_pid( dw );	/* get child pid */
	if( dw->chpid <= 0 )
		return;

	kill( current->window->debug_window->chpid, SIGINT );
}


void debug_stop_debug( GtkWidget *widget, gpointer data )
{
	gI_document *current;
	gI_debug_window *dw;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	if( !current->window->debug_window )
		return;	

	dw = current->window->debug_window;

	debug_stop_gdb( dw );

	/* close all debug windows */
	if( dw->brkpnt_window != NULL && GTK_WIDGET_VISIBLE( dw->brkpnt_window->window ) )
	{
		gtk_widget_destroy( dw->brkpnt_window->window );
		gI_brkpnt_window_destroy( widget, dw->brkpnt_window ); 
	}

	if( dw->watch_window != NULL && GTK_WIDGET_VISIBLE( dw->watch_window->window ) )
	{
		gtk_widget_destroy( dw->watch_window->window );
		gI_watch_window_destroy( widget, dw->watch_window );
	}

	if( dw->bt_window != NULL && GTK_WIDGET_VISIBLE( dw->bt_window->window ) )
	{
		gtk_widget_destroy( dw->bt_window->window );
		gI_bt_window_destroy( widget, dw->bt_window );
	}

	if( GTK_WIDGET_VISIBLE( dw->window ) )
	{
		gI_debug_window_destroy( widget, dw );
	}
}


static void debug_attach_ok( GtkWidget *widget, GtkWidget *entry )
{
	gchar cmdstr[25];
	gint pid;
	gI_document *current;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	if( !current->window->debug_window )
		return;

	pid = atoi( gtk_entry_get_text( GTK_ENTRY( entry ) ) );
	if( pid > 0 )
	{
		g_snprintf( cmdstr, 25, "attach %d", pid );
		send_gdb_cmd( current->window->debug_window, cmdstr, TRUE );
	}
	else
	{
		gI_error_dialog( _("Invalid PID!") );
	}
}


void debug_attach( GtkWidget *widget, gpointer data )
{
	/* FIXME: offer process (c)list here... */

	entry_dialog( _("Enter PID: "), _("Attach to Process"), debug_attach_ok );
}


gI_brkpnt *gI_brkpnt_new()
{
	gI_brkpnt *brkpnt;

	brkpnt = g_malloc0( sizeof( gI_brkpnt ) );
	brkpnt->file = NULL;
	brkpnt->line = -1;
	brkpnt->no = -1;

	return( brkpnt );
}


gI_gdbcmd *gI_gdbcmd_new()
{
	gI_gdbcmd *gdbcmd;

	gdbcmd = g_malloc0( sizeof( gI_gdbcmd ) );
	gdbcmd->cmd = NULL;
	gdbcmd->show = 1;
	gdbcmd->status = N_GDBCMD_TODO;

	return( gdbcmd );
}


/*
 ---------------------------------------------------------------------
     Function: get_child_pid()
     Desc: Returns the the child PID of a process running under GDB
		   control (uses ps) 
 ---------------------------------------------------------------------
*/
static gint get_child_pid( gI_debug_window *dw )
{
	FILE *fp;
	gchar buf[STRLEN];
	glong pid, ppid;
	glong child_pid = -1;

	fp = popen( "ps -j", "r" );
	if( !fp )
	{
		perror( "popen" );
		return( -1 );
	}	

	/* skip title line */
	fgets( buf, sizeof(buf), fp );

	while( fgets( buf, sizeof(buf), fp ) )
	{
		if( sscanf( buf, "%ld %ld", &ppid, &pid ) == 2 )
		{
			if( dw->pid == ppid )
			{
				child_pid = pid;
				break;
			}
		}
	}	

	if( pclose( fp ) != 0 )
	{
		perror( "pclose" );
	}

	return( child_pid );
}


void gI_debug_window_parse_state( gI_debug_window *dw, gchar *buf )
{
	gchar *ptr;

	ptr = &buf[strlen(buf)];

	ptr -= strlen( GDB_PROMPT );

	if( !strcmp( ptr, GDB_PROMPT ) )
	{
		if( strstr( buf, "\032\032" ) )
		{
			dw->state = N_GDB_BREAK;
		}
		else
		{
			dw->state = N_GDB_IDLE;
		}
	}
	else
	{
		dw->state = N_GDB_RUNNING;
	}
}


static gint check_for_data(void)
{
	gint iteration = 0;

	while( gtk_events_pending() || gdk_events_pending() )
		iteration += gtk_main_iteration();

	return( iteration );
}


gI_watch_window *gI_watch_window_new()
{
	gI_watch_window *ww;
	gchar *list_titles[] = { "Expression", "Type", "Value" };

	ww = (gI_watch_window *) g_malloc0( sizeof( gI_watch_window ) );
	ww->data = NULL;

	ww->window = gtk_window_new( GTK_WINDOW_TOPLEVEL );
	gtk_window_set_title( GTK_WINDOW( ww->window ), _("Expressions") );
	gtk_widget_set_usize( ww->window, 600, 250 );
	gtk_signal_connect( GTK_OBJECT( ww->window ), "destroy",
						GTK_SIGNAL_FUNC( gI_watch_window_destroy ), (gpointer) ww );

	install_watch_popup( ww );

	ww->list = gtk_clist_new_with_titles( 3, list_titles );
	gtk_clist_column_titles_passive( GTK_CLIST( ww->list ) );
	gtk_clist_set_column_width( GTK_CLIST( ww->list ), 0, 100 );
	gtk_clist_set_column_width( GTK_CLIST( ww->list ), 1, 75 );
	gtk_container_add( GTK_CONTAINER( ww->window ), ww->list );
	gtk_signal_connect( GTK_OBJECT( ww->list ), "select_row",
						GTK_SIGNAL_FUNC( watch_list_select ), (gpointer) ww );
	gtk_signal_connect( GTK_OBJECT( ww->list ), "button_press_event",
						GTK_SIGNAL_FUNC( watch_event_handler ), (gpointer) ww );
	gtk_widget_show( ww->list );

	gtk_widget_show( ww->window );	

	return( ww );
}


void gI_watch_window_destroy( GtkWidget *widget, gI_watch_window *ww )
{
	free( ww );
	ww = NULL;

	if( main_window->debug_window )
		main_window->debug_window->watch_window = NULL;
}


static void watch_list_select( GtkWidget *widget, gint row, gint column, GdkEventKey *bevent, gI_watch_window *ww )
{
	gchar *expr;

	if( !bevent )
		return;

	if( bevent->type == GDK_2BUTTON_PRESS )
	{
		gtk_clist_get_text( GTK_CLIST( ww->list ), row, 0, &expr );		
		entry_dialog_data( expr, _("Set Expression"), set_expression, (gpointer) expr );
	}
}


static void set_expression( GtkWidget *widget, entry_data *ed )
{
	gchar cmdstr[STRLEN];
	gchar *ptr;
	gchar *value;

	ptr = gtk_entry_get_text( GTK_ENTRY( ed->entry ) ); 
	value = (gchar *) ed->data;

	g_snprintf( cmdstr, sizeof(cmdstr), "set %s = %s", value, ptr );

	/* destroy entry dialog and free memory */
	gtk_widget_destroy( ed->dialog );
	g_free( ed ); 

	/* set expression */
	send_gdb_cmd( main_window->debug_window, cmdstr, TRUE );

	/* update expressions window */
	send_gdb_cmd( main_window->debug_window, "info locals", TRUE );
	main_window->debug_window->wait_for = N_WAIT_EXPRESSIONS;
}


gI_brkpnt_window *gI_brkpnt_window_new()
{
	gI_brkpnt_window *bpw;
	gchar *list_titles[] = { "No.", "Function", "File", "Line" };

	bpw = (gI_brkpnt_window *) g_malloc0( sizeof( gI_brkpnt_window ) );
	
	bpw->window = gtk_window_new( GTK_WINDOW_TOPLEVEL );
	gtk_window_set_title( GTK_WINDOW( bpw->window ), _("Breakpoints") );
	gtk_widget_set_usize( bpw->window, 550, 225 );
	gtk_signal_connect( GTK_OBJECT( bpw->window ), "destroy",
					    GTK_SIGNAL_FUNC( gI_brkpnt_window_destroy ), (gpointer) bpw ); 
	
	install_bpw_popup( bpw );
	
	bpw->list = gtk_clist_new_with_titles( 4, list_titles );
	gtk_clist_set_column_width( GTK_CLIST( bpw->list ), 0, 30 );
	gtk_clist_set_column_width( GTK_CLIST( bpw->list ), 1, 200 );
	gtk_clist_set_column_width( GTK_CLIST( bpw->list ), 2, 200 );
	gtk_clist_set_column_width( GTK_CLIST( bpw->list ), 3, 50 );
	gtk_clist_column_titles_passive( GTK_CLIST( bpw->list ) );
	gtk_container_add( GTK_CONTAINER( bpw->window ), bpw->list );
	gtk_signal_connect( GTK_OBJECT( bpw->list ), "select_row",
						GTK_SIGNAL_FUNC( brkpnt_list_select ), (gpointer) bpw );
	gtk_signal_connect( GTK_OBJECT( bpw->list ), "button_press_event",
						GTK_SIGNAL_FUNC( brkpnt_event_handler ), (gpointer) bpw );
	gtk_widget_show( bpw->list );

	gtk_widget_show( bpw->window ); 

	return( bpw );
}

 
void gI_brkpnt_window_destroy( GtkWidget *widget, gI_brkpnt_window *bpw )
{
	g_free( bpw );
	bpw = NULL;

	main_window->debug_window->brkpnt_window = NULL;
}


static void debug_brkpnt_win( GtkWidget *widget, gI_debug_window *dw )
{
	if( !dw )	/* since we decided that there can be only one debug window */
		dw = main_window->debug_window;
	if( !dw )
		return;

	send_gdb_cmd( dw, "info breakpoints", TRUE );
	dw->wait_for = N_WAIT_BREAKPOINTS;
}


static void add_brkpnt( gI_debug_window *dw, gchar *brkpnt )
{
	gchar *row[4];
	gchar brkno[10];
	gchar function[256];
	gchar fileln[STRLEN];
	gchar file[STRLEN];
	gchar line[10];
	gchar *ptr;
	glong count;
	gI_brkpnt *_brkpnt;

	if( !dw )	/* since we decided that there can be only one debug window */
		dw = main_window->debug_window;
	if( !dw )
		return;

	/* this should not happen, abort */
	if( isspace( brkpnt[0] ) )
		return;

	/* only breakpoints, no watchpoints */
	if( strstr( brkpnt, "watchpoint" ) )
		return;

	count = sscanf( brkpnt, "%s %*s %*s %*s %*s in %s at %s", brkno, function, fileln );
	if( count == 3 )
	{
		/* get file and line no */
		ptr = strchr( fileln, ':' );
		ptr++;
		strcpy( line, ptr );
		ptr--;
		*ptr = '\0';	
		strcpy( file, fileln ); 

		/* add breakpoint to list */
		row[0] = brkno;
		row[1] = function;
		row[2] = file;
		row[3] = line; 
		gtk_clist_append( GTK_CLIST( dw->brkpnt_window->list ), row );

		_brkpnt = gI_brkpnt_new();
		_brkpnt->file = g_strdup( file );
		_brkpnt->line = atoi( line );	
		_brkpnt->no = atoi( brkno );	

/***
		if( !dw->breakpoints )
			dw->breakpoints = g_list_alloc();
***/

		dw->breakpoints = g_list_prepend( dw->breakpoints, (gpointer) _brkpnt );
	}
}


static void brkpnt_list_select( GtkWidget *widget, gint row, gint column, GdkEventButton *bevent, gI_brkpnt_window *bpw )
{
	gchar *file;
	gchar *line;
	
	if( !bevent )
		return;

	if( bevent->type == GDK_2BUTTON_PRESS )
	{
		/* get file & line from list */
		gtk_clist_get_text( GTK_CLIST( bpw->list ), row, 2, &file );
		gtk_clist_get_text( GTK_CLIST( bpw->list ), row, 3, &line );

		/* go to file & line */
		goto_file( file );
		goto_line( atoi(line) );
	}
}


static void install_debug_popup( gI_debug_window *dw )
{
	GtkAccelGroup *group;
	GtkItemFactoryEntry debug_popup_items[] =
	{
	    {"/Breakpoints/Set",              "F9",	debug_brkpnt,   	0},
	    {"/Breakpoints/Clear",            NULL,	debug_brkpnt_clear,	0},
	    {"/Breakpoints/View",	     "<alt>F9",	debug_brkpnt_win,	0},
		{"/Program/Break",				 "F12",	debug_break,		0},
		{"/Program/Continue",			  "F6",	debug_cont,			0},
		{"/Program/File",		  		  NULL,	debug_file,			0},
		{"/Program/Kill",		  	"<ctrl>F5",	debug_kill,			0},
		{"/Program/Run",				  "F5",	debug_run,			0},
		{"/sep1",						  NULL, NULL,				0, "<Separator>"},
		{"/Backtrace",					  NULL, debug_backtrace,	0},
		/*{"/Exceptions",					  NULL, debug_exceptions,	0},*/
		{"/Expressions",				  NULL, debug_expressions,	0},
		{"/sep2",						  NULL, NULL,				0, "<Separator>"},
		{"/Run To Cursor",			  NULL,		debug_run_to_cursor,0},
		{"/Step Into",				  "F11",	debug_step_into,	0},
		{"/Step Out",				  NULL,		debug_step_out,		0},
		{"/Step Over",				  "F10",	debug_step_over,	0}
	};

	group = gtk_accel_group_new();

    dw->popup_menu = gtk_item_factory_new( GTK_TYPE_MENU, "<Debug>", group );
    gtk_item_factory_create_items(dw->popup_menu, (sizeof(debug_popup_items)/sizeof(debug_popup_items[0])), debug_popup_items, NULL);

	gtk_accel_group_attach( group, GTK_OBJECT( dw->window ) );
}


static gint popup_debug_menu( GtkWidget *widget, GdkEvent *ev )
{
	if( ev->type == GDK_BUTTON_PRESS )
	{
		GdkEventButton *event = (GdkEventButton *) ev;
		if( event->button == 3 )
		{
			gtk_item_factory_popup(GTK_ITEM_FACTORY(widget), (guint) event->x_root, (guint) event->y_root, event->button, event->time);
			return( TRUE );
		}
	}

	return( FALSE );
}


static void debug_kill( GtkWidget *widget, gI_debug_window *dw )
{
	send_gdb_cmd( dw, "kill", TRUE );
}


static void debug_step_out( GtkWidget *widget, gI_debug_window *dw )
{
	send_gdb_cmd( dw, "finish", TRUE );
}


static void debug_run_to_cursor( GtkWidget *widget, gI_debug_window *dw )
{
	gI_document *current;
	gchar cmdstr[50];
	glong line;
	gchar *file;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	if( !current->filename )
		return;

	file = current->filename;
	line = get_line_from_point( gI_text_get_point( current->text ) );

	g_snprintf( cmdstr, sizeof(cmdstr), "until %s:%ld", file, line ); 

	send_gdb_cmd( dw, cmdstr, TRUE );
}


gI_bt_window *gI_bt_window_new()
{
	gI_bt_window *bt;
	gchar *list_titles[] = { "No.", "Function/Address", "File", "Line" };

	bt = (gI_bt_window *) g_malloc0( sizeof( gI_bt_window ) );

	bt->window = gtk_window_new( GTK_WINDOW_TOPLEVEL );
	gtk_window_set_title( GTK_WINDOW( bt->window ), "Backtrace" );
	gtk_widget_set_usize( bt->window, 550, 225 );
	gtk_signal_connect( GTK_OBJECT( bt->window ), "destroy",
						GTK_SIGNAL_FUNC( gI_bt_window_destroy ), (gpointer) bt );
	install_bt_popup( bt );
	
	bt->list = gtk_clist_new_with_titles( 4, list_titles );
	gtk_clist_set_column_width( GTK_CLIST( bt->list ), 0, 30 );
	gtk_clist_set_column_width( GTK_CLIST( bt->list ), 1, 200 );
	gtk_clist_set_column_width( GTK_CLIST( bt->list ), 2, 200 );
	gtk_clist_set_column_width( GTK_CLIST( bt->list ), 3, 50 );
	gtk_clist_column_titles_passive( GTK_CLIST( bt->list ) );
	gtk_container_add( GTK_CONTAINER( bt->window ), bt->list );
	gtk_signal_connect( GTK_OBJECT( bt->list ), "button_press_event",
						GTK_SIGNAL_FUNC( bt_list_select ), (gpointer) bt ); 
	gtk_widget_show( bt->list );

	gtk_widget_show( bt->window );

	return( bt );
}


void gI_bt_window_destroy( GtkWidget *widget, gI_bt_window *bt )
{
	g_free( bt );
	bt = NULL;
	main_window->debug_window->bt_window = NULL;
}


static void debug_backtrace( GtkWidget *widget, gI_debug_window *dw )
{
	if( !dw )	/* since we decided that there can be only one debug window.. */
		dw = main_window->debug_window;
	if( !dw )
		return;

	send_gdb_cmd( dw, "bt", TRUE );
	dw->wait_for = N_WAIT_BACKTRACE; 
}


static void debug_goto_frame( GtkWidget *widget, gI_debug_window *dw )
{
	GList *selection;
	gchar *frameno;
	gchar cmdstr[100];

	if( !dw )	/* since we decided that there can be only one debug window */
		dw = main_window->debug_window;
	if( !dw )
		return;

	selection = GTK_CLIST( dw->bt_window->list )->selection;
	if( !selection )
		return; 

	gtk_clist_get_text( GTK_CLIST( dw->bt_window->list ), (gint) selection->data, 0, &frameno );	

	g_snprintf( cmdstr, sizeof(cmdstr), "frame %s", frameno );

	send_gdb_cmd( dw, cmdstr, TRUE );
}


static void add_frame( gI_debug_window *dw, gchar *frame )
{
	glong count;
	gchar frameno[10];
	gchar funcaddr[STRLEN];
	gchar _funcaddr[STRLEN];
	gchar *row[4];
	gchar *ptr;
	gchar *fn[3];
	gchar *file = NULL;
	gchar *line = NULL;

	if( !dw )
		return;
	if( !dw->bt_window )
		return;

	count = sscanf( frame, "%s %s", frameno, funcaddr );
	if( count == 2 )
	{
		if( sscanf( frame, "%*s %*s in %s", _funcaddr ) == 1 )
		{
			strcpy( funcaddr, _funcaddr );
		}

		ptr = &frame[strlen(frame)-1];
		if( isspace( *ptr ) )
		{	
			while( isspace(*ptr) )
				ptr--;
		}

		while( !isspace(*ptr) )
			ptr--;

		if( strchr( ptr, ':' ) )
		{
			SK_GetFields( ptr, fn, ':' );
	
			file = fn[1];
			line = fn[2];

			while( isspace(*file) )
				file++;
			while( isspace(*line) )
				line++;
		}
		else
		{
			fn[1] = NULL;
			fn[2] = NULL;
		}	
	
		if( !dw->bt_window )
			return;

		row[0] = frameno;
		row[1] = funcaddr;
		row[2] = file;
		row[3] = line;		
		gtk_clist_append( GTK_CLIST( dw->bt_window->list ), row );

		if( fn[0] != NULL )
			free( fn[0] );
		if( fn[1] != NULL )
			free( fn[1] );
		if( fn[2] != NULL )
			free( fn[2] );
	}
	else
	{
		return;
	}
}


static gint bt_list_select( GtkWidget *widget, GdkEventButton *event )
{
	gI_bt_window *bt;

	bt = main_window->debug_window->bt_window;
	if( !bt )
		return( FALSE );

	if( event->button == 3 )
	{
		gtk_item_factory_popup(GTK_ITEM_FACTORY(bt->popup_menu), (guint) event->x_root, (guint) event->y_root, event->button, event->time);
		return( TRUE );
	}

	return( FALSE );
}


static void install_bt_popup( gI_bt_window *bt )
{
	GtkAccelGroup *group;
	GtkItemFactoryEntry bt_popup_items[] =
	{
		{"/Goto Frame",			  NULL, debug_goto_frame,	0},
		{"/View Source",		  NULL, debug_bt_view_source,	0}
	};

	group = gtk_accel_group_new();

    bt->popup_menu = gtk_item_factory_new( GTK_TYPE_MENU, "<Backtrace>", group );
    gtk_item_factory_create_items(bt->popup_menu, (sizeof(bt_popup_items)/sizeof(bt_popup_items[0])), bt_popup_items, NULL);

	gtk_accel_group_attach( group, GTK_OBJECT( bt->window ) );
}


static void debug_bt_view_source( GtkWidget *widget, gI_debug_window *dw )
{
	GList *selection;
	gchar *file, *line;

	if( !dw )	/* since we decided that there can be only one debug window */
		dw = main_window->debug_window;
	if( !dw )
		return;

	selection = GTK_CLIST( dw->bt_window->list )->selection;
	if( !selection )
		return; 

	gtk_clist_get_text( GTK_CLIST( dw->bt_window->list ), (gint) selection->data, 2, &file );	
	gtk_clist_get_text( GTK_CLIST( dw->bt_window->list ), (gint) selection->data, 3, &line );	

	if( !file || !line )
		return;

	if( check_file_open( file ) )
	{
		goto_file( file );
	}
	else
	{
		if( !file_exist( file ) )
			return;

		file_open_by_name( main_window, file );
	}

	goto_line( atoi(line) );
}


static void debug_brkpnt_clear_ok( GtkWidget *widget, GtkWidget *entry )
{
	gchar *ptr;
	gI_debug_window *dw;
	gchar cmdstr[STRLEN];

	dw = main_window->debug_window;
	if( !dw )
		return;

	ptr = gtk_entry_get_text( GTK_ENTRY( entry ) );
	if( !ptr || isempty( ptr ) )
		return;

	g_snprintf( cmdstr, sizeof(cmdstr), "clear %s", ptr );

	send_gdb_cmd( dw, cmdstr, TRUE );
}


static void debug_brkpnt_clear( GtkWidget *widget, gI_debug_window *dw )
{
	entry_dialog( _("Enter Breakpoint: "), _("Debug - Clear Breakpoint"), debug_brkpnt_clear_ok );	
}


static void install_bpw_popup( gI_brkpnt_window *bpw )
{
	GtkAccelGroup *group;
	GtkItemFactoryEntry bpw_popup_items[] =
	{
		{"/Clear Breakpoint",	  "Delete", debug_bpw_clear_bp,		0},
		{"/View Source",		  NULL, debug_bpw_view_source,		0}
	};

	group = gtk_accel_group_new();

    bpw->popup_menu = gtk_item_factory_new( GTK_TYPE_MENU, "<Breakpoint>", group );
    gtk_item_factory_create_items(bpw->popup_menu, (sizeof(bpw_popup_items)/sizeof(bpw_popup_items[0])), bpw_popup_items, NULL);

	gtk_accel_group_attach( group, GTK_OBJECT( bpw->window ) );
}


static gint brkpnt_event_handler( GtkWidget *widget, GdkEventButton *event )
{
	if( !event )
		return( FALSE );

	if( event->button == 3 )
	{
		gI_brkpnt_window *bpw = main_window->debug_window->brkpnt_window;
		if( !bpw )
			return( FALSE );
	
		gtk_item_factory_popup(GTK_ITEM_FACTORY(bpw->popup_menu), (guint) event->x_root, (guint) event->y_root, event->button, event->time);
		return( TRUE );
	}
	
	return( FALSE );
}


static void debug_bpw_clear_bp( GtkWidget *widget, gpointer data )
{
	GList *selection;
	gI_brkpnt_window *bpw = main_window->debug_window->brkpnt_window;
	gI_debug_window *dw = main_window->debug_window;
	gchar *brkno;
	gchar cmdstr[50];

	if( !bpw )
		return;

	selection = GTK_CLIST( bpw->list )->selection;
	if( !selection )
		return;

	gtk_clist_get_text( GTK_CLIST( bpw->list ), (gint) selection->data, 0, &brkno );
	g_snprintf( cmdstr, sizeof(cmdstr), "del %s", brkno );
	
	send_gdb_cmd( dw, cmdstr, TRUE );	

	/* update breakpoint window */
	gtk_clist_clear( GTK_CLIST( bpw->list ) );
	send_gdb_cmd( main_window->debug_window, "info breakpoints", TRUE );
	main_window->debug_window->wait_for = N_WAIT_BREAKPOINTS;
}


static void debug_bpw_view_source( GtkWidget *widget, gpointer data )
{
	GList *selection;
	gI_brkpnt_window *bpw = main_window->debug_window->brkpnt_window;
	gchar *file;
	gchar *line;

	if( !bpw )
		return;

	selection = GTK_CLIST( bpw->list )->selection;
	if( !selection )
		return;

	gtk_clist_get_text( GTK_CLIST( bpw->list ), (gint) selection->data, 2, &file );
	gtk_clist_get_text( GTK_CLIST( bpw->list ), (gint) selection->data, 3, &line );


	if( !file_exist( file ) )
		return;

	if( check_file_open( file ) )
	{
		goto_file( file );
	}
	else
	{
		file_open_by_name( main_window, file );
	}

	goto_line( atoi(line) );
}


static void install_watch_popup( gI_watch_window *ww )
{
	GtkAccelGroup *group;
	GtkItemFactoryEntry ww_popup_items[] =
	{
		{"/Evaluate",			  NULL, debug_ww_eval_expr,			0},
		{"/Set",				  NULL, debug_ww_set_expr,			0}
	};

	group = gtk_accel_group_new();

    ww->popup_menu = gtk_item_factory_new( GTK_TYPE_MENU, "<Watch>", group );
    gtk_item_factory_create_items(ww->popup_menu, (sizeof(ww_popup_items)/sizeof(ww_popup_items[0])), ww_popup_items, NULL);

	gtk_accel_group_attach( group, GTK_OBJECT( ww->window ) );
}


static gint watch_event_handler( GtkWidget *widget, GdkEventButton *event, gI_watch_window *ww )
{
	if( !event )
		return( FALSE );

	if( event->button == 3 )
	{
		if( !ww )
			return( FALSE );
	
		gtk_item_factory_popup(GTK_ITEM_FACTORY(ww->popup_menu), (guint) event->x_root, (guint) event->y_root, event->button, event->time);
		return( TRUE );
	}
	
	return( FALSE );
}


static void debug_ww_eval_expr( GtkWidget *widget, gpointer data )
{
	gchar *expr, *type;
	GList *selection;
	gI_watch_window *ww;
	gchar cmdstr[STRLEN];

	if( !main_window->debug_window )
		return;

	ww = main_window->debug_window->watch_window;
	if( !ww )
		return;	

	selection = GTK_CLIST( ww->list )->selection;
	if( !selection )
		return; 

	gtk_clist_get_text( GTK_CLIST( ww->list ), (gint) selection->data, 0, &expr );		
	gtk_clist_get_text( GTK_CLIST( ww->list ), (gint) selection->data, 1, &type );		

	if( !strchr( type, '*' ) )
	{
		g_snprintf( cmdstr, sizeof(cmdstr), "print (%s)", expr );
	}
	else
	{
		g_snprintf( cmdstr, sizeof(cmdstr), "print *(%s)", expr );
	}

	send_gdb_cmd( main_window->debug_window, cmdstr, TRUE );
	main_window->debug_window->watch_window->data = g_strdup( expr );
	main_window->debug_window->wait_for = N_WAIT_EVALEXPR;
}


static void debug_ww_set_expr( GtkWidget *widget, gpointer data )
{
	gchar *expr;
	GList *selection;
	gI_watch_window *ww;

	if( !main_window->debug_window )
		return;

	ww = main_window->debug_window->watch_window;
	if( !ww )
		return;	

	selection = GTK_CLIST( ww->list )->selection;
	if( !selection )
		return; 

	gtk_clist_get_text( GTK_CLIST( ww->list ), (gint) selection->data, 0, &expr );		
	entry_dialog_data( expr, _("Set Expression"), set_expression, (gpointer) expr );
}


static void show_evalexpr( gchar *expr, gchar *data )
{
	GtkWidget *window;
	GtkWidget *root, *root_item, *item;
	gchar *start = NULL, *end = NULL;
	gchar *ptr;
	gchar subexpr[STRLEN];
	glong endofexpr = 0;
	gchar *retptr;
	gchar *exprptr;
	gchar *title;
	item_data *id;
	glong count = 0;

	window = gtk_window_new( GTK_WINDOW_TOPLEVEL );
	title = g_strconcat( "Expression: ", expr, NULL );
	gtk_window_set_title( GTK_WINDOW( window ), title );
	g_free( title );
	gtk_widget_set_usize( window, 350, 250 );
	gtk_signal_connect_object( GTK_OBJECT( window ), "destroy",
							GTK_SIGNAL_FUNC( gtk_widget_destroyed ), NULL );

	root = gtk_tree_new();
	gtk_container_add( GTK_CONTAINER( window ), root );
	gtk_widget_show( root );

	root_item = gtk_tree_item_new_with_label( expr );
	gtk_tree_append( GTK_TREE( root ), root_item );

	root = gtk_tree_new();
	gtk_tree_item_set_subtree( GTK_TREE_ITEM( root_item ), root );
	gtk_widget_show( root );


	/* remove all '\n' from data */
	while( (retptr = strchr( data, '\n' )) )
		*retptr = ' ';

	/* parse data and create tree */
	ptr = data;
	start = ptr;

	while( *ptr != '\0' )
	{
		switch( *ptr )
		{
			case '"':
			{
				ptr++;
				while( *ptr != '"' )
					ptr++;
				ptr++;
				break;
			}		
	
			case ',':
			{
				if( count > 1 )
				{
					ptr++;
				}
				else
				{
					/* set retptr & abort here */
					end = ptr - 1;
				}
				break;
			}

			case '{':
			{
				count++;
				ptr++;
				break;
			}

			case '}':
			{
				count--;
				if( count == 1 )
				{
					/* toplevel */
					end = ptr;
				}
				if( count == 0 )
				{
					/* end */
					endofexpr = 1;
					end = ptr - 1;
				}
				ptr++;
				break;
			}

			default:
			{
				ptr++;
				break;
			}
		}

		if( !start || !end )
			continue;

		if( *start == '{' )
			start++;

		strncpy( subexpr, start, (end-start)+1 );
		subexpr[(end-start)+1] = '\0';

		/* strip whitespace */
		exprptr = &subexpr[0];
		while( *exprptr == ' ' )
			exprptr++;

		/* warning: memory is not free'd *LEAK* */
		id = (item_data *) g_malloc( sizeof( item_data ) );
		id->expr = g_strdup( expr );
		id->item = g_strdup( exprptr );
		if( strstr( subexpr, "= 0x" ) && !strstr( subexpr, "= {" ) )
			id->ptr = 1;
		else
			id->ptr = 0; 

		/* new tree item */
		item = gtk_tree_item_new_with_label( exprptr );
		gtk_tree_append( GTK_TREE( root ), item );
		gtk_signal_connect( GTK_OBJECT( item ), "button_press_event",
							GTK_SIGNAL_FUNC( evalwin_item_bp_handler ), (gpointer) id );
		gtk_widget_show( item );

		if( endofexpr )
			break;

		ptr += 2;
		start = ptr;
		end = NULL;
	}

	gtk_tree_item_expand( GTK_TREE_ITEM( root_item ) );
	gtk_widget_show( root_item );	
	gtk_widget_show( window );
}


static void evalwin_item_bp_handler( GtkWidget *widget, GdkEventButton *event, item_data *id )
{
	gchar cmdstr[STRLEN];
	gchar *ptr;
	gchar *newitem;

	if( !event )
		return;

	if( event->type != GDK_2BUTTON_PRESS )
		return;

	ptr = id->item;
	while( *ptr != ' ' && *ptr != '\0' )
		ptr++;
	*ptr = '\0';

	newitem = g_strconcat( id->expr, "->", id->item, NULL );

	if( id->ptr )
	{
		g_snprintf( cmdstr, sizeof(cmdstr), "print *(%s)", newitem );
	}
	else
	{
		g_snprintf( cmdstr, sizeof(cmdstr), "print (%s)", newitem );
	}

	send_gdb_cmd( main_window->debug_window, cmdstr, TRUE );
	main_window->debug_window->watch_window->data = g_strdup( newitem );
	main_window->debug_window->wait_for = N_WAIT_EVALEXPR;

	g_free( newitem );
}


static void debug_expressions( GtkWidget *widget, gI_debug_window *dw )
{
	if( !dw )	/* since we decided that there can be only one debug window.. */
		dw = main_window->debug_window;
	if( !dw )
		return;

	send_gdb_cmd( dw, "info locals", TRUE );
	dw->wait_for = N_WAIT_EXPRESSIONS;
}


static void debug_exceptions( GtkWidget *widget, gI_debug_window *dw )
{
	if( !dw )	/* since we decided that there can be only one debug window.. */
		dw = main_window->debug_window;
	if( !dw )
		return;

	send_gdb_cmd( dw, "info catch", TRUE );
	dw->wait_for = N_WAIT_EXCEPTIONS;

	/* needed for c++ debugging, output processing is not yet implemented */
}


static void debug_file_ok( GtkWidget *widget, GtkWidget *entry )
{
	gchar *ptr;
	gchar cmdstr[STRLEN];
	gI_debug_window *dw;

	dw = main_window->debug_window;
	if( !dw )
		return;

	ptr = gtk_entry_get_text( GTK_ENTRY( entry ) );
	if( !ptr || isempty( ptr ) )
		return;

	g_snprintf( cmdstr, sizeof(cmdstr), "file %s", ptr );
	
	send_gdb_cmd( dw, cmdstr, TRUE );	
}


static void debug_file( GtkWidget *widget, gpointer data )
{
	entry_dialog( _("Enter File: "), _("Debug - Set Program File"), debug_file_ok );
}
