/* ----------------------------------------------------------------------
 * a small program to reorder the DATA statements in f77 code for f2c
 * (place them in strict f77 order)
 * ----------------------------------------------------------------------
 * author   : Wolfgang Wander
 * copyright: Wolfgang Wander (1995-1996)
 * ----------------------------------------------------------------------
 * Freely Distributable.
 * May not be copied without this header.
 * No warranties for the functionality of the code are given!
 * ----------------------------------------------------------------------
 * Warning: This code relies on my understanding of f77 grammar. If there
 *          really is some f77 grammar, it might be completely different.
 * ---------------------------------------------------------------------- */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#define _BSD_SOURCE
#define _GNU_SOURCE

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#ifdef NEED_STRNCASECMP_PROTO
#   ifdef __STDC__
       int strncasecmp(const char *s1, const char *s2, int n);
#   else
       int strncasecmp();
#   endif
#endif

/* ----------------------------------------------------------------------
 * the standard continue statement
 */

#define CONTINUE "     > "


/* ----------------------------------------------------------------------
 *  all this keywords have to appear _before_ a data statement in strict
 *  f77.
 */

typedef struct  {
  char *name;
  int  type;
} f77word;


int line_counter = 0;
int outline_counter = 0;
int inflow;
int commainsert = 0;

FILE *linefile = 0;
char *filename = "";
int cppline = 1;
int cpprefline = 1;
char *cppfilename = "";


#define f77substart  0x0001
#define f77variable  0x0002
#define f77parameter 0x0004
#define f77common    0x0008
#define f77entry     0x0010

f77word before[] = {
  { "program",    f77substart },
  { "include",    0           },
  { "function",   f77substart },
  { "subroutine", f77substart },
  { "entry",      f77entry    },
  { "block",      f77substart },
  { "parameter",  f77parameter},
  { "integer",    f77variable },
  { "byte",       f77variable },
  { "real",       f77variable },
  { "doubleprecision",     f77variable },
  { "doublecomplex",    f77variable },
  { "complex",    f77variable },
  { "logical",    f77variable },
  { "character",  f77variable },
  { "dimension",  0           },
  { "implicit",   0           },
  { "common",     f77common   },
  { "namelist",   0           },
  { "equivalence",0           },
  { "save",       0           },
  { "external",   0           },
  { "intrinsic",  0           }
};

/* ----------------------------------------------------------------------
 * the states of the `grammar'
 */

typedef enum {
  init_region,			/* initialization code         */
  variable_def,			/* a variable definition       */
  data_statement,		/* the statement we search for */
  common_statement,		/* common block definitions    */
  flow_region,			/* execution statements        */
  comment,			/* a comment                   */
  continued,			/* a continued line            */
  statementfun,			/* statement function          */
  entry,			/* entry statement             */
  eof_found			/* end of file found           */
} state;

FILE *infile, *outfile;		/* the input and output of the
				 * f77reorder pipe             */

#define flinesize 1024		/* the maximum allowed line length */

/* ----------------------------------------------------------------------
 * a linked list of lines. Will be filled with the data statements
 * up to (but excluding) the first execution statement in a subroutine
 */

typedef struct _data {
  char *line;
  int number;
  char *filename;
  struct _data *next;
} dataline;


/* --- protos --- */
void freeparameter( void );

/* --- some things from configure: --- */

#ifndef HAVE_STRDUP
char *strdup(char *s)
{
   int l = strlen(s);
   char *m = malloc(l+1);
   strcpy(m,s);
   return m;
}
#endif

#ifndef HAVE_STRSPN
int strspn(char *s, char *chrs)
{
   int i = 0;
   while(*s && strchr(chrs, *s)) /* if you want an efficient implementation */
      i++,s++;                   /* get yourself a real libc :-) */
   return i;
}
#endif

#ifndef HAVE_STRNCASECMP

/* don't get into more configure trouble by using ctype :-) */
#define _scc_tolower(x) ((x)>='A'&&(x)<='Z' ? (x)-'A'+'a' : (x))

int strncasecmp(char *s, char *t, int n)
{
   while(n-- & *s && _scc_tolower(*s)==_scc_tolower(*p))
      s++,p++;
   if(n < 0) return 0;
   return _scc_tolower(*s) < _scc_tolower(*p) ? -1 : 1;
}

#undef _scc_tolower
#endif

/* ----------------------------------------------------------------------
 * Append a line to the linked list of initialization statements
 * ---------------------------------------------------------------------- */

dataline *
append(dataline *head, char *line)
{
  dataline *walk;
  if( (walk = head) ) {		/* find the end of the linked list */
    while( walk->next)		/* optimize! */
      walk = walk->next;
    walk = walk->next = calloc( 1, sizeof(dataline) );
  } else {
    walk = head = calloc( 1, sizeof(dataline) );
  }
  if( ! walk  || 0 == (walk->line = malloc( strlen(line)+1) )) {
    fprintf(stderr,"f77reorder: out ouf memory. (9)\n");
    exit(1);
  }
  strcpy(walk->line, line);	/* copy the line */
  walk->number = line_counter - cpprefline + cppline;
  walk->filename = malloc( strlen(cppfilename)+1);
  if( ! walk->filename ) {
     fprintf(stderr,"f77reorder: out ouf memory. (9)\n");
    exit(1);
  }
  strcpy( walk->filename, cppfilename);
  return head;
}

/* ----------------------------------------------------------------------
 * free the heap used by the linked list of initialization statements
 * ---------------------------------------------------------------------- */

void
freeline( dataline *line ) {
  dataline *next;
  while( line ) {
    next = line->next;
    free(line->line);
    free(line->filename);
    free(line);
    line = next;
  }
}

/* ----------------------------------------------------------------------
 * output routines
 * ---------------------------------------------------------------------- */

void
_outputline( char *text,  int linenumber, char *filename )
{
  outline_counter++;
  if( linefile )
    fprintf(linefile,"%d %d %s\n", outline_counter, linenumber, filename);
  fprintf(outfile,"%s",text);
}

void
outputdata( dataline *line )
{
  _outputline( line->line, line->number, line->filename);
}

void
outputline( char *text )
{
  _outputline( text, -cpprefline+line_counter + cppline, cppfilename);
} 

/* ----------------------------------------------------------------------
 * string compare of a string with a tag in f77 manner (disregarding
 * spaces) and case senstitivity
 * ---------------------------------------------------------------------- */

int
strf77cmp( char *line, char *tag, char **end )
{
  while( *tag ) {
    while( isspace(*line) )
      line++;
    if( toupper(*line) == toupper(*tag) ) {
      line++;
      tag++;
    } else
      return 1;
  }
  if( isalnum(*line) )
    return 1;
  if( end )
    *end = line;
  return 0;
}
	

/* ----------------------------------------------------------------------
 * check if a f77 code line starts with a string tag
 * ---------------------------------------------------------------------- */

int
firsttag( char *text, char *tag)
{
  int l;
  l = strspn(text," \t");	/* find the beginning of the line.
				 * assumes there are no labels allowed
				 * before initializer statements */
  return strf77cmp( text+l, tag, 0) == 0;
}

/* ----------------------------------------------------------------------
 * skips a fortran tag
 * ---------------------------------------------------------------------- */

char *
skiptag( char *text, char *tag)
{
  while( *text && *tag ) {
    if( tolower( *text ) == tolower( *tag ) ) 
      tag++;
    else if( !isspace(*text ) )
      return 0;
    text++;
  }
  return text;
}     


/* ----------------------------------------------------------------------
 * get the line number from a cpp line
 * ---------------------------------------------------------------------- */

void
parsecppline( char *text )
{
  char *p, *q;
  static char buf[512];
  if( text[1] == ' ' && isdigit( text[2] )) {
   p = strchr(text, '"');
    if( p ) {
      q = strchr(p+1,'"');
      if( q ) {
	cppline     = atoi(text+2);
	cpprefline  = line_counter+1;
 	strncpy( buf, p+1, q-p-1)[q-p-1] = 0;
	if( !*buf )
	  strcpy( buf, filename);
	cppfilename = buf;
      }
    }
  }
} 
    

/* ----------------------------------------------------------------------
 * read a fortran line with the possibility to have an ungetline funtion
 * i.e. to put a complete line back to the stream.
 * ---------------------------------------------------------------------- */
char *ungetstr = 0;

int
fgetline( char *text )
{
  if( ungetstr ) {		/* is the ungetstr line filled? */
    strcpy( text, ungetstr);	/* if so - don't read from file */
    free( ungetstr );
    ungetstr = 0;
  } else {
    once_more:
    if( fgets( text, flinesize, infile) == 0 ) /* read from file */
      return 1;
    line_counter ++;
    if( *text == '#' ) {
      parsecppline( text );
      goto once_more;
    }
  }
  return 0;
}

/* ----------------------------------------------------------------------
 * the above mentioned ungetline function
 * ---------------------------------------------------------------------- */

void
fungetline( char *text )
{
  if( ungetstr != 0 ) {	        /* something went wrong - we cannot
				 * unget twice*/
    fprintf(stderr,"f77reorder: Internal error. Ungetline called twice.\n");
    exit(1);
  }
  ungetstr = malloc( strlen( text ) + 1);
  if( ! ungetstr ) {
    fprintf(stderr,"f77reorder: out ouf memory (10)\n");
    exit(1);
  }
  strcpy( ungetstr, text);
}

/* ----------------------------------------------------------------------
 * a function which tests, if a line is a comment. A comment (in my
 * grammar ;-) is defined as a line starting with either !, c, C or *
 * in the first column or beeing a line starting with ! somewhere
 * following only TABs and spaces or being an empty line
 * -----------
 * Comments don't cause a transition from init_region to flow_region!
 * ----------------------------------------------------------------------*/

int 
isacomment( char *text )
{
  int l = strspn(text," \t");
  if( strchr( "!cC*\n", text[0] ) )     /* is a comment */
    return 1;
  if( text[l] == '!' )		/* a line starting with ! after only spaces */
    return 1;
  if( text[l] == '\n' || text[l] == 0 )	/*  an empty line */
    return 1;  
  return 0;
}

/* ----------------------------------------------------------------------
 * check if a line is a label.
 * In my grammar a label starts in the first 4 columns of a fortran
 * code line. A label always starts with a digit.
 * ---------------------------------------------------------------------- */

int
islabel( char *text )
{
  int l = strspn(text," \t");
  if( l < 4 && isdigit(text[l]) )
     return 1;
  return 0;
}


/* ----------------------------------------------------------------------
 * check if a line is continued from the previous one
 * ---------------------------------------------------------------------- */

int
iscontinued( char *text )
{
  int clm;
  for( clm = 0; clm < 5; ) {
    if( *text == ' ' ) {
      text++;
      clm++;
    } else
      return 0;
  }
  return *text != ' ';
}

/* ----------------------------------------------------------------------
 * Get a data statement and append it to the list of dataline
 * a DATA statement may exceed several lines, therefore a line
 * continuation is checked for.
 * ---------------------------------------------------------------------- */

state
getdataline( char *text , dataline **line)
{
  char ingetstr[flinesize];
  
  *line = append(*line, text);	/* append current text to the dataline list */
 
  for( ;; ) {
    if(fgetline( ingetstr ) )	/* get a new line from the input */
      return eof_found;
    if( isacomment(ingetstr) )  {        /* is a comment */
      outputline(ingetstr);
      continue;
    }
    if( iscontinued( ingetstr))
      *line = append( *line, ingetstr);	/* continued line - append it */
    else
      break;			/* a new statement - return */
  }
  fungetline( ingetstr );	/* put the new statement in the unget buffer */
  return data_statement;
}

/* ----------------------------------------------------------------------
 * skips the 6 first columns of a line
 * ---------------------------------------------------------------------- */
char *
skip6( char *line )
{
  int col = 0;
  while( *line && col < 6 ) {
    if( *line == '\t' )
      col = 8;
    else
      col ++;
    line++;
  }
  return line;
}


/* ----------------------------------------------------------------------
 * checks, if a continued line contains valid common variable statements
 * and the new line does NOT start with a comma
 * ---------------------------------------------------------------------- */

int
cont_common_wo_comma( dataline *line )
{
  char *p;
  for( ; line; line = line->next ) {
    for( p = skip6( line->line ); *p; p++ ) {
      if( *p == '!' )		/* a comment */
	break;
      if( *p == ',' )
        return 0;
      if( !isspace (*p ) )
	return 1;
    }
  }
  return 0;
}

/* ----------------------------------------------------------------------
 * checks if a common block statements lacks of a comma at the end of
 * line
 */

int
comma_missing( dataline *line )
{
  char *p;
  int miss = 0;
  p = skip6( line->line );
  if( firsttag( p, "common") )
    p = skiptag( p, "common");
  
  for( ; *p; p++ ) {
    if( *p == '!' )
      break;
    if( isalpha(*p) )
      miss = 1;
    if( *p == ',' || *p == '/')
      miss = 0;
  }
  return miss;
}

/* ----------------------------------------------------------------------
 * inserts a comma to the right place if one is missing
 */

int
insert_comma( dataline *line )
{
  char *p;
  int miss = 0;
  p = skip6( line->line );
  if( firsttag( p, "common") )
    p = skiptag( p, "common");
  
  for( ; *p; p++ ) {
    if( *p == '!' || *p == '\n' )
      break;
    if( isalpha(*p) )
      miss = 1;
    if( *p == ',' )
      miss = 0;
  }
  outputdata( line );
  if( miss && commainsert)
    _outputline(CONTINUE " , !inserted by f77reorder \n",
		line->number, line->filename);
  return miss;
}


/* ----------------------------------------------------------------------
 * Gets a complete common block definition (including continued lines)
 * and checks for missing commas at the end of the lines. Some compilers
 * seem to allow line breaks as a variable name separator
 * ---------------------------------------------------------------------- */

state
getcommonline( char *text )
{
  char ingetstr[flinesize];
  dataline *line = 0, *walk;
  
  line = append(0 , text);	/* append current text to the dataline list */
 
  for( ;; ) {
    if(fgetline( ingetstr ) )	/* get a new line from the input */
      return eof_found;
    if( isacomment(ingetstr) )  {       /* is a comment */
      outputline(ingetstr);	/* mismatch in order (REPAIR!)  */
      continue;
    }
    if( iscontinued( ingetstr))
      line = append( line, ingetstr);	/* continued line - append it */
    else
      break;			/* a new statement - return */
  }
  fungetline( ingetstr );	/* put the new statement in the unget buffer */

  for( walk = line; walk; walk = walk->next ) {	/* now print the stuff */
    if( cont_common_wo_comma( walk->next ) && comma_missing( walk ) )
      insert_comma( walk );
    else
      outputdata( walk );
  }
    
  freeline( line );
  return common_statement;
}


/* ----------------------------------------------------------------------
 * skips the variable type up to the first variable name
 * ---------------------------------------------------------------------- */

char *
skiptypedec( char *line)
{
  char *end = 0;
  int i;
  
  while( isspace(*line))
    line++;
  for( i = 0; i < sizeof( before) / sizeof(before[0]); i ++ ) {
    if( strf77cmp( line, before[i].name, &end) == 0 ) {
      line = end;
      break;
    }
  }
  while( isspace(*line))
    line++;
  if( *line == '*' ) {
    line++;
    while( isspace(*line))
      line++;
    if( *line == '(' ) {
      while( *line && *line != ')' )
	line++;
      if( *line == ')' )
	line++;
    } else {
      while( isdigit(*line))
	line++;
    }
    while( isspace(*line))
      line++;
  }
  return line;
}


/* ======================================================================
 * functions for parsing variable initializations
 * ---------------------------------------------------------------------- */

typedef enum {
  v_name, v_dimension, v_data, v_comma, v_datadone
} vstate;

typedef struct _vl {
  char *name;
  char *dimension;
  int   idim;
  char *data;
  int   ddim;
  vstate   commasep;
  struct _vl *next;
  struct _vl *prev;
} variable;

typedef struct _al {
  struct _al *next;
  char name[1];
} arrayname;

arrayname *Ahead = 0;

/* ----------------------------------------------------------------------
 * extracts variable names from the variable list
 * ---------------------------------------------------------------------- */
void
keep_arraynames(variable *var)
{
  arrayname *aname;
  for( ; var; var = var->next ) {
    aname = malloc( sizeof(arrayname)+strlen(var->name));
    if(!aname ) {
      fprintf(stderr,"f77reorder: out ouf memory. (13a)\n");
      exit(1);
    }
    strcpy( aname->name, var->name);
    aname->next = Ahead;
    Ahead = aname;
  }
}

/* ----------------------------------------------------------------------
 * checks if an array name exists
 * ---------------------------------------------------------------------- */

int
array_exists( char *line )
{
  arrayname *aname;
  for( aname = Ahead; aname; aname = aname->next ) {
    if( strf77cmp( line, aname->name, 0 ) == 0 )
      return 1;
  }
  return 0;
}
   
/* ----------------------------------------------------------------------
 * check if a line contains a statement function
 * ---------------------------------------------------------------------- */

int
isstatementfun( char *text )
{
  char *pequal;
  char *fstart = 0, *fend = 0;
  int bra;
  text = skip6( text);
  pequal = strchr( text, '=' );
  if( pequal ) {
    while( isspace(*text))	/* skip whitespace */
      text++;
    
    /* get the function name */
    fstart = text;
    while( isalpha(*text) || *text == '$' || *text == '_')
      text++;
    while( isalnum(*text) || *text == '$' || *text == '_')
      text++;
    fend = text;
    if( fend == fstart )	/* a null name */
      return 0;
    
    while( isspace(*text))
      text++;
    if( *text++ == '(' ) {	/* the arguments */
      bra = 1;
      while( bra && *text ) {
	if( *text == '(' )
	  bra++;
	if( *text == ')' )	/* must match braces */
	  bra--;
	text ++;
      }
      while( isspace(*text))
	text++;
      if( *text == '=' ) {
	if( array_exists( fstart ) ) { /* only if no array */
	  return 0;
	} else {
	  return 1;
	}
      }
    }
  }
  return 0;
}



/* ----------------------------------------------------------------------
 * frees a list of variables
 * ---------------------------------------------------------------------- */

void
freevarlist( variable *var )
{
  variable *next;
  for( ; var ; var = next ) {
    next = var->next;
    if( var->name )
      free(var->name);
    if( var->dimension)
      free(var->dimension);
    if( var->data )
      free( var->data);
    free(var);
  }
}
	
 

/* ----------------------------------------------------------------------
 * scans a line for a variable name and fills the current variable with
 * it, the ending of the name determines the further proceeding
 * (search for dimension statement, data or next variable)
 * ---------------------------------------------------------------------- */

char *
vargetname( char *text, variable *current )
{
  char *start, *new, *end;
  int len;
  
  while( isspace(*text))
    text++;
  start = text;

  while( isalpha(*text) || *text == '$' || *text == '_')
    text++;
  label:
  while( isalnum(*text) || *text == '$' || *text == '_')
    text++;
  end = text;
  while( isspace(*text))
    text++;
  if( isalnum(*text) || *text == '$' || *text == '_')
    goto label;
  if( *text == '*' ) {
    text++;
    while( isspace(*text))
      text++;
    while(isalnum(*text))
      text++;
  } else
    text = end;

  if( current->name )
    len = strlen( current->name);
  else
    len = 0;

  if( text - start > 0 ) {
    len += text-start + 1;
    new = malloc( len );
    if(!new) {
      fprintf(stderr,"f77reorder: out of memory (1)\n");
      exit(1);
    }
    *new = 0;
    if( current->name) {
      strcpy(new, current->name);
      free( current->name );
    }
    strncat( new, start, text-start);
    current->name = new;
  }
  while( isspace(*text))
    text++;
  if( *text == '!' || *text == '\0' )
    return 0;
  else if( *text == '(' )
    current->commasep = v_dimension;
  else if( *text == '/' )
    current->commasep = v_data;
  else if( *text == ',' ) {
    current->commasep = v_comma;
    text++;
  } else {
    fprintf(stderr,
	    "f77reorder: error in variable statement (1) - %s line %d\n",
	    cppfilename,line_counter-cpprefline+cppline);
    exit(1);
  }
  return text;
}


/* ----------------------------------------------------------------------
 * scans a line for a dimension field and fills the current variable with
 * it, the ending of the field determines the further proceeding
 * (search for data or next variable)
 * ---------------------------------------------------------------------- */

char *
vargetdimensions( char *text, variable *current )
{
  char *start, *new, *end;
  int len, brack = 0;

  new = strchr(text,'\n');
  if( new )
    *new = 0;
  
  if( current->dimension ) {
    for( start = current->dimension; *start; start++ ) {
      if( *start == '(' )
	brack++;
      if( *start == ')' )
	brack--;
    }
  }

  while( isspace(*text))
    text++;
  start = text;
  
  if( *text == '(' || brack) {
    for( ;; ) {
      while( *text ) {
	if( *text == '(' )
	  brack++;
	if( *text == ')' && --brack == 0 ) {
	  text++;
	  break;
	}
	text++;
      }
      end = text;
      while( isspace(*end ))
	end++;
      if( *end == '*' ) {
	end++;
	while( isspace(*end ))
	  end++;
	if( *end == '(' ) {
	  text = end;
	  continue;
	}
	if(isdigit(*end) ) {
	  while( isdigit(*end ))
	    end ++;
	  text = end;
	}
      }
      break;
    }
  }
    
  
  if( current->dimension )
    len = strlen( current->dimension);
  else
    len = 0;

  if( text - start > 0 ) {
    len += text-start + 1;
    new = malloc( len );
    if(!new) {
      fprintf(stderr,"f77reorder: out of memory (2)\n");
      exit(1);
    }
    *new = 0;
    if( current->dimension) {
      strcpy(new, current->dimension);
      free( current->dimension );
    }
    strncat( new, start, text-start);
    current->dimension = new;
  }
  while( isspace(*text))
    text++;
  if( *text == '!' || *text == '\0' )
    return 0;
  else if( *text == '/' )
    current->commasep = v_data;
  else if( *text == ',' ) {
    current->commasep = v_comma;
    text++;
  } else {
    fprintf(stderr,
	    "f77reorder: error in variable statement (2) - %s line %d\n",
	    cppfilename,line_counter-cpprefline+cppline);
    exit(1);
  }
  return text;
}

/* ----------------------------------------------------------------------
 * scans a line for a data field and fills the current variable with
 * it, the ending of the field determines the further proceeding
 * (search for next variable or error)
 * ---------------------------------------------------------------------- */

char *
vargetdata( char *text, variable *current )
{
  char *start, *new;
  int len, begin = *text == '/';
  int quote = 0;

  
  new = strchr(text,'\n');
  if( new )
    *new = 0;
  
  start = text;
  if( begin )
    text++;
  
  while( *text ) {
    if( ! quote && (*text == '/' || *text == '!')  )
      break;
    if( *text == '\'' ) {
      if( text[1] == '\'' )
	text ++;
      else
	quote ^= 1;
    }
    text++;
  }


  if( current->data )
    len = strlen( current->data)+6;
  else
    len = 0;

  if( text - start > 0 ) {
    len += text-start + 2;
    if( !*text )
      len++;
    new = malloc( len );
    if(!new) {
      fprintf(stderr,"f77reorder: out of memory. (3)\n");
      exit(1);
    }
    *new = 0;
    if( current->data) {
      strcpy(new, current->data);
      strcat(new, CONTINUE );
      free( current->data );
    }
    strncat( new, start, text-start+1);
    if( !*text )
      strcat( new, "\n");
    current->data = new;
  }
  if( *text == '/' ) {
    current->commasep = v_datadone;
    text ++;
  }
  while( isspace(*text))
    text++;
  if( *text == '!' || *text == '\0' )
    return 0;
  else if( *text == ',' ) {
    current->commasep = v_comma;
    text++;
  } else {
    fprintf(stderr,
	    "f77reorder: error in variable statement (3) - %s line %d\n",
	    cppfilename,line_counter-cpprefline+cppline);
    exit(1);
  }
  return text;
}

/* ----------------------------------------------------------------------
 * called after reading a complete initializer - a comma might follow
 */

char *
vargetcomma( char *text, variable *current )
{
  while( *text ) {
    if( *text == ',' ) {
      current->commasep = v_comma;
      return text+1;
    }
    if( isspace(*text ) ) {
      text++;
      continue;
    } else {
      fprintf(stderr,
	      "f77reorder: error in variable statement (4) - %s line %d\n",
	      cppfilename,line_counter-cpprefline+cppline);
      exit(1);
    }
  }
  return 0;			/* not reached */
}

/* ----------------------------------------------------------------------
 * scan a (interupted) line of text for variable definitions containing
 * name + optional dimension + optional data
 * integer i (1:1,2:2) /1/
 * ---------------------------------------------------------------------- */

char *
parsevariables( char *text, variable **list, int skiptype, char **type )
{
  variable *current = *list;
  char *tstart, *p;

  if( skiptype ) {
    tstart = text;
    text = skiptypedec(text);
    *type = malloc( text - tstart + 1);
    strncpy( *type, tstart, text - tstart)[text - tstart] = 0;
    p = strchr( *type,0);
    while( p-- > *type ) {
      if( isspace(*p) )
	*p = 0;
      else
	break;
    }
    *list = current = calloc( 1, sizeof( variable ));
    if( ! current ) {
      fprintf(stderr,"f77reorder: out of memory. (4)\n");
      exit(1);
    }
    current->commasep = v_name;
  }
  while( text ) {
    switch( current->commasep ) {
    case v_name:
      text = vargetname( text, current);
      break;
    case v_dimension:
      text = vargetdimensions( text, current );
      break;
    case v_data:
      text = vargetdata( text, current );
      break;
    case v_datadone:
      text = vargetcomma( text, current );
      break;
    case v_comma:
      current->next = calloc( 1, sizeof( variable));
      if( ! current->next ) {
	fprintf(stderr,"f77reorder: out of memory. (5)\n");
	exit(1);
      }
      current->next->prev = current;
      *list = current = current->next;
      current->commasep = v_name;
      break;
    }
  }
  return text;
}


/* ----------------------------------------------------------------------
 * gets an integer for a given parameter or integer
 * ---------------------------------------------------------------------- */

int
getvalue(char*buf)
{
  int ret;
  int getparameter( char * );
  
  char *dup = strdup(buf), *p;
  for( p = dup; *p; p++ )
    if( *p == ' ' || *p == '\t')
      strcpy( p, p+1);
  if( isdigit( *dup ) || *dup == '-' )
    ret = atoi(dup);
  else
    ret = getparameter( dup );
  free(dup);
  return ret;
}
 

/* ----------------------------------------------------------------------
 * calculates the dimension of a variable from its () statement
 * ---------------------------------------------------------------------- */

void
calcvardimension( variable *var)
{
  char *buf, *tok, *up, *p;
  int dim = 1, onedim;
  if( var->dimension ) {
    buf = strdup( var->dimension );
    p = strchr(buf,')');
    if( p )
      p[1] = 0;
    if(! buf ) {
      fprintf(stderr,"f77reorder: out of memory. (6)\n");
      exit(1);
    }
    for( tok = buf; *tok; tok++ )
      if( *tok == ' ' || *tok == '\t' )
	strcpy( tok, tok+1);
    
    for( tok = strtok( buf, "(),"); tok ; tok = strtok(0,"(),")) {
      up = strchr(tok,':');
      if( up ) {
	*up = 0;
	onedim = getvalue( up+1) - getvalue( tok ) + 1;
      } else
	onedim = getvalue( tok);
      if( onedim <= 0 ) {
	fprintf(stderr,
		"f77reorder: Non positive array dimension in %s line %d\n"
		"    variable %s%s\n",
		cppfilename,line_counter-cpprefline+cppline,
		var->name,var->dimension);
	exit(1);
      }
      dim *= onedim;
    }
    free(buf);
  }
  var->idim = dim;
}


/* ----------------------------------------------------------------------
 * calculates the dimension of a single data field
 * ---------------------------------------------------------------------- */

int
dimdata( char **field )
{
  char *ptr = *field;
  char *mulstart = ptr;
  int inquotes = 0;
  int mult = 0;
  int dobreak = 0;
  while( *ptr && !dobreak) {
    switch( *ptr ) {
    case '\n':
      ptr = skip6(ptr+1);	/* line continuation */
      if( !mult )
	mulstart = ptr;
      break;
    case '*':
      if( !inquotes ) {
	if( !mult ) {
	  *ptr = 0;
	  mult = getvalue( mulstart);
	  *ptr = '*';
	  if( mult < 1 ) {
	    fprintf(stderr,
		    "f77reorder: non positive dimension "
		    "in data statement %s line %d: %s\n",
		    cppfilename,line_counter-cpprefline+cppline,
		    *field);
	    exit(1);
	  }
	}
      }
      ptr++;
      break;
    case ',':
      ptr++;
    case '/':
      if( !inquotes )
	dobreak = 1;
      else
	ptr++;
      break;
    case '\'':
      if( inquotes ) {
	if( ptr[1] == '\'' )
	  ptr++;
	else
	  inquotes = 0;
      } else
	inquotes = 1;
      ptr++;
      break;
    default:
      ptr++;
    }

  }
  if( ! mult ) 
    mult = 1;
  if( *ptr == 0 || *ptr == '/' )
    *field = 0;
  else
    *field = ptr;
  return mult;
}
	
/* ----------------------------------------------------------------------
 * calculates the dimension of a data statement
 * ---------------------------------------------------------------------- */

void
calcdatadimension( variable *var )
{
  int dim = 0;
  char *ptr;
  if( var->data ) {
    ptr = var->data+1;
    while( ptr ) {
      dim += dimdata( &ptr );
    }
  }
  var->ddim = dim;
}
  

/* ----------------------------------------------------------------------
 * calculates the dimensions (data+variable) for a variable list
 * ---------------------------------------------------------------------- */

void
calcdimensions( variable *var )
{
  for( ; var; var = var->next) {    
    calcvardimension(var);
    calcdatadimension(var);
  }
}

/* ----------------------------------------------------------------------
 * write out the variable list (without data statements included)
 * ---------------------------------------------------------------------- */

void
variableprint( char *type, variable *var, int number, char *filename )
{
  int len = 7 + strlen(type);
  int newlen;
  char buf[80];
  
  sprintf(buf,"      %s ", type);
  while(var) {
    if( ! var->name ) {
      fprintf(stderr,"f77reorder: syntax in %s line %d\n",
	      cppfilename,line_counter-cpprefline+cppline);
      exit(1);
    }
    newlen = strlen(var->name);
    if( var->dimension )
      newlen += strlen(var->dimension);
    if( var->next )
      newlen += 1;
    if( len + newlen > 72 && len > 7 ) {
      /* linebreak required */
      strcat( buf, "\n");
      _outputline( buf, number, filename);
      strcpy( buf, CONTINUE);
      len = 7;
      continue;
    }
    sprintf(strchr(buf,0),"%s%s%s", var->name,
	    var->dimension? var->dimension:"",
	    var->next?",":"");
    len += newlen;
    var = var->next;
  }
  strcat( buf, "\n");
  _outputline( buf, number, filename);
}

/* ----------------------------------------------------------------------
 * append the data statements in the variables to the data line list
 * ---------------------------------------------------------------------- */

dataline *
dataprint( variable *var, dataline *line)
{
  char buf[flinesize], *lbuf;
  variable *backvar;
  int thisdim, len, newlen;
  while( var ) {
    if( var->ddim ) {
      thisdim = 0;
      for( backvar = var; backvar; backvar = backvar->prev) {
	if( backvar != var && backvar->ddim > 0 ) {
	  fprintf(stderr,
		  "f77reorder: not enough initializers in %s line %d\n",
		  cppfilename,line_counter-cpprefline+cppline);
	  exit(1);
	}
	thisdim += backvar->idim;
	if( thisdim == var->ddim )
	  break;
	if( thisdim > var->ddim ) {
	  fprintf(stderr,
		  "f77reorder: Too many initializers in %s line %d\n",
		  cppfilename,line_counter-cpprefline+cppline);
	  exit(1);
	}
      }

      strcpy( buf, "      data ");
      len = strlen(buf);
      for(;;) {
	newlen = strlen(backvar->name);
	if( backvar != var)
	  newlen += 1;
	if( len + newlen > 72 && len > 7 ) {
	  /* linebreak required */
	  strcat( buf,"\n");
	  line = append( line, buf);
	  strcpy(buf, CONTINUE );
	  len = 7;
	  continue;
	}
	strcat( buf, backvar->name);
	if( backvar != var )
	  strcat(buf,",");
	len += newlen;
	if( var == backvar )
	  break;
	backvar = backvar->next;
      }
      strcat( buf,"\n");
      line = append( line, buf);
      if( strlen( var->data ) > flinesize - 10 ) 
	lbuf = malloc( strlen( var->data ) + 10 );
      else
	lbuf = buf;
      strcpy( buf, CONTINUE );
      strcat( buf, var->data);
      strcat( buf,"\n");
      line = append( line, buf);
      if( lbuf != buf )
	free(lbuf);
    }
    var = var->next;
  }
  return line;
}
	  

/* ----------------------------------------------------------------------
 * checks whether a line contains a function start
 * ---------------------------------------------------------------------- */

int
isafunction( char *text )
{
  text = skip6( text);
  text = skiptypedec( text);
  if( strncasecmp( text, "function", 8 ))
    return 0;
  return 1;
}
  

/* ----------------------------------------------------------------------
 * scans for data statements in variable declarations (where they don't
 * belong) but who knows about rules in this language...
 * ---------------------------------------------------------------------- */

state
getotherdata( char *text, dataline **line )
{
  int skiptype = 1;
  int acomment = 0;
  int needsredo = 0;
  char *type = 0;
  variable *variables = 0, *varlast = 0;
  dataline *intermed = 0, *walk;
  char *ptr;

  if( isafunction( text )) {
    freeparameter();
    return init_region;
  }
  
  for(;;) {

    if( acomment ) {
      outputline(text);
    } else {
      intermed = append(intermed, text);/* append current text to line list */ 
      ptr = skip6( text);
      while( ptr ) {
	ptr = parsevariables( ptr, &variables, skiptype, &type);
	skiptype = 0;
      }
    }
    if( fgetline( text) )
      return eof_found;

    if( isacomment(text) ) { /* is a comment */
      acomment = 1;
      continue;
    }
    acomment = 0;
    if( iscontinued( text))
      continue;
    else
      break;		 /* a new statement - return */
  }
  fungetline( text );    /* put the new statement in the unget buffer */

  varlast = 0;
  /* --- rewind --- */
  for( ; variables; variables = variables->prev ) {
    varlast = variables;
    if( variables->data )
      needsredo = 1;
    if( !variables->name ) {
      fprintf(stderr,"f77reorder: Syntax error in variable declaration:"
	      " %s line %d\n",
	      cppfilename,line_counter-cpprefline+cppline-1);
      exit(1);
    }
  }
  variables = varlast;
  keep_arraynames(variables);
  if( ! needsredo ) {		/* no reorder needed - spit out the lines */
    for( walk = intermed; walk; walk = walk->next)
      outputdata( walk);
  } else {
    calcdimensions( variables);
    variableprint( type, variables, intermed->number, intermed->filename );
    *line = dataprint( variables, *line);
  }
  freeline( intermed );    
  if( type )
    free(type);
  freevarlist( variables );
  return variable_def;
}


/* ----------------------------------------------------------------------
 * scan a line for parameter definitions and extract all the integer
 * variable definitions
 * ---------------------------------------------------------------------- */

typedef struct _ip
{
  char *name;
  int  value;
  struct _ip *next;
} iparameter;

iparameter *Phead = 0;


state
scanparameter( char *text )
{
  char *p, *q, *v;
  char *name;
  int  namelen;
  int  lineret;
  int  ival;
  int  hadbracket=0;
  char buffer[flinesize];
  iparameter *par;
  
  outputline(text);  
  strcpy( buffer, text);
  
  p = strchr(buffer,'(');
  for( ;; )  {

    if( p ) {
      hadbracket = 1;
      q = strchr(p,')');
    }

    while( p ) {
      p++;
      /* --- remove the spaces for our convenience --- */
      for( v = p; *v; v++ )
	if( *v == ' ' || *v == '\t') {
	  strcpy( v, v+1);
	  v--;
	}
      /* --- now search for the variable content - only look for integers */
      v = strchr( p,'=');
      name = p;
      namelen = v-p;
      if( v ) {
	v++;
	if( *v == '\'' ) {	/* a character contant */
	  for(;;) {
	    v = strchr(v+1,'\'');
	    if( v && v[1] == '\'' )
	      v++;
	    else
	      break;
	  }
	  if( v )
	    v = strchr(v+1,',');
	} else if( isdigit(*v) || *v == '-' ) {	/* an 'integer' */
	  ival = atoi(v);
	  par  = malloc( sizeof( iparameter));
	  if(!par ) {
	    fprintf(stderr,"f77reorder: out of memory. (7)\n");
	    exit(1);
	  }
	  par->value = ival;
	  par->name = malloc( namelen+1);
	  if(!par->name ) {
	    fprintf(stderr,"f77reorder: out of memory. (8)\n");
	    exit(1);
	  }
	  strncpy( par->name, name, namelen)[namelen] = 0;
	  par->next = Phead;
	  Phead = par;
	  if( v )
	    v = strchr(v+1,',');
	}
      }
      if( v )
	p = v;
      else
	p = 0;
    }
    while( !(lineret = fgetline( text) )) {
      if( isacomment(text) ) {
	outputline(text);
	continue;
      } else
	break;
    }
    if( lineret )
      return eof_found;
    if( iscontinued( text)) {
      outputline(text);
      strcpy( buffer, text);
      p = skip6(buffer);
      if( ! hadbracket )
	p = strchr(text,'(');
      continue;
    } else
      break;			/* a new statement - return */
  }
  fungetline( text );	/* put the new statement in the unget buffer */
  return variable_def;
}

/* ----------------------------------------------------------------------
 * return the integer content of a parameter. If not defined - exit.
 * ---------------------------------------------------------------------- */

int
getparameter( char *pname )
{
  iparameter *list;
  
  for( list = Phead; list; list = list->next ) {
    if( !strf77cmp( pname, list->name, 0) )
      return list->value;
  }
  fprintf(stderr,"f77reorder: Unknown parameter '%s' in %s line %d\n",
	  pname, cppfilename,line_counter-cpprefline+cppline);
  exit(1);
  return 0;
}


/* ----------------------------------------------------------------------
 * clean the parameter list at a new function start
 * ---------------------------------------------------------------------- */

void
freeparameter( )
{
  iparameter *list, *next;
  arrayname *alist, *anext;
  extern int inflow;

  inflow = 0;
  for( list = Phead; list; list = next ) {
    next = list->next;
    free(list->name);
    free(list);
  }
  Phead = 0;
  for( alist = Ahead; alist; alist = anext ) {
    anext = alist->next;
    free(alist);
  }
  Ahead = 0;
}

/* ----------------------------------------------------------------------
 * function to be called at a new subroutine/... start - cleans the
 * defined parameters
 * ---------------------------------------------------------------------- */

state
subroutinestart(void)
{
  freeparameter();
  return init_region;
}
  

/* ----------------------------------------------------------------------
 * scan a f77 code line in `text' for its content. If it is a DATA
 * statement, append the data statement to the dataline list
 * The status of the f77 grammar (see above) is returned.
 * ---------------------------------------------------------------------- */

state
scanline( char *text, dataline **line)
{
  int i;
  char *f;
  if( isacomment(text) )              /* is a comment */
    return comment;
  for( i = 0; i < sizeof( before) / sizeof(before[0]); i ++ ) 
    if( firsttag( text, before[i].name) ) { /* an initializer line? */
      switch( before[i].type) {
      case f77variable:
	f = skiptypedec( text);
	if( strf77cmp( f, "function", 0 ) != 0 ) {
	  if( line ) 
	    return getotherdata( text, line);
	  else
	    return flow_region;	/* we are already in the flow region */
	}
	/* no break */
      case f77substart:
	return subroutinestart( );
      case f77parameter:
	if( line )
	  return scanparameter( text );
	else
	  return flow_region;
      case f77common:
	if( line )
	  return getcommonline( text );
	else
	  return flow_region;
      case f77entry:
	return entry;
      default:  
	return init_region;
      }
    }
  if( line && firsttag( text, "data" ) ) { /* a data statement? */
    return getdataline( text, line );     
  }
  
  if( text[0] != '\t' && strncmp(text,"      ",6)) /* a continued line */
    return continued;

  if( isstatementfun(text) )
    return statementfun;

  if( islabel(text) )		/* a label can only be found in the */
    return flow_region;		/* flow_region */
  
  return flow_region;		/* everything else is an execution statement */
}

/* ----------------------------------------------------------------------
 * the main function. Reads always from stdin and writes to stdout
 * argument passing to the program is not yet fully enabled ,-)
 * -----------------------------------------------------------------------
 * strongly relies on f77 style input and my understanding of f77
 * grammer.
 * ---------------------------------------------------------------------- */

int
main(int argc, char **argv) {
  char text[flinesize];
  int i;
  extern int  inflow;
  state st = init_region;	/* a f77 starts always with some kind
				 * of initialization*/
  dataline *line = 0, *walk;

  for( i = 1; i < argc; i++ ) {
    if( !strcmp(argv[i],"--line") && argc > i + 1 ) {
      linefile=fopen( argv[++i], "w");
      if( !linefile ) {
	perror( argv[++i] );
	exit(1);
      }
    } else if( !strcmp(argv[i],"--file") && argc > i + 1 ) {
      cppfilename = filename = argv[++i];
    } else if( !strcmp(argv[i],"--commainsert")) {
      commainsert = 1;
    } else {
      fprintf( stderr,"f77reorder: Unknown parameter %s\n",
	       argv[i]);
    }
  }

  inflow = 0;
  infile = stdin;
  outfile = stdout;

  while( !fgetline( text ) && st != eof_found ) { /* read a line until the
						   * end of input */
    switch( st ) {		/* the state machine */
    case init_region:		/* if the current state is in the */
    case data_statement:	/* initialization region */
    case common_statement:
    case variable_def:
    case comment:
    case continued:
      st = scanline( text, &line); /* scan for DATA lines and keep them */
      if( inflow && (st == variable_def ||
		     st == common_statement )) {
	fprintf( stderr,
		 "f77reorder: Declaration statement among "
		 "executables - %s line %d\n",
		 cppfilename,line_counter-cpprefline+cppline-1);

	exit(1);
      }
      switch( st ) {
      case data_statement:
      case variable_def:
      case common_statement:
      case eof_found:	       
	break;
      case flow_region:		/* reached the flow region - spit out
				 * the data statements first */
	for( walk = line; walk; walk = walk->next)
	  outputdata( walk);
	freeline( line );
	line = 0;
	/* no break */
      case init_region:
      case entry:
      case comment:
      case statementfun:
      case continued:
	outputline(text); /* then write out the current read line */
	if( st == statementfun || st == entry)
	  st = init_region;
	break;
      }
      break;
    case flow_region:		/* in the flow region - just write what
				 * was read in */
      outputline(text);
      inflow = 1;
      if( ( st = scanline( text, 0) ) == init_region ) /* and check for the */
	st = init_region;	/* next subroutine */
      if( ( st == variable_def ||
	    st == common_statement )) {
	fprintf( stderr,
		 "f77reorder: Declaration statement among "
		 "executables - %s line %d\n",
		 cppfilename,line_counter-cpprefline+cppline-1);

	exit(1);
      }
      if( st == statementfun || st == entry )
	st = flow_region;
	
      break;
    default:
      break;
    }
  }
  if( linefile )
    fclose(linefile);
  return 0;
}
  
