(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                propre.ml                                 *)
(****************************************************************************)

open Pp;;

open Std;;
open Names;;
open Vectops;;
open Term;;
open Mach;;
open Trad;;
open Generic;;
open Printer;;
open Reduction;;
open Himsg;;
open Constrtypes;;
open Environ;;
open Termenv;;
open Machops;;
open Typing;;
open CoqAst;;
open Ast;;
open Pfedit;;
open Command;;
open Vernacinterp;;


(* ====== De ProPre a Coq ====== *)
(* ============================= *)

(* === A FAIRE :
   - Gerer les variables implicites
   - Utiliser Match lorsque Match_rec est inutile
   - Autoriser les parametres fonctionnels
*)

(* === PREREQUIS :
   - La clause 'Recursive Definition' fait appel aux
   constructeurs 'refl_equal' et 'refl_eqT' (cf fonction
   "refl_eqC")
*)

(* === BUGS :
   ...
*)

(* === Exceptions *)
exception ProPre_error of (int * std_ppcmds)
;;

exception Error_arbrec
;;

exception Error_filtre
;;

let perror (n,pps) = raise (ProPre_error (n,pps))
;;

(* === futil *)
let rec mapapp f = function
  [] -> []
 |x::xs -> (f x)@(mapapp f xs)
;;

let cmap f aas bs = List.map f (List.combine aas bs)
;;

let coupe_list_n n l = 
 let rec cln_rec ys = function
  0,_ -> anomaly "coupe_list_n(0)"
 |1,(x::xs) -> (List.rev ys,x,xs)
 |n,(x::xs) -> cln_rec (x::ys) (pred n,xs)
 |_ -> anomaly "coupe_list_n"
 in cln_rec [] (n,l)
 
;;

let map3 f l1 l2 l3  = 
 let rec map3_rec = function
 [],[],[] -> []
|x1::l1',x2::l2',x3::l3'
 -> (f x1 x2 x3)::(map3_rec (l1',l2',l3'))
| _ -> anomaly "map3"
 in map3_rec (l1,l2,l3)

;;

let map4 f l1 l2 l3 l4 = 
 let rec map4_rec = function
 [],[],[],[] -> []
|x1::l1',x2::l2',x3::l3',x4::l4'
 -> (f x1 x2 x3 x4)::(map4_rec (l1',l2',l3',l4'))
| _ -> anomaly "map4"
 in map4_rec (l1,l2,l3,l4)

;;

let map5 f l1 l2 l3 l4 l5 = 
 let rec map5_rec = function
 [],[],[],[],[] -> []
|x1::l1',x2::l2',x3::l3',x4::l4',x5::l5'
 -> (f x1 x2 x3 x4 x5)::(map5_rec (l1',l2',l3',l4',l5'))
| _ -> anomaly "map5"
 in map5_rec (l1,l2,l3,l4,l5)

;;

let split' xyzs = 
 let rec split_rec xs ys zs = function
 [] -> (List.rev xs,List.rev ys, List.rev zs)
|(x,y,z)::xyzs' -> split_rec (x::xs) (y::ys) (z::zs) xyzs'
 in split_rec [] [] [] xyzs

;;

let rec mapconc s f = function
  [] -> ""
 |x::[] -> f x
 |x::xs -> (f x)^s^(mapconc s f xs)
;;

(* map_or :
*)
let rec map_or f = function
 [] -> false
|x::xs -> (f x)or(map_or f xs)
;;

(* map_and : 
   ('a -> bool) -> 'a list -> bool *)
let rec map_and f = function
 [] -> true
|x::xs -> (f x)&(map_and f xs)
;;

(* map2_and : 
   ('a -> 'b -> bool) -> 'a list -> 'b list -> bool *)
let rec map2_and f = fun
 p_0 p_1 -> match p_0,p_1 with ([], []) -> true
|((x::xs), (y::ys)) -> (f x y)&(map2_and f xs ys)
|(_, _) -> anomaly "map2_and"
;;

(* rem_assoc : 
   'a -> ('a * 'b) list -> ('a * 'b) list *)
let rec rem_assoc x = function
 [] -> []
|(y,z)::s -> if x=y then s else (y,z)::(rem_assoc x s)
;;

(* except_last : 
   'a list -> 'a list *)
let rec except_last = function
 [] -> []
|[x] -> []
|x::xs -> x::(except_last xs)
;;

(* add_last : 
   'a -> 'a list -> 'a list *)
let rec add_last x = function
 [] -> [x]
|x'::xs -> x'::(add_last x xs)
;;

(* n_tl : 
   int -> 'a list -> 'a list *)
let rec n_tl = fun
 p_0 p_1 -> match p_0,p_1 with (0, xs) -> xs
|(n, (x::xs)) -> n_tl (pred n) xs
|(_, _) -> anomaly "n_tl"
;;

(* mkset : 
  'a list -> 'a list *)
let rec mkset = function
 [] -> []
|x::xs -> if (List.mem x xs) then mkset xs else x::(mkset xs)
;;

(* string_of_name : 
   name -> string *)
let string_of_name = function
 Name id -> string_of_id id
|_ -> anomaly "string_of_name"
;;

(* id_of_ints : 
   string -> int list -> identifier *)
let id_of_ints s x =
 id_of_string (s^(mapconc "_" string_of_int x))
;;

(* null_id : 
   identifier *)
let null_id = id_of_string "..."
;;

(* id_of_name : 
   name -> identifier *)
let id_of_name = function
 Anonymous -> null_id
|Name (x) -> x
;;

type global_name = GLOBALNAME of constr;;

let mkPropC ctnts = ope("PROP",[ide ctnts ]);;
let mkLambdaC(id,t,u) = ope("LAMBDA",[t;slam(Some (string_of_id id),u)]);;
let mkProdC(id,t,u) = ope("PROD",[t;slam(Some (string_of_id id),u)]);;
let mkArrowC(t,u) = ope("PROD",[t;slam(None,u)]);;
let mkAppLC(f,l) = ope("APPLIST",f::l);;
let mkAppC(f,a) = mkAppLC(f,[a]);;
let mkRecC(pred,c,lf) = ope("XTRA", (str "REC")::pred::c::lf);;

(* === De 'construct' a 'etype' *)
type etype =
  Eset
 |Eprop
 |Evar of identifier
 |Ecste of global_name * etype list
 |Eprod of identifier * etype
 |Efleche of etype list * etype
;;

(* etype_of_constr_aux : 
   etype list * etype -> etype *)
let etype_of_constr_aux = function
 [],t -> t
|ts,t -> Efleche(List.rev ts,t)
;;

(* Evar0 : 
   etype *)
let evar0 = Evar null_id
;;

(* etype_of_constr : 
   etype list -> etype list -> etype list -> sorts oper term 
   -> etype *)
let rec etype_of_constr env args_stack  prem_stack = function
 Rel n -> etype_of_constr_aux(prem_stack, List.nth env (n-1))
|DOP0(Sort(Prop Pos)) -> etype_of_constr_aux(prem_stack,Eset)
|DOP0(Sort(Prop Null)) -> etype_of_constr_aux(prem_stack,Eprop)
|VAR(id) as x
 ->  etype_of_constr_aux(prem_stack,Ecste(GLOBALNAME x,args_stack))
|DOPN(Const sp,_) as x
 ->  etype_of_constr_aux(prem_stack,Ecste(GLOBALNAME x,args_stack))
|DOPN(MutInd (x_0,x_1),_) as x
 ->  etype_of_constr_aux(prem_stack,Ecste(GLOBALNAME x,args_stack))
|DOPN(AppL,ts) 
 -> let us = list_of_tl_vect ts in
    let u =  hd_vect ts in
     etype_of_constr env ((List.map (etype_of_constr env [] []) us)@args_stack) prem_stack u
|DOP2(Prod,t1,DLAM(Anonymous,t2))
 -> let t = etype_of_constr env [] [] t1 in
     (etype_of_constr (evar0::env) args_stack (t::prem_stack) t2)
|DOP2(Prod,t1,DLAM(Name id,t2))
 -> (let x = Evar id in 
     let t = etype_of_constr env [] [] t1 in
     if (dependent (Rel 1) t2) then 
      (etype_of_constr (x::env) args_stack ((Eprod(id,t))::prem_stack) t2)
     else
      (etype_of_constr (evar0::env) args_stack (t::prem_stack) t2))
|DOP2(Cast,c,_) -> etype_of_constr env args_stack prem_stack c
|_ -> anomaly "etype_of_constr"
;;

(* etype_of : 
   constr signature -> identifier -> etype list * etype *)
let etype_of sign x =
let j = fconstruct (Evd.mt_evd()) sign (nvar (string_of_id x)) in
 match j._VAL with
  (DOPN(Const _,_)|VAR _ |DOPN(MutInd _,_)|DOPN(MutConstruct _,_))
   -> let t = j._TYPE in
      (match (etype_of_constr [] [] [] t) with
        Efleche(ts,t) -> (ts,t)
       |t -> ([],t))
  | _ -> anomaly "etype_of"
;;

(* === Controle types fonctions *)
(* check_params_f : 
   constr signature -> (identifier * ast) list 
   -> (ast -> ast) *) 
let rec check_params_rec sign funC = function
 [] -> funC
|(x,t)::xTs
 -> (fconstruct (Evd.mt_evd()) sign (funC t);
     check_params_rec sign (fun x_0 -> funC (mkProdC(x,t,x_0))) xTs)
;;

let check_params_f sign xSs = 
 check_params_rec sign (fun x -> x) xSs
;;

(* check_type_exprC :
   ast -> ast *)
let rec check_type_exprC ast =
  match ast with
      Nvar _ -> ast
    | Node(loc,"APPLIST",ts) -> Node(loc,"APPLIST",List.map check_type_exprC ts) 
    | x -> perror(1,[<'sTR"bad type expression \""; gencompr CCI x;
                      'sTR"\"">])
;;

(* check_type_f' : 
   constr signature -> (ast -> ast) -> ast -> ast *)
let rec check_type_f' sign funC = function
    Node(_,"PROP",[Id(_,x)]) as t ->
      (match x with
           "Null" -> funC (mkPropC "Null")
         |_ -> perror(1,[<'sTR"bad type expression \""; gencompr CCI t;
                          'sTR"\".">]))
  | Node(_,"PROD",[t;Slam(_,None,u)]) ->
      let t' = funC (check_type_exprC t) in
        (match (fconstruct (Evd.mt_evd()) sign t')._TYPE with
             DOP0(Sort(Prop Pos)) ->
               check_type_f' sign (fun x -> funC (mkArrowC(t,x))) u
           | _ -> perror(1,[<'sTR"type of "; gencompr CCI t;
                             'sTR" is not of type Set">]))
  | t ->
      let t' = funC (check_type_exprC t) in
        (match (fconstruct (Evd.mt_evd()) sign t')._TYPE with
             DOP0(Sort(Prop Pos)) -> t'
           |_ -> perror(1,[<'sTR"type of ";  gencompr CCI t;
                            'sTR" is not of type Set">]))
;;

(* check_type_f : 
   constr signature -> (identifier * ast) list -> ast -> ast *)
let rec check_type_f sign xSs t =
try
  check_type_f' sign (check_params_f sign xSs) t
with
 ProPre_error(0,x) -> perror(20,[<'sTR"(parameters list) :\n"; x; 'sTR".">])
|ProPre_error(_,x) -> perror(20,[<'sTR"(type specification) :\n"; x; 'sTR".">])
|UserError(_,x) -> perror(20,[<'sTR"(type specification) :\n"; x; 'sTR".">])
|_ -> anomaly "check_type_f"
;;
 
(* === Des 'command's aux 'xterm's*)
type xterm = 
  Xvar of identifier
 |Xcc of identifier
 |Xcf of identifier * xterm list
 |Xct of identifier
 |Xft of identifier * xterm list
 |Xfv of identifier * xterm list
 |Xfx of identifier * xterm list
 |Xind of identifier * xterm list
;;

type symb = Is_var | Is_cc | Is_cf | Is_ft | Is_ct | Is_fx | Is_ind | Is_x;;

(* is_var : 
   constr signature -> string -> bool *)
let is_var sign x = 
 try (search_reference (gLOB sign) (id_of_string x);false)
 with (Not_found|UserError _) -> true
;;

(* symb_of_rec : constr signature -> sorts oper term -> symb *)
let rec symb_of_rec sign = function
 DOPN(MutConstruct _,_) as cx 
 -> (match whd_betaiota (type_of (Evd.mt_evd()) sign cx) with
      DOP2(Prod,_,_) -> Is_cf
     |_ -> Is_cc)
|DOPN(MutInd _,_) -> Is_ind
|DOPN(Const _,_) as k -> Is_ft
|VAR id ->  let val_0 = (fconstruct (Evd.mt_evd()) sign (nvar (string_of_id id)))._VAL in
   begin match val_0 with VAR id' -> Is_var | _ -> symb_of_rec sign val_0 end
|_ -> Is_x
;;

(* symb_of : 
   identifier -> identifier -> identifier list 
   -> constr signature -> symb *)
let rec symb_of x fx xs sign = 
try
(if x=fx then
  Is_fx
 else
  if (List.mem x xs) then
   Is_var
  else
   try 
    symb_of_rec sign (global (gLOB sign) x)
   with 
    _ -> Is_var)
with
 UserError _ 
 -> perror(31, [< 'sTR"unknown symbol \""; 'sTR(string_of_id x); 'sTR"\"" >])
|_ -> anomaly "symb_of"
;;

(* add_last_arg : 
   xterm -> xterm -> xterm *)
let add_last_arg t = function 
 Xcf(x,ts) -> Xcf(x,add_last t ts)
|Xft(x,ts) -> Xft(x,add_last t ts)
|Xfv(x,ts) -> Xfv(x,add_last t ts)
|Xfx(x,ts) -> Xfx(x,add_last t ts)
|Xind(x,ts) -> Xind(x,add_last t ts)
|_ -> anomaly "add_last_arg"
;;

(* xterm_of_command : 
   identifier -> identifier list -> constr signature 
   -> ast -> xterm *)
let rec xterm_of_command fx xs sign = function
    Nvar(_,x) ->
      let idx = id_of_string x in
        (match (symb_of idx fx xs sign) with
             Is_var -> Xvar idx
           |Is_cc  -> Xcc idx
           |Is_ind -> Xind (idx,[])
           |Is_ft -> Xct idx (* devrait etre Is_ct (cf symb_of_rec) *)
           |_ -> perror(8,[<'sTR"illegal constant \""; 'sTR x; 'sTR"\"">]))
  | Node(_,"APPLIST",(Nvar(_,x))::ts) as t ->
      let idx = id_of_string x in
        (match (symb_of idx fx xs sign) with
             Is_var -> Xfv(idx,List.map (xterm_of_command fx xs sign) ts)
           |Is_cf -> Xcf(idx,List.map (xterm_of_command fx xs sign) ts)
           |Is_ft -> Xft(idx,List.map (xterm_of_command fx xs sign) ts)
           |Is_fx -> Xfx(idx,List.map (xterm_of_command fx xs sign) ts)
           |Is_ind -> Xind(idx,List.map (xterm_of_command fx xs sign) ts)
           |_ -> perror(8,[<'sTR"illegal constant \""; 'sTR x; 'sTR"\" in \"";
                            gencompr CCI t; 'sTR"\"">]))
  | t -> perror(3,[<'sTR"illegal expression \""; gencompr CCI t; 'sTR"\"">])
;; 

(* === Controle equations *)
(* flat_AppC : 
   ast -> string * ast list *)
let flat_AppC t = 
 let rec flat_AppC_rec ll = function
     Nvar(_,x) -> (x,ll)
   | Node(_,"APPLIST",t::l) -> flat_AppC_rec (l@ll) t
   | t ->  perror(3,[< 'sTR"illegal expression " ; gencompr CCI t >])
 in flat_AppC_rec [] t
 
;;

(* id_of_contents : 
   contents -> identifier *)
let string_of_contents c =
    match c with
    Pos -> "Pos"
  | Null -> "Null"
;;

(* valC : 
   name list -> sorts oper term -> ast *)
let rec valC e = function
    Rel n -> nvar (string_of_id (id_of_name (List.nth e (n-1))))
  | VAR id -> nvar (string_of_id id)
  | DOP0(Sort(Prop c)) -> ope("PROP",[ide (string_of_contents c)])
  | DOP2(Prod,t1,DLAM(Anonymous,t2)) ->
      ope("PROD",[valC e t1;slam(None,valC (Anonymous::e) t2)])
  | DOP2(Prod,t1,DLAM(Name id,t2)) ->
      ope("PROD",[valC e t1;slam(Some (string_of_id id),
                                 valC ((Name id)::e) t2)])
  | DOPN(AppL,cl) -> ope("APPLIST",map_vect_list (valC e) cl)
  | DOPN(MutInd _ as x,_) -> nvar (string_of_id (id_of_global x))
  | DOPN((MutConstruct(_) as val_0),_) -> nvar (string_of_id (id_of_global val_0))
  | DOPN(Const sp,_) -> nvar (string_of_id (basename sp))
  | t ->
      let pt = pTERM t in perror(2,[< 'sTR"illegal type expression " ; pt >])
;;

(* flatC_Prod : 
   string -> sorts oper term -> (name * ast) list * ast *)
let rec flatC_Prod_rec e ts = function
 DOP2(Prod,t,DLAM(x,u))
 -> flatC_Prod_rec (x::e) ((x,valC e t)::ts) u
|t -> (List.rev ts,valC e t)
;;

let flatC_Prod x t = 
try
 flatC_Prod_rec [] [] t
with
  ProPre_error(2,y) 
  -> perror(0,[<y; 'sTR" while checking type of "; 'sTR(x)>])
;;

(* Subst_idC : 
   ast -> identifier -> ast -> ast *)
(* Et la capture de variables ? *)
let rec subst_idC y x ast =
  match ast with
    Node(lp,"PROP",[c]) -> Node(lp,"PROP",[c])
  | Nvar(lv,z) -> if x=(id_of_string z) then y else ast
  | Node(la,"APPLIST",l) -> Node(la,"APPLIST",List.map (subst_idC y x) l)
  | Node(lp,"PROD",[t;Slam(ls,None,u)]) ->
      Node(lp,"PROD",[subst_idC y x t; Slam(ls,None,subst_idC y x u)])
  | Node(lp,"PROD",[t;Slam(ls,Some na,u)]) ->
      if x=(id_of_string na) then ast
      else Node(lp,"PROD",[subst_idC y x t;Slam(ls,Some na,subst_idC y x u)])
  |_ -> anomaly "subst_idC"
;;

(* it_subst_idC :
   ast -> (name * 'a) list -> ast list -> ast *)
let rec it_subst_idC t = fun
 p_0 p_1 -> match p_0,p_1 with ([], []) -> t
|(((Name y,_)::xTs), (t_0::ts))
 -> it_subst_idC (subst_idC t_0 y t) xTs ts
|((_::xTs), (_::ts)) -> it_subst_idC t xTs ts
| (_, _) -> anomaly "it_subst_idC"
;;

(* typed_vars : 
   constr signature -> ast list -> (name * ast) list 
   -> (identifier * ast) list *)
let rec typed_vars sign astl na =
  match (astl,na) with
      ([], []) -> []
    | (((Nvar(_,y) as vy)::ts), ((Name x,t)::xTs)) ->
        let xTs' = List.map (fun (x',t') -> (x',subst_idC vy x t')) xTs in
          if (is_var sign y) then (id_of_string y,t)::(typed_vars sign ts xTs')
          else (typed_vars sign ts xTs')
    | (((Nvar(_,y) as vy)::ts), ((_,t)::xTs)) ->
        if (is_var sign y) then (id_of_string y,t)::(typed_vars sign ts xTs)
        else (typed_vars sign ts xTs)
    | ((t::ts), ((Name x,_)::xTs)) ->
        let xTs' = List.map (fun (x',t') -> (x',subst_idC t x t')) xTs in
        let (f,ts') = flat_AppC t in
        let (xTs'',_) =
          flatC_Prod f ((fconstruct (Evd.mt_evd()) sign (nvar f))._TYPE) in
          typed_vars sign (ts'@ts) (xTs''@xTs')
    | ((t::ts), (_::xTs)) ->
        let (f,ts') = flat_AppC t in
        let (xTs',_) =
          flatC_Prod f ((fconstruct (Evd.mt_evd()) sign (nvar f))._TYPE) in
          typed_vars sign (ts'@ts) (xTs'@xTs)
    | ([], _) -> perror(5,[<'sTR"missing argument(s)">])
    | (_, []) -> perror(6,[<'sTR"extra argument(s)">])
;;

(* AppC_fx :
   identifier -> ast list -> ast *)
let appC_fx fx = function
 [] -> anomaly "AppC_fx"
|t::ts -> 
 let rec appC_rec t = function
 [] -> t
|t'::ts -> appC_rec (mkAppC(t,t')) ts
 in appC_rec (mkAppC(nvar (string_of_id fx),t)) ts

;;

(* refl_eqC : 
   sorts oper term -> ast *)
let refl_eqC = function
 DOP0(Sort(Type(_))) -> nvar ("refl_eqT")
|DOP0(Sort(Prop Pos)) -> nvar ("refl_equal")
|_ -> anomaly "refl_eqC"
;;

(* eqC :  
   ast -> ast -> ast -> sorts oper term -> ast *)
let eqC t u t_0 = function
    DOP0(Sort(Type(_))) -> ope("APPLIST",[nvar("eqT");t_0; t; u])
  | DOP0(Sort(Prop Pos)) -> ope("APPLIST",[nvar("eq"); t_0; t; u])
|_ -> anomaly "eqC"
;;

(* closeC : 
   ast -> (identifier * ast) list -> ast *)
let rec closeC t = function
 [] -> t
|(x,t_0)::xTs -> mkProdC(x,t_0,closeC t xTs)
;;


(* check_left : 
   xterm list -> xterm list *)
let rec check_left = function
 [] -> []
|(Xvar x)::ts -> (Xvar x)::(check_left ts)
|(Xcc x)::ts -> (Xcc x)::(check_left ts)
|(Xcf(x,us))::ts -> (Xcf(x,check_left us))::(check_left ts)
|(Xind (x,us))::ts -> (Xind (x,check_left us))::(check_left ts)
|(Xct x)::_ -> perror(9,[< 'sTR"illegal symbol \"" ; 'sTR (string_of_id x) ; 'sTR"\" on left of \"=>\"" >]) 
|(Xft(f,_))::_ -> perror(9,[< 'sTR"illegal symbol \"" ; 'sTR (string_of_id f) ; 'sTR"\" on left of \"=>\"" >])
|(Xfv(f,_))::_ -> perror(9,[< 'sTR"illegal symbol \"" ; 'sTR (string_of_id f) ; 'sTR"\" on left of \"=>\"" >])
|(Xfx(f,_))::_ -> perror(9,[< 'sTR"illegal symbol \"" ; 'sTR (string_of_id f) ; 'sTR"\" on left of \"=>\"" >])
;;

(* check_vars_eq : 
   int -> identifier list -> (identifier * ast) list 
   -> (identifier * ast) list *)
let check_vars_eq i id_params_f typed_vars_eq =
 
 let rec check_vars_eq_rec xTs_acc = function
 [] -> List.rev xTs_acc
|(x,(Node(_,"PROP",[p]) as propp))::xTs
 -> if (List.mem x id_params_f) then
     if (List.mem_assoc x xTs_acc) then
       check_vars_eq_rec xTs_acc xTs
     else
       check_vars_eq_rec ((x,propp)::xTs_acc) xTs
    else
     perror(0,[<'sTR"(equation "; 'sTR(string_of_int i); 'sTR") :\n variable \""; 'sTR(string_of_id x); 'sTR"\" should be declared in parameters list">])
|(x,t)::xTs
 -> if (List.mem_assoc x xTs_acc) then
     if (List.mem x id_params_f) then
      perror(0,[< 'sTR"(equation "; 'sTR(string_of_int i); 'sTR") :\n parameter \"";
                  'sTR(string_of_id x); 'sTR"\" can't occur on left of \"=>\"">])
     else
      perror(0,[<'sTR"(equation "; 'sTR(string_of_int i); 'sTR") :\n variable \""; 
                 'sTR(string_of_id x); 
                 'sTR"\" can't have more than one occurence on left of \"=>\"">])
     else
      check_vars_eq_rec ((x,t)::xTs_acc) xTs
 in check_vars_eq_rec [] typed_vars_eq

;;

(* check_vars_eqs : 
   constr signature -> identifier list -> (name * ast) list 
   -> (ast list * 'a) list -> (identifier * ast) list list *)
let rec check_vars_eqs sign id_params_f typed_args_fC eqs_fC =
 
 let rec check_vars_eqs_rec i = function
 [] -> []
|(ts,_)::eqs
 -> try 
     (check_vars_eq i id_params_f (typed_vars sign ts typed_args_fC))
     ::(check_vars_eqs_rec (succ i) eqs)
    with
     ProPre_error(0,x) -> perror(0,x) 
    |ProPre_error(_,x) -> perror(0,[< 'sTR"(equation "; 'sTR(string_of_int i); 'sTR") :\n "; x>])
 in check_vars_eqs_rec 1 eqs_fC

;;

(* check_type_eqs : 
   constr signature -> ast list -> constr list *)
let check_type_eqs sign eqs_f =
 
 let rec check_type_eqs_rec i = function
 [] -> []
|e::es
 ->((try 
      fconstruct (Evd.mt_evd()) sign e
    with
     (UserError (_,x)) 
     -> perror(0,[<'sTR"(equation "; 'sTR(string_of_int i); 'sTR") :\n "; x>])
     | _ -> anomaly "check_type_eqs");
      check_type_eqs_rec (succ i) es)
 in check_type_eqs_rec 1 eqs_f

;;

(* pair_of_eq : 
   ast -> ast list * ast *)
(*===
let rec pair_of_eq = function
 ASTnode("PROD",[_;(ASTslam(_,t))]) -> pair_of_eq t
|ASTnode("APPLIST",[_;_;ASTnode("APPLIST",_::ts);u]) -> (ts,u)
|_ -> anomaly "pair_of_eq"
;;
===*)

(* xterm_of_eqs : 
   constr signature -> identifier -> identifier list 
   -> ast list -> (xterm list * xterm) list *)
let xterm_of_eqs sign id_params_f nom_f eqs_fC =
 
 let rec xterm_of_eqs_rec i = function
 [] -> []
|(ts,u)::eqs
 -> try 
     (let ts' = List.map (xterm_of_command id_params_f nom_f sign) ts in
       (check_left ts',xterm_of_command id_params_f nom_f sign u)
       ::(xterm_of_eqs_rec (succ i) eqs))
    with
     ProPre_error(_,x) -> perror(0,[<'sTR"(equation "; 'sTR(string_of_int i); 'sTR") :\n"; x>])
 in xterm_of_eqs_rec 1 eqs_fC

;;     

(* === Etiquetage des equations  *)
type etiqt = L | De | D | C | N
;;

type etiql = Nl | Ll | Vl
;;

type eterm = 
  Qvar of etiqt * identifier
 |Qcc of etiqt * identifier
 |Qcf of etiqt * identifier * eterm list
 |Qct of etiqt * identifier
 |Qft of etiqt * identifier * eterm list
 |Qfv of etiqt * identifier * eterm list
 |Qfx of etiqt * identifier * eterm list
 |Qind of etiqt * identifier * eterm list
;;
 
(* arg_par_rec : 
   'a -> 'b list * 'a list -> 'b list *)
let rec arg_par_rec d = function
 [],[] -> []
|(t::ts),(di::dis)
   -> if d=di then (arg_par_rec d (ts,dis)) else t::(arg_par_rec d (ts,dis))
|_ -> anomaly "arg_par_rec"
;;

(* arg_par : 
   identifier -> 'a list -> constr signature -> 'a list *)
let arg_par c ts sign =
 let (dis,d) = etype_of sign c in
 arg_par_rec d (ts,dis) 
;;

(* arg_ind_rec :
   'a -> 'b list * 'a list -> 'b list *)
let rec arg_ind_rec d = function
 [],[] -> []
|(t::ts),(di::dis)
   -> if d=di then t::(arg_ind_rec d (ts,dis)) else (arg_ind_rec d (ts,dis))
|_ -> anomaly "arg_ind_rec"
;;

(* arg_ind : 
   identifier -> 'a list -> constr signature -> 'a list *)
let arg_ind c ts sign =
 let (dis,d) = etype_of sign c in
 arg_ind_rec d (ts,dis) 
;;

(* appels_ind : 
   xterm -> xterm list list *)
let rec appels_ind = function
 Xfx(_,ts) -> ts::(appels_inds ts)
|Xft(_,ts) -> (appels_inds ts)
|Xfv(_,ts) -> (appels_inds ts)
|Xcf(_,ts) -> (appels_inds ts)
|Xind(_,ts) -> (appels_inds ts)
|_ -> []

(* appels_inds : 
   xterm list -> xterm list list *)
and appels_inds = function
  [] -> []
 |t::ts 
  -> (appels_ind t)@(appels_inds ts)
;;

(* etier : 
   etiqt -> xterm list -> eterm list *)
let rec etier e ts = 
 let rec etier0 = function
 Xvar x -> Qvar(e,x)
|Xcc c -> Qcc(e,c)
|Xct c -> Qct(e,c)
|Xcf(c,ts') -> Qcf(e,c,etier e ts')
|Xft(f,ts') -> Qft(e,f,etier e ts')
|Xfv(f,ts') -> Qfv(e,f,etier e ts')
|Xfx(f,ts') -> Qfx(e,f,etier e ts')
|Xind (x,xs) -> Qind(e,x,etier e xs)
 in List.map etier0  ts

;;

(* est_D : 
   eterm -> bool *)
let est_D = function
 Qvar(De,_) -> true
|Qvar(D,_) -> true
|Qcc(De,_) -> true
|Qcc(D,_) -> true
|Qcf(De,_,_) -> true
|Qcf(D,_,_) -> true
|_ -> false
;;

(* est_De : 
   eterm -> bool *)
let est_De = function
  Qvar(De,_) -> true
 |Qcc(De,_) -> true
 |Qcf(De,_,_) -> true
 |_ -> false
;;

(* est_C : 
   eterm -> bool *)
let est_C = function
  Qvar(C,_) -> true
 |Qcc(C,_) -> true
 |Qcf(C,_,_) -> true
 |_ -> false
;;

(* sont_C : 
   eterm list -> bool *)
let sont_C ets = map_and est_C ets
;;

(* un_De : 
   eterm list -> bool *)
let un_De ets = 
 let rec un_De_rec = function
   (b,[]) -> b
  |(true,(et::ets))
   -> if (est_C et) then
        un_De_rec (true,ets)
      else
        false
  |(false,(et::ets))
   -> if (est_De et) then
        un_De_rec (true,ets)
      else
        if (est_C et) then
          un_De_rec (false,ets)
        else
          false
 in un_De_rec (false,ets)
 
;;

(* etik : 
   constr signature -> xterm * xterm -> eterm *)
let rec etik sign = function
  (Xvar x,u)
  -> if (Xvar x)=u then Qvar(C,x) else Qvar(L,x)
 |(Xcc c,u)
  ->  if (Xcc c)=u then Qcc(C,c) else Qcc(L,c)
 |(Xcf(c,ts),Xvar x)
  -> if (List.mem (Xvar x) ts) then Qcf(De,c,etier N ts) else Qcf(L,c,etier N ts)
 |(Xcf(c,ts),Xcf(c',ts'))
  -> if (c=c') then
       let ets = cmap (etik sign) ts ts' in
       let eis = arg_ind c ets sign in
        if (sont_C eis) then 
          Qcf(C,c,ets)
        else
          if (un_De eis) then
            Qcf(D,c,ets)
          else 
            Qcf(L,c,ets) 
     else
      Qcf(L,c,etier L ts)
 |(Xcf(c,ts),_)
  -> Qcf(L,c,etier L ts)
 |(Xind (x,xs),u) -> Qind(L,x,etier N xs)
 | _ -> anomaly "etik"
;;

(* etikrecs : 
   constr signature -> xterm list -> xterm list list 
   -> eterm list list *)
let rec etikrecs sign ts uss = 
 List.map (cmap (etik sign) ts) uss
;;

(* etikeqs : 
   constr signature -> int -> (xterm list * xterm) list 
   -> (int * etiql * eterm list) list *)
let rec etikeqs sign n = function
  [] -> []
 |(ts,u)::es 
  -> let uss' = appels_ind u in
      if uss'=[] then
        (n,Nl,etier N ts)::(etikeqs sign (succ n) es)
      else
        (List.map (fun us' -> (n,Vl,us')) (etikrecs sign ts uss'))
        @(etikeqs sign (succ n) es)
;;

(* etikf : 
   constr signature -> (xterm list * xterm) list 
   -> (int * etiql * eterm list) list *)
let etikf sign es = etikeqs sign 1 es
;;

(* === Matrice de recurrence *)
type etiqc = P | I
;; 

type imatrec = int list * etiqc * etype
;;

type lmatrec = int * etiql * eterm list
;;

(* Qvar0 : 
   eterm *)
let qvar0 = Qvar(N,null_id)
;;

(* etype_env : 
   etype * etype -> (identifier * etype) list *)
let rec etype_env = function
  (Evar x), d 
  -> [x,d]
 |(Ecste(d,ds)), (Ecste(d',ds'))
  -> mapapp etype_env (List.combine ds ds')
 |Eset,_ -> []
 |Eprop,_ -> []
 |_ -> anomaly "etype_env"
;;

(* s_etype : 
   (identifier * etype) list -> etype -> etype *)
let rec s_etype env = function
  Eset -> Eset
 |Eprop -> Eprop
 |Evar x 
  -> if (List.mem_assoc x env) then
       (List.assoc x env)
     else 
       (Evar x)
 |Ecste(d,ds)
  -> Ecste(d,List.map (s_etype env) ds)
 |Eprod(x,d) 
  -> if (List.mem_assoc x env) then
      Eprod(x,d)
     else
      Eprod(x,s_etype env d)
 |Efleche(ds,d)
  -> Efleche(List.map (s_etype env) ds,d)
;;

(* subst_idE : 
   etype -> identifier -> etype -> etype *)
let rec subst_idE d x = function
 Evar y -> if x=y then d else (Evar y)
|Eprod(y,d') -> Eprod(y,subst_idE d x d')
|Ecste(y,ds) -> Ecste(y,List.map (subst_idE d x) ds)
|Efleche(ds,d') -> Efleche(List.map (subst_idE d x) ds,subst_idE d x d')
|d -> d
;;

(* cons_etypes_args_inst : 
*)
let rec cons_etypes_args_inst = fun
 p_0 p_1 -> match p_0,p_1 with ((d::ds), ((Eprod(x,d'))::ds'))
 -> d'::(cons_etypes_args_inst ds (List.map (subst_idE d x) ds'))
|((d::ds), (_::ds')) -> anomaly "cons_etypes_args_inst1"
|((d::ds), []) -> anomaly "cons_etypes_args_inst2"
|([], ((Eprod(x,d'))::ds')) -> anomaly "cons_etypes_args_inst3"
|([], ds') -> ds'
;;

(* cons_etypes_args_aux : 
   etype list -> etype -> etype list *)
let cons_etypes_args_aux ds = function
 Efleche(us,_) -> cons_etypes_args_inst ds us
|Ecste _ -> []
|_ -> anomaly "cons_etypes_args_aux"
;;

(* esets : 
   'a list -> etype list *)
let rec esets = function
 [] -> []
|x::xs -> Eset::(esets xs)
;;

(* cons_etypes_args_of : 
   etype -> etype list list *)
let cons_etypes_args_of  = function
 Ecste((GLOBALNAME (DOPN(MutInd _,_) as mind),ds)) ->
    let mis = mind_specif_of_mind mind in 
    let (_,typesconstr) = mis_type_mconstructs mis in 
         List.map (cons_etypes_args_aux ds)
             (map_vect_list (etype_of_constr (List.rev ds) [] []) typesconstr)
|_ ->  anomaly "cons_etypes_args_of 2"
;;

(* icols_ci' : 
   int list -> int -> 'a -> 'a list 
   -> (int list * etiqc * 'a) list *)
let rec icols_ci' x n d = function
  [] -> []
 |di::dis 
  -> if di=d then
      (add_last n x,I,d)::(icols_ci' x (succ n) d dis)
     else
      (add_last n x,P,di)::(icols_ci' x (succ n) d dis)
;;

(* icols_ci : 
   (int list * etiqc * etype) list 
   -> (int list * etiqc * etype) list 
   -> int list -> etype -> etype list 
   -> (int list * etiqc * etype) list *)
let icols_ci ic1 ic2 x d = function
 [] -> ic1@((add_last 1 x,P,evar0)::ic2)
|ds -> ic1@(icols_ci' x 1 d ds)@ic2
;;

(* dev_icol : 
   int -> (int list * etiqc * etype) list 
   -> (int list * etiqc * etype) list list *)
let dev_icol i icol =
  (fun (ic1,(x,e,d),ic2) ->
     List.map (icols_ci ic1 ic2 x d) (cons_etypes_args_of d))
   (coupe_list_n i icol)
;;

(* place_ligne : 
   'a -> int -> 'a list list -> 'a list list *)
let place_ligne l n ssmat = 
  
 let rec place_ligne_rec = function
  1,ls::lss -> (l::ls)::lss
 |n,ls::lss -> ls::(place_ligne_rec (pred n,lss))
 |_ -> anomaly "place_ligne"
 in place_ligne_rec (n,ssmat)

;;

(* num_of_cons : 
   identifier -> constr signature -> int *)
let num_of_cons c sign =
 match search_reference (gLOB sign) c with
   DOPN(MutConstruct(_,i),_) -> i
 |_ -> anomaly "num_of_cons"
;;

(* dev_lmat0 : 
   ('a * etiql * eterm list) list list 
   -> 'a -> eterm list -> eterm list 
   -> constr signature -> etiql * eterm 
   -> ('a * etiql * eterm list) list list *)
let dev_lmat0 sslmat ne ts1 ts2 sign = function
  el,Qcc(_,c)
  -> place_ligne (ne,el,ts1@(qvar0::ts2)) (num_of_cons c sign) sslmat
 |Vl,Qcf(De,c,ts)
  -> place_ligne (ne,Nl,ts1@ts@ts2) (num_of_cons c sign) sslmat
 |Ll,Qcf(De,c,ts)
  -> place_ligne (ne,Nl,ts1@ts@ts2) (num_of_cons c sign) sslmat
 |Vl,Qcf(D,c,ts)
  -> place_ligne (ne,Ll,ts1@ts@ts2) (num_of_cons c sign) sslmat
 |el,Qcf(_,c,ts)
  -> place_ligne (ne,el,ts1@ts@ts2) (num_of_cons c sign) sslmat
 |_ -> anomaly "dev_lmat0"
;;

(* dev_lmat : 
   int -> int -> constr signature 
   -> ('a * etiql * eterm list) list 
   -> ('a * etiql * eterm list) list list *)
let dev_lmat i nc sign lmat =
 
 let rec dev_lmat_rec lmat0 = function
    [] -> lmat0
   |(ne,el,ts)::lgs
    -> let (ts1,t,ts2) = coupe_list_n i ts in
        (dev_lmat_rec 
         (dev_lmat0 lmat0 ne ts1 ts2 sign (el,t))
         lgs)
 in (dev_lmat_rec (iterate (fun x -> []::x) nc []) lmat)
  
;;

(*  nb_of_cons : 
    identifier ->  constr signature -> int *)
let nb_of_cons d sign =
 match (search_reference (gLOB sign) d) with
 (DOPN(MutInd _,_)) as ind 
  ->  mis_nconstr (mind_specif_of_mind ind)
 |_ -> anomaly "nb_of_cons (2)"
;;

(* id_of_Ecste : 
   etype -> identifier *)
let id_of_Ecste = function
 Ecste(GLOBALNAME (VAR id),_) -> id
|Ecste(GLOBALNAME (DOPN(MutInd _ as mind,_)),_) -> id_of_global mind
|Ecste(GLOBALNAME (DOPN(Const sp,_)),_) -> basename sp
|_ -> anomaly "id_of_ecste"
;;
 
(* numcol : 
   'a -> ('a * 'b * 'c) list -> int *)
let rec numcol x = function
 [] -> anomaly "numcol"
|(y,_,_)::ys 
 -> if x=y then 1 else (succ (numcol x ys))
;;

(* dev_mat : 
   constr signature
   -> int list * etiqc * etype 
   -> (int list * etiqc * etype) list 
     * ('a * etiql * eterm list) list 
   -> ((int list * etiqc * etype) list 
     * ('a * etiql * eterm list) list) list *)
let dev_mat sign (x,_,d) (icol,lmat) =
  let i = numcol x icol in
  let nc = nb_of_cons (id_of_Ecste d) sign in
  let lmats = (dev_lmat i nc sign lmat) in
  if (List.mem [] lmats) then
    perror(14,[<'sTR"pattern matching is not exhaustive">])
  else
   List.combine (dev_icol i icol) lmats
;;

(* icolini : 
   int -> (identifier * etype) list 
   -> (int list * etiqc * etype) list *)
let rec icolini n = function
 [] -> []
|d::ds
-> ([n],P,d)::(icolini (succ n) ds)
;;

(* matini :
   State ref -> etype list -> (xterm list * xterm) list 
   -> (int list * etiqc * etype) list 
     * (int * etiql * eterm list) list *)
let matini st ds es =
  (icolini 1 ds,etikf st es)
;;

(* === Variable eligible *)
(* est_Qvar : 
   eterme -> bool *)
let est_Qvar = function
  Qvar(_) -> true
 |_ -> false
;;

(* tout_I_var : 
   ('a * etiqc * 'b) list * eterm list 
   * ('c * etiqc * 'd) list * eterm list 
   -> bool *)
let rec tout_I_var = function
  [],[],[],[] -> true
 |[],[],(_,P,_)::ics,_::ts -> tout_I_var ([],[],ics,ts)
 |[],[],(_,I,_)::ics,(Qvar(_))::ts -> tout_I_var ([],[],ics,ts)
 |[],[],(_,I,_)::ics,_::ts -> false
 |(_,P,_)::ics0,_::ts0,ics,ts -> tout_I_var (ics0,ts0,ics,ts)
 |(_,I,_)::ics0,(Qvar(_))::ts0,ics,ts -> tout_I_var (ics0,ts0,ics,ts)
 |(_,I,_)::ics0,_::ts0,ics,ts -> false
 |_ -> anomaly "tout_i_var"
;;

(* var_elig : 
   ('a * etiqc * 'b) list -> eterm list 
   -> etiql * bool list * eterm list * ('a * etiqc * 'b) list 
   -> bool list *)
let rec var_elig ics0 ts0 = function
  _, [], [], [] -> []
 |el, (false::bs), (t::ts), (ic::ics)
  -> false::(var_elig (ic::ics0) (t::ts0) (el,bs,ts,ics))
 |el, (true::bs), (Qvar((x_0,x_1))::ts), (ic::ics)
  -> false::(var_elig (ic::ics0) ((Qvar((x_0,x_1)))::ts0) (el,bs,ts,ics))
 |el, (true::bs), (Qind((x_0,x_1,x_2))::ts), (ic::ics)
  -> false::(var_elig (ic::ics0) ((Qind((x_0,x_1,x_2)))::ts0) (el,bs,ts,ics))
 |Nl, (true::bs), (t::ts), (ic::ics)
  -> true::(var_elig (ic::ics0) (t::ts0) (Nl,bs,ts,ics))
 |el, (true::bs), (Qcc(L,c)::ts), ic::ics
  -> false::(var_elig (ic::ics0) ((Qcc(L,c))::ts0) (el,bs,ts,ics))
 |el, (true::bs), (Qcc((x_0,x_1))::ts), ic::ics
  -> true::(var_elig (ic::ics0) ((Qcc((x_0,x_1)))::ts0) (el,bs,ts,ics))
 |el, (true::bs), (Qcf(L,d,ts'))::ts, ic::ics 
  -> false::(var_elig (ic::ics0) ((Qcf(L,d,ts'))::ts0) (el,bs,ts,ics))
 |el, (true::bs), (Qcf(De,d,ts'))::ts, ic::ics
  -> (tout_I_var (ics0,ts0,ics,ts))
     ::(var_elig (ic::ics0) ((Qcf(De,d,ts'))::ts0) (el,bs,ts,ics))
 |Ll, (true::bs), (Qcf(D,d,ts'))::ts, (x,P,d')::ics 
  -> false::(var_elig ((x,P,d')::ics0) ((Qcf(D,d,ts'))::ts0) (Ll,bs,ts,ics))
 |el, (true::bs), (Qcf(e,d,ts'))::ts, ic::ics
  -> true::(var_elig (ic::ics0) ((Qcf(e,d,ts'))::ts0) (el,bs,ts,ics))
 |_ -> anomaly "var_elig"
;;

(* select_var_eligs : 
   bool list * 'a list -> 'a list *)
let rec select_var_eligs = function
 [],[] -> []
|true::bs, x::xs -> x::(select_var_eligs (bs,xs))
|false::bs, x::xs -> select_var_eligs (bs,xs)
|_ -> anomaly "select_var_eligs"
;;

(* vars_elig0 : 
   bool list -> ('a * etiqc * 'b) list 
   -> ('c * etiql * eterm list) list 
   -> ('a * etiqc * 'b) list *)
let rec vars_elig0 bs ics = function
  [] -> select_var_eligs (bs,ics)
 |(n,el,ts)::ls
  -> vars_elig0 (var_elig [] [] (el,bs,ts,ics)) ics ls
;;

(* == Controle Inductive Set *)
(* check_ind_Set : 
   sorts oper term -> bool *)
let rec check_ind_Set = function
 DOP0(Sort(Prop Pos)) -> true
|DOP2(Prod,_,DLAM(_,t)) -> check_ind_Set t
|DOP2(Cast,t,_) -> check_ind_Set t
|_ -> perror(0,[<>])
;;

(* check_ind_arity : 
   int -> sorts oper term -> bool *)
let rec check_ind_arity = fun
 p_0 p_1 -> match p_0,p_1 with (0, t) -> check_ind_Set t
|(n, (DOP2(Cast,t,_))) -> check_ind_arity n t
|(n, (DOP2(Prod,DOP0(Sort(Prop Pos)),DLAM(_,t))))
 -> check_ind_arity (pred n) t
|(n, (DOP2(Prod,DOP2(Cast,DOP0(Sort(Prop Pos)),_),DLAM(_,t))))
 -> check_ind_arity (pred n) t
|(n, (DOP2(Prod,_,DLAM(_,t)))) -> perror(0,[<>])
|(_, _) -> anomaly "check_ind_arity"
;;

(* rm_nProd : 
   int -> 'a oper term -> 'a oper term *)
let rec rm_nProd = fun
 p_0 p_1 -> match p_0,p_1 with (0, t) -> t
|(n, (DOP2(Cast,t,_))) -> rm_nProd n t
|(n, (DOP2(Prod,_,DLAM(_,t)))) -> rm_nProd (pred n) t
|(_, _) -> anomaly "rm_nProd"
;;

(* check_ind_def : 
   sorts oper term -> bool *)
let rec check_ind_def = function
DOPN(MutInd (x_0,x_1),_) as t
 -> let mispec = mind_specif_of_mind t in 
    let n = mis_nparams mispec in
    let (lC,arity) = mis_lc_arity mispec in 
    let (_,cs) = decomp_all_DLAMV_name lC in
    (check_ind_arity n arity;
     map_and check_cons_type (map_vect_list (rm_nProd n) cs))
|DOP2(Cast,t,_) -> check_ind_def t
|_ -> false

(* check_cons_type : 
   sorts oper term -> unit *)
and check_cons_type = function
 DOP2(Prod,t,DLAM(Anonymous,u))
 -> (check_type_expr t; check_cons_type u)
|DOP2(Cast,t,_) -> check_cons_type t
|t -> (check_type_expr t)

(* check_type_expr : 
   sorts oper term -> bool *)
and check_type_expr = function
 Rel _ -> true
|DOP0(Sort(Prop Pos)) -> true
|DOP0(Sort(Prop Null)) -> true
|DOP2(Cast,t,_) -> check_type_expr t
|DOPN(MutInd _,_) as x -> check_ind_def x
|DOPN(AppL,ts) -> map_and check_type_expr (Array.to_list ts)
|DOPN(Const _,_) as c 
 -> let c' = whd_betadelta (Evd.mt_evd()) c in
    (c=c') or (check_type_expr c')
|VAR _ as x 
 -> check_type_expr (type_of_rel (Evd.mt_evd()) (gLOB (initial_sign())) x)
|t -> false
;;

(* init_vars_eligs : 
    *)
let rec init_vars_eligs = function
 [] -> []
|(_,_,Ecste(GLOBALNAME (DOPN(MutInd _,_) as t),_))::ics 
 -> (check_ind_def t)::(init_vars_eligs ics)
|_::ics 
 -> false::(init_vars_eligs ics)
;;
 
(* vars_elig : 
   ....
   ('a * etiqc * etype) list 
   * ('b * etiql * eterm list) list 
   -> ('a * etiqc * etype) list *)
let vars_elig (icol,lmat) =
  vars_elig0 (init_vars_eligs icol) icol lmat
;;

(* === Arbre de recurrence *)
type arbrec =
  Frec of int 
 |Brec of ((int list * etype) * arbrec list)
;;

(* l_partoutvar : 
   eterm list -> bool *)
let rec l_partoutvar = function
  [] -> true
 |(Qvar(_))::ts -> l_partoutvar ts
 |_ -> false
;;

(* check_dep_params_bis :
*)
let rec check_dep_params_bis = fun
 p_0 p_1 -> match p_0,p_1 with ([], _) -> true
|((true::bs), (t::ts)) -> (est_C t)&(check_dep_params_bis bs ts)
|((_::bs), (t::ts)) -> (check_dep_params_bis bs ts)
;;

(* check_dep_params :
*)
let rec check_dep_params bs = function
 [] -> []
|(n,Vl,ts)::ls
 -> if (check_dep_params_bis bs ts) then
     (n,Vl,ts)::(check_dep_params bs ls)
    else
     perror(11,[<'sTR"(equation "; 'sTR(string_of_int n); 'sTR") :\n can't change value of dependent parameter">])
|l::ls -> l::(check_dep_params bs ls)
;;

(* contient_decr_bis : 
   constr signature -> eterm list -> bool *)
let rec contient_decr_bis sign = function
 [] -> false
|Qcf(De,_,_)::ts -> true
|Qcf(D,_,_)::ts -> true
|Qcf(C,c,ts')::ts 
 -> (contient_decr_bis sign (arg_par c ts' sign)) 
   or (contient_decr_bis sign ts)
|t::ts -> (contient_decr_bis sign ts)
;;

(* contient_decr : 
   ... *)
let rec contient_decr sign = function
  [] -> []
 |(n,Vl,ts)::ls 
  -> if (contient_decr_bis sign ts) then
      (n,Vl,ts)::(contient_decr sign ls)
     else
      perror(11,[<'sTR"(equation "; 'sTR(string_of_int n); 'sTR") :\n can't find argument on which apply structural induction">])
 |l::ls -> l::(contient_decr sign ls)
;;

(* memeq : 
   ('a * 'b) list -> ('a * 'c * 'b) list -> ('a * 'b) list *)
let rec memeq nts = function
 [] -> nts
|(n,_,ts)::ls
 -> if (List.mem_assoc n nts) then (memeq nts ls) else (memeq ((n,ts)::nts) ls)
;;

(* numeqs : 
   ('a * 'b) list -> 'a list *)
let numeqs ls = 
 let rec numeqs_rec ns = function
 [] -> (List.rev ns)
|(n,_)::ls
 -> if (List.mem n ns) then (numeqs_rec ns ls) else (numeqs_rec (n::ns) ls)
 in numeqs_rec [] ls

;;

(* unifQ : 
   eterm -> eterm -> bool *)
let rec unifQ = fun
 p_0 p_1 -> match p_0,p_1 with ((Qvar _), _) -> true
|(_, (Qvar _)) -> true
|((Qcc(_,x)), (Qcc(_,y))) -> x=y
|((Qcf(_,x,ts)), (Qcf(_,y,us))) -> (x=y)&(map2_and unifQ ts us)
|((Qind(_,x,ts)), (Qind(_,y,us))) -> (x=y)&(map2_and unifQ ts us)
|(_, _) -> false
;;

(* unif_eq : 
   eterm list -> ('a * eterm list) list -> 'a list *)
let rec unif_eq ts = function
 [] -> []
|(n,us)::nus
 -> if (map2_and unifQ ts us) then n::(unif_eq ts nus) else (unif_eq ts nus)
;;

(* unif_eqs : 
   ('a * eterm list) list -> 'a list *)
let rec unif_eqs = function
 [] -> []
|(n,ts)::nts
 -> match (unif_eq ts nts) with
     [] -> unif_eqs nts
    |ns -> (n::ns)@(unif_eqs nts)
;;

(* cons_Frec : 
   (int * eterm list) list -> arbrec *)
let cons_Frec = function
 [] -> perror(14,[<'sTR" :\n pattern matching is not exhaustive">])
|(n,_)::[] -> Frec n
|nts 
 -> let ns = unif_eqs nts in
    match ns with
     [] -> perror(13,[<'sTR"(equation(s) "; 
                       'sTR(mapconc "," string_of_int (List.rev (numeqs nts))); 
                       'sTR") ;\n can't find argument on which apply structural induction">])
    |_ -> perror(12,[< 'sTR"(equations "; 'sTR(mapconc "," string_of_int ns); 
                       'sTR") :\n overlapping">])
;;

(* cons_arbrec0 : 
   constr signature
   -> identifier
   -> ((int list * etiqc * etype) list 
       -> int list * etiqc * etype) 
   -> (int list * etiqc * etype) list 
     * (int * etiql * eterm list) list 
   -> (int list * etiqc * etype) list 
   -> arbrec *)
let rec cons_arbrec0 sign f choix_var mat = function
 [] -> cons_Frec (memeq [] (snd mat))
|vs -> cons_arbrec0s sign f choix_var mat vs
   
(* cons_arbrec0s : 
   constr signature
   -> string 
   -> ((int list * etiqc * etype) list 
       -> int list * etiqc * etype) 
   -> (int list * etiqc * etype) list 
     * (int * etiql * eterm list) list 
   -> (int list * etiqc * etype) list 
   -> arbrec *)
and cons_arbrec0s sign f choix_var mat = function
 [] -> raise Error_arbrec
|vs 
 -> let (x,e,d) = choix_var vs in
    try (Brec((x,d),
              (List.map (fun m -> (cons_arbrec0 sign f choix_var m (vars_elig m)))
		   (dev_mat sign (x,e,d) mat))))
    with 
     Error_arbrec -> cons_arbrec0s sign f choix_var mat (except (x,e,d) vs)
;;

(* dep_args_f :
*)
let rec dep_args_f_rec = function
 (Eprod _)::ds -> true::(dep_args_f_rec ds)
|_::ds -> false::(dep_args_f_rec ds)
|[] -> []
;;

let dep_args_f = function
 Efleche(ds,d) -> dep_args_f_rec ds
|_ -> anomaly "dep_args_f"
;;

(* cons_arbrec : 
   constr signature 
   -> identifier * etype * (xterm list * xterm) list 
   -> arbrec *)
let cons_arbrec sign f d es = 
 match d with
  Efleche(ds,_)
  -> (let (ics,ls) = (matini sign ds es) in 
      cons_arbrec0 sign f List.hd 
                  (ics,
                   contient_decr sign (check_dep_params (dep_args_f d) ls))
                  (vars_elig (ics,ls)))
 |_ -> anomaly "cons_arbrec"
;;

(* === Extraction du terme (de 'arbrec' a 'command') *)
type proto_form =
 At of identifier * etype
|Hr of identifier * identifier list * xterm
;;

(* coupe_prem : 
   'a -> ('a * 'b) list -> ('a * 'b) list * ('a * 'b) list *)
let coupe_prem x xTs = 
 let rec coupe_prem_rec xTs' = function
 [] -> (List.rev xTs',[])
|(_,x',t)::xTs
 -> if (x=x') then
     (List.rev ((x,t)::xTs'),xTs)
    else
     coupe_prem_rec ((x',t)::xTs') xTs
 in coupe_prem_rec [] xTs

;;

(* subst_idX : 
   xterm -> identifier -> xterm -> xterm *)
let rec subst_idX t x = function
 Xvar y -> if x=y then t else (Xvar y)
|Xcf(y,ts) -> Xcf(y,List.map (subst_idX t x) ts)
|Xft(y,ts) -> Xft(y,List.map (subst_idX t x) ts)
|Xfv(y,ts) -> Xfv(y,List.map (subst_idX t x) ts)
|Xfx(y,ts) -> Xfx(y,List.map (subst_idX t x) ts)
|Xind(y,ts) -> Xind(y,List.map (subst_idX t x) ts)
|t' -> t'
;;

(* etype_actuel : 
   etype list -> etype -> etype list -> etype list -> etype list * etype *)
let rec etype_actuel acc_ds d = fun
 p_0 p_1 -> match p_0,p_1 with (((Eprod(x,_))::ds'), (d''::ds''))
 -> etype_actuel acc_ds (subst_idE d'' x d) 
                 (List.map (subst_idE d'' x) ds') ds''
|(ds', []) -> (((List.rev acc_ds)@ds'),d)
|((d'::ds'), ds'') -> etype_actuel (d'::acc_ds) d ds' ds''
;;

(* etype_actuel_of : 
   constr signature -> etype list -> identifier -> etype list * etype *)
let etype_actuel_of sign ds x =
 let (ds',d) = etype_of sign x in etype_actuel [] d ds' ds
;;

(* hyp_cons : 
   etype -> identifier list ->  xterm -> identifier 
   -> int list -> int -> etype list -> proto_form list *)
(* NB : on suppose que toutes les variables param du type du cons
   sont instanciees *)
let rec hyp_cons d xs t x ns i ds = 
 
 let rec hyp_cons_rec i = function
 [] -> []
|(d'::ds)
 -> if d=d' then
     (let nsi = add_last i ns in
      let xi = id_of_ints "x" nsi in
      (At(xi,d'))
      ::(Hr(id_of_ints "h" nsi,xs,subst_idX (Xvar xi) x t))
      ::(hyp_cons_rec (succ i) ds))
    else
     hyp_cons_rec (succ i) ds
 in hyp_cons_rec i ds

;;

(* xterm_of_etype : 
   etype -> xterm *)
let rec xterm_of_etype = function
 Evar x -> Xvar x
|Ecste(x,ds) as e -> Xind(id_of_Ecste e,List.map xterm_of_etype ds)
|_ -> anomaly "xterm_of_etype"
;;

(* xid_list : 
   int list -> int -> int -> xterm list *)
let rec xid_list ns i = function
 0 -> []
|n 
 -> (Xvar (id_of_ints "x" (add_last i ns)))
    ::(xid_list ns (succ i) (pred n))
;;

(* term_cons : 
   identifier -> xterm list -> xterm list -> xterm *)
let term_cons c = fun
 p_0 p_1 -> match p_0,p_1 with ([], []) -> Xcc c
|(xps, xs) -> Xcf(c,xps@xs)
;;

(* it_AppC : 
   command -> command list -> command *)
let rec it_AppC t us = 
 List.fold_left (fun t u -> mkAppC(t,u)) t us
;;

(* ProdC_of_etypes : 
   ast -> etype list -> command *)
let rec prodC_of_etypes t = function
 [] -> t
|(Eprod(x,e))::es 
 -> mkProdC(x,com_of_etype e,prodC_of_etypes t es)
|e::es
 -> mkArrowC(com_of_etype e,prodC_of_etypes t es) 

(* com_of_etype : 
   etype -> ast *)
and com_of_etype = function
 Evar x -> nvar (string_of_id x)
|Eset -> mkPropC "Pos"
|Eprop -> mkPropC "Null"
|Eprod(x,_) -> nvar (string_of_id x)
|Ecste(d,[]) as e -> nvar (string_of_id (id_of_Ecste e))
|Ecste(d,es) as e ->
    it_AppC (nvar (string_of_id (id_of_Ecste e))) (List.map com_of_etype es)
|Efleche([],e) -> com_of_etype e
|Efleche(es,e) -> prodC_of_etypes (com_of_etype e) es
;;

(* prem_cons : 
   int -> etype -> ast -> int list -> etype list 
   -> (identifier * ast) list * (bool * identifier * ast) list*)
let prem_cons i d hrC ns dacs = 
 let rec prem_cons_rec pcs ppcs i = function
 [] -> (List.rev pcs, List.rev ppcs)
|da::das
 -> let nsi = add_last i ns in
    let (x,t) = (id_of_ints "x" nsi,com_of_etype da) in
    if d=da then
     prem_cons_rec ((id_of_ints "h" nsi,hrC)::(x,t)::pcs) ppcs (succ i) das
    else
     prem_cons_rec ((x,t)::pcs) ((false,x,t)::ppcs) (succ i) das
 in prem_cons_rec [] [] i dacs

;;

(* hd_const_id_of :
   sorts oper term -> identifier *)
let rec hd_const_id_of = function
 VAR id -> id
|DOPN(Const _,_) as x -> basename (path_of_const x)
|DOPN(MutInd _ as x,_)  -> id_of_global x
|DOPN(MutConstruct _ as x,_) ->  id_of_global x
|DOP2(Prod,_,DLAM(_,t)) -> hd_const_id_of t
|DOPN(AppL,cl) -> hd_const_id_of (hd_vect cl) 
|_ -> anomaly "hd_const_id_of"
;;

(* hd_symb_of :
   'a oper term -> 'a oper term *)
let rec hd_symb_of = function
 DOP2(Lambda,_,DLAM(_,t)) -> hd_symb_of t
|DOPN(AppL,cl) -> hd_symb_of (hd_vect cl) 
|t -> t
;;

(* ids_cons_of : 
   etype -> constr signature -> identifier list *)
let ids_cons_of d sign = 
 let x = id_of_Ecste d in
 let n = nb_of_cons x sign in
  match (search_reference (gLOB sign) x) with
   (DOPN(MutInd _,_)) as mind 
    -> let mispec = mind_specif_of_mind mind in 
        Array.to_list (mis_consnames mispec)
  |_ -> anomaly "ids_of_cons"
;; 

(* etype_of_com : 
   constr signature -> ast -> etype *)
let rec etype_of_com_rec sign = function
    Node(_,"PROP",[Id(_,"Pos")]) -> Eset 
  | Node(_,"PROP",[Id(_,"Null")]) -> Eprop 
  | Nvar(_,x) -> Evar (id_of_string x)
  | Node(_,"APPLIST",_) as x ->
      let (f,ts) = flat_AppC x in
      let glob_f = search_reference (gLOB sign) (id_of_string f) in
        Ecste(GLOBALNAME glob_f,List.map (etype_of_com_rec sign) ts)
  | Node(_,"PROD",[t;Slam(_,None,u)]) ->
      Efleche([etype_of_com_rec sign t],etype_of_com_rec sign u)
  | Node(_,"PROD",[t;Slam(_,Some x,u)]) ->
      Efleche([Eprod(id_of_string x,etype_of_com_rec sign t)],
              etype_of_com_rec sign u)
  | _ -> anomaly "etype_of_com_rec"
;;

let rec flat_Efleche = function
 Efleche(ts,Efleche(us,v)) -> flat_Efleche (Efleche(ts@us,v))
|Ecste(x,ts) -> Ecste(x,List.map flat_Efleche ts)
|Eprod(x,t) -> Eprod(x,flat_Efleche t)
|Efleche(ts,t) -> Efleche(List.map flat_Efleche ts,flat_Efleche t)
|t -> t
;;

let etype_of_com sign t = flat_Efleche (etype_of_com_rec sign t)
;;

exception Erreur_instance
;;

(* maj_assoc : 
   'a -> 'b -> ('a * 'b) list -> ('a * 'b) list *)
let rec maj_assoc x y xys = 
 let rec maj_assoc_rec = function
 [] -> []
|(x',y')::xys
 -> if x=x' then (x,y)::xys else (x',y')::(maj_assoc_rec xys)
 in maj_assoc_rec xys

;;

let xvar0 = Xvar null_id
;;

(* instance : 
   identifier list -> xterm -> xterm -> (identifier * xterm) list
 NB : (instance [x1;..;xk] t1 t2) -> [(x1,u1);..;(xk,uk)] 
           ssi t2=t1[u1/x1,..,xk/ux] *)
let rec instance xs t1 t2 = 
 let rec instance_rec xus = fun
 p_0 p_1 -> match p_0,p_1 with ((Xvar x), t)
 -> if (List.mem_assoc x xus) then
     (let u = List.assoc x xus in
       if u=xvar0 then
        maj_assoc x t xus
       else
        if u=t then xus else raise Erreur_instance)
    else     
     if t=(Xvar x) then xus else raise Erreur_instance
|((Xcc x), (Xcc x')) 
 -> if x=x' then xus else raise Erreur_instance
|((Xct x), (Xct x')) 
 -> if x=x' then xus else raise Erreur_instance
|((Xcf(x,ts)), (Xcf(x',ts')))
 -> if x=x' then
     List.fold_right2 (fun t1 t2 x -> instance_rec x t1 t2) ts ts' xus
    else
     raise Erreur_instance
|((Xft(x,ts)), (Xft(x',ts')))
 -> if x=x' then
     List.fold_right2 (fun t1 t2 x -> instance_rec x t1 t2) ts ts' xus
    else
     raise Erreur_instance
|((Xfv(x,ts)), (Xfv(x',ts')))
 -> if x=x' then
     List.fold_right2 (fun t1 t2 x -> instance_rec x t1 t2) ts ts' xus
    else
     raise Erreur_instance
|((Xfx(x,ts)), (Xfx(x',ts')))
 -> if x=x' then
     List.fold_right2 (fun t1 t2 x -> instance_rec x t1 t2) ts ts' xus
    else
     raise Erreur_instance
|((Xind(x,ts)), (Xind(x',ts')))
 -> if x=x' then
     List.fold_right2 (fun t1 t2 x -> instance_rec x t1 t2) ts ts' xus
    else
     raise Erreur_instance
|(_, _) -> raise Erreur_instance
 in instance_rec (List.map (fun x -> (x,xvar0)) xs) t1 t2

;;

(* cherche_instance : 
   xterm -> proto_form list -> identifier * xterm list *)
let rec cherche_instance t = function
 [] -> anomaly "cherche_instance"
|(Hr(hi,xs,u))::hs
 -> (try 
      let xus = instance xs u t in
       (hi,List.map snd xus)
     with
      Erreur_instance -> cherche_instance t hs)
|h::hs -> cherche_instance t hs
;;
 
(* etendre : 
   identifier -> proto_form list -> xterm -> ast *)
let rec etendre f hs = function
 Xvar x -> nvar (string_of_id x)
|Xcc x -> nvar (string_of_id x)
|Xct x -> nvar (string_of_id x)
|Xcf(x,ts) 
 -> it_AppC (nvar (string_of_id x)) (List.map (etendre f hs) ts)
|Xft(g,ts)
 -> if f=g then
     (let (h,us) = cherche_instance (Xft(g,ts)) hs in
       it_AppC (nvar (string_of_id h)) (List.map (etendre f hs) us))
    else
     it_AppC (nvar (string_of_id g)) (List.map (etendre f hs) ts)
|Xfv(x,ts) -> it_AppC (nvar (string_of_id x)) (List.map (etendre f hs) ts)
|Xfx(f',ts)
 -> let (h,us) = cherche_instance (Xft(f',ts)) hs in
      it_AppC (nvar (string_of_id h)) (List.map (etendre f hs) us)
|Xind(x,[]) -> nvar (string_of_id x)
|Xind(x,ts)
 -> it_AppC (nvar (string_of_id x)) (List.map (etendre f hs) ts)
;;

(* renomme_idX : 
   identifier -> identifier -> xterm -> xterm *)
let rec renomme_idX x y = function
 Xvar z -> Xvar (if z=y then x else z)
|Xcf(c,ts) -> Xcf(c,List.map (renomme_idX x y) ts)
|Xft(f,ts) -> Xft(f,List.map (renomme_idX x y) ts)
|Xfv(z,ts) -> Xfv((if z=y then x else z),List.map (renomme_idX x y) ts)
|Xfx(f,ts) -> Xfx(f,List.map (renomme_idX x y) ts)
|Xind(i,ts) -> Xind(i,List.map (renomme_idX x y) ts)
|t -> t
;;

(* renommeX :
   xterm -> xterm -> xterm -> xterm *)
let rec renommeX u v w =
  match (v,w) with
 ((Xvar x), (Xvar y)) -> renomme_idX x y u
|((Xcf(_,ts)), (Xcf(_,us)))
 -> List.fold_right2 (fun ts us u -> renommeX u ts us) ts us u
|((Xft(_,ts)), (Xft(_,us)))
 -> List.fold_right2 (fun ts us u -> renommeX u ts us) ts us u
|((Xfv(_,ts)), (Xfv(_,us)))
 -> List.fold_right2 (fun ts us u -> renommeX u ts us) ts us u
|((Xfx(_,ts)), (Xfx(_,us)))
 -> List.fold_right2 (fun ts us u -> renommeX u ts us) ts us u
|((Xind(_,ts)), (Xind(_,us)))
 -> List.fold_right2 (fun ts us u -> renommeX u ts us) ts us u
|(_, _) -> u
;;
     
(* perm_intros : 
   'a -> ('b * 'a * 'c) list -> ('b * 'a * 'c) list * ('b * 'a * 'c) list *)
let rec perm_intros_rec ps1 x = function
 [] -> ([],List.rev ps1)
|(b,x',t)::ps2
 -> if (x=x') then
     (List.rev ((b,x',t)::ps1),ps2)
    else
     perm_intros_rec ((b,x',t)::ps1) x ps2
;;

let perm_intros x ps = perm_intros_rec [] x ps
;;

(* perm_elims : 
   constr signature -> (bool * identifier * ast) list 
   -> proto_form list * (bool * identifier * ast) list *)
let rec perm_elims_rec sign hs ps = function
 [] -> (List.rev hs,List.rev ps)
|(true,x,t)::ps'
 ->  perm_elims_rec sign ((At(x,etype_of_com sign t))::hs) ps ps'
|p::ps'
 -> perm_elims_rec sign hs (p::ps) ps'
;;

let perm_elims sign ps = perm_elims_rec sign [] [] ps
;;

(* args_cons : 
   int list -> int -> etype list -> xterm list
 NB : on suppose que les parametres du type sont tous en tete *)
let rec args_cons ns i = function
 [] -> []
|(Eprod(x,_))::das
 -> args_cons ns (succ i) das
|da::das 
 -> (Xvar (id_of_ints "x" (add_last i ns)))::(args_cons ns (succ i) das)
;;

(* com_of_arbrec : 
   constr signature -> identifier -> (xterm list * xterm) list
   -> proto_formlist -> (bool * identifier * ast) list -> etype 
   -> xterm -> arbrec -> ast *)
let rec com_of_arbrec sign f es hs ps d t a = 
 
 let rec com_of_arbrec_rec hs ps d t = function
 Frec n 
 -> let (ts,u) = List.nth es (n-1) in
    let u' = renommeX u t (Xft(f,ts)) in
    let u_0 = etendre f hs u' in
     List.fold_right (fun (_,x,t_0) u -> mkLambdaC(x,t_0,u)) ps u_0
|Brec((ns,d'),ars)
 -> (* pour Perm *)
    let x = id_of_ints "x" ns in
    let (ps1,ps2) = perm_intros x ps in
    let (hs',ps1') = perm_elims sign (except_last ps1) in
    (* pour Rec *)
    (* liste des arg de d' *)
    let ds = match d' with Ecste(_,ds) -> ds |_ -> [] in
    (* liste des cons de d' *)
    let cs = ids_cons_of d' sign in
    (* liste des types actuels des arg des cons *)
    let dacss = List.map fst (List.map (etype_actuel_of sign ds) cs) in
    (* args param actuels des cons *)
    let xpcs = List.map xterm_of_etype ds in
    (* liste des autres args des cons *)
    let i0 = succ (List.length ds) in
    let xss = List.map (args_cons ns i0) dacss in
    (* liste des xterms des cons *)
    let cs = List.map2 (fun c xics -> (term_cons c xpcs xics)) cs xss in
    (* listes des termes pour les cas des cons *)
    let tcs = List.map (fun c -> subst_idX c x t) cs in
    (* type(commmande) des hyp rec *)
    let hrC 
    = List.fold_right (fun (_,x,t_0) u -> mkProdC(x,t_0,u)) (ps1'@ps2) (com_of_etype d) in
    (* liste des premisses et des arg param des cons *)
    let (pcss,ppcss) 
    = List.split (List.map (prem_cons i0 d' hrC ns) dacss) in
    (* liste des prem des hr *)
    let xs = List.map (fun (_,x,_) -> x) (ps1'@ps2) in
    (* liste des hyp rec des cons *)
    let hiscs = List.map (hyp_cons d' xs t x ns i0) dacss in
    (* liste des res pour les appels rec de com_of_arbrec *)
    let us 
    = map4 (fun hisc ppcs tc ar 
            -> com_of_arbrec_rec (hisc@hs'@hs) (ppcs@ps1'@ps2) d tc ar)
           hiscs ppcss tcs ars in
    (* liste des res paraclos *)
    let us' 
    = map3 (fun u pcs ppcs 
            -> let u' = List.fold_left (fun t (_,x,_) -> mkAppC(t,nvar (string_of_id x))) u ppcs in
               List.fold_right (fun (x,t_0) u_0 -> mkLambdaC(x,t_0,u_0)) pcs u')
           us pcss ppcss in 
    let u =  mkRecC(hrC,nvar (string_of_id x),us') in
    let u' = List.fold_left (fun t (_,x,_) -> mkAppC(t,nvar (string_of_id x))) u ps1' in
      List.fold_right (fun (_,x,t_0) u_0 -> mkLambdaC(x,t_0,u_0)) ps1 u'
 in com_of_arbrec_rec hs ps d t a

;;

(* === Theoremes equationels *)
(* thm_eq :
   identifier -> identifier -> constr -> command -> State -> State *)
(* === Pour memoire
let thm_eq nom_f nom_th kind_f eq st =
 let st1 = theorem nom_th (Strength 0) st in
 let st2 = statement eq st1 in
  (* silent_by silent_trivial st2 *)
  by (THEN intros (THEN (unfold_nth [[],nom_f]) (apply (refl_eqC kind_f)))) st2
;;
=== *)

(* thm_eqs : 
   identifier -> command list -> unit *)
let rec thm_eqs nom_f eqs_f = 
 
 let rec thm_eqs_rec i = function
 [] -> ()
|(eq::eqs) 
 -> try
     let nom_th = make_ident ((string_of_id nom_f)^"_eq") i in
     (start_proof (string_of_id nom_th) NeverDischarge eq;
      solve_nth 1 Auto.full_trivial;
      save_named true;
      thm_eqs_rec (succ i) eqs)
    with
     UserError _ -> perror(0,[<'sTR"(equation "; 'sTR(string_of_int i);'sTR") :\n left and right members are not convertible">])
    |e -> raise e
 in thm_eqs_rec 1 eqs_f

;;

(* === Fonction principale *)
(* thf : 
   constr signature -> ast -> (bool * identifier * ast) list * etype *)
let rec thf sign t = 
 let rec thf_rec ps i = function
     Node(_,"PROD",[t;Slam(_,None,u)]) ->
       thf_rec ((false,id_of_ints "x" [i],t)::ps) (succ i) u
   | Node(_,"PROD",[t;Slam(_,Some x,u)]) ->
       let x'= id_of_ints "x" [i] in
         thf_rec ((true,x',t)::ps) (succ i)
           (subst_idC (nvar (string_of_id x')) (id_of_string x) u)
   | t -> (List.rev ps,etype_of_com sign t)
 in thf_rec [] 1 t

;;

(* close_eq : 
   identifier -> ast -> sorts oper term -> (identifier * ast) list
   -> ast list * ast -> ast *)
let close_eq nom_f type_res_fC kind_f typed_vars_eq eq_fC =
 let (ts,u) = eq_fC in
  closeC (eqC (appC_fx nom_f ts) u type_res_fC kind_f) typed_vars_eq
;;

(* renomme_rec : 
   etype list -> int -> etype -> etype list -> etype *)
let rec renomme_rec ds i d = function
 [] -> Efleche(List.rev ds,d)
|(Eprod(x,d'))::ds'
 -> let x'= id_of_ints "x" [i] in
    renomme_rec ((Eprod(x',d'))::ds) (succ i) 
                (subst_idE (Evar x') x d)(List.map (subst_idE (Evar x') x) ds')
|d'::ds' -> renomme_rec (d'::ds) (succ i) d ds'
;;

(* renommeE : 
   etype -> etype *)
let renommeE = function 
 Efleche(ds,d) -> renomme_rec [] 1 d ds
|_ -> anomaly "renommeE"
;;

(* Pour debuggage ..... *)
let it_ProPre = (ref (nvar("_")));;

(* ast_free_occ :
*)
let rec ast_free_occ x = function
    Nvar(_,y) -> x=(id_of_string y)
  | Node(_,"PROD",[t;Slam(_,None,u)]) ->
      ((ast_free_occ x t)or(ast_free_occ x u))
  | Node(_,"PROD",[t;Slam(_,Some y,u)]) ->
      (id_of_string y)<>x
      &((ast_free_occ x t)or(ast_free_occ x u))
  | Node(_,"APPLIST",ts) -> map_or (ast_free_occ x) ts
  | Node(_,"PROP",_) -> false
  | _ -> anomaly "ast_free_occ"
;;

(* update_ast_dep :
*)
let rec update_ast_dep = function
    Node(lp,"PROD",[t;Slam(ls,None,u)]) ->
      Node(lp,"PROD",[t;Slam(ls,None,update_ast_dep u)])
  | Node(lp,"PROD",[t;Slam(ls,Some x,u)]) ->
      if (ast_free_occ (id_of_string x) u)
      then Node(lp,"PROD",[t;Slam(ls,Some x,update_ast_dep u)])
      else Node(lp,"PROD",[t;Slam(ls,None,update_ast_dep u)])
  | Node(la,"APPLIST",ts) ->
      Node(la,"APPLIST",List.map update_ast_dep ts)
  |t -> t
;;

(* ProPre_rec_def : 
   (string * ast) list * identifier * ast * (ast list * ast) list 
   -> unit *)
let proPre_rec_def (nom_f,params_f,type_fC,eqs_fC) =
try
 let sign0 = initial_sign() in
 let params_f' 
  = mapapp (fun (xs,t) -> List.map (fun x -> (x,t)) xs) params_f in 
 let type_fC' = update_ast_dep (check_type_f sign0 params_f' type_fC) in
 let j1 = fconstruct (Evd.mt_evd()) sign0 type_fC' in
 let type_f = j1._VAL in
 let kind_f = j1._TYPE in
 let (typed_args_fC,type_res_fC) = flatC_Prod (string_of_id nom_f) type_f in
 let id_params_f = List.map fst params_f' in
 let params_RefC = List.map (fun (x,_) -> nvar (string_of_id x)) params_f' in
 let eqs_fC' = List.map (fun (ts,u) -> ((params_RefC@ts),u)) eqs_fC in
 let typed_vars_eqs_f 
  = check_vars_eqs sign0 id_params_f typed_args_fC eqs_fC' in
 let sign1 
  = add_sign 
      (nom_f,assumption_of_judgement (Evd.mt_evd()) (gLOB sign0) j1) sign0 in
 let eqs_f 
  = List.map2 (close_eq nom_f type_res_fC kind_f) typed_vars_eqs_f eqs_fC' in 
 let _ = check_type_eqs sign1 eqs_f in
 let eqs_fX = xterm_of_eqs sign0 nom_f id_params_f eqs_fC' in
 let type_fE = renommeE (etype_of_constr [] [] [] type_f) in
 let arbrec_f = cons_arbrec sign1 nom_f type_fE eqs_fX in
 let (idxtype_args_fC,type_res_fE) = thf sign0 type_fC' in
 let id_args_fX = List.map (fun (_,x,_) -> Xvar x) idxtype_args_fC in
 let fX = Xft(nom_f,id_args_fX) in 
 let fC = 
  com_of_arbrec sign1 nom_f eqs_fX [] idxtype_args_fC type_res_fE fX arbrec_f in
(* let fj = fconstruct (Evd.mt_evd()) (initial_sign()) fC in *)
  (it_ProPre := fC;
   definition_body nom_f NeverDischarge fC;
   thm_eqs nom_f eqs_f;
   message ((string_of_id nom_f)^" is recursively defined."))
with
 ProPre_error(_,x) ->
 errorlabstrm ""
 [< 'sTR"in Recursive Definition \"" ; 'sTR(string_of_id nom_f); 'sTR"\" " ; x >]
|UserError(s,pps) ->
 errorlabstrm ""
 [< 'sTR"in Recursive Definition \"" ; 'sTR(string_of_id nom_f) ; 'sTR"\" :\n" ; 
    'sTR s; 'sPC ; pps >]
|e -> raise e
;;

(* === Grammaire *)
vinterp_add
(
"RecursiveDefinition",
function
 ((VARG_IDENTIFIER f)::(VARG_BINDERLIST prms)::
  (VARG_COMMAND t)::es)
   -> (let pair_es = 
       (function 
         (VARG_COMMANDLIST (u::ts)) -> (ts,u) 
        |_ -> anomaly "vinterp_add.RecursiveDefinition") 
       in 
       let es' = List.map pair_es es in
        fun() -> proPre_rec_def(f,prms,t,es'))
|_ -> anomaly "vinterp_add.RecursiveDefinition"
);;

(* $Id: propre.ml,v 1.14 1999/06/29 07:48:00 loiseleu Exp $ *)
