(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                mimick.ml                                 *)
(****************************************************************************)

open Initial;;
open Std;;
open More_util;;
open Names;;
open Generic;;
open Evd;;
open Term;;
open Proof_trees;;
open Refiner;;
open Tacmach;;
open Pfedit;;
open Himsg;;
open Pp;;


type ttactic = 
  proof_tree -> proof_tree list * (proof_tree list -> proof_tree);;


let (outProof,inProof) =
  let (raw_outProof,raw_inProof) = System.extern_intern(1003,".pf") in
  let outProof filename (pf:proof_tree) = raw_outProof filename pf in
  let inProof filename = (raw_inProof filename : proof_tree) in
  (outProof,inProof)
;;

let saveProof s =
    let pts = get_pftreestate() in
    let pf = proof_of_pftreestate pts in
        outProof s pf
;;


let readProof s = inProof s;;

let associateProof pf gls = 
    context (set_mimick (Some pf) (pf_ctxt gls)) gls
;;

let eraseProof gls =
    match (pf_ctxt gls).mimick with
    None -> tclIDTAC gls
  | Some _ -> context (set_mimick None (pf_ctxt gls)) gls
;;

let eq_type {body=c1;typ=s1} {body=c2;typ=s2} = eq_constr c1 c2 & s1 =s2

let eq_goal g1 g2 =
    eq_constr g1.concl g2.concl &
    for_all2eq eq_type (vals_of_sign g1.hyps) (vals_of_sign g2.hyps) &
    for_all2eq (=) (ids_of_sign g1.hyps) (ids_of_sign g2.hyps)
;;


let first_result = fun f -> 
 let rec aux = function
    [] -> None
  | (h::t) -> try (Some (f h))
              with Failure _ | UserError _ -> aux t
 in aux
;;

let rec mimick pf gls =
    (match pf.ref with
     None -> tclIDTAC gls
   | Some (r,spfl) ->
     (tclTHEN
      (tclTRY (refiner r))
      (fun gls ->
           match first_result (fun pf -> if eq_goal (sig_it gls) pf.goal then pf
                                         else failwith "caught") spfl with
           Some pf -> mimick pf gls
         | None -> tclIDTAC gls))
     gls)
;;

let flat1 pf =
    (match (pf.subproof,pf.ref) with
     (None,_) -> error "flat1"
   | (Some subpf,Some(_,spfl)) ->
     let (cl,v) = frontier subpf
     in v spfl)
;;

(* for all x in S1 there exists y in S2 such that P(x,y) *)
let for_all_exists p s1 s2 =
    List.for_all (fun x -> List.exists (fun y -> p x y) s2) s1
;;

(* [Tactic Mimick] will mimick a proof, and whenever a step fails to
    produce children which are equal to the ones in the original
    proof, it will revert to the proof which is behind the current
    proof, failing if it doesn't find such a proof.

    The rules which we can handle are PRIM, CONTEXT, and TACTIC.  The
    others cause immediate failure.
 *)
let rec tacticMimick pf gls =
    (match pf.ref with
     None -> tclIDTAC gls
   | Some (r,spfl) ->
     (let (cls,v) = (tclTRY (refiner r) gls) in
      let cl = sig_it cls
      in if (not((for_all_exists (fun g spf -> eq_goal g spf.goal) cl spfl))) then
          begin
          mSGNL [< 'sTR"Mimick step " ; 'cUT ; pr_rule  r ;
                   'sTR" failed on goal " ; 'cUT ; prgl pf.goal >];
          ((tclORELSE (tacticMimick (flat1 pf)) (tclIDTAC))) gls
          end
         else
     ((tclTHEN ((tclTRY (refiner r))) (selectMimickSubgoal spfl))) gls))

and selectMimickSubgoal spfl gls =
    match first_result (fun pf -> if eq_goal (sig_it gls) pf.goal then pf
                                  else failwith "caught") spfl with
    Some pf -> tacticMimick pf gls
  | None -> tclIDTAC gls
;;


let rec mapshape nl (fl:((proof_tree list) -> proof_tree) list) 
                    (l:proof_tree list) =
    match nl with
    [] -> []
  | h::t ->
    let m,l = chop_list h l
    in (List.hd fl m)::(mapshape t (List.tl fl) l)
;;

let tTHEN (tac1:ttactic) (tac2:ttactic) g =
  let gl,p = tac1 g in
  let gll,pl = List.split(List.map tac2 gl) in
    (List.flatten gll) ,  ((comp p (mapshape(List.map List.length gll)pl)))
;;



let tIDTAC p = [p],(function [p] -> p | _ -> error "TIDTAC");;

let rec frontier p =
    match p.ref with
    None -> ([p],
                fun [p'] ->
                    if p'.goal = p.goal then p'
                    else error "frontier was handed back a ill-formed proof")
  | Some(r,pfl) ->
    let gll,vl = List.split(List.map frontier pfl)
    in (List.flatten gll,
        fun retpfl ->
            let pfl' = mapshape (List.map List.length gll) vl retpfl
            in {status = Refiner.and_status (List.map Refiner.pf_status pfl');
                subproof=p.subproof;
                goal = p.goal;
                ref = Some(r,pfl')})
;;

let leaf g = {status=INCOMPLETE_PROOF;
              subproof=None;
              goal=g;
              ref=None};;

let lEAF p = tIDTAC (leaf p.goal);;

let unTTAC ttac p = let (cl,v) = ttac p in v cl;;

let descend p =
    match p.ref with
    None -> error "descend"
  | Some(r,pfl) ->
    (pfl,
     (fun retpfl ->
          if for_all2eq (fun p1 p2 -> p1.goal = p2.goal) pfl retpfl then
              {status = Refiner.and_status (List.map Refiner.pf_status retpfl);
                subproof=p.subproof;
                goal = p.goal;
                ref = Some(r,retpfl)}
          else error "descend"))
;;

let expand p =
    match p.ref with
    None -> error "expand"
  | Some(TACTIC _,spfl) ->
    let (_,v) = frontier (outSOME p.subproof)
    in (spfl,v)
  | _ -> error "expand"
;;

let elideTactics l = 
 let rec eliderec pf =
    match pf.ref with
    None -> tIDTAC pf
  | Some (r,_) ->
    (match r with
     TACTIC(s,_) ->
     if List.mem s l then
         (tTHEN expand eliderec) pf
     else (tTHEN descend eliderec) pf
   | _ -> (tTHEN descend eliderec) pf)
 in eliderec
;;

let show_mimick_script () =
    let pts = get_pftreestate () in
    let pf = proof_of_pftreestate pts and
        evc = evc_of_pftreestate pts in

    let mimick_names = ["Mimick";"TacticMimick"]@
                       ["DumpMimick";"DumpTacticMimick"]@
                       ["DumpMimickWith";"DumpTacticMimickWith"] in
    let pf = unTTAC (elideTactics (mimick_names@["Interpret"])) pf in
    let pf = unTTAC (elideTactics mimick_names) pf
    in mSG(print_script true (ts_it evc) pf.goal.hyps pf)
;;

open Vernacinterp;;
open Constrtypes;;

vinterp_add
("WriteMimick",
 fun [VARG_STRING filename] ->
 fun () -> saveProof filename)
;;

vinterp_add
("ShowMimick",
 fun [] ->
 fun () -> show_mimick_script())
;;

let readDumpedProof () =
let ident = id_of_string(get_proof()) in
    let sp = Lib.make_path OBJ ident in
    let fname = "$MIMICKDIR/"^(string_of_path sp) in
        readProof fname
;;

let readDumpedProofFrom ident =
    let sp = Lib.make_path OBJ ident in
    let fname = "$MIMICKDIR/"^(string_of_path sp) in
        readProof fname
;;


hide_string_tactic "Mimick" (fun s -> mimick (readProof s));;
hide_string_tactic "TacticMimick" (fun s -> tacticMimick (readProof s));;
hide_atomic_tactic "DumpMimick" (fun gls -> mimick (readDumpedProof()) gls);;
hide_atomic_tactic "DumpTacticMimick" 
 (fun gls -> tacticMimick (readDumpedProof()) gls);;
hide_ident_tactic "DumpMimickWith" 
 (fun id -> mimick (readDumpedProofFrom id));;
hide_ident_tactic "DumpTacticMimickWith" 
 (fun id -> tacticMimick (readDumpedProofFrom id));;

let save_proof_to_standard_mimick_spot ident =
    let pf = proof_of_pftreestate(get_pftreestate()) in
    if (not(is_complete_proof pf)) then
        error "Cannot save incomplete proof of mimicking";
    let sp = Lib.make_path OBJ ident in
    let fname = "$MIMICKDIR/"^(string_of_path sp) in
        saveProof fname
;;

open Command;;
open Discharge;;
open Vernacentries;;

overwriting_vinterp_add("SaveNamed",
    fun [] ->
    fun () -> show_script() ;
              save_proof_to_standard_mimick_spot (id_of_string(get_proof()));
              save_named true);;

overwriting_vinterp_add("SaveAnonymousThm",
    fun [VARG_IDENTIFIER id] ->
    fun () -> show_script() ;
              save_proof_to_standard_mimick_spot id;
              save_anonymous_thm true (string_of_id id));;
overwriting_vinterp_add("SaveAnonymousRmk",
    fun [VARG_IDENTIFIER id] ->
    fun () -> show_script() ;
              save_proof_to_standard_mimick_spot id;
              save_anonymous_remark true (string_of_id id));;

overwriting_vinterp_add("TheoremProof",
    fun [VARG_STRING kind;VARG_IDENTIFIER s;VARG_COMMAND c;
         VARG_VARGLIST coml] ->
        let calls = (List.map (fun (VCALL(na,l)) -> (na,l)) coml) in
        let (stre,opacity) = (match kind with
                 "THEOREM" -> (NeverDischarge,true)
               | "LEMMA" -> (make_strength(safe_cdddr (Library.cwd())),true)
               | "FACT" -> (make_strength(safe_cdr (Library.cwd())),true)
               | "REMARK" -> (make_strength(Library.cwd()),true)
               | "DEFINITION" -> (NeverDischarge,false)
               | "LET" -> (make_strength(safe_cddr (Library.cwd())),false)
               | "LOCAL" -> (make_strength(Library.cwd()),false))
        in fun () ->
        try
        Library.with_heavy_rollback
        (fun () ->
            (start_proof (string_of_id s) stre c;
             show_open_subgoals();
             List.iter Vernacinterp.call calls;
             show_script();
             save_proof_to_standard_mimick_spot (id_of_string(get_proof()));
             save_named opacity))
        ()
        with e ->
            mSGNL [< 'sTR"Error during checking of theorem " ; print_id s ;
                          'sPC ; Errors.explain_sys_exn e ; 
                          'sPC ; 'sTR"... trying to mimick" >];
            restart();
            show_open_subgoals();
            solve_nth 1 (tacticMimick (readDumpedProof()));
            show_script();
            if (not((is_complete_proof(proof_of_pftreestate(get_pftreestate()))))) then
                error "Mimick failed"
            else save_named opacity)
;;

(* $Id: mimick.ml,v 1.20 1999/08/06 20:49:19 herbelin Exp $ *)
