(**************************************************************************
  *********                  ntcommand.ml                         *********
  **************************************************************************)

open More_util;;
open Generic;;
open Term;;
open Names;;
open Std;;
open Pp;;
open Machops;;
open Ntdef;;
open Nttop;;
open Ntpprinter;;
open Ntparam;;

(************************************************************************
  |  Pretty-print of text can be done in different languages.           |
  |  The rules that are common to all langauges are implemented in      |
  |   ppprooftext.ml.                                                   |
  |  The rule for french are in ppfrench.ml and for english in          |
  |   ppenglish.ml.                                                     |
  ************************************************************************)
let select_language_text id =
 match string_of_id id with
 | "English" -> Ppenglish.nc_set_pprinter_text ()
 | "French" -> Ppfrench.nc_set_pprinter_text ()
 | s -> errorlabstrm "ntcommand__select_language_text"
                     [< 'sTR "Unknown language: "; 'sTR s >];;

(************************************************************************
  | print_text print a proof (a term of kind Prop).                     |
  |                                                                     |
  | show_proof_text print the current proof (if of kind Prop)           |
  ************************************************************************)
let print_text id = pP (!p_definition (def_from_Rel_to_VAR (nt_def_of_id id)));;

let print_current_text () =
 pP (!p_definition (def_from_Rel_to_VAR (nt_def_of_current ())));;

(***********************************************************************
  ***********************************************************************)
let id_to_c id = global (gLOB (Vartab.initial_sign())) id;;

(************************************************************************
  |  The theorems or constructors in apply_omit_.._set are omitted      |
  | when they are in head position of an application                    |
  |  ex :                                                               |
  |    Using T with H1 and H2 we get P.  (T not omitted)                |
  |    With H1 and H2 we have P.         (T ommited)                    |
  |                                                                     |
  |  the inductively defined constants in elim_omit_cst_set are omitted |
  |  in elimination.                                                    |
  |  ex :                                                               |
  |    Using H with definition of C we get two cases. (C not omitted)   |
  |    Using H we get two cases. (C omitted)                            |
  ************************************************************************)
let mem_text_param_omit id = if match id_to_c id with
 | DOPN ((Const sp), _) -> ref_set_mem sp apply_omit_const_set
 | DOPN ((MutConstruct (x0,x1)), _) -> ref_set_mem (x0,x1) apply_omit_construct_set
 | DOPN ((MutInd (x0,x1)), _) -> ref_set_mem (x0,x1) elim_omit_cst_set
 | _ -> errorlabstrm "ntcommand__mem_text_param_omit" 
       [< 'sTR "Parametrization by this kind of construction not available" >]
 then
 print_string (string_of_id id ^ " is implicit in texts")
 else print_string (string_of_id id ^ " is not implicit in texts");;

let add_text_param_omit id =
 let p=match id_to_c id with
 | DOPN ((Const sp), _) -> Apply_omit_const sp
 | DOPN ((MutConstruct (x0,x1)), _) -> Apply_omit_construct (x0,x1)
 | DOPN ((MutInd (x0,x1)), _) -> Elim_omit_cst (x0,x1) 
 | _ -> errorlabstrm "ntcommand__add_text_param_omit"
       [< 'sTR "Parametrization by this kind of construction not available" >]
 in add_param p;;

let remove_text_param_omit id =
 let p=match id_to_c id with
 | DOPN ((Const sp), _) -> Apply_omit_const sp
 | DOPN ((MutConstruct (x0,x1)), _) -> Apply_omit_construct (x0,x1)
 | DOPN ((MutInd (x0,x1)), _) -> Elim_omit_cst (x0,x1) 
 | _ -> errorlabstrm "ntcommand__add_text_param_omit"
       [< 'sTR "Parametrization by this kind of construction not available" >]
 in remove_param p;;

let init_text_param_omit () =
 ref_set_init apply_omit_const_set;
 ref_set_init apply_omit_construct_set;
 ref_set_init elim_omit_cst_set;
 let f str = add_text_param_omit (id_of_string str) in
 List.iter f
  ["proj1";
  "proj2";
  "sym_eq";
  "sym_eqT";
  "eq_ind";
  "eq_ind_r";
  "eqT_ind";
  "eqT_ind_r";
  "conj";
  "or_introl";
  "or_intror";
  "ex_intro";
  "exT_intro";
  "refl_equal";
  "refl_eqT";
  "exist"; "nat"; "and"; "or"; "sig"; "False"; "eq"; "eqT"; "ex"; "exT"];;

(************************************************************************
  | The arguments of theorems or contructors in apply_rec_sub_*_set     |
  | are printed in a recursive way.                                     |
  |  ex :                                                               |
  |    * Using T with H1 and H2 we get P.                               |
  |    * Using T with H3 and H4 we get P'.                              |
  |    Using T with these two resulta we get P''.  (T is not in .._set) |
  |                                                                     |
  |    Using T with H1, H2, H3 and H4 we get P''.  (T is in .._set)     |
  |                                                                     |
  | Elimination on inductive constants in elim_rec_sub_set are used     |
  | in a similar way.                                                   |
  ************************************************************************)
let mem_text_param_rec_sub id = if match id_to_c id with
 | DOPN ((Const sp), _) -> ref_set_mem sp apply_rec_sub_const_set
 | DOPN ((MutConstruct (x0,x1)), _) -> ref_set_mem (x0,x1) apply_rec_sub_construct_set
 | DOPN ((MutInd (x0,x1)), _) -> ref_set_mem (x0,x1) elim_rec_sub_set
 | _ -> errorlabstrm "ntcommand__mem_text_param_rec_sub"
       [< 'sTR "Parametrization by this kind of construction not available" >]
 then
 print_string (string_of_id id ^ " arguments are contractible")
 else
 print_string
 (string_of_id id ^ " arguments are not contractible");;

let add_text_param_rec_sub id =
 let p=match id_to_c id with
 | DOPN ((Const sp), _) -> Apply_rec_sub_const sp
 | DOPN ((MutConstruct (x0,x1)), _) -> Apply_rec_sub_construct (x0,x1)
 | DOPN ((MutInd (x0,x1)), _) -> Elim_rec_sub (x0,x1)
 | _ -> errorlabstrm "ntcommand__add_text_param_rec_sub"
       [< 'sTR "Parametrization by this kind of construction not available" >]
 in add_param p;;

let remove_text_param_rec_sub id =
 let p=match id_to_c id with
 | DOPN ((Const sp), _) -> Apply_rec_sub_const sp
 | DOPN ((MutConstruct (x0,x1)), _) -> Apply_rec_sub_construct (x0,x1)
 | DOPN ((MutInd (x0,x1)), _) -> Elim_rec_sub (x0,x1)
 | _ -> errorlabstrm "ntcommand__add_text_param_rec_sub"
       [< 'sTR "Parametrization by this kind of construction not available" >]
 in remove_param p;;

let init_text_param_rec_sub () =
 ref_set_init apply_rec_sub_const_set;
 ref_set_init apply_rec_sub_construct_set;
 ref_set_init elim_rec_sub_set;
 let f str = add_text_param_rec_sub (id_of_string str) in
 List.iter f ["conj"; "or_introl"; "or_intror"; "proj1"; "proj2"];;

(************************************************************************
  | The dela-reduction or elimination on constant in immediate_*_set    |
  | are not explicit in texts.                                          |
  |  ex :                                                               |
  |   ... we get (T a b) which is equivalent to P. (T is not in .._set) |
  |   ... we get P. (T is in .._set)                                    |
  |                                                                     |
  |  ex :                                                               |
  |   Assume (and A B) (h).                                             |
  |   By definition of and, with this hypothesis we have A (h1)         |
  |     and B (h2).                             (and is not in .._set)  |
  |                                                                     |
  |   Asume A (h1) and B (h2).                  (and is in .._set)      |
  ************************************************************************)
let mem_text_param_immediate id = if match id_to_c id with
 | DOPN ((Const sp), _) -> ref_list_mem sp immediate_delta_red_list
 | DOPN ((MutInd (x0,x1)), _) -> ref_set_mem (x0,x1) immediate_elim_set
 | _ -> errorlabstrm "ntcommand__mem_text_param_immediate"
       [< 'sTR "Parametrization by this kind of construction not available" >]
 then
 print_string
 ("reduction or elimination on " ^ string_of_id id ^
   " are supposed to be transparent ")
 else
 print_string
 ("reduction or elimination on " ^ string_of_id id ^
   " are not supposed to be transparent ");;

let add_text_param_immediate id =
 let p=match id_to_c id with
 | DOPN ((Const sp), _) -> Immediate_delta_red sp
 | DOPN ((MutInd (x0,x1)), _) -> Immediate_elim (x0,x1)
 | _ -> errorlabstrm "ntcommand__add_text_param_immediate"
       [< 'sTR "Parametrization by this kind of construction not available" >]
 in add_param p;;

let remove_text_param_immediate id =
 let p=match id_to_c id with
 | DOPN ((Const sp), _) -> Immediate_delta_red sp
 | DOPN ((MutInd (x0,x1)), _) -> Immediate_elim (x0,x1)
 | _ -> errorlabstrm "ntcommand__add_text_param_immediate"
       [< 'sTR "Parametrization by this kind of construction not available" >]
 in remove_param p;;

let init_text_param_immediate () =
 ref_set_init immediate_elim_set;
 ref_list_init immediate_delta_red_list;
 let f str = add_text_param_immediate (id_of_string str) in
 List.iter f ["and"; "or"; "ex"; "exT"; "not"];;

let print_table table_name printer table =
  mSG (hOV 0 [< 'sTR table_name ;
                  if table=[] then [< 'sTR "None" >]
                  else [< hOV 0 (prlist_with_sep 
                              (fun () -> [< 'fNL >])
                              printer
                              table) >];
                  'fNL >]);;

let print_text_param_omit () =
  print_table "Implicit lemmas and theorems: "
    (fun sp -> [< 'sTR(string_of_path sp) >]) 
    (ref_set_elements apply_omit_const_set);
  print_table "Implicit proof constructors: "
    (fun ((sp,_),_) -> [< 'sTR(string_of_path sp) >]) 
    (ref_set_elements apply_omit_construct_set);
  print_table "Implicit inductive constants: "
    (fun (sp,_) -> [< 'sTR(string_of_path sp) >]) 
    (ref_set_elements elim_omit_cst_set);;

let print_text_param_rec_sub () =
  print_table "Contractible lemmas and theorems: "
    (fun sp -> [< 'sTR(string_of_path sp) >]) 
    (ref_set_elements apply_rec_sub_const_set);
  print_table "Contractible proof constructors: "
    (fun ((sp,_),_) -> [< 'sTR(string_of_path sp) >])
    (ref_set_elements apply_rec_sub_construct_set);
  print_table "Contractible inductive constants: "
    (fun (sp,_) -> [< 'sTR(string_of_path sp) >]) 
    (ref_set_elements elim_rec_sub_set);;

let print_text_param_immediate () =
  print_table "Definitions transparent for Natural: "
    (fun sp -> [< 'sTR(string_of_path sp) >]) 
    (ref_list_elements immediate_delta_red_list);
  print_table "Inductive constants transparent for Natural: "
    (fun (sp,_) -> [< 'sTR(string_of_path sp) >]) 
    (ref_set_elements immediate_elim_set);;

(***********************************************************************
  ***********************************************************************)

(* static initialization (by ML - not saved in Coq's environment) *)
Ppprooftext.nc_set_pprinter_text ();;
select_language_text (id_of_string "English");;

(* dynamic initialization (by vernacular - saved in Coq's environment) *)
let init_text () =
 init_False "False";
 select_language_text (id_of_string "English");
 init_text_param_omit ();
 init_text_param_rec_sub ();
 init_text_param_immediate ();;

