(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                multcase.ml                               *)
(****************************************************************************)

(* This file is linked to the system through  trad.ml in order to be able
 * to type during macro expansion. The functions here receive the closure
 * of exemeta_rec and some other functions from trad.ml.
 *)

open Std;;
open More_util;;
open Himsg;;
open Pp;;
open Vectops;;
open Generic;;
open Term;;
open Termenv;;
open Reduction;;
open Typing;;
open Names;;
open Multcase_astterm;; 
open Astterm;;
open Indrec;;
open Tradevar;;

(* Fonctions temporaires pour relier la forme caste de la forme jugement *)
let cenv_of_tenv (ENVIRON ((idl,tl),b)) =
  ENVIRON((idl,tl),
          List.map (fun (n,j) -> (n,DOP2(Cast,j.body,DOP0(Sort j.typ)))) b);;
let tenv_of_cenv (ENVIRON ((idl,tl),b)) =
  ENVIRON((idl,tl),
  List.map (fun (n,DOP2(Cast,c,DOP0(Sort s))) -> (n,{body=c;typ=s})) b);;
(* FIN TMP *)


(* == General purpose functions == *)

(* J'ai remplace strip_outer_cast par whd_castapp. A revoir. PAPAGENO 01/1999
let whd_cast = strip_outer_cast;;
***********)
let whd_cast = whd_castapp;;

let starts_with_underscore id = 
 let s=string_of_id id in (String.get s 0)='_';;


let makelist n elem =
 let rec makerec n  =
   if n=0 then []
    else elem::(makerec (n-1)) 
 in makerec n 
;;
 
let make_list k v = List.map (fun _ -> v)  (rel_list 0 k)
;;


let rec map_append_vect f v = 
  let length = Array.length v in 
  let rec map_rec i =
    if i>=length then [] 
    else (f v.(i))@ map_rec (i+1) 

  in map_rec 0  
;;

let rec  has_duplicates = function 
    [] -> None
  | x::l -> if List.mem x l then (Some x) else has_duplicates l
;;



(* behaves as elam_and_popl but receives an environment instead of a 
 *  dbenvironment
 *)
let elam_and_popl n env body l =
  let ENVIRON(sign,dbenv)=env  in
  let ndbenv,a,l = lam_and_popl n dbenv body l
  in ENVIRON(sign,ndbenv),a,l
;;


(* behaves as lam_and_popl_named but receives an environment instead of a 
 *  dbenvironment
 *)
let elam_and_popl_named n env body l  =
  let ENVIRON(sign,dbenv)=env  in
  let ndbenv,a,l = lam_and_popl_named n dbenv body l
  in ENVIRON(sign,ndbenv),a,l
;;
 

(* product_ize  [x1:A1]..[xn:An]u build (x1:A1)..(xn:An)u *)
let rec productize  lam  = 
  match strip_outer_cast lam with 
    DOP2(Lambda,ty,DLAM(n,bd)) -> mkProd n ty (productize bd)
  | x  -> x                      
;;

(* General functions on inductive types *)

let find_mutind_data sigma env ty =
  let (ity,largs) = find_mrectype sigma ty in
  let mispec = mind_specif_of_mind ity in 
  let nparams = mis_nparams mispec
  and arity = mis_arity mispec
  and nconstr = mis_nconstr mispec in 
  (ity,largs,nparams,arity,nconstr)
;;


let find_more_mutind_data sigma env ty = 
  let (ity,largs,nparams,arityind,nconstr) = find_mutind_data sigma env ty in 
  let (params,args) = chop_list nparams largs in 
  (ity,(params,args),nparams,arityind,nconstr)
;;

let mind_consnamev ity = mis_consnames (mind_specif_of_mind ity);;

let mind_nconstr ity = mis_nconstr (mind_specif_of_mind ity);;


(* yields the number of indexes of the inductive type ty *)
let nb_localargs sigma env ty =
try let (ity,_,nparams,arity,_)=find_mutind_data sigma env ty  in 
    max 0 (nb_prod arity - nparams)
with _ -> errorlabstrm "nb_localargs" [<'sTR "Not inductive type here" >]
;; 

let mutind_of_constructor c = 
  let (sp,i,j,args) = destMutConstruct c in 
  mkMutInd sp i args
;;

let ith_constructor i ity = 
  let (sp, tyi, cl) = destMutInd ity in
  mkMutConstruct sp tyi i cl
;;

(* determines whether is a predicate or not *)
let is_predicate sigma env ity = 
  let (_,_,nparams,arity,_) =  find_mutind_data sigma env ity
  in  (nb_prod arity) - nparams > 0
;;

(* Tests that hd of current pattern is a constructor of ity.
 * The number of arguments of the constructor has been checked in
 * pat_of_constr.
 *)
let is_constructor_of c ity = 
  let (sp,tyi,_) = destMutInd ity in
  let (sp',tyi',_,_) = destMutConstruct c in
  sp=sp' & tyi=tyi'
;;


(*---------------------------------------------------------------*
 *          Code for expansion of Cases macros                   *
 *---------------------------------------------------------------*)


(* == functions for replacing variables except in patterns == *)
(* (replace_var id t c) replaces for t all those occurrences of (VAR id) in c
 * that are not in patterns. It lifts t across binders.
 * The restriction is to avoid restoring parameters in patterns while treating
 * rhs.
 *)
(* PB: (replace_var_nolhs (VAR x) T <<Cases y of (C x) => x end>>)
 * will return <<Cases y of (C x) => T end>>, which is wrong!
 *)
let replace_var_nolhs var t x = 
 let rec substrec n = function
     (VAR(x)  as c)    -> if c=var then lift n t else c
   | DOPN(XTRA("EQN",cl),v) -> DOPN(XTRA("EQN",cl),Array.concat 
                                  [[|substrec n v.(0)|]; tl_vect v]) 
   | DOPN(oper,cl)    -> DOPN(oper,Array.map (substrec n) cl)
   | DOPL(oper,cl)    -> DOPL(oper,List.map (substrec n) cl)
   | DOP1(oper,c)     -> DOP1(oper,substrec n c)
   | DOP2(oper,c1,c2) -> DOP2(oper,substrec n c1,substrec n c2)
   | DLAM(na,c)       -> DLAM(na,substrec (n+1) c)
   | DLAMV(na,v)      -> DLAMV(na,Array.map (substrec (n+1)) v)
   | x                -> x in
 if eq_constr var t then x
 else substrec 0 x
;;

(* replaces occurrences of [(VAR id1)..(VAR idn)] respectively by t  in c *)
let replace_lvar_nolhs lid t c =
  List.fold_right (fun var c -> replace_var_nolhs var t c) lid c
;;


(* === Closures imported from trad.ml to perform typing === *)

type 'a trad_functions  = 
  { exemeta_rec : trad_constraint -> constr assumptions -> constr -> judgement;
    inh_ass_of_j : constr assumptions -> judgement -> constr;
    get_ise :     unit -> 'a Evd.evar_map
    };;


let mssg_should_be_constructor env patt =
  let pp = pTERMINENV(env,patt) in
  [<'sTR "Expecting a constructor in head of pattern " ; pp; 'sTR "." >]
;;

let mssg_wrong_num_arg_in_constr env constr patt n =
  let pp = pTERMINENV(env,patt) in
  let pc = pTERMINENV(env,constr) in
  [<'sTR "In pattern "; pp; 'sTR " the constructor "; pc;
    'sTR " expects " ; 'iNT n ; 'sTR " arguments. ">]
;;



(* The type of patterns.
 * Makes pattern-matching safer than on a constr, and we can do a few
 * checks we won't have to do afterwards.
 *)

(* invariant: the head of Cstr should be a MutConstruct and applied to the
 * right number of args.
 *)
type pattern =
    Var of identifier
  | Cstr of constr * pattern list
  | As of identifier * pattern;;

let rec lift_pattern n b p =
  match p with
    Var _ -> p
  | Cstr(c,args) -> Cstr(liftn n b c, List.map (lift_pattern n b) args)
    | As(id,p') -> As(id, lift_pattern n b p')
;;

let rec pat_vars vl = function
    Var id -> id :: vl
  | Cstr(_,pl) -> List.fold_right (fun p l -> pat_vars l p) pl vl
  | As(id,p) -> id :: pat_vars vl p
;;

let p_extract_lifted (k,p) = lift_pattern k 1 p;;
let p_insert_lifted p = (0,p)

let p_varsl pll =
  List.fold_right (fun (_,p) l -> pat_vars l p) pll []
;;

(* For pretty-printing. We keep the aliases. The ouput is not a well typed
 * term: in patterns, constructors are not applied to their args.
 *)
let rec pconstr_of_pat = function
    Var id -> VAR id
  | Cstr(c,args) -> applist(c, List.map pconstr_of_pat args)
  | As(id,p) -> DOPN(XTRA("AS",[]),[|VAR id; pconstr_of_pat p|])
;;

(* Partial check on patterns *)
let rec pat_of_constr_stk sigma env c appl =
  match c with
    DOPN(MutConstruct (_,i),_) ->
      let ity = mutind_of_constructor c in
      let nparams = mis_nparams (mind_specif_of_mind ity) in
      let tyconstr = type_mconstruct sigma i ity in
      let nb_args_constr = (nb_prod tyconstr) - nparams in
      let _ =
 	if List.length appl <> nb_args_constr
        then errorlabstrm "pat_of_constr" 
            (mssg_wrong_num_arg_in_constr env c (applist(c,appl))
	       nb_args_constr) in
      Cstr(c, List.map (pat_of_constr sigma env) appl)
  | DOPN(AppL,v) ->
      pat_of_constr_stk sigma env v.(0) (Vectops.app_tl_vect v appl)
  | DOP1(XTRA("!",[]),t) -> pat_of_constr_stk sigma env t appl
  | _ -> errorlabstrm "pat_of_constr"
	(mssg_should_be_constructor env (applist(c,appl)))

and pat_of_constr sigma env = function
  | VAR id -> Var id
  | DOPN(XTRA("AS",[]),[|VAR id; c|]) -> As(id, pat_of_constr sigma env c)
  | c -> pat_of_constr_stk sigma env c []
;;


(* renaming different occurrences of _ in pattern (before checking linearity!)
 *)
let rec rename_patt acc patt =
  match patt with
    Var x ->
      if starts_with_underscore x
      then let nid = next_ident_away x acc in (nid::acc, Var nid)
      else (acc, patt)
  | Cstr(c,args) ->
      let (nacc,nargs) = rename_lpatt acc args in
      (nacc,Cstr(c,nargs))
  | As(id,p) ->
      let (nacc,np) = rename_patt acc p in
      (nacc,As(id,np))

and rename_lpatt vl lp =
  List.fold_right
    (fun patt (acc,nlpatt) ->
      let (nacc,np) = rename_patt acc patt in (nacc, np::nlpatt))
    lp
    (vl,[])
;;

(* checking linearity of a list of ids in patterns *)
let non_linearl_mssg env lpatt id =
  let ppl = prlist_with_sep pr_spc (fun patt -> pTERMINENV(env,patt)) lpatt in
  [<'sTR "The variable " ; 'sTR(string_of_id id);
    'sTR " is bound several times in pattern(s) " ; ppl >] 
;;

(* lhs is used to report errors *)
let check_linearity env lpatt lhs =
  let ids = List.fold_right (fun p vl -> pat_vars vl p) lpatt [] in
  match has_duplicates ids with
    None -> () 
  | (Some id) -> 
      errorlabstrm "linearize_matx_patterns" (non_linearl_mssg env lhs id)
;;


(* Usage of this function should be avoided, because it requires that the
 * params are infered.
 *)
let rec constr_of_pat = function
    Var id -> VAR id
  | Cstr(c,args) ->
      let ity = mutind_of_constructor c in
      let mispec = mind_specif_of_mind ity in 
      let nparams = mis_nparams mispec in
      applist(c, makelist nparams mkExistential @ List.map constr_of_pat args)
  | As(id,p) -> constr_of_pat p
;;


(* == Error messages == *)

let mssg_ill_typed_branch (innermsg) =
 errorlabstrm "compile_multcase" 
   [<'sTR "Expansion strategy failed to build a well typed case expression.";
     'cUT; 'sTR "There is a branch that mistmatches the expected type."; 'fNL;
     hOV 1 [<'sTR "The risen type error on the result of expansion was:";
             'sPC; innermsg >]
   >];;

let mssg_number_of_patterns env lhs n =
  let plhs = prlist_with_sep pr_spc (fun patt -> pTERMINENV(env,patt)) lhs in
  [< 'sTR"Expecting ";'iNT n ; 'sTR" pattern(s) but found ";
    'iNT (List.length lhs); 'sTR" in "; plhs >]

let mssg_not_induc_here env constr ctyp =
  let pc = pTERMINENV(env,constr) in
  let pct = pTERMINENV(env,ctyp) in
      [< 'sTR "In Cases expression"; 'bRK(1,1); pc; 'sPC; 
         'sTR "has type"; 'bRK(1,1); pct; 'sPC; 
         'sTR "which is not an inductive definition" >]
;;

let mssg_wrong_predicate_arity env pred nondep_arity dep_arity=
  let pp = pTERMINENV(env,pred) in
    [<'sTR "The elimination predicate " ; pp; 'cUT;
      'sTR "should be of arity " ; 'iNT nondep_arity ; 
      'sTR " (for non dependent case)  or " ;
      'iNT dep_arity ; 'sTR " (for dependent case).">]
;;


let mssg_hd_is_not_constructor hd patt ity env =
  let pi = pTERMINENV(env,ity) in
  let pp = pTERMINENV(env,pconstr_of_pat patt) in
  let ph = pTERMINENV(env,hd) in
    [< 'sTR "Expecting a constructor of type "; pi; 
       'sTR " in pattern "; pp; 'bRK(1,1) ;
       'sTR " but found the head symbol " ; ph >]
;;

let warning_needs_dep_elim() =
  warning
"This pattern matching may need dependent elimination to be compiled.
I will try, but if fails try again giving dependent elimination predicate."
;;

let mssg_needs_inversion x t env =
  let px = pTERMINENV(env,x) in
  let pt = pTERMINENV(env,t) in
    [< 'sTR "Sorry, I need inversion to compile pattern matching of term ";
       px ; 'sTR " of type: "; pt>]
;;

let mssg_may_need_inversion () =
  [< 'sTR "This pattern-matching is not exhaustive.">]
;;

let mssg_mistmatch_type patt ity env =
  let pi = pTERMINENV(env,ity) in
  let pp = pTERMINENV(env,patt) in
    [< 'sTR "Constructor pattern: "; pp; 'bRK(1,1); 
       'sTR " cannot match values of type "; pi >]
;;


(* == Errors concerning mlcase == *)

let mssg_mlcase_infer_failure c env  =
  let pc = pTERMINENV(env,c) in
    (hOV 3 [<'sTR "Cannot infer type of expression :";'wS 1; pc>])
;;

let mssg_mlcase_not_inductive c env =
  let pc = pTERMINENV(env,c) in
    (hOV 3 [<'sTR "ML Case on a non inductive type :";'wS 1; pc>])
;;

let lift_list m  = List.map (fun (n,c) -> m+n,c) ;;



(* eta-expands the term t *)

let rec eta_expand0 n t = function
     DOP2(Prod,a,DLAM(na,b)) -> DOP2(Lambda,a,DLAM(na,eta_expand0 (n+1) t b))
   | DOP2(Cast,tt',_)        -> eta_expand0 n t tt'
   | _ -> appvect (lift n t, rel_vect 0 n)
;;

let rec eta_expand = function
     (DOP2(Cast,t',_), tt) -> eta_expand (t',tt)
   | (t, DOP2(Cast,tt',_)) -> eta_expand (t,tt')
   | (DOP2(Lambda,ta,DLAM(na,tb)), DOP2(Prod,_,DLAM(_,ttb))) ->
        DOP2(Lambda,ta,DLAM(na,eta_expand (tb,ttb)))
   | (t, tt) -> eta_expand0 0 t tt
;;

let eta_reduce_if_rel c =
  match eta_reduce_head c with
      Rel n -> Rel n
    | _ -> c;;

(* ===================================== *)
(*        DATA STRUCTURES                *)
(* ===================================== *)
let push a s = a::s;;
let push_lifted a s = (insert_lifted a)::s;;
let pop = function (a::s) -> (a,s) | _ -> error "pop";;
let empty = function [] -> true | _ -> false;;
let top = function (a::s) -> a | _ -> error "pop";;

(* Check *)
type 'a check = {past:'a list; 
                 future:'a list};;


let empty_ck: 'a check = {past=[]; future=[]};;

let to_past {past=p;future=f} = 
  if not (empty f) 
   then let (a,r)=pop f in  {past=push a p; future=r}
   else   error "move_left:right is empty"
;;


(* dependencies: terms on which the type of patterns depends
   patterns: list of patterns to analyse
   rhs: right hand side of equations
   Current pattern to analyse is placed in the top of patterns.future 
*)

type row = {dependencies: (int * constr) list; 
            patterns: (int * pattern) check; 
            rhs: int * constr};;


let row_current r = p_extract_lifted (top r.patterns.future);;
let pop_row_current ck = {past= ck.past; future= snd (pop ck.future)};;



(* ====================================================== *)
(* Functions to absolutize alias names of as-patterns in  *)
(*  a term                                                *)
(* ====================================================== *)
let rec whd_as = function
   As(_,p) -> whd_as p
 | x -> x
;;

let rec alias_ids = function
   As(id,p) -> (VAR id) :: (alias_ids p)
 | x -> []
;;

(* (expand_alias_and_hdname t rhs patt =
 *   if patt=(VAR id)  then  rhs[(VAR id) <- t]
 *   if patt = (p as id) then  (expand p t rhs)[(VAR id)<-t]
 *)
let absolutize_alias_and_hdname value rhs = function
    Var id -> replace_var_nolhs (VAR id) value rhs
  | As _ as patt ->
      let lid = alias_ids patt in
      let hd_id = constr_of_pat (whd_as patt)
      in List.fold_right (fun id rhs -> replace_var_nolhs id value rhs) 
               (hd_id::lid) rhs
 |  _ -> anomalylabstrm "absolutize_alias"
                  [<'sTR "pattern should be a variable or an as-pattern">]
;;


(* (absolutize_alias t rhs patt =
 *   if patt = (p as id) then  (expand p t rhs)[(VAR id)<-t] else rhs
 *)
let absolutize_alias value rhs = function
    Var _ -> rhs
  | As _ as patt ->
      let lid = alias_ids patt in
      List.fold_right (fun id rhs ->  replace_var_nolhs id value rhs) lid rhs
 |  _ -> anomalylabstrm "absolutize_alias"
                  [<'sTR "pattern should be a variable or an as-pattern">]
;;

 
let pop_and_prepend_future lpatt chk =
  let (_,fut) = pop chk.future 
  in {past=chk.past; 
       future= (List.map p_insert_lifted lpatt) @ fut}
;;

type matrix = row list ;;

(* info_flow allos to tag "tomatch-patterns" during expansion:
 * INH means that tomatch-pattern is given by the user
 * SYNT means that tomatch-pattern arises from destructuring a constructor
 *      (i.e. comes during top-down analysis of patterns)
 * INH_FIRST is used ONLY during dependent-Cases compilation. it tags the 
 *           first tomatch-pattern
 *)
type info_flow = INH | SYNT | INH_FIRST;;



(* If the case is non-dependent, the algorithm of compilation generates 
   the predicate P of the analysis using la 0eme definition. When 
   the case is dependent it should use the same strategy than rec.
   For that terms to match are tagged with INH or SYNT so decide 
   if pred should be inherited to branches or synthetised. 
   While left to right analysis
   of patterns the predicate is inherited, while top-down analysis of
   patterns predicate is synthetised, by doing anonymous abstractions when
   the non-dependent case is applied to an object of dependent type.
*)

type general_data = {case_dep : bool;
                     pred:  int* Term.constr;
                     deptm: (int*Term.constr) list; 
                     tomatch: (Term.constr*info_flow)  list;
                     mat:  matrix };;


let gd_current gd = top gd.tomatch;;
let pop_current gd = snd (pop gd.tomatch);;

let replace_gd_current cur gd =
let tm = pop_current gd 
in  {case_dep = gd.case_dep; pred=gd.pred; deptm=gd.deptm;
    tomatch = cur:: tm; mat = gd.mat}
;;

let replace_gd_pred pred gd = 
 {case_dep = gd.case_dep; 
  pred= insert_lifted pred; 
  deptm=gd.deptm; tomatch = gd.tomatch; mat = gd.mat}
;;


let prepend_tomatch ltm gd =
 {case_dep = gd.case_dep; pred=gd.pred; deptm=gd.deptm;
  tomatch = ltm@ gd.tomatch; mat = gd.mat}
;;

let pop_and_prepend_tomatch ltm gd =
let _,tm= pop  gd.tomatch  
in {case_dep= gd.case_dep; pred= gd.pred;   deptm = gd.deptm;
    tomatch = ltm@tm;
    mat = gd.mat}
;;


(* ========================================== *)
(*    Lifiting operations for general data    *) 
(* ========================================== *)

(* == Ops lifting all db-indexes == *)
let lift_check n chk = {past= lift_list n chk.past;   
                        future =  lift_list n chk.future}
;;

 
(* lifts n the the row *)
let lift_row n {dependencies=s; patterns=patt; rhs=(m,c)} = 
 {dependencies=lift_list n s; patterns=lift_check n patt ; rhs=(m+n, c)};;


(* pushes (na,t) to dbenv (that is a stack of (name,constr)  and lifts
 * tomach's dependencies, tomatch, pred and rhs in matrix 
 *)
let push_and_lift_gdl trad  (na,t) 
    (env, {case_dep=b; pred=(n,p); deptm=s; tomatch=tm; mat=m}, l) =
let tj = trad.exemeta_rec def_vty_con env t in
let assum = trad.inh_ass_of_j env tj in
let v = (na,assum) in 
(add_rel v env, {case_dep=b;
                 pred = (n+1,p);
                 deptm = lift_list 1 s;
                 tomatch = List.map (fun (c,x) -> (lift 1 c),x) tm ;
                 mat = List.map (lift_row 1) m
                }, 
                 List.map (fun (x,c) -> (x+1,c)) l)
;;

let push_and_lift_gd trad  (na,t) env gd =
let nenv,ngd,_ = push_and_lift_gdl trad (na,t) (env, gd, [])
in (nenv,ngd) ;;

(* if t is not (x1:A1)(x2:A2)....(xn:An)t' then (push_and_liftl n env t gd l)
 * raises an error else it gives ([x1,A1 ; x2,A2 ; ... ; xn,An]@env,t',gd')
 * where gd' is gd lifted n steps and l' is l lifted n steps
 *)
let push_and_lift_gdnl trad  n t envgdl =
let rec pushrec = function
    (0, t, envgdl) -> (envgdl, t)
  | (n, (DOP2(Prod,t,DLAM(na,b))), envgdl) 
       -> pushrec (n-1, b, (push_and_lift_gdl trad (na,t) envgdl))
  | (n, (DOP2(Cast,t,_)), envgdl) -> pushrec (n, t, envgdl)
  | (_, _, _) -> error "push_and_lift_gdn"
in pushrec (n,t,envgdl)
;;

let push_and_lift_gdn trad  n t dbenv gd  =
let (nenv,ngd,_),nt = 
      push_and_lift_gdnl trad n t (dbenv, gd, [])
in (nenv,nt,ngd)
;;

let push_and_lift_gdnl1 trad  n t (dbenv, gd, a) =
  match push_and_lift_gdnl trad n t (dbenv, gd, [a]) with
      (nenv,ngd,[na]),_ -> (nenv,ngd,na)
    | _ -> assert false
;;

let push_env_and_liftl_gdn trad n dbenv_base dbenv  gd l  =
  let rec pushrec base gd l dbenv n = 
    if n=0 then  base,gd,l
    else 
      try (match dbenv with 
	     (na, t):: x  ->
               let ndbenv,ngd,nl = pushrec base gd l x (n-1) in
		 push_and_lift_gdl  trad (na,t) (ndbenv, ngd, nl)
	     | _ -> assert false)
      with UserError _ -> error "push_env_and_lift_gd"
  in 
    pushrec dbenv_base  gd l (get_rels dbenv)  n
;;

let push_env_and_liftl2_gdn trad n dbenv_base dbenv  gd a1 a2 =
  match push_env_and_liftl_gdn trad n dbenv_base dbenv  gd [a1;a2] with
      (_,gd',[a1';a2']) -> (gd',a1',a2')
    | _ -> assert false;;

(* == Ops. pushing  patterns to tomatch and lifting == *)

(* adds a new pattern to check and lifts the matrix (i.e. rhs)*)

let push_patt_and_liftl trad (na,ty) info env gd l=
let (nenv,{case_dep=b; pred=p; deptm=d; tomatch=tm; mat=mt}, nl)= 
          push_and_lift_gdl trad  (na,ty) (env, gd, l)
in  (nenv,{case_dep=b;pred=p; deptm=d; 
           tomatch= ((Rel 1),info)::tm; mat=mt},nl)
;;


let push_patt_and_lift trad (na,ty) info env gd =
let (env,gd,_)=  push_patt_and_liftl trad (na,ty) info env gd []
in (env,gd)
;;

(* if t=(x1:P1)..(xn:Pn)Q behaves as push_and_lift_gd but if gd.patterns=patt 
   then the resutling gd'.patterns = (Rel n)..(Rel 1)patt
*)
let push_lpatt_and_liftl trad n t info env gd l=
let rec pushrec = fun
    p_0 p_1 p_2 -> 
     match p_0,p_1,p_2 with 
         (0, t, (env,gd,l)) -> (env,t,gd,l)
       | (n, (DOP2(Prod,t,DLAM(na,b))), (env,gd,l)) 
                -> pushrec (n-1) b 
             (push_patt_and_liftl trad  (na,t) info env gd l)
       | (n, (DOP2(Cast,t,_)), (env,gd,l)) -> pushrec n t (env,gd,l)
       | (_, _, _) -> error "push_and_lift_gdn"
in let nenv,body,ngd, nl = pushrec n t (env,gd,l) in
   let newpatt,oldpatt = chop_list n ngd.tomatch
   in (nenv,body,{case_dep=ngd.case_dep; pred=ngd.pred; 
                 deptm=ngd.deptm; 
                 tomatch= (List.rev newpatt) @ oldpatt; 
                 mat= ngd.mat},nl)
;;

let push_lpatt_and_lift trad n t info env gd =
let (env,b,gd,_) =
  push_lpatt_and_liftl trad n t info env gd []
in (env,b,gd)
;;



(* == lifting db-indexes greater equal a base index in gd == *)
(*  Ops. lifting indexes under b bindings (i.e. we lift only db-indexes
 *  that are >=  b
 *)

let extract_blift_ins n b c = 
      insert_lifted (liftn n b (extract_lifted c))
;;

let blift_list k b l =  List.map (extract_blift_ins k b) l ;;

let p_extract_blift_ins n b c = 
  (0, lift_pattern n b (p_extract_lifted c))
;;

let p_blift_list k b l =  List.map (p_extract_blift_ins k b) l ;;

let blift_check n b chk =
 {past= p_blift_list n b chk.past;   future = p_blift_list n b chk.future}
;;

(* lifts n the the indexes >= b of  row *)
let blift_row n b r = 
 {dependencies = blift_list n b r.dependencies;
  patterns=blift_check n b  r.patterns;
  rhs= extract_blift_ins n b r.rhs}
;;

 
  
let push_and_blift_gdl trad (na,t) b env gd l =
let tj = trad.exemeta_rec def_vty_con env t in
let assum = trad.inh_ass_of_j env tj in
let v = (na,assum) in 
(add_rel v env,{case_dep= gd.case_dep;
                pred = extract_blift_ins 1 b gd.pred;
                deptm = blift_list 1 b gd.deptm;
                tomatch = List.map (fun (c,x) -> (liftn 1 b c),x) gd.tomatch ;
                mat = List.map (blift_row 1 b) gd.mat
                }, 
                blift_list 1 b l)
;;


(* adds a new pattern to check and lifts those indexes of the matrix 
 * >= b (i.e. rhs)
 *)
let push_patt_and_bliftl trad (na,ty) info b env gd l=
let (nenv,{case_dep=cd; pred=p; deptm=d; tomatch=tm; mat=mt}, nl)= 
          push_and_blift_gdl trad (na,ty) b env gd l
in  (nenv,{case_dep=cd;
           pred=p; 
           deptm=d; 
           tomatch= ((Rel b,info))::tm; 
           mat=mt},nl)
;;


(* if t=(x1:P1)..(xn:Pn)Q behaves as push_lpatt_liftl but
 * it lifts only those indexes >= b. if gd.patterns=patt 
 *  then the resutling gd'.patterns = (Rel b+n)..(Rel b+1)patt
 *)
let push_lpatt_and_bliftl trad n t info b env gd l =
let rec pushrec = fun
    p_0 p_1 p_2 -> 
     match p_0,p_1,p_2 with 
         (0, t, (env,gd,l)) -> (env,t,gd,l)
       | (n, (DOP2(Prod,t,DLAM(na,body))), (env,gd,l)) 
                -> pushrec (n-1) body
                     (push_patt_and_bliftl trad (na,t) info b env gd l)
       | (n, (DOP2(Cast,t,_)), (env,gd,l)) -> pushrec n t (env,gd,l)
       | (_, _, _) -> error "push_and_lift_gdn"
in if b=1 then  
     push_lpatt_and_liftl trad n  t info env gd l
   else let nenv,body,ngd, nl = pushrec n t (env,gd,l) in
   let newpatt,oldpatt = chop_list n ngd.tomatch
   in (nenv,body,{case_dep=ngd.case_dep; pred=ngd.pred; 
                 deptm=ngd.deptm; 
                 tomatch= (List.rev newpatt) @ oldpatt; 
                 mat= ngd.mat},nl)
;;




(* =============================================================== *)


(* if tomatch=[Rel i1;...Rel in] of type 
   [(Ti1 p1_bar u1_bar);..(Tij pj_bar uj_bar)] then it yields
   [u1_bar;...uj_bar] 
*)
let find_depargs trad env tomatch =
let dep c =
 let tyj = trad.exemeta_rec mt_tycon env c
 in try let (ity,largs) = find_mrectype (trad.get_ise()) tyj._TYPE in 
        let nparams =  mind_nparams ity in 
        let (params,args) = chop_list nparams largs 
         in args
    with  Induc -> []
in map_append dep (List.map (fun (x,_) -> x) tomatch)
;;

(* == to treat  ml-case == *)

let make_pre_mlcase c lf = 
  (DOPN(XTRA("MLCASE",[Ast.str "REC"]), Array.append [|c|] lf))
;;

let dummy_mark =  (DOP0(XTRA ("SYNTH", [])));;

let rec hd_of_prodlam = function
   DOP2(Prod,_,DLAM(_,c))   -> hd_of_prodlam c
 | DOP2(Lambda,t,DLAM(_,c)) -> hd_of_prodlam c
 | DOP2(Cast,c,t)     -> hd_of_prodlam t
 | c   -> c
;;

let is_for_mlcase p =  (hd_of_prodlam p)=dummy_mark;; 

let has_synth t = dependent dummy_mark t;;

let rec dummy_lam n t = 
  if n=0 then t
    else (DOP2(Lambda,dummy_mark ,DLAM(Anonymous, dummy_lam (n-1) t)))
;;


(* == Special functions to deal with mlcase on objects of dependent types == *)

(* ================================================= *)
(*   Exceptions and functions for Error messages     *)
(* ================================================= *)


(* (CCError "function", env, gd, mssg) whre mssg=(string,Pp.std_ppcmds)
 * is the inner risen error_mssg
 *)
exception CCError of 
 string * Term.constr Term.assumptions * general_data * (Term.constr option) 
        * (string * Pp.std_ppcmds);;


let mssg_build_leaf  errenv errgd errrhs innermssg =
  let s,pstream= innermssg in
  let rhs = 
    (match errrhs with (Some rhs) -> rhs | 
       None -> invalid_arg "mssg_build_leaf") in
  let prhs = pTERMINENV(errenv,rhs) in
  let ermsg =
    [< 'sTR "The term "; prhs; 'sTR " is not well typed."; 'fNL; 
       hOV 1 [<'sTR "The risen type error on the result of expansion was:";
               'sPC; pstream >] 
    >]
in (errorlabstrm s ermsg)
;;
   

(* == functions for syntactic correctness test of patterns == *)


let patt_are_var =
  List.for_all
    (fun r -> match (whd_as (row_current r)) with
      Var _ -> true
    |_ -> false)
;;

let check_pattern sigma env (cur_tomatch,ity,params) row =
  let rec check_rec p =
    match p with
      Var id -> ()
    | Cstr(c,args) ->
 	if not (is_constructor_of c ity)
        then errorlabstrm "check_pattern"
	    (mssg_hd_is_not_constructor c p ity env)
    | As(_,p') -> check_rec p' 

  in check_rec (row_current row) 
;;

let patt_are_correct sigma env (cur_tomatch,ity,params) mat =
  List.iter (check_pattern sigma env (cur_tomatch,ity,params)) mat 
;;

(*The only variables that patterns can share with the environment are 
  parameters of inducive definitions!. Thus patterns should also be 
  lifted when pushing inthe context. *)


(* == functions to deal with names in contexts == *)

(* If cur=(Rel j) then 
 * if env = ENVIRON(sign,[na_h:Th]...[na_j:Tj]...[na_1:T1])
 * then it yields ENVIRON(sign,[na_h:Th]...[Name id:Tj]...[na_1:T1])
 *)
let change_name_rel env current id=
 match current with 
  (Rel j) -> 
    if starts_with_underscore id 
    then env 
    else let  ENVIRON(sign,db) = env in
     ( match chop_list (j-1) db with
	   db1,((_,ty)::db2) -> (ENVIRON(sign,db1@(Name id,ty)::db2))
	 | _ -> assert false)
 |  _  -> env
;;
 

(* == Function dealing with constraints in compilation of dep case == *)

let xtra_tm = DOP0(XTRA("TMPATT",[]));;
let is_xtra_tm tm = match tm with DOP0(XTRA("TMPATT",[])) -> true |_ -> false;;

(* represents the constraint cur=ci *)
let build_constraint cur ci = DOP2(XTRA("CONSTRAINT",[]),cur,ci);;

let top_constraint gd =
 match (extract_lifted gd.pred) with 
  DOP2(Prod,(DOP2(XTRA("CONSTRAINT",[]),cur,ci)),_) -> true 
 |   _ -> false
;;

let destruct_constraint gd =
 match (extract_lifted gd.pred) with 
  DOP2(Prod,(DOP2(XTRA("CONSTRAINT",[]),cur,ci)),bd) -> cur,ci,(lift (-1) bd)
 |   _ ->  anomalylabstrm "destruct_constraint" [<>]
;;

let rec whd_constraint = function
     DOP2(Prod,(DOP2(XTRA("CONSTRAINT",[]),_,_)),(DLAM(_,bd))) 
          -> whd_constraint (lift (-1) bd)
   | DOP2(Lambda,(DOP2(XTRA("CONSTRAINT",[]),_,_)),(DLAM(_,bd))) 
           -> whd_constraint (lift (-1) bd)
   | VAR(x)           -> (VAR x)
   | DOPN(oper,cl)    -> DOPN(oper,Array.map whd_constraint  cl)
   | DOPL(oper,cl)    -> DOPL(oper,List.map whd_constraint  cl)
   | DOP1(oper,c)     -> DOP1(oper,whd_constraint  c)
   | DOP2(oper,c1,c2) -> DOP2(oper,whd_constraint c1,whd_constraint c2)
   | DLAM(na,c)       -> DLAM(na,whd_constraint  c)
   | DLAMV(na,v)      -> DLAMV(na,Array.map whd_constraint  v)
   | x                -> x
;;

(* if next_pred = [d_bar:D_bar][h:(I~d_bar)]Q 
 * and bty = (y_bar:S_bar)(I~dep_ci)
 * and ci_params = (ci p_bar)
 * then it builds the next predicate containing the constraints in the correct
 * environment:
 * (y_bar:S_bar)
 *   (XTRA cur=(ci_param y_bar))->(next_pred dep_ci (ci_param dep_ci))
 *) 

(* PRE: gd.pred is a product correspondent to dependent elimination preidcate
 * productized (i.e. (d_bar:D_bar)(y:(I d_bar))Q 
 * It returns a product of the form
 * (s_bar:S_bar)Constraint->(whd_beta ([d_bar:D_bar][y:(I d_bar)]Q  dep_ci ci))
 * if info=INH then any constraints are generated
 *)
let  insert_constraint next_env gd brty ((cur,info), ci_param) = 
  let ENVIRON(_,dbenv)= next_env  in
  let k = nb_prod brty in
  let l = [insert_lifted cur; insert_lifted ci_param; gd.pred] in
  let dbenv0,body,l' = push_and_liftl k dbenv brty l in
  let cur0,cip0,npred0 =
    match l' with [a;b;c] -> (a,b,c) | _ -> assert false in
  let dep_ci = args_app body in
  let cip1 = appvect ((extract_lifted cip0),(rel_vect 0 k)) in 

  let npred1 = to_lambda (Array.length dep_ci) (extract_lifted npred0) in  

  let ndbenv,bodypred,nk =
    if Array.length dep_ci=1 (* type of cur is non-dependent *)  then 
      dbenv0, appvect (npred1, dep_ci),k 
    else   
      let app_pred = appvect (npred1, dep_ci) in
      if info = SYNT then  
       	let c = build_constraint (extract_lifted cur0) cip1 in
       	let dbenv1 = (Anonymous,c)::dbenv0
       	in dbenv1, (lift 1 app_pred), (k+1)

      else dbenv0,app_pred,k  (* we avoid generating the constraint*) in 
 
 (* we productize the constraint if someone has been generated *)
  let _,npred,_ = prod_and_popl nk ndbenv (whd_beta bodypred) []
  in npred
;;


(*--------------------------------------------------------------------------*
 * A few functions to infer the inductive type from the patterns instead of *
 * checking that the patterns correspond to the ind. type of the            *
 * destructurated object. Allows type inference of examples like            *
 *  [n]Cases n of O => true | _ => false end                                *
 *--------------------------------------------------------------------------*)

(* Computing the inductive type from the matrix of patterns *)
let strip_first_row mat =
  List.fold_right (fun pl (rw,smat) -> (List.hd pl::rw, List.tl pl::smat))
    mat ([],[])
;;

let rec find_row_ind = function
    [] -> None
  | (Var _) :: l -> find_row_ind l
  | Cstr(c,_) :: _ -> Some(mutind_of_constructor c)
  | As(_,p)::l -> find_row_ind (p::l)
;;

let find_pretype mat =
  let lpatt =
    List.map (fun r -> p_extract_lifted (List.hd r.patterns.future)) mat in
  match find_row_ind lpatt with
    Some ity -> ity
  | None -> anomalylabstrm "find_pretype" 
       	[<'sTR "Expecting a patt. in constructor form and found empty list">]
;;

(* Tries to unify ty (typed by a sort) with (tyi ?...?) *)
(* To improve the dirty hacks below, either put the_conv_x in the trad
 * record, or (better) extract the conversion function of Trad and link
 * it before multcase
 *)
let unify_with_inductive trad env ty tyi =
  let (ar,_) = decomp_prod (trad.get_ise()) (mind_arity tyi) in
  (* dirty (but easy) hack to declare the evars with the right type *) 
  let tyia = mkAppL (cons_vect tyi (Array.create ar mkExistential)) in
  let ity = (trad.exemeta_rec mt_tycon env tyia)._VAL in
  (* dirty hack to unify types ity and ty *)
  let dum_tm = Environ.lambda_create(ity,mkCast(mkRel 1) (lift 1 ty)) in
  let _ =
    try trad.exemeta_rec mt_tycon env dum_tm
    with UserError _ -> raise Induc in
  ()
;;

(* Split type ty as the application of an inductive type to arguments.
 * If ty is not an inductive type, we look if we can infer if from the
 * constructor names in the first row of mat. We define evars that will
 * stand for the params & args of this inductive type. Works fine for
 * the params, but we cannot get information from the branches to find
 * the value of the arguments...
 *)
let evar_find_mrectype trad env mat ty =
  try find_mrectype (trad.get_ise()) ty
  with Induc ->
    (unify_with_inductive trad env ty (find_pretype mat);
     find_mrectype (trad.get_ise()) ty)
;;

(* Same as above, but we do the unification for all the rows that contain
 * constructor patterns. This is what we do at the higher level of patterns.
 * For nested patterns, we do this unif when we ``expand'' the matrix, and we
 * use the function above.
 *)
let coerce_first_row trad env mat j =
  let (row,smat) = strip_first_row mat in
  (try let _ = find_mrectype (trad.get_ise()) j._TYPE in ()
  with Induc ->
    (match find_row_ind row with
      Some tyi -> unify_with_inductive trad env j._TYPE tyi
    | _ -> ()));
  smat
;;

let coerce_to_indtype trad env matx lj =
  let pats =
    List.map (fun r -> List.map p_extract_lifted r.patterns.future) matx in
  let _ = List.fold_left (coerce_first_row trad env) pats lj in
  ()
;;


(********************************)
(*  == compilation functions == *)
(********************************)

(* nbargs_iconstr is the number of arguments of the constructor i that are not
 * parameters. Current is the current value to substitute  "as" binders in
 * as-patterns 
 * About Expansion of as patterns: 
 * we absolutize alias names (x1..xn) in pattern in rhs by 
 * rhs [x1..xn <- gd.current]  if the type of current is not dep.
 * rhs [x1..xn <- (ci params lid)] otherwise
 * (in first case we share in second we rebuild the term)
 * Note: in the case of (VAR id) we use sharing whenver the type is non-dep
 * or we reconstruct the term when its dependent.
 * depcase tells if the current Case  to compile is dependent or not (this is
 * because during dependent compilation terms are always reconstructed and not
 * shared)
 * TODO: find a better criterion, or let the user choose...
 *)
let submat depcase sigma env i (ity, params) current mat =
  let ci = ith_constructor i ity in
  let k = List.length params in
  let nbargs_iconstr = nb_prod (type_mconstruct sigma i ity) - k in
  let paramsv = Array.of_list params in

(* builds the term to substitute those patterns that are variables or alias 
 * and the list of new patterns to push to the future *)
  let build_term r =
    let lid = global_vars (extract_lifted r.rhs) in
    let new_ids = get_new_ids nbargs_iconstr (id_of_string "t") lid in
    let lpatt = List.map (fun id -> Var id) new_ids in
    let largs = List.map (fun id -> VAR id) new_ids  
    in (appvect (ci, Array.concat [paramsv; Array.of_list largs])),lpatt in

  let rec expand_row aliasvar r =
    let cur = row_current r in
    match cur with
    | Var id ->
 	let (t,lpatt) = build_term r  in
        let value =
	  if (is_predicate sigma env ity) or depcase then t 
          else current in
        let nrhs = (*replace_lvar_nolhs [VAR id] t *)
		       (extract_lifted r.rhs) in
	let nnrhs = replace_lvar_nolhs (VAR id :: aliasvar) value nrhs in
	(None, [{dependencies=r.dependencies;
                  patterns = pop_and_prepend_future lpatt r.patterns;
                  rhs = insert_lifted nnrhs }])
    | Cstr(c,largs) ->
 	if c <> ci then (None,[])
	else
	  let value =
	    if (is_predicate sigma env ity) or depcase then constr_of_pat cur
            else current in
	  let nrhs =
	    replace_lvar_nolhs aliasvar value (extract_lifted r.rhs) in 
	  (None,[{ dependencies=r.dependencies;
                   patterns = pop_and_prepend_future largs r.patterns;
                   rhs = insert_lifted nrhs }])
    | As(id1,_) ->
       let npatt = whd_as cur in
       let aliasvar = (alias_ids cur) in
       let _,nrow = expand_row aliasvar
                      {dependencies=r.dependencies;
                       patterns = pop_and_prepend_future [npatt] r.patterns;
                       rhs = r.rhs } in
       let rnenv = change_name_rel env current id1 in
       (Some rnenv),nrow in
  let lenv,llrows = List.split (List.map (expand_row []) mat) in
  let lrows = map_append (fun x -> x) llrows in
  let lsome= filter (fun op -> op <> None) lenv in
  let rnenv = 
    if lsome = [] then env 
    else outSOME (List.hd lsome)
  in  rnenv, lrows
;;


type status =  Any_Tomatch | All_Variables | Match_Current 
            |  Any_Tomatch_Dep | All_Variables_Dep | Solve_Constraint ;;

let not_of_empty_type trad env (current,_) =
  let ct = (trad.exemeta_rec mt_tycon env current)._TYPE in
  try
    let (_,_,_,_,nconstr) = find_mutind_data (trad.get_ise()) env ct in
    nconstr <> 0
  with Induc -> true;;
  
let gdstatus trad env gd =
  if top_constraint gd  then Solve_Constraint
  else
    match gd.tomatch with
     [] -> if gd.case_dep then Any_Tomatch_Dep else Any_Tomatch
  | (current,info)::tl_tm ->
   if gd.mat=[] then 
     if (List.for_all (not_of_empty_type trad env) gd.tomatch)
     then errorlabstrm "gdstatus" 
       [< 'sTR "One should match a term in an empty inductive type"; 'fNL;
	  'sTR "to get an empty list of pattern" >]
     else Match_Current (* to treat empty types *)
   else
     if (patt_are_var gd.mat)
     then  if gd.case_dep then  All_Variables_Dep else  All_Variables
     else 
       if is_xtra_tm current then  Match_Current
       else
	 let curj=trad.exemeta_rec mt_tycon env current in
	 try
           let (ity,largs) = evar_find_mrectype trad env gd.mat curj._TYPE in
           let nparams =  mind_nparams ity in 
           let (params,args) = chop_list nparams largs in
           let _ =
	     patt_are_correct (trad.get_ise()) env (current,ity,params) gd.mat
           in Match_Current
	 with Induc  -> errorlabstrm "gdstatus" 
	     (mssg_not_induc_here env current curj._TYPE)
	   | UserError(a,b) -> errorlabstrm "gdstatus" b
;;


(* If S is the type of x and I that of y, then
 * solve_dep (x,S)(y,I) =
 * if S=(I u1.uj..un) and T=(I w1..wj..wn) where I has j parameters then
 *     u1=w1 &...& uj=wj & (solve u_j+1 w_j+1)& ..& (solve u_j+n w_j+n)
 *  else if S=(Rel i) and T=(Rel j) 
 *       else fail!
 * Note: this succeds only if uj+1...ur are all variables, otherwise 
 * inversion is needed.
 * WARNING!: here we compare using whd_cast is this enough or should we
 *  erase all cast before comparing??
 *)  
let  rec solve_dep sigma env (x,t) (y,s)  =
  let rec solve_args  = function
      [],[] -> []
    | (e1::l1), (e2::l2) ->
     	let e = whd_cast e1 in 
      	(match e with
          (Rel i) -> ((Rel i), e2)::(solve_args (l1,l2))
        | _ ->
	    if  e1= whd_cast e2 then (solve_args (l1,l2)) 
            else errorlabstrm "solve_dep" (mssg_needs_inversion  x t env))
    | _ -> anomaly  "solve_dep"  in
  try  
    let (ityt,argst)= find_mrectype sigma  t
    and (itys,argss)= find_mrectype sigma (hd_of_prod s)  in
    if whd_cast ityt= whd_cast itys & (List.length argst = List.length argss)
    then 
      let nparams = mind_nparams ityt in
      let (paramt,largst) = chop_list nparams argst
      and (params,largss) = chop_list nparams argss in
      if for_all2 
	  (fun a b -> strong whd_cast a= strong whd_cast b) paramt params 
      then  solve_args  (largst,largss) 
      else anomalylabstrm "solve_dep" 
          [<'sTR "mistake in the code building the branch!!" >]
    else anomalylabstrm "solve_dep" 
        [<'sTR "mistake in the code building the branch (pb:parameters)!">]
  with Induc -> anomalylabstrm "solve_dep" 
      [<'sTR "mistake in the code building the branch (pb: constructor)!">]
;; 

let apply_subst subst dep =
  let rec subst_term subst c = 
    match subst with 
      [] -> c
    | (Rel i, t)::s -> 
        if dependent (Rel i) c then
          if i = 1 then subst1 t c  
          else
	    let lsubstituend =
	      (List.rev (Array.to_list (rel_vect 0 (i-1))))@[t]
            in  subst_term s (substl lsubstituend  c)
        else  subst_term s c 
    | _ -> assert false   in
  let extr_subst_ins c =  insert_lifted (subst_term subst (extract_lifted c)) 
  in   List.map extr_subst_ins dep 
;;


let solve_dependencies trad env gd (current,ci)=
  if gd.deptm = [] then gd.deptm
  else
    let currentj = trad.exemeta_rec mt_tycon env current in
    let cij= trad.exemeta_rec mt_tycon env ci in 
    try
      let subst= solve_dep (trad.get_ise()) env 
          (currentj._VAL,currentj._TYPE) (cij._VAL,cij._TYPE) 
      in apply_subst subst  gd.deptm
    with UserError(a,b)-> errorlabstrm "solve_dependencies" b
;;


let substitute_dependencies dep row =
  let rec subst rhs = function 
      [], [] -> rhs
    | (t::x), (var::y) -> 
 	(match (extract_lifted var) with
	  (VAR id) as v  ->
	    subst (replace_var_nolhs v (extract_lifted t) rhs) (x,y)
  	| _ -> anomalylabstrm "substitute_dependencies" [<>] )
    | _,_ -> anomalylabstrm "substitute_dependencies" 
          [< 'sTR "Dep. tomatch mistmatch dependencies of patterns" >]

  in subst (extract_lifted row.rhs) (dep, row.dependencies)
;;


(* depends .. env cur tm = true if the the type of some element of tm
 * depends on cur and false otherwise
 *) 
(* I think that cur<>(Rel n) only during the first application of cc 
 * because later constructors in gd.tomatch are applied to variables
 *)
let depends trad env cur tm =
  let ENVIRON(sign,dbenv) = env  in
  match cur with
    (Rel n) ->
      let (gamma2,(na,t),_)= split_list n dbenv in
      let abs = lamn (List.length gamma2) gamma2 mkImplicit
      in dependent (Rel 1) abs 
  | _ -> false
;;


let lift_ctxt  k env =
  let ENVIRON(sign,dbenv) = env  in
  let delta,_ = decompose_prod (lift k (prod_it mkImplicit dbenv)) in
  ENVIRON(sign,delta)
;;


let split_ctxt j (ENVIRON(sign,db)) =
  let db1,db2= chop_list j db in
  (ENVIRON(sign,db1)), (ENVIRON(sign,db2))
;;

let prepend_db_ctxt (ENVIRON(sign1,db1)) (ENVIRON(sign2,db2)) =  
  ENVIRON(sign1, db1@db2)
;;


 
(* substitute_ctxt ci ENVIRON(sign, ([n1:U1]..[nj:Uj]))i = 
 *  substitutes ci by (Rel 1) in U1...Uj
 *)
let subst1_ctxt  ci env =
  let ENVIRON(sign,dbenv) = env  in
  let delta,_ = decompose_prod (subst1 ci (prod_it mkImplicit dbenv)) in
    ENVIRON(sign,delta)
;;

(* yields env if current pattern of first row is not a dummy var,
 * otherwise it undumize its identifier and renames the variable cur 
 * in context with the name of the current of the
 * first row.
 *)
let rename_cur_ctxt env cur gd =
  if gd.mat =[] then env
  else
    let r = List.hd gd.mat in
    let current = row_current r in
    match current with
      (Var id) when not (starts_with_underscore id) ->
 	change_name_rel env cur id
    | _ -> env
;;

(* supposes that if |env|=n then the prefix of branch_env coincides with env
 * except for the name of db variables
 *)
let common_prefix_ctxt env branch_env =
  let (ENVIRON(sign,db))=env in
  let (ENVIRON(_,branch_db)) = branch_env in 
  let j = List.length db in
  let rndb,_= chop_list j (List.rev branch_db)
  in (ENVIRON(sign, List.rev  rndb))
;;

let tm_depends trad env current ltm= 
  let rec dep_rec = function 
      [] -> false
    | (tm,_)::ltm -> 
   	let tmj = trad.exemeta_rec mt_tycon env tm 
   	in (dependent current (nf_ise1 (trad.get_ise()) tmj._TYPE))
	  or (dep_rec ltm)
  in dep_rec ltm
;;



(* substitutes the current of the row by t in rhs. If the current of the row 
 * is an as-pattern, (p AS id) then it   expands recursively al as in such
 * patern by by  (expand rhs p)[id<- t]
 *)
(* t is the current tomatch (used for expanding as-patterns) *)
let subst_current value r =
  let cur = row_current r in
  let nrhs = absolutize_alias_and_hdname value (extract_lifted r.rhs) cur 
  in   {dependencies = r.dependencies;
	 patterns = pop_row_current r.patterns;
	 rhs = insert_lifted nrhs} 
;;

(* t is the current tomatch (used for expanding as-patterns) *)
let shift_current_to_dep  r  =
  let curpatt = row_current r in
  let value = constr_of_pat (whd_as curpatt) in 
  let nrhs = extract_lifted r.rhs 
  in {dependencies = push (insert_lifted value) r.dependencies;
      patterns = pop_row_current r.patterns;
      rhs = insert_lifted (absolutize_alias value nrhs curpatt)}
;;



let castify_tylist trad  env l = 
  let rec cast_rec = function
      [] -> env,[]
    | ty::l -> 
    	let lenv,lcast = cast_rec  l in
    	let tyj =  trad.exemeta_rec def_vty_con lenv ty in
    	let ntyj =trad.inh_ass_of_j env tyj 
    	in (add_rel (Anonymous,ntyj) lenv), (ntyj::lcast)
  in snd (cast_rec l)
;; 


(* =========================================================================
 * the following functions determine the context of dependencies of a 
 * a vector of terms. They are useful to build abstractions wrt to dependencies
 * =========================================================================*)
(* tj._VAL = s and tj._TYPE = (T p1..pn u1..um) (s.t. p1..pn are parameters of T
 *  and T:(x1:P1)...(xn:Pn)(y1:B1)..(ym:Bm)s) then
 * (tyenv_and_args tj) = 
 *     ([B1;..;Bm;(T (lift m p1)..(lift m pn) (Rel m)..(Rel 1))], 
 *      [u1;..um; tj._VAL])
 *  in order to build later [y1:B1]..[ym:Bm](T p1'..pm' y1..ym)
 *  (where pi' = lift m pi)
 *)

let tyenv_and_args trad  env tj =
  let (ity,largs,nparams,arity,nconstr) = 
    find_mutind_data (trad.get_ise()) env tj._TYPE in
  let (params,args) = chop_list nparams largs in
  let lam_arity = to_lambda nparams arity in 
  let arity0 = whd_beta (appvect (lam_arity, Array.of_list params)) in
  let k = nb_prod arity0  in
  let env0,_,params0 =
    push_and_liftl k [] arity0 (List.map insert_lifted params) in
  let paramsv = Array.of_list (List.map extract_lifted params0) in
  let tyenv0 = List.map (fun (id,ty) -> ty) env0 in
  let t = appvect (ity, Array.concat [paramsv; rel_vect 0 k]) in
  let tyenv= castify_tylist trad  env (t::tyenv0) 
  in  tyenv, args@[tj._VAL]
;;


(* tj._VAL = s and tj._TYPE = (T p1..pn u1..um) (s.t. p1..pn are parameters of T
 *  and T:(x1:P1)...(xn:Pn)(y1:B1)..(ym:Bm)s) then
 * (tyenv_of_term tj) = [B1;..;Bm]
 *      [u1;..um; tj._VAL])
 *  in order to build later [y1:B1]..[ym:Bm]someterm
 *)

let tyenv_of_term sigma env tj =
  try 
    let (ity,largs,nparams,arity,nconstr) =
      find_mutind_data sigma env tj._TYPE in
    let (params,args) = chop_list nparams largs in
    let lam_arity = to_lambda nparams arity in 
    let arity0 = whd_beta (appvect (lam_arity, Array.of_list params)) in
    let env0,_ = decompose_prod arity0 
    in env0
  with Induc -> []
;;


(* =============================================== *)
(*   for mlcase                                    *)
(* =============================================== *)
(*
let abstract_pred sigma  env (pred,ltmj) =
let rec abst = function
    [] -> dummy_mark
 | (tj::ltm) -> 
     let res = abst ltm in
     let envty = tyenv_of_term sigma env tj in
     let _,nres,_ = lam_and_popl (List.length envty) envty res []
     in nres

in abst ltmj
;;
*)


(* if ltmj=[j1;...jn] then this builds the abstraction
 *  [d1_bar:D1_bar] ...[dn_bar:Dn_bar](lift m pred)
 *  where di_bar are the dependencies of the type ji._TYPE and m is the sum
 *  of the lengths of d1_bar ... d1_n. 
 * The abstractions are not binding, because the predicate is properly lift
 *) 
let abstract_predj  sigma env (pred,ltmj) =
  let rec abst = function
      [] -> pred
    | (tj::ltm) -> 
     	let res = abst ltm in
     	let envty = tyenv_of_term sigma env tj in
     	let k = (List.length envty)  in
     	let _,nres,_ = lam_and_popl k envty (lift k res) []
     	in nres
  in abst ltmj;;


(* idem as abstract_predj but receives a list of terms instead of judgments*)

let abstract_pred_lterms  trad  env (pred,ltm) = 
  let ltmj = List.map (trad.exemeta_rec mt_tycon env) ltm
  in  abstract_predj (trad.get_ise()) env (pred,ltmj)
;;

let abstract_pred_dummy sigma  env ltmj =
  abstract_predj sigma env (dummy_mark,ltmj)
;;





let info_abstract_predj  sigma env (pred,ltmj) =
  let rec abst linfo = function
      [] -> linfo,pred
    | (tj::ltm) -> 
     	let linfo,res = abst linfo ltm in
     	let envty = tyenv_of_term sigma env tj in
     	let k = (List.length envty)  in
     	let info = if k=0 then SYNT else INH in
     	let _,nres,_ = lam_and_popl k envty (lift k res) []
     	in (info::linfo),nres
  in abst [] ltmj;;


(* idem as abstract_predj but receives a list of terms instead of judgments*)

let info_abstract_pred_lterms  trad  env (pred,ltm) = 
let ltmj = List.map (trad.exemeta_rec mt_tycon env) ltm
in  info_abstract_predj (trad.get_ise()) env (pred,ltmj)
;;


(* if the type of cur is : (I p_bar d_bar) where d_bar are d_bar are 
 * dependencies, then this function builds (pred d_bar)
 * Precodition: pred has at least the same number of abstractions as d_bars 
 * length
 *)  
let apply_to_dep trad  env pred cur=
  let curj=trad.exemeta_rec mt_tycon env cur in 
  try
    (let k = nb_localargs (trad.get_ise())  env curj._TYPE in
    if k=0 then pred
    else
      let _,tyargs = tyenv_and_args trad env curj in
      let (ldep,_) = chop_list k tyargs
      in whd_beta (appvect (pred, Array.of_list ldep)))
  with _ -> pred
;;


(* if dummy_pred = [x1:Implicit]..[xk:Implicit]...[xk+j:Implicit]P 
 * and ipred = [y1:T1]..[yk:Tk]T
 * the result is [y1:T1]..[yk:Tk][xk+1:Implicit]...[xk+j:Implicit](lift j T)

 *** CADUC ? ***
let replace_dummy_abstractions dummy_pred ipred =
  let k = nb_lam ipred in
  let j = (nb_lam dummy_pred) - k in 
  let (env,body) = decompose_lam ipred in
  let (_,ndpred,_) = push_and_liftl k [] dummy_pred [] in
  let dummy_env = decompose_lam ndpred in
    lam_it env (lam_it dummy_env (lift j body));;
 *)

(* ==   == *)



(* analogue strategy as Christine's MLCASE *)
let find_implicit_pred trad ith_branch_builder env cj nconstr=
  let isrec = false in 
  let rec findtype i = 
    if i > nconstr then 
      errorlabstrm "find_implicit_pred"
 	(mssg_mlcase_infer_failure (Rel 80) env)
    else 
      try
 	(let expti = Indrec.branch_scheme (trad.get_ise()) isrec i cj._TYPE in
       	let _,bri= ith_branch_builder i  in
       	let fi = trad.exemeta_rec (mk_tycon expti) env bri in 
       	let efit = nf_ise1 (trad.get_ise()) fi._TYPE in 
       	let pred =
          (*Indrec.*)pred_case_ml_onebranch (tenv_of_cenv env) (trad.get_ise()) isrec 
            (cj._VAL,cj._TYPE) (i,fi._VAL,efit) in
 	if has_ise pred or (has_synth pred) then error"isevar" else pred)
      with UserError _ -> findtype (i+1)   in
  try  findtype 1  
  with Induc ->
    error "find_implicit_pred" (mssg_mlcase_not_inductive (Rel 80) env) 
;;

(* =============================================== *)
(* Strategies for building elimination predicates  *)
(* =============================================== *)
(* we build new predicate p for elimination  
 * by 0-splitting (we use inheritance or synthesis) 
 *)
let build_nondep_predicate trad env cty gd =
  let (current,info),tl_tm = pop gd.tomatch in
  let (ity,largs,nparams,arityind,_) =
    find_mutind_data (trad.get_ise()) env cty in 
  let (params,args) = chop_list nparams largs in 
  let abs = to_lambda nparams arityind in  
  let narity = whd_beta (appvect (abs, Array.of_list params)) in
  let n =  nb_prod narity  in 


(* gd.pred has the form [deptm_1]..[deptm_r]P (as given by the user, this is
 * an invariant of the algorithm) 
 * then p = [deptm_1] ([demptm_2]...[deptm_r]P   val_deptm_2...val_dpetm_r)
 *      next_pred = [deptm_1]..[deptm_r]P
 *)
  if not gd.case_dep then 
(* this computations is done now in build_nondep_branch
 let next_pred = if info=SYNT then lambda_ize n narity (extract_lifted gd.pred)                  else  extract_lifted gd.pred in
*)
    let next_pred = extract_lifted gd.pred  in
    let depargs = List.map insert_lifted (find_depargs trad  env tl_tm) in
    let (env0,pred0,depargs0) = push_lam_and_liftl n [] next_pred depargs in
    let depargs1= List.map extract_lifted depargs0 in
    let (_,p,_) =
      if depargs1=[] (*or n=0*)  
      then lam_and_popl n env0 pred0 []
      else lam_and_popl n env0
	  (whd_beta (appvect (pred0,Array.of_list depargs1))) [] 
    in (p,next_pred)
  else
    let pp = pTERMINENV(env, (extract_lifted gd.pred)) in
    errorlabstrm "build_nondep_predicate" 
      [<'sTR "Predicate  "; pp;
      	'sTR " is not correct for non-dependent elimination.">]
;;


(* TODO: Display in the message the nested dependent pattern found *)
let mssg_nested_dep_patt env mat =
[< 'sTR "Compilation of Dependent Cases fails when there are";'cUT;
   'sTR "nested patterns (in constructor form)  of dependent types."; 'cUT;
   'sTR "Try to transform your expression in a sequence of Dependent Cases"; 
   'cUT ; 'sTR "with simple patterns.">]
;;

(* ity is closed *)
let rec to_lambda_unif trad env n prod ity =
  if n=0 then prod 
  else match prod with 
    (DOP2(Prod,ty,DLAM(na,bd))) -> 
      let _ = if n=1 then unify_with_inductive trad env ty ity in
      mkLambda na ty (to_lambda_unif trad (add_rel (na,ty) env) (n-1) bd ity)
  | DOP2(Cast,c,_) -> to_lambda_unif trad env n c ity
  | _  -> error "to_lambda_unif"
;;

let build_dep_predicate trad env cty gd =
  if gd.case_dep then  
    let (current,info),tl_tm = pop gd.tomatch  in
    let (ity,largs,nparams,arityind,_) =
      find_mutind_data (trad.get_ise()) env cty in 
    let (params,args) = chop_list nparams largs in 
    let abs = to_lambda nparams arityind in  
    let narity = whd_beta (appvect (abs, Array.of_list params)) in
    let n =  nb_prod narity  in 

    if n=0 then (* the predicate is already depdent *)
      let npred = to_lambda_unif trad env 1 (extract_lifted gd.pred) ity
      in npred,npred
    else 
      if info=SYNT then 
     	errorlabstrm "build_dep_predicate" (mssg_nested_dep_patt env gd.mat)
      else 
  	let npred = to_lambda_unif trad env (n+1) (extract_lifted gd.pred) ity
  	in npred,npred
  
  else anomalylabstrm "build_dep_predicate"
      [<'sTR "build_dep_predicate was called with gd.case_dep=false ">]
;;

(* =================================== *)
(*   Principal functions               *)
(* =================================== *)

let my_norec_branch_scheme sigma i mind = 
  let typc =  type_inst_construct sigma i mind in
  let rec crec typc =
    match whd_betadeltaiota sigma typc with 
      DOP2(Prod,c,DLAM(name,t)) -> DOP2(Prod,c,DLAM(name,crec t))
    | (DOPN(AppL,v))  ->  appvect (mkExistential, tl_vect v)
    | _ -> mkExistential
  in crec typc
;;



let find_type_case_branches trad env  nconstr cj p = 
  if not (is_for_mlcase p) then
    let pj= trad.exemeta_rec mt_tycon env p in
    let evalct = nf_ise1 (trad.get_ise()) cj._TYPE in 
    let evalpt = nf_ise1 (trad.get_ise()) pj._TYPE in
    let (_,bty,rsty)=
      type_case_branches env (trad.get_ise()) evalct evalpt pj._VAL cj._VAL
    in bty
  else
    let build_branch i = my_norec_branch_scheme (trad.get_ise()) i cj._TYPE  in
    let lbr = List.map build_branch (interval 1 nconstr)  
    in  Array.of_list lbr
;; 

(* if ityparam :(d_bar:D_bar)s
 * then we abstract the dependencies and the object building the non-binding
 * abstraction  [d_bar:D_bar][_:(I param d_bar)]body_br 
 *)
(****************************************************
let abstract_generalized_args trad ityparam env body_br =
let arity= (trad.exemeta_rec (None,None) env ityparam)._TYPE in
let m =  nb_prod  arity in
let arityenv = decompose_prod arity in 
let nity = appvect (ityparam, (rel_vect 0 m)) in
let narityenv = (Anonymous,nity)::arityenv in
let (_,nbodybr,_) = lam_and_popl_named (m+1) narityenv
                         (lift (m+1) nbodybr) []
in nbodybr 
;;
**********************************************************)

(**************************************************************************)
(**** To allow arbitrary expression in tomatch and not only (Rel i) *******)

(* returns env',gd',args'  where:
 * If v1<>..<>vn and v1 ..vn are all variables then we push
 * (v1,INH)..(vn,INH) to gd.tomatch (gd') and  env'=env and args'=[]
 * Otherwise we push new variables ((Rel n),INH)...((Rel m),INH)
 *  to gd.tomatch (gd'),
 * we return their env (env') and args'=[v1..vn]
 *)

let  push_tomatch_rel trad env gd  ltmj = 
  let rec push_rec acc_ids env gd args = function 
      [] -> env,acc_ids,gd,args
    | tmj::l -> 
  	let tyenv,tyargs = tyenv_and_args trad  env tmj in
  	let k =  List.length  tyenv in
  	let idl = get_new_ids k  (id_of_string "v") acc_ids in
  	let dbenv0 = List.map2 (fun id t -> (Name id,t)) idl tyenv in
  	let (_,pr,_)= prod_and_popl k  dbenv0 mkImplicit [] in 
  	let env0,pr0,gd0 = push_and_lift_gdn trad  (k-1) pr env gd in
  	let env1,_,gd1 = push_lpatt_and_lift trad   1 pr0 INH env0 gd0 in
  	push_rec (acc_ids@idl) env1 gd1 (args@tyargs) l
  in 
  if List.for_all isRel (List.map (fun tmj -> tmj._VAL) ltmj)
  then (env, prepend_tomatch (List.map (fun tmj -> (tmj._VAL,INH)) ltmj) gd,[])
  else 
    let nenv,_,ngd,nargs = push_rec (ids_of_env env) env gd [] ltmj in
    let newpatt,oldpatt = chop_list (List.length ltmj) ngd.tomatch 
    in (nenv, {case_dep= ngd.case_dep; pred= ngd.pred;  deptm = ngd.deptm;
              	tomatch = (List.rev newpatt) @ oldpatt; 
              	mat = ngd.mat}, nargs)
;;

let  push_tomatch_dep trad  env gd  ltmj = 
  let rec push_rec gd args = function 
      [] -> (gd,args)
    | tmj::l -> 
  	let tyenv,tyargs = tyenv_and_args trad env tmj in
  	let ngd,resargs =  push_rec gd args l in
  	(prepend_tomatch [(tmj._VAL,INH)] ngd, (tyargs@resargs)) in
  let ngd,nargs = push_rec gd [] (List.tl ltmj) in
  let fst_tm = (List.hd ltmj)._VAL ,INH_FIRST in
  (env, prepend_tomatch [fst_tm] ngd, nargs)
;;

(* returns env,gd',args'  where:
 * if ltmj= e1...en then
 * we prepend (e1,INH_FIRST)(e2:INH)..(en,INH) to gd.tomatch (gd') and  
 * if not gd.case_dep  then env'=env and args'=[]
 *)
(*************************************************************
let  push_tomatch trad env gd ltmj = 
let env,args = 
  if not gd.case_dep  then env,[]
   else let env,_,args = push_tomatch_rel trad env gd ltmj 
        in env,args
in env, prepend_tomatch (List.map (fun tmj -> (tmj._VAL,INH)) ltmj) gd,args
;;
*****************************************************************)
let push_tomatch trad  env gd  ltmj = 
  if not gd.case_dep  then  
    env, prepend_tomatch (List.map (fun tmj -> (tmj._VAL,INH)) ltmj) gd,[]
  else push_tomatch_dep trad  env gd ltmj 
      
;;
(* ---------------------------------------------------------*)




type dependency_option = DEP | NONDEP;;

(* if ityparam :(d_bar:D_bar)s
 * then we abstract the dependencies and the object building the non-binding
 * abstraction  [d_bar:D_bar]body_br 
 *)
let abstract_generalized_args dependency_kind trad ityparam env body_br =
let arity= (trad.exemeta_rec mt_tycon env ityparam)._TYPE in
let m =  nb_prod arity in
  match dependency_kind with
    NONDEP ->  (lift m body_br)
  | DEP    ->
      let nity = appvect (ityparam, (rel_vect 0 m)) in
      let arityenv,_ = decompose_prod arity in
      let narityenv = (Anonymous, nity)::arityenv in
      let (_,nbodybr,_) = lam_and_popl_named (m+1) narityenv
                              (lift (m+1) body_br) [] in
        nbodybr 
;;


(* *)
let rec cc trad env gd = 
 match (gdstatus trad  env gd) with
   Match_Current -> 
     if not gd.case_dep then
       let (current,info),tl_tm= pop gd.tomatch 
       in if (isRel current) & (tm_depends trad env current tl_tm)
           then (* match_current  trad  env gd *)
                 (warning_needs_dep_elim() ;match_current  trad  env gd) 
           else match_current trad  env gd 
     else match_current_dep trad  env gd 

 | All_Variables -> substitute_rhs trad  env gd
 | Any_Tomatch   -> build_leaf trad env gd 

     (* for compiling dependent elimination *) 
 | All_Variables_Dep -> substitute_rhs_dep trad  env gd 
 | Any_Tomatch_Dep   -> build_leaf_dep     trad  env gd 
 | Solve_Constraint -> solve_constraint    trad  env gd 


and solve_constraint trad env gd =
 let cur,ci,npred= destruct_constraint gd in
 let ngd =  {case_dep = gd.case_dep; 
             pred = insert_lifted npred;
             deptm = solve_dependencies trad env gd (cur,ci);
             tomatch = gd.tomatch;
             mat = gd.mat}
 in cc trad env ngd


and build_dep_branch trad env gd bty ty i = 
 let (ity,(params,args),nparams,arityind,nconstr) =
        find_more_mutind_data (trad.get_ise()) env ty in 
 let n =  (List.length args)+1 in
 let k = nb_prod (type_mconstruct (trad.get_ise()) i ity) - nparams in
 let lpatt = make_list k (xtra_tm,SYNT) in
 let (current,info),_= pop  gd.tomatch  in 
 let gd0 =  pop_and_prepend_tomatch lpatt gd in

 let next_env,_,ngd,cur0,lp0 = 
   env,dummy_mark,gd0,insert_lifted current,(List.map insert_lifted params) in

 let cur = extract_lifted cur0 in
 let lp = List.map extract_lifted lp0 in 
 let lifted_params = Array.of_list lp in
 let ci_param = appvect (ith_constructor i ity, lifted_params) in

 let rnnext_env0,next_mat = submat ngd.case_dep (trad.get_ise()) next_env i 
                                    (ity,lp) cur ngd.mat in
 let next_predicate = insert_constraint next_env ngd bty.(i-1) 
                                       ((cur,info),ci_param)  in
 let next_gd = {case_dep = ngd.case_dep;
                pred =  insert_lifted next_predicate;
                deptm = ngd.deptm;
                tomatch = ngd.tomatch;
                mat = next_mat} in
 let brenv,body_br = cc trad rnnext_env0 next_gd in
 let branch =
  if empty next_gd.tomatch 
  then body_br (* all the generalisations done in the elimination predicate 
                * have been consumed *)
  else let (_,nextinfo),_ = pop  next_gd.tomatch  in
  if nextinfo=SYNT then body_br (* there's no generalisation to consume *)
   else (* consume generalisations in the elim pred. through abstractions *)
   match (gdstatus trad  rnnext_env0 next_gd) with
     Match_Current | All_Variables | All_Variables_Dep -> body_br
    | _ -> (* we abstract the generalized argument tomatch of 
            * elimination predicate [d_bar:D_bar][_:(I param d_bar)]body_br 
            *)
        let ityparam = appvect (ity,lifted_params) 
        in abstract_generalized_args DEP trad ityparam rnnext_env0 body_br
 in
 let rnnext_env = common_prefix_ctxt next_env brenv in
 rnnext_env,(eta_reduce_if_rel branch)
                           
(**************************************************************
 * to deal with multiple patterns of dependent families (ex. Nacira)!!
     
and  build_nondep_branch trad env gd next_pred bty ty i=
 let (ity,(params,args),nparams,arityind,nconstr) =
        find_more_mutind_data (trad.get_ise()) env ty in 
 let n =  (nb_prod arityind) -  nparams in 
 let k = nb_prod (type_mconstruct (trad.get_ise()) i ity) - nparams in
 let (current,info) = gd_current gd in 
 let (next_env,body,ngd,curlp) = 
   push_lpatt_and_liftl trad  k bty.(i-1) SYNT env
                 {case_dep= gd.case_dep; 
                  pred= insert_lifted next_pred; 
                  deptm = gd.deptm;
                  tomatch = snd (pop gd.tomatch);
                  mat = gd.mat}  
                  (List.map insert_lifted (current::params)) in
 let (cur::lp) = List.map extract_lifted curlp in 
 let lifted_params = Array.of_list lp in
 let ci = appvect (ith_constructor i ity, 
                   Array.concat [lifted_params; rel_vect 0 k]) in 

 let rnnext_env0,next_mat=submat ngd.case_dep (trad.get_ise()) next_env i 
                                 (ity,lp) cur ngd.mat in

 let ninfo,npred = 
  if n=0 then (SYNT, ngd.pred) (* as we treat only non-dep. elimination *)
  else let  dep_ci = args_app body in 
       if Array.length dep_ci=0   then  (info,ngd.pred) else
        let brpred = whd_beta (appvect (extract_lifted ngd.pred, dep_ci)) in
        let ciargs_patt =  List.map fst (fst (chop_list k ngd.tomatch)) in 
          (*we put pred. under same format that should be given by user
           * and set info to be INH, to indicate that dependecies have been
           * generalized *)
           let pred0 = abstract_pred_lterms trad 
                          next_env (brpred, ciargs_patt) 
           in (INH,(insert_lifted pred0))  in
 
 (* we change info of next current as current my pass from SYNT to INH
  * whenver dependencies are generalized in elimination predicate *)
 let ntomatch = 
   if empty ngd.tomatch then ngd.tomatch 
    else let ((next_cur ,_),nltm)= pop ngd.tomatch
         in push (next_cur, ninfo) nltm     in
 let next_gd = 
  {case_dep = ngd.case_dep;
   pred =  npred;
   deptm = solve_dependencies trad next_env ngd (cur, ci);
   tomatch = ntomatch ;
   mat = next_mat} in
 let brenv,body_br = cc trad rnnext_env0 next_gd in
 let rnnext_env = common_prefix_ctxt next_env brenv in
 let _,branch,_ = lam_and_popl_named  k (get_rels rnnext_env) body_br []
 in rnnext_env,(eta_reduce_if_rel branch)

******************************************************)

(* build__nondep_branch ensures the invariant that elimination predicate in 
 * gd is always under the same form the user is expected to give it.
 * Thus, whenever an argument is destructed, for each
 * synthesed argument, the corresponding predicate is computed assuring
 * that the invariant.
 * Whenever we match an object u of type (I param t_bar),where I is a dependent
 * family of arity (x_bar:D_bar)s, in order to compile we
 *  1) replace the current element in gd by  (Rel 1) of type (I param x_bar)
 *     pushing to the environment env the new declarations  
 *     [x_bar : D_bar][_:(I param x_bar)]
 *  2) compile new gd in the environment env[x_bar : D_bar][_:(I param x_bar)]
 *     using the type (I param x_bar) to solve dependencies 
 *  3) pop the declarations [x_bar : D_bar][_:(I param x_bar)] from the
 *     environment by abstracting the result of compilation. We obtain a
 *     term ABST. Then apply the abstraction ABST to t_bar and u.
 *     The result is whd_beta (ABST t_bar u)
 *)

and build_nondep_branch trad env gd next_pred bty ty i =
 let (ity,(params,args),nparams,arityind,nconstr) =
        find_more_mutind_data (trad.get_ise()) env ty in 
 let n =  (nb_prod arityind) -  nparams in 
 let k = nb_prod (type_mconstruct (trad.get_ise()) i ity) - nparams in
 let (current,info) = gd_current gd in 
  
 (* if current is not rel, then we replace by rel so to solve dependencies *)
 let (nenv,ncurargs,ncur,ncurgd,npred,nbty) = 
  if isRel current 
   then (env, [], current, gd, (insert_lifted next_pred), bty.(i-1))
   else let curj  = trad.exemeta_rec mt_tycon env current in  
        (* we push current and dependencies to environment *)
        let (relenv,_,relargs) = push_tomatch_rel trad env gd [curj] in

        (* we lift predicate and type branch w.r.t. to pushed arguments *) 
        let nrelargs = List.length relargs in
        let (curgd,lpred,lbty) = push_env_and_liftl2_gdn trad nrelargs 
             env relenv gd (insert_lifted next_pred) (insert_lifted bty.(i-1))
        in (relenv, relargs, (Rel 1), (replace_gd_current((Rel 1),info) curgd),
             lpred,  extract_lifted lbty)     in

 let (next_env,body,ngd,curlp) = 
   push_lpatt_and_liftl trad  k nbty SYNT nenv
                 {case_dep= ncurgd.case_dep; 
                  pred= npred;
                  deptm = ncurgd.deptm;
                  tomatch = snd (pop ncurgd.tomatch);
                  mat = ncurgd.mat}  
                  (List.map insert_lifted (ncur::params)) in

 match List.map extract_lifted curlp with
     [] -> assert false
   | (cur::lp) ->
 let lifted_params = Array.of_list lp in
 let ci = appvect (ith_constructor i ity, 
                   Array.concat [lifted_params; rel_vect 0 k]) in 

 let rnnext_env0,next_mat=submat ngd.case_dep (trad.get_ise()) next_env i 
                                 (ity,lp) cur ngd.mat in

 if next_mat = [] then
      (* there is no row in the matrix corresponding to the ith constructor *)
      errorlabstrm "build_nondep_branch" (mssg_may_need_inversion())
 else

  let (linfo,npred) = 
    let  dep_ci = args_app body in
    let brpred = if (n=0 or Array.length dep_ci=0) then 
                    (* elmination predicate of ngd has correct number
                     * of abstractions *)
                     extract_lifted ngd.pred 
                 else whd_beta (appvect (extract_lifted ngd.pred, dep_ci)) in
    let ciargs_patt =  List.map fst (fst (chop_list k ngd.tomatch)) in 
           (*we put pred. under same format that should be given by user
            * and set info to be INH, to indicate that dependecies have been
            * generalized *)
    let linfo,pred0 = info_abstract_pred_lterms trad next_env 
                                         (brpred, ciargs_patt) 
    in
    (linfo,(insert_lifted pred0))
  in
 
 (* we change info of next current as current my pass from SYNT to INH
  * whenver dependencies are generalized in elimination predicate *)
 let ntomatch = 
  let synt_tomatch, inh_tomatch = chop_list k ngd.tomatch in
  let nsynt_tomatch = List.map2 (fun info (tm,_) -> (tm,info))
                            linfo synt_tomatch
  in nsynt_tomatch @ inh_tomatch  in 
   
 let next_gd = 
  {case_dep = ngd.case_dep;
   pred =  npred;
   deptm = solve_dependencies trad next_env ngd (cur, ci);
   tomatch = ntomatch ;
   mat = next_mat}
 in

 let final_env, final_branch =
  let brenv,body_br = cc trad rnnext_env0 next_gd in
  let rnnext_env = common_prefix_ctxt next_env brenv in
  let branchenv,branch,_ = lam_and_popl_named  k (get_rels rnnext_env) 
                                body_br []
  in ENVIRON(get_globals rnnext_env, branchenv),branch
 in

 (* we restablish initial current by abstracting and applying  *)
 let p = List.length ncurargs  in
 let abstenv,abst,_ = lam_and_popl_named p (get_rels final_env)
                         final_branch [] in
 let app = whd_beta (appvect (abst, Array.of_list ncurargs)) in
 ENVIRON(get_globals final_env, abstenv),  (eta_reduce_if_rel app)


and match_current trad env gd =
  let (current,info),tl_tm = pop gd.tomatch in
  let cj = trad.exemeta_rec mt_tycon env current in
  let (ity,_,nparams,arity,nconstr) =
    find_mutind_data (trad.get_ise()) env cj._TYPE in 

  (* we build new predicate p for elimination  *)
  let (p,next_pred) = 
   build_nondep_predicate trad  env cj._TYPE gd    in

  (* we build the branches *)
  let bty = find_type_case_branches trad env nconstr cj p  in

  let build_ith_branch gd = build_nondep_branch trad env gd 
                               next_pred bty cj._TYPE   in
  let build_case ()=
  (* to print with Cases instead of Case we cast the matched element *)
    if nconstr=0 
    then env,(mkMutCaseA (ci_of_mind ity) (eta_reduce_if_rel p) current [||])
    else
    match List.map (build_ith_branch gd) (interval 1 nconstr) with
	[] -> assert false
      | (bre1,f)::lenv_f -> 
        let lf = f::(List.map snd lenv_f) in
        (common_prefix_ctxt env bre1,
         mkMutCaseA (ci_of_mind ity) (eta_reduce_if_rel p)
	   current (Array.of_list lf))
  in

  let build_mlcase () =
    if nconstr=0 
    then errorlabstrm "match_current" [< 'sTR "cannot treat ml-case">]
    else 
      let n = nb_localargs (trad.get_ise()) env ity in
      let np= extract_lifted gd.pred  in
      let k = nb_lam np in
      let (_,npred) = decompose_lam np in
      let next_gd = {case_dep= gd.case_dep; 
                     pred= insert_lifted npred;
                     deptm = gd.deptm;
                     tomatch = gd.tomatch;
                     mat = gd.mat}
      in
      try (* we try to find out the predicate and recall match_current *)
       (let ipred = find_implicit_pred trad (build_ith_branch  next_gd) 
                            env cj (Array.length bty) in
         (* we abstract with the rest of tomatch *)
        let env0,bodyp,_ = push_lam_and_liftl  n [] ipred [] in
         (*we put pred. under same format that should be given by user*)
        let ipred0 =
          abstract_pred_lterms trad env (bodyp, List.map fst tl_tm) in
        let (_,nipred,_) = lam_and_popl n env0 ipred0 []  in
        let explicit_gd = {case_dep= gd.case_dep; 
                           pred= insert_lifted nipred;
                           deptm = gd.deptm;
                           tomatch = gd.tomatch;
                           mat = gd.mat} 
 
        in match_current trad env explicit_gd)

      with UserError(_,s) -> errorlabstrm "build_mlcase" 
                  [<'sTR "Not enough information to solve implicit Case" >] 

  in if is_for_mlcase p then  build_mlcase ()
     else   build_case ()


and match_current_dep trad env gd = 
  let (current,info),tl_tm= pop gd.tomatch in
  let sigma = trad.get_ise() in 

  let nenv,current,ngd=  
    if info=SYNT then   (* we try to guess the type of current *)
      if nb_prod (extract_lifted gd.pred) >0  then 
       	let nenv,_,ngd= push_and_lift_gdn  trad  
                                1 (extract_lifted gd.pred) env gd
       	in nenv,(Rel 1),ngd 
      else anomalylabstrm "match_current_dep" 
          [<'sTR "sth. wrong with gd.predicate">] 
    else env,current,gd    (* i.e. current is typable in current env *)
  in       
  let cj = trad.exemeta_rec mt_tycon nenv current in
  let (ity,(params,args),nparams,arity,nconstr) = 
    find_more_mutind_data (trad.get_ise()) nenv cj._TYPE in 
  let _ = patt_are_correct sigma nenv (current,ity,params) gd.mat in
  (* we replace current implicit by (Rel 1) *) 
  let ngd0 = replace_gd_current (current,info) ngd in

  (* we build new predicate p for elimination  *)
  let (p,next_pred) = build_dep_predicate trad  nenv cj._TYPE ngd0  in

  let np=whd_constraint p in
  
  (* we build the branches *)
  let bty = find_type_case_branches trad nenv nconstr cj np  in

  let build_ith_branch env gd = build_dep_branch trad env gd bty cj._TYPE in

  let ngd1 = replace_gd_pred (to_prod (nb_lam next_pred) next_pred) ngd0 in

  let build_dep_case () = 
    if info=SYNT then 
      (*= builds [d_bar:D_bar][h:(I~d_bar)]<..>Cases current of lf end =*)

      let lf = List.map (fun i -> snd  (build_ith_branch nenv ngd1 i))
                            (interval 1 nconstr) in
      let case_exp =
 	mkMutCaseA (ci_of_mind ity) (eta_reduce_if_rel np)
	  current (Array.of_list lf) in
      let _,rescase,_ = elam_and_popl_named 1 nenv case_exp []
      in nenv,rescase
    else  
      if info = INH_FIRST then
     (*= Consumes and initial tomatch so builds <..>Cases current of lf end =*)
       let lf = List.map (fun i -> (snd (build_ith_branch nenv ngd1 i))) 
                        (interval 1 nconstr) 
       in nenv, (mkMutCaseA (ci_of_mind ity) (eta_reduce_if_rel np) current
		   (Array.of_list lf))
      else  (* we treat an INH value *)
       (* Consumes and initial tomatch abstracting that was generalized 
        *  [m:T] <..>Cases current of lf end  *) 
        let n = (List.length args)+1 in
        let nenv1,ngd2,np1 = push_and_lift_gdnl1 
                            trad   
                             n (extract_lifted gd.pred)
			     (nenv, ngd1, insert_lifted np) in
        let lf = List.map (fun i -> (snd (build_ith_branch nenv1 ngd2 i))) 
                       (interval 1 nconstr)   in
        (* Now we replace the initial INH tomatch value (given by the user) 
         * by (Rel 1) so to link it to the product. The instantiation of the
         * this (Rel 1) by initial value will be done by the application 
         *)
        let case_exp =
           mkMutCaseA None (extract_lifted np1) (Rel 1) (Array.of_list lf) in 
        let nenv2,rescase,_ = elam_and_popl_named n nenv1 case_exp []
        in nenv2,rescase
  in build_dep_case ()


and bare_substitute_rhs trad tm_is_dependent env gd =
 let (cur,info),tm = pop gd.tomatch in
 let nenv = rename_cur_ctxt env cur gd in
 let npred = 
   if gd.case_dep then gd.pred
    else (* le prochain argument n'est pas filtrable (i.e. parce que les
          *  motifs sont tous des variables ou parce qu'il n'y a plus de
          *  motifs), alors npred est gd.pred *)    
     let tmp_gd ={case_dep = gd.case_dep; pred = gd.pred; deptm = gd.deptm;
                  tomatch = tm; 
                  mat = List.map (subst_current cur)  gd.mat} in
     let pred0 = extract_lifted gd.pred in
     let pred1 = apply_to_dep trad  env pred0 cur 
     in insert_lifted pred1 in

 let ngd = if tm_is_dependent then
              {case_dep = gd.case_dep; 
               pred = npred;
               deptm = push_lifted cur gd.deptm;
               tomatch = tm; 
               mat = List.map shift_current_to_dep gd.mat}  
           else {case_dep = gd.case_dep; 
                 pred = npred;
                 deptm = gd.deptm;
                 tomatch = tm; 
                 mat = List.map (subst_current cur)  gd.mat}  
  in let brenv,res =  cc trad nenv ngd
     in (common_prefix_ctxt nenv brenv), res


(* to preserve the invariant that elimination predicate is under the same
 * form we ask to the user, berfore substitution we compute the correct
 * elimination predicat whenver the argument is not inherited (i.e. info=SYNT)
 *)
and substitute_rhs trad env gd =
 let (cur,info),tm = pop gd.tomatch in
 let nenv = rename_cur_ctxt env cur gd in
 let npred = 
  match info with
   SYNT -> (* we abstract dependencies in elimination predicate, to maintein
             * the invariant, that gd.pred has always the correct number of
             * dependencies *)
            (*we put pred. under same format that should be given by user*)
   (try let curj= trad.exemeta_rec mt_tycon env cur in 
         let npred = abstract_predj (trad.get_ise()) env
                               ((extract_lifted gd.pred),[curj])
         in insert_lifted npred
    with _ -> gd.pred )

  | _          -> gd.pred  in

 let ngd = {case_dep=gd.case_dep;
            pred= npred;
            deptm = gd.deptm; tomatch = gd.tomatch; mat = gd.mat} in
 let tm_is_dependent = depends trad  env cur tm 
 in bare_substitute_rhs trad tm_is_dependent env ngd


and substitute_rhs_dep trad env gd =
 let ((cur,info),_) = pop gd.tomatch in
 let nenv,npred,ngd =
   push_and_lift_gdn trad 1 (extract_lifted gd.pred) env gd in
 let ncur = (Rel 1) in
 let (_,ntm) = pop ngd.tomatch in
 let next_gd = {case_dep= gd.case_dep;
                 pred = insert_lifted npred;
                 deptm = ngd.deptm;
                 tomatch = [(ncur,info)]@ntm;
                 mat= ngd.mat}  in
 let tm_is_dependent = dependent ncur npred in
 let nenv0,body= bare_substitute_rhs trad tm_is_dependent nenv next_gd in
 let nenv1,nbody,_ = elam_and_popl_named 1 nenv0 body [] in
 let nbody = if info=INH_FIRST then applist(nbody,[cur]) else nbody in
 (nenv1, nbody)

and build_leaf trad env gd =
 if empty gd.mat then errorlabstrm "build_leaf" (mssg_may_need_inversion())
 else let t= if empty gd.deptm then extract_lifted ((top gd.mat).rhs)
              else  substitute_dependencies gd.deptm (top gd.mat) 
      in try env, (trad.exemeta_rec (mk_tycon (extract_lifted  gd.pred))
                                     env t)._VAL
        with UserError(s,stream) -> 
                raise (CCError("build_leaf",env,gd,(Some t),(s,stream)))

and build_leaf_dep trad env gd  = build_leaf trad env gd 
;;

(* *)

(* determines wether the multiple case is dependent or not. For that
 * the predicate given by the user is eta-expanded. If the result
 * of expansion is pred, then :
 * if pred=[x1:T1]...[xn:Tn]P and tomatchj=[|[e1:S1]...[ej:Sj]] then
 * if n= SUM {i=1 to j} nb_prod (arity Sj)
 *  then case_dependent= false
 *  else if n= j+(SUM (i=1 to j) nb_prod(arity Sj))
 *        then case_dependent=true
 *        else error! (can not treat mixed dependent and non dependnet case
 *)
(***********to accept elimination predicates that are not eta expanded **
let case_dependent trad env pred tomatchj =
let nb_dep_ity =
 (fun tmj -> try let (ity,largs)=find_mrectype (trad.get_ise()) tmj._TYPE  in
                 let arity=mind_arity ity in 
                 let nparams = mind_nparams ity in
                 nb_prod arity - nparams
             with _ -> errorlabstrm "case_dependent" 
                        (mssg_not_induc_here env tmj.Val tmj._TYPE))  in
let n=nb_lam pred in
let ndepv = List.map nb_dep_ity tomatchj in
let sum= List.fold_right (fun i j -> i+j)  ndepv 0 
in if sum= n then false 
   else if n=sum+ List.length tomatchj
         then true 
         else errorlabstrm "case_dependent" 
          (mssg_wrong_predicate_arity env pred sum (sum+ List.length tomatchj))
;;
****************************************)


let case_dependent trad env pred tomatchj =
  let nb_dep_ity tmj =
    try
      let (ity,largs)=find_mrectype (trad.get_ise()) tmj._TYPE  in
      let arity=mind_arity ity in 
      let nparams = mind_nparams ity in
      nb_prod arity - nparams
    with Induc -> errorlabstrm "case_dependent" 
	            (mssg_not_induc_here env tmj._VAL tmj._TYPE) in

  let predjudg = trad.exemeta_rec mt_tycon env pred in
  let etapred = eta_expand (pred, predjudg._TYPE) in
  let n=nb_lam etapred in
  let ndepv = List.map nb_dep_ity tomatchj in
  let sum= List.fold_right (fun i j -> i+j)  ndepv 0  in
  if sum= n then (false,etapred)
  else if n=sum+ List.length tomatchj
  then (true,etapred)
  else errorlabstrm "case_dependent" 
      (mssg_wrong_predicate_arity env etapred sum (sum+ List.length tomatchj))
;;


(* builds the matrix of equations testing that each row has n patterns
 * and linearizing the _ patterns.
 *)
let matx_of_eqns sigma env eqns n =
  let build_row = function
      (rhs::lhs) ->
      	let lpatt = List.map (pat_of_constr sigma env) lhs in
	let (_,rlpatt) = rename_lpatt [] lpatt in
      	let _ = check_linearity env rlpatt lhs in
    	if List.length rlpatt = n
    	then { dependencies = []; 
               patterns =
	       { past = [];
		 future = 
		 List.map p_insert_lifted rlpatt};
               rhs = insert_lifted rhs}
    	else errorlabstrm "matx_of_eqns" (mssg_number_of_patterns env lhs n)
    | _ -> errorlabstrm "expand_mastx_patterns"
	  [<'sTR "empty list of patterns">]

  in List.map build_row eqns
;;


let initial_gd trad env (pred,tomatchj,eqns) =
  let sigma = trad.get_ise() in
  let matx =
    matx_of_eqns sigma env (Array.to_list eqns) (List.length tomatchj) in
  let _ = coerce_to_indtype trad env matx tomatchj in
  let cd,npred  = 
    if is_for_mlcase pred
    then (false, abstract_pred_dummy sigma env tomatchj) 
    else
      let cdep,etapred = case_dependent trad env pred tomatchj in
      if cdep then (cdep, productize etapred) else (cdep,etapred) in
  { case_dep=cd;
    pred=insert_lifted npred; 
    deptm = []; 
    tomatch = []; 
    mat = matx }
;;




(* (mach_)compile_multcase:
 * The multcase that are in rhs are under the form of XTRA, as all ops as
 * substitution estc work under XTRA, when we arrive at the leaves we call
 * trad.exemeta_rec that will  call compile recursively.
 * compile (<pred>Case e1..en end of ...)= ([x1...xn]t' e1...en)
 * where t' is the result of the translation. 
 * INVARIANT for NON-DEP COMPILATION: predicate in gd is always under the same
 * form we ask the user to write <..>.
 * Thus, during the algorithm, whenever the argument to match is inherited
 * (i.e. info<>SYNT) the elimination predicate in gd should have the correct
 * number of abstractions. Whenever the argument to match is synthesed we have
 * to abstract all the dependencies in the elimination predicate, before 
 * processing the tomatch argument. The invariant is thus preserved in the
 * functions build_nondep_branch y substitute_rhs.
 * Note: this function is used by trad.ml 
 *)
let compile_multcase_fail trad vtcon env macro =
 let (pred,tomatch, eqns) = destruct_multcase macro in
 let tomatchl= Array.to_list tomatch in 
 let tomatchj = List.map (trad.exemeta_rec mt_tycon env) tomatchl in
 let npred =
   if is_for_mlcase pred then 
     (match vtcon with 
       (_,(_,Some p)) -> abstract_predj (trad.get_ise()) env (p,tomatchj) 
     | _ -> pred)
   else (trad.exemeta_rec mt_tycon env pred)._VAL in

 let gdi = initial_gd  trad env (npred, tomatchj, eqns) in

 (* we push the terms to match to gd *)
 let env1,gd,args = push_tomatch trad env gdi tomatchj in

 (* Now we compile, abstract and reapply the terms tomatch *)
 let brenv,body = cc trad env1 gd in 
 let rnenv1 = common_prefix_ctxt env1 brenv in
 let k = List.length (get_rels env1) - List.length (get_rels env) in
 let env2,abst,_ = elam_and_popl k rnenv1 body [] in
 let res = if args=[] then abst 
           else whd_beta (appvect (abst, Array.of_list args)) 
 in trad.exemeta_rec vtcon env2 res
;;


let compile_multcase (exemeta_rec,inh_ass_of_j,get_ise) vtcon env macro =
 let trad = {exemeta_rec=(fun vtyc env -> exemeta_rec vtyc (tenv_of_cenv env));
             inh_ass_of_j=(fun env j -> incast_type 
		 (inh_ass_of_j (tenv_of_cenv env) j));
             get_ise = get_ise} in
 try  compile_multcase_fail trad vtcon (cenv_of_tenv env) macro
 with UserError("Ill-formed branches", s) ->  mssg_ill_typed_branch (s)
    | CCError ("build_leaf", errenv, errgd,errrhs,innermssg) ->
         mssg_build_leaf  errenv  errgd errrhs innermssg
;;
