(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                               coqmktop.ml                                *)
(****************************************************************************)

(* Afin de rendre Coq plus portable, ce programme Caml remplace le script
   coqmktop. 
   
   Ici, on trie la ligne de commande pour en extraire les options spcifiques
    coqmktop, puis on appelle "ocamlc" (ou "ocamlopt")
   avec les autres options et la liste des fichiers  linker.

   On essaye au maximum d'utiliser les modules Sys et Filename pour que la
   portabilit soit maximale. On n'utilise pas Unix.
*)


(*******************)
(* Objects to link *)
(*******************)

(* 1. Core objects *)
let ocamlobjs = ["unix.cma"];;
let dynobjs=["dynlink.cma"];;
let camlp4objs = ["gramlib.cma"];;
let configobjs = ["coq_config.cmo"];;
let launchobjs = ["usage.cmo"];;
let utilobjs = ["utils.cma"];;
let libobjs = ocamlobjs @ camlp4objs @ configobjs @ launchobjs @ utilobjs;;

let parsingobjs = [
  "clexer.cmo"; "pcoq.cmo"; "ast.cmo"; "g_prim.cmo"; "extend.cmo";
  "egrammar.cmo"; "esyntax.cmo"; "g_command.cmo"; "g_multiple_case.cmo";
  "g_tactic.cmo"; "g_basevernac.cmo"; "coqAst.cmo"];;
let metaobjs = ["meta.cma"];;
let constrobjs = ["constr.cma"];;
let typingobjs = ["typing.cma"];;
let proofsobjs = ["proofs.cma"];;
let env1objs = ["env1.cma"];;
let env2objs = ["env2.cma"];;
let core_objs =
  libobjs @ parsingobjs @ metaobjs @ constrobjs @ typingobjs @ env1objs
  @ proofsobjs;;

(* 2. Files only in coqtop *)
let coq_parsing = ["g_vernac.cmo"];;
let coq_env = ["cmd_line.cmo"];;
let coq_tactics = ["tactics.cma"];;
let coq_src = coq_parsing @ coq_env @ coq_tactics;;

(* 3. Files only in coqsearchisos (if option -searchisos is used) *)
let coqsearch = ["version_searchisos.cmo"; "cmd_searchisos_line.cmo"];;

(* 4. Toplevel objects *)
let camlp4objs =
  ["camlp4_top.cma"; "pa_o.cmo"; "pa_op.cmo"; "pa_extend.cmo";
   "q_CoqAst.cmo" ];;
let topobjs = camlp4objs;;

let gramobjs = ["g_zsyntax.cmo"; "g_natsyntax.cmo"];;
let notopobjs = gramobjs;;

(* 5. High-level tactics objects *)
let cmotacsobjs = [
  "prolog.cmo"; "equality.cmo"; "inv.cmo"; "leminv.cmo"; 
  "point.cmo"; "progmach.cmo"; "program.cmo"; "propre.cmo";
  "tauto.cmo"; "gelim.cmo"; "eqdecide.cmo"];;

(* environment *)
let src_coqtop = ref Coq_config.coqtop
let notactics  = ref false
let opt        = ref false
let top        = ref false
let searchisos = ref false
let echo       = ref false

let includes () = 
  List.fold_right
    (fun d l -> "-I" :: List.fold_left Filename.concat !src_coqtop d :: l)
    [ [ "src" ; "config" ] ;
      [ "src" ; "launch" ] ;
      [ "src" ; "lib" ; "util" ] ;
      [ "src" ; "meta" ] ;
      [ "src" ; "constr" ] ;
      [ "src" ; "typing" ] ;
      [ "src" ; "proofs" ] ;
      [ "src" ; "parsing" ] ;
      [ "src" ; "env" ] ;
      [ "src" ; "tactics" ] ;
      [ "tactics" ] ;
      [ "tactics" ; "tcc" ] ;
      [ "tactics" ; "programs" ] ;
      [ "theories" ; "ZARITH" ];
      [ "theories" ; "ARITH" ]
    ]
    ["-I"; Coq_config.camlp4lib]

(* Transform bytecode object file names in native object file names *)
let native_suffix f =
  if Filename.check_suffix f ".cmo"
  then (Filename.chop_suffix f ".cmo") ^ ".cmx"
  else if Filename.check_suffix f ".cma"
  then (Filename.chop_suffix f ".cma") ^ ".cmxa"
  else failwith ("File "^f^" has not extension .cmo or .cma")

(* Transforms a file name in the corresponding Caml module name. *)
let rem_ext_regexpr = Str.regexp "\(.*\)\.\(cm..?\|ml\)"

let module_of_file name =
  let s = Str.replace_first rem_ext_regexpr "\1" (Filename.basename name) in
    String.capitalize s

(* Build the list of files to link and the list of modules names *)
let files_to_link userfiles =
  let dyn_objs=if not(!opt) then dynobjs else [] in
  let command_objs = if !searchisos then coqsearch else coq_src in
  let tactic_objs = if !notactics then [] else cmotacsobjs in
  let toplevel_objs = if !top then topobjs else if !opt then notopobjs else []
  in
  let objs = core_objs @ dyn_objs @ env2objs @ command_objs @ tactic_objs @
    toplevel_objs in
  let tolink =
    if !opt then (List.map native_suffix objs) @ userfiles
    else objs @ userfiles in
  let modules = List.map module_of_file tolink in
    (modules, tolink)

(*Gives the list of all the directories under dir*)
(*Uses Unix, sorry... But you can try to do the same thing without it.*)
let alldir dir=
  let ini=Unix.getcwd()
  and tmp=Filename.temp_file "coq" "rec"
  and lst=ref []
  in
    Unix.chdir dir;
    (let bse=Unix.getcwd()
     in
     let _ = Sys.command ("find "^bse^" -type d >> "^tmp) in
       let inf=open_in tmp
       in
         try
           (while true do
              lst:=(!lst)@[input_line inf]
            done;
            [])
         with
            End_of_file ->
              close_in inf;
              Sys.remove tmp;
              Unix.chdir ini;
              !lst);;

(* usage *)
let usage () =
  prerr_endline "Usage: coqmktop <options> <ocaml options> files
Options are:
  -srcdir dir   Specify where the Coq source files are
  -notactics    Do not link high level tactics
  -o exec-file  Specify the name of the resulting toplevel
  -opt          Compile in native code
  -top          Build Coq on a ocaml toplevel (incompatible with -opt)
  -searchisos   Build a toplevel for SearchIsos (implies -notactics)
  -R dir        Specify recursively directories for Ocaml\n";
  exit 1

(* parsing of the command line *)
let parse_args () =
  let rec parse (op,fl) = function
   
      [] -> List.rev op, List.rev fl

    | "-srcdir" :: d :: rem -> src_coqtop := d ; parse (op,fl) rem
    | "-srcdir" :: _        -> usage ()

    | "-notactics" :: rem -> notactics := true ; parse (op,fl) rem

    | "-opt" :: rem -> opt := true ; parse (op,fl) rem

    | "-top" :: rem -> top := true ; parse (op,fl) rem

    | "-searchisos" :: rem ->
        searchisos := true;
        notactics := true;
        parse (op,fl) rem

    | "-echo" :: rem -> echo := true ; parse (op,fl) rem

    | ("-cclib"|"-ccopt"|"-I"|"-o" as o) :: rem' ->
	begin
	  match rem' with
	      a :: rem -> parse (a::o::op,fl) rem
	    | []       -> usage ()
	end

    | "-R" :: rem' ->
        begin
	  match rem' with
	      a :: rem ->
                parse ((List.rev(List.flatten (List.map (fun d -> ["-I";d])
                  (alldir a))))@op,fl) rem
	    | []       -> usage ()
	end

    | ("-compact"|"-g"|"-p" as o) :: rem -> parse (o::op,fl) rem

    | ("-h"|"--help") :: _ -> usage ()

    | f :: rem ->
	if Filename.check_suffix f ".ml" 
	  or Filename.check_suffix f ".cmx" 
	  or Filename.check_suffix f ".cmo"
	  or Filename.check_suffix f ".cmxa"
	  or Filename.check_suffix f ".cma" then
	  parse (op,f::fl) rem
	else begin
	  prerr_endline ("Don't know what to do with " ^ f);
	  exit 1
	end

  in
    parse ([Coq_config.osdeplibs],[]) (List.tl (Array.to_list Sys.argv))

let clean file =
  let rm f = if Sys.file_exists f then Sys.remove f in
  let basename = Filename.chop_suffix file ".ml" in
    if not !echo then
      begin
        rm file;
        rm (basename ^ ".o");
        rm (basename ^ ".cmi");
        rm (basename ^ ".cmo");
        rm (basename ^ ".cmx")
      end

(*Gives all modules in dir. Uses .cmi. Unix again sorry again*)
let all_modules_in_dir dir=
  try
    (let lst=ref []
     and stg=ref ""
     and dh=Unix.opendir dir
     in
       try
         (while true do
	    stg:=Unix.readdir dh;
            if (Filename.check_suffix !stg ".cmi") then
              lst:=(!lst)@[String.capitalize (Filename.chop_suffix !stg
                ".cmi")]
          done;
          [])
       with
          End_of_file -> Unix.closedir dh; !lst)
  with
     Unix.Unix_error (_,"opendir",_) ->
       failwith ("all_modules_in_dir: directory "^dir^" not found");;

(*Gives a part of command line (corresponding to dir) for extract_crc*)
let crc_cmd dir=
  " -I "^dir^(List.fold_right (fun x y -> " "^x^y) (all_modules_in_dir dir)
  "");;

(*Same as crc_cmd but recursively*)
let rec_crc_cmd dir=
  List.fold_right (fun x y -> x^y) (List.map crc_cmd (alldir dir)) "";;

(*Creates another temporary file for Dynlink if needed*)
let tmp_dynlink()=
  let tmp = Filename.temp_file "coqdynlink" ".ml" in
  let _ = Sys.command ("echo \"Dynlink.init();;\" > "^tmp) in
  let _ = Sys.command (Coq_config.camllib^"/extract_crc"^(crc_cmd
      Coq_config.camllib)^(crc_cmd Coq_config.camlp4lib)^(rec_crc_cmd
      (Coq_config.coqtop^"/src"))^" >> "^tmp) in
  let _ = Sys.command ("echo \";;\" >> "^tmp) in
  let _ = 
    Sys.command ("echo \"Dynlink.add_available_units crc_unit_list;;\" >> "^
		 tmp) in
    tmp;;

(*Initializes the kind of loading in the main program*)
let declare_loading_string()=
  if !opt then
    "Mltop.set Mltop.Native;;\n"
  else
    if not(!top) then
      "Mltop.set Mltop.WithoutTop;;\n"
    else
      "Mltop.set (Mltop.WithTop {Mltop.load_obj=Topdirs.dir_load;
                                 Mltop.use_file=Topdirs.dir_use;
                                 Mltop.add_dir=Topdirs.dir_directory});;\n";;

(* create a temporary main file to link *)
let create_tmp_main_file modules =
  let main_name = Filename.temp_file "coqmain" ".ml" in
  let oc = open_out main_name in
    try
      (* Add the pre-linked modules *)    
      output_string oc "List.iter Mltop.add_known_module [\"";
      output_string oc (String.concat "\";\"" modules);
      output_string oc "\"];;\n";

      (*Initializes the kind of loading*)
      output_string oc (declare_loading_string());
      (* Start the right toplevel loop: Coq or Coq_searchisos *)
      if !searchisos
      then output_string oc "Cmd_searchisos_line.start();;\n"
      else output_string oc "Cmd_line.start();;\n";
      (* Start the Ocaml toplevel if it exists *)
      if !top then
        output_string oc "Printexc.catch Toploop.loop(); exit 1;;\n";
      close_out oc;
      main_name
    with e -> clean main_name; raise e;;

(* main part *)
let main () =
  let (options, userfiles) = parse_args () in

  (* which ocaml command to invoke *)
  let prog =
    if !opt then
      (* native code *)
      if !top then
	failwith "no custom toplevel in native code !"
      else "ocamlopt -linkall"
    else
      (* bytecode *)
      if !top then "ocamlmktop -custom -linkall" else "ocamlc -custom -linkall"
  in

  (* files to link *)
  let (modules, tolink) = files_to_link userfiles in

  (*file for dynlink *)
  let dynlink=
    if not((!opt)||(!top)) then
      [tmp_dynlink()]
    else
      []
  in

  (* the list of the loaded modules *)
  let main_file = create_tmp_main_file modules in

    try
      let args = (includes ()) @ options @ tolink @ dynlink @ [ main_file ] in

      (*Now, with the .cma, we MUST use the -linkall option*)
      let command = String.concat " " ((prog^" -linkall")::args) in
        if !echo then begin print_endline command; flush stdout end;
        let retcode = Sys.command command in
          clean main_file;
          (* command gives the exit code in HSB, and signal in LSB !!! *)
          if retcode > 255 then retcode lsr 8 else retcode 
    with
        e -> clean main_file; raise e
;;

let retcode =
  try Printexc.print main ()
  with _ -> 1
;;

exit retcode;;

(* $Id: coqmktop.ml,v 1.33 1999/06/29 07:47:10 loiseleu Exp $ *)
