(* camlp4r pa_extend.cmo *)
(***********************************************************************)
(*                                                                     *)
(*                             Camlp4                                  *)
(*                                                                     *)
(*        Daniel de Rauglaudre, projet Cristal, INRIA Rocquencourt     *)
(*                                                                     *)
(*  Copyright 1998 Institut National de Recherche en Informatique et   *)
(*  Automatique.  Distributed only by permission.                      *)
(*                                                                     *)
(***********************************************************************)

(* $Id: pcaml.ml,v 2.8 1999/11/23 21:05:00 ddr Exp $ *)

value version = "2.04";

value gram =
  Grammar.create
    {Token.func = fun _ -> failwith "no loaded parsing module";
     Token.using = fun _ -> (); Token.removing = fun _ -> ();
     Token.tparse = fun []; Token.text = fun _ -> ""}
;

value interf = Grammar.Entry.create gram "interface file";
value implem = Grammar.Entry.create gram "implementation file";
value top_phrase = Grammar.Entry.create gram "toplevel phrase";
value use_file = Grammar.Entry.create gram "source file";
value sig_item = Grammar.Entry.create gram "sig_item";
value str_item = Grammar.Entry.create gram "str_item";
value module_type = Grammar.Entry.create gram "module_type";
value module_expr = Grammar.Entry.create gram "module_expr";
value expr = Grammar.Entry.create gram "expression";
value patt = Grammar.Entry.create gram "pattern";
value ctyp = Grammar.Entry.create gram "type";
value let_binding = Grammar.Entry.create gram "let_binding";

value class_sig_item = Grammar.Entry.create gram "class_sig_item";
value class_str_item = Grammar.Entry.create gram "class_str_item";
value class_type = Grammar.Entry.create gram "class_type";
value class_expr = Grammar.Entry.create gram "class_expr";

value input_file = ref "";
value output_file = ref None;

value undef x = ref (fun _ -> failwith x);

type printer_t 'a = ref ('a -> unit);
value print_interf = undef "no printer";
value print_implem = undef "no printer";

List.iter (fun (n, f) -> Quotation.add n f)
  [("id", Quotation.ExStr (fun _ s -> "$0:" ^ s ^ "$"));
   ("string", Quotation.ExStr (fun _ s -> "\"" ^ String.escaped s ^ "\""))];

value quotation_dump_file = ref (None : option string);

type err_ctx =
  [ Expanding | ParsingResult of (int * int) and string | Locating ]
;
exception Qerror of string and err_ctx and exn;

value expand_quotation loc expander shift name str =
  try expander str with
  [ Stdpp.Exc_located (p1, p2) exc ->
      let exc1 = Qerror name Expanding exc in
      raise (Stdpp.Exc_located (shift + p1, shift + p2) exc1)
  | exc ->
      let exc1 = Qerror name Expanding exc in
      raise (Stdpp.Exc_located loc exc1) ]
;

value parse_quotation_result entry loc shift name str =
  let cs = Stream.of_string str in
  try Grammar.Entry.parse entry cs with
  [ Stdpp.Exc_located iloc (Qerror _ Locating _ as exc) ->
      raise (Stdpp.Exc_located (shift + fst iloc, shift + snd iloc) exc)
  | Stdpp.Exc_located iloc (Qerror _ Expanding exc) ->
      let ctx = ParsingResult iloc str in
      let exc1 = Qerror name ctx exc in raise (Stdpp.Exc_located loc exc1)
  | Stdpp.Exc_located _ (Qerror _ _ _ as exc) ->
      raise (Stdpp.Exc_located loc exc)
  | Stdpp.Exc_located iloc exc ->
      let ctx = ParsingResult iloc str in
      let exc1 = Qerror name ctx exc in raise (Stdpp.Exc_located loc exc1) ]
;

value handle_quotation loc proj in_expr entry reloc (name, str) =
  let shift =
    match name with
    [ "" -> String.length "<<"
    | _ -> String.length "<:" + String.length name + String.length "<" ]
  in
  let shift = fst loc + shift in
  let expander =
    try Quotation.find name with
    [ Not_found ->
        let exc1 = Qerror name Expanding Not_found in
        raise (Stdpp.Exc_located loc exc1) ]
  in
  let ast =
    match expander with
    [ Quotation.ExStr f ->
        let new_str = expand_quotation loc (f in_expr) shift name str in
        parse_quotation_result entry loc shift name new_str
    | Quotation.ExAst fe_fp ->
        expand_quotation loc (proj fe_fp) shift name str ]
  in
  reloc (fun _ -> loc) shift ast
;

value parse_locate entry shift str =
  let cs = Stream.of_string str in
  try Grammar.Entry.parse entry cs with
  [ Stdpp.Exc_located (p1, p2) exc ->
      let ctx = Locating in
      let exc1 = Qerror (Grammar.Entry.name entry) ctx exc in
      raise (Stdpp.Exc_located (shift + p1, shift + p2) exc1) ]
;

value handle_locate loc entry ast_f (pos, str) =
  let s = str in
  let loc = (pos, pos + String.length s) in
  let x = parse_locate entry (fst loc) s in ast_f loc x
;

value expr_anti loc e = MLast.ExAnt loc e;
value patt_anti loc p = MLast.PaAnt loc p;
value expr_eoi = Grammar.Entry.create gram "expression";
value patt_eoi = Grammar.Entry.create gram "pattern";
EXTEND
  expr_eoi:
    [ [ x = expr; EOI -> x ] ]
  ;
  patt_eoi:
    [ [ x = patt; EOI -> x ] ]
  ;
END;

value handle_expr_quotation loc x =
  handle_quotation loc fst True expr_eoi Reloc.expr x
;

value handle_expr_locate loc x = handle_locate loc expr_eoi expr_anti x;

value handle_patt_quotation loc x =
  handle_quotation loc snd False patt_eoi Reloc.patt x
;

value handle_patt_locate loc x = handle_locate loc patt_eoi patt_anti x;

value expr_reloc = Reloc.expr;
value patt_reloc = Reloc.patt;

value find_line (bp, ep) str =
  find 0 1 0 where rec find i line col =
    if i == String.length str then (line, 0, col)
    else if i == bp then (line, col, col + ep - bp)
    else if str.[i] == '\n' then find (succ i) (succ line) 0
    else find (succ i) line (succ col)
;

value report_quotation_error name ctx =
  let name = if name = "" then Quotation.default.val else name in
  do Format.print_flush ();
     Format.open_hovbox 2;
     Printf.eprintf "While %s \"%s\":"
       (match ctx with
        [ Expanding -> "expanding quotation"
        | ParsingResult _ _ -> "parsing result of quotation"
        | Locating -> "parsing" ])
       name;
  return
  match ctx with
  [ ParsingResult (bp, ep) str ->
      match quotation_dump_file.val with
      [ Some dump_file ->
          do Printf.eprintf " dumping result...\n"; flush stderr; return
          try
            let (line, c1, c2) = find_line (bp, ep) str in
            let oc = open_out_bin dump_file in
            do output_string oc str;
               output_string oc "\n";
               flush oc;
               close_out oc;
               Printf.eprintf "File \"%s\", line %d, characters %d-%d:\n"
                 dump_file line c1 c2;
               flush stderr;
            return ()
          with _ ->
            do Printf.eprintf "Error while dumping result in file \"%s\""
                 dump_file;
               Printf.eprintf "; dump aborted.\n";
               flush stderr;
            return ()
      | None ->
          do if input_file.val = "" then
               Printf.eprintf
                 "\n(consider setting variable Pcaml_quotation_dump_file)\n"
             else Printf.eprintf " (consider using option -QD)\n";
             flush stderr;
          return () ]
  | _ -> do Printf.eprintf "\n"; flush stderr; return () ]
;

value print_format str =
  let rec flush ini cnt =
    if cnt > ini then Format.print_string (String.sub str ini (cnt - ini))
    else ()
  in
  let rec loop ini cnt =
    if cnt == String.length str then flush ini cnt
    else
      match str.[cnt] with
      [ '\n' ->
          do flush ini cnt;
             Format.close_box ();
             Format.force_newline ();
             Format.open_box 2;
          return loop (cnt + 1) (cnt + 1)
      | ' ' ->
          do flush ini cnt; Format.print_space (); return
          loop (cnt + 1) (cnt + 1)
      | _ -> loop ini (cnt + 1) ]
  in
  do Format.open_box 2; loop 0 0; return Format.close_box ()
;

value print_exn =
  fun
  [ Out_of_memory -> Format.print_string "Out of memory\n"
  | Match_failure (file, first_char, last_char) ->
      do Format.print_string "Pattern matching failed, file ";
         Format.print_string file;
         Format.print_string ", chars ";
         Format.print_int first_char;
         Format.print_char '-';
         Format.print_int last_char;
      return ()
  | Stream.Error str -> print_format ("Parse error: " ^ str)
  | Stream.Failure -> Format.print_string "Parse failure"
  | Token.Error str ->
      do Format.print_string "Lexing error: "; Format.print_string str; return
      ()
  | Failure str ->
      do Format.print_string "Failure: "; Format.print_string str; return ()
  | Invalid_argument str ->
      do Format.print_string "Invalid argument: "; Format.print_string str;
      return ()
  | Sys_error msg ->
      do Format.print_string "I/O error: "; return Format.print_string msg
  | x ->
      do Format.print_string "Uncaught exception: ";
         Format.print_string
           (Obj.magic (Obj.field (Obj.field (Obj.repr x) 0) 0));
         if Obj.size (Obj.repr x) > 1 then
           do Format.print_string " (";
              for i = 1 to Obj.size (Obj.repr x) - 1 do
                if i > 1 then Format.print_string ", " else ();
                let arg = Obj.field (Obj.repr x) i in
                if not (Obj.is_block arg) then
                  Format.print_int (Obj.magic arg : int)
                else if Obj.tag arg = 252 then
                  do Format.print_char '"';
                     Format.print_string (Obj.magic arg : string);
                  return Format.print_char '"'
                else Format.print_char '_';
              done;
           return Format.print_char ')'
         else ();
      return () ]
;

value report_error exn =
  match exn with
  [ Qerror name ctx exn ->
      do report_quotation_error name ctx; return print_exn exn
  | e -> print_exn exn ]
;

value warning_default_function (bp, ep) txt =
  do Printf.eprintf "<W> loc %d %d: %s\n" bp ep txt; flush stderr; return ()
;

value warning = ref warning_default_function;

value no_constructors_arity = Ast2pt.no_constructors_arity;
value no_assert = ref False;

value arg_spec_list_ref = ref [];
value arg_spec_list () = arg_spec_list_ref.val;
value add_option name spec descr =
  arg_spec_list_ref.val := arg_spec_list_ref.val @ [(name, spec, descr)]
;
