#pragma implementation
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <misc.h>
#include <configf.h>
#include <userconf.h>
#include <translat.h>
#include <subsys.h>
#include <netconf.h>
#include <fviews.h>
#include "inetdconf.h"
#include "inetdconf.m"
#include "etcservice.h"
#include "fwport.h"
#include "../../paths.h"

#define PROC_SYS_NET_IPV4_IPLOCAL_PORT_RANGE	"/proc/sys/net/ipv4/ip_local_port_range"

/*
 * class DAEMON_DIALD
 */
#include <netconf.h>
#include <daemoni.h>
#include <signal.h>

static HELP_FILE help_firewall ("inetdconf","firewall");

static CONFIG_FILE f_config_file(
		"/usr/lib/linuxconf/lib/inetdconfFW.sh"
		,help_firewall
		,CONFIGF_OPTIONAL|CONFIGF_GENERATED
		,"root"
		,"root"
		,0700
		,subsys_inetdconf);

class FW {
private:
	void append_to_dialog( DIALOG *dia, FWPORT *fwport );
	void check_status( );
	int config_file_existed;	// Flag:  file exists before write
public:
	SSTRING internet_ip;		// Local internet adress
	SSTRING internet_interface;	// Local internet interface
	SSTRINGS other_ip;		// Other network routes
	SSTRINGS other_if;		// Other interface routes
	SSTRINGS other_mask;		// Other routes masks
	FWPORTLIST *fwportlist;		// Active servers
	ETCSERVICELIST *etcservicelist;	// /etc/services
	char active;			// 1: Set firewall
	char dynamic_ip;		// 1: Dynamic ip address
	char log_denied;		// 1: Log denied packets in syslog
	int ip_forward;			// /proc/sys/net/ipv4/ip_forward
	int ip_dynaddr;			// /proc/sys/net/ipv4/ip_dynaddr
	int port_low;			// ip_local_port_range: From
	int port_high;			// ip_local_port_range: To
public:
	FW( void );
	int internet_route( void );
	int edit( void );
	int write( void );
	int input_error( void );
	~FW( void );
};


PUBLIC FW::FW( void )
{
	etcservicelist = new ETCSERVICELIST();
	etcservicelist->read();
	fwportlist = new FWPORTLIST();
	fwportlist->read();
	fwportlist->sort();
	fwportlist->update_from_db();

	FILE *fp = NULL;
#if 0
	// Apparently this does not work or there is something I dont
	// understand. A firewall I set up used default values 1024-4999
	// which denied packets on ports above 61000 ... which should
	// not have been allocated in the first place!
	FILE *fp = fopen( PROC_SYS_NET_IPV4_IPLOCAL_PORT_RANGE, "r" );
	if ( fp != NULL ) {
		fscanf( fp, "%d %d", &port_low, &port_high );
		fclose( fp );
	} else {
		port_low = 1024;
		port_high = 65535;
	}
#endif
	port_low = 1024;
	port_high = 65535;

	fp = fopen( PROC_SYS_NET_IPV4_DYNADDR, "r" );
	if ( fp != NULL ) {
		fscanf( fp, "%d", &ip_dynaddr );
		fclose( fp );
	} else {
		ip_dynaddr = 0;
	}
	fp = fopen( PROC_SYS_IP_FORWARD, "r" );
	if ( fp != NULL ) {
		fscanf( fp, "%d", &ip_forward );
		fclose( fp );
	} else {
		ip_forward = 0;
	}
//fprintf(stderr, "FW::FW: port_low=%d port_high=%d ip_forward=%d ip_dynaddr=%d\n", port_low, port_high, ip_forward, ip_dynaddr );
	active = (char)linuxconf_getvalnum( K_FIREWALL, K_ACTIVE, 0 );
	dynamic_ip=(char)linuxconf_getvalnum(K_FIREWALL,K_DYNAMIC_IP,ip_dynaddr?1:0);
	log_denied=(char)linuxconf_getvalnum(K_FIREWALL,K_LOG_DENIED,0);
}

PUBLIC FW::~FW( void )
{
	delete fwportlist;
	delete etcservicelist;
}

/*
 * Find route to the internet.
 */
PUBLIC int FW::internet_route( void )
{
	ROUTES routes;
	ROUTE *defaultroute = NULL;
	routes.readactive();
	for ( int i=0; i<routes.getnb(); i++ ) {
		ROUTE *route = routes.getitem(i);
//fprintf(stderr,"%s %s %s %s %s\n", route->getdst(), route->getgateway(), route->getmask(), route->getiface(), (route->dst_is_host())?"Host":"Net");
		if ((strcmp( route->getdst(), "default" ) == 0 )
		/*
		 * Do not accept sl0 as the route to internet as this will be
		 * replaced by ppp0 by diald
		 */
		&& ( strncmp( route->getiface(), "lo", 2 ) != 0 ) ) {
			if ( strcmp( route->getgateway(), route->getiface() ) == 0 ) {
				defaultroute = route;
				break;
			} else {
				internet_ip.setfrom( route->getgateway() );
				internet_interface.setfrom( route->getiface() );
//fprintf(stderr,"default: %s %s %s %s\n", route->getdst(), route->getgateway(), route->getmask(), route->getiface());
				return( 1 );
			}
		} else if ( ! route->dst_is_host() ) {
			other_ip.add( new SSTRING( route->getdst() ));
			other_if.add( new SSTRING( route->getiface() ));
			other_mask.add( new SSTRING( route->getmask() ));
		}
	}
	if ( defaultroute == NULL ) {
		return( 0 );
	}
	for ( int i=0; i<routes.getnb(); i++ ) {
		ROUTE *route = routes.getitem(i);
		if ((strcmp( route->getiface(), defaultroute->getiface() ) == 0)
		&& ( route->dst_is_host() )) {
//fprintf(stderr,"internet: %s %s %s %s\n", route->getdst(), route->getgateway(), route->getmask(), route->getiface());
			internet_ip.setfrom( route->getdst() );
			internet_interface.setfrom( route->getiface() );
			return( 1 );
		}
	}
	return( 0 );
}


PUBLIC int FW::write( void )
{
	if ( ! perm_rootaccess( MSG_U(P_EDITFIREWALL,
			"change internet firewall") ) ) {
		return( -1 );
	}
	config_file_existed = f_config_file.exist();
	/*
	 * Update linuxconf database
	 */
	linuxconf_setcursys( subsys_inetdconf );
	linuxconf_replace( K_FIREWALL, K_ACTIVE, active );
	linuxconf_replace( K_FIREWALL, K_DYNAMIC_IP, dynamic_ip );
	linuxconf_replace( K_FIREWALL, K_LOG_DENIED, log_denied );
	for (int i=0; i<fwportlist->getnb(); i++ ) {
		FWPORT *fwport = fwportlist->getitem( i );
		char service[100];
		snprintf( service, sizeof(service),
			"%s/%d", fwport->protocol.get(), fwport->port);
		if ( fwport->active ) {
			linuxconf_replace( K_FIREWALL, service, fwport->active);
		} else {
			linuxconf_removeall( K_FIREWALL, service );
		}
	}
	linuxconf_save( (PRIVILEGE *)NULL );
	if ( ! active ) {
		return( 0 );
	}

	const char *ipchains = daemon_findpath( "ipchains" );
	FILE *fout = f_config_file.fopen( (PRIVILEGE *)NULL, "w" );
	fprintf( fout, "#!/bin/sh\n" );
	fprintf( fout, "# Created by Linuxconf (module: inetdconf)\n" );
	fprintf( fout, "#\n" );
	fprintf( fout, "command=$1;\n" );
	fprintf( fout, "internet_interface=$2;\n" );
	fprintf( fout, "ip=$3;\n" );
	fprintf( fout, "\n" );
	fprintf( fout, "\n" );

	fprintf( fout, "start_firewall() {\n" );
	/*
	 * Forward: Flush & set policy DENY, allow local interfaces with MASQ.
	 * Log the rest on this particular interface.
	 */
	fprintf( fout, "\t%s -F forward;\n", ipchains );
	fprintf( fout, "\t%s -P forward DENY;\n", ipchains );
	for (int i=0; i<other_ip.getnb(); i++ ) {
		SSTRING *iface = other_if.getitem( i );
		SSTRING *ip = other_ip.getitem( i );
		SSTRING *mask = other_mask.getitem( i );
		if ( strncmp( iface->get(), "lo", 2 ) != 0 ) {
			fprintf( fout,
				"\t%s -A forward -s %s/%s -j MASQ -b;\n",
				ipchains, ip->get(), mask->get() );
			fprintf( fout, "\t%s -A forward -j DENY -l;\n", ipchains );
		}
	}

	/*
	 * Input: Flush & set policy DENY, allow local interfaces.
	 */
	fprintf( fout, "\t%s -F input;\n", ipchains );
	fprintf( fout, "\t%s -P input DENY;\n", ipchains );
	for (int i=0; i<other_ip.getnb(); i++ ) {
		SSTRING *iface = other_if.getitem( i );
//		SSTRING *ip = other_ip.getitem( i );
//		SSTRING *mask = other_mask.getitem( i );
		fprintf( fout, "\t%s -A input -i %s -j ACCEPT;\n",
			ipchains, iface->get() );
	}
	/*
	 * Allow icmp
	 */
	fprintf( fout, "\t%s -A input -p %s -d $ip/32 -j ACCEPT;\n",
		ipchains,
		"icmp"
		);
	/*
	 * Allow specified ports
	 */
	for (int i=0; i<fwportlist->getnb(); i++ ) {
		FWPORT *fwport = fwportlist->getitem( i );
		if ( ! fwport->active ) continue;
		ETCSERVICE *etcservice = etcservicelist->getitem( fwport->protocol.get(), fwport->port);
		char port[10];
		snprintf( port, sizeof(port), "%d", fwport->port );
		const char *service = "";
		if ( etcservice != NULL ) {
			service = etcservice->service_name.get();
		} else {
			service = port;
		}
		fprintf( fout, "\t%s -A input -i %s -p %s -d $ip/32 %s -j ACCEPT;\n",
			ipchains,
			internet_interface.get(),
			fwport->protocol.get(),
			service
			);
	}
	/*
	 * Allow ports from port_low to port_high : tcp and udp
	 */
	fprintf( fout, "\t%s -A input -i %s -p %s -d $ip/32 %d:%d -j ACCEPT;\n",
		ipchains,
		internet_interface.get(),
		"tcp",
		port_low,
		port_high
		);
	fprintf( fout, "\t%s -A input -i %s -p %s -d $ip/32 %d:%d -j ACCEPT;\n",
		ipchains,
		internet_interface.get(),
		"udp",
		port_low,
		port_high
		);
	/*
	 * Log denied packets to syslog
	 */
	if ( log_denied ) {
		fprintf( fout, "\t%s -A input -j DENY -l;\n",
			ipchains
			);
	}
	fprintf( fout, "}\n" );
	fprintf( fout, "\n" );
	fprintf( fout, "\n" );
	fprintf( fout, "stop_firewall() {\n" );
	fprintf( fout, "\t%s -F input;\n", ipchains );
	fprintf( fout, "\t%s -P input ACCEPT;\n", ipchains );
	fprintf( fout, "}\n" );
	fprintf( fout, "\n" );
	fprintf( fout, "\n" );
	fprintf( fout, "case $command in\n" );
	fprintf( fout, "\tstart)\n" );
	fprintf( fout, "\t\tcase $internet_interface in\n" );
	fprintf( fout, "\t\t\t%s)\n", internet_interface.get() );
	fprintf( fout, "\t\t\t\tstart_firewall;\n" );
	fprintf( fout, "\t\t\t\t;;\n" );
	fprintf( fout, "\t\tesac\n" );
	fprintf( fout, "\t\t;;\n" );
	fprintf( fout, "\tstatus)\n" );
	fprintf( fout, "\t\t%s -L -v;\n", ipchains );
	fprintf( fout, "\t\t;;\n" );
	fprintf( fout, "\tstop)\n" );
	fprintf( fout, "\t\tstop_firewall;\n" );
	fprintf( fout, "\t\t;;\n" );
	fprintf( fout, "\t*)\n" );
	fprintf( fout, "\t\techo \"Usage: $0 {start %s ip|stop|status}\"\n",
		internet_interface.get() );
	fprintf( fout, "\t\texit 1\n" );
	fprintf( fout, "\t;;\n" );
	fprintf( fout, "esac\n" );

	fprintf( fout, "exit 0\n" );

	return( fclose( fout ) );
}

PRIVATE void FW::check_status( )
{
	if ( ! config_file_existed ) {
		xconf_notice( MSG_U(I_SCRIPT_CREATED,
		"Executable script (%s) has been created.\n"
		"This script should be executed when the internet interface %s is brought up.\n"
		"\n"
		"For a dynamic ip address start it by (in %s):\n"
		"%s start $1 $4\n"
		"\n"
		"or for a static ip address in a rc-script:\n"
		"%s start %s %s\n"
		),	f_config_file.getpath(),
			internet_interface.get(),
			ETC_PPP_IPUP,
			f_config_file.getpath(),
			f_config_file.getpath(),
			internet_interface.get(),
			internet_ip.get()
			 );
	}
}

PRIVATE void FW::append_to_dialog( DIALOG *dia, FWPORT *fwport )
{
	ETCSERVICE *etcservice = etcservicelist->getitem( fwport->protocol.get(), fwport->port);
	const char *service = "";
	if ( etcservice != NULL ) {
		service = etcservice->service_name.get();
	}
	char text[100];
	snprintf( text, sizeof( text ), "%d/%s %s %s:%d %s %s",
		fwport->port,
		service,
		MSG_U(T_IN,"in"),
		fwport->program.get(),
		fwport->pid,
		MSG_U(T_BY,"by"),
		fwport->user.get()
	);
	dia->newf_chk( "", fwport->active, text);
}

PUBLIC int FW::edit( void )
{
//fprintf(stderr,"FW::edit: inferface=%s ip=%s fwportlist->getnb()=%d\n",internet_interface.get(), internet_ip.get(), fwportlist->getnb());
	
	DIALOG dia;


	dia.newf_title (MSG_U(T_BASICFWINFO,"Basic information")
		,1,"",MSG_R(T_BASICFWINFO));
	dia.newf_chk( MSG_U(F_ENABLE_FIREWALL,"Internet input filter firewall"), active, MSG_U(F_ACTIVE,"Active" ) );
	dia.newf_chk( MSG_U(F_DYNAMIC_IP,"Dynamic ip address"), dynamic_ip, "" );
	dia.newf_chk( MSG_U(F_LOG_DENIED,"Denied packets logged in system log"), log_denied, "" );
	dia.newf_info( MSG_U(F_INTERNET_INTERFACE, "Internet interface"),
		internet_interface.get() );
	dia.newf_info( MSG_U(F_INTERNET_IP, "Internet ip address"),
		internet_ip.get() );
	dia.newf_info( MSG_U(F_IP_FORWARD, "Kernel ip forward"),
		ip_forward?MSG_U(F_YES,"Yes"):MSG_U(F_NO,"No") );
	dia.newf_info( MSG_U(F_IP_DYNADDR, "Kernel dynamic ip"),
		ip_forward?MSG_R(F_YES):MSG_R(F_NO) );

	dia.newf_title (MSG_U(T_INTERNET,"Internet")
		,1,"",MSG_R(T_INTERNET));
	dia.newf_title ("",MSG_U(T_ALLOWED_SERVICES,
		"Allowed services from the internet"));
#if 0
#endif
	dia.newf_title( MSG_U(T_ACTIVE_SERVICES_TCP,"TCP") ,2,"",MSG_R(T_ACTIVE_SERVICES_TCP));
	for (int i=0; i<fwportlist->getnb(); i++ ) {
		FWPORT *fwport = fwportlist->getitem( i );
		if ( fwport->protocol.cmp( "tcp" ) == 0 ) {
			append_to_dialog( &dia, fwport );
		}
	}
#if 0
#endif
	dia.newf_title( MSG_U(T_ACTIVE_SERVICES_UDP,"UDP") ,2,"",MSG_R(T_ACTIVE_SERVICES_UDP));
	for (int i=0; i<fwportlist->getnb(); i++ ) {
		FWPORT *fwport = fwportlist->getitem( i );
		if ( fwport->protocol.cmp( "udp" ) == 0 ) {
			append_to_dialog( &dia, fwport );
		}
	}
	for (int i=0; i<other_ip.getnb(); i++ ) {
		SSTRING *ip = other_ip.getitem( i );
		SSTRING *iface = other_if.getitem( i );
		SSTRING *mask = other_mask.getitem( i );
		dia.newf_title ( iface->get(),1,"",iface->get());
		dia.newf_info( MSG_U(I_INTERFACE,
			"Full access to this interface"), iface->get() );
		dia.newf_info( MSG_U(F_NETWORK_ADDRESS,
			"Network address"), ip->get());
		dia.newf_info( MSG_U(F_NETWORK_MASK,
			"Network mask"),mask->get());
	}
//fprintf(stderr,"FW::edit: OK\n");
	int ret = 0;
	int choice = 0;
	while (1) {
		MENU_STATUS code = dia.edit(
		MSG_U(T_FIREWALL, "Input firewall")
		,MSG_U(I_FIREWALL,
		"This simple input filter firewall for internet interface blocks every connection unless it is\n"
		"allowed in this dialog. Internal network is masqueraded and icmp is allowed."
				)
			,help_firewall
			,choice
			,(MENUBUT_CANCEL|MENUBUT_ACCEPT)
			);
		if (code == MENU_CANCEL || code == MENU_ESCAPE){
			ret = -1;
			break;
		}else if (code == MENU_ACCEPT ) {
			if ( input_error( ) ) {
				continue;
			}
			ret = 0;
			write( );
			check_status();
			break;
		}
	}
	return ret;
}

PUBLIC int FW::input_error( void )
{
	if ( dynamic_ip && ip_dynaddr == 0 ) {
		xconf_error( MSG_U(E_IP_DYNADDR,
			"You want dynamic ip address but this is\n"
			"not active in the kernel\n"));
		return( 1 );
	}
	if ( ip_forward == 0 ) {
		xconf_notice( MSG_U(E_IP_FORWARD,
			"Forwarding of ip traffic is not active in\n"
			"the kernel. This is not needed for the\n"
			"firewall but you will not reach the internet\n"
			"from a local network.\n") );
	}
	return( 0 );
}
/*
 * DAEMON_DIALD
 */
class PROC;
class DAEMON_DIALD {
private:
public:
	PROC *myProcess;
public:
	DAEMON_DIALD();
	PROC *getdaemonprocess( void );
};

PUBLIC DAEMON_DIALD::DAEMON_DIALD()
{
	myProcess = getdaemonprocess();
}

PUBLIC PROC *DAEMON_DIALD::getdaemonprocess()
{
	PROC *myProcess = NULL;
	char pidpath[PATH_MAX];
	sprintf (pidpath,"%s.pid","/var/run/diald");
	CONFIG_FILE f_pid ( pidpath, help_nil
		,CONFIGF_OPTIONAL|CONFIGF_MANAGED
		,"root","root",0600);
	if ( f_pid.exist ()){
		myProcess = process_find ("diald", &f_pid);
	}
	return( myProcess );
}

PUBLIC void firewall_edit( )
{
	if ( ! kernel_newer( 2,2,0 ) ) {
		xconf_error( MSG_U( E_OLD_KERNEL,
			"Sorry, your kernel is too old for this function.\n"
			"Version at least 2.2.0 required\n"
			) );
		return;
	}

	FW fw;
	fw.internet_route();

	if ( strncmp( fw.internet_interface.get(), "sl", 2 ) == 0 ) {
		DAEMON_DIALD myDiald;
		if ( myDiald.myProcess != NULL ) {
			xconf_error( MSG_U( E_NOT_UP,
			"Route to the internet is not valid.\n"
			"Diald is running but the link is not active.\n"
			"Can not determine the route to the internet!\n"
			) );
			return;
		}
	}
	if ( fw.internet_interface.is_empty() ) {
		xconf_error( MSG_U( E_NODEFAULT_ROUTE,
			"Can not determine the route to the internet!\n"
			"The search is based on internet being the default\n"
			"route.\n") );
		return;
	}
//fprintf(stderr,"firewall_edit: fw.internet_ip=%s fw.internet_interface=%s\n", fw.internet_ip.get(), fw.internet_interface.get());
	if ( fw.fwportlist->getnb() == 0 ) {
		xconf_error( MSG_U( E_NOPORTLIST,
			"No active services could be located in this system.\n"
			"If this is correct there is no need for a firewall.\n"
			) );
		return;
	}

	fw.edit( );
}
