/*
    ext2.c -- generic ext2 stuff
    Copyright (C) 1998,99 Lennert Buytenhek <lbuijten@cs.leidenuniv.nl>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include "ext2.h"

/* ext2 stuff ****************************************************************/

unsigned char _bitmap[8] = {0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80};

void ext2_copy_block(struct ext2_fs *fs, blk_t from, blk_t to)
{
	unsigned char buf[fs->blocksize];

	ext2_bcache_flush(fs, from);
	ext2_bcache_flush(fs, to);

	ext2_read_blocks(fs, buf, from, 1);
	ext2_write_blocks(fs, buf, to, 1);
}

int ext2_get_block_state(struct ext2_fs *fs, blk_t block)
{
	struct ext2_buffer_head *bh;
	int group;
	int offset;
	int state;

	block -= fs->sb.s_first_data_block;
	group = block / fs->sb.s_blocks_per_group;
	offset = block % fs->sb.s_blocks_per_group;

	bh = ext2_bread(fs, fs->gd[group].bg_block_bitmap);
	state = bh->data[offset>>3] & _bitmap[offset&7];
	ext2_brelse(bh, 0);

	return state;
}

void ext2_move_blocks(struct ext2_fs *fs, blk_t src, blk_t num, blk_t dest)
{
	unsigned char *buf;
	blk_t i;

	if ((buf = malloc(num << fs->logsize)) != NULL)
	{
		ext2_bcache_flush_range(fs, src, src+num);
		ext2_bcache_flush_range(fs, dest, dest+num);

		ext2_read_blocks(fs, buf, src, num);
		ext2_write_blocks(fs, buf, dest, num);

		free(buf);

		return;
	}

	if (src > dest)
		for (i=0;i<num;i++)
			ext2_copy_block(fs, src+i, dest+i);
	else
		for (i=num-1;i>=0;i--)
			ext2_copy_block(fs, src+i, dest+i);
}

int ext2_read_blocks(struct ext2_fs *fs, void *ptr, blk_t block, blk_t num)
{
	return fs->devhandle->ops->read(fs->devhandle->cookie, ptr, block, num);
}

void ext2_set_block_state(struct ext2_fs *fs, blk_t block, int state, int updatemetadata)
{
	struct ext2_buffer_head *bh;
	int                      group;
	int                      offset;

	block -= fs->sb.s_first_data_block;
	group = block/fs->sb.s_blocks_per_group;
	offset = block%fs->sb.s_blocks_per_group;

	bh = ext2_bread(fs, fs->gd[group].bg_block_bitmap);
	bh->dirty = 1;
	if (state)
		bh->data[offset>>3] |= _bitmap[offset&7];
	else
		bh->data[offset>>3] &= ~_bitmap[offset&7];
	ext2_brelse(bh, 0);

	if (updatemetadata)
	{
		int diff;

		diff = state ? -1 : 1;

		fs->gd[group].bg_free_blocks_count += diff;
		fs->sb.s_free_blocks_count += diff;
		fs->metadirty = 1;
	}
}

int ext2_write_blocks(struct ext2_fs *fs, void *ptr, blk_t block, blk_t num)
{
	return fs->devhandle->ops->write(fs->devhandle->cookie, ptr, block, num);
}

void ext2_zero_blocks(struct ext2_fs *fs, blk_t block, blk_t num)
{
	unsigned char *buf;
	blk_t i;

	if ((buf = malloc(num << fs->logsize)) != NULL)
	{
		memset(buf, 0, num << fs->logsize);
		ext2_bcache_flush_range(fs, block, block+num);
		ext2_write_blocks(fs, buf, block, num);
		free(buf);
		return;
	}

	if ((buf = malloc(fs->blocksize)) != NULL)
	{
		memset(buf, 0, fs->blocksize);

		for (i=0;i<num;i++)
		{
			ext2_bcache_flush(fs, block+i);
			ext2_write_blocks(fs, buf, block+i, 1);
		}

		free(buf);
		return;
	}

	for (i=0;i<num;i++)
	{
		struct ext2_buffer_head *bh;

		bh = ext2_bcreate(fs, block+i);
		bh->dirty = 1;
		ext2_brelse(bh, 1);
	}
}





















off_t ext2_get_inode_offset(struct ext2_fs *fs, ino_t inode, blk_t *block)
{
	int group;
	int offset;

	inode--;

	group = inode / fs->sb.s_inodes_per_group;
	offset = (inode % fs->sb.s_inodes_per_group) * sizeof(struct ext2_inode);

	*block = fs->gd[group].bg_inode_table + (offset >> fs->logsize);

	return offset & (fs->blocksize - 1);
}

int ext2_get_inode_state(struct ext2_fs *fs, ino_t inode)
{
	struct ext2_buffer_head *bh;
	int                      group;
	int                      offset;
	int                      ret;

	inode--;
	group = inode / fs->sb.s_inodes_per_group;
	offset = inode % fs->sb.s_inodes_per_group;

	bh = ext2_bread(fs, fs->gd[group].bg_inode_bitmap);
	ret = bh->data[offset>>3] & _bitmap[offset&7];
	ext2_brelse(bh, 0);

	return ret;
}

void ext2_read_inode(struct ext2_fs *fs, ino_t inode, struct ext2_inode *data)
{
	struct ext2_buffer_head *bh;
	blk_t			 blk;
	off_t			 off;

	off = ext2_get_inode_offset(fs, inode, &blk);

	bh = ext2_bread(fs, blk);
	memcpy(data, bh->data + off, sizeof(struct ext2_inode));
	ext2_brelse(bh, 0);
}

void ext2_set_inode_state(struct ext2_fs *fs, ino_t inode, int state, int updatemetadata)
{
	struct ext2_buffer_head *bh;
	int                      group;
	int                      offset;

	inode--;
	group = inode / fs->sb.s_inodes_per_group;
	offset = inode % fs->sb.s_inodes_per_group;

	bh = ext2_bread(fs, fs->gd[group].bg_inode_bitmap);
	bh->dirty = 1;
	if (state)
		bh->data[offset>>3] |= _bitmap[offset&7];
	else
		bh->data[offset>>3] &= ~_bitmap[offset&7];
	ext2_brelse(bh, 0);

	if (updatemetadata)
	{
		int diff;

		diff = state ? -1 : 1;

		fs->gd[group].bg_free_inodes_count += diff;
		fs->sb.s_free_inodes_count += diff;
		fs->metadirty = 1;
	}
}

void ext2_write_inode(struct ext2_fs *fs, ino_t inode, const struct ext2_inode *data)
{
	struct ext2_buffer_head *bh;
	blk_t			 blk;
	off_t			 off;

	off = ext2_get_inode_offset(fs, inode, &blk);

	bh = ext2_bread(fs, blk);
	bh->dirty = 1;
	memcpy(bh->data + off, data, sizeof(struct ext2_inode));
	ext2_brelse(bh, 0);
}

void ext2_zero_inode(struct ext2_fs *fs, ino_t inode)
{
	struct ext2_inode buf;

	memset(&buf, 0, sizeof(struct ext2_inode));
	ext2_write_inode(fs, inode, &buf);
}





/* formula grabbed from linux ext2 kernel source */
static int is_root(int x, int y)
{
	if (!x)
		return 1;

	while (1)
	{
		if (x == 1)
			return 1;

		if (x % y)
			return 0;

		x /= y;
	}
}

int ext2_is_group_sparse(struct ext2_fs *fs, int group)
{
	if (!fs->sparse)
		return 1;

	if (is_root(group, 3) || is_root(group, 5) || is_root(group, 7))
		return 1;

	return 0;
}









void ext2_close(struct ext2_fs *fs)
{
	ext2_commit_metadata(fs, 1);
	ext2_sync(fs);

	ext2_bcache_deinit(fs);

	fs->devhandle->ops->close(fs->devhandle->cookie);

	free(fs->gd);
	free(fs);
}

void ext2_commit_metadata(struct ext2_fs *fs, int all_copies)
{
	int	      i;
	int	      num;
	unsigned char sb[fs->blocksize];
	blk_t	      sboffset;

	if (!fs->metadirty || (!all_copies && fs->metadirty == 2))
		return;

	fs->sb.s_r_blocks_count = (fs->r_frac * (loff_t)fs->sb.s_blocks_count) / 100;

	memset(sb, 0, fs->blocksize);

	sboffset = fs->sb.s_first_data_block;
	if (sboffset)
		memcpy(sb, &fs->sb, 1024);
	else
		memcpy(sb+1024, &fs->sb, 1024);

	num = 1;
	if (all_copies)
		num = fs->numgroups;

	i = 1;
	if (fs->metadirty == 1)
		i = 0;

	for (;i<num;i++)
	{
		struct ext2_buffer_head *bh;
		int                      sb_block;
		int                      j;

		if (!ext2_is_group_sparse(fs, i))
			continue;

		sb_block = sboffset + i * fs->sb.s_blocks_per_group;

		bh = ext2_bcreate(fs, sb_block);
		memcpy(bh->data, sb, fs->blocksize);
		ext2_brelse(bh, 1);

		for (j=0;j<fs->gdblocks;j++)
		{
			bh = ext2_bcreate(fs, sb_block + j + 1);
			memcpy(bh->data, ((unsigned char *)fs->gd) + (j << fs->logsize), fs->blocksize);
			ext2_brelse(bh, 1);
		}
	}

	if (all_copies)
		fs->metadirty = 0;
	else
		fs->metadirty = 2;
}

void ext2_sync(struct ext2_fs *fs)
{
	ext2_commit_metadata(fs, 0);
	ext2_bcache_sync(fs);
	fs->devhandle->ops->sync(fs->devhandle->cookie);
}

struct ext2_fs *ext2_open(struct ext2_dev_handle *handle)
{
	struct ext2_fs *fs;

	if ((fs = (struct ext2_fs *)malloc(sizeof(struct ext2_fs))) == NULL)
	{
		fprintf(stderr, "ext2_open: MAerror!\n");
		goto error;
	}

	if ((fs->gd = (struct ext2_group_desc *)malloc(ext2_max_groups * sizeof(struct ext2_group_desc))) == NULL)
	{
		fprintf(stderr, "ext2_open: MAerror!\n");
		goto error_free_fs;
	}

	handle->ops->set_blocksize(handle->cookie, 10);

	if (!handle->ops->read(handle->cookie, &fs->sb, 1, 1)
	    || fs->sb.s_magic != EXT2_SUPER_MAGIC)
	{
		fprintf(stderr, "ext2_open: invalid superblock\n");
		goto error_free_gd;
	}

	if (fs->sb.s_state & EXT2_ERROR_FS)
	{
		fprintf(stderr, "ext2_open: filesystem has errors! e2fsck first!\n");
		goto error_free_gd;
	}

	if (!(fs->sb.s_state & EXT2_VALID_FS))
	{
		fprintf(stderr, "ext2_open: filesystem was not cleanly unmounted! e2fsck first!\n");
		goto error_free_gd;
	}

	if (fs->sb.s_feature_compat ||
	    (fs->sb.s_feature_incompat & ~EXT2_FEATURE_INCOMPAT_FILETYPE) ||
	    (fs->sb.s_feature_ro_compat & ~EXT2_FEATURE_RO_COMPAT_SPARSE_SUPER))
	{
		fprintf(stderr, "ext2_open: fs has incompatible feature enabled\n");
		goto error_free_gd;
	}

	fs->devhandle = handle;
	fs->logsize = fs->sb.s_log_block_size + 10;
	handle->ops->set_blocksize(handle->cookie, fs->logsize);

	if (!ext2_bcache_init(fs))
	{
		fprintf(stderr, "ext2_open: MAerror!\n");
		goto error_free_gd;
	}

	/* From this point on all is well. No more memory allocations
	   are made */

	fs->blocksize = 1 << fs->logsize;

	fs->numgroups = howmany(fs->sb.s_blocks_count-fs->sb.s_first_data_block, fs->sb.s_blocks_per_group);
	fs->gdblocks = howmany(fs->numgroups * sizeof(struct ext2_group_desc), fs->blocksize);
	fs->inodeblocks = howmany(fs->sb.s_inodes_per_group * sizeof(struct ext2_inode), fs->blocksize);
	fs->r_frac = howmany(100 * (loff_t)fs->sb.s_r_blocks_count, fs->sb.s_blocks_count);
	fs->adminblocks = 3 + fs->gdblocks + fs->inodeblocks;

	fs->sparse = 0;
	if (fs->sb.s_feature_ro_compat & EXT2_FEATURE_RO_COMPAT_SPARSE_SUPER)
		fs->sparse = 1;

	ext2_read_blocks(fs, fs->gd, fs->sb.s_first_data_block + 1, fs->gdblocks);

	return fs;

error_free_gd:
	free(fs->gd);
error_free_fs:
	free(fs);
error:
	return NULL;
}
