// vs_tree.h  (this is -*-c++-*-)
//
//  Copyright 1999 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  A simple tree displayer that is a vscreen by itself.  Note that
// you can't modify a tree's contents once it has been created (although you
// can change the display order), and that everything put into the tree until
// the first subtree will display as a flat list.
//
//  Displaying the tree-ness is up to the derived classes (a generic
// indentation policy probably won't work)
//
//  One more point: after much consideration, I have decided that
// *HETEROGENOUS TREES ARE NOT ALLOWED*.  There's just no good way to flexibly
// sort such a beast.  I don't anticipate this being a major problem, though.
//  (note that while not allowed in general, they can sort of be done -- just
// sort with the key item.  A default class that does just this is provided)

#ifndef VS_TREE_H
#define VS_TREE_H

#include "vscreen.h"
#include "vs_treeitem.h"
#include "vs_subtree.h"

#include <list>
#include <assert.h>

class vs_tree_status;

class vs_tree:public vscreen
{
  vs_treeitem *root;
  vs_treeiterator begin, end;

  vs_treeiterator top;
  vs_treeiterator selected;
  // The top item on the current page and the currently selected item.
  // NOTE: it's implicitly assumed in many places in the code that the 
  // currently selected item is visible (ie, on the screen).

  string header,status;
  // The values of the header and status lines (they're stored internally to
  // make it easier to repaint the window correctly)

  string last_search_term;

  int line_of(vs_treeiterator item);
  bool item_visible(vs_treeiterator item);

  void move_selection(vs_treeiterator from, vs_treeiterator to);
  // Moves the on-screen representation of the selection (you should verify
  // that the new location is on the current page)

  list<vs_tree_status *> status_stack,delete_list;
  // Status items are stored on a stack, which is stored in a list.  The
  // deletion list is important since it's how we avoid "delete this;"
protected:
  virtual void paint_header();
  virtual void paint_status();
  // Responsible for actually /drawing/ the header and status lines

  string get_header() {return header;}
  string get_status() {return status;}

  void add_status_widget(vs_tree_status *widget, bool front=true);
  // Adds a widget.
  void remove_status_widget(vs_tree_status *widget);
  // Removes a widget
  vscreen_widget *get_focus();

  vs_treeiterator get_selected() {return selected;}
  vs_treeiterator get_begin() {return begin;}
  vs_treeiterator get_end() {return end;}

  void sync_bounds();
  // This is an awful hack; I've been thinking about an alternate design of
  // the tree code for a while, and this just confirms it.  Yuck! :)
  //  It'll be the first thing to be removed in the next version..
  //  -- well, it wasn't.
public:
  vs_tree();
  vs_tree(vs_treeitem *_root, bool showroot=false);

  void show_header();
  void show_status();
  // Should mainly be used if it's necessary to force an update of the status
  // line

  static int get_header_color() {return get_color("tree_header_color", COLOR_YELLOW, COLOR_BLUE);}
  static int get_background_color() {return get_color("tree_background_color", COLOR_WHITE, COLOR_BLACK);}
  static int get_status_color() {return get_color("tree_status_color", COLOR_YELLOW, COLOR_BLUE);}

  void set_root(vs_treeitem *_root, bool showroot=false);

  virtual void dispatch_char(chtype ch);
  virtual void repaint();

  void set_status(string new_status);
  void set_header(string new_header);
  // Set the status and header lines of the window, respectively.
  // These routines do NOT call refresh()!

  void set_selection(vs_treeiterator to);
  // Directly sets the selection to a given item.  [ the item must be
  // visible -- ie, all its parents must be expanded ]
  // Causes a redraw.

  void delete_status_widget(vs_tree_status *widget);
  // Moves the widget to a queue for later deletion (ie, when we aren't
  // executing a method inside it)

  virtual ~vs_tree() {delete root; root=NULL;}

  void search_for(string s);
};

class vs_tree_status:public vscreen_widget
// used for the vs_tree's status bar.
{
protected:
  vs_tree *get_tree() {return (vs_tree *) get_owner();}
  // Yes, an ugly hack.
public:
  vs_tree_status(vs_tree *_owner):vscreen_widget(_owner) {}
  virtual void display()=0;
};

class vs_tree_msg:public vs_tree_status
// Displays a transient message -- grabs the input focus and vanishes when a
// key is pressed.
{
  string msg;
  int attr;  
public:
  vs_tree_msg(vs_tree *_owner, string _msg, int _attr)
    :vs_tree_status(_owner),msg(_msg),attr(_attr)
  {}
  virtual bool dispatch_char(chtype ch);
  virtual void display();
  virtual void removed() {}
};

#endif
