/*******************************************************************
 * 							 Pilot Software
 *
 *		 Copyright(c) 1995-1998, Palm Computing Inc., All Rights Reserved
 *
 *-------------------------------------------------------------------   
 * FileName:
 *		Hardware.h
 *
 * Description:
 *		General Hardware Equates. This header file
 *  contains general hardware configuration equates that are not dependent on
 *  the version of the hardware we're running on. Equates specific to
 *  a certain component like the 68328, etc. is contained in separate
 *  header files.
 *
 * History:
 *		 4/27/95	RM		Created by Ron Marianetti
 *		 6/12/98	jhl	Updated definitions for hwrNVPrefSet/Get
 *		 6/17/98	jhl	added 'mbtx' ROM token def
 *		 8/17/98	SCL	Synchronized HwrMiscFlags between Sumo/Eleven
 *		 9/14/98	jfs	Added PrvKeyBootKeys def (see ROMBoot.c for info)
 *		10/06/98	jfs	Added software-only hardware ID and features detection defs
 *		10/07/98	jfs	Cleaned up hardware ID bit defs and made unused combos obvious
 *
 *******************************************************************/  

#ifdef	NON_PORTABLE	   // So app's don't mistakenly include this

#ifndef __HARDWARE_H__
#define __HARDWARE_H__



/**************************************************************************
 * General Equates
 ***************************************************************************/
 
// Bits in the low memory global GHwrWakeUp
#define	hwrWakeUpGeneral			0x0001
#define	hwrWakeUpPWM				0x0002
#define	hwrWakeUpFromKey			0x0004				// set if we woke from a key press
#define	hwrWakeUpReturnToSleep	0x0008				// set if we woke from 24 hr. interrupt
																	//  in which case we want to go right back to sleep
#define	hwrWakeUpLCD				0x0010				// <chg 4-1-98 RM> New flag set if LCD needs
																	//  waking. Set by HwrLCDSleep() and cleared 
																	//  by HwrLCDWake()
																	//  See comments for EvtGotUserEvent(). 

// ----------------------------------------------------------------------
// NOTE: Each of the following defines now has a corresponding ROM token
// to support the new software-only hardware ID feature detect mechanism.
// Be sure to create corresponding hardware ROM tokens when adding new
// GHwrMiscFlags and GHwrMiscFlagsExt bit flags, and modify the
// HwrIdentifyFeatures routine to look for them as appropriate.
// ----------------------------------------------------------------------

// Bits in the low memory global GHwrMiscFlags
#define	hwrMiscFlagHasBacklight		0x0001			// set if backlight is present
#define	hwrMiscFlagHasMbdIrDA		0x0002			// set if IrDA is present (on main board)
#define	hwrMiscFlagHasCardIrDA		0x0004			// set if IrDA is present (on memory card)
#define	hwrMiscFlagHasBurrBrown		0x0008			// set if BurrBrown A/D is present
#define	hwrMiscFlagHasJerryHW		0x0010			// set if Jerry Hardware is present
#define	hwrMiscFlagNoRTCBug			0x0020			// set if using rev of DragonBall (3G or later)
																	//  that doesn't require the RealTimeClock
																	//  bug work-around (see TimeMgr68328.c).
																	//  <chg 3-27-98 RM>
#define	hwrMiscFlagHas3vRef			0x0040			// set if switchable 3v reference is present
#define	hwrMiscFlagHasAntennaSw		0x0080			// set if viewer has an antenna raised switch
#define	hwrMiscFlagHasCradleDetect	0x0100			// set if we can detect the cradle
#define	hwrMiscFlagHasSWContrast	0x0200			// set if UI should support software contrast
#define	hwrMiscFlagInvertLCDForBL	0x0400			// set if we need to invert LCD w/Backlight
#define	hwrMiscFlagHasMiscFlagExt	0x0800			// set if we have new hwrMiscFlagsExt

			// The following bit flags are set by HwrIdentifyFeatures.
			// They allow software to read the hardware ID without poking at hardware.
			// They also provide some isolation from different ID detection schemes
			// such as if the ID detection mechanism should change with EZ...
#define	hwrMiscFlagID1				0x1000				// set if ID bit keyBitHard1 was set
#define	hwrMiscFlagID2				0x2000				// set if ID bit keyBitHard2 was set
#define	hwrMiscFlagID3				0x4000				// set if ID bit keyBitHard3 was set
#define	hwrMiscFlagID4				0x8000				// set if ID bit keyBitHard4 was set
#define	hwrMiscFlagIDMask			0xF000
#define	hwrMiscFlagIDOffset		12						// Bits to shift to get a numeric ID


// NOTE: Hardwawre ID hwrMiscFlagIDOther indicates that the old mechanism of using the app
// keys to read the hardware ID at boot time is NOT SUPPORTED. Instead, HwrIdentifyFeatures
// will use the hwrROMTokenHardwareID ROM token to discover the manufacturer and product ID
// of the device. This ROM token will also activate the new software-only hardware features
// detection code. See HwrIdentifyFeatures for details.

// Hardware ID's for DragonBall 328 based products
#define	hwrMiscFlagIDOther		(0)
#define	hwrMiscFlagIDUnused1		(hwrMiscFlagID1)
#define	hwrMiscFlagIDUnused2		(hwrMiscFlagID2)
#define	hwrMiscFlagIDUnused3		(hwrMiscFlagID2 | hwrMiscFlagID1)
#define	hwrMiscFlagIDUnused4		(hwrMiscFlagID3)
#define	hwrMiscFlagIDUnused5		(hwrMiscFlagID3 | hwrMiscFlagID1)
#define	hwrMiscFlagIDUnused6		(hwrMiscFlagID3 | hwrMiscFlagID2)
#define	hwrMiscFlagIDUnused7		(hwrMiscFlagID3 | hwrMiscFlagID2 | hwrMiscFlagID3)
#define	hwrMiscFlagIDUnused8		(hwrMiscFlagID4)
#define	hwrMiscFlagIDUnused9		(hwrMiscFlagID4 | hwrMiscFlagID1)
#define	hwrMiscFlagIDThumper		(hwrMiscFlagID4 | hwrMiscFlagID2)
#define	hwrMiscFlagIDRazor		(hwrMiscFlagID4 | hwrMiscFlagID2 | hwrMiscFlagID1)
#define	hwrMiscFlagIDJerry		(hwrMiscFlagID4 | hwrMiscFlagID3)
#define	hwrMiscFlagIDKarma		(hwrMiscFlagID4 | hwrMiscFlagID3 | hwrMiscFlagID1)
#define	hwrMiscFlagIDRocky		(hwrMiscFlagID4 | hwrMiscFlagID3 | hwrMiscFlagID2)
#define	hwrMiscFlagIDTouchdown	(hwrMiscFlagID4 | hwrMiscFlagID3 | hwrMiscFlagID2 | hwrMiscFlagID1)


// Hardware ID's for DragonBall EZ based products
#define	hwrMiscFlagIDSumo			(hwrMiscFlagID4 | hwrMiscFlagID2 | hwrMiscFlagID1)
#define	hwrMiscFlagIDBrad			(hwrMiscFlagID4 | hwrMiscFlagID3 | hwrMiscFlagID1)


// Bits in the low memory global GHwrMiscFlagsExt
#define	hwrMiscFlagExtSubID1		0x0001				// subtype ID (for feature select in dev.)
#define	hwrMiscFlagExtSubID2		0x0002				// subtype ID (for feature select in dev.)
#define	hwrMiscFlagExtSubID3		0x0004				// subtype ID (for feature select in dev.)
#define	hwrMiscFlagExtSubIDMask	0x0007				// sybtype ID Mask

#define	hwrMiscFlagExtHasLiIon	0x0010				// set if we have Lithium Ion battery
#define	hwrMiscFlagExtHasRailIO	0x0020				// set if we have Rail I/O hardware
#define	hwrMiscFlagExtHasFlash	0x0040				// set if we have Flash ROM
#define	hwrMiscFlagExtHasFParms	0x0080				// set if we have Flash ROM parms area

#define	hwrMiscFlagExt115KIrOK	0x0100				// device supports 115K IR transfers


// Bits for the call HwrDockSignals
#define	hwrDockInSyncButton	0x0001					// Dock button
#define	hwrDockInGeneric1		0x0002					// General Purpose input #1

#define	hwrDockOutGeneric0	0x0001					// General Purpose output #0

// Enum returned by hwrDockStatus() call
typedef enum {
		DockStatusNotDocked = 0,
		DockStatusInModem,
		DockStatusInCharger,
		DockStatusUnknown = 0xFF
		} hwrDockStatusState, *hwrDockStatusStateP;
		
// Define # of card slots
#if EMULATION_LEVEL == EMULATION_NONE
#define	hwrNumCardSlots	1								// 1 card slot
#else
#define	hwrNumCardSlots	2								// 2 card slots for emulation
#endif


// Screen Dimensions
#define	hwrDisplayWidth		160					// Width of display
#define	hwrDisplayHeight		160					// Height of display

// Screen depth info
#define	hwrDisplayBootDepth	1						// Boot time bits per pixel (system default)
#define	hwrDisplayMaxDepth	2						// Maximum bits per pixel hardware supports
#define	hwrDisplayAllDepths	0x0003				// All display depths (in bit map format)

// Screen refresh rate
#define	hwrDisplayPeriod		12						// frame period in ms.


// Battery scaling:
//    Voltage = (level + hwrVoltStepsOffset) / hwrStepsPerVolt
// One of these is loaded into new Global GHwrBatteryStepsPerVolt at boot -soe- 10/11/98
#define	hwrStepsPerVolt33Ref		78						// 78 = 1 volt. - 3.3 volt reference
#define	hwrStepsPerVolt30Ref		85						// 85 = 1 volt. - 3.0 volt reference

#define	hwrVoltStepsOffset	0						// Add to level before scaling		

//----------------------------------------------------------------------------
// Digitizer Dimensions
//----------------------------------------------------------------------------
#define	hwrSystemTop			(hwrDisplayHeight+4)
#define	hwrSystemHeight		56


// If the device supports a Text Services Managers then the are 
// addition buttons in the silk screen area, between the left-most app buttons
// and the left edge of the Graffiti alpha area.
// kwk - this should be a resource, except that Graffiti needs to know about
// the pane split location and the two 'pop up the keyboard' areas when it's
// processing strokes in GraffitiGlue.c

#if TEXT_SERVICES == TEXT_SERVICES_ON

	#define	hwrBtnCount					8
	#define	hwrBtnWidth					20
	#define	hwrGraffitiAlphaWidth	50
	#define	hwrTSMBtnWidth				22
	#define	hwrTSMBtnHeight			16				// Std. height...last button is 1/2 this.
	#define	hwrTSMBtnHeightSumo		14				// Heights change for Sumo silkscreen
	#define	hwrCalcChr					keyboardChr

#else

	#define	hwrBtnCount					4
	#define	hwrBtnWidth					27
	#define	hwrGraffitiAlphaWidth	62
	#define	hwrTSMBtnWidth				0
	#define	hwrTSMBtnHeight			0
	#define	hwrTSMBtnHeightSumo		0
	#define	hwrCalcChr					calcChr

#endif

// Graffiti Area
#define	hwrGraffitiTop						hwrSystemTop															// Top coordinate of Graffiti area
#define	hwrGraffitiLeft					(hwrBtnWidth + hwrTSMBtnWidth)									// Left coordinate of Graffiti area
#define	hwrGraffitiWidth					(hwrDisplayWidth - (hwrBtnWidth *2) - hwrTSMBtnWidth)		// Width of Graffiti area
#define	hwrGraffitiHeight					hwrSystemHeight														// Height of Graffiti area
#define	hwrGraffitiSplit					(hwrGraffitiLeft + hwrGraffitiAlphaWidth)												// Split mode dividing line

// Launch Button
#define	hwrLaunchBtnLeft					0							
#define	hwrLaunchBtnTop					hwrSystemTop						
#define	hwrLaunchBtnWidth					hwrBtnWidth
#define	hwrLaunchBtnHeight				(hwrSystemHeight/2)

// Menu Button
#define	hwrMenuBtnLeft						0							
#define	hwrMenuBtnTop						(hwrSystemTop+hwrMenuBtnHeight)  							
#define	hwrMenuBtnWidth					hwrBtnWidth
#define	hwrMenuBtnHeight					(hwrSystemHeight/2)

// Calculator Button
#define	hwrCalcBtnLeft						(hwrDisplayWidth - hwrBtnWidth)
#define	hwrCalcBtnTop						hwrSystemTop 						
#define	hwrCalcBtnWidth					hwrBtnWidth
#define	hwrCalcBtnHeight					(hwrSystemHeight/2)

// Find Button
#define	hwrFindBtnLeft						(hwrDisplayWidth - hwrBtnWidth)
#define	hwrFindBtnTop						(hwrSystemTop+hwrCalcBtnHeight)	 						
#define	hwrFindBtnWidth					hwrBtnWidth
#define	hwrFindBtnHeight					(hwrSystemHeight/2)

// Keyboard Alpha Button
#define	hwrKeyboardAlphaBtnLeft			(hwrBtnWidth + hwrTSMBtnWidth)						
#define	hwrKeyboardAlphaBtnTop			(hwrSystemTop+hwrSystemHeight-hwrSystemHeight/4)	 						
#define	hwrKeyboardAlphaBtnWidth		18
#define	hwrKeyboardAlphaBtnHeight		(hwrSystemHeight/4)

// Keyboard Numeric Button
#define	hwrKeyboardNumericBtnLeft		(hwrFindBtnLeft-hwrKeyboardNumericBtnWidth)						
#define	hwrKeyboardNumericBtnTop		hwrKeyboardAlphaBtnTop	 						
#define	hwrKeyboardNumericBtnWidth		18
#define	hwrKeyboardNumericBtnHeight	hwrKeyboardAlphaBtnHeight

// TSM Buttons 1
#define	hwrTSMBtn1Left						hwrBtnWidth							
#define	hwrTSMBtn1Top						(hwrSystemTop + (hwrTSMBtnHeight * 0))
#define	hwrTSMBtn1Width					hwrTSMBtnWidth
#define	hwrTSMBtn1Height					hwrTSMBtnHeight

#define	hwrTSMBtn1TopSumo					(hwrSystemTop + (hwrTSMBtnHeightSumo * 0))
#define	hwrTSMBtn1HeightSumo				hwrTSMBtnHeightSumo

// TSM Buttons 2
#define	hwrTSMBtn2Left						hwrBtnWidth							
#define	hwrTSMBtn2Top						(hwrSystemTop + (hwrTSMBtnHeight * 1))
#define	hwrTSMBtn2Width					hwrTSMBtnWidth
#define	hwrTSMBtn2Height					hwrTSMBtnHeight

#define	hwrTSMBtn2TopSumo					(hwrSystemTop + (hwrTSMBtnHeightSumo * 1))
#define	hwrTSMBtn2HeightSumo				hwrTSMBtnHeightSumo

// TSM Buttons 3
#define	hwrTSMBtn3Left						hwrBtnWidth							
#define	hwrTSMBtn3Top						(hwrSystemTop + (hwrTSMBtnHeight * 2))
#define	hwrTSMBtn3Width					hwrTSMBtnWidth
#define	hwrTSMBtn3Height					hwrTSMBtnHeight

#define	hwrTSMBtn3TopSumo					(hwrSystemTop + (hwrTSMBtnHeightSumo * 2))
#define	hwrTSMBtn3HeightSumo				hwrTSMBtnHeightSumo

// TSM Buttons 4 (on Sumo, the last button is full height)
#define	hwrTSMBtn4Left						hwrBtnWidth							
#define	hwrTSMBtn4Top						(hwrSystemTop + (hwrTSMBtnHeight * 3))
#define	hwrTSMBtn4Width					hwrTSMBtnWidth
#define	hwrTSMBtn4Height					(hwrTSMBtnHeight/2)

#define	hwrTSMBtn4TopSumo					(hwrSystemTop + (hwrTSMBtnHeightSumo * 3))
#define	hwrTSMBtn4HeightSumo				hwrTSMBtnHeightSumo


// HwrLCDGetDepth requests
#define	hwrLCDGetCurrentDepth			0				// Get depth display is currently set to

//--------------------------------------------------------------------------
// The battery data in the ROM is stored in a structure like this.
// 
// There is an array of these structures in the ROM.
//
// WARNING: DO NOT CHANGE THE STRUCTURE OF THIS RECORD WITHOUT MODIFYING THE
//          ASSOCIATED ROUTINES IN HwrBatteryTD1.c !!!
//----------------------------------------------------------------------------
typedef struct {
	Word	sysBattDataStructVersion;	// rev level of the structure (this one is 1)
	Word	sysBattMildWrngVoltage;		// in centivolts-- this is the voltage where the warning dialog appears
	Word	sysBattSevereWrngVoltage;	// in cVolts-- this is the voltage where the critical warning dialog appears
	Word	sysBattMildWrngPercent;		// Mild warning percentage for 3.2 and newer (rev 2)
	Word	sysBattSevereWrngPercent;	// Severe warning percentage for 3.2 and newer (rev 2)
	Word	sysBattShutdownEnVoltage;	// in cV-- This voltage causes us to do a software shutdown
	Word	sysBattShutdownDisVoltage;	// in cV-- This voltage allows us to wakeup from shutdown
	Word	sysBattMinIRVoltage;			// prevents IR from working if below this threshold
	DWord	sysBattMildWrngTimeout;		// in ticks-- this is the number of ticks to wait between displaying warning dialogs
	DWord	sysBattSevereWrngTimeout;	// in ticks-- this is the number of ticks to wait between displaying critical warning dialogs
	Word	sysBattCritWarnOffDays;		// in days-- number of days between mild and severe warnings
	DWord	sysBattCritWarnOnTicks;		// in ticks-- number of ticks between mild and severe warnings
	Word	sysBattStepsPerVolt;			// for scaling A/D to Voltages
	SWord	sysBattVoltageStepOffset;	// for scaling A/D to Voltages
	DWord	sysBattReserved1;				// future use
	DWord	sysBattReserved2;				// future use
	Word	sysBattVoltageCurve[11];	// in cV; this is the voltage at the specified percentage/10; i.e. if the voltage
												// with 20% remaining is 2.5 volts, battVoltageCurve[2] = 250.
	} SysBatteryDataStruct, *SysBatteryDataStructP;

//--------------------------------------------------------------------
// Layout of the A to D convertor calibration ROM token.
// This data is calculated for each unit during production, and burnt
// into the ROM.  It consists of 2 precise voltages which are 
// applied to the A to D convertor, as well as their respective outputs
// from the A/D convertor.  Because of manufacturing tolerances in the 
// convertor, its important that the conversion of the output (0-255) 
// into centivolts is tailored to each device during manufacturing.
//--------------------------------------------------------------------
typedef struct {
	Word	lowCentivolts;
	Word	lowAToDOutput;
	Word	highCentivolts;
	Word	highAToDOutput;
} SysAToDCalibrationTokenType;



//----------------------------------------------------------------------------
// The ROM token list is an array of zero or more HwrROMTokenRec records
// terminated by a record having token 0xFFFFFFFF. Each record is variable
// length (6 bytes minimum) and must begin on an even byte boundry.
// 
// WARNING: DO NOT CHANGE THE STRUCTURE OF THIS RECORD WITHOUT MODIFYING THE
//          ASSOCIATED ASSEMBLY ROUTINES IN ROMHardwareTD1.c !!!
//          PrvGetROMToken, HwrGetROMToken, PrvExecuteROMToken1
//----------------------------------------------------------------------------
typedef struct {
	ULong	token;
	UInt	len;
	UChar	data[2];		// variable length data goes here (0 or more chars)
} HwrROMTokenRec, *HwrROMTokenRecPtr;


#define	hwrROMTokenSnum		'snum'	// MUST be the first token, should be 12 bytes long
													//   Snum == Memory Card Flash ID (serial number)
#define	hwrROMTokenIrda		'irda'	// present ONLY on memory card w/IrDA (Gordon)
#define	hwrROMTokenFlex		'flex'	// present ONLY on memory card w/Flex pager (Cessna)
#define	hwrROMTokenBrst		'brst'	// present ONLY when Sharp PSRAM is used (triggers PrvHwrBurstRefresh)
#define	hwrROMTokenBoot1		'cd01'	// boot time code plug-in #1 (See RomBoot.c)
#define	hwrROMTokenBoot2		'cd02'	// boot time code plug-in #2 (See RomBoot.c)
#define	hwrROMTokenBoot3		'cd03'	// boot time code plug-in #3 (See RomBoot.c)
#define	hwrROMTokenBoot4		'cd04'	// boot time code plug-in #4 (See RomBoot.c)
#define	hwrROMTokenBoot5		'cd05'	// boot time code plug-in #5 (See RomBoot.c)
#define	hwrROMTokenAToD		'adcc'	// Data for the A to D Convertor calibration
// 6/17/98 jhl
#define	hwrROMTokenMobitex	'mbtx'	// Mobitex calibration and serial number
#define	hwrROMTokenAlkCal		'calk'	// alkaline calibration value
#define	hwrROMTokenNicadCal	'cncd'	// nicad calibration value


// 3/2/99 SCL
// The following defines are for the first ("manufacturer") digit of the 'snum' token.
#define hwrManu
typedef enum {
	hwrSnumManu3Com1		= '1',
	hwrSnumManuTRG			= '4',
	hwrSnumManu3Com5		= '5',
	hwrSnumManu3Com6		= '6',
	hwrSnumManuMotorola	= '7',
	hwrSnumManuSymbol		= '8',
	hwrSnumManuQualcomm	= '9',
	hwrSnumManu3ComE		= 'E'
	} HwrSnumManufacturerEnum;


// 10/06/98	jfs
// The following structure describes the format of the data field of the hwrROMTokenHardwareID ROM token.
// Note the featureIDs field is a variable length structure in which the OEM can place any number
// of IDs representing features of THEIR particular hardware. Only the oemID and productID fields
// are mandatory, the values of which are assigned to the OEM partner by Palm Computing.
typedef struct {
	DWord	oemID;
	DWord	productID;
	DWord	featureIDs[1];		// OEM-specific variable length data goes here (0 or more DWords)
} HwrIDRec, *HwrIDRecP;

// The following ROM token activates the software-only hardware ID and features detection code.
// See HwrIdentifyFeatures for details.
#define	hwrROMTokenHardwareID		'hwid'

// GHwrMiscFlags ROM tokens (each has a corresponding hwrMiscFlagXXX bit definition)
#define	hwrROMTokenHasBacklight		'hbkl'
#define	hwrROMTokenHasMbdIrDA		'hirm'
#define	hwrROMTokenHasCardIrDA		'hirc'
#define	hwrROMTokenHasBurrBrown		'hbbd'
#define	hwrROMTokenHasJerryHW		'hjrm'
#define	hwrROMTokenNoRTCBug			'hrtc'
#define	hwrROMTokenHas3vRef			'h3vr'
#define	hwrROMTokenHasAntennaSw		'hant'

#define	hwrROMTokenHasCradleDetect	'hcrd'
#define	hwrROMTokenHasSWContrast	'hswc'
#define	hwrROMTokenInvertLCDForBL	'hibl'
#define	hwrROMTokenHasMiscFlagExt	'hmfe'

// GHwrMiscFlagsExt ROM tokens (each has a corresponding hwrMiscFlagExtXXX bit definition)
#define	hwrROMTokenExtHasLiIon		'hbli'
#define	hwrROMTokenExtHasRailIO		'hrio'
#define	hwrROMTokenExtHasFlash		'hflr'
#define	hwrROMTokenExtHasFParms		'hfpr'

#define	hwrROMTokenExt115KIrOK		'h115'



//----------------------------------------------------------------------------
// Equates for the PalmOS 4.0 HwrBattery() call.
//
// This is a multi-purpose call which supercedes HwrBatteryLevel(). Not
// only will it read the Alkaline battery level but it can also be
// used to read the RAM radio NiCad, to schedule a recharge of the
// radio NiCad, etc.
//
//----------------------------------------------------------------------------
typedef enum {
	hwrBatteryCmdInitStage1=0,			// Initialize. Called by system during reset
												// cmdP unused

	hwrBatteryCmdInitStage2,			// 2nd stage of initialize. Called by system during reset
												// after alarm manager is up.  cmdP unused

	hwrBatteryCmdMainRead,				// Read main battery level
												// cmdP = &HwrBatteryCmdReadType
												//    .mVolts <- returned level in millivolts
												//    .abs    <- returned level (0 - 255)

	hwrBatteryCmdRadioRead,				// Read Radio battery level
												// cmdP = &HwrBatteryCmdReadType
												//    .mVolts <- returned level in millivolts
												//    .abs    <- returned level (0 - 255)

	hwrBatteryCmdRadioChargeStart,	// Start Radio battery charging
												// cmdP = &HwrBatteryCmdChargeStart
												//    .startSeconds -> when to start
													
	hwrBatteryCmdRadioChargeStop,		// Stop Radio battery charging
												// cmdP unused
												
	hwrBatteryCmdRadioChargeStatus,	// Return current charging status
												// cmdP = &HwrBatteryCmdChargeStatus
												//    .startSeconds -> scheduled start time
												//    .charging -> true if currently charging
												
	hwrBatteryCmdRadioSetLoaded,		// Inform HwrBattery if radio battery is under
												//   load or not. This call should be used by the radio
												//   interface ONLY!. 
												// cmdP = load boolean
												//  Nil pointer 		-> not loaded
												//  Non-nil pointer 	-> loaded
												
	hwrBatteryCmdRadioSetCalValues,	// Set the battery calibration values
												// cmdP = DWordPtr	(DWord calValue[2])
												//  calValue[0] == alkCal
												//  calValue[1] == nicadCal
	hwrBatteryCmdCalcPercent,			// Ask for the battery percent global to be
												// updated
	hwrBatteryCmdNMI,						// Inform the battery mgr of an NMI event
												
	hwrBatteryCmdAddload,				// Request that a high power device be turned on
	hwrBatteryCmdRemoveload,			// Inform the battery that a high power device is turned off
	
	hwrBatteryCmdKeyWake,				// Sent by Key Mgr when a key wakes up the device - used to restart charging after
												// alkaline batteries get too low suspending charge (on the assumption a user will
												// always hit a kard key when they replace alkalines)
	
	hwrBatteryCmdChargeAlertClear,	// Sent to indicate that user has been notified that
												// charging has been suspended to to low alkalines.

	hwrBatteryCmdSetBatteryKind,		// Request to set the battery kind
	hwrBatteryCmdGetBatteryState,		// Request to return battery state (selects warnings)
	hwrBatteryCmdPeriodicUpdate		// Timer tick routine use to update timers/counters/voltages

	} HwrBatteryCmdEnum;
	
	 
// Parameter blocks for above Battery commands:
typedef struct HwrBatCmdBatteryKindType {
	SysBatteryKind	kind;						// Battery type
	} HwrBatCmdBatteryKindType;
	
typedef struct HwrBatCmdBatteryStateType {
	SysBatteryState state;					// State of system wrt battery capacity
	} HwrBatCmdBatteryStateType;
	
typedef struct HwrBatCmdReadType {
	Word		mVolts;						// level in millivolts (2500 = 2.5 volts)
	Word		abs;							// absolute level (0 -> 255)
	} HwrBatCmdReadType;
	
typedef struct  HwrBatCmdChargeStartType {
	DWord		startSeconds;
	Word		startCode;					// from enum below
	} HwrBatCmdChargeStartType;
	
typedef struct HwrBatCmdChargeStatusType {
	Word		structSize;					// caller must init to sizeof(HwrBatCmdChargeStatusType)

	Word		lastLevel;					// last sampled voltage in millivolts
	Word		flags;						// hwrBatStatusFlagXXX defined below
	DWord		startSeconds;				// Scheduled time of next charge, or 0 
	
	DWord		lastChargeStart;			// Time that last charge cycle started
	DWord		lastChargeFinish;			// Time that last charge cycle finished
												//  or 0 if currently charging
	Word		lastChargePeak;			// Peak voltage seen during last/current
												//  charge cycle
	Word		lastChargeFinal;			// Final voltage at end of last charge
											   //  cycle
												
	DWord		nextAlarmTime;				// Next scheduled alarm for internal maintenance
	Word		totalPower;					// total load being delivered by alk power supply in mW
	DWord		remainingChargeTime;		// the maximum time left on the charge. 
	Word		kicksDone;					// number of Nicad "kicks" (2 second charges every 30 minutes) done.
	Word		kicksSkipped;				// number of kicks desired but skipped for any reason (both of these since last charge)

	} HwrBatCmdChargeStatusType;

// Flag bits for the flags field in above structure
#define	hwrBatStatusFlagCharging	0x0001	// True if charging
#define	hwrBatStatusFlagLoaded		0x0002	// True if battery under load by the radio
#define	hwrBatStatusFlagSuspended	0x0004	// True if current charge is suspended
#define	hwrBatStatusFlagAlkLoad		0x0008	// True if alkaline has another load blocking charging
#define	hwrBatStatusNMIEvent			0x0010	// True if charging suspended by NMI
#define	hwrBatStatusFlagNotify		0x0020	// True if battery dialog should notify the
															// user of charging suspended due to low AAA's.
	
// enumeration for 3 bit stop code field (how did charging terminate)
// will be in upper byte of the flags word
typedef enum {
	hwrAlmStopCodeNul = 0,						// not stopped yet
	hwrAlmStopCodeMin,							// voltage flattened but we kept charing for min time
	hwrAlmStopCodeFlatV,							// normal end - voltage on nicad flattened
	hwrAlmStopCodeMax,							// exceeded our maximum charge time
	hwrAlmStopCodeAbort,							// aborted by outside call
	hwrAlmStopCodeLowAlkSuspended,			// aborted alkaline too low, charging will resume
	hwrAlmStopCodeLowAlkCompleted,			// aborted alkaline too low, charge completed prvMinChargeTimeIfNMI minutes
														// charging will not resume.
	hwrAlmStopCodeBadNicad						// aborted nicad defective
	} PrvAlmStopCodeEnum;

// enumeration for 3 bit start code field (how did charging start)
// will be in upper nibble of upper byte of the flags word
typedef enum {
	hwrAlmStartCodeNul = 0,						// not started yet
	hwrAlmStartCodeTxLoadCrit,					// voltage went below critical level on Tx
	hwrAlmStartCodeTxLoadSched,				// voltage went below level for "charge desired" on Tx
	hwrAlmStartCodeRxLoadCrit,					// voltage went below critical level on Rx (8mA load)
	hwrAlmStartCodeNotRF						// background check of level while Rf not up below no load critical
	} PrvAlmStartCodeEnum; 

// structs, bit defs and types for calling hwrbattery to add/remove power loads
typedef enum {
	hwrBatLoadOK = 0,						// OK to turn on requested device
	hwrBatLoadDeniedBatTooLow,			// can't turn on new load - current Alk Voltaghe too low
	hwrBatLoadDeniedPowerTooHi,		// can't turn on new load - exceeds power supply rating
	hwrBatLoadOKAlreadyOn,				// OK to turn on - but alread on
	hwrBatLoadOKAlreadyOff,				// already off
	hwrBatLoadDeniedCharging,			// Only given when requesting radio operation and currently
												// charging.
	hwrBatLoadDeniedChargingSuspended // Charging but suspended due to low AAA
	} PrvBatResultCodeEnum;   


// Bit definitions for various hi current load devices
#define	hwrBatLoadBitCPULCD			0x0001	// normal CPU and LCD load
#define	hwrBatLoadBitBacklight		0x0002	// Backlight load
#define	hwrBatLoadBitIRDA				0x0004	// IRDA load during Tx (not beam receive)
#define	hwrBatLoadBitHotSync			0x0008	// additional load when hot syncing (to normal CPU/LCD)
#define	hwrBatLoadBitCharger			0x0010	// Charger load
#define	hwrBatLoadBitRadioRx			0x0020	// Jerry Radio in Rx mode
#define	hwrBatLoadBitRadioTx			0x0040	// Jerry Radio in Tx mode
#define	hwrBatLoadBitFlashWrite		0x0080	//	Flash write/erase
#define	hwrBatLoadBitOther			0x8000	// non-standard load (must pass in specific mWatt)

// enum with matches above bits (value = bit position)
typedef enum {
	hwrBatLoadCPULCD = 0,			// (=bit00) normal CPU and LCD load
	hwrBatLoadBacklight,				// (=bit01) Backlight load
	hwrBatLoadIRDA,					// (=bit02) IRDA load during Tx (not beam receive)
	hwrBatLoadHotSync,				// (=bit03) additional load when hot syncing (to normal CPU/LCD)
	hwrBatLoadCharger,				// (=bit04) Charger load
	hwrBatLoadRadioRx,				// (=bit05) Jerry Radio in Rx mode
	hwrBatLoadRadioTx,				// (=bit06) Jerry Radio in Tx mode
	hwrBatLoadFlashWrite,			// (=bit07) is what AMD calls a program or erase operation
	hwrBatLoadUnused2,				// (=bit08) unused
	hwrBatLoadUnused3,				// (=bit09) unused
	hwrBatLoadUnused4,				// (=bit10) unused
	hwrBatLoadUnused5,				// (=bit11) unused
	hwrBatLoadUnused6,				// (=bit12) unused
	hwrBatLoadUnused7,				// (=bit13) unused
	hwrBatLoadUnused8,				// (=bit14) unused
	hwrBatLoadOther					// (=bit15) non-standard load (must pass in specific mWatt)
	} PrvBatLoadTypeEnum;

typedef struct  HwrBatCmdAddLoadType {
	PrvBatLoadTypeEnum		loadType;			// load type bit from enum above - predefined or other
	Byte 							reserved1;
	Word							load;					// power load (in mW - only if type is other)
	PrvBatResultCodeEnum		resultCode;			// result code from enum above
	Byte 							reserved2;
	} HwrBatCmdAddLoadType;
	
// The following include has to appear here, because MemoryPrvNew.h
// needs hwrNumCardSlots, which is defined above...

#include <MemoryPrvNew.h>	// for CardHeaderPtr below


/**************************************************************************
 * Prototypes of functions used only when running on the real hardware
 ***************************************************************************/
#ifdef __cplusplus
extern "C" {
#endif


	// These routines are only accessed from the Debugger or boot code,
	//  so they don't need traps.
	void		HwrPreRAMInit(void); 
	
	void		HwrPreDebugInit(void); 
	
	Err		HwrChecksumBigROM (CardHeaderPtr cardP);

	void 		HwrInit(void); 

	void		HwrPostDebugInit(void); 
	
	void		HwrIdentifyFeatures(void); 
	
	void		HwrResetNMI(void);
	
	DWord		HwrGetRAMSize(UInt cardNo, DWord limit);
	
	void		HwrDrawSplashScreen(void);

	void   	HwrDisableInts(void);
	
	void		HwrShutDownPLL(void);
	
	void		HwrLowBatteryHandler(void);
	
	void		HwrModelSpecificInit(void);

	void		PrvExecuteROMToken1(void);

	void		HwrBurstRefresh(void);			// called (directly) by HwrSleep()

	DWord		PrvKeyBootKeys(void);
	
	void		HwrWakeupIntHandler(void);

	void		HwrRadioShutdown(void);


	// Interrupt handlers should NOT patch the low-memory interrupt vectors.
	// Instead, they should be installed using the following traps, which the
	// kernel will call whenever the given interrupt triggers.
	// Additionally, since the handlers are limited in what they can do (no calls
	// to Memory Manager or other routines that use it, including FeatureMgr),
	// a pointer to globals should be stored in the corresponding LowMem global.
	// See irq1GlobalsP, irq2GlobalsP, irq3GlobalsP, & irq6GlobalsP in <Globals.h>
	void		HwrIRQ1Handler(DWord param)
							SYS_TRAP(sysTrapHwrIRQ1Handler);
	void		HwrIRQ2Handler(DWord param)
							SYS_TRAP(sysTrapHwrIRQ2Handler);
	void		HwrIRQ3Handler(DWord param)
							SYS_TRAP(sysTrapHwrIRQ3Handler);
	void		HwrIRQ4Handler(DWord param)
							SYS_TRAP(sysTrapHwrIRQ4Handler);
	void		HwrIRQ5Handler(DWord param)
							SYS_TRAP(sysTrapHwrIRQ5Handler);
	void		HwrIRQ6Handler(DWord param)
							SYS_TRAP(sysTrapHwrIRQ6Handler);
							
	
	// These routines are for "general" use.
	
	void 		HwrGetRAMMapping(Ptr /*CardInfoPtr*/ cardInfoP, UInt* numBlocksP, 
							DWordPtr physBlockInfoP, DWordPtr logBlockInfoP)
							SYS_TRAP(sysTrapHwrGetRAMMapping);
				
	DWord		HwrMemWritable(VoidPtr addr)
							SYS_TRAP(sysTrapHwrMemWritable);
	
	DWord		HwrMemReadable(VoidPtr addr)
							SYS_TRAP(sysTrapHwrMemReadable);
							
	Err		HwrCursor(Boolean set, Int* modeP, PointType* whereP, PointType* sizeP, 
						UInt* blinkP)
							SYS_TRAP(sysTrapHwrCursor);
	
	void		HwrDelay(ULong	microseconds)
							SYS_TRAP(sysTrapHwrDelay);
	
	void		HwrDoze(Boolean onlyNMI)
							SYS_TRAP(sysTrapHwrDoze);
	
	void		HwrSleep(Boolean untilReset, Boolean emergency)
							SYS_TRAP(sysTrapHwrSleep);
	
	Err		HwrWake(void)
							SYS_TRAP(sysTrapHwrWake);
							
	Err		HwrSetSystemClock(ULong*	freqP)
							SYS_TRAP(sysTrapHwrSetSystemClock);
							
	Err		HwrSetCPUDutyCycle(UInt*	dutyP)
							SYS_TRAP(sysTrapHwrSetCPUDutyCycle);
							
	Err		HwrLCDInit(UInt framePeriod, VoidPtr baseP, UInt width,
							UInt height, UInt depth)
							SYS_TRAP(sysTrapHwrLCDInit);
							
	Err		HwrLCDBaseAddr(Ptr* newAddrP)
							SYS_TRAP(sysTrapHwrLCDBaseAddr);
							
	Err		HwrLCDDrawBitmap(UInt x, UInt y, Ptr bitmapP, Boolean clearFirst)
							SYS_TRAP(sysTrapHwrLCDDrawBitmap);
							
	Err		HwrLCDSleep(Boolean untilReset, Boolean emergency)
							SYS_TRAP(sysTrapHwrLCDSleep);
							
	Err		HwrLCDWake(void)
							SYS_TRAP(sysTrapHwrLCDWake);
							
	Word		HwrLCDGetDepth(Word request)
							SYS_TRAP(sysTrapHwrLCDGetDepth);
							
	Err		HwrTimerInit(UInt timer)
							SYS_TRAP(sysTrapHwrTimerInit);
							
	Err		HwrTimerSleep(UInt timer)
							SYS_TRAP(sysTrapHwrTimerSleep);
							
	Err		HwrTimerWake(UInt timer)
							SYS_TRAP(sysTrapHwrTimerWake);
							
	UInt		HwrBatteryLevel(void)
							SYS_TRAP(sysTrapHwrBatteryLevel);
							
	Boolean	HwrPluggedIn(void)
							SYS_TRAP(sysTrapHwrPluggedIn);
							
	Boolean	HwrBacklight(Boolean set, Boolean newState)
							SYS_TRAP(sysTrapHwrBacklight);
							
	Boolean	HwrEnableDataWrites(void)
							SYS_TRAP(sysTrapHwrEnableDataWrites);
	
	void		HwrDisableDataWrites(void)
							SYS_TRAP(sysTrapHwrDisableDataWrites);
							
	Err		HwrDockSignals(WordPtr inputsP, Word outputSet, Word outputClr)
							SYS_TRAP(sysTrapHwrDockSignals);
	
	Err		HwrGetROMToken (Word cardNo, DWord token, BytePtr *dataP, WordPtr sizeP )
							SYS_TRAP(sysTrapHwrGetROMToken);

	Err		HwrNVPrefSet(DWord creator, Word id, VoidPtr dataP, Word length)
							SYS_TRAP(sysTrapHwrNVPrefSet);
					
	Err		HwrNVPrefGet(DWord creator, Word id, VoidPtr dataP, WordPtr lengthP)
							SYS_TRAP(sysTrapHwrNVPrefGet);
							
	Err		HwrBattery(Word /* HwrBatterCmdEnum*/ cmd, VoidPtr cmdP)
							SYS_TRAP(sysTrapHwrBattery);
							
	Byte		HwrLCDContrast(Boolean set, Byte NewContrastLevel)
							SYS_TRAP(sysTrapHwrLCDContrast);
	
	hwrDockStatusState	HwrDockStatus(void)
							SYS_TRAP(sysTrapHwrDockStatus);
	
#ifdef __cplusplus 
}
#endif


/************************************************************
 * Assembly Function Prototypes
 *************************************************************/
#define	_HwrWake		\
				ASM_SYS_TRAP(sysTrapHwrWake)

#define	_HwrDelay	\
				ASM_SYS_TRAP(sysTrapHwrDelay)


#endif 	//__HARDWARE_H__

#endif 	// NON_PORTABLE
