/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "Miscellaneous.h"

#include "Byteswapping.h"		// Canonical
#include "CPU_REG.h"			// LowMem
#include "Logging.h"			// LogDump
#include "Platform.h"			// Platform::AllocateMemory
#include "RAM_ROM.h"			// Memory::MapPhysicalMemory
#include "Strings.r.h"			// kStr_INetLibTrapBase, etc.
#include "UAE_Utils.h"			// uae_strcpy

#include "SocketMessaging.h"	// CSocket::Startup
#include "DebugMgr.h"			// Debug::Startup
#include "EmRPC.h"				// RPC::Startup

#include <algorithm>			// sort()

extern "C" {
#include "gzip.h"
#include "lzw.h"

int (*write_buf_proc)(char *buf, unsigned size);

DECLARE(uch, inbuf,  INBUFSIZ +INBUF_EXTRA);
DECLARE(uch, outbuf, OUTBUFSIZ+OUTBUF_EXTRA);
DECLARE(ush, d_buf,  DIST_BUFSIZE);
DECLARE(uch, window, 2L*WSIZE);
DECLARE(ush, tab_prefix, 1L<<BITS);

int test = 0;		  /* test .gz file integrity */
int level = 1;		  /* compression level */
int exit_code = OK; 	 /* program exit code */
int verbose = 2;		/* be verbose (-v) */
int quiet = 0;			/* be quiet (-q) */

char ifname[] = "ifname";
char ofname[] = "ofname";
char* progname = "progname";

long bytes_in;			   /* number of input bytes */
long bytes_out; 		   /* number of output bytes */
int  ifd;				   /* input file descriptor */
int  ofd;				   /* output file descriptor */
unsigned insize;		   /* valid bytes in inbuf */
unsigned inptr; 		   /* index of next byte to be processed in inbuf */
unsigned outcnt;		   /* bytes in output buffer */

#include <setjmp.h>
jmp_buf env;

RETSIGTYPE abort_gzip()
{
	LogDump ();
	abort ();
//	longjmp (env, 1);
}

int my_fprintf (FILE*, const char* fmt, ...)
{
	int 	n;
	va_list arg;

	va_start (arg, fmt);
	n = LogGetStdLog ()->VPrintf (fmt, arg);
	va_end (arg);

	return n;
}

}	// extern "C"

#undef get_byte
#undef put_byte
#undef put_long


const uae_u16	kOpcode_ADD 	= 0x0697;	// ADD.L X, (A7)
const uae_u16	kOpcode_LINK	= 0x4E50;
const uae_u16	kOpcode_RTE 	= 0x4E73;
const uae_u16	kOpcode_RTD 	= 0x4E74;
const uae_u16	kOpcode_RTS 	= 0x4E75;
const uae_u16	kOpcode_JMP 	= 0x4ED0;

const uae_u16	kOpcode_CMPW	= 0xB47C;		// CMP.W #$xxxx,D2
const uae_u16	kOpcode_ADDW	= 0xD442;		// ADD.W D2,D2
const uae_u32	kOpcode_JMPPC	= 0x4EFB2002;	// JMP *+$0004(D2.W)
const uae_u16	kOpcode_JMPREL	= 0x4EFA;		// JMP *+$xxxx

uae_u32 gLibErrorBase = 0;

const char* gPalmOSLibraries [] =
{
	"INet.lib",
	"IrDA Library", // Also includes Exchange Lib
	"Security.lib",
	"Web.lib",
//	serIrCommLibNameP,	// ???SerIrCommLib.h doesn't declare the functions as SYSTRAPs!
	"Debugger Comm",
	"IrSerial Library",
	"Net.lib",
	"PADHTAL.lib",
	"Rail I/O Library",
	"RMP NetPlugIn.lib",
	"Serial Lib68681",
	"Serial Library",
	"TCPHTAL.lib",

	NULL
};

// This table should match up with the one above.
const long gResourceBases [] =
{
	kStr_INetLibTrapBase,
	kStr_IrLibTrapBase, // Also includes Exchange Lib
	kStr_SecLibTrapBase,
	kStr_WebLibTrapBase,
//	kStr_serIrCommLibTrapBase,	// ???SerIrCommLib.h doesn't declare the functions as SYSTRAPs!
	kStr_SerLibTrapBase,
	kStr_SerLibTrapBase,
	kStr_NetLibTrapBase,
	kStr_HTALLibTrapBase,
	kStr_RailLibTrapBase,
	kStr_NPILibTrapBase,
	kStr_SerLibTrapBase,
	kStr_SerLibTrapBase,
	kStr_HTALLibTrapBase
};

typedef struct SysLibTblEntryTypeV10 {
	Ptr*			dispatchTblP;					// pointer to library dispatch table
	void*			globalsP;						// Library globals
	} SysLibTblEntryTypeV10;
typedef SysLibTblEntryTypeV10*	SysLibTblEntryV10Ptr;


extern "C"
{
	int PrvGzipReadProc 	(char* buf, unsigned size);
	int PrvGzipWriteProc	(char* buf, unsigned size);
}

static void*	gSrcP;
static void*	gDstP;
static long 	gSrcBytes;
static long 	gDstBytes;
static long 	gSrcOffset;
static long 	gDstOffset;

void CommonStartup (void)
{
	CSocket::Startup ();
	Debug::Startup ();	// Create our sockets
	RPC::Startup ();	// Create our sockets
}

void CommonShutdown (void)
{
	Debug::Shutdown ();
	RPC::Shutdown ();
	CSocket::Shutdown ();

	delete LogGetStdLog ();	// Dumps it to a file, too.
}


// ===========================================================================
//	 StMemory Class
// ===========================================================================
//	Constructor allocates the Ptr
//	Destructor disposes of the Ptr

StMemory::StMemory(
	char*	inPtr)
{
	mIsOwner = (inPtr != NULL);
	mPtr	 = inPtr;
}

StMemory::StMemory(
	long	inSize, 				// Bytes to allocate
	Bool	inClearBytes)			// Whether to clear all bytes to zero
{
	mIsOwner = true;

	if (inClearBytes)
		mPtr = (char*) Platform::AllocateMemoryClear (inSize);
	else
		mPtr = (char*) Platform::AllocateMemory (inSize);
}


StMemory::StMemory(
	const StMemory	&inPointerBlock)
{
	UNUSED_PARAM(inPointerBlock)
}


StMemory::~StMemory()
{
	Dispose ();
}


StMemory&
StMemory::operator = (
	const StMemory	&inPointerBlock)
{
	UNUSED_PARAM(inPointerBlock)

	return *this;
}


void
StMemory::Adopt(
	Ptr 	inPtr)
{
	if (inPtr != mPtr)
	{
		Dispose ();
	
		mIsOwner = (inPtr != NULL);
		mPtr	 = inPtr;
	}
}


Ptr
StMemory::Release() const
{
	mIsOwner = false;
	return mPtr;
}


void
StMemory::Dispose()
{
	if (mIsOwner && (mPtr != NULL))
	{
		Platform::DisposeMemory (mPtr);
	}
	
	mIsOwner = false;
	mPtr	 = NULL;
}



StMemoryMapper::StMemoryMapper (const void* memory, long size) :
	fMemory (memory)
{
	Memory::MapPhysicalMemory (fMemory, size);
}

StMemoryMapper::~StMemoryMapper (void)
{
	Memory::UnmapPhysicalMemory (fMemory);
}


void*	Platform_AllocateMemory (size_t size)
{
	return Platform::AllocateMemory (size);
}


void*	Platform_ReallocMemory	(void* p, size_t size)
{
	return Platform::ReallocMemory (p, size);
}


void	Platform_DisposeMemory	(void* p)
{
	Platform::DisposeMemory (p);
}


// ---------------------------------------------------------------------------
//		 LoadAnyFiles
// ---------------------------------------------------------------------------

void LoadAnyFiles (const FileRefList& fileList)
{
	try
	{
		int 	prcCount = 0;
		int 	psfCount = 0;
		int 	romCount = 0;
		int 	otherCount = 0;

		FileRefList::const_iterator	iter = fileList.begin();
		while (iter != fileList.end())
		{
			// egcs 1.1.1 can't seem to handle iter->Method() here...
			if ((*iter).IsPRC())	prcCount++;
			else if ((*iter).IsPDB()) prcCount++;
			else if ((*iter).IsPQA()) prcCount++;
			else if ((*iter).IsPSF()) psfCount++;
			else if ((*iter).IsROM()) romCount++;
			else otherCount++;

			++iter;
		}

		if ((prcCount > 0) + (psfCount > 0) + (romCount > 0) + (otherCount > 0) > 1)
		{
			Errors::Throw (kError_OnlySameType);
		}
		else if (prcCount > 0)
		{
			::LoadPalmFileList (fileList);
		}
		else if (psfCount > 0)
		{
			if (psfCount > 1)
			{
				Errors::Throw (kError_OnlyOnePSF);
			}
			else
			{
				gEmuPrefs->UpdateRAMMRU (fileList[0]);
				Platform::OpenSession ();
			}
		}
		else if (romCount > 0)
		{
			if (romCount > 1)
			{
				Errors::Throw (kError_OnlyOneROM);
			}
			else
			{
				Preference<Configuration>	pref (kPrefKeyLastConfiguration);
				Configuration	cfg = *pref;
				cfg.fROMFile = (fileList[0]);
				pref = cfg;

				Platform::CreateSession ();
			}
		}
		else
		{
			Errors::Throw (kError_UnknownType);
		}
	}
	catch (ErrCode errCode)
	{
		Errors::ReportIfError (kStr_GenericOperation, errCode, 0, false);
	}
}


// ---------------------------------------------------------------------------
//		 CollectOKObjects
// ---------------------------------------------------------------------------

void CollectOKObjects(FormPtr frm, vector<Word>& okObjects, Bool reportErrors)
{
	Int16 winWidth, winHeight;
	::WinGetDisplayExtent(&winWidth, &winHeight);

	Word	numObjects = FrmGetNumberOfObjects (frm);
	for (Word objIndex = 0; objIndex < numObjects; ++objIndex)
	{
		FormObjectKind	objType = FrmGetObjectType (frm, objIndex);
		Word			objID = FrmGetObjectId (frm, objIndex);

		switch (objType)
		{
			case frmBitmapObj:
			case frmLineObj:
			case frmFrameObj:
			case frmRectangleObj:
			case frmLabelObj:
			case frmTitleObj:
			case frmPopupObj:
				// do nothing for these
				break;
			
			default:
			{
				// Check for completely offscreen objects.
				// (The jury is still out on partially offscreen object.)
				RectangleType	bounds;
				FrmGetObjectBounds (frm, objIndex, &bounds);

				if (bounds.extent.x <= 0 ||
					bounds.extent.y <= 0)
				{
					// Allow zero-sized gadgets and tables.  The former are often
					// used as dummy objects merely to hold references to custom
					// data.  The latter exist because there's no other way to
					// hide a table (there's no "usable" bit).

					if (objType == frmGadgetObj ||
						objType == frmTableObj)
					{
						break;
					}

					// Allow zero-width (but not zero-height) popup triggers.

					if (objType == frmControlObj)
					{
						uaecptr	ctrlPtr = (uaecptr) FrmGetObjectPtr (frm, objIndex);
						uae_u8	style = get_byte (ctrlPtr + offsetof (ControlType, style));

						if (style == popupTriggerCtl)
						{
							if (bounds.extent.x == 0 &&
								bounds.extent.y > 0)
							{
								break;
							}
						}
					}

					// Allow zero-height lists if the number of objects in them is zero.

					if (objType == frmListObj)
					{
						uaecptr	listPtr = (uaecptr) FrmGetObjectPtr (frm, objIndex);
						Word	numItems = get_word (listPtr + offsetof (ListType, numItems));
							// !!! TBD: call LstGetNumberOfItems instead...

						if (numItems == 0)
						{
							if (bounds.extent.x > 0 &&
								bounds.extent.y == 0)
							{
								break;
							}
						}
					}

					// Report any errors.  For now, don't report errors on 1.0
					// devices.  They may not follow the rules, either.  In
					// particular, someone noticed that the Graffiti state
					// indicator has a size of 0,0.

					if (reportErrors && Patches::OSMajorVersion () > 1)
					{
						Errors::ReportSizelessFormObject (objID, bounds);
					}
				}
				else if (bounds.topLeft.x >= winWidth ||
					bounds.topLeft.y >= winHeight ||
					bounds.topLeft.x + bounds.extent.x <= 0 ||
					bounds.topLeft.y + bounds.extent.y <= 0)
				{
					if (reportErrors && Patches::OSMajorVersion () > 1)
					{
						Errors::ReportOffscreenFormObject (objID, bounds);

						FormObjectKind	objType = FrmGetObjectType (frm, objIndex);
						Word			objID = FrmGetObjectId (frm, objIndex);
						RectangleType	bounds;
						FrmGetObjectBounds (frm, objIndex, &bounds);
					}
				}
				else
				{
					okObjects.push_back (objIndex);
				}
				break;
			}
		}
	}
}


// ---------------------------------------------------------------------------
//		 PinRectInRect
// ---------------------------------------------------------------------------

static void PrvOffsetRect (AbsRectType& r, long dh, long dy)
{
	r.left += dh;
	r.right += dh;

	r.top += dy;
	r.bottom += dy;
}

Bool PinRectInRect (AbsRectType& inner, const AbsRectType& outer)
{
	// Do this in a way such that if the incoming rectangle is
	// taller or wider than gdRect that we ensure we see the
	// top and left.

	Bool	result = false;

	if (inner.bottom > outer.bottom)
	{
		::PrvOffsetRect (inner, 0, outer.bottom - inner.bottom);	// Move it up
		result = true;
	}

	if (inner.right > outer.right)
	{
		::PrvOffsetRect (inner, outer.right - inner.right, 0);	// Move it left
		result = true;
	}

	if (inner.top < outer.top)
	{
		::PrvOffsetRect (inner, 0, outer.top - inner.top);	// Move it down
		result = true;
	}

	if (inner.left < outer.left)
	{
		::PrvOffsetRect (inner, outer.left - inner.left, 0);	// Move it right
		result = true;
	}

	return result;
}

// ---------------------------------------------------------------------------
//		 StartsWith
// ---------------------------------------------------------------------------

Bool StartsWith (const char* s, const char* p)
{
	if (strlen (s) < strlen (p))
		return false;

	return (_strnicmp (s, p, strlen (p)) == 0);
}


// ---------------------------------------------------------------------------
//		 EndsWith
// ---------------------------------------------------------------------------

Bool EndsWith (const char* s, const char* p)
{
	if (strlen (s) < strlen (p))
		return false;

	const char* buffer = s + strlen(s) - strlen(p);
	return (_stricmp (buffer, p) == 0);
}


// ---------------------------------------------------------------------------
//		 Platform::GetIndCommPort
// ---------------------------------------------------------------------------

string GetIndCommPort (int index)
{
	StringList	gCachedPorts;

	if (gCachedPorts.size () == 0)
	{
		Platform::CreatePortNameCache (gCachedPorts);
	}

	if (index >= gCachedPorts.size ())
	{
		return string ();
	}

	return gCachedPorts[index];
}


// For "data" databases, we look to see if they have the following structure
// in their appInfo block. If so, we grab the icon and version string from here.
// WARNING: This structure contains variable size fields and must be generated
//	and parsed at programmatically.
#define 	dataAppInfoSignature 'lnch'
#define 	dataAppInfoVersion	3  // current version of header

#include "PalmPack.h"

struct DataAppInfoType
{
	DWord			signature;					// must be 'lnch' (0x6C6E6338)
	Word			hdrVersion; 				// version of this header - must be 3 
	Word			encVersion; 				// encoder version
	Word			verStrWords;				// length of version string array that 
												//	follows in 16-bit words.
	//Byte			verStr[verStrWords];		// 0 terminated version string with
												//	possible extra NULL byte at end for
												//	padding
														
	//--- The title is only present in version 2 or later
	Word			titleWords; 				// length of title string array that follows
												//	 in 16-bit words.
	//Byte			title[titleWords];			// 0 terminated title string with possible
												//	extra NULL at end for padding. 


	Word			iconWords;					// length of icon data that follows in 16-bit
												//	 words.
	//Byte			icon[iconWords];			// icon in "BitmapType" format with possible NULL
												//	byte at end for even Word padding
	Word			smIconWords;				// length of small icon data that follows in
												//	 16-bit words
	//Byte			smIcon[smIconWords];		// small icon in "BitmapType" format with
												// possible NULL byte at end for even Word
												//	padding
												//--------- Version 2 Fields ------------
};

// Size of appInfo block. 
#define 	dataAppInfoVersionSize (sizeof(DataAppInfoType))

#include "PalmPackPop.h"


/***********************************************************************
 *
 * FUNCTION:	IsVisible
 *
 * DESCRIPTION: Returns whether or not the given database represents
 *				an item we want to display.
 *
 * PARAMETERS:	dbType - type of the database
 *
 *				dbCreator - creator signature of the database
 *
 *				dbAttrs - attributes of the database
 *
 * RETURNED:	True if we should include this database in our list.
 *
 ***********************************************************************/

Bool IsVisible (DWord dbType, DWord dbCreator, Word dbAttrs)
{
	UNUSED_PARAM(dbCreator)

	// Don't show anything concerning the Launcher
	// (That comment and the following commented out code was from the
	// Launcher application. I've take it out so that we can run
	// Gremlins over the Launcher).
//	if (dbCreator == sysFileCLauncher)
//		return false;

	// The following test can come and go.	Currently, it's here
	// so that things like Clipper don't show up in the list (just
	// as it doesn't show up in the Launcher).	However, there may
	// be time when we want to show it.  An example would be in
	// an experiemental version of the New Gremlin dialog that
	// separated apps and documents.  Selecting an app in one list
	// would display a list of its documents in the other list.  In
	// that case, we'd need to show clipper in order to be able to
	// display its documents.
	if (dbAttrs & dmHdrAttrHidden)
		return false;

	if (dbAttrs & dmHdrAttrLaunchableData)
		return true;

	if (dbType == sysFileTApplication)
		return true;

	if (dbType == sysFileTPanel)
		return true;

	return false;
}


/***********************************************************************
 *
 * FUNCTION:	IsExecutable
 *
 * DESCRIPTION: Returns whether or not the given database contains an
 *				application in which case we want to present only
 *				the database with the most recent version number (in
 *				case there's more than one database with this one's
 *				type and creator).
 *
 * PARAMETERS:	dbType - type of the database
 *
 *				dbCreator - creator signature of the database
 *
 *				dbAttrs - attributes of the database
 *
 * RETURNED:	True if this database is an executable.
 *
 ***********************************************************************/

Bool IsExecutable (DWord dbType, DWord dbCreator, Word dbAttrs)
{
	UNUSED_PARAM(dbCreator)
	UNUSED_PARAM(dbAttrs)

	if (dbType == sysFileTApplication)
		return true;

	if (dbType == sysFileTPanel)
		return true;

	return false;
}


/***********************************************************************
 *
 * FUNCTION:	AppGetExtraInfo
 *
 * DESCRIPTION: Returns additional information on the application.	This
 *				information is usually pretty expensive to get, so we
 *				defer getting it until as late as possible.
 *
 *				This function is derived from one in the Launcher with
 *				the same name.	That function returned a lot more
 *				information.  This one has been simplified to return
 *				only the application's (or special database's) name.
 *
 * PARAMETERS:	infoP - pointer to the DatabaseInfo struct for the application
 *				we need to get more information on.
 *
 * RETURNED:	Any errors encountered while processing this request.
 *				The requested information is returned back in the DatabaseInfo
 *				struct.
 *
 ***********************************************************************/

static Err AppGetExtraInfo (DatabaseInfo* infoP)
{
	Err err = errNone;

	infoP->name[0] = 0;

	//====================================================================
	// If it's a resource database, we must open it to get the appName
	//====================================================================
	if (infoP->dbAttrs & dmHdrAttrResDB)
	{
		DmOpenRef	appDB = NULL;
		VoidHand	strH;
		
		// Open database 
		appDB = DmOpenDatabase (infoP->cardNo, infoP->dbID, dmModeReadOnly);
		if (appDB == NULL)
		{
			err = DmGetLastErr ();
			goto Exit;
		}

		//...............................
		// Get app name if we don't already have it.
		//...............................
		strH = DmGet1Resource (ainRsc, ainID);
		
		// copy launcher name, if present
		if (strH != NULL)
		{
			uaecptr strP = (uaecptr) MemHandleLock (strH);
			uae_strcpy (infoP->name, strP);
			MemHandleUnlock (strH);
			DmReleaseResource (strH);
		}

		DmCloseDatabase (appDB);
	} // if resource database

	//====================================================================
	// If it's a record database, we look in the appInfo block.
	//====================================================================
	else
	{
		LocalID 	appInfoID;
		VoidHand	appInfoH = 0;
		VoidPtr 	appInfoP = 0;
		uaecptr 	specialInfoP;
		uaecptr 	bP;
		Word		verStrWords, titleWords;

		// Look for app info
		err = DmDatabaseInfo (infoP->cardNo, infoP->dbID, 0,
					0, 0, 0, 0, 0, 0, &appInfoID, 0, 0, 0);

		if (!err && appInfoID)
		{
			// Get handle (if RAM based) and ptr to app Info
			if (MemLocalIDKind (appInfoID) == memIDHandle)
			{
				appInfoH = (VoidHand) MemLocalIDToGlobal (appInfoID, infoP->cardNo);
				appInfoP = MemHandleLock (appInfoH);
			}
			else
			{
				appInfoP = MemLocalIDToGlobal(appInfoID, infoP->cardNo);
			}

			// See if this is the special launcher info and if so, get the icons
			//	out of that.
			specialInfoP = (uaecptr) appInfoP;
			DataAppInfoType specialInfo;

			specialInfo.signature = get_long (specialInfoP + offsetof (DataAppInfoType, signature));
			specialInfo.hdrVersion = get_word (specialInfoP + offsetof (DataAppInfoType, hdrVersion));
			specialInfo.encVersion = get_word (specialInfoP + offsetof (DataAppInfoType, encVersion));

			if (MemPtrSize (appInfoP) >= dataAppInfoVersionSize &&
				specialInfo.signature == dataAppInfoSignature &&
				specialInfo.hdrVersion >= dataAppInfoVersion)
			{
				// Get ptr to version string
				bP = specialInfoP + offsetof (DataAppInfoType, verStrWords);
				verStrWords = get_word (bP);
				bP += sizeof(Word);
				bP += verStrWords * sizeof(Word);

				// Get ptr to name string
				titleWords = get_word (bP);
				bP += sizeof(Word);
				if (titleWords)
				{
					uae_strcpy (infoP->name, bP);
				}
			} // If valid appInfo

			if (appInfoH)
			{
				MemHandleUnlock(appInfoH);
			}
		} // if (!err && appInfoID)
	} // Record Database. 

Exit:

	// If no luck getting the visible name, put in default
	if (infoP->name[0] == 0)
	{
		// Get DB name
		strcpy (infoP->name, infoP->dbName);
	}

	return err; 
}



/***********************************************************************
 *
 * FUNCTION:	AppCompareDataBaseNames
 *
 * DESCRIPTION: sort() callback function to sort entries by name.
 *
 * PARAMETERS:	a, b - references to two DatabaseInfo's to compare.
 *
 * RETURNED:	True if a should appear before b, false otherwise.
 *
 ***********************************************************************/

static bool AppCompareDataBaseNames (const DatabaseInfo& a, const DatabaseInfo& b)
{
	return _stricmp (a.name, b.name) < 0;
}


/***********************************************************************
 *
 * FUNCTION:	GetDatabases
 *
 * DESCRIPTION: Collects the list of entries that should be displayed
 *				in the New Gremlin dialog box.
 *
 *				This function is derived from the Launcher function
 *				AppCreateDataBaseList, as rewritten by Ron for the
 *				3.2 ROMs.
 *
 * PARAMETERS:	dbList -- collection into which we store the found
 *				DatabaseInfo entries.
 *
 * RETURNED:	nothing.
 *
 ***********************************************************************/

void GetDatabases (DatabaseInfoList& dbList, Bool applicationsOnly)
{
	UInt			cardNo;
	Word			numCards;
	Word			numDBs;
	Int				dbIndex;		// UInt results in a bug
	LocalID			dbID;
	Err				err = errNone;
	DatabaseInfo	dbInfo;
	Boolean			needToAddNewEntry;

	//=======================================================================
	// Cycle through all databases in the ROM and RAM and place them into our list.
	//=======================================================================
	numCards = ::MemNumCards ();
	for (cardNo = 0; cardNo < numCards; ++cardNo)
	{
		numDBs = ::DmNumDatabases (cardNo);

		//---------------------------------------------------------------
		// Loop through databases on this card, DmGetDatabase() returns ROM 
		// databases first, followed by RAM databases. 
		//---------------------------------------------------------------
		for (dbIndex = 0; dbIndex < numDBs; ++dbIndex)
		{
			//--------------------------------------------------------
			// Get info on the next database and see if it should be visible.
			//--------------------------------------------------------
			dbID = ::DmGetDatabase (cardNo, dbIndex);
			err = ::DmDatabaseInfo (
						cardNo,
						dbID,
						dbInfo.dbName,		/*nameP*/
						&dbInfo.dbAttrs,
						&dbInfo.version,
						NULL,				/*create date*/
						&dbInfo.modDate,
						NULL,				/*backup date*/
						NULL,				/*modNum*/
						NULL,				/*appInfoID*/
						NULL,				/*sortInfoID*/
						&dbInfo.type,
						&dbInfo.creator);

			Errors::ThrowIfPalmError (err);


			// If it's not supposed to be visible, skip it
			if (applicationsOnly && !::IsVisible (dbInfo.type, dbInfo.creator, dbInfo.dbAttrs))
			{
				continue;
			}

			//--------------------------------------------------------------
			// Save info on this database
			//--------------------------------------------------------------
			dbInfo.dbID = dbID;
			dbInfo.cardNo = cardNo;

			//--------------------------------------------------------------
			// If it's an executable, make sure it's the most recent version in our
			// list
			//--------------------------------------------------------------
			needToAddNewEntry = true;
			if (applicationsOnly && ::IsExecutable (dbInfo.type, dbInfo.creator, dbInfo.dbAttrs))
			{
				// Search for database of same type and creator and check version
				DatabaseInfoList::iterator	thisIter = dbList.begin ();
				while (thisIter != dbList.end ())
				{
					if ((*thisIter).type == dbInfo.type &&
						(*thisIter).creator == dbInfo.creator)
					{
						// If this new one is a newer or same version than the previous one, 
						// replace the previous entry. Checking for == version allows RAM
						// executables to override ROM ones. 
						if (dbInfo.version >= (*thisIter).version)
						{
							::AppGetExtraInfo (&dbInfo);
							*thisIter = dbInfo;
						}

						// Since there's already an item with this type/creator
						// already in the list, there's no need to add another one.
						needToAddNewEntry = false;

						break;
					}

					++thisIter;
				}
			}


			//--------------------------------------------------------------
			// If we still need to add this entry, do so now.
			//--------------------------------------------------------------
			if (needToAddNewEntry)
			{
				::AppGetExtraInfo (&dbInfo);
				dbList.push_back (dbInfo);
			}
		} // for (dbIndex = 0; dbIndex < numDBs; dbIndex++)
	} // for (cardNo = 0; cardNo < MemNumCards(); cardNo++)


	//===========================================================================
	// Sort the list by name
	//===========================================================================
	// Sort the databases by their name.
	sort (dbList.begin (), dbList.end (), AppCompareDataBaseNames);
}



/***********************************************************************
 *
 * FUNCTION:	InstallCalibrationInfo
 *
 * DESCRIPTION: Sets the pen calibration info to be "perfect": no
 *				translation or scaling.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void InstallCalibrationInfo (void)
{
	// Open the preferences database.  If the new version of PrefOpenPreferenceDB
	// exists, then call it.  Otherwise, call the old version.	We can't just
	// unconditionally call the old version, as that has a bug in the newer
	// ROMs that causes it to create the database incorrectly if it doesn't
	// already exist.

	DmOpenRef	dbP;
	if (LowMem::TrapExists (sysTrapPrefOpenPreferenceDB))
		dbP = ::PrefOpenPreferenceDB (false);
	else
		dbP = ::PrefOpenPreferenceDBV10 ();

	if (dbP)
	{
		// Get the calibration information.

		VoidHand	resourceH = ::DmGetResource (sysResTSysPref, sysResIDSysPrefCalibration);

		// If that information doesn't exist, go about creating it.

		if (!resourceH)
		{
			resourceH = ::DmNewResource (dbP, sysResTSysPref, sysResIDSysPrefCalibration,
						4 * sizeof(UInt));
		}

		if (resourceH)
		{
			// Write in the calibration information.  The information has the
			// following format and values:
			//
			//		scaleX	: 256
			//		scaleY	: 256
			//		offsetX :	0
			//		offsetY :	0
			//
			// We encode that data here as a string of bytes to avoid endian problems.

			VoidPtr resP = ::MemHandleLock (resourceH);

			unsigned char data[] = { 1, 0, 1, 0, 0, 0, 0, 0 };

			::DmWrite (resP, 0, data, 4 * sizeof (UInt));

			::MemHandleUnlock (resourceH);
			::DmReleaseResource (resourceH);
		}

		::DmCloseDatabase (dbP);
	}
}


/***********************************************************************
 *
 * FUNCTION:	ResetCalibrationInfo
 *
 * DESCRIPTION: Sets the pen calibration info to be "perfect": no
 *				translation or scaling.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void ResetCalibrationInfo (void)
{
	// Reset the pen calibration info by calling PenCalibrate with the right
	// parameters.  Unfortunately, due presumably to division rounding errors,
	// doing this just once doesn't necessarily get the scaling and offset
	// values exactly.  However, making a few calls to PenCalibrate seems
	// to get us to home in on the perfect calibration values.

	Bool	perfect = false;

	for (int kk = 0; kk < 3 && !perfect; ++kk)
	{
		#define target0X	10				// top left
		#define target0Y	10
		#define target1X	(160-10)		// bottom right
		#define target1Y	(160-10)

		Err			err;
		PointType	digPoints[2];
		PointType	scrPoints[2];

		scrPoints[0].x = target0X;
		scrPoints[0].y = target0Y;
		scrPoints[1].x = target1X;
		scrPoints[1].y = target1Y;

		digPoints[0].x = 0x100 - target0X;
		digPoints[0].y = 0x100 - target0Y;
		digPoints[1].x = 0x100 - target1X;
		digPoints[1].y = 0x100 - target1Y;

		err = ::PenRawToScreen(&digPoints[0]);
		err = ::PenRawToScreen(&digPoints[1]);
		err = ::PenCalibrate(&digPoints[0], &digPoints[1], &scrPoints[0], &scrPoints[1]);

		DmOpenRef	dbP;
		if (LowMem::TrapExists (sysTrapPrefOpenPreferenceDB))
			dbP = ::PrefOpenPreferenceDB (false);
		else
			dbP = ::PrefOpenPreferenceDBV10 ();

		if (dbP)
		{
			VoidHand	resourceH = ::DmGetResource (sysResTSysPref, sysResIDSysPrefCalibration);

			if (resourceH)
			{
				VoidPtr			resP = ::MemHandleLock (resourceH);
				unsigned char	perfect_pattern[] = { 1, 0, 1, 0, 0, 0, 0, 0 };

				perfect = (uae_memcmp ((void*) perfect_pattern, (uaecptr) resP, 8) == 0);

				::MemHandleUnlock (resourceH);
				::DmReleaseResource (resourceH);
			}

			::DmCloseDatabase (dbP);
		}
	}
}


/***********************************************************************
 *
 * FUNCTION:	SetHotSyncUserName
 *
 * DESCRIPTION: Calls the Data Link Manager to set the user's HotSync
 *				name.  Many applications key off this name for things
 *				like copy protection, so we set this value when they
 *				boot up.
 *
 * PARAMETERS:	userNameP - the user's name.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

const UInt	kMagicRefNum	= 0x666;	// See comments in HtalLibSendReply.

void SetHotSyncUserName (const char* userNameP)
{
	if (LowMem::GetTrapAddress (sysTrapDlkDispatchRequest) == UAE_NULL)
		return;

	if (!userNameP)
		return;

	size_t	userNameLen = strlen (userNameP) + 1;

	// We need to prepare a command block for the DataLink Manager.
	// Define one large enough to hold all the data we'll pass in.
	//
	// The format of the data block is as follows:
	//
	//		[byte] DlpReqHeaderType.id			: Command request number (== dlpWriteUserInfo)
	//		[byte] DlpReqHeaderType.argc		: # of arguments for this command (== 1)
	//
	//		[byte] DlpTinyArgWrapperType.bID	: ID of first argument (== dlpWriteUserInfoReqArgID)
	//		[byte] DlpTinyArgWrapperType.bSize	: Size in bytes of first argument (== whatever)
	//
	//		[long] DlpWriteUserInfoReqHdrType.userID		: Not used here - set to zero
	//		[long] DlpWriteUserInfoReqHdrType.viewerID		: Not used here - set to zero
	//		[long] DlpWriteUserInfoReqHdrType.lastSyncPC	: Not used here - set to zero
	//		[8byt] DlpWriteUserInfoReqHdrType.lastSyncDate	: Not used here - set to zero
	//		[long] DlpWriteUserInfoReqHdrType.modFlags		: Bits saying what values are being set
	//		[byte] DlpWriteUserInfoReqHdrType.userNameLen	: Length of user name + NULL
	//
	//		[str ] userName

	char	buffer[ sizeof (DlpReqHeaderType) +
					sizeof (DlpTinyArgWrapperType) +
					sizeof (DlpWriteUserInfoReqHdrType) +
					dlpMaxUserNameSize];

	// Get handy pointers to all of the above.
	DlpReqHeaderType*			reqHdr = (DlpReqHeaderType*) buffer;
	DlpTinyArgWrapperType*		reqWrapper = (DlpTinyArgWrapperType*) (((char*) reqHdr) + sizeof(DlpReqHeaderType));
	DlpWriteUserInfoReqHdrType* reqArgHdr = (DlpWriteUserInfoReqHdrType*) (((char*) reqWrapper) + sizeof(DlpTinyArgWrapperType));
	char*						reqName = ((char*) reqArgHdr) + sizeof (DlpWriteUserInfoReqHdrType);

	// Fill in request header
	reqHdr->id		= dlpWriteUserInfo;
	reqHdr->argc	= 1;

	// Fill in the request arg wrapper
	reqWrapper->bID 	= (Byte) dlpWriteUserInfoReqArgID;
	reqWrapper->bSize	= (Byte) (sizeof (*reqArgHdr) + userNameLen);

	// Fill in request arg header
	reqArgHdr->modFlags 	= dlpUserInfoModName;
	reqArgHdr->userNameLen	= userNameLen;

	// Copy in the user name.
	strcpy (reqName, userNameP);

	// Build up a session block to hold the command block.
	DlkServerSessionType	session;
	memset (&session, 0, sizeof (session));
	session.htalLibRefNum = kMagicRefNum;	// See comments in HtalLibSendReply.
	session.gotCommand = true;
	session.cmdLen = sizeof (buffer);
	session.cmdP = buffer;

	// For simplicity, byteswap here so that we don't have to reparse all
	// that above data in DlkDispatchRequest.

	Canonical (session.htalLibRefNum);
	Canonical (session.gotCommand);
	Canonical (session.cmdLen);
	Canonical (session.cmdP);

	Canonical (reqHdr->id);
	Canonical (reqHdr->argc);

	Canonical (reqWrapper->bID);
	Canonical (reqWrapper->bSize);

	Canonical (reqArgHdr->modFlags);
	Canonical (reqArgHdr->userNameLen);

	// Finally, install the name.
	Err err = DlkDispatchRequest (&session);

#if 0
	ULong	lastSyncDate;
	char	userName[dlkUserNameBufSize];
	Err 	err = DlkGetSyncInfo(0/*succSyncDateP*/, &lastSyncDate, 0/*syncStateP*/,
								userName, 0/*logBufP*/, 0/*logLenP*/);
#endif
}


/***********************************************************************
 *
 * FUNCTION:    SeperateList
 *
 * DESCRIPTION: Break up a comma-delimited list of items, returning the
 *				pieces in a StringList.
 *
 * PARAMETERS:  stringList - the StringList to receive the broken-up
 *					pieces of the comma-delimited list.  This collection
 *					is *not* first cleared out, so it's possible to add
 *					to the collection with this function.
 *
 *				str - the string containing the comma-delimited items.
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void SeperateList (StringList& stringList, string str, char delimiter)
{
	string::size_type	offset;

	while ((offset = str.find (delimiter)) != string::npos)
	{
		string	nextElement = str.substr (0, offset);
		str = str.substr (offset + 1);
		stringList.push_back (nextElement);
	}

	stringList.push_back (str);
}


/***********************************************************************
 *
 * FUNCTION:	RunLengthEncode
 *
 * DESCRIPTION: Pack data according to the scheme used in QuickDraw's
 *				PackBits routine.  This is the format, according to
 *				Macintosh Technote 1023:
 *
 *				The first byte is a flag-counter byte that specifies
 *				whether or not the following data is packed, and the
 *				number of bytes involved.
 *
 *				If this first byte is a negative number, the following
 *				data is packed and the number is a zero-based count of
 *				the number of times the data byte repeats when expanded.
 *				There is one data byte following the flag-counter byte
 *				in packed data; the byte after the data byte is the next
 *				flag-counter byte.
 *
 *				If the flag-counter byte is a positive number, then the
 *				following data is unpacked and the number is a zero-based
 *				count of the number of incompressible data bytes that
 *				follow. There are (flag-counter+1) data bytes following
 *				the flag-counter byte. The byte after the last data byte
 *				is the next flag-counter byte.
 *
 *				Consider the following example:
 *
 *				Unpacked data:
 *
 *					AA AA AA 80 00 2A AA AA AA AA 80 00
 *					2A 22 AA AA AA AA AA AA AA AA AA AA
 *
 *				After being packed by PackBits:
 *
 *					FE AA			; (-(-2)+1) = 3 bytes of the pattern $AA
 *					02 80 00 2A 	; (2)+1 = 3 bytes of discrete data
 *					FD AA			; (-(-3)+1) = 4 bytes of the pattern $AA
 *					03 80 00 2A 22	; (3)+1 = 4 bytes of discrete data
 *					F7 AA			; (-(-9)+1) = 10 bytes of the pattern $AA
 *
 *				or
 *
 *					FE AA 02 80 00 2A FD AA 03 80 00 2A 22 F7 AA
 *					*	  * 		  * 	*			   *
 *
 *				The bytes with the asterisk (*) under them are the
 *				flag-counter bytes. PackBits packs the data only when
 *				there are three or more consecutive bytes with the same
 *				data; otherwise it just copies the data byte for byte
 *				(and adds the count byte).
 *
 * PARAMETERS:	srcPP - pointer to the pointer to the source bytes.  The
 *					referenced pointer gets udpated to point past the
 *					last byte included the packed output.
 *
 *				dstPP - pointer to the pointer to the destination buffer.
 *					The referenced pointer gets updated to point past
 *					the last byte stored in the output buffer.
 *
 *				srcBytes - length of the buffer referenced by srcPP
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void RunLengthEncode (void** srcPP, void** dstPP, long srcBytes, long dstBytes)
{
	UNUSED_PARAM(dstBytes)

	enum { kBeginRun, kRepeatRun, kCopyRun };

	uae_u8* srcP = (uae_u8*) *srcPP;
	uae_u8* dstP = (uae_u8*) *dstPP;
	uae_u8* opP;
	long	sample[3] = { -1, -1, -1 }; // Type must be > uae_u8 so that it can hold -1.
	long	opCount;
	long	state = kBeginRun;

	for (srcBytes += 1; srcBytes >= 0; --srcBytes)
	{
		sample[0] = sample[1];
		sample[1] = sample[2];
		sample[2] = -1;

		if (srcBytes > 1)
		{
			sample[2] = *srcP++;
		}

		switch (state)
		{
			case kBeginRun: // Determine whether or not to pack the bytes
				if (sample[2] == sample[0] && sample[2] == sample[1])
				{
					state		= kRepeatRun;
					opCount 	= -2;
				}
				else if (sample[0] != -1)
				{
					state		= kCopyRun;
					opCount 	= 0;
					opP 		= dstP++;
					*dstP++ 	= (uae_u8) sample[0];
				}
				break;

			case kRepeatRun:	// We're packing bytes
				if (sample[2] == sample[1])
				{
					--opCount;

					if (opCount > -127)
					{
						break;
					}

					sample[2]	= -1;
				}

				sample[1]	= -1;
				*dstP++ 	= (uae_u8) opCount;
				*dstP++ 	= (uae_u8) sample[0];
				state		= kBeginRun;
				break;

			case kCopyRun:	// We're copying bytes
				if (sample[0] != sample[1] || sample[0] != sample[2])
				{
					*dstP++ = (uae_u8) sample[0];
					++opCount;

					if (opCount >= 127)
					{
						*opP	= (uae_u8) opCount;
						state	= kBeginRun;
						break;
					}
				}
				else
				{
					*opP	= (uae_u8) opCount;
					state	= kRepeatRun;
					opCount = -2;
				}
				break;
		}
	}

	if (state == kCopyRun)
	{
		*opP = (uae_u8) opCount;
	}

	*srcPP = (void*) srcP;
	*dstPP = (void*) dstP;
}


/***********************************************************************
 *
 * FUNCTION:	RunLengthDecode
 *
 * DESCRIPTION: Decode the data packed by RunLengthEncode.
 *
 * PARAMETERS:	srcPP -
 *
 *				dstPP -
 *
 *				dstBytes -
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void RunLengthDecode (void** srcPP, void** dstPP, long srcBytes, long dstBytes)
{
	UNUSED_PARAM(srcBytes)

	uae_s8* 	srcP	= (uae_s8*) *srcPP;
	uae_s8* 	dstP	= (uae_s8*) *dstPP;
	uae_s8* 	limitP	= dstP + dstBytes;

	while (dstP < limitP)
	{
		int op = *srcP++;
		if (op == -128)
		{
			// Nothing
		}
		else if (op >= 0)
		{
			int count = op + 1;
			do
			{
				*dstP++ = *srcP++;
			} while (--count);
		}
		else
		{
			int 	count = 1 - op;
			uae_u8	fillData = *srcP++;
			do
			{
				*dstP++ = fillData;
			} while (--count);
		}
	}

	*srcPP = (void*) srcP;
	*dstPP = (void*) dstP;
}


/***********************************************************************
 *
 * FUNCTION:	RunLengthWorstSize
 *
 * DESCRIPTION: Calculate the largest buffer needed when packing a
 *				buffer "srcBytes" long.  The algorithm is based on
 *				that found in Macintosh Technote 1023.
 *
 * PARAMETERS:	srcBytes - number of bytes in the buffer to be encoded.
 *
 * RETURNED:	Largest buffer size needed to encode source buffer.
 *
 ***********************************************************************/

long RunLengthWorstSize (long srcBytes)
{
	long	maxDestBytes = (srcBytes + (srcBytes + 126) / 127);

	return maxDestBytes;
}


/***********************************************************************
 *
 * FUNCTION:	GzipEncode
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	srcPP - pointer to the pointer to the source bytes.  The
 *					referenced pointer gets udpated to point past the
 *					last byte included the packed output.
 *
 *				dstPP - pointer to the pointer to the destination buffer.
 *					The referenced pointer gets updated to point past
 *					the last byte stored in the output buffer.
 *
 *				srcBytes - length of the buffer referenced by srcPP
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void GzipEncode (void** srcPP, void** dstPP, long srcBytes, long dstBytes)
{
	gSrcP		= *srcPP;
	gDstP		= *dstPP;
	gSrcBytes	= srcBytes;
	gDstBytes	= dstBytes;
	gSrcOffset	= 0;
	gDstOffset	= 0;

	bytes_in = srcBytes;	// (for gzip internal debugging)

	ush 	attr = 0;		   /* ascii/binary flag */
	ush 	deflate_flags = 0; /* pkzip -es, -en or -ex equivalent */
	int 	method;

	clear_bufs ();

	read_buf		= &::PrvGzipReadProc;
	write_buf_proc	= &::PrvGzipWriteProc;

	bi_init (NULL);
	ct_init (&attr, &method);
	lm_init (level, &deflate_flags);

	deflate ();

	// Perform the equivalent of a put_byte(0) to pad out the
	// compressed buffer.  gzip apparently can skid off the
	// end of the compressed data when inflating it, so we need
	// an extra zero.
	//
	// We do the put_byte operation by hand because we've undef'ed
	// put_byte so that it doesn't conflict with the UAE put_byte function.

	outbuf[outcnt++] = 0;

	flush_outbuf ();

	*srcPP = ((char*) gSrcP) + gSrcOffset;
	*dstPP = ((char*) gDstP) + gDstOffset;
}


/***********************************************************************
 *
 * FUNCTION:	GzipDecode
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	srcPP -
 *
 *				dstPP -
 *
 *				dstBytes -
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void GzipDecode (void** srcPP, void** dstPP, long srcBytes, long dstBytes)
{
	gSrcP		= *srcPP;
	gDstP		= *dstPP;
	gSrcBytes	= srcBytes;
	gDstBytes	= dstBytes;
	gSrcOffset	= 0;
	gDstOffset	= 0;

	clear_bufs ();

	read_buf		= &::PrvGzipReadProc;
	write_buf_proc	= &::PrvGzipWriteProc;

	inflate ();

	*srcPP = ((char*) gSrcP) + gSrcOffset;
	*dstPP = ((char*) gDstP) + gDstOffset;
}


/***********************************************************************
 *
 * FUNCTION:	GzipWorstSize
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	srcBytes - number of bytes in the buffer to be encoded.
 *
 * RETURNED:	Largest buffer size needed to encode source buffer.
 *
 ***********************************************************************/

long GzipWorstSize (long srcBytes)
{
	long	maxDestBytes = srcBytes * 2;

	return maxDestBytes;
}


/***********************************************************************
 *
 * FUNCTION:	PrvGzipReadProc
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	.
 *
 * RETURNED:	.
 *
 ***********************************************************************/

int PrvGzipReadProc (char* buf, unsigned size)
{
	if (gSrcOffset == gSrcBytes)
		return EOF;

	if (size > (gSrcBytes - gSrcOffset))
		size = gSrcBytes - gSrcOffset;

	if (size > 0)
	{
		memcpy (buf, ((char*) gSrcP) + gSrcOffset, size);
		gSrcOffset += size;
	}

	return size;
}


/***********************************************************************
 *
 * FUNCTION:	PrvGzipWriteProc
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	.
 *
 * RETURNED:	.
 *
 ***********************************************************************/

int PrvGzipWriteProc (char* buf, unsigned size)
{
	if (gDstOffset == gDstBytes)
		return EOF;

	if (size > (gDstBytes - gDstOffset))
		size = gDstBytes - gDstOffset;

	if (size > 0)
	{
		memcpy (((char*) gDstP) + gDstOffset, buf, size);
		gDstOffset += size;
	}

	return size;
}


#pragma mark -

static int kBitCount[16] =
{
	0, 1, 1, 2,
	1, 2, 2, 3,
	1, 2, 2, 3,
	2, 3, 3, 4
};

int CountBits (uae_u32 v)
{
	return	kBitCount[ (v >> 0) & 0x0F ] +
			kBitCount[ (v >> 4) & 0x0F ] +
			kBitCount[ (v >> 8) & 0x0F ] +
			kBitCount[ (v >> 12) & 0x0F ];
}


/***********************************************************************
 *
 * FUNCTION:	NextPowerOf2
 *
 * DESCRIPTION: Calculates the next power of two above the given number
 *				If the given number is already a power of two, it is
 *				returned.
 *
 * PARAMETERS:	n - probe number
 *
 * RETURNED:	Next power of two above the probe number, or the number
 *				itself if it is a power of two.
 *
 ***********************************************************************/

//	Seven implementations!	No waiting!

uae_u32 NextPowerOf2 (uae_u32 n)
{
	// Smear down the upper 1 bit to all bits lower than it.

	uae_u32 n2 = n;

	n2 |= n2 >> 1;
	n2 |= n2 >> 2;
	n2 |= n2 >> 4;
	n2 |= n2 >> 8;
	n2 |= n2 >> 16;

	// Now use itself to clear all the lower bits.

	n2 &= ~(n2 >> 1);

	// If n2 ends up being the same as what we started with, keep it.
	// Otherwise, we need to bump it by a factor of two (round up).

	if (n2 != n)
		n2 <<= 1;

	return n2;
}


#if 0
uae_u32 NextPowerOf2 (uae_u32 n)
{
	uae_u32 startn = n;
	uae_u32 prevn = 0;

	while (n)		// Loop until we're out of bits
	{
		prevn = n;	// Remember what we're starting with.  When "n"
					// reaches zero, prevn will hold the previous value,
					// which will have a single bit in it.	Since we're
					// whacking off bits from the bottom, this will be
					// the highest bit.
		n &= n - 1; // Mask off the low bit
	}

	// If prevn ends up being the same as what we started with, keep it.
	// Otherwise, we need to bump it by a factor of two.

	if (prevn != startn)
		prevn <<= 1;

	return prevn;
}
#endif


#if 0
	// This was my own first attempt.  Pretty lame...

uae_u32 NextPowerOf2 (uae_u32 x)
{
	// Figure out the next power-of-2 higher than or equal to 'x'. We do
	// this by continually shifting x to the left until we get a '1' in
	// the upper bit. At the same time, we shift 0x80000000 to the right.
	// When we find that '1' in the upper bit, the shifted 0x80000000
	// pattern should hold our power-of-two.
	//
	// That approach is good for finding the next power-of-2 higher than
	// a given value, so in order to deal with the 'or equal to' part
	// of the task, we decrement x by 1 before embarking on this adventure.
	//
	// This function'll fail for x == 0 or x == 1, as well x > 0x80000000,
	// so handle those cases up front:

	if (x > 0x80000000)
		return -1;

	if (x == 0 || x == 1)
		return x;

	--x;

	unsigned long	result = 0x80000000;
	while (((x <<= 1) & 0x80000000) == 0)	// check for the highest set bit.
		result >>= 1;

	return result;
}
#endif

#if 0
	// This one was posted to the net.	Seems most reasonable.
	// Fails when n == 0.
uae_u32 HighBitNumber (uae_u32 n)
{
	uae_u32 i = (n & 0xffff0000) ? 16 : 0;

	if ((n >>= i) & 0xff00)
	{
		i |= 8;
		n >>= 8;
	}

	if (n & 0xf0)
	{
		i |= 4;
		n >>= 4;
	}

	if (n & 0xc)
	{
		i |= 2;
		n >>= 2;
	}

	return (i | (n >> 1));
}
#endif

#if 0
	// This one was posted to the net.	Uses a loop; not quite as effecient.
	// Seems pretty buggy, since it doesn't work for x == 0, 1, or 2.
uae_u32 HighBitNumber (uae_u32 x)
{
	unsigned long mask=2, numBits=1;
	while (mask < x)
	{
		mask += mask;
		numBits++;
	}

	return numBits;
}
#endif

#if 0
	// This one was posted to the net. Make up to 5 comparisons, which is
	// more than the one we're using.
uae_u32 HighBitNumber (uae_u32 x)
{
#define hi_bit(n)\
	((n)>=1<<16?(n)>=1<<24?(n)>=1<<28?(n)>=1<<30?(n)>=1<<31?31:30:(n)>=1<<29?\
	29:28:(n)>=1<<26?(n)>=1<<27?27:26:(n)>=1<<25?25:24:(n)>=1<<20?(n)>=1<<22?\
	(n)>=1<<23?23:22:(n)>=1<<21?21:20:(n)>=1<<18?(n)>=1<<19?19:18:(n)>=1<<17?\
	17:16:(n)>=1<<8?(n)>=1<<12?(n)>=1<<14?(n)>=1<<15?15:14:(n)>=1<<13?13:12:(\
	n)>=1<<10?(n)>=1<<11?11:10:(n)>=1<<9?9:8:(n)>=1<<4?(n)>=1<<6?(n)>=1<<7?7:\
	6:(n)>=1<<5?5:4:(n)>=1<<2?(n)>=1<<3?3:2:(n)>=1<<1?1:(n)>=1<<0?0:-1)

	return hi_bit (x);
}
#endif

#if 0
	// This one was posted to the net (by the same guy posting that macro).
	// Pretty neat until that divide by 37.
uae_u32 HighBitNumber (uae_u32 x)
{
	static const int t[] =
	{
		-1,  0, 25,  1, 22, 26, 31,  2, 15, 23, 29, 27, 10, -1, 12,  3,  6, 16,
		-1, 24, 21, 30, 14, 28,  9, 11,  5, -1, 20, 13,  8,  4, 19,  7, 18, 17
	};

	return t[(n |= n >> 1, n |= n >> 2, n |= n >> 4, n |= n >> 8, n |= n >> 16) % 37];
}
#endif


/***********************************************************************
 *
 * FUNCTION:	DateToDays
 *
 * DESCRIPTION: Convert a year, month, and day into the number of days
 *				since 1/1/1904.
 *
 *				Parameters are not checked for valid dates, so it's
 *				possible to feed in things like March 35, 1958.  This
 *				function also assumes that year is at least 1904, and
 *				will only work up until 2040 or so.
 *
 * PARAMETERS:	year - full year
 *
 *				month - 1..12
 *
 *				day - 1..31
 *
 * RETURNED:	Number of days since 1/1/1904.
 *
 ***********************************************************************/

uae_u32 DateToDays (uae_u32 year, uae_u32 month, uae_u32 day)
{
	static const int	month2days[] =
	{
		  0,	 31,	 59,	 90,	120,	151,
		181,	212,	243,	273,	304,	334
	};


	// Normalize the values.

	year -= 1904;
	month -= 1;
	day -= 1;

	// Not counting any possible leap-day in the current year, figure out
	// the number of days between now and 1/1/1904.

	const uae_u32	kNumDaysInLeapCycle = 4 * 365 + 1;

	uae_u32 days =	day + month2days[month] +
					(year * kNumDaysInLeapCycle + 3) / 4;

	// Now add in this year's leap-day, if there is one.

	if ((month >= 2) && ((year & 3) == 0))
		days++;

	return days;
}


/***********************************************************************
 *
 * FUNCTION:	GetLibraryName
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	The libraries name, or an empty string if the library
 *				could not be found.
 *
 ***********************************************************************/

string GetLibraryName (uae_u16 refNum)
{
	if (refNum == sysInvalidRefNum)
		return string();

	CEnableFullAccess	munge;	// Remove blocks on memory access.

	/*
		The System Library Table (sysLibTableP) is an array of
		sysLibTableEntries entries.  Each entry has the following
		format:

			Ptr*		dispatchTblP;	// pointer to library dispatch table
			void*		globalsP;		// Library globals
			LocalID 	dbID;			// database id of the library
			VoidPtr 	codeRscH;		// library code resource handle for RAM-based libraries

		The latter two fields are present only in Palm OS 2.0 and
		later.	So our first steps are to (a) get the pointer to 
		the array, (b) make sure that the index into the array (the
		refNum passed as the first parameter to all library calls)
		is within range, (c) get a pointer to the right entry,
		taking into account the Palm OS version, and (d) getting the
		dispatchTblP field.

		The "library dispatch table" is an array of 16-bit offsets.  The
		values are all relative to the beginning of the table (dispatchTblP).
		The first entry in the array corresponds to the library name.  All
		subsequent entries are offsets to the various library functions,
		starting with the required four: sysLibTrapOpen, sysLibTrapClose,
		sysLibTrapSleep, and sysLibTrapWake.
	*/

	uae_u32 sysLibTableP		= LowMem_GetGlobal (sysLibTableP);
	Word	sysLibTableEntries	= LowMem_GetGlobal (sysLibTableEntries);

	if (sysLibTableP == UAE_NULL)
	{
		// !!! No library table!
		assert (false);
		return string();
	}

	if (refNum >= sysLibTableEntries)
	{
		// !!! RefNum out of range!
		assert (false);
		return string();
	}

	uaecptr libEntry;
	uaecptr dispatchTblP;

	if (Patches::OSMajorVersion () > 1)
	{
		libEntry		= sysLibTableP + refNum * sizeof (SysLibTblEntryType);
		dispatchTblP	= get_long (libEntry + offsetof (SysLibTblEntryType, dispatchTblP));
	}
	else
	{
		libEntry		= sysLibTableP + refNum * sizeof (SysLibTblEntryTypeV10);
		dispatchTblP	= get_long (libEntry + offsetof (SysLibTblEntryTypeV10, dispatchTblP));
	}

	// The first entry in the table is always the offset from the
	// start of the table to the library name.	Use this information
	// get the library name.

	uae_s16 	offset = get_word (dispatchTblP + LibTrapIndex (sysLibTrapName) * 2);
	uaecptr 	libNameP = dispatchTblP + offset;

	char		libName[256];
	uae_strcpy (libName, libNameP);

	return string (libName);
}


/***********************************************************************
 *
 * FUNCTION:	GetFunctionAddress
 *
 * DESCRIPTION: Determines the address of a system function,
 *
 * PARAMETERS:	trapWord - the dispatch number used to invoke the
 *					function(the number after the TRAP $F opocde)
 *
 *				extra - an optional extra parameter.  This parameter can
 *					be something like a library reference number, or a
 *					register to be used in a sub-dispatcher.
 *
 *				digDeep - a boolean saying how hard we should look for
 *					the function address.  If false, we just return the
 *					address generated by the ROM TrapDispatcher
 *					function.  Otherwise, for certain special ROM
 *					routines (like the Floating Point Manager entry
 *					point), we look deeper.
 *
 * RETURNED:	The function's address, or UAE_NULL if it could not
 *				be determined.
 *
 ***********************************************************************/

uaecptr GetFunctionAddress (uae_u16 trapWord, uae_u32 extra, Bool digDeep)
{
	// Ensure that it's in Canonical format.

	trapWord = SysTrapIndex (trapWord) | sysTrapBase;

	// If it's a regular system function (0xA000-0xA7FFF), handle it.

	if (IsSystemTrap (trapWord))
		return GetSysFunctionAddress (trapWord, extra, digDeep);

	// Assume it's a library function call.

	return GetLibFunctionAddress (trapWord, (Word) extra, digDeep);
}


/***********************************************************************
 *
 * FUNCTION:	GetLibFunctionAddress
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	The function's address, or UAE_NULL if it could not
 *				be determined.
 *
 ***********************************************************************/

uaecptr GetLibFunctionAddress (uae_u16 trapWord, Word refNum, Bool digDeep)
{
	if (refNum == sysInvalidRefNum)
		return UAE_NULL;

	CEnableFullAccess	munge;	// Remove blocks on memory access.

	/*
		The System Library Table (sysLibTableP) is an array of
		sysLibTableEntries entries.  Each entry has the following
		format:

			Ptr*		dispatchTblP;	// pointer to library dispatch table
			void*		globalsP;		// Library globals
			LocalID 	dbID;			// database id of the library
			VoidPtr 	codeRscH;		// library code resource handle for RAM-based libraries

		The latter two fields are present only in Palm OS 2.0 and
		later.	So our first steps are to (a) get the pointer to 
		the array, (b) make sure that the index into the array (the
		refNum passed as the first parameter to all library calls)
		is within range, (c) get a pointer to the right entry,
		taking into account the Palm OS version, and (d) getting the
		dispatchTblP field.

		The "library dispatch table" is an array of 16-bit offsets.  The
		values are all relative to the beginning of the table (dispatchTblP).
		The first entry in the array corresponds to the library name.  All
		subsequent entries are offsets to the various library functions,
		starting with the required four: sysLibTrapOpen, sysLibTrapClose,
		sysLibTrapSleep, and sysLibTrapWake.
	*/

	uae_u32 sysLibTableP		= LowMem_GetGlobal (sysLibTableP);
	Word	sysLibTableEntries	= LowMem_GetGlobal (sysLibTableEntries);

	if (sysLibTableP == UAE_NULL)
	{
		// !!! No library table!
		assert (false);
		return UAE_NULL;
	}

	if (refNum >= sysLibTableEntries)
	{
		// See comments in HtalLibHeadpatch::HtalLibSendReply.
		if (refNum == kMagicRefNum)
		{
			return 1;
		}

		// !!! RefNum out of range!
		assert (false);
		return UAE_NULL;
	}

	uaecptr libEntry;
	uaecptr dispatchTblP;

	if (Patches::OSMajorVersion () > 1)
	{
		libEntry		= sysLibTableP + refNum * sizeof (SysLibTblEntryType);
		dispatchTblP	= get_long (libEntry + offsetof (SysLibTblEntryType, dispatchTblP));
	}
	else
	{
		libEntry		= sysLibTableP + refNum * sizeof (SysLibTblEntryTypeV10);
		dispatchTblP	= get_long (libEntry + offsetof (SysLibTblEntryTypeV10, dispatchTblP));
	}

	// Validate the dispatch number.  See if the library is one that
	// we know about.  If so, compare the dispatch number against
	// the list of known dispatch number limits.
	
	// The first entry in the table is always the offset from the
	// start of the table to the library name.	Use this information
	// get the library name.

	uae_s16 	offset = get_word (dispatchTblP + LibTrapIndex (sysLibTrapName) * 2);
	uaecptr 	libNameP = dispatchTblP + offset;
	char		libName[256];
	uae_strcpy (libName, libNameP);

	// Iterate over our list of Palm OS-supplied libraries to see
	// if this is one of them.	If so, the library is known to follow
	// a format that allows us to determine the size of the table.

	const char**	knownLibNameP = gPalmOSLibraries;

	while (*knownLibNameP)
	{
		if (strcmp (*knownLibNameP, libName) != 0)
		{
			knownLibNameP++;
			continue;
		}

		// OK, it's one of our libraries.  Now get the *second*
		// entry in the table.	This is the offset from the start
		// of the table to the entry point for the standard
		// "library open" function.  All Palm OS libraries follow
		// the convention where this offset points to a JMP
		// instruction immediately following the function table.
		// We can use this information to calculate the size of
		// the table.

		offset = get_word (dispatchTblP + LibTrapIndex (sysLibTrapOpen) * 2);

		if (LibTrapIndex (trapWord) < (offset / 2))
		{
			// It's a known library, and the offset is in range,
			// so break out of here so that we can get one with
			// actually calling the function.
			break;
		}

		// It's a known library, but the offset is out of range,
		// so report a problem.

		gLibErrorBase = knownLibNameP - gPalmOSLibraries;

		return UAE_NULL;
	}

	// Either we didn't know about that library, or we did and the
	// version test passed.  So go ahead and jump.

	offset = get_word (dispatchTblP + LibTrapIndex (trapWord) * 2);
	uaecptr result = dispatchTblP + offset;

	if (digDeep && get_word (result) == kOpcode_JMPREL)
	{
		result += 2;
		result += (uae_s16) get_word (result);
	}

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	GetSysFunctionAddress
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	The function's address, or UAE_NULL if it could not
 *				be determined.
 *
 ***********************************************************************/

uaecptr GetSysFunctionAddress (uae_u16 trapWord, uae_u32 extra, Bool digDeep)
{
	uaecptr unimplementedAddress	= LowMem::GetTrapAddress (sysTrapSysUnimplemented);
	uaecptr result					= LowMem::GetTrapAddress (trapWord);

	if (result == UAE_NULL ||
		((result == unimplementedAddress) && (trapWord != sysTrapSysUnimplemented)))
	{
		// sysTrapHostControl is always implemented, so make sure we
		// don't return a NULL address.

		if (SysTrapIndex (trapWord) == SysTrapIndex (sysTrapHostControl))
		{
			return 0x12345678;
		}

		return UAE_NULL;
	}

	if (digDeep)
	{
		uaecptr result2 = UAE_NULL;

		if (trapWord == sysTrapIntlDispatch 	||
			trapWord == sysTrapOmDispatch		||
			trapWord == sysTrapTsmDispatch		||
			trapWord == sysTrapFlpDispatch 		||
			trapWord == sysTrapSerialDispatch)
		{
			result2 = GetStdDispatchAddress (result, extra);
		}

		else if (trapWord == sysTrapFlpEmDispatch)
		{
			result2 = GetFlpEmDispatchAddress (result, extra);
		}

		if (result2)
			result = result2;
	}

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	GetStdDispatchAddress
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	The function's address, or UAE_NULL if it could not
 *				be determined.
 *
 ***********************************************************************/

uaecptr GetStdDispatchAddress (uaecptr entryPt, uae_u32 regD2)
{
	CEnableFullAccess	munge;	// Remove blocks on memory access.

	/*
		The standard dispatch routine code looks like this...

			+$0000	10D5D7FE  *CMP.W	 #$0018,D2		| B47C 0018
			+$0004	10D5D802   BHI.S	 @error 		| 626C
			+$0006	10D5D804   ADD.W	 D2,D2			| D442
			+$0008	10D5D806   ADD.W	 D2,D2			| D442
			+$000A	10D5D808   JMP		 *+$0004(D2.W)	| 4EFB 2002

			+$000E	10D5D80C   JMP		 function1		| 4EFA F732
			+$0012	10D5D810   JMP		 function2		| 4EFA 0AB8
			...
	*/

	// check for expected opcodes

	if (get_word (entryPt + 0x00) != kOpcode_CMPW ||
		get_word (entryPt + 0x06) != kOpcode_ADDW ||
		get_word (entryPt + 0x08) != kOpcode_ADDW ||
		get_long (entryPt + 0x0A) != kOpcode_JMPPC)
	{
		return UAE_NULL;
	}

	// load the max selector value, and compare to D2.W

	uae_u16 maxSelector = get_word (entryPt + 2);

	if ((regD2 & 0x0000FFFF) > maxSelector)
	{
		return UAE_NULL;
	}

	// point to appropriate JMP xxx instruction.

	uaecptr result = entryPt + 14 + (regD2 * 4);

	if (get_word (result) != kOpcode_JMPREL)
	{
		return UAE_NULL;
	}

	result += 2;
	result += (uae_s16) get_word (result);

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	GetFlpEmDispatchAddress
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	The function's address, or UAE_NULL if it could not
 *				be determined.
 *
 ***********************************************************************/

uaecptr GetFlpEmDispatchAddress (uaecptr entryPt, uae_u32 regD2)
{
	CEnableFullAccess	munge;	// Remove blocks on memory access.

	/*
		The FlpEm dispatch routine code looks like this...

			+$0000	10D5D804   ADD.W	 D2,D2				| D442
			+$0002	10D5D806   ADD.W	 D2,D2				| D442
			+$0004	10D5D808   JMP		 *+$0004(D2.W)		| 4EFB 2002

			+$0008	10D5D80C   JMP		 _fp_round			| 4EFA F732
			+$000A	10D5D810   JMP		 _fp_get_fpscr		| 4EFA 0AB8
			...
	*/

	// check for expected opcodes

	if (get_word (entryPt + 0x00) != kOpcode_ADDW ||
		get_word (entryPt + 0x02) != kOpcode_ADDW ||
		get_long (entryPt + 0x04) != kOpcode_JMPPC)
	{
		return UAE_NULL;
	}

	// point to appropriate JMP xxx instruction.

	uaecptr result = entryPt + 8 + (regD2 * 4);

	if (get_word (result) != kOpcode_JMPREL)
	{
		return UAE_NULL;
	}

	result += 2;
	result += (uae_s16) get_word (result);

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	GetTrapName
 *
 * DESCRIPTION: Returns a pointer to a string appropriate for
 *				identifying a function.  If the function has a Macsbug
 *				name, that name is returned.  If the function is a
 *				known Palm OS function, that name is looked up in a
 *				string resource list.  Otherwise, a default "unknown"
 *				string is returned.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

char* GetTrapName (const SystemCallContext& context, Bool digDeep)
{
	return GetTrapName (context.fTrapWord, context.fExtra, digDeep);
}


char* GetTrapName (uae_u16 trapWord, uae_u32 extra, Bool digDeep)
{
	static char name[sysPktMaxNameLen];

	::FindTrapName (trapWord, name, extra, digDeep);

	if (strlen (name) == 0)
	{
		strcpy (name, ">>> Unknown function name <<<");

		if (IsSystemTrap (trapWord))
		{
			string	knownName = Platform::GetString (kStr_SysTrapBase + SysTrapIndex (trapWord));
			if (knownName[0] != '<')
			{
				strcpy (name, knownName.c_str ());
			}
		}
	}

	return name;
}


/***********************************************************************
 *
 * FUNCTION:	FindTrapName
 *
 * DESCRIPTION: Finds the Macsbug name for the given function.	If the
 *				name cannot be found, an empty string is returned.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void FindTrapName (uae_u16 trapWord, char* nameP, uae_u32 extra, Bool digDeep)
{
	uaecptr addr = ::GetFunctionAddress (trapWord, extra, digDeep);

	if (addr)
		::FindFunctionName (addr, nameP);
	else
		nameP[0] = '\0';
}


/***********************************************************************
 *
 * FUNCTION:	FindFunctionName
 *
 * DESCRIPTION: Returns information about the function containing the
 *				the given memory address, including the function's
 *				start (the LINK instruction, usually), the function's
 *				end (just after the RTS, usually), and the function's
 *				name (that follows the function).
 *
 * PARAMETERS:	addr - address contained within the function.
 *
 *				nameP - storage for the returned function name.  Must
 *					be at least 32 characters.
 *
 *				startAddrP - storage for the returned function start.
 *
 *				endAddrP - storage for the returned function end.
 *
 *				nameCapacity - bytes of storage available at nameP
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/
 
// Llamagraphics, Inc:	Added nameCapacity argument so that callers
// can retrieve more than 31 character function names.	The default
// capacity is 32, so callers that don't provide the extra argument
// will get the same result as before.

void FindFunctionName (uaecptr addr, char* nameP, 
			uaecptr* startAddrP, uaecptr* endAddrP,
			long nameCapacity)
{
	// Get the start address only if requested.

	if (startAddrP)
		*startAddrP = ::FindFunctionStart (addr);

	// Get the end address if requested or if we need it to
	// get the Macsbug name.

	if (nameP || endAddrP)
	{
		uaecptr endAddr = ::FindFunctionEnd (addr);

		// Return the end address if requested.

		if (endAddrP)
			*endAddrP = endAddr;

		// Return the Macsbug name if requested.

		if (nameP)
			::CopyMacsbugName (endAddr, nameP, nameCapacity);
	}
}


/***********************************************************************
 *
 * FUNCTION:	FindFunctionStart
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

uaecptr FindFunctionStart (uaecptr addr)
{
	long	maxLength = 0x02000;

	while (--maxLength)
	{
		addr -= 2;

		// Make sure the address is valid
		if (!valid_address (addr, 2))
		{
			return UAE_NULL;
		}

		if (EndOfFunctionSequence (addr))
		{
			addr += 2;	// skip past the final RTS (or whatever).

			// Skip over the Macsbug name.
			Bool	isFixed;
			uae_u8	len = MacsbugNameLength (addr, &isFixed);

			addr += len;

			// Make sure the final address is word-aligned.
			if ((addr & 1) != 0)
				addr += 1;

			// If in variable format, the name is followed by
			// a word indicating the size of a chunk of "local data".
			if (!isFixed)
			{
				len = get_word (addr);
				addr += 2 + len;
			}

			return addr;
		}
	}

	return UAE_NULL;
}


/***********************************************************************
 *
 * FUNCTION:	FindFunctionEnd
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

uaecptr FindFunctionEnd (uaecptr addr)
{
	long	maxLength = 0x02000;

	while (--maxLength)
	{
		// Make sure the address is valid
		if (!valid_address (addr, 2))
		{
			return UAE_NULL;
		}

		if (EndOfFunctionSequence (addr))
		{
			return addr + 2;
		}

		addr += 2;
	}

	return UAE_NULL;
}


/***********************************************************************
 *
 * FUNCTION:	CopyMacsbugName
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

// Llamagraphics, Inc:	Added nameCapacity argument so that callers
// can retrieve more than 31 character function names.	The default
// capacity is 32, so callers that don't provide the extra argument
// will get the same result as before.

void CopyMacsbugName (uaecptr eof, char* name, long nameCapacity)
{
	// Macsbug names follow the end of the function.

	// Get the length of the macsbug symbol.

	uae_u8	length = MacsbugNameLength (eof, NULL);

	if (length > nameCapacity - 1)
		length = nameCapacity - 1;

	// copy the string

	char*	dest = name;
	while (length--)
	{
		uae_u8	ch = get_byte (eof++);

		if (!ValidMacsbugChar (ch))
		{
			// All bets are off.
			*name = 0;
			return;
		}

		*dest++ = (char) ch;
	}

	// Terminate the string.

	*dest = 0;
}


/***********************************************************************
 *
 * FUNCTION:	EndOfFunctionSequence
 *
 * DESCRIPTION: Test the given memory location to see if it contains
 *				a valid-looking end of function sequence.  A valid
 *				end of function sequence is an RTE, JMP (A0), RTD, or
 *				RTS (as long as it doesn't look like the sequnce
 *				CodeWarrior uses for 32-bit jumps.
 *
 * PARAMETERS:	addr - memory location to test
 *
 * RETURNED:	True if it looks like this is an end of function
 *				sequence.
 *
 ***********************************************************************/

Bool EndOfFunctionSequence (uaecptr addr)
{
	uae_u16 opcode = get_word (addr);

	// Look for the special 32-bit relative jumps which
	// CodeWarrior puts in. These are of the following form:
	//
	// +$0022  10CD3C2A   PEA		*+$0010 		; 10CD3C3A		| 487A 000E
	// +$0026  10CD3C2E   PEA		*+$0006 		; 10CD3C34		| 487A 0004
	// +$002A  10CD3C32   ADDI.L	#$00000CD2,(A7) ; '....'		| 0697 0000 0CD2
	// +$0030  10CD3C38   RTS										| 4E75

	if (opcode == kOpcode_RTS &&
		get_word (addr - 6) != kOpcode_ADD)
	{
		return true;
	}

	if (opcode == kOpcode_RTE ||
		opcode == kOpcode_JMP ||	// JMP (A0)
		opcode == kOpcode_RTD)
	{
		return true;
	}

	return false;
}


/***********************************************************************
 *
 * FUNCTION:	MascsbugNameLength
 *
 * DESCRIPTION: Returns the length of the Macsbug symbol starting at
 *				the given memory location.	Also bumps "addr" to skip
 *				over the length byte or bytes.
 *
 * PARAMETERS:	addr - start of the Macsbug symbol.  Should contain
 *					the symbols length byte or bytes.
 *
 *				isFixed - receives a boolean indicating if this name
 *					is in "fixed" or "variable" format.  If the former,
 *					the name is padded with spaces which need to be
 *					accounted for.	If the latter, spaces are not
 *					allowed at all.  Also, variable-length names are
 *					followed by a word containing the length of a
 *					chunk of "local data".
 *
 * RETURNED:	The symbol length (zero on error).
 *
 ***********************************************************************/

uae_u8 MacsbugNameLength (uaecptr& addr, Bool* isFixed)
{
	// The Macsbug name can be in one of three forms:
	//
	// Variable length: The first byte is in the range $80 to $9F and is a length in the
	//					range 0 to $1F. The high order bit must be set. A length of 0
	//					implies the second byte is the actual length in the range $01 
	//					thru $FF. The length byte(s) and name may be an odd number of
	//					bytes. However, the data after the name is word aligned.
	//
	// Fixed length 8:	The first byte is in the range $20 to $7F and is an ASCII character.
	//					The high order bit may be set but is not required to be.
	//
	// Fixed length 16: The first byte is in the range $20 to $7F and is an ASCII character.
	//					The high order bit may be set but is not required to be.
	//					The high order bit in the second byte is required to be set.
	//					This distinguishes the two types of fixed names.

	uae_u8	length = get_byte (addr);

	if (length < 0x20)
	{
		length = 0;

		if (isFixed)
			*isFixed = true;
	}
	else if (length >= 0x80 && length < 0xA0)
	{
		++addr;

		length &= 0x7F;

		if (length == 0)
			length = get_byte (addr++);

		if (isFixed)
			*isFixed = false;
	}
	else
	{
		length &= 0x7F;

		if (get_byte (addr + 1) < 0x80)
			length = 8;
		else
			length = 16;

		if (isFixed)
			*isFixed = true;
	}

	return length;
}


/***********************************************************************
 *
 * FUNCTION:	ValidMacsbugChar
 *
 * DESCRIPTION: Returns whether or not the given character is a
 *				character in a valid Macsbug symbol.  Valid characters
 *				are: [a-zA-Z0-9_%.]
 *
 * PARAMETERS:	ch - the character to test
 *
 * RETURNED:	True if the character is valid.
 *
 ***********************************************************************/

Bool ValidMacsbugChar (uae_u8 ch)
{
	static Bool initialized = false;
	static Bool validChar[256];

	if (!initialized)
	{
		initialized = true;

		memset (validChar, false, sizeof (validChar));

		validChar ['_'] = true;
		validChar ['%'] = true;
		validChar ['.'] = true;

		uae_u8	ii;

		for (ii = 'a'; ii <= 'z'; ++ii)
			validChar [ii] = true;

		for (ii = 'A'; ii <= 'Z'; ++ii)
			validChar [ii] = true;

		for (ii = '0'; ii <= '9'; ++ii)
			validChar [ii] = true;
	}

	return validChar [ch];
}


/***********************************************************************
 *
 * FUNCTION:	GetSystemCallContext
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool GetSystemCallContext (uaecptr pc, SystemCallContext& context)
{
	context.fPC = pc;

	// Determine how the system function is being called.  There are two ways:
	//
	//		* Via SYS_TRAP macro:
	//
	//			TRAP	$F
	//			DC.W	$Axxx
	//
	//		* Via SYS_TRAP_FAST macro:
	//
	//			MOVE.L	struct(LowMemType.fixed.globals.sysDispatchTableP), A1
	//			MOVE.L	((trapNum-sysTrapBase)*4)(A1), A1
	//			JSR 	(A1)	; opcode == 0x4e91
	//
	// The PC is current pointing to either the TRAP $F or the JSR (A1),
	// so we can look at the opcode to determine how we got here.

	uae_u8* realMem 	= get_real_address (pc);
	uae_u16 opcode		= do_get_mem_word (realMem);

	context.fViaTrap 	= opcode == (m68kTrapInstr + sysDispatchTrapNum);
	context.fViaJsrA1	= opcode == (0x4e91);

	
	if (context.fViaTrap)
	{
		// Not all development systems generate the correct dispatch
		// numbers; some leave off the preceding "A".  Make sure it's
		// set so that we can recognize it as a trap dispatch number.
		// (This code is here specifically so that the profiling routines
		//	will work, which check for trap numbers masquerading as function
		//	addresses by checking to see if they are in the sysTrapBase range.)

		context.fTrapWord	= get_word (pc + 2) | sysTrapBase;
		context.fNextPC 	= pc + 4;
	}
	else if (context.fViaJsrA1)
	{
		context.fTrapWord	= (get_word (pc - 2) / 4) | sysTrapBase;
		context.fNextPC 	= pc + 2;
	}
	else
	{
		assert (false);
		return false;
	}

	if (::IsSystemTrap (context.fTrapWord))
	{
		context.fTrapIndex	= SysTrapIndex (context.fTrapWord);
		context.fExtra		= m68k_dreg (regs, 2);
	}
	else
	{
		context.fTrapIndex	= LibTrapIndex (context.fTrapWord);
		context.fExtra		= get_word (m68k_areg (regs, 7));
	}

	assert ((context.fTrapWord >= sysTrapBase) && (context.fTrapWord < sysTrapBase + 0x1000));

	context.fDestPC = ::GetFunctionAddress (context.fTrapWord, context.fExtra, true);

	return true;
}


/***********************************************************************
 *
 * FUNCTION:	GetHostTime
 *
 * DESCRIPTION: Returns the current time in hours, minutes, and seconds.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void GetHostTime (long* hour, long* min, long* sec)
{
	time_t t;
	struct tm tm;
	
	time (&t);
	tm = *localtime (&t);
	
	*hour = tm.tm_hour; 	// 0...23
	*min =	tm.tm_min;		// 0...59
	*sec =	tm.tm_sec;		// 0...59
}


/***********************************************************************
 *
 * FUNCTION:	GetHostDate
 *
 * DESCRIPTION: Returns years since 1900, month as 0-11, and day as 1-31
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void GetHostDate (long* year, long* month, long* day)
{
	time_t t;
	struct tm tm;
	
	time (&t);
	tm = *localtime (&t);
	
	*year =  tm.tm_year + 1900; 	// 1904...2040
	*month = tm.tm_mon + 1; 		// 1...12
	*day =	 tm.tm_mday;			// 1...31
}


/***********************************************************************
 *
 * FUNCTION:	LaunchCmdToString
 *
 * DESCRIPTION: Convert the given launch command (the command that's
 *				passed to PilotMain in a Palm OS application) into a
 *				text form suitable for displaying to a user.
 *
 * PARAMETERS:	cmd - the launch command (from SystemMgr.h)
 *
 * RETURNED:	A pointer to a string representing the command.  If
 *				the command is unrecognized (probably because this
 *				function is out-of-date and needs to be updated for
 *				newly-added commands), a default string containing
 *				the command number is returned.
 *
 ***********************************************************************/

const char* LaunchCmdToString (Word cmd)
{
#undef DOCASE
#define DOCASE(name)	\
	case name:			\
		return #name;

	switch (cmd)
	{
		DOCASE (sysAppLaunchCmdNormalLaunch)
		DOCASE (sysAppLaunchCmdFind)
		DOCASE (sysAppLaunchCmdGoTo)
		DOCASE (sysAppLaunchCmdSyncNotify)
		DOCASE (sysAppLaunchCmdTimeChange)
		DOCASE (sysAppLaunchCmdSystemReset)
		DOCASE (sysAppLaunchCmdAlarmTriggered)
		DOCASE (sysAppLaunchCmdDisplayAlarm)
		DOCASE (sysAppLaunchCmdCountryChange)
		DOCASE (sysAppLaunchCmdSyncRequestLocal)
		DOCASE (sysAppLaunchCmdSaveData)
		DOCASE (sysAppLaunchCmdInitDatabase)
		DOCASE (sysAppLaunchCmdSyncCallApplicationV10)
		DOCASE (sysAppLaunchCmdPanelCalledFromApp)
		DOCASE (sysAppLaunchCmdReturnFromPanel)
		DOCASE (sysAppLaunchCmdLookup)
		DOCASE (sysAppLaunchCmdSystemLock)
		DOCASE (sysAppLaunchCmdSyncRequestRemote)
		DOCASE (sysAppLaunchCmdHandleSyncCallApp)
		DOCASE (sysAppLaunchCmdAddRecord)
		DOCASE (sysSvcLaunchCmdSetServiceID)
		DOCASE (sysSvcLaunchCmdGetServiceID)
		DOCASE (sysSvcLaunchCmdGetServiceList)
		DOCASE (sysSvcLaunchCmdGetServiceInfo)
		DOCASE (sysAppLaunchCmdFailedAppNotify)
		DOCASE (sysAppLaunchCmdEventHook)
		DOCASE (sysAppLaunchCmdExgReceiveData)
		DOCASE (sysAppLaunchCmdExgAskUser)
		DOCASE (sysDialLaunchCmdDial)
		DOCASE (sysDialLaunchCmdHangUp)
		DOCASE (sysSvcLaunchCmdGetQuickEditLabel)
		DOCASE (sysAppLaunchCmdURLParams)
		DOCASE (sysAppLaunchCmdNotify)
		DOCASE (sysAppLaunchCmdOpenDB)
		DOCASE (sysAppLaunchCmdAntennaUp)
		DOCASE (sysAppLaunchCmdGoToURL)
	}

	static char 	buffer[20];
	sprintf (buffer, "#%ld", (long) cmd);
	return buffer;
}


/***********************************************************************
 *
 * FUNCTION:    GetDeviceList
 *
 * DESCRIPTION: Return the list of items that should appear in the
 *				various listboxes and menus that allow the user to
 *				select a device.  The list is returned as a collection
 *				of pairs.  The first element of the pair is the
 *				DeviceType, and the second element of the pair is
 *				the name of the device in text form, suitable for
 *				displaying to the user.
 *
 * PARAMETERS:  deviceList - reference to the collection to receive
 *					the results.
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void GetDeviceTextList (DeviceTextList& deviceList)
{
// -*- NEW DEVICE -*-

	deviceList.push_back (make_pair (kDevicePilot1000, string ("Pilot")));
	deviceList.push_back (make_pair (kDevicePalmPilotPersonal, string ("PalmPilot")));
	deviceList.push_back (make_pair (kDevicePalmIII, string ("Palm III")));
	deviceList.push_back (make_pair (kDevicePalmIIIx, string ("Palm IIIx")));
	deviceList.push_back (make_pair (kDevicePalmV, string ("Palm V")));
	deviceList.push_back (make_pair (kDevicePalmVII, string ("Palm VII")));
	deviceList.push_back (make_pair (kDevicePalmVIIEZ, string ("Palm VII EZ")));
	deviceList.push_back (make_pair (kDeviceAustin, string ("Color Device")));
}


/***********************************************************************
 *
 * FUNCTION:    GetMemoryList
 *
 * DESCRIPTION: Return the list of items that should appear in the
 *				various listboxes and menus that allow the user to
 *				select a memory size.  The list is returned as a
 *				collection of pairs.  The first element of the pair is
 *				the memory size (in K), and the second element of the
 *				pair is the same thing in textform, suitable for
 *				displaying to the user.
 *
 * PARAMETERS:  memoryList - reference to the collection to receive
 *					the results.
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void GetMemoryTextList (MemoryTextList& memoryList)
{
	memoryList.push_back (make_pair (RAMSizeType (128), string ("128K")));
	memoryList.push_back (make_pair (RAMSizeType (256), string ("256K")));
	memoryList.push_back (make_pair (RAMSizeType (512), string ("512K")));
	memoryList.push_back (make_pair (RAMSizeType (1024), string ("1024K")));
	memoryList.push_back (make_pair (RAMSizeType (2048), string ("2048K")));
	memoryList.push_back (make_pair (RAMSizeType (4096), string ("4096K")));
	memoryList.push_back (make_pair (RAMSizeType (8192), string ("8192K")));
}
