/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#ifndef _CHUNKFILE_H_
#define _CHUNKFILE_H_

#include "PPStream.h"


class StreamHandle;
class Chunk;


/*
	ChunkFile is a class for managing files containing tagged
	chunks of data.  It contain member functions for writing
	tagged data to a file, finding that data later, and reading
	the data back into user-specified buffers.  There are also
	member functions for writing common data types (numbers and
	strings) in a platform-independent fashion.
	
	There is no facility for *updating* chunk files.  That is,
	there is no way to modify the contents of a chunk that already
	exists in a a chunk file (regardless of whether or not the
	length of the chunk changes).
	
	The format of a chunk file is simple.  It's essentially a
	variable-length array of:
	
		tag: 4 bytes
		size: 4 bytes
		data: "size" bytes of data

	"tag", "size", and any integer values written with WriteInt
	are stored in Big Endian format.  Strings are stored without
	the NULL terminator.  Data is packed as tightly as possible;
	there's no word or longword alignment.
 */

class ChunkFile
{
	public:
								ChunkFile		(StreamHandle& s);
								~ChunkFile		(void);

		typedef uae_u32	Tag;

//		static const long		kChunkNotFound = -1;	// VC++ is a bit medieval here...
		enum { kChunkNotFound = -1 };
		long					FindChunk		(Tag tag);	// Returns chunk size

		Bool					ReadChunk		(Tag tag, Chunk&);
		void					ReadChunk		(uae_u32 size, void* data);
		Bool					ReadInt			(Tag tag, uae_u8&);
		Bool					ReadInt			(Tag tag, uae_s8&);
		Bool					ReadInt			(Tag tag, uae_u16&);
		Bool					ReadInt			(Tag tag, uae_s16&);
		Bool					ReadInt			(Tag tag, uae_u32&);
		Bool					ReadInt			(Tag tag, uae_s32&);
		Bool					ReadString		(Tag tag, char*);
		Bool					ReadString		(Tag tag, string&);

		void					WriteChunk		(Tag tag, const Chunk&);
		void					WriteChunk		(Tag tag, uae_u32 size, const void* data);
		void					WriteInt		(Tag tag, uae_u8);
		void					WriteInt		(Tag tag, uae_s8);
		void					WriteInt		(Tag tag, uae_u16);
		void					WriteInt		(Tag tag, uae_s16);
		void					WriteInt		(Tag tag, uae_u32);
		void					WriteInt		(Tag tag, uae_s32);
		void					WriteString		(Tag tag, const char*);
		void					WriteString		(Tag tag, const string&);

		StreamHandle&			GetStream		(void) const;

	private:
		StreamHandle&			fStream;
};


class Chunk
{
	public:
								Chunk			(void);
								Chunk			(long inLength);
								~Chunk			(void);

		void*					GetPointer		(void) const;
		long					GetLength		(void) const;
		void					SetLength		(long inLength);

	private:
		void*					fPtr;
		long					fUsedSize;
		long					fAllocatedSize;
};

class ChunkStream : public PPStream
{
	public:
								ChunkStream		(Chunk&);
		virtual					~ChunkStream	(void);

		virtual void			SetLength		(uae_s32 inLength);

		virtual ErrCode			PutBytes		(const void	*inBuffer,
												 uae_s32	ioByteCount);
		virtual ErrCode			GetBytes		(void		*outBuffer,
												 uae_s32	ioByteCount);

	private:
		Chunk&					fChunk;
};


#endif	// _CHUNKFILE_H_
