/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "Bank_Dummy.h"

#include "Bank_DRAM.h"			// DRAMBank::ValidAddress
#include "Bank_SRAM.h"			// SRAMBank::GetMemoryStart
#include "CPU_REG.h"			// Software::BusError
#include "PalmHeap.h"			// PalmHeap::ChunkInfo
#include "RAM_ROM.h"			// gValidateMemory


static Bool PrvLooksLikeA5Access(uaecptr iAddress, long size, Bool forRead);


// ===========================================================================
//		 Dummy Bank Accessors
// ===========================================================================
// Dummy banks are non-existent blocks of memory.  Dummy bank accessors do
// not do anything.

static AddressBank	gAddressBank =
{
	DummyBank::GetLong,
	DummyBank::GetWord,
	DummyBank::GetByte,
	DummyBank::SetLong,
	DummyBank::SetWord,
	DummyBank::SetByte,
	DummyBank::GetRealAddress,
	DummyBank::ValidAddress,
	DummyBank::GetMetaAddress,
	DummyBank::AddOpcodeCycles
};

struct MapRange
{
	uaecptr	addr;
	uae_u32	size;
};

// Let us map in at least eight segments of emulator RAM into the "virtual
// RAM" of the emulated ROM.  Each time we call a ROM function via the
// ATrap class, two entries are used up.  By setting kNumPhysicalBanks,
// we allow for four of those to go on at the same time.
//
// I'm now bumping this up to something way bigger.  The idea behind keeping
// the number small was to catch places where ranges were allocated but not
// de-allocated.  There doesn't seem to be much problem with that, so this
// debugging aid is not really needed any more.  On the other hand, we now
// map in strings from getenv.  These mappings stay around indefinitely, so
// we need to allow for such unbounded growth.

static const int	kNumPhysicalBanks	= 100;
static MapRange		gPhysicalBanks [kNumPhysicalBanks];

inline Bool HackForHwrGetRAMSize (uaecptr address)
{
//	if ((address & 0xFF000000) == SRAMBank::GetMemoryStart ())
	if (address == SRAMBank::GetMemoryStart () + gRAMBank_Size)
		return true;

	return false;
}


/***********************************************************************
 *
 * FUNCTION:	DummyBank::Initialize
 *
 * DESCRIPTION:	Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void DummyBank::Initialize (void)
{
	// Clear the memory banks.

	Memory::InitializeBanks (gAddressBank, 0, 0xFFFF);

	memset (gPhysicalBanks, 0, sizeof (gPhysicalBanks));
}


/***********************************************************************
 *
 * FUNCTION:	DummyBank::Reset
 *
 * DESCRIPTION:	Standard reset function.  Sets the sub-system to a
 *				default state.  This occurs not only on a Reset (as
 *				from the menu item), but also when the sub-system
 *				is first initialized (Reset is called after Initialize)
 *				as well as when the system is re-loaded from an
 *				insufficient session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void DummyBank::Reset (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	DummyBank::Save
 *
 * DESCRIPTION:	Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void DummyBank::Save (SessionFile&)
{
	// The only state we have to save is in gPhysicalBanks.  We shouldn't
	// have to actually save this because there shouldn't be anything in
	// there at the moment when we save files.  We map in memory ranges
	// at the following times:
	//
	//		* Calling system functions
	//		* Loading .prc, etc., files
	//		* Mapping in environment strings
	//
	// The first two should not be in "effect" at the time we save a file.
	// The last one we'll leave up to the HostControl system to re-establish.
}


/***********************************************************************
 *
 * FUNCTION:	DummyBank::Load
 *
 * DESCRIPTION:	Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void DummyBank::Load (SessionFile&)
{
}


/***********************************************************************
 *
 * FUNCTION:	DummyBank::Dispose
 *
 * DESCRIPTION:	Standard dispose function.  Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void DummyBank::Dispose (void)
{
}


// ---------------------------------------------------------------------------
//		 DummyBank::GetLong
// ---------------------------------------------------------------------------

uae_u32 DummyBank::GetLong (uaecptr iAddress)
{
	// Hack to keep HwrGetRAMSize working: it runs off
	// the end of RAM while testing it.

	if (HackForHwrGetRAMSize (iAddress))
		return 0;

#if (VALIDATE_DUMMY_GET)
	if (gMemAccessFlags.fValidate_DummyGet && !ValidAddress (iAddress, sizeof (long)))
	{
		if (!PrvLooksLikeA5Access (iAddress, sizeof (long), true))
			InvalidAccess (iAddress, sizeof (long), true);
	}
#endif

#if HAS_PROFILING
	CYCLE_GETLONG (WAITSTATES_DUMMYBANK);
#endif

	if (!ValidAddress (iAddress, sizeof (long)))
	{
		return ~0;
	}

	uae_u8*	p = (uae_u8*) iAddress;
	return (((uae_u32) p[0]) << 24) | (((uae_u32) p[1]) << 16) | (((uae_u32) p[2]) << 8) | p[3];
}


// ---------------------------------------------------------------------------
//		 DummyBank::GetWord
// ---------------------------------------------------------------------------

uae_u32 DummyBank::GetWord (uaecptr iAddress)
{
	// Hack to keep HwrGetRAMSize working: it runs off
	// the end of RAM while testing it.

	if (HackForHwrGetRAMSize (iAddress))
		return 0;

#if (VALIDATE_DUMMY_GET)
	if (gMemAccessFlags.fValidate_DummyGet && !ValidAddress (iAddress, sizeof (short)))
	{
		if (!PrvLooksLikeA5Access (iAddress, sizeof (short), true))
			InvalidAccess (iAddress, sizeof (short), true);
	}
#endif

#if HAS_PROFILING
	CYCLE_GETWORD (WAITSTATES_DUMMYBANK);
#endif

	if (!ValidAddress (iAddress, sizeof (short)))
	{
		return ~0;
	}

	uae_u8*	p = (uae_u8*) iAddress;
	return (((uae_u32) p[0]) << 8) | p[1];
}


// ---------------------------------------------------------------------------
//		 DummyBank::GetByte
// ---------------------------------------------------------------------------

uae_u32 DummyBank::GetByte (uaecptr iAddress)
{
	// Hack to keep HwrGetRAMSize working: it runs off
	// the end of RAM while testing it.

	if (HackForHwrGetRAMSize (iAddress))
		return 0;

#if (VALIDATE_DUMMY_GET)
	if (gMemAccessFlags.fValidate_DummyGet && !ValidAddress (iAddress, sizeof (char)))
	{
		if (!PrvLooksLikeA5Access (iAddress, sizeof (short), true))
			InvalidAccess (iAddress, sizeof (char), true);
	}
#endif

#if HAS_PROFILING
	CYCLE_GETBYTE (WAITSTATES_DUMMYBANK);
#endif

	if (!ValidAddress (iAddress, sizeof (char)))
	{
		return ~0;
	}

	uae_u8*	p = (uae_u8*) iAddress;
	return p[0];
}


// ---------------------------------------------------------------------------
//		 DummyBank::SetLong
// ---------------------------------------------------------------------------

void DummyBank::SetLong (uaecptr iAddress, uae_u32 iLongValue)
{
	// Hack to keep HwrGetRAMSize working: it runs off
	// the end of RAM while testing it.

	if (HackForHwrGetRAMSize (iAddress))
		return;

#if (VALIDATE_DUMMY_SET)
	if (gMemAccessFlags.fValidate_DummySet && !ValidAddress (iAddress, sizeof (long)))
	{
		if (!PrvLooksLikeA5Access (iAddress, sizeof (long), false))
			InvalidAccess (iAddress, sizeof (long), false);
	}
#endif

#if HAS_PROFILING
	CYCLE_PUTLONG (WAITSTATES_DUMMYBANK);
#endif

	if (!ValidAddress (iAddress, sizeof (long)))
	{
		return;
	}

	uae_u8*	p = (uae_u8*) iAddress;
	p[0] = (uae_u8) (iLongValue >> 24);
	p[1] = (uae_u8) (iLongValue >> 16);
	p[2] = (uae_u8) (iLongValue >> 8);
	p[3] = (uae_u8) (iLongValue >> 0);
}


// ---------------------------------------------------------------------------
//		 DummyBank::SetWord
// ---------------------------------------------------------------------------

void DummyBank::SetWord (uaecptr iAddress, uae_u32 iWordValue)
{
	// Hack to keep HwrGetRAMSize working: it runs off
	// the end of RAM while testing it.

	if (HackForHwrGetRAMSize (iAddress))
		return;

#if (VALIDATE_DUMMY_SET)
	if (gMemAccessFlags.fValidate_DummySet && !ValidAddress (iAddress, sizeof (short)))
	{
		if (!PrvLooksLikeA5Access (iAddress, sizeof (short), false))
			InvalidAccess (iAddress, sizeof (short), false);
	}
#endif

#if HAS_PROFILING
	CYCLE_PUTWORD (WAITSTATES_DUMMYBANK);
#endif

	if (!ValidAddress (iAddress, sizeof (short)))
	{
		return;
	}

	uae_u8*	p = (uae_u8*) iAddress;
	p[0] = (uae_u8) (iWordValue >> 8);
	p[1] = (uae_u8) (iWordValue >> 0);
}


// ---------------------------------------------------------------------------
//		 DummyBank::SetByte
// ---------------------------------------------------------------------------

void DummyBank::SetByte (uaecptr iAddress, uae_u32 iByteValue)
{
	// Hack to keep HwrGetRAMSize working: it runs off
	// the end of RAM while testing it.

	if (HackForHwrGetRAMSize (iAddress))
		return;

#if (VALIDATE_DUMMY_SET)
	if (gMemAccessFlags.fValidate_DummySet && !ValidAddress (iAddress, sizeof (char)))
	{
		if (!PrvLooksLikeA5Access (iAddress, sizeof (char), false))
			InvalidAccess (iAddress, sizeof (char), false);
	}
#endif

#if HAS_PROFILING
	CYCLE_PUTBYTE (WAITSTATES_DUMMYBANK);
#endif

	if (!ValidAddress (iAddress, sizeof (char)))
	{
		return;
	}

	uae_u8*	p = (uae_u8*) iAddress;
	p[0] = (uae_u8) (iByteValue >> 0);
}


// ---------------------------------------------------------------------------
//		 DummyBank::ValidAddress
// ---------------------------------------------------------------------------

int DummyBank::ValidAddress (uaecptr iAddress, uae_u32 iSize)
{
	int	result = false;

	// Allow access to the physical range of memory that we've
	// mapped into Palm OS-emulated space.

	for (int ii = 0; ii < kNumPhysicalBanks; ++ii)
	{
		if (gPhysicalBanks[ii].addr)
		{
			uaecptr	start = gPhysicalBanks[ii].addr;
			uaecptr	end = start + gPhysicalBanks[ii].size;

			if ((iAddress >= start) && (iAddress + iSize <= end))
			{
				result = true;
				break;
			}
		}
	}

	assert (result);

	return result;
}


// ---------------------------------------------------------------------------
//		 DummyBank::GetRealAddress
// ---------------------------------------------------------------------------

uae_u8* DummyBank::GetRealAddress (uaecptr iAddress)
{
	return (uae_u8*) iAddress;
}


// ---------------------------------------------------------------------------
//		 DummyBank::GetMetaAddress
// ---------------------------------------------------------------------------

uae_u8* DummyBank::GetMetaAddress (uaecptr iAddress)
{
	UNUSED_PARAM(iAddress)

	static uae_u8	dummyBits[4] = {0, 0, 0, 0};

	return dummyBits;
}


// ---------------------------------------------------------------------------
//		 DummyBank::AddOpcodeCycles
// ---------------------------------------------------------------------------

void DummyBank::AddOpcodeCycles (void)
{
}


// ---------------------------------------------------------------------------
//		 DummyBank::InvalidAccess
// ---------------------------------------------------------------------------

void DummyBank::InvalidAccess (uaecptr iAddress, long size, Bool forRead)
{
	UNUSED_PARAM(iAddress)
	UNUSED_PARAM(size)
	UNUSED_PARAM(forRead)

	Software::BusError ();
}


// ---------------------------------------------------------------------------
//		 DummyBank::MapPhysicalMemory
// ---------------------------------------------------------------------------
// Maps a range of physical memory to appear at the same location of the
// emulated Palm OS's virtual memory.

void DummyBank::MapPhysicalMemory (const void* addr, uae_u32 size)
{
	// Start mapping from longword boundaries.  It appears that MWDebug
	// rounds down to those boundaries on occassion.

	uaecptr	roundedAddress = ((uaecptr) addr) & ~0x03;
	size += ((uaecptr) addr) - roundedAddress;

	
	// Make sure that the passed in memory range doesn't overlap with
	// memory ranges already in use.  The following loop gets the
	// memory handlers for the range of memore we're mapping in, and
	// makes sure that they are DummyBank memory handlers.

	uae_s32	startIndex = bankindex(roundedAddress);
	uae_s32	endIndex = bankindex(roundedAddress + size);

	for (uae_s32 ii = startIndex; ii <= endIndex; ++ii)
	{
		assert (mem_banks[ii] == &gAddressBank);
	}

	// We're cool...map in the range.

	int	jj;
	for (jj = 0; jj < kNumPhysicalBanks; ++jj)
	{
		// If we've already mapped this one in, just return.

		if (gPhysicalBanks[jj].addr == roundedAddress)
		{
			break;
		}

		// Found an empty record; use it.

		if (gPhysicalBanks[jj].addr == UAE_NULL)
		{
			gPhysicalBanks[jj].addr = roundedAddress;
			gPhysicalBanks[jj].size = size;
			break;
		}
	}

	if (jj == kNumPhysicalBanks)
	{
		assert (!"Too many physical memory mappings.");
		abort ();
	}
}


// ---------------------------------------------------------------------------
//		 DummyBank::UnmapPhysicalMemory
// ---------------------------------------------------------------------------
// Unmaps a range of physical memory from appearing at the same location of
// the emulated Palm OS's virtual memory.

void DummyBank::UnmapPhysicalMemory (const void* addr)
{
	// Start mapping from longword boundaries.  It appears that MWDebug
	// rounds down to those boundaries on occassion.

	uaecptr	roundedAddress = ((uaecptr) addr) & ~0x03;

	for (int ii = 0; ii < kNumPhysicalBanks; ++ii)
	{
		if (gPhysicalBanks[ii].addr == roundedAddress)
		{
			gPhysicalBanks[ii].addr = UAE_NULL;
			gPhysicalBanks[ii].size = 0;
		}
	}
}


// ---------------------------------------------------------------------------
//		 DummyBank::GetMappingInfo
// ---------------------------------------------------------------------------

void DummyBank::GetMappingInfo (const void* addr, void** start, uae_u32* len)
{
	for (int ii = 0; ii < kNumPhysicalBanks; ++ii)
	{
		if ((uaecptr) addr >= gPhysicalBanks[ii].addr &&
			(uaecptr) addr < gPhysicalBanks[ii].addr + gPhysicalBanks[ii].size)
		{
			*start = (void*) gPhysicalBanks[ii].addr;
			*len = gPhysicalBanks[ii].size;

			return;
		}
	}

	*start = NULL;
	*len = 0;
}


// ---------------------------------------------------------------------------
//		 PrvLooksLikeA5Access
// ---------------------------------------------------------------------------

Bool PrvLooksLikeA5Access(uaecptr iAddress, long size, Bool forRead)
{
	// The OS sets the high bit of the A5 register when calling PilotMain
	// with a launch code that doesn't allow global variable access.

	if ((iAddress & 0x80000000) != 0 &&
		(m68k_areg (regs, 5) & 0x80000000) != 0)
	{
		uaecptr	strippedAddress = iAddress & 0x7FFFFFFF;
		uaecptr	strippedA5 = m68k_areg (regs, 5) & 0x7FFFFFFF;

		// See if the stripped test address and the real A5 both reside in
		// the dynamic heap.

		if (DRAMBank::ValidAddress (strippedA5, 1) &&
			DRAMBank::ValidAddress (strippedAddress, size))
		{
			// Now see if they both point into the same chunk.

			uaecptr	heapHdr = PalmHeap::FindHeapHeader (strippedA5);

			PalmHeap::HeapInfo	heapInfo;
			PalmHeap::GetHeapInfo (heapHdr, &heapInfo);

			uaecptr	chunkHdr1 = PalmHeap::FindChunkHeader (strippedA5, heapInfo);
			uaecptr	chunkHdr2 = PalmHeap::FindChunkHeader (strippedAddress, heapInfo);

			PalmHeap::ChunkInfo	info1;
			PalmHeap::GetChunkInfo (chunkHdr1, heapInfo, &info1);

			PalmHeap::ChunkInfo	info2;
			PalmHeap::GetChunkInfo (chunkHdr2, heapInfo, &info2);

			if ((info1.chunkHdrStart != UAE_NULL) &&
				(info1.chunkHdrStart == info2.chunkHdrStart))
			{
				// All tests pass, so report an attempt to utilize A5
				// in a verbotten situation.

				int button = Errors::ReportA5Access (iAddress, size, forRead);
				Emulator::HandleDlgButton (button, m68k_getpc ());
				return true;	// (I don't think we ever get here...)
			}
		}
	}

	return false;
}
