/*
 *  File:        bmtable.c
 *  Purpose:     Implementation of the XForms object bmtable. 
 *  Author:      Alejandro Aguilar Sierra <asierra@servidor.unam.mx> 
 *  Created:     November 1995
 *  Description: A bitmap table uses a single bitmap to simulate a 2d array 
 *               of bitmap buttons. It can be used to build bitmap menus.
 *               
 *  Copyright (C) 1995,1996 Alejandro Aguilar Sierra 
 *
 *  You are free to use and modify this code under the terms of
 *  the GNU General Public Licence version 2 or later. 
 *  
 */ 

static char rcsid[] = "$Id: bmtable.C,v 1.3 1996/06/03 23:52:58 larsbj Exp $";

#include <stdlib.h>

#include "bmtable.h"

/*  XForms compatibility macros  */

#ifndef FL_INCLUDE_VERSION
#define BT_LINE fl_line
#else
#define BT_LINE fl_diagline
#endif

typedef struct   {   
   int nx, ny;   /* Dimensions of the table */
   int dx, dy;   /* Size of each item */ 
   int bx, by;   /* Bitmap's position */
   int bw, bh;   /* Bitmap dimensions */
   char* bdata;  /* Bitmap data */
   int maxi;     /* Number of items */
   int i;        /* Current position */
   int mousebut; /* mouse button pushed */  
   Pixmap pix;   /* Pixmap from data (temporal) */
} BMTABLE_SPEC;
                 

static int handle_bitmaptable(FL_OBJECT *ob, int event, FL_Coord mx, 
			      FL_Coord my, int key, void *xev);


FL_OBJECT *fl_create_bmtable(int type, FL_Coord x, FL_Coord y, 
				 FL_Coord w, FL_Coord h, const char *label)
{
   FL_OBJECT *ob;
   
   ob = fl_make_object(FL_BMTABLE, type, x, y, w, h, label, handle_bitmaptable);
   ob->boxtype = FL_BMTABLE_BOXTYPE;
   ob->spec = fl_calloc(1, sizeof(BMTABLE_SPEC));
   ((BMTABLE_SPEC *)ob->spec)->pix = 0;
   ((BMTABLE_SPEC *)ob->spec)->bdata= 0;
   ((BMTABLE_SPEC *)ob->spec)->mousebut= -1;
   return ob;
}


FL_OBJECT *fl_add_bmtable(int type, FL_Coord x, FL_Coord y, 
			      FL_Coord w, FL_Coord h, const char *label)
{
   FL_OBJECT *ob;
   
   ob = fl_create_bmtable(type, x, y, w,h, label);  
   fl_add_object(fl_current_form, ob); 
   
   return ob;
}


static void draw_bitmaptable(FL_OBJECT *ob)
{
   int i, j;
   FL_Coord mx, my;
   FL_Coord xx, yy, ww, hh;
   BMTABLE_SPEC *sp = (BMTABLE_SPEC *)ob->spec;
   GC gc = fl_state[fl_get_vclass()].gc[0];
   
   /* draw the bounding box first */
   fl_drw_box(ob->boxtype, ob->x, ob->y, ob->w, ob->h, ob->col1, ob->bw);
   
   /* draw the background bitmap */
   if (sp->bdata)  {
      if (!sp->pix) 
	 sp->pix = XCreatePixmapFromBitmapData(fl_display, fl_winget(), 
			sp->bdata, sp->bw, sp->bh,
		        fl_get_flcolor(ob->lcol), fl_get_flcolor(ob->col1),
		        DefaultDepth(fl_display, DefaultScreen(fl_display)));
	      
      if (sp->pix) { 
	 /* Adjust position */ 
	 if (sp->bx < FL_abs(ob->bw) + 1) {
	    xx = FL_abs(ob->bw) - sp->bx + 1;
	    mx = ob->x + FL_abs(ob->bw) + 1;
	 } else  {
	    xx =0;
	    mx = ob->x + sp->bx;
	 }
	 if (sp->by < FL_abs(ob->bw) + 1)  {	 
	    yy = FL_abs(ob->bw) - sp->by + 1;
	    my = ob->y + FL_abs(ob->bw) + 1;
	 } else   { 
	    yy =0;
	    my = ob->y + sp->by;
	 }                 
	 ww = (mx + sp->bw < ob->x + ob->w - FL_abs(ob->bw)) ? 
	   sp->bw: ob->x + ob->w - FL_abs(ob->bw) - mx;
	 hh = (my + sp->bh < ob->y + ob->h - FL_abs(ob->bw)) ?
	   sp->bh: ob->y + ob->h - FL_abs(ob->bw) - my; 
      
	 XCopyArea(fl_display, sp->pix, fl_winget(), gc, xx, yy, ww, hh, mx, my);
      }
   }
   
   /* draw the grid if type > FLAT */   
   if (ob->type > FL_BMTABLE_FLAT)  {
      mx = ob->x + ob->w; 
      my = ob->y + ob->h; 
      for (yy=ob->y; yy<=my; yy+=sp->dy) {
	 if (ob->boxtype!=FL_FLAT_BOX && (yy==ob->y || yy>my-sp->dy)) 
	     continue;
	 BT_LINE(ob->x, yy, ob->w, 1, FL_GRAY16);
	 BT_LINE(ob->x, yy+1, ob->w-2, 1, FL_GRAY75); 
      }	   
      for (xx=ob->x; xx<=mx; xx+=sp->dx)  {
	 if (ob->boxtype!=FL_FLAT_BOX && (xx==ob->x || xx>mx-sp->dx))
             continue; 
	 BT_LINE(xx, ob->y, 1, ob->h, FL_BLACK);
	 BT_LINE(xx+1, ob->y+1, 1, ob->h-2, FL_WHITE); 
      }	 
   }  
   
   /* Simulate a pushed button */
   if (ob->pushed && 0 <= sp->i && sp->i < sp->maxi)  {  
      i = sp->i % sp->nx;
      j = sp->i/sp->nx;
      ww = sp->dx-2*FL_abs(ob->bw);
      hh = sp->dy-2*FL_abs(ob->bw);
      xx = ob->x + sp->dx*i + FL_abs(ob->bw);
      yy = ob->y + sp->dy*j + FL_abs(ob->bw);
      fl_drw_frame(FL_DOWN_FRAME, xx, yy, ww, hh, ob->col1, ob->bw);
   }
}


static int handle_bitmaptable(FL_OBJECT *ob, int event, FL_Coord mx, 
			      FL_Coord my, int key, void *xev)
{
   int i, j;
   BMTABLE_SPEC *sp = (BMTABLE_SPEC *)ob->spec;
   
   switch (event)  {
    case FL_DRAW: 
      draw_bitmaptable(ob);
      break;
    case FL_MOUSE:
      if (!ob->belowmouse) {    /* This never happens. Why? */
	 sp->i = -1;
      	 fl_redraw_object(ob);
	 break;
      }
      i = (mx - ob->x)/sp->dx;  j = (my - ob->y)/sp->dy;
      if (i>=0 && i< sp->nx && j>=0 && j< sp->ny)   {
	 i += j*sp->nx;
	 if (i >= sp->maxi) i = -1;
	 if (sp->i !=  i)  {
	    sp->i = i;
	    fl_redraw_object(ob);
	 }
      }
      break;        
    case FL_PUSH:
      sp->mousebut = key;
      i = (mx - ob->x)/sp->dx + ((my - ob->y)/sp->dy)*sp->nx; 
      if (0 <= i && i < sp->maxi)  {
	 sp->i =  i;
	 fl_redraw_object(ob);
      } else
	sp->i =  -1; 
      break;
    case FL_RELEASE:    
      fl_redraw_object(ob);
      return 1;
    case FL_FREEMEM:
      if (sp->pix) 
	 XFreePixmap(fl_display, sp->pix);
      fl_free(((BMTABLE_SPEC*)ob->spec));      
      break;
   }
   return 0;
}


/*
 * The table has nx columns of dx width each and ny rows of dy height each. 
 * Initially the position of the firts item is supposed to be the same that
 * the object position (x, y), and the number of items is supposed to be
 * exactly nx*ny.
 * 
 * The user could change these later. See below.
 */ 
void fl_set_bmtable_data(FL_OBJECT *ob, int nx, int ny, int bw, int bh, 
			char *bdata)
{
   BMTABLE_SPEC *sp = (BMTABLE_SPEC *)ob->spec;
   sp->nx = nx;
   sp->ny = ny; 
   sp->bx = FL_abs(ob->bw);
   sp->by = FL_abs(ob->bw);
   sp->dx = ob->w/nx; 
   sp->dy = ob->h/ny;
   sp->i = -1;
   sp->maxi = sp->nx * sp->ny;
   sp->bw = bw;
   sp->bh = bh;
   sp->bdata = bdata;
}


/*
 *  This function works only for X11R6 or later
 */
void fl_set_bmtable_file(FL_OBJECT *ob, int nx, int ny, const char *filename)
{	
   int bw, bh, xh, yh;
   char *bdata;
   
#if XlibSpecificationRelease > 5 
   if(XReadBitmapFileData(filename, (unsigned int *) &bw, (unsigned int *) &bh,
        (unsigned char **) &bdata, &xh, &yh)==BitmapSuccess)
     fl_set_bmtable_data(ob, nx, ny, bw, bh, bdata);
#else
   fprintf(stderr, "Set bmtable file: Sorry, I need X11 release 6 to do " 
	   "work!\n");
#endif
}


/*
 * This function allows to adjust the position of the first item and its
 * size (dx, dy). The input values are incremental, not absolute.
 */
void fl_set_bmtable_adjust(FL_OBJECT *ob, int px, int py, int dx, int dy)
{
   BMTABLE_SPEC *sp = (BMTABLE_SPEC *)ob->spec;   
   sp->bx += px;
   sp->by += py;                         
   sp->dx += dx;
   sp->dy += dy; 
}

/* 
 * This function returns the table's selected position.
 */
int fl_get_bmtable(FL_OBJECT *ob)
{ 
   return  ((BMTABLE_SPEC *)ob->spec)->i;
}


/* 
 * You can change the max number of items if you want.
 */
void fl_set_bmtable_maxitems(FL_OBJECT *ob, int i)
{
   if (i>0)
     ((BMTABLE_SPEC *)ob->spec)->maxi = i;
}   


int fl_get_bmtable_maxitems(FL_OBJECT *ob)
{
   return  ((BMTABLE_SPEC *)ob->spec)->maxi;
}


void fl_replace_bmtable_item(FL_OBJECT *ob, int id, int  cw, int ch, char *data)
{
   fprintf(stderr, "Replace bmtable item: Sorry, not yet implemented!\n");
}


void fl_get_bmtable_item(FL_OBJECT *ob, int id, int *cw, int *ch, char *data)
{
   fprintf(stderr, "Get bmtable item: Sorry, not yet implemented!\n");
}  

void fl_set_bmtable(FL_OBJECT *ob, int pushed, int pos)
{
   ((BMTABLE_SPEC *)ob->spec)->i = (pushed) ? pos: -1;
}


int fl_get_bmtable_numb(FL_OBJECT *ob)
{
   return ((BMTABLE_SPEC *)ob->spec)->mousebut;
}


Pixmap fl_get_bmtable_pixmap(FL_OBJECT *ob)
{
   return ((BMTABLE_SPEC *)ob->spec)->pix;
}


void fl_draw_bmtable_item(FL_OBJECT *ob, int i, Drawable d, int xx, int yy)
{
   int x, y, w, h;
   GC gc = fl_state[fl_get_vclass()].gc[0];
   BMTABLE_SPEC *sp = (BMTABLE_SPEC *)ob->spec;
   
   if (sp->pix) {
      x = (i % sp->nx)*sp->dx + FL_abs(ob->bw);
      y = (i/sp->nx)*sp->dy + FL_abs(ob->bw);
      w = sp->dx-2*FL_abs(ob->bw);
      h = sp->dy-2*FL_abs(ob->bw);	 
      XCopyArea(fl_display, sp->pix, d, gc, x, y, w, h, xx, yy);
   }
}

#undef BT_LINE
