/* The code should compile with either ANSI C or K&R compilers. */

/*
 *      Copyright (c) 1993 by California Institute of Technology.
 *      Written by William Deich.  Not derived from licensed software.

 *    You may distribute under the terms of either the GNU General Public
 *    License or the Artistic License, as specified in the README file.
 */

#include "super.h"
#include "version.h"

#ifdef USE_NETGROUP
#define netgrp_u_compare(pattern, user) innetgr(pattern, NULL, user, NULL)
#define netgrp_h_compare(pattern, host) innetgr(pattern, host, NULL, NULL)
#else
#define netgrp_u_compare(p, u) 0
#define netgrp_h_compare(p, h) 0
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Check that an environment variable only includes allowed characters */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Returns 0 if pat matched; -1 otherwise.  */
int
checkenv(name, value, pat)
char *name;	/* variable name to check (e.g. "TERM") */
char *value;	/* contents of variable (e.g. "vt100") */
char *pat;	/* pattern that value must match */
{

    if (!value)
	return -1;

    if (debug)
	(void) fprintf(stderr,
	    "\tcheckenv args: name=\"%s\"; value=\"%s\"; pat=\"%s\"\n",
	    name, value, pat);
    
    /* Environment variables are always checked with re_comp/re_exec:
     * the patterns are fixed internally, not supplied by the user.
     */
    if (re_comp(pat))
	return Error(0, 0,
	    "%t\n\tcheckenv(): couldn't compile pattern `%-.500s'.\n", pat);

    if (re_exec(value) != 1)
	return Error(0, 0,
	    "checkenv(): $%.100s (=%.100s) doesn't match pattern %-.500s.\n",
			name, value, pat);
    return 0;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Check ownership of the file */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
check_owner(file)
char *file;
{
    /* Return 0 if file ownership ok (or if no check required); -1 if not ok */
    struct stat st;
    struct passwd *owner_pw;

    if (!file || *file == '\0')
	return Error(1, 0, "check_owner(): passed null ptr or empty string\n");

    if (*li.owner == '\0')
	return 0;	/* no checks required */

    if (stat(file, &st) == -1)
	return Error(1, 0, "stat() failed on file `%s': ", file);

    /* Convert owner string to a uid */
    owner_pw = getpwnam(li.owner);
    if (!owner_pw) {
	/* Maybe owner was a uid, not a username */
	char c;
	int numeric, i;
	numeric = (sscanf(li.owner, "%d%c", &i, &c) == 1);
	if (numeric)
	    owner_pw = getpwuid(i);
    }
    if (!owner_pw)
	return Error(0, 0,
	"Can't check owner=%s: no such user or uid as `%s' in password file.\n",
	li.owner);

    if (st.st_uid != owner_pw->pw_uid)
	return Error(0, 0,
	    "Actual owner of `%s' is uid %d, but superfile \
requires owner to be %d (%s).\n", file, st.st_uid,
	    owner_pw->pw_uid, owner_pw->pw_name);
    
    return 0;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Set user and group according to the specified args */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
set_u_g()
{
    /* Return 0 on success, -1 on failure */
    struct passwd *uid_pw, *pw;
    struct group *gp;
    SETGRENT_T setgrent();
    void endgrent();
    int i, found_gid, numeric=0;

    ui.new_uid = ui.orig_uid;
    ui.new_gid = ui.orig_gid;

    /* Check gid=xxx */
    if (*li.group) {
	char c;
	numeric = (sscanf(li.group, "%d%c", &i, &c) == 1);
	ui.new_gid = i;

	setgrent();
	for (found_gid=0, gp = getgrent(); gp; gp = getgrent()) {
	    if ((strcmp(li.group, gp->gr_name) == 0) ||
		(numeric && ui.new_gid == gp->gr_gid)) {
		/* Found the gid in the group file */
		ui.new_gid = gp->gr_gid;
		found_gid = 1;
		break;
	    }
	}
	endgrent();
	if (!found_gid && numeric) {
	    /* Look through the password file */
	    for (found_gid=0, pw = getpwent(); pw; pw = getpwent()) {
		if (ui.new_gid == pw->pw_gid) {
		    found_gid = 1;
		    break;
		}
	    }
	    endpwent();
	}
	if (!found_gid)
	    return Error(0, 0,
	    "%t\n\tCan't set gid: \
no such group or gid as `%s' in group%s file.\n",
		li.group, numeric ? " or password" : "");
    }

    /* Check uid=xxx u+g=yyy */
    if (*li.user || *li.u_g) {
	if (!*li.user)
	    strcpy(li.user, li.u_g);
	uid_pw = getpwnam(li.user);
	if (!uid_pw) {
	    char c;
	    numeric = (sscanf(li.user, "%d%c", &i, &c) == 1);
	    ui.new_uid = i;
	    if (numeric)
		uid_pw = getpwuid(ui.new_uid);
	}
	if (uid_pw) {
	    ui.new_uid = uid_pw->pw_uid;
	    if (*li.u_g)
		ui.new_gid = uid_pw->pw_gid;
	} else {
	    return Error(0, 0, "%t\n\tCan't set uid: \
no such user or uid as `%s' in password file.\n", li.user);
	}
    }

    /* Now set uid & gid */
    if (*li.group || *li.u_g)
	if (setgid(ui.new_gid) == -1)
	    return Error(1, 0, "setgid(gid=%d) failed: ", ui.new_gid);
    if (*li.user || *li.u_g)
	if (setuid(ui.new_uid) == -1)
	    return Error(1, 0, "setuid(uid=%d) failed: ", ui.new_uid);

    if (li.passinfo.required) {
	/* Get caller's encrypted password */
	i = get_encrypted_pw();
	if (i != 0)
	    return i;
    }

    return 0;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Puts the encrypted password in ui.encr, and the salt in ui.salt.
 * Returns 0 on success, -1 on failure to obtain the password.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifdef SUNOS5 
int
get_encrypted_pw()
{
    /* Shadow passwords are always used on Sunos 5.x */
    struct spwd *caller_pw;
    if (!(caller_pw = getspnam(ui.caller)))
	return Error(1, 0,
	    "Failed to obtain shadow password entry for user %s: ",
	    ui.caller);
    strcpy(ui.encr, caller_pw->sp_pwdp);
    strncpy(ui.salt, caller_pw->sp_pwdp, 2);
    ui.salt[2] = '\0';
    return 0;
}

#else
#ifdef _HPUX_SOURCE
int
get_encrypted_pw()
{
    struct passwd *caller_pw;

    /* See if we can do shadow password lookup for HPUX 9.x.
     * The rule is that if /.secure/etc/passwd exists, we have to use it;
     * otherwise, fall through to regular password file lookup.
     */
    static struct stat st;
    if (stat("/.secure/etc/passwd", &st) == 0) {
	/* Shadow password file exists; use it */
	struct s_passwd *caller_pw;
	if (!(caller_pw = getspwnam(ui.caller)))
	    return Error(1, 0,
		"Failed to obtain shadow password entry for user %s: ",
		ui.caller);
	strcpy(ui.encr, caller_pw->pw_passwd);
	strncpy(ui.salt, caller_pw->pw_passwd, 2);
	ui.salt[2] = '\0';
    } else {

	/* Fall through to regular password file lookup. */
	if (!(caller_pw = getpwnam(ui.caller)))
	    return Error(0, 0, "No password entry for user %s.\n", ui.caller);
	strcpy(ui.encr, caller_pw->pw_passwd);
	strncpy(ui.salt, caller_pw->pw_passwd, 2);
	ui.salt[2] = '\0';
    }
    return 0;
}

#else
#ifdef SCO
int
get_encrypted_pw()
{
    struct passwd *caller_pw;
    struct spwd *caller_pw;

    if (!(caller_pw = getpwnam(ui.caller)))
	return Error(0, 0, "No password entry for user %s.\n", ui.caller);

    /* SCO 3.2v4 has "x" in password field to indicate shadow password
     * file has to be consulted.
     */
    if (strcmp(caller_pw->sp_pwdp, "x") == 0) {
	/* Shadow password in use... */
	if (!(caller_pw = getspnam(ui.caller)))
	    return Error(1, 0,
	    "Failed to obtain shadow password entry for user %s: ",
	    ui.caller);
	strcpy(ui.encr, caller_pw->sp_pwdp);
	strncpy(ui.salt, caller_pw->sp_pwdp, 2);
	ui.salt[2] = '\0';

    } else {
	/* Fall through to regular password file lookup. */
	strcpy(ui.encr, caller_pw->pw_passwd);
	strncpy(ui.salt, caller_pw->pw_passwd, 2);
	ui.salt[2] = '\0';
    }
    return 0;
}

#else

int
get_encrypted_pw()
{
    /* Vanilla password file lookup */
    struct passwd *caller_pw;

    if (!(caller_pw = getpwnam(ui.caller)))
	return Error(0, 0, "No password entry for user %s.\n", ui.caller);

    strcpy(ui.encr, caller_pw->pw_passwd);
    strncpy(ui.salt, caller_pw->pw_passwd, 2);
    ui.salt[2] = '\0';
    return 0;
}

#endif
#endif
#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Checks if password is needed, and if so, prompts for same.
 * Returns 0 on success, -1 on error.

 * The timestamp directory faces the same problem as the logfile: if the
 * administrator wants to share an NFS-mounted directory across hosts
 * on which root is translated to nobody for NFS access, we have to be
 * able to create the timestamp file under a special uid.  This is done
 * just as in open_writer(): we fork, setuid(), and do the file
 * manipulation in the child.  This allows us to implement a special uid
 * for the timestamp file, without needing the operating system to
 * offer saved uid's or interprocess file-descriptor passing, etc.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
check_pass(cmd)
char *cmd;
{
    char file[MAXPATHLEN];
    struct stat st;
    int l, istat, file_exists, got_pass;
    int timed_out = 0;
    int status;
    PID_T child;

    if (!li.passinfo.required)
	return 0;			/* don't need password */

    /* Create or update the timestamp file even if the lifetime is 0
     * (always ask for password).  We do this because the user may
     * execute _another_ command which has a password expiry > 0,
     * and which will be happy to use the password that was already
     * entered with the 0-lifetime command.
     */
    child = fork();
    if (child == -1) {
	return Error(1, 0, "Failed to create child for timestamp processing: ");

    } else if (child > 0) {
	/* In parent -- wait to see if the child succeeded */
	if (wait(&status) < 0)
	    return Error(1, 0, "wait() on timestamp creation process failed: ");
	else if (status == 0)
	    return 0;
	else
	    return Error(0, 0, "Timestamp creation failed\n");
	
    } else {
	/* In child.  setuid, then make and/or test the directory */
	if (*li.passinfo.user != '\0') {
	    stringcopy(li.user, li.passinfo.user, sizeof(li.user));
	    *li.group = '\0';
	    *li.u_g = '\0';
	    if (set_u_g() == -1) {
		return Error(1, 0,
		"failed to setuid %s before setting timestamp file: ",
		li.user);
	    }
	}
	/* Make the timestamp directory name */
	if (!makedirname(gi.passinfo.perhost ? TIMESTAMP_DIR : "",
							ui.hostname, file))
	    return -1;

	/* Make the timestamp directory */
	if (makedir(file) == -1)
	    return -1;

	/* Make the file in the timestamp directory */
	l = strlen(file) + 1 + strlen(ui.caller);
	if (l >= MAXPATHLEN)
	    return Error(1, 0,
	    "Can't create timestamp file: it would exceed MAXPATHLEN = %d\n",
	    MAXPATHLEN);
	strcat(file, "/");
	strcat(file, ui.caller);

	istat = stat(file, &st);
	if (istat != 0 && errno != ENOENT)
	    return Error(1, 0, "Failed to stat timestamp file `%s': ", file);

	file_exists = (istat == 0);
	if (file_exists)
	    timed_out = (li.passinfo.timeout < 1) ||
			((time(NULL)-st.st_mtime) > li.passinfo.timeout*60);

	got_pass=0;
	if (!file_exists || timed_out) {
	    got_pass = (get_password(cmd, ui.caller, ui.salt, ui.encr) == 1);
	    if (!got_pass)
		return -1;
	}

	/* NOTE: A race condition is possible between two super's, with the
	 * worst-case effect of an error message and failure to run the
	 * requested command.
	 */

	/* If file exists, and we haven't (a) gotten the password again, or
	 * (b) supposed to automatically refresh the timestamp, do nothing to
	 * the file except ensure that we own it.

	 * Otherwise create the file (unlink it first if it exists).
	 */
	if (file_exists && !(got_pass || li.passinfo.renewtime)) {
	    if (st.st_uid != geteuid())
		return Error(0, 0,
	    "Timestamp file `%s' is owned by uid=%d, but expected owner=%d.\n\
\tN.B. If you recently changed the value of timestampuid=xxx, all existing\n\
\tfiles in the timestamp directory _may_ have the wrong owner; delete them.\n\
\t(No security hole appears when you delete a timestamp file.)\n",
		file, st.st_uid, geteuid());

	} else {
	    if (file_exists) {
		if (unlink(file) != 0)
		    return Error(1, 0,
			"Failed to unlink() timestamp file `%s': ", file);
	    }
	    if (open(file, O_WRONLY|O_CREAT|O_TRUNC|O_EXCL, 0200) == -1)
		return Error(1, 0,
		    "Failed to open() timestamp file `%s': ", file);
	}
	exit(0);
    }
    /* UNREACHABLE */
    Error(0, 1, "Unreachable code!\n");
    return 0;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Gets a user's encrypted password. Returns -1 on failure, +1 on success */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
get_password(cmd, user, salt, encr)
char *cmd, *user, *salt, *encr;
{
    /* No such file or password timed out -- get password */
    int ntry, match;
    char msg[500];
    char *crypt(), *getpass();
    char *encrypted = NULL;
    if (strcmp(encr, "") == 0) {
	return Error(0, 0,
	    "Command requires a password, but user `%s' has no password\n",
    user);
    }
    for (ntry=0, match=0; ntry < MAXTRY && !match; ntry++) {
	if (ntry == 0) {
	    (void) sprintf(msg,
    "Your password is required for super command `%s'...\nPassword: ",
		cmd);
	} else {
	    strcpy(msg, "Password incorrect\nPassword: ");
	}

	encrypted = crypt(getpass( msg ), salt);
	if (encr && encrypted)
	    match = (strcmp(encr, encrypted) == 0);
	else
	    match = 0;
    }
    if (!match)
	return Error(0, 0, "Password incorrect\n");
    return 1;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Adds condition to condition list.
 *	returns -1 on syntax error, malloc error, etc;
 *	returns 0 otherwise.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
InsertCondition(condition, s, isglobal)
char *condition;	/* condition to insert: xxx~yyy */
char *s;		/* pts to yyy in condition */
int isglobal;		/* Is this a per-command or global condition? */
{
    int invert = (*condition == '!');
    TimeList *tl;

    if ( invert )
	condition++;

    if (STRMATCH3("time", condition, s-1)) {
	tl = isglobal ? &gi.timeafter : &li.time;
	if (InsertTimeList(s, tl, isglobal ? "global" : "local", invert) == -1)
	    return -1;

    } else if (STRMATCH3("user", condition, s-1)) {
	if (InsertUserList(s, &li.userpats, &li.origtext, invert) == -1)
	    return -1;

    } else {
	return Error(0, 0,
		"%t\n\tInternal error: unrecognized condition <%s>.\n",
		condition);
    }
    return 0;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Add a user/group/host pattern to a list.
 *	returns -1 on syntax error, malloc error, etc;
 *	returns 0 otherwise.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
InsertUserList(wd, sl, otl, invert)
char *wd;	/* Pattern to match; must NOT have leading '!'; braces ok;
		 * disallow '<', '>'.
		 */
Simple2List *sl;/* Insert user list elements at sl->next */
SimpleList *otl;/* Insert original text to this list, at otl->next */
int invert;	/* Inverts the test */
{
    int i, iwd;
    char *tok, *s;
    char **wdlist;
    int globbraces();
    SimpleList *new;
    Simple2List *new2;

    /* Check for illegal characters */
    if ((s=strchr(wd, '>')) || (s=strchr(wd, '<'))) {
	if (s-wd == 4 && strncmp(wd, "time", 4) == 0) {
	    return Error(0, 0,
	    "%t\n\tPermittedUser patterns may not use '>' or '<';\n\
\tyou used '%s'; perhaps you meant to write 'time~%s'\n", wd, s);
	} else {
	    return Error(0, 0,
	    "%t\n\tPermittedUser patterns may not use '>' or '<';\n\
\tyou used '%s'.\n", wd);
	}
    }
    new = (SimpleList *) malloc(sizeof(SimpleList));
    if (!new)
	return Error(0, 0, "%t\n\tFailed to malloc space for PermittedUser\n");
    new->next = otl->next;
    new->pat = (char *) malloc(strlen(wd) + 1);
    if (!new->pat)
	return Error(0, 0,
		"%t\n\tFailed to malloc space for PermittedUser pat\n");
    strcpy(new->pat, wd);
    otl->next = new;

    /* Do brace globbing */
    if ((i=globbraces(wd, &wdlist)) != 0)
	return Error(0, 0, "%tMissing `%c'.\n", i);

    for (iwd=0; (tok=wdlist[iwd]); iwd++) {
	new2 = (Simple2List *) malloc(sizeof(Simple2List));
	if (!new2)
	    return Error(0, 0,
		"%t\n\tFailed to malloc space for PermittedUser\n");
	new2->next = sl->next;
	new2->other = otl->next;
	new2->pat = (char *) malloc(strlen(tok) + (invert ? 2 : 1));
	if (!new2->pat)
	    return Error(0, 0,
		"%t\n\tFailed to malloc space for PermittedUser pat\n");
	if (invert) {
	    *new2->pat = '!';
	    strcpy(new2->pat+1, tok);
	} else {
	    strcpy(new2->pat, tok);
	}
	sl->next = new2;
    }
    return 0;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Match a list of user/group/host pattern against the present user. */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void
match_ugh_user(sl, isglobal)
Simple2List *sl;/* A list of user pats to match against, starting at sl->next */
int isglobal;	/* !0 means its from a global def */
{
    /* Checks if user is matched against each elt in sl.
     * Sets matches.match_user if user matches; sets it to 1 if the last
     * match is non-inverting, 0 otherwise.

     * BUT! the time list created by the InsertUserList function
     * is in reverse order, so we only need to find if the first
     * entry in the list that is a match (+ or -) and stop there!
     */

    int invert, match;
    int check_ugh __P((char *, char *));

    for (match=0, sl=sl->next; sl && !match; sl=sl->next) {
	invert = *sl->pat == '!';
	if (check_ugh(sl->other->pat, invert ? sl->pat+1 : sl->pat) == 0) {
	    match = 1;
	    matches.user = invert ? 0 : 1;
	    if (debug)
		(void) fprintf(stderr,
			"\tPermission %s: %s pattern %suser~%s\n",
			invert ? "denied" : "allowed",
			isglobal ? "global" : "per-cmd",
			invert ? "!" : "", sl->pat);
	} else if (debug) {
	    (void) fprintf(stderr, "\tNot applicable: %s pattern %suser~%s\n",
			isglobal ? "global" : "per-cmd",
			invert ? "!" : "", sl->pat);
	}
    }
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Check pattern against a hostname.  If the hostname is fully-qualified,
 * then try stripping off each of the domains to find a match.
 * Return -1 on failure to match; 0 on success.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
check_host(pat, host)
char *pat, *host;
{
    int is_netgroup = 0;
    int match;
    char *p, *dotp;

    if (*pat == '+') {
       is_netgroup = 1;
       ++pat;
    }

    match = (is_netgroup ? netgrp_h_compare(pat, host)
						: (*pat_compare)(host));

    dotp = strrchr (host, '.');
    while (dotp && !match) {
       *dotp = 0;
       match = (is_netgroup ? netgrp_h_compare(pat, host)
						: (*pat_compare)(host));
       p = strrchr (host, '.');
       *dotp = '.';
       dotp = p;
    }
    return (match ? 0 : -1);
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Try to match a string to a pattern. */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
match_pattern(match, str, pattern)
int match;	/* Return input value of match on failure; 1 on success */
char *str;
char *pattern;
{
    int i, ipat;
    char *tok, tokbuf[1000];
    char **patlist;
    int globbraces();
    char chkbuf[1024];

    /* Do brace globbing on the pattern */
    /* fprintf(stderr, "globbraces -> ``%s''\n", pattern); */
    if ((i=globbraces(pattern, &patlist)) != 0) {
	Error(0, 0, "%tMissing `%c'.\n", i);
    } else {
	for (ipat=0; (tok=patlist[ipat]); ipat++) {
	    strcpy(tokbuf, tok);
	    anchor(tok, chkbuf);			/* Anchor all matches */
	    if ((*pat_compile)(chkbuf) != NULL) {
		return Error(0, 0,
			"%t\n\tBad command pattern: `%s'.\n", pattern);

	    } else if ((*pat_compare)(str) == 1) {
		    if (debug)
			(void) fprintf(stderr,
			    "\tMatched user's command=%s to CmdPattern=%s\n",
			    str, pattern);
		    return 1;

	    } else if (debug) {
		(void) fprintf(stderr,
			"\tNo match user's command=%s to CmdPattern=%s\n",
			    str, pattern);
	    }
	}
    }
    return match;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Check a single user/group/host string
 * Return -1 on failure to match; 0 on success.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
check_ugh(origtext, token)
char *origtext;		/* original text -- for error messages */
char *token;		/* user/group/host pattern */
{
    char chkbuf[1024];
    char *userpat, *grouppat, *hostpat;
    int match, i;

    if (ui.orig_uid == 0)
	return 0;			/* root is always legit */

    /* Split into user:group@host; check host part first (if it exists) */
    if ((hostpat = strchr(token, '@'))) {
	if (hostpat[1] == 0)
	return Error(0, 0, "%tMissing hostname in pattern `%s'.\n", origtext);
	*hostpat++ = 0;
 
	match = -1;
	if (hostpat[0] == '+') {

#ifdef USE_NETGROUP
	    if (hostpat[1] == 0)
		return Error(0, 0,
			"%tMissing netgroupname in pattern `%s'.\n", origtext);
	    match = check_host(hostpat, ui.hostname);
#else
		return Error(0, 0, 
	"%thostnames may not begin with `+' since this super() was compiled\n\
without -DUSE_NETGROUP.\n");
#endif
	} else {
	    strtolower(hostpat);
	    anchor(hostpat, chkbuf);    /* Force all matches to be anchored */
	    if ((*pat_compile)(chkbuf) != NULL)
		return Error(0, 0, "%tbad host pattern: `%s'.\n", origtext);
	}
 
	if (match == -1)
	    match = check_host(hostpat, ui.lc_hostname);

	if (match == -1)
	    return -1;
    }

    grouppat = strchr(token, ':');
    userpat = token;
    if (*token == '\0' && !hostpat) {
	/* Nothing in pattern?! */
	return Error(0, 0, "%t\n\tUnacceptable pattern `%s'.\n", origtext);

    } else if (*token == '\0') {
	userpat = grouppat = "^.*$";		/* only hostname given */

    } else if (grouppat && *(grouppat+1)) {	/* pat is "uuu:ggg or ":ggg" */
	if (token == grouppat)
	    userpat = "^.*$";			/* pat is ":ggg" */
	*grouppat++ = '\0';

    } else {					/* pat is "uuu" or "uuu:" */
	if (grouppat)
	    *grouppat = '\0';			/* pat is "uuu:" */
	grouppat = "^.*$";
    }
    if (strchr(grouppat, ':'))
	return Error(0, 0, "%t\n\tGroup pattern `%s' contains a colon!\n",
			grouppat);

    if (gi.group_slash == 0 && strchr(grouppat, '/'))
		return Error(0, 0, "%t\n\tFormat error in super.tab file: \
group pattern `%s' contains a slash.\n\
\tPerhaps you meant to use Cmd::Filename, but forgot one colon,\n\
\tso it looks like User:Group?  If you really need to allow\n\
\tslashes in group patterns, use global option group_slash=y.\n", grouppat);


#ifdef USE_NETGROUP
    if (userpat[0] == '+') {
	if (userpat[1] == 0)
	    return Error(0, 0,
			"%tMissing netgroupname in pattern `%s'.\n", origtext);
	match = netgrp_u_compare(&userpat[1], ui.caller);
    } else
#endif
    {
	anchor(userpat, chkbuf);		/* Anchor all matches */
	if ((*pat_compile)(chkbuf) != NULL)
	    return Error(0, 0, "%t\n\tbad user pattern: `%s'.\n", origtext);
	match = (*pat_compare)(ui.caller);
#ifdef MATCH_DECIMAL_UID
	if (match != 1) {
	    /* Enabling MATCH_DECIMAL_UID allows the userpat to be
	     * numeric, as an alternative to being interpreted as a
	     * user name: after checking the username, we check if the
	     * user's uid, as a decimal text value, matches the user
	     * pattern userpat.
	     */
	    char buf[20];
	    (void) sprintf(buf, "%d", ui.orig_uid);
	    match = (*pat_compare)(buf);
	}
#endif
    }
    if (match != 1)
	return -1;

    anchor(grouppat, chkbuf);
    i = ingroup(ui.caller, ui.orig_gid, chkbuf);
    if (i == -1)
	return Error(0, 0, "%t\n\tbad group pattern\n", origtext);
    else if (i != 1)
	return -1;
    
    return 0;				/* Success! */
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Determines if user's group matches a group pattern. */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
ingroup(user, gid, gp_pat)
char *user;
GID_T gid;
char *gp_pat;	/* pattern to match */
{
    /* Use:
     *	ingroup(user, gid, gp_pat)
     * Returns:
     *	1 if the user is in a group matching the regex pattern gp_pat.
     *	0 if the user isn't in a group matching the pattern.
     *	-1 if pattern failed to compile.

     * SIDE-EFFECT: uses pat_compile/pat_compare!
     *			-- messes up caller's use of same!

     * Examples:
     *	ingroup("joe", joes_gid, "xyz")
     * returns !0 if user joe is in group "xyz".
     *	ingroup("joe", joes_gid, "xy.*")
     * returns !0 if user joe is in any group matching "xy.*".

     */

    struct group *gp;
    char **mem;
    char buf[20];
    SETGRENT_T setgrent();
    void endgrent();

    if ((*pat_compile)(gp_pat) != (char *)0 )
	return -1;

    /* Search group file for groups user is in.  For each group of which
     * the user is a member, test a match to the pattern.
     */
    setgrent();
    for (gp = getgrent(); gp; gp = getgrent()) {
	/* The gr_mem list only shows usernames added in the /etc/group file,
	 * and not any users assigned to the group in the passwd file.
	 * Thus discover group membership by first checking the user's
	 * group from the password file (gp->gr_gid) against this group's
	 * gid, then check to see if this user is in the gp->gr_mem list.
	 */
	if (gid != gp->gr_gid) {
	    for (mem = gp->gr_mem; *mem ; mem++)
		if (strcmp(*mem, user) == 0)
		    break;
	    if (!*mem)
		continue;			/* not in group */
	}
	/* if here, the user is in group gp; now check if group
	 * name gp->gr_name matches group pattern gp_pat.
	 */
	if ((*pat_compare)(gp->gr_name) == 1) {
	    /* successful match -- user is in a group that matches gp_pat */
	    endgrent();
	    return 1;
	}
#ifdef MATCH_DECIMAL_GID
	else {
	    /* Enabling MATCH_DECIMAL_GID allows the gp_pat to be
	     * numeric, as an alternative to being interpreted as a
	     * group name: we check if the group id gp->gr_gid, as a
	     * decimal text value, matches the group pattern gp_pat.
	     */
	    (void) sprintf(buf, "%d", gp->gr_gid);
	    if ((*pat_compare)(buf) == 1){
		/* successful match -- user is in a group that matches gp_pat */
		endgrent();
		return 1;
	    }
	}
#endif
    }

#ifdef MATCH_DECIMAL_GID
    /* We haven't found any group from /etc/group to which we belong that
     * matches the pattern.  It is possible that the user's group id from the
     * password file isn't in the /etc/group file at all, in which case the
     * user's group won't have matched the pattern since we've only checked
     * /etc/group entries so far.  Now check the numeric id from the
     * /etc/passwd file against the pattern.
     */
    (void) sprintf(buf, "%d", gid);
    if ((*pat_compare)(buf) == 1){
	endgrent();
	return 1;
    }
#endif

    endgrent();
    return 0;
}

