/*************************************************************************\
**  source       * embed.c
**               * 
**  description  * SOLID SQL API usage sample:
**               * 
**               * Embedded SQL statements translated to dynamic CLI
**               * function calls
**               * 
**               * (C) Copyright Solid Information Technology Ltd 1996
\*************************************************************************/

/* Environment specific headers */
#ifdef SS_WINDOWS
#include <windows.h>
#include <stdio.h>
#endif

/* Common system headers */
#include <c.h>
#include <assert.h>


/* SOLID SQL API headers */
#include <cli0defs.h>
#include <cli0core.h>
#include <cli0ext1.h>

#include <c.h>
#include <assert.h>

/* Constants */
#define MAX_NAME_LEN 50
#define MAX_STMT_LEN 100

/* Utility macros - very crude error handling */
#define CHECK(r)  \
        if ( ((r) != SQL_SUCCESS) && ((r) != SQL_SUCCESS_WITH_INFO) ) \
            print_error((r), henv, hdbc, hstmt); \
        assert( ((r) == SQL_SUCCESS) || ((r) == SQL_SUCCESS_WITH_INFO) )

      

/* Function prototypes */
static void usage(void);
static void welcome(void);
static void goodbye(UCHAR* network_name);
static void ask_connection_info(
        UCHAR*  network_name, UCHAR* username, UCHAR* password
);
static void embedded_to_dynamic_SQL_example(
        UCHAR * server, UCHAR * uid, UCHAR * pwd
);
static void print_error(
        RETCODE rc, HENV henv, HDBC hdbc, HSTMT hstmt
);
static void error_exit(
        RETCODE rc, HENV henv, HDBC hdbc, HSTMT hstmt
);
static UCHAR* read_line(UCHAR* string, SWORD maxlen);


/*##**********************************************************************\
 *
 *              main
 *
 * ad hoc SQL query execution program main function.
 *
 * Parameters : none
 *
 * Return value - give :
 *       0      = Success return
 *      -1      = Error occurred
 */
int SS_CDECL main(int argc, UCHAR* argv[])
{
        UCHAR   network_name[MAX_NAME_LEN];
        UCHAR   username[MAX_NAME_LEN];
        UCHAR   password[MAX_NAME_LEN];

        if (argc > 1) {
            usage();
            return(-1);
        }

        welcome();

        ask_connection_info(network_name, username, password);

        embedded_to_dynamic_SQL_example(network_name, username, password);

        goodbye(network_name);

        return(0);
}


/*##**********************************************************************\
 *
 *              usage
 *
 * Print usage instructions on standard output.
 */
static void usage(void)
{
        printf("\nUsage:  embed \n\n");
        welcome();
}


/*##**********************************************************************\
 *
 *              welcome
 *
 * Print welcome texts on standard output. 
 */
static void welcome(void)
{
        printf("This program automatically executes a set of static SQL \n");
        printf("statements and checks their success.\n\n");
}


/*##**********************************************************************\
 *
 *              ask_connection_info
 *
 * Prompt the user for SOLID Server network name, username and password.
 *
 * Parameters - output:
 *      network name, username, password: character strings 
 */
static void ask_connection_info(
        UCHAR*  network_name, UCHAR* username, UCHAR* password
) {
        printf("Try connecting with the default network name ");
#ifdef SS_WINDOWS
        printf("'shmem solid'.\n");
#else
        printf("'tcp [host] 1313'.\n");
#endif

        printf("\nConnect to SOLID Server with network name:\n");
        read_line(network_name, MAX_NAME_LEN);

        printf("\nUsername:\n");
        read_line(username, MAX_NAME_LEN);

        printf("\nPassword:\n");
        read_line(password, MAX_NAME_LEN);
}


/*##**********************************************************************\
 *
 *              goodbye
 *
 * Output normal termination message.
 */
static void goodbye(UCHAR* network_name)
{
        printf("Disconnected from '%s'.\n", network_name);
}


/*##**********************************************************************\
 *
 *              embedded_to_dynamic_SQL_example
 *
 *  Example C function that constructs SQL statements 
 *  within the application. The example comments include
 *  equivalent embedded SQL calls for illustrative purposes. 
 *
 *  Please see SOLID Server Programmer's Guide and Reference
 *  for detailed information on functions and their parameters.
 */
static void embedded_to_dynamic_SQL_example(
        UCHAR * server, UCHAR * uid, UCHAR * pwd
) {
	HENV   henv = NULL;
	HDBC   hdbc = NULL;
	HSTMT  hstmt = NULL;

	SDWORD  id;
	UCHAR   name[MAX_NAME_LEN + 1];
	UCHAR   create[MAX_STMT_LEN];
	UCHAR   insert[MAX_STMT_LEN];
	UCHAR   select[MAX_STMT_LEN];
	UCHAR   delete[MAX_STMT_LEN];
	UCHAR   drop[MAX_STMT_LEN];
	SDWORD  namelen;
	SDWORD  rowcount;
	RETCODE rc;

        /********************************************************/
        printf("Connecting to server '%s' ...\n", server);

	/* EXEC SQL CONNECT TO :server USER :uid USING :pwd; */
	/* Allocate an environment handle. */
	/* Allocate a connection handle. */
	/* Connect to a data source. */
	/* Allocate a statement handle. */

	rc = SQLAllocEnv(&henv);
        CHECK(rc);
	rc = SQLAllocConnect(henv, &hdbc);
        CHECK(rc);
	rc = SQLConnect(hdbc, server, SQL_NTS, uid, SQL_NTS,  pwd, SQL_NTS);
        CHECK(rc);
	rc = SQLAllocStmt(hdbc, &hstmt);
        CHECK(rc);

        /********************************************************/
        printf("Creating table 'NAMEID' ...\n");

	/* EXEC SQL CREATE TABLE NAMEID       */
	/*    (ID integer, NAME varUCHAR(50)); */
	/* Execute the SQL statement. */

	strcpy(create, (UCHAR *)
               "CREATE TABLE NAMEID (ID INTEGER, NAME VARCHAR(50))"
        );
	rc = SQLExecDirect(hstmt, create, SQL_NTS);
        CHECK(rc);

        /********************************************************/
        printf("Commit.\n");

	/* EXEC SQL COMMIT WORK;      */
	/* Commit the table creation. */

	/* Note that the default transaction mode for drivers */
	/* that support SQLSetConnectOption is auto-commit    */
	/* and SQLTransact has no effect. */

	rc = SQLTransact(henv, hdbc, SQL_COMMIT);
        CHECK(rc);

        /********************************************************/
        printf("Inserting a row to table 'NAMEID' (500, Babbage) ...\n");

	/* EXEC SQL INSERT INTO NAMEID VALUES ( :id, :name ); */
	/* Show the use of the SQLPrepare/SQLExecute method:  */
	/* Prepare the insertion and bind parameters. */
	/* Assign parameter values. */
	/* Execute the insertion. */

	strcpy((UCHAR *)insert, (UCHAR *)"INSERT INTO NAMEID VALUES (?, ?)");
	rc = SQLPrepare(hstmt, insert, SQL_NTS);
        CHECK(rc);
	rc = SQLBindParameter(hstmt, 1, SQL_PARAM_INPUT, SQL_C_SLONG, 
		                  SQL_INTEGER, 0, 0, &id, 0, NULL);
        CHECK(rc);
	rc = SQLBindParameter(hstmt, 2, SQL_PARAM_INPUT, SQL_C_CHAR, 
                        	  SQL_VARCHAR, MAX_NAME_LEN, 0, name, 0, NULL);
        CHECK(rc);

	id=500;
	strcpy((UCHAR *)name, (UCHAR *)"Babbage");
	rc = SQLExecute(hstmt);
        CHECK(rc);

        /********************************************************/
        printf("Commit.\n");

	/* EXEC SQL COMMIT WORK; */
	/* Commit the insertion. */

	rc = SQLTransact(henv, hdbc, SQL_COMMIT);
        CHECK(rc);

        /* Close statement */
        rc = SQLFreeStmt(hstmt, SQL_CLOSE );
        CHECK(rc);

        /* Unbind Parameters */
        rc = SQLFreeStmt(hstmt, SQL_RESET_PARAMS );
        CHECK(rc);
        id = 0;
        name[0] = '\0';

        /********************************************************/
        printf("Retrieving data from table 'NAMEID'...\n");

	/* EXEC SQL DECLARE c1 CURSOR FOR */
	/* SELECT ID, NAME FROM NAMEID;   */
	/* EXEC SQL OPEN c1; */
	/* Show the use of the SQLExecDirect method. */
	/* Execute the selection. */
	/* Note that the application does not explicitly declare a cursor. */

	strcpy((UCHAR *)select, (UCHAR *)"SELECT ID, NAME FROM NAMEID");
	rc = SQLExecDirect(hstmt, select, SQL_NTS);
        CHECK(rc);

	/* EXEC SQL FETCH c1 INTO :id, :name; */
	/* Bind the columns of the result set */
	/* with SQLBindCol.     */
	/* Fetch the first row. */
	rc = SQLBindCol(hstmt, 1, SQL_C_SLONG, &id, 0, NULL);
        CHECK(rc);
	rc = SQLBindCol(hstmt, 2, SQL_C_CHAR, name,
                   (SDWORD)sizeof(name), &namelen);
        CHECK(rc);
	rc = SQLFetch(hstmt);
        CHECK(rc);

        printf("A row found (%d, %s).\n", id, name);

	rc = SQLFetch(hstmt);
        assert(rc == SQL_NO_DATA_FOUND);

        printf("End of result set reached.\n");
        

        /********************************************************/
        printf("Commit.\n");

	/* EXEC SQL COMMIT WORK;   */
	/* Commit the transaction. */

	rc = SQLTransact(henv, hdbc, SQL_COMMIT);
        CHECK(rc);

        /********************************************************/
        printf("Closing cursor ...\n");

	/* EXEC SQL CLOSE c1;         */
	/* Free the statement handle. */

        /* Close statement */
        rc = SQLFreeStmt(hstmt, SQL_CLOSE );
        CHECK(rc);

        /* Unbind variables */
        rc = SQLFreeStmt(hstmt, SQL_UNBIND );
        CHECK(rc);

        /********************************************************/
        printf("Deleting all rows from table 'NAMEID' ...\n");

	/* EXEC SQL DELETE FROM NAMEID   */
	/* Execute the SQL statement.
           Check from sqlca structure the number of rows deleted. */

	strcpy(delete, (UCHAR *) "DELETE FROM NAMEID" );
	rc = SQLExecDirect(hstmt, delete, SQL_NTS);
        CHECK(rc);

        rc = SQLRowCount(hstmt, &rowcount);
        CHECK(rc);
        assert( rowcount == (SDWORD)1 );
        printf("1 row deleted.\n");


        /********************************************************/
        printf("Commit.\n");

	/* EXEC SQL COMMIT WORK;      */

	rc = SQLTransact(henv, hdbc, SQL_COMMIT);
        CHECK(rc);


        /********************************************************/
        printf("Dropping table 'NAMEID' ...\n");

	/* EXEC SQL DROP TABLE NAMEID       */
	/* Execute the SQL statement. */

	strcpy(drop, (UCHAR *) "DROP TABLE NAMEID" );
	rc = SQLExecDirect(hstmt, drop, SQL_NTS);
        CHECK(rc);

        /********************************************************/
        printf("Commit.\n");

	/* EXEC SQL COMMIT WORK;      */

	rc = SQLTransact(henv, hdbc, SQL_COMMIT);
        CHECK(rc);


        /* Release statement resource allocation */
	rc = SQLFreeStmt(hstmt, SQL_DROP);
        CHECK(rc);

        /********************************************************/
        printf("Disconnecting from server ...\n");

	/* EXEC SQL DISCONNECT;             */
	/* Disconnect from the data source. */
	/* Free the connection handle.      */
	/* Free the environment handle.     */

	SQLDisconnect(hdbc);
	SQLFreeConnect(hdbc);
	SQLFreeEnv(henv);

	return;
}


/*##**********************************************************************\
 *
 *              terminate_connection
 *
 * Close connection and free resources.
 */
static void terminate_connection(
        HENV henv, HDBC hdbc, HSTMT hstmt
) {
	/* Free the statement handle.       */
	/* Disconnect from the database.    */
	/* Free the connection handle.      */
	/* Free the environment handle.     */

	if (hstmt) {
            SQLFreeStmt(hstmt, SQL_DROP );
            SQLDisconnect(hdbc);
        }
        if (hdbc)	SQLFreeConnect(hdbc);
        if (henv)   SQLFreeEnv(henv);
}


/*##**********************************************************************\
 *
 *              print_error
 *
 * Output error message and disconnect from database.
 */
static void print_error(RETCODE rc, HENV henv, HDBC hdbc, HSTMT hstmt)
{
        UCHAR   SQLstate[MAX_NAME_LEN];
        SDWORD  errorcode;
        UCHAR   errormsg[SQL_MAX_MESSAGE_LENGTH];
        SWORD   msglen;
        RETCODE r;

        printf("Error (%d) encountered!\n", rc);

        /* Print more error info, if available */
        if (henv && hdbc && hstmt) {
            r = SQLError(henv, hdbc, hstmt, SQLstate, &errorcode, errormsg,
                        SQL_MAX_MESSAGE_LENGTH - 1, &msglen);
            if (r != SQL_NO_DATA_FOUND) {
                printf("%s\n", errormsg);
            }
        }
/*        terminate_connection(henv, hdbc, hstmt); */
}            


/*##**********************************************************************\
 *
 *              read_line
 *
 * Read one line from standard input into the parameter buffer.
 * Allows continuing to the following by preceding the end-of-line with '\'.
 */
static UCHAR* read_line(
        UCHAR* string, SWORD maxlen
) {

	static UCHAR retbuf[MAX_NAME_LEN + 1];
	static UCHAR buf[MAX_NAME_LEN + 1];
	UCHAR* sp = NULL; 
	UCHAR* tp = NULL;
	bool loopend = FALSE;

	*string = '\0';
	retbuf[0] = '\0';
	retbuf[MAX_NAME_LEN] = '\0';
	buf[0] = '\0';
	buf[MAX_NAME_LEN] = '\0';

	while(!loopend) {

            sp = buf;

	 	/* read a line */
		fgets(sp, MAX_NAME_LEN - 1, stdin);

		/* trim leading non-alpha UCHARacters */
		while (*sp && *sp != '\n' && isspace(*sp)) 
			sp++;

            if (*sp == '\n') *sp = '\0';

            if (!*sp) {
                /* empty string */
                break;
            }

		/* trim trailing non-alpha UCHARacters */
		tp = sp + strlen(sp) - 1;
		while (tp > sp && *tp != '\\' && isspace(*tp) )
			tp--;

		/* check if input is continued on the following line */
            if (*tp == '\\') {
			*tp = '\0';
		} else {
			*++tp = '\0';
			loopend = TRUE;
		}
		strcat(retbuf, sp);
	}

	/* check and enforce maxlen */
	if (strlen(retbuf) > maxlen)
		retbuf[maxlen] = '\0';

	strcpy(string, retbuf);
	return(string);
}

