;==============================================================================

; file: "_t-c-3.scm"

; Copyright (C) 1994-1998 by Marc Feeley, All Rights Reserved.

(include "generic.scm")

;------------------------------------------------------------------------------
;
; Back end for C language (part 3)
; -----------------------

(define targ-bits-per-byte 8) ; 8 bits per byte

; FIXNUM representation.

; Limits of fixnums (assumes 2's complement representation):

(define targ-min-fixnum32
  (* -2
     (expt 2 (- (* 4 targ-bits-per-byte)
                (+ targ-tag-bits 2)))))

(define targ-max-fixnum32
  (- (+ targ-min-fixnum32 1)))

(define (targ-fixnum32? x)
  (and (>= x targ-min-fixnum32)
       (<= x targ-max-fixnum32)))

(define targ-min-fixnum64
  (* -2
     (expt 2 (- (* 8 targ-bits-per-byte)
                (+ targ-tag-bits 2)))))

(define targ-max-fixnum64
  (- (+ targ-min-fixnum64 1)))

(define (targ-fixnum64? x)
  (and (>= x targ-min-fixnum64)
       (<= x targ-max-fixnum64)))

(define (targ-u32-to-i32 n)
  (if (< 2147483647 n)
    (- n 4294967296)
    n))

(define (targ-i32->hex-string n)
  (if (< n 0)
    (string-append "-0x" (number->string (- n) 16))
    (string-append "0x" (number->string n 16))))

; BIGNUM representation.

; Radix must be a power of two and targ-bignum-radix^2 - 1 must fit in
; a 32 bit fixnum.

(define targ-bignum-radix
  (expt 2
        (quotient (- (* 4 targ-bits-per-byte)
                     (+ targ-tag-bits 1))
                  2)))

(define (targ-bignum-digits obj)

  (define (integer->digits n)
    (if (= n 0)
      '()
      (cons (remainder n targ-bignum-radix)
            (integer->digits (quotient n targ-bignum-radix)))))

  (if (< obj 0)
    (cons 0 (integer->digits (- obj)))
    (cons 1 (integer->digits obj))))

; FLONUM representation.

(define targ-inexact-+2   (exact->inexact 2))
(define targ-inexact--2   (exact->inexact -2))
(define targ-inexact-+1   (exact->inexact 1))
(define targ-inexact-+1/2 (exact->inexact (/ 1 2)))
(define targ-inexact-0    (exact->inexact 0))

(define (targ-float->inexact-exponential-format x f64?)
  (let* ((e-bits (if f64? 11 8))
         (e-bias (- (expt 2 (- e-bits 1)) 1)))

    (define (exp-form-pos x y i)
      (let ((i*2 (+ i i)))
        (let ((z (if (and (not (< e-bias i*2))
                          (not (< x y)))
                   (exp-form-pos x (* y y) i*2)
                   (vector x 0 1))))
          (let ((a (vector-ref z 0)) (b (vector-ref z 1)))
            (let ((i+b (+ i b)))
              (if (and (not (< e-bias i+b))
                       (not (< a y)))
                (begin
                  (vector-set! z 0 (/ a y))
                  (vector-set! z 1 i+b)))
              z)))))

    (define (exp-form-neg x y i)
      (let ((i*2 (+ i i)))
        (let ((z (if (and (< i*2 (- e-bias 1))
                          (< x y))
                   (exp-form-neg x (* y y) i*2)
                   (vector x 0 1))))
          (let ((a (vector-ref z 0)) (b (vector-ref z 1)))
            (let ((i+b (+ i b)))
              (if (and (< i+b (- e-bias 1))
                       (< a y))
                (begin
                  (vector-set! z 0 (/ a y))
                  (vector-set! z 1 i+b)))
              z)))))

    (define (exp-form x)
      (if (< x targ-inexact-+1)
        (let ((z (exp-form-neg x targ-inexact-+1/2 1)))
          (vector-set! z 0 (* targ-inexact-+2 (vector-ref z 0)))
          (vector-set! z 1 (- -1 (vector-ref z 1)))
          z)
        (exp-form-pos x targ-inexact-+2 1)))

    (if (negative? (float-copysign targ-inexact-+1 x))
      (let ((z (exp-form (float-copysign x targ-inexact-+1))))
        (vector-set! z 2 -1)
        z)
      (exp-form x))))

(define (targ-float->exact-exponential-format x f64?)
  (let* ((z      (targ-float->inexact-exponential-format x f64?))
         (m-bits (if f64? 52 23))
         (e-bits (if f64? 11 8)))

    (let ((y (vector-ref z 0)))
      (if (not (< y targ-inexact-+2)) ; +inf. or +nan.?
        (begin
          (if (< targ-inexact-0 y)
            (vector-set! z 0 (expt 2 m-bits))              ; +inf.
            (vector-set! z 0 (- (* (expt 2 m-bits) 2) 1))) ; +nan.
          (vector-set! z 1 (expt 2 (- e-bits 1))))
        (vector-set! z 0
          (truncate
            (inexact->exact
             (* (vector-ref z 0) (exact->inexact (expt 2 m-bits)))))))
      (vector-set! z 1 (- (vector-ref z 1) m-bits))
      z)))

(define (targ-float->bits x f64?)
  (let ((m-bits (if f64? 52 23))
        (e-bits (if f64? 11 8)))

    (define (bits a b)
      (let ((m-min (expt 2 m-bits)))
        (if (< a m-min)
          a
          (+ (- a m-min)
             (* (+ (+ b m-bits) (- (expt 2 (- e-bits 1)) 1))
                m-min)))))

    (let* ((z (targ-float->exact-exponential-format x f64?))
           (y (bits (vector-ref z 0) (vector-ref z 1))))
      (if (negative? (vector-ref z 2))
        (+ (expt 2 (+ e-bits m-bits)) y)
        y))))

(define (targ-f32->bits x)
  (targ-float->bits x #f))

(define (targ-f64->hi-lo-bits x)
  (let ((bits (targ-float->bits x #t)))
    (cons (quotient bits #x100000000)
          (modulo   bits #x100000000))))

(define (targ-unusual-float? x)
  (cond ((zero? x)
         (negative? (float-copysign targ-inexact-+1 x)))
        ((negative? x)
         (not (< x (/ x targ-inexact-+2))))
        (else
         (not (< (/ x targ-inexact-+2) x)))))

; RATNUM representation.

(define (targ-numerator x)
  (numerator x))

(define (targ-denominator x)
  (denominator x))

; CPXNUM representation.

(define (targ-real-part x)
  (real-part x))

(define (targ-imag-part x)
  (imag-part x))

; Extraction of object's type and subtype.

(define (targ-obj-type obj)
  (cond ((false-object? obj)
         'boolean)
        ((eq? obj #t)
         'boolean)
        ((null? obj)
         'null)
        ((absent-object? obj)
         'absent)
        ((void-object? obj)
         'void)
        ((end-of-file-object? obj)
         'eof)
        ((optional-object? obj)
         'optional)
        ((rest-object? obj)
         'rest)
        ((key-object? obj)
         'key)
        ((script-object? obj)
         'script)
        ((symbol-object? obj)
         'subtyped)
        ((keyword-object? obj)
         'subtyped)
        ((proc-obj? obj)
         'procedure)
        ((pair? obj)
         'pair)
        ((number? obj)
         (cond ((and (integer? obj) (exact? obj) (targ-fixnum32? obj))
                'fixnum32)
               (else
                'subtyped)))
        ((char? obj)
         'char)
        (else
         'subtyped)))

(define (targ-obj-subtype obj)
  (cond ((symbol-object? obj)
         'symbol)
        ((keyword-object? obj)
         'keyword)
        ((number? obj)
         (cond ((and (integer? obj) (exact? obj))
                (if (targ-fixnum64? obj)
                  'bigfixnum
                  'bignum))
               ((and (rational? obj) (exact? obj))
                'ratnum)
               ((and (inexact? (targ-real-part obj))
                     (exact? (targ-imag-part obj))
                     (zero? (targ-imag-part obj)))
                'flonum)
               (else
                'cpxnum)))
        ((u8vect? obj)
         'u8vector)
        ((u16vect? obj)
         'u16vector)
        ((u32vect? obj)
         'u32vector)
        ((f32vect? obj)
         'f32vector)
        ((f64vect? obj)
         'f64vector)
        ((vector? obj)
         'vector)
        ((string? obj)
         'string)
        (else
         (compiler-internal-error
           "targ-obj-subtype, unknown object 'obj'" obj))))

; Note: The following hashing function must return the same value as the
; one in "lib/setup.c".

(define (targ-hash str)

  (define two^32 4294967296) ; to emulate C's unsigned 32 bit arithmetic
  (define two^24 16777216)
  (define two^8 256)

  (define (xor x y)
    (cond ((= x 0)
           y)
          ((= y 0)
           x)
          (else
           (+ (* 2 (xor (quotient x 2) (quotient y 2)))
              (if (eq? (odd? x) (odd? y)) 0 1)))))

  (let ((len (string-length str)))
    (let loop ((h 0) (i 0))
      (if (< i len)
        (let ((n (modulo (+ (* h two^8)
                            (character->unicode (string-ref str i)))
                         two^32)))
          (loop (xor (quotient n two^24) n)
                (+ i 1)))
        (modulo h (+ (max-fixnum32) 1))))))
