/* GiMd2Viewer - Quake2 model viewer
 * Copyright (C) 1998  Lionel ULMER <bbrox@mygale.org>
 *
 * Based on code Copyright (C) 1997 Trey Harrison <trey@crack.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <GL/gl.h>
#include <GL/glu.h>

#include "global.h"
#include "model.h"

extern zvVertex3f(float a, float b, float c) ;

typedef struct {
  float x, y, z;
} Vec3;

typedef struct {
  char ident[4];
  int version;
  int skinwidth;
  int skinheight;
  int framesize;    /* byte size of each frame*/
  int num_skins;
  int num_xyz;
  int num_st;       /* greater than num_xyz for seams*/
  int num_tris;
  int num_glcmds;   /* dwords in strip/fan command list*/
  int num_frames;
  int ofs_skins;    /* each skin is a MAX_SKINNAME string */
  int ofs_st;       /* byte offset from start for stverts */
  int ofs_tris;     /* offset for dtriangles */
  int ofs_frames;   /* offset for first frame */
  int ofs_glcmds;
  int ofs_end;      /* end of file */
} ModelHeader;

typedef struct {
  u_char x;
  u_char y;                /* X,Y,Z coordinate, packed on 0-255 */
  u_char z;
  u_char lightnormalindex; /* index of the vertex normal */
} Trivertex;

typedef struct {
  Vec3 scale;          /* multiply byte verts by this */
  Vec3 origin;         /* then add this */
  char name[16];       /* frame name from grabbing */
  Trivertex verts[1];  /* variable sized */
} FrameInfo;

static unsigned char buffer[512];
static int buf_pos = 0;
static int buf_len = 0;

static inline
unsigned char http_getbyte(void *handle)
{
  if (buf_pos >= buf_len) {
    buf_len = httpRead(handle, buffer, 512);
    buf_pos = 0;
  }
  return buffer[buf_pos++];
}

static inline
unsigned int http_getint(void *handle)
{
  unsigned int ret;
  
  if (buf_pos >= buf_len) {
    buf_len = httpRead(handle, buffer, 512);
    buf_pos = 0;
  }
  ret = buffer[buf_pos++];

  if (buf_pos >= buf_len) {
    buf_len = httpRead(handle, buffer, 512);
    buf_pos = 0;
  }
  ret |= (buffer[buf_pos++] << 8);

  if (buf_pos >= buf_len) {
    buf_len = httpRead(handle, buffer, 512);
    buf_pos = 0;
  }
  ret |= (buffer[buf_pos++] << 16);

  if (buf_pos >= buf_len) {
    buf_len = httpRead(handle, buffer, 512);
    buf_pos = 0;
  }
  ret |= (buffer[buf_pos++] << 24);
  return ret;
}

static inline
float http_getfloat(void *handle)
{
  unsigned int nb = http_getint(handle);
  return *((float *) &nb);
}

static inline
unsigned int http_getbuffer(void *handle, unsigned char *buf, int maxlength)
{
  int in_buffer = buf_len - buf_pos;

  if (in_buffer >= maxlength) {
    memcpy(buf, buffer, maxlength);
    buf_pos += maxlength;

    return maxlength;
  }
  else {
    memcpy(buf, buffer, in_buffer);
    buf_pos = buf_len;
    return in_buffer + httpRead(handle, buf + in_buffer, maxlength - in_buffer);
  }
}

static inline
unsigned int http_skip(void *handle, int length)
{
  int in_buffer = buf_len - buf_pos;

  if (in_buffer >= length) {
    buf_pos += length;

    return 0;
  } else {
    length -= in_buffer;

    while (length > 0) {
      buf_len = httpRead(handle, buffer, 512);

      if (buf_len == 0)
	break;
      if (length >= buf_len) {
	length -= buf_len;
	buf_pos = buf_len;
      } else {
	buf_pos = length;
	length = 0;
      }
    }
    return length;
  }
}

void DrawModel(Model *mdl, int frame, int nextframe, float ratio, float scale)
{
  int nb_vert;
  long *command;
  Frame *frame1, *frame2;
  
  /* Keeps the frame value valid */
  frame = frame % mdl->numframes;
  nextframe = nextframe % mdl->numframes;
  
  /* Gets the frames information */
  frame1 = mdl->frames + frame;
  frame2 = mdl->frames + nextframe;
  
  /* Do the gl commands */
  command = mdl->glcmds;
  nb_vert = 0;
  while (*command != 0) {
    int num_verts, i;
    
    /* Determine the command to draw the triangles */
    if (*command > 0) {
      /* Triangle strip */
      num_verts = *command;
      glBegin(GL_TRIANGLE_STRIP);
    } else {
      /* Triangle fan */
      num_verts = -(*command);
      glBegin(GL_TRIANGLE_FAN);
    }
    command++;
    
    for (i = 0; i < num_verts; i++) {
      Vec3 p;  /* Interpolated point */
      int vert_index;

      /* Grab the vertex index */
      vert_index = *command; command++;
      
      /* Interpolate */
      p.x = frame1->vert_table[vert_index].x +
	(frame2->vert_table[vert_index].x -
	 frame1->vert_table[vert_index].x) * ratio;
      p.y = frame1->vert_table[vert_index].y +
	(frame2->vert_table[vert_index].y -
	 frame1->vert_table[vert_index].y) * ratio;
      p.z = frame1->vert_table[vert_index].z +
	(frame2->vert_table[vert_index].z -
	 frame1->vert_table[vert_index].z) * ratio;
      
      glTexCoord2f(mdl->texinfo[nb_vert].s, mdl->texinfo[nb_vert].t);
      zvVertex3f(p.x * scale, p.y * scale, p.z * scale);

      nb_vert++;
    }
    glEnd();
  }
}

static
int GetFrames(ModelHeader *mdl_header, Model *mdl, void *handle)
{
  FrameInfo *frames;
  int frame;
  int offset = 0;

  trace(DBG_ZV, "Getting Frames");
  
  /* Converts the FrameInfos to Frames */
  mdl->frames = (Frame *) malloc(sizeof(Frame) * mdl->numframes);
  for (frame = 0; frame < mdl->numframes; frame++) {
    Vec3 scale, origin;
    int i;

    scale.x = http_getfloat(handle); offset += 4;
    scale.y = http_getfloat(handle); offset += 4;
    scale.z = http_getfloat(handle); offset += 4;
    origin.x = http_getfloat(handle); offset += 4;
    origin.y = http_getfloat(handle); offset += 4;
    origin.z = http_getfloat(handle); offset += 4;
    
    http_getbuffer(handle, (char *) &(mdl->frames[frame].name), 16);
    offset += 16;
    mdl->frames[frame].vert_table = (Vertex *) malloc(sizeof(Vertex) *
						      mdl_header->num_xyz);

    /* Loads the vertices */
    for (i = 0; i < mdl_header->num_xyz; i++) {
      Trivertex cur_vert;
      Vertex *p = (mdl->frames[frame].vert_table) + i;
      
      cur_vert.x = http_getbyte(handle); offset += 1;
      cur_vert.y = http_getbyte(handle); offset += 1;
      cur_vert.z = http_getbyte(handle); offset += 1;
      cur_vert.lightnormalindex = http_getbyte(handle); offset += 1;
      
      p->x = ((cur_vert.x * scale.x) + origin.x);
      p->y = ((cur_vert.y * scale.y) + origin.y);
      p->z = ((cur_vert.z * scale.z) + origin.z);
    }
  }
  return offset;
}

static
int GetGLCmds(ModelHeader *mdl_header, Model *mdl, void *handle)
{
  long command, *cmd_copy;
  TexInfo *texinfo;
  int num_vertices;
  int offset = 0;

  trace(DBG_ZV, "Getting GL commands");
  
  /* We keep only the commands and the index in the glcommands. We
     'pre-parse' the texture coordinates */
  /* Do not ask me how I found this formula :-)) */
  num_vertices = ((mdl_header->num_tris + 2 * mdl_header->num_glcmds - 2) / 7);
  mdl->texinfo = (TexInfo *) malloc(sizeof(TexInfo) * num_vertices);
  mdl->glcmds = (long *)
    malloc(sizeof(long) * (mdl_header->num_glcmds - 2 * num_vertices));
  
  /* Now transform the GL commands */
  cmd_copy = mdl->glcmds;
  texinfo = mdl->texinfo;
  while (1) {
    int nb_verts, i;

    command = http_getint(handle); offset += 4;
    if (command == 0)
      break;
    
    /* Determine the command to draw the triangles */
    if (command > 0)
      /* Triangle strip */
      nb_verts = command;
    else
      /* Triangle fan */
      nb_verts = -command;
    *(cmd_copy++) = command;
    
    for (i = 0; i < nb_verts; i++) {
      /* Gets the texture information */
      texinfo->s = http_getfloat(handle); offset += 4;
      texinfo->t = http_getfloat(handle); offset += 4;
      texinfo++;
      
      /* We keep the vertex index */
      command = http_getint(handle); offset += 4;
      *(cmd_copy++) = command;
    }
  }
  /* Do not forget to copy the zero :-) */
  *(cmd_copy++) = command;
  return offset;
}

Private
int ModelLoad_callback(Model **mdl, void *handle)
{
  int frame;
  ModelHeader mdl_header;
  int offset = 0;
  
  trace(DBG_ZV, "Loading model...");
  /* Failsafe... */
  if (handle == NULL) {
    *mdl = NULL;
    return 0;
  }

  /* Read the header */
  http_getbuffer(handle, (char *) &(mdl_header.ident), 4); offset += 4;
  mdl_header.version = http_getint(handle); offset += 4;
  mdl_header.skinwidth = http_getint(handle); offset += 4;
  mdl_header.skinheight = http_getint(handle); offset += 4;
  mdl_header.framesize = http_getint(handle); offset += 4;
  mdl_header.num_skins = http_getint(handle); offset += 4;
  mdl_header.num_xyz = http_getint(handle); offset += 4;
  mdl_header.num_st = http_getint(handle); offset += 4;
  mdl_header.num_tris = http_getint(handle); offset += 4;
  mdl_header.num_glcmds = http_getint(handle); offset += 4;
  mdl_header.num_frames = http_getint(handle); offset += 4;
  mdl_header.ofs_skins = http_getint(handle); offset += 4;
  mdl_header.ofs_st = http_getint(handle); offset += 4;
  mdl_header.ofs_tris = http_getint(handle); offset += 4;
  mdl_header.ofs_frames = http_getint(handle); offset += 4;
  mdl_header.ofs_glcmds = http_getint(handle); offset += 4;
  mdl_header.ofs_end = http_getint(handle); offset += 4;
  
  /* Check if this is really a .MD2 file :-) */
  if (strncmp(mdl_header.ident, "IDP2", 4)) {
    *mdl = NULL;
    return 0;
  }
  
  /* Create the model */
  (*mdl) = (Model *) malloc(sizeof(Model));
  memset((*mdl), 0, sizeof(Model));
  
  /* We do not need all the info from the header, just some of it*/
  (*mdl)->numframes = mdl_header.num_frames;
  (*mdl)->skinwidth = mdl_header.skinwidth;
  (*mdl)->skinheight = mdl_header.skinheight;

  /* Check which info is first */
  if (mdl_header.ofs_frames > mdl_header.ofs_glcmds) {
    http_skip(handle, mdl_header.ofs_glcmds - offset);
    offset = mdl_header.ofs_glcmds;
    offset += GetGLCmds(&mdl_header, *mdl, handle);
    http_skip(handle, mdl_header.ofs_frames - offset);
    offset = mdl_header.ofs_frames;
    offset += GetFrames(&mdl_header, *mdl, handle);
  } else {
    http_skip(handle, mdl_header.ofs_frames - offset);
    offset = mdl_header.ofs_frames;
    offset += GetFrames(&mdl_header, *mdl, handle);
    http_skip(handle, mdl_header.ofs_glcmds - offset);
    offset = mdl_header.ofs_glcmds;
    offset += GetGLCmds(&mdl_header, *mdl, handle);
  }
  return 0;
}

/* Frees the memory used by a model */
void FreeModel(Model *mdl)
{
  int frame;

  for (frame = 0; frame < mdl->numframes; frame++)
    free(mdl->frames[frame].vert_table);
  free(mdl->frames);
  free(mdl->glcmds);
  free(mdl->texinfo);
  free(mdl);
}

int GetModelDList(Model *mdl, int frame, int nextframe, float inter, float scale)
{
  int ret;

  if (mdl == NULL)
    return -1;
  ret = glGenLists(1);
  glNewList(ret, GL_COMPILE);
  glCullFace(GL_FRONT);
  DrawModel(mdl, frame, nextframe, inter, scale);
  glEndList();
  return ret;
}

Model *GetModel(char *url)
{
  Model *model;
  
  httpOpen(url, (void *) ModelLoad_callback, &model, NO_BLOCK);
  return model;
}
