#include "global.h"
#include "net_shm.h"
#include "stat.h"


/****************************/
/*** send commands        ***/ 
/****************************/

/*
 * envoi d'un message de type 'd' = Delta, sur le mcast
 * concernant la propriete prop_id de l'objet donne.
 * On fait le resetDates (pas version) 
 * Format actuel d'un Delta: 
 * 'd', nom de l'objet (n), no de propriete (h), version (h),
 * donnees du world.
 */
Private
void sendDelta(NetObject *pnoh, u_int8 prop_id)
{
  Property *pprop;  
  Payload pl;
  Channel *pchannel = channels_list;

  assert(pnoh); assert(pnoh->prop);
  pprop = pnoh->prop + prop_id;
  pprop->responsible = 1;
  resetDates(pprop);
  initPayload(&pl);
#ifdef OLD_PAYLOAD
  writePayload(&pl, "cnhh", DELTA, pnoh->noid, prop_id, pprop->version);
#else
  writePayload(&pl, "cnch", DELTA, pnoh->noid, prop_id, pprop->version);
#endif

  /*** TODO: the timevals ***/

  assert(prop_id < countProperties(pnoh->type));
  getProperty(pnoh, prop_id, &pl);
  trace(DBG_NET, "send delta: port=%d", pchannel->sa[SOCK_RTP]->sin_port);
  sendPacket(pchannel->sa[SOCK_RTP], &pl);
}

/*
 * send a 'C' to mentioned address for the mentionned object
 * we don't touch version and dates
 * C format:
 * 'C', object type (c), object name (n), permanent (c)
 * then the initialization payload, obtained by a getAllProperties()
 * then the version vector (nprop *pnoh)
 */
Private
void sendCreate(struct sockaddr_in *to, NetObject *pnoh)
{
  Payload pl;
  int i;
  u_int8 nprop;

  initPayload(&pl);
#ifdef OLD_PAYLOAD
  writePayload(&pl, "cdnc", CREATE, pnoh->type, pnoh->noid, pnoh->permanent);
#else
  writePayload(&pl, "ccnc", CREATE, pnoh->type, pnoh->noid, pnoh->permanent);
#endif
  getAllProperties(pnoh, &pl);  
  nprop = countProperties(pnoh->type);
  for (i = 0; i < nprop; i++)
    writePayload(&pl, "h", pnoh->prop[i].version);
#ifdef DEBUG_NET
  notice("sendCreate on object: %s", nameByID(pnoh->noid));
#else
  trace(DBG_NET, "sendCreate on object: %s", nameByID(pnoh->noid));
#endif

  trace(DBG_NET, "send Create: port=%d", to->sin_port);
  sendPacket(to, &pl);
}

/*
 * send a query '?'
 * we try to avoid to send too many
 * Format:
 * '?', object name (n)
 */
Private
void sendQuery(struct sockaddr_in *to, NetObjectId noid)
{
  Payload pl;
  static NetObjectId oldnoid;
  static int count = 0;

  /* Heuristique pour eviter d'envoyer des rafales de ? */
  if (equalNetObjectId(noid, oldnoid)) { 
    /* last request was on this name */
    count++;
    if (count <= 15)
      return;    /* cancel this request */
    /* 15: 10 proprietes en moyenne, avec 15 on saute donc
     * sans doute un "bloc" de deltas, mais sans doute pas deux 
     */ 
  } else { /* it's another one */
    oldnoid = noid;
  }
  count = 0;
  initPayload(&pl);
  writePayload(&pl, "cn", QUERY, noid);
#ifdef DEBUG_NET
  notice("sendQuery on object: %s", nameByID(noid));
#else
  trace(DBG_NET, "sendQuery on object: %s", nameByID(noid));
#endif

  trace(DBG_NET, "send Query: port=%d", to->sin_port);
  sendPacket(to, &pl);
}

/*
 * send a Delete 'D'
 * Format: 'D', object name (n) 
 */
Private
void sendDelete(struct sockaddr_in *to, NetObjectId noid)
{
  Payload pl;

  initPayload(&pl);
  writePayload(&pl, "cn", DELETE, noid);
#ifdef DEBUG_NET
  notice("sendDelete on object: %s", nameByID(noid));
#else
  trace(DBG_NET, "sendDelete on object: %s", nameByID(noid));
#endif

  trace(DBG_NET, "send Delete: port=%d", to->sin_port);
  sendPacket(to, &pl);
} 


/*****************************************/
/*** functions "declare" our "request" ***/
/*****************************************/

/*
 * we assume the header yet initialized with createNetObject()
 * Exported to wmgt, to call for each new objects
 */
Public
void declareCreation(NetObject *pnoh)
{
  Channel *pchannel = channels_list;

  assert(pnoh);
  if (getNetObjectId(pnoh->noid) == NULL) {
    warning("declareCreation: unnamed or deleted object (type=%d)", pnoh->type);
    return;
  }
  if (ntohl(pnoh->noid.host_id) == 1) {
    warning("declareCreation: not a new object (type=%d)", pnoh->type);
    return;
  }
  sendCreate(pchannel->sa[SOCK_RTP], pnoh);
}  

/*
 * Update object version
 * Exported to wmgt, to call at each modification
 */
Public
void declareDelta(NetObject *pnoh, u_int8 prop_id)
{
  Property *pprop;  

  assert(pnoh); 
  if (pnoh->noid.host_id == 0) {
    warning("declareDelta: unnamed object (type=%d)", pnoh->type);
    return;
  }
  if (prop_id >= countProperties(pnoh->type)) {
    warning("declareDelta: invalid property_id=%d (type=%d)",
             prop_id, pnoh->type); 
    return;
  }
  assert(pnoh->prop);
  pprop = pnoh->prop + prop_id;
  pprop->responsible = 1;
  pprop->version += 1 + abs(rand() % MAX_VERSION_JUMP); /* %10 */
  sendDelta(pnoh, prop_id);
}

/*
 * Exported to wmgt 
 * Destroy the object (local copy)
 * object must be valid (nom & co) in input
 * Au retour, il faut faire un deleteNetObject et le free() final.
 */
Public
void declareDeletion(NetObject *pnoh)
{
  Channel *pchannel = channels_list;

  assert(pnoh);
  if (!getNetObjectId(pnoh->noid)) {
    warning("declareDeletion: unnamed/already deleted object %d", pnoh->type);
    return;
  }
  if (pnoh->permanent) {
    warning("declareDeletion: on permanent object (type=%d)", pnoh->type);
    return;
  }
  sendDelete(pchannel->sa[SOCK_RTP], pnoh->noid);
}


/****************/	  
/*** incoming ***/
/****************/	  

/*
 * Incoming Delta 'd'
 * content:
 *         format (3 bytes)
 *         noid (8 bytes: 4 + 2 + 2)
 *         prop_id (1 byte)
 *         version_id (2 bytes)
 *         total (14 bytes)
 */
Private
void incomingDelta(struct sockaddr_in *from, Payload *ppl) 
{
  NetObjectId noid;	/* name received */
  NetObject *pnoh;	/* header of object */
  Property *pprop;	/* property found */
  u_int8 prop_id;	/* property number */
  int16 vers_id;	/* version received */
  int16 d;		/* versions difference */
  int nprop;		/* number of properties */

  if (scanPayload(ppl, "nch", &noid, &prop_id, &vers_id) < 0) {
    if (scanPayload(ppl, "nhh", &noid, &prop_id, &vers_id) < 0)
      return;
  }
#ifdef DEBUG_NET
  notice("incomingDelta on: %s, p=%d, v=%d", 
	 nameByID(noid), prop_id, vers_id);
#endif

  if ((pnoh = getNetObjectId(noid)) == NULL) { /* delta on an unknown */
    sendQuery(from, noid);	/* send to source in unicast */
    return;
  }

  /* verify prop_id */
  nprop = countProperties(pnoh->type);
  if (prop_id >= nprop) {
    warning("incomingDelta: invalid property id (%d)"
	    "(type is %d, nprop is %d)", prop_id, pnoh->type, nprop);
    return;
  }
  pprop = pnoh->prop + prop_id;

  /* depends on version ... */
  /* in complement to 2: d gives the distance, same throught the boundary */
  d = pprop->version - vers_id;
  if (abs(d) > 5000) { /* very far */
    warning("incomingDelta: very different versions: mine is %d, received %d",
             pprop->version, vers_id);
  }
  if (d > 0)
    return;	/* mine is more recent */
  resetDates(pprop);
  if (d < 0) {	/* its own is more recent */
    setProperty(pnoh, prop_id, ppl);
    pprop->responsible = 0;	/* leave responsibility to an other one */
    pprop->version = vers_id;
  }
  else if (pprop->responsible) {
    /* same versions, 2 responsibles: conflict
     * resolved by getting a new random version
     */
    declareDelta(pnoh, prop_id); /* publish new version to source in unicast */
    warning("Conflict resolution: object=%s, prop=%d,"
	   " changing version number from %d to %d",
	   nameByID(pnoh->noid), prop_id, vers_id, pnoh->prop[prop_id].version);
  } /* else, it's just a "recall" (then nothing to do) */
}

/*
 * Incoming 'C'
 * content:
 *         format (3 bytes)
 *         type_id (1 byte)
 *         noid (8 bytes: 4 + 2 + 2)
 *         permanent_flag (1 byte)
 *	   properties (nprop * 20 bytes: 2 + 2 + 8 + 8)
 * Create the object's local copy if it doesn't exist
 */
Private
void incomingCreate(struct sockaddr_in *from, Payload *ppl) 
{
  NetObjectId noid;
  NetObject *pnoh;
  u_int8 type_id, permanent;
  int i;
  u_int8 nprop;

  if (scanPayload(ppl, "cnc", &type_id, &noid, &permanent) < 0) {
    if (scanPayload(ppl, "dnc", &type_id, &noid, &permanent) < 0)
      return;
  }
  if (getNetObjectId(noid) != NULL)
    return;		/* local copy already exists -> ignore this request */

#ifdef DEBUG_NET
  notice("incomingCreate on object: %s (type=%d), permanent=%d", 
	 nameByID(noid), type_id, permanent);
#else
  trace(DBG_NET, "incomingCreate on object: %s (type=%d), permanent=%d", 
	 nameByID(noid), type_id, permanent);
#endif
  
  pnoh = createObjectFromNetwork(type_id, noid, ppl);
  assert(pnoh->type == type_id);
  assert(equalNetObjectId(pnoh->noid, noid));
  pnoh->permanent = permanent;
  initProperties(pnoh, 0);	/* 0 == we are not responsible */
  insertNetObject(pnoh);
  nprop = countProperties(type_id);
  for (i = 0; i < nprop; i++) {	/* get properties from network */
    if (scanPayload(ppl, "h", &(pnoh->prop[i].version)) < 0)
      return;
  }
}

/*
 * Incoming '?'
 * content:
 *         format (1 byte)
 *         noid (8 bytes: 4 + 2 + 2)
 *         total (9 bytes)
 */
Private
void incomingQuery(struct sockaddr_in *from, Payload *ppl) 
{
  NetObjectId noid;
  NetObject *pnoh;

  if (scanPayload(ppl, "n", &noid) < 0)
    return;
#ifdef DEBUG_NET
  notice("incomingQuery on object: %s", nameByID(noid));
#else
  trace(DBG_NET, "incomingQuery on object: %s", nameByID(noid));
#endif
  if ((pnoh = getNetObjectId(noid)) == NULL)
    /* unknown object: may be we have deleted it, we advertize the requester */
    sendDelete(from, noid);
  else
    sendCreate(from, pnoh);
}

/*
 * Incoming 'D'
 * content:
 *         format (1 byte)
 *         noid (8 bytes: 4 + 2 + 2)
 *         total (9 bytes)
 */
Private
void incomingDelete(struct sockaddr_in *from, Payload *ppl) 
{
  NetObjectId noid;
  NetObject *pnoh;

  if (scanPayload(ppl, "n", &noid) < 0)
    return;
#ifdef DEBUG_NET
  notice("incomingDelete on object: %s", nameByID(noid));
#else
  trace(DBG_NET, "incomingDelete on object: %s", nameByID(noid));
#endif
  if (pnoh = getNetObjectId(noid))
     requestDeletionFromNetwork(pnoh);
}


/*
 * When something is available on file-descriptor "on"
 * Exported to gui
 */
Public
void incoming(int on) 
{
  fd_set set;
  struct timeval timeout;
  Payload pl;
  struct sockaddr_in from;	/* source unicast addesses */
  u_int8 cmd;

  while (1) {
    FD_ZERO(&set);
    FD_SET(on, &set);
    timeout.tv_sec = timeout.tv_usec = 0;
    if (select(FD_SETSIZE, &set, NULL, NULL, &timeout) == 0)
      break;

    if (recvPacket(on, &from, &pl) < 1)	/* read packet */
      return;
    if (my_hostid == ntohl(from.sin_addr.s_addr) &&
        my_portid == ntohs(from.sin_port))
      return;	/* Loopback */
    if (pl.len == 0)
      return;	/* empty payload */
    if (scanPayload(&pl, "c", &cmd) < 0)
      return;	/* invalid payload */

    #ifdef DEBUG_NET
    notice("Incoming from %lx/%x (mine is %lx/%x)",
	   ntohl(from.sin_addr.s_addr), ntohs(from.sin_port),
	   my_hostid, my_portid);
    #endif

    switch (cmd) {
    case CREATE: incomingCreate(&from, &pl); break;
    case DELTA : incomingDelta(&from, &pl); break;
    case QUERY : incomingQuery(&from, &pl); break;
    case DELETE: incomingDelete(&from, &pl); break;
    default: warning("Incoming cmd unknown: x%02x", cmd);
    }
  }
}


/***************/
/*** Timeout ***/ 
/***************/

/*
 * Check if some responsibilities must not to be taken
 * check if some objects must not to be deleted
 * do the refresh. 
 * return the delay in ms before recall
 */
Public
int networkTimeout()
{
  NetObject *pnoh, *next;
  Property *pprop;
  int i;
  u_int8 nprop;
  struct timeval now;
  
  statAdjust();

  gettimeofday(&now, NULL);
  /* scan objects list */
  for (pnoh = objects_list; pnoh != NULL; ) {
    next = pnoh->next;	/* save it now, pnoh may disapear */
    nprop = countProperties(pnoh->type);
    /* scan properties */
    for (i=0; i < nprop; i++) {
      pprop = pnoh->prop + i;
      if (pprop->responsible && diffDates(pprop->last_seen, now) > REFRESH_TIMEOUT) {
	/* we are responsible, we must refresh */
	sendDelta(pnoh, i);
      }

      if (diffDates(pprop->assume_at, now) > 0) {
	/* assume's timeout on this property */
	if (pnoh->permanent) {
	  if (pprop->version != 0) {
	    /* if permanent prop hasn't its initial value,
	     * somebody must assume responsibility */
	    notice("Assuming responsibility for %s, prop_id %d unseen for %5.2fs", 
		    nameByID(pnoh->noid), i, diffDates(pprop->last_seen, now));
	    if (pprop->responsible) {
	      warning("networkTimeout: should assume responsibility "
		    "of something I am responsible for");
              return (-1);
            }

	    declareDelta(pnoh, i);   /* assume responsibility: publish */ 
	  }
	} else { 
	  /* volatile */
	  notice("Assuming death of %s (unseen for %5.2f s)", 
		 nameByID(pnoh->noid), diffDates(pprop->last_seen, now));
	  if (pprop->responsible) {
	    warning("networkTimeout: Should assume death "
		  "of something I am responsible for");
            return (-1);
	  } 
	  declareDeletion(pnoh);
	  requestDeletionFromNetwork(pnoh);  /* elimine le mort */

	  i = nprop;
	  /* no reason to continue on h after requestDeletion */
	} 
      }
    }
    pnoh = next;
  }
  return (int) (ceil(REFRESH_TIMEOUT * 1000));
}

