#include "global.h"
#include "net_shm.h"


/*
 * Do REUSEADDR on the socket
 * return sock if OK, else -1
 */
Private
int setReuseAddr(int sock)
{
  int one = 1;
  
  if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, 
		 (char *) &one, sizeof(one)) < 0) {
    warning("SO_REUSEADDR: %s (%d)", strerror(errno), errno);
    return -1;
  }
  return sock;
}

/*
 * Control blocking(1)/non blocking(0)
 * return sock if OK, else -1
 */
Private
int setBlocking(int sock, int block)
{
  int flags;

  if ((flags = fcntl(sock, F_GETFL)) < 0) {
    warning("F_GETFL: %s (%d)", strerror(errno), errno);
    return -1;
  }
  if (block)
    flags &= ~O_NONBLOCK;
  else
    flags |= O_NONBLOCK;
  if (fcntl(sock, F_SETFL, flags) < 0) {
    warning("F_SETFL: %s (%d)", strerror(errno), errno);
    return -1;
  }
  return sock;
}

/*
 * Set the ttl
 * return sock if OK, else -1
 */
Private
int setScope(int sock, u_int8 ttl)
{
  if (setsockopt(sock, IPPROTO_IP, IP_MULTICAST_TTL, &ttl, sizeof(ttl)) < 0) {
    warning("IP_MULTICAST_TTL: %s (%d)", strerror(errno), errno);    
    return -1;
  }
  return sock;
}

/*
 * Set loopback: active (1) either inactive (0)
 * return sock if OK, else -1
 */
Public
int setLoopback(int sock, u_int8 loop)
{
  if (setsockopt(sock, IPPROTO_IP, IP_MULTICAST_LOOP, 
		 &loop, sizeof(loop)) < 0) {
    warning("IP_MULTICAST_LOOP: %s (%d)", strerror(errno), errno);
    return -1;
  }
  return sock;
}

/*
 * Join group (ADD_MEMBERSHIP)
 * "group" in ip network format
 * return sock if OK, else -1
 */
Private
int joinGroup(int sock, Channel *pchannel)
{
  /*
   * TODO: test if unicast tunnel
   */
  memset((char *) &(pchannel->mreq), 0, sizeof(struct ip_mreq));
  pchannel->mreq.imr_multiaddr.s_addr = pchannel->group;
  pchannel->mreq.imr_interface.s_addr = htonl(INADDR_ANY);
  if (setsockopt(sock, IPPROTO_IP, IP_ADD_MEMBERSHIP, 
		 (void *) &(pchannel->mreq), sizeof(struct ip_mreq)) < 0) {
    warning("IP_ADD_MEMBERSHIP: %s (%d)", strerror(errno), errno);
    return -1;
  }
  return sock;
}

/*
 * Leave group (DROP_MEMBERSHIP)
 * return sock if OK, else -1
 */
Private
int leaveGroup(int sock, Channel *pchannel)
{
  /*
   * TODO: test if unicast tunnel
   */
  if (setsockopt(sock, IPPROTO_IP, IP_DROP_MEMBERSHIP, 
		 (void *) &(pchannel->mreq), sizeof(struct ip_mreq)) < 0) {
    warning("IP_DROP_MEMBERSHIP: %s (%d)", strerror(errno), errno);
    return -1;
  }
  return sock;
}


/*
 * cree une socket d'ecoute de channel designe par group/port.
 * On realise le joinGroup. Prevue pour la reception seulement.
 * group fmt network et port fmt host.
 * Return sock else -1 if problem
 */
Public
/* int createMcastRecvSocket(Channel *pchannel, u_int16 port) */
int createMcastRecvSocket(Channel *pchannel, struct sockaddr_in *sa)
{
  int sock = -1;

  if ((sock = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
    warning("receive socket: %s", strerror(errno));
    return -1;
  }
  if (setReuseAddr(sock) < 0)
    perror("reuse failed");
  if (bind(sock, (struct sockaddr *) sa, sizeof(struct sockaddr_in)) < 0)
    warning("receive bind: %s", strerror(errno));
  if (joinGroup(sock, pchannel) < 0)
    perror("join failed");

  return sock;
}

/*
 * Create Unicast socket
 * return fd, -1 if problem
 */
Public
int createUcastSocket(u_int32 uni_addr, u_int16 port)
{
  int sock = -1;
  struct sockaddr_in ad;

  if ((sock = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
    warning("receive unicast socket: %s", strerror(errno));
    return -1;
  }
  memset((char *) &ad, 0, sizeof(ad));
  ad.sin_family = AF_INET; 
  ad.sin_port = htons(port);
  ad.sin_addr.s_addr = uni_addr;

  setReuseAddr(sock);
  if (bind(sock, (struct sockaddr *) &ad, sizeof(ad)) < 0) {
    warning("receive inicast bind: %s", strerror(errno));
  }
  return sock;
}

/*
 * Create an UDP socket
 * Prevue pour emettre (uni et mcast) et recevoir (unicast donc)
 * Effectue le setScope (le ttl,format host) et setLoopback (off)
 * return sock if OK, else -1
 */
Public
int createMcastSendSocket(u_int8 ttl)
{
  int sock = -1;

  if ((sock = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
    warning("send socket: %s", strerror(errno));    
    return -1;
  }
  setScope(sock, ttl);
  setLoopback(sock, 0);		/* no loopback */
  setBlocking(sock, 0);
  return sock;
}

Public
void closeMcastSocket(Channel *pchannel)
{
  leaveGroup(pchannel->sd[RECV_RTP], pchannel);
  close(pchannel->sd[RECV_RTP]);
  pchannel->sd[RECV_RTP] = -1;
  close(pchannel->sd[SEND_RTP]);
  pchannel->sd[SEND_RTP] = -1;

  leaveGroup(pchannel->sd[RECV_RTCP], pchannel);
  close(pchannel->sd[RECV_RTCP]);
  pchannel->sd[RECV_RTCP] = -1;
  close(pchannel->sd[SEND_RTCP]);
  pchannel->sd[SEND_RTCP] = -1;
}

Public
void closeUcastSocket(Channel *pchannel)
{
  close(pchannel->sd[RECV_UDP]);
  pchannel->sd[RECV_UDP] = -1;
}
