#include "global.h"
#include "defaults.h"
#include "net_shm.h"
#include "rtpsess.h"
#include "stat.h"
#include "channel.h"


/*
 * Decode the string format "group[/port[/ttl]]". 
 * return group, port, ttl 
 * If chan_str == "" or NULL, we get the default values
 */
Private void
decodeChannel(const char *chan_str, u_int32 *group, u_int16 *port, u_int8 *ttl)
{
  char ch[CHAN_LEN];
  char *ttlstr, *portstr, *groupstr;

  assert(group); assert(port); assert(ttl);
  strncpy(ch, chan_str, CHAN_LEN - 1);
  groupstr = ch;
  portstr = strchr(ch, '/');
  if (portstr != NULL)
    *portstr = '\0';
  *group = inet_addr(groupstr);
  if (portstr != NULL) {
    portstr++;
    ttlstr = strchr(portstr, '/');
    if (ttlstr != NULL)
      *ttlstr = '\0';
    *port = atoi(portstr);
    if (ttlstr != NULL) {
      *ttlstr = '\0';
      ttlstr++;
      *ttl = atoi(ttlstr);
    }
  }
}

/*
 * Network Initialization
 */
Public
void ChannelInit()
{
  int i, j;
  static first = 1;

  if (first) {
    gettimeofday(&start_time, NULL);
    first = 0;
  }
  channels_list = NULL;
  sessions_list = NULL;
  objects_list = NULL;

  for (i=0; i < MAX_CHAN; i++) {
    for (j=0; j < MAX_FD; j++) {
      tab_chan_fd[i][j] = -1 ;
    }
  }
  /*
   * TODO
   * getenv("tunnel");
   */
}

/*
 * Alloc a Channel
 */
Public
Channel *
ChannelAlloc()
{
  Channel *pchan;

  if ((pchan = (Channel *) malloc(sizeof(Channel))) == NULL)
    return ((Channel *) NULL);
  memset(pchan, 0, sizeof(Channel));
  if (channels_list == (Channel *) NULL) {
    channels_list = pchan;
    pchan->next = NULL;
  }
  else {
    pchan->next = channels_list;
    channels_list = pchan;
  }
  trace(DBG_RTP, "ChannelAlloc: pchan=%x", pchan);
  return pchan;
}

/*
 * Free a Channel
 */
Public
void ChannelFree(Channel *pchannel)
{
  Channel *pchan;

  trace(DBG_RTP, "ChannelFree: pchannel=%x", pchannel);
  if (pchannel == NULL) {
    trace(DBG_RTP, "ChannelFree: channel already free");
    return;
  }
  for (pchan = channels_list; pchan != NULL; pchan = pchan->next) {
    if (pchan == pchannel) {
      channels_list = pchan->next;
      pchan->next = NULL;
      if (pchan == channels_list)
        channels_list = NULL;
      free(pchan);
      return;
    }
  }
  channels_list = NULL;
}

/*
 * Channel naming
 * init my_hostid, my_portid, new_objectid
 */
Private
void channelNaming(Channel *pchan)
{
  struct sockaddr_in ad;
  int adlen = sizeof(ad);
  Payload pl;			/* dummy payload */
  char hostname[MAXHOSTNAMELEN];
  struct hostent *ph;
  struct in_addr *pa;

  gethostname(hostname, sizeof(hostname));
  ph = gethostbyname(hostname); assert(ph);
  pa = (struct in_addr*) (ph->h_addr); assert(pa);
  my_hostid = pa->s_addr;   
 
  initPayload(&pl);
#ifdef FIRST_PACKET
  sendPacket(pchan->sa[SOCK_RTP], &pl); /* needed for naming (port number) */
#endif
  if (getsockname(pchan->sd[SEND_RTP], (struct sockaddr *) &ad, &adlen) < 0)
    fatal("getsockname: %s", strerror(errno));

  my_portid = ntohs(ad.sin_port);
  trace(DBG_NET, " my_portid = %d", my_portid);
  new_objectid = 0;
}

/*
 * Open a Channel
 * channel string is given by decodeChannel
 * channel structure is given bye Channel
 * return number of fd to survey
 *
 * Usage:
 * int *fds,i;
 * int count = ChannelOpen(pchan, chan_str, &fds);
 * for (i=0; i < count; i++) {
 *   addThisFdToWatchList(fds[i]);
 * }
 * create sockets mcast_recv_rtp and mcast_send_rtp
 *        sockets mcast_recv_rtcp and mcast_send_rtcp
 */
Public
int ChannelOpen(Channel *pchan, const char *chan_str, int **pfds)
{
  u_int8 cntfds = 0;		/* number of fd */
  u_int32 group;		/* IP Mcast addr in network format */
  u_int16 port;			/* port */
  u_int8 ttl;
  
  assert(chan_str); assert(pchan);
  decodeChannel(chan_str, &group, &port, &ttl);
  pchan->group = group;

  port &= ~1;	/* RTP compliant: even port */

  /* RTP channel */
  memset((char *) &(pchan->sin_mcast_rtp), 0, sizeof(struct sockaddr_in));
  pchan->sin_mcast_rtp.sin_family = AF_INET; 
  pchan->sin_mcast_rtp.sin_port = htons(port);
  pchan->sin_mcast_rtp.sin_addr.s_addr = group; 
  pchan->sa[SOCK_RTP] = &(pchan->sin_mcast_rtp);
  if ((pchan->sd[RECV_RTP] = createMcastRecvSocket(pchan, pchan->sa[SOCK_RTP])) < 0)
    fatal("can't open receive socket RTP");
  pchan->sd[SEND_RTP] = createMcastSendSocket(ttl);
  cntfds += 2;
  trace(DBG_RTP, "pchan->sa[0]=%x, pchan->sd[0]=%d, pchan->sd[1]=%d",
        pchan->sa[SOCK_RTP], pchan->sd[RECV_RTP], pchan->sd[SEND_RTP]);
  trace(DBG_RTP, "SOCK_RTP: port=%d", pchan->sa[SOCK_RTP]->sin_port);

  /* RTCP channel */
  memset((char *) &(pchan->sin_mcast_rtcp), 0, sizeof(struct sockaddr_in));
  pchan->sin_mcast_rtcp.sin_family = AF_INET;
  pchan->sin_mcast_rtcp.sin_port = htons(port + 1);	/* not ++port !!! */
  pchan->sin_mcast_rtcp.sin_addr.s_addr = group;
  pchan->sa[SOCK_RTCP] = &(pchan->sin_mcast_rtcp);
  if ((pchan->sd[RECV_RTCP] = createMcastRecvSocket(pchan, pchan->sa[SOCK_RTCP])) < 0)
    fatal("can't open receive socket RTCP");
  pchan->sd[SEND_RTCP] = createMcastSendSocket(ttl);
  cntfds += 2;
  trace(DBG_RTP, "pchan->sa[1]=%x, pchan->sd[2]=%d, pchan->sd[3]=%d",
        pchan->sa[SOCK_RTCP], pchan->sd[RECV_RTCP], pchan->sd[SEND_RTCP]);
  trace(DBG_RTP, "SOCK_RTCP: port=%d", pchan->sa[SOCK_RTCP]->sin_port);

  /* UDP channel */
  pchan->sa[SOCK_UDP] = NULL;

  /* RTP session initialization */
  if ((pchan->session = RtpAllocSession()) == NULL)
    return -1;
  RtpCreateSession(pchan->session, group, port, ttl);

  channelNaming(pchan);

  *pfds = pchan->sd;
  return cntfds;
}

/*
 * Closing all sockets
 */
Public
void ChannelClose(Channel *pchannel)
{
  /* respect this order! */
  RtpSendBye(pchannel);
  statRTP(pchannel->session);
  RtpCloseSession(pchannel->session);
  closeMcastSocket(pchannel);
  ChannelFree(pchannel);
}

/*
 * getfdbysa
 *
 * get a socket fd by sockaddr_in*
 */
Public
int getfdbysa(Channel *pchannel, struct sockaddr_in *sa, int idx_fd)
{
  Channel *pchan;

  assert(pchannel); assert(sa);
  for (pchan = pchannel; pchan != NULL; pchan = pchan->next) {
    if (pchan->sa[SOCK_RTP] == sa || pchan->sa[SOCK_RTCP] == sa) {
      if (pchan->sd[idx_fd] < 0) {
        break;
      }
      return(pchan->sd[idx_fd]);
    }
  }
  pchan = channels_list;
  trace(DBG_RTP, "getfdbysa no match: pchan->sa[0]=%x, pchan->sa[1]=%x, sa=%x, idx_fd=%d fd=%d", pchan->sa[SOCK_RTP], pchan->sa[SOCK_RTCP], sa, idx_fd, pchan->sd[idx_fd]);
  return(-1);
}

/*
 * getsabysa
 *
 * get a socket address by sockaddr_in*
 */
Public
struct sockaddr_in *
getsabysa(Channel *pchannel, struct sockaddr_in *sa, int idx_sa)
{
  Channel *pchan;

  assert(pchannel); assert(sa);
  for (pchan = pchannel; pchan != NULL; pchan = pchan->next) {
    if (pchan->sa[SOCK_RTP] == sa || pchan->sa[SOCK_RTCP] == sa) {
      return(pchan->sa[idx_sa]);
    }
    trace(DBG_FORCE, "getsabysa error: pchan->sa[0]=%x, pchan->sa[1]=%x, sa=%x, idx_sa=%d pchan->sa[idx_sa]=%x", pchan->sa[SOCK_RTP], pchan->sa[SOCK_RTCP], sa, idx_sa, pchan->sa[idx_sa]);
  }
  return((struct sockaddr_in *) NULL);
}

