/***************************************************************************
                          store.cpp  -  general purpose file storage class
                             -------------------                                         
    begin                : Thu Jun 3 1999                                           
    copyright            : (C) 1999 by David Johnson                         
    email                : arandir@meer.net                                     

    This software licensed under the Berkeley Software Distribution License
 ***************************************************************************/

#include <qobject.h>
#include <qtextstream.h>
#include "store.h"

// This is version 0.3 of the Store class

/////////////////////////////////////////////////////////////////////////////
// Construction

// A minimal default constructor. Should not normally be used
Store::Store()
{
	smode = IO_ReadWrite; // an invalid mode
	sgood = false;
	sfilename = "";
	sgroup = "";
	sname = "";
	svalue = "";
	sversion = "";
	sformat = "";
}

// The typical constructor
Store::Store(int mode, const QString &filename, const QString &format, QString version="")
{
	smode = mode;
	sversion = version;
	sformat = format;
	sfilename = filename;
	sgood = false;
	sgroup = "";
	sname = "";
	svalue = "";
	
	if (!((smode == IO_ReadOnly) || (smode == IO_WriteOnly))) {
		// invalid mode, don't proceed
		return;
	}
	// open file according to mode
	sfile.setName(filename);
	if (!sfile.open(smode)) {
		// error opening file
		sgood = false;
		sfile.close();
	} else {
		// file opened successfully
		sgood = true;
		if (!((smode == IO_ReadOnly) ? getHeader() : putHeader())) {
			// error in header
			sgood = false;
			sfile.close();
		}
	}
}

// The deconstructor
Store::~Store()
{
	// close any open files
	if (sfile.isOpen())
		sfile.close();
}
	
/////////////////////////////////////////////////////////////////////////////
// Utility functions

// Return true if no file error exists
bool Store::good()
{
	if (!sgood) {
		// badness was set somewhere else
		return false;
	}
	// check for file errors
	return (sfile.status() ==  IO_Ok);
}

// Returns the file version as a string, NULL if invalid header
const QString Store::getVersion()
{
	return sversion;
}

// check for a valid header
bool Store::getHeader()
{
	if (smode != IO_ReadOnly) {
		// wrong mode
		return false;
	}
	QString line, token;
	// get first line of text
	sfile.readLine(line, MAX_LENGTH);
	// set up text stream
	QTextStream istr(&line, IO_ReadOnly);
	// Check for comment character
	istr >> token;
	if (token[0] != STORE_COMMENT) {
		warning(QT_TR_NOOP("Store: Missing or bad header in file ") + sfilename + "\n");
		return false;
	}
	// check for format
	istr >> token;
	if (QString::compare(token.lower(), sformat.lower())) { // reminder: with compare, zero equals true
		warning(QT_TR_NOOP("Store: Bad header in file ") + sfilename + "\n");
		return false;
	}
	// check for version keyword
	// may be "v" or "version"
	istr >> token;
	token.lower(); // take it to lower case
	if ((token.compare("v")) && (token.compare("version"))) {
		// no optional version
		sversion = "NONE";
		return true;
	} else {
		// get version (accepting next token as valid)
		istr >> token;
		sversion = token;
		return true;
	}		
}

// write a header to file
bool Store::putHeader()
{
	// check mode
	if (smode != IO_WriteOnly) {
		// wrong mode
		return false;
	}
	// write out the header (no newline)
	QTextStream ostr(&sfile);
	ostr << "# " << sformat;
	if (sversion.length() > 0)
		ostr << " v " << sversion;
	// is everything still ok?
	return (sfile.status() ==  IO_Ok);
}

/////////////////////////////////////////////////////////////////////////////
// Storage functions
	
// Write a new element group
bool Store::putGroup(const QString &group)
{
	if (smode != IO_WriteOnly) {
		// wrong mode
		return false;
	}
	// put group in brackets all by itself
	QTextStream ostr(&sfile);
	ostr << "\n[" << group << "]";
	// is everything still ok?
	return (sfile.status() == IO_Ok);
}

// Write a new line with name and value pair
bool Store::putLine(const QString &name, const QString &value)
{
	if (smode != IO_WriteOnly) {
		// wrong mode
		return false;
	}
	// start a newline
	QTextStream ostr(&sfile);
	ostr << "\n";
	// write name, delimiter and value
	ostr << name << " " << STORE_DELIM << " " << value;
	// is everything still ok?
	return (sfile.status() == IO_Ok);
}

// I don't like this duplication, find out proper way to do it
bool Store::putLine(const QString &name, const int &value)
{
	if (smode != IO_WriteOnly) {
		// wrong mode
		return false;
	}
	// start a newline
	QTextStream ostr(&sfile);
	ostr << "\n";
	// write name, delimiter and value
	ostr << name << " " << STORE_DELIM << " " << value;
	// is everything still ok?
	return (sfile.status() == IO_Ok);
}

// Write a comment line
bool Store::putComment(const QString &comment)
{
	if (smode != IO_WriteOnly) {
		// wrong mode
		return false;
	}
	QTextStream ostr(&sfile);
	ostr << "\n" << STORE_COMMENT << comment;
	// is everything still ok?
	return (sfile.status() == IO_Ok);
}

bool Store::putElement(const QString &element)
{
	if (smode != IO_WriteOnly) {
		// wrong mode
		return false;
	}
	QTextStream ostr(&sfile);
	ostr << ", " << element;
	// is everything still ok?
	return (sfile.status() == IO_Ok);
}

// I don't like this duplication, find out proper way to do it
bool Store::putElement(const int &element)
{
	if (smode != IO_WriteOnly) {
		// wrong mode
		return false;
	}
	QTextStream ostr(&sfile);
	ostr << ", " << element;
	// is everything still ok?
	return (sfile.status() == IO_Ok);
}

/////////////////////////////////////////////////////////////////////////////
// Retrieval functions

// Read in next valid line
QString Store::nextValidLine()
{
	QString line ="";
	QString buffer = "";
	QString token = "";
	bool valid = false;
	while (!valid) {
		// check for eof
		if (sfile.atEnd()) {
			// no more lines
			return "";
		}
		sfile.readLine(line, MAX_LENGTH);
		if (line.length() == 0) {
			// line with no content
			valid = false;
		} else {
			// do some checking on line
			buffer = line;
			QTextStream istr(&buffer, IO_ReadOnly);
			// get first element
			istr >> token;
			if (token.length() == 0) {
				// line must be whitespace only
				valid = false;
			} else if (token[0] == STORE_COMMENT) {
				// it's a comment, ignore line
				valid = false;
			} else {
				// a valid token, thus valid line
				valid = true;
			}
		}
	}
	return line;
}

// Read the next line
bool Store::getLine()
{
	if (smode != IO_ReadOnly) {
		// wrong mode
		return false;
	}
	
	// read in next valid line
	QString line = nextValidLine();
	if (line.length() == 0) {
		//didn't get a valid line, return
		return false;
	}
	// line is now valid
	QString token = "";
	QString element = "";
	QTextStream istr(&line, IO_ReadOnly);
	
	// process name element
	// get first token
	istr >> token;
	if ((token[0] != STORE_DELIM) && (token.length() > 0))
		element = token;
	// get rest of tokens
	istr >> token;
	while ((token[0] != STORE_DELIM) && (token.length() > 0)) {
		element += " ";
		element += token;
		istr >> token;
	}
	// is element a group?
	if ((element[0] == '[') && (element[element.length()-1] == ']')) {
		//get rid of brackets
		int len = element.length();
		element = element.mid(1, len-2);
		sgroup = element;
	}
	// set name to element
	sname = element;
	// everything else is now the value element
	element = "";
	// get next token
	istr >> token;
	if (token.length() > 0)
		element = token;
	// get rest of tokens
	istr >> token;
	while (token.length() > 0) {
		element += " ";
		element += token;
		istr >> token;
	}
	svalue = element;	
	
	return true;
}

// Return the current element group
const QString Store::getGroup()
{
	if (smode != IO_ReadOnly) {
		// wrong mode
		return NULL;
	}
	return sgroup;
}

// Return the current name element
const QString Store::getName()
{
	if (smode != IO_ReadOnly) {
		// wrong mode
		return NULL;
	}
	return sname;
}

// Return the current value element
const QString Store::getValue()
{
	if (smode != IO_ReadOnly) {
		// wrong mode
		return NULL;
	}
	return svalue;
}	







