/* GKrellM
|  Copyright (C) 1999 Bill Wilson
|
|  Author:	Bill Wilson		bill@gkrellm.net
|  Latest versions might be found at:
|		http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation, Inc.,
|  59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
|
*/
/* 10/30/99	
 Added check_maildir() function contributed by Patrick Crosby <xb@dotfiles.com>
*/

#include "gkrellm.h"
#include <utime.h>
#include <dirent.h>

#define MUTE_FLAG	-1

/* msg_count_mode has 3 states
*/
#define	MSG_NEW_TOTAL_COUNT	0
#define	MSG_NEW_COUNT		1
#define	MSG_NO_COUNT		2

/* animation_select_mode states
*/
#define	ANIMATION_NONE			0
#define	ANIMATION_ENVELOPE		1
#define	ANIMATION_PENGUIN		2
#define	ANIMATION_BOTH			3

typedef struct
	{
	gchar	*path;
	time_t	last_mtime;
	off_t	last_size;
	gint	mail_count;
	gint	new_mail_count;
	gint	old_mail_count;
	gint	need_animation;
	}
	Mailbox;

static GList	*mailbox_list;

typedef struct
	{
	gchar	*command;
	FILE	*pipe;
	gint	timeout;
	gint	messages;
	gint	new;
	gint	seen;
	}
	Mailproc;

static Mailproc	mail_user_agent,
				mail_fetch;			/* fetchmail	*/
static gchar	*mail_notify;		/* Sound		*/

static gint		run_animation,
				fetch_animation,
				decal_frame;

static gint		mail_fetch_timeout = 5;			/* Minutes */
static gint		mail_check_timeout = 4;			/* Seconds */

static Panel	mailcheck;

static gint		mailcheck_visible = FALSE;
static gint		mute_mode;
static gint		mua_inhibit_mode;		/* Inhibit checking if MUA launched */
static gint		msg_count_mode;
static gint		animation_select_mode	= ANIMATION_BOTH;
static gint		fetch_check_only_mode;
static gint		reset_remote_mode;
static gint		force_mail_fetch;
static gint		new_mail_count, total_mail_count;
static gint		check_timeout;

static gint		anim_frame,
				anim_dir,
				anim_pause;



  /* Look at a From line to see if it is valid, lines look like:
  |  From sending_address dayofweek month dayofmonth timeofday year
  |  eg: From bill@gkrellm.net Fri Oct 22 13:52:49 1999
  */
static gint
is_From_line(gchar *buf)
	{
	gint	dayofmonth = 0;

	if (strncmp(buf, "From ", 5))
		return FALSE;

	/* In case sending address missing, look for a day of month
	|  number in field 3 or 4 (0 based).
	*/
	if (sscanf(buf, "%*s %*s %*s %d", &dayofmonth) != 1)
		if (sscanf(buf, "%*s %*s %*s %*s %d", &dayofmonth) != 1)
			return FALSE;
	if (dayofmonth < 1 || dayofmonth > 31)
		return FALSE;
	return TRUE;
	}


  /* 
  |  Modified by Patrick Crosby <xb@dotfiles.com> 10/28/1999
  |  
  |  check_maildir checks a Maildir directory for new/old mail.
  |  
  |  mbox->path should be the path to the Maildir directory
  |  (e.g. /home/pcrosby/Maildir)
  */
static gint
check_maildir(Mailbox *mbox)
	{
	gchar			buf[256];
	struct dirent	*entry;
	DIR 			*folder_new, *folder_cur;
	gint			i;

	snprintf(buf, sizeof(buf), "%s/new", mbox->path);
	if (!(folder_new = opendir(buf)))
		return FALSE;

	snprintf(buf, sizeof(buf), "%s/cur", mbox->path);
	if (!(folder_cur = opendir(buf)))
		{
		closedir(folder_new);
		return FALSE;
		}

	mbox->mail_count = 0;
	mbox->old_mail_count = 0;

	while ((entry = readdir(folder_new)) != NULL)
		{
		gchar *tmp		= entry->d_name;
		gint is_message = TRUE;

		/* all messages in a maildir folder have their first */
		/* 9 chars as digits */
		for (i = 0; i < 9; i++)
			{
			if (isdigit(*tmp++) == 0)
				is_message = FALSE;
			}
		if (is_message)
			{
			mbox->mail_count++;
			}
		}

	while ((entry = readdir(folder_cur)) != NULL)
		{
		gchar *tmp		= entry->d_name;
		gint is_message = TRUE;

		/* all messages in a maildir folder have their first */
		/* 9 chars as digits */
		for (i = 0; i < 9; i++)
			{
			if (isdigit(*tmp++) == 0)
				is_message = FALSE;
			}
		if (is_message)
			{
			mbox->mail_count++;
			mbox->old_mail_count++;
			}
	}

	closedir(folder_new);
	closedir(folder_cur);
	mbox->new_mail_count = mbox->mail_count - mbox->old_mail_count;
    return True;
	}

  /* Count total mail and old mail in a mailbox.  Old mail can be read
  |  with a Status: R0, or can be accessed and not read with Status: O
  |  So, new mail will be the diff - note that unread mail is not
  |  necessarily new mail.  According to stat() man page:
  |  st_atime is changed by mknod(), utime(), read(), write(), truncate()
  |  st_mtime is changed by mknod(), utime(), write()
  |  But, new mail arriving (writing mailbox) sets st_mtime while reading
  |  the mailbox (mail program reading) sets st_atime.  So the test
  |  st_atime > st_mtime is testing if mbox has been read since last new mail.
  |  Mail readers may restore st_mtime after writting status.
  */
static gint
check_mailbox(Mailbox *mbox, struct stat *s)
	{
	FILE			*f;
	struct utimbuf	ut;
	gchar			buf[1024];
	gint			n;
	gint			in_header	= FALSE;

	/* Message no-counting mode reports new mail based on mailbox
	|  modified time and size.
	*/
	if (msg_count_mode == MSG_NO_COUNT)
		{
		if (   s->st_size > 0
			&& s->st_size >= mbox->last_size
			&& s->st_mtime >= s->st_atime
		   )
			mbox->new_mail_count = TRUE;
		else
			mbox->new_mail_count = FALSE;
		mbox->mail_count = (s->st_size > 0) ? 1 : 0;	/* boolean, not used */
		mbox->old_mail_count = 0;			/* not used */
		mbox->last_size = s->st_size;
		mbox->last_mtime = s->st_mtime;
		return TRUE;
		}

	/* If the mailboxes have been modified since last check, count
	|  the new/total messages.
	*/
	if (s->st_mtime != mbox->last_mtime || s->st_size != mbox->last_size)
		{
		if ((f = fopen(mbox->path, "r")) == NULL)
			return FALSE;
		mbox->mail_count = 0;
		mbox->old_mail_count = 0;
		while(fgets(buf, sizeof(buf), f))
			{
			if (buf[0] == '\n')
				in_header = FALSE;
			else if (is_From_line(buf))
				{
				mbox->mail_count += 1;
				in_header = TRUE;
				}
			else if (   in_header
					 && strncmp(buf, "Status:", 7) == 0
					 && (strchr(buf, 'R') || strchr(buf, 'O'))	/* R0 0R O */
					)
				mbox->old_mail_count += 1;
			}
		fclose(f);

		/* Restore the mbox stat times for other mail checking programs and
		|  so the (st_atime > st_mtime) animation override below will work.
		*/
		ut.actime = s->st_atime;
		ut.modtime = s->st_mtime;
		utime(mbox->path, &ut);

		mbox->last_mtime = s->st_mtime;
		mbox->last_size = s->st_size;
		}
	/* Set the animation state when new mail count changes, and override
	|  the animation to false if mbox has been accessed since last modify
	|  (A MUA has probably read the mbox).
	*/
	n = mbox->mail_count - mbox->old_mail_count;
	if (n != mbox->new_mail_count)
		mbox->need_animation = (n > mbox->new_mail_count);
	if (s->st_atime > s->st_mtime)
		mbox->need_animation = FALSE;
	mbox->new_mail_count = n;

    return True;
	}

static Decal	mail_label_decal;
static Decal	mail_icon_decal;

static void
draw_mail_label_decal(gint new_mail_count, gint mail_count)
	{
	GdkFont		*font;
	gint		x, w;
	Panel		*p;
	Decal		*d;
	Style		*style;
	glong		new_value;
	gchar		buf[32], nbuf[16], tbuf[16];

	p = &mailcheck;
	d = &mail_label_decal;

	/* Decals have only a single value, but I need a two valued decal here.
	|  So, just pack the two numbers and use that as a test for value.
	*/
	new_value = (mail_count << 16) | new_mail_count;
	if (new_value == d->value)
		return;
	d->value = new_value;
	d->modified = TRUE;

	if (new_mail_count == MUTE_FLAG)
		sprintf(buf, "mute");
	else
		{
		if (msg_count_mode == MSG_NO_COUNT)
			buf[0] = '\0';
		else if (msg_count_mode == MSG_NEW_COUNT)
			{
			if (new_mail_count == 0)
				strcpy(buf, "-");
			else
				snprintf(buf, sizeof(buf), "%d", new_mail_count);
			}
		else /* MSG_NEW_TOTAL_COUNT */
			{
			if (new_mail_count == 0)
				strcpy(nbuf, "-");
			else
				sprintf(nbuf, "%d", new_mail_count);
			if (mail_count == 0)
				strcpy(tbuf, "-");
			else
				sprintf(tbuf, "%d", mail_count);
			snprintf(buf, sizeof(buf), "%s/%s", nbuf, tbuf); 
			}
		}
	font = d->text_style.font;
	w = gdk_string_width(font, buf);
	if (w > d->w)
		{
		font = GK.alt_font;
		w = gdk_string_width(font, buf);
		}
	style = GK.meter_style[MAIL_IMAGE];
	x = UC.chart_width * p->label.position / LABEL_MAX;
	x -= style->border_panel.left + w / 2;
	if (p->label.position >= 50)
		x -= mail_icon_decal.w;
	if (x > d->w - w)
		x = d->w - w;
	if (x < 0)
		x = 0;
	gdk_draw_pixmap(d->pixmap, GK.draw1_GC, p->background,
			d->x, d->y,  0, 0,  d->w, d->h);
	draw_string(d->pixmap, font, &d->text_style.color,
			d->text_style.effect, x, d->y_baseline, buf);
	}


static void
update_krell_animation_frame()
	{
	/* Run the animation.  Just go back and forth with a pause on
	|  frames 1 and full_scale - 1 (frames 0 and full_scale are cut off).
	|  Frame 0 is blank anyway, and frame full_scale is just not used.
	*/
	if (anim_pause-- <= 0)
		{
		if (anim_frame <= 1)
			anim_dir = 1;
		if (anim_frame >= mailcheck.krell->full_scale - 1)
			anim_dir = -1;
		anim_frame += anim_dir;
		anim_frame %= mailcheck.krell->full_scale;
		if (anim_frame == 1 || anim_frame == mailcheck.krell->full_scale - 1)
			anim_pause = 4;
		}
	}

  /* I popen the mail_user_agent so I can know if it is running and
  |  the mail_fetch so I can process fetchmail's output.
  */
static void
pipe_command(Mailproc *mp)
	{
	gchar	buf[128];

	if (mp->pipe)	/* Still running?  */
		{
		if (GK.debug)
			printf("pipe_command: old command still running.\n");
		return;
		}
	if (mp->command == NULL || *(mp->command) == '\0')
		return;
	snprintf(buf, sizeof(buf), "%s 2>&1", mp->command);
	if (GK.debug)
		printf("pipe_command: %s\n", buf);
	if ((mp->pipe = popen(buf, "r")) == NULL)
		return;
	fcntl(fileno(mp->pipe), F_SETFL, O_NONBLOCK);
	}

  /* Read chars from a pipe until no more to read or until a newline.
  |  Return number of chars read or -1 for pipe is closed.
  |  XXX What if I get incomplete lines??
  */
static gint
fgets_pipe(gchar *buf, gint size, Mailproc *mp)
	{
	gint	i, n, count;

	if (mp->pipe == NULL)
		return -1;
	count = 0;
	for (i = 0; i < size - 1;  ++i)
		{
		n = fread(&buf[i], 1, 1, mp->pipe);
		count += n;
		if (n == 1 && buf[i] != '\n')
			continue;
		if (n == 0)
			{
			if (feof(mp->pipe))
				{
				pclose(mp->pipe);
				mp->pipe = NULL;
				if (count == 0)
					count = -1;
				}
			}
		else
			++i;
		break;
		}
	buf[i] = '\0';
	return count;
	}

static gint
mua_is_launched()
	{
	gchar	buf[128];
	gint	n;

	if (mail_user_agent.pipe == NULL)
		return FALSE;
	while ((n = fgets_pipe(buf, sizeof(buf), &mail_user_agent)) > 0)
		;
	return (n < 0) ? FALSE : TRUE;
	}

  /* Read mail_fetch pipe and if fetch_check_only_mode, look for output
  |  of fetchmail -c so I can report new mail still on a remote server.
  |  If no remote mail, I get:
  |		fetchmail: No mail for billw at mail.wt.net
  |  If remote mail i will get lines like:
  |		1 message for billw at mail.wt.net (32743 octets).
  |	  or
  |		26 messages (25 seen) for billw at mail.wt.net
  |  If the remote mail is fetched, I get additional lines like:
  |		reading message 1 of 1 (32743 octets) .................... flushed
  |  Note: above 26 messages (25 seen) should show as 1/26 on panel.
  */

/* Temp variables to accumulate results while fetch program is launched.
*/
static gint	fetch_messages, fetch_seen;

static void
read_mail_fetch()
	{
	gchar	buf[128];
	gint	n, msg, seen;

	while ((n = fgets_pipe(buf, sizeof(buf), &mail_fetch)) > 0)
		{
		if (GK.debug)
			printf("read_mail_fetch(%d): %s\n", fetch_check_only_mode, buf);
		if (fetch_check_only_mode)
			{
			msg = 0;
			seen = 0;
			sscanf(buf, "%d %*s (%d", &msg, &seen);
			if (msg > 0)
				{
				fetch_messages += msg;
				fetch_seen += seen;
				}
			}
		}
	if (n == 0 && GK.debug)
		printf("read_mail_fetch: no read data.\n");

	/* Don't set the new values until fetch program is done.
	*/
	if (fetch_check_only_mode && mail_fetch.pipe == NULL)
		{
		mail_fetch.messages = fetch_messages;
		mail_fetch.seen = fetch_seen;
		mail_fetch.new = mail_fetch.messages - mail_fetch.seen;		
		}
	}


void
update_mailcheck(void)
	{
	Mailbox		*mbox;
	struct stat	s;
	GList		*list;
	gchar		cmd[256];
	gint		prev_fetch_new, prev_new_mail_count;
	static gint	mailcheck_primed;

	if (UC.enable_mailcheck == FALSE || mailbox_list == NULL)
		return;

	if (   force_mail_fetch
		|| (   GK.minute_tick && --mail_fetch.timeout <= 0
			&& (mute_mode && UC.super_mute_mailcheck) == FALSE
		   )
	   )
		{
		if ((mua_is_launched() && mua_inhibit_mode) == FALSE)
			{
			if (GK.debug)
				printf("Launching mail fetch\n");
			fetch_messages = 0;
			fetch_seen = 0;
			pipe_command(&mail_fetch);
			mail_fetch.timeout = mail_fetch_timeout;
			force_mail_fetch = FALSE;
			}
		}
	if (   GK.second_tick && --check_timeout <= 0
		&& (mute_mode && UC.super_mute_mailcheck) == FALSE
		&& (mua_is_launched() && mua_inhibit_mode) == FALSE
	   )
		{
		check_timeout = mail_check_timeout;
		prev_new_mail_count = new_mail_count;
		prev_fetch_new = mail_fetch.new;
		if (mail_fetch.pipe)
			read_mail_fetch();
		new_mail_count = mail_fetch.new;
		total_mail_count = mail_fetch.messages;
		if (mail_fetch.new != prev_fetch_new)
			fetch_animation = (mail_fetch.new > prev_fetch_new);

		run_animation = fetch_animation;
		for (list = mailbox_list; list; list = list->next)
			{
			mbox = (Mailbox *) list->data;
			if (stat(mbox->path, &s) < 0)
				continue;
			if (   (S_ISDIR(s.st_mode) && check_maildir(mbox))
				|| (! S_ISDIR(s.st_mode) && check_mailbox(mbox, &s))
			   )
				{
				total_mail_count += mbox->mail_count;
				new_mail_count += mbox->new_mail_count;
				run_animation |= mbox->need_animation;
				}
			}
		if (mailcheck_primed == FALSE)
			{
			mailcheck_primed = TRUE;
			force_mail_fetch = TRUE;	/* Do fetch at startup 	*/
			return;
			}
		/* Run the notify (sound) command if the new mail count goes up,
		|  and if the various modes permit.
		*/
		if (   (new_mail_count > prev_new_mail_count)
			&& mail_notify && mail_notify[0] != '\0' && mute_mode == FALSE
		   )
			{
			snprintf(cmd, sizeof(cmd), "%s &", mail_notify);
			system(cmd);
			}
		}
	if (mute_mode && (GK.timer_ticks % 15) < 3)	/* Asymmetric blink */
		draw_mail_label_decal(MUTE_FLAG, MUTE_FLAG);
	else
		draw_mail_label_decal(new_mail_count, total_mail_count);

	if (run_animation && (animation_select_mode & ANIMATION_PENGUIN))
		update_krell_animation_frame();
	else
		anim_frame = 0;

	if ((GK.timer_ticks % GK.decal_mail_delay) == 0)
		{
		if (run_animation)
			{
			if (animation_select_mode & ANIMATION_ENVELOPE)
				{
				++decal_frame;
				if (decal_frame >= GK.decal_mail_frames)
					decal_frame = 1;
				}
			else
				decal_frame = 0;
			}
		else
			decal_frame = 1;
		}
	draw_decal(&mailcheck, mailcheck.decal, decal_frame);

	/* All the animation frame drawing is done with the general krell code.
	*/
	mailcheck.krell->previous = 0;
	update_krell(&mailcheck, mailcheck.krell, anim_frame);

	draw_layers(&mailcheck);
	}


static gint
mailcheck_expose_event(GtkWidget *widget, GdkEventExpose *event)
	{
	if (widget == mailcheck.drawing_area)
		{
		gdk_draw_pixmap(widget->window,
		      widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		      mailcheck.pixmap,
		      event->area.x, event->area.y,
		      event->area.x, event->area.y,
		      event->area.width, event->area.height);
		}
	return FALSE;
	}

static gint
cb_mailcheck_panel_click(GtkWidget *widget, GdkEventButton *ev)
	{
	if (ev->button == 1)
		{
		if (reset_remote_mode)
			{
			mail_fetch.messages = 0;
			mail_fetch.new = 0;
			mail_fetch.seen = 0;
			}
		fetch_animation = FALSE;
		run_animation = FALSE;
		if (mua_is_launched() == FALSE)
			pipe_command(&mail_user_agent);
		else
			{
			check_timeout = 0;
			if (GK.debug)
				printf("Mail user agent is already running.\n");
			}
		}
	if (ev->button == 2)
		{
		mute_mode = 1 - mute_mode;
		/* If coming out of mute mode and checking was inhibited, force fetch.
		*/
		if (! mute_mode && UC.super_mute_mailcheck)
			force_mail_fetch = TRUE;
		}
	return TRUE;
	}

static void
add_mailbox(gchar *path)
	{
	Mailbox	*mbox;

	mbox = g_new0(Mailbox, 1);
	mbox->path = g_strdup(path);
	mailbox_list = g_list_append(mailbox_list, mbox);
	}

void
create_mailcheck(GtkWidget *vbox)
	{
	Style		*style;
	Panel		*p;
	Decal		*d;
	gchar		*s, buf[128];
	gint		h0;

	p = &mailcheck;
	if (mail_fetch.command == NULL)
		mail_fetch.command = g_strdup("");
	if (mail_user_agent.command == NULL)
		mail_user_agent.command = g_strdup("");
	if (mail_notify == NULL)
		mail_notify = g_strdup("");

	if (mailbox_list == NULL)
		{
		if ((s = getenv("MAIL")) == NULL)
			if ((s = getenv("USER")) != NULL)
				{
				sprintf(buf, "/var/spool/mail/%s", s);
				s = buf;
				}
		if (s)
			add_mailbox(s);
		}

	style = GK.meter_style[MAIL_IMAGE];

	create_krell("mail", GK.krell_meter_image[MAIL_IMAGE], &p->krell, style);
	p->krell->full_scale = style->krell_depth - 1;

	/* Get vertical space forced by krell height.  Use to center decals.
	*/
	h0 = (p->krell->h - style->border_panel.top - style->border_panel.bottom)
			/ style->krell_depth;

	d = &mail_icon_decal;
	p->decal = d;
	if (style->label_position < 50)
		d->x = UC.chart_width - GK.decal_mail_width - style->border_panel.right;
	else
		d->x = style->border_panel.left;
	d->w = GK.decal_mail_width;
	d->h = GK.decal_mail_height;
	d->y = 0;			/* Don't fit inside of panel borders */
	d->pixmap = GK.decal_mail_pixmap;
	d->mask = GK.decal_mail_mask;
	d->value = -1;			/* Force initial draw */

	/* The new/total mail text needs to be a decal because the text changes
	|  and must be drawn as a layer in update_layers().
	*/
	d->next = &mail_label_decal;
	d = &mail_label_decal;
	d->next = NULL;
	default_textstyle(&d->text_style, TEXTSTYLE_TIME);
	d->y_baseline = gdk_char_height(d->text_style.font, '0');
	d->h = d->y_baseline + d->text_style.effect;
	d->y = style->border_panel.top;		/* Do fit inside of panel borders. */
	if (style->label_position < 50)
		d->x = style->border_panel.left;
	else
		d->x = style->border_panel.left + mail_icon_decal.w;
	d->w = UC.chart_width - mail_icon_decal.w
			- style->border_panel.left - style->border_panel.right;
	d->pixmap = gdk_pixmap_new(top_window->window, d->w, d->h, -1);
	d->value = -1;				/* Force initial draw */
	d->type = DECAL_OPAQUE;		/* Text decals are opaque	*/

	default_textstyle(&p->label.textstyle, TEXTSTYLE_TIME);
	configure_panel(p, NULL, style);
	create_panel_area(vbox, p, GK.bg_meter_image[MAIL_IMAGE]);

	/* Adjust the decals to be centered.
	*/
	mail_icon_decal.y = (p->h - mail_icon_decal.h) / 2;
	mail_label_decal.y = (p->h - mail_label_decal.h) / 2;

	if(UC.enable_mailcheck)
		{
		GK.monitor_height += p->h;
		mailcheck_visible = TRUE;
		}
	else
		gtk_widget_hide(p->hbox);

	gtk_signal_connect(GTK_OBJECT (p->drawing_area), "expose_event",
			(GtkSignalFunc) mailcheck_expose_event, NULL);

	gtk_signal_connect(GTK_OBJECT(p->drawing_area),"button_release_event",
			(GtkSignalFunc) cb_mailcheck_panel_click, NULL);
	}


void
write_mail_config(FILE *f)
	{
	Mailbox	*mbox;
	GList	*list;

	for (list = mailbox_list; list; list = list->next)
		{
		mbox = (Mailbox *) list->data;
		fprintf(f, "mail mailbox %s\n", mbox->path);
		}
	fprintf(f, "mail mua %s\n", mail_user_agent.command);
	fprintf(f, "mail notify %s\n", mail_notify);
	fprintf(f, "mail fetch_command %s\n", mail_fetch.command);
	fprintf(f, "mail fetch_timeout %d\n", mail_fetch_timeout);
	fprintf(f, "mail check_timeout %d\n", mail_check_timeout);
	fprintf(f, "mail msg_count_mode %d\n", msg_count_mode);
	fprintf(f, "mail animation_select_mode %d\n", animation_select_mode);
	fprintf(f, "mail fetch_check_only_mode %d\n", fetch_check_only_mode);
	fprintf(f, "mail reset_remote_mode %d\n", reset_remote_mode);
	fprintf(f, "mail mua_inhibit_mode %d\n", mua_inhibit_mode);
	}

void
load_mail_config(gchar *arg)
	{
	gchar	mail_config[64], item[196];
	gint	n;

	n = sscanf(arg, "%s %[^\n]", mail_config, item);
	if (n == 2)
		{
		if (GK.debug)
			printf("mail_config=%s item=<%s>\n", mail_config, item);
		if (strcmp(mail_config, "mailbox") == 0)
			add_mailbox(item);
		else if (strcmp(mail_config, "mua") == 0)
			mail_user_agent.command = g_strdup(item);
		else if (strcmp(mail_config, "notify") == 0)
			mail_notify = g_strdup(item);
		else if (strcmp(mail_config, "fetch_command") == 0)
			mail_fetch.command = g_strdup(item);
		else if (strcmp(mail_config, "fetch_timeout") == 0)
			sscanf(item, "%d", &mail_fetch_timeout);
		else if (strcmp(mail_config, "check_timeout") == 0)
			sscanf(item, "%d", &mail_check_timeout);
		else if (strcmp(mail_config, "msg_count_mode") == 0)
			sscanf(item, "%d", &msg_count_mode);
		else if (strcmp(mail_config, "animation_select_mode") == 0)
			sscanf(item, "%d", &animation_select_mode);
		else if (strcmp(mail_config, "fetch_check_only_mode") == 0)
			sscanf(item, "%d", &fetch_check_only_mode);
		else if (strcmp(mail_config, "reset_remote_mode") == 0)
			sscanf(item, "%d", &reset_remote_mode);
		else if (strcmp(mail_config, "mua_inhibit_mode") == 0)
			sscanf(item, "%d", &mua_inhibit_mode);
		}
	}


/* ---------------------------------------------------------------------*/
static GtkWidget		*mbox_path_entry;

static GtkWidget		*mail_user_agent_entry;
static GtkWidget		*mail_fetch_entry;
static GtkWidget		*mail_notify_entry;
static GtkWidget		*enable_mailcheck_button;
static GtkWidget		*super_mute_button;
static GtkWidget		*mbox_clist;
static GtkWidget		*mail_fetch_spin_button;
static GtkWidget		*mail_check_spin_button;
static GtkWidget		*msg_count_button[3];
static GtkWidget		*animation_select_button[4];
static GtkWidget		*check_only_button;
static GtkWidget		*reset_remote_button;
static GtkWidget		*mua_inhibit_button;

static gint				mailboxes_modified,
						selected_row;

static void
cb_mailbox_selected(GtkWidget *clist, gint row, gint column,
					GdkEventButton *bevent, gpointer data)
	{
	gchar	*s;

	gtk_clist_get_text(GTK_CLIST(clist), row, 0, &s);
	gtk_entry_set_text(GTK_ENTRY(mbox_path_entry), s);
	selected_row = row;
	}

static void
cb_mailbox_unselected(GtkWidget *clist, gint row, gint column,
					GdkEventButton *bevent, gpointer data)
	{
	selected_row = -1;
	gtk_entry_set_text(GTK_ENTRY(mbox_path_entry), "");
	}

static void
cb_enter_mailbox()
	{
	gchar			*buf[2], *s;

	s = gtk_entry_get_text(GTK_ENTRY(mbox_path_entry));
	if (*s == '\0')
		return;
	buf[0] = s;
	buf[1] = NULL;
	if (selected_row >= 0)
		{
		gtk_clist_set_text(GTK_CLIST(mbox_clist), selected_row, 0, s);
		gtk_clist_unselect_row(GTK_CLIST(mbox_clist), selected_row, 0);
		selected_row = -1;
		}
	else
		gtk_clist_append(GTK_CLIST(mbox_clist), buf);
	gtk_entry_set_text(GTK_ENTRY(mbox_path_entry), "");
	mailboxes_modified = TRUE;
	}

static void
cb_delete_mailbox(GtkWidget *widget, gpointer *data)
	{
	gtk_entry_set_text(GTK_ENTRY(mbox_path_entry), "");
	if (selected_row >= 0)
		{
		gtk_clist_remove(GTK_CLIST(mbox_clist), selected_row);
		mailboxes_modified = TRUE;
		selected_row = -1;
		}
	}

void 
apply_mail_config()
	{
	Mailbox			*mbox;
	gchar			*s;
	gint			r, i;
	GtkSpinButton	*spin;

	UC.super_mute_mailcheck = GTK_TOGGLE_BUTTON(super_mute_button)->active;
	UC.enable_mailcheck = GTK_TOGGLE_BUTTON(enable_mailcheck_button)->active;
	enable_visibility(UC.enable_mailcheck, &mailcheck_visible,
					mailcheck.hbox, mailcheck.h);
	for (i = 0; i < 3; ++i)
		if (GTK_TOGGLE_BUTTON(msg_count_button[i])->active)
			msg_count_mode = i;
	for (i = 0; i < 4; ++i)
		if (GTK_TOGGLE_BUTTON(animation_select_button[i])->active)
			animation_select_mode = i;
	mail_label_decal.value += 1000;		/* Force a redraw */
	mua_inhibit_mode = GTK_TOGGLE_BUTTON(mua_inhibit_button)->active;
	fetch_check_only_mode = GTK_TOGGLE_BUTTON(check_only_button)->active;
	reset_remote_mode = GTK_TOGGLE_BUTTON(reset_remote_button)->active;

	s = gtk_entry_get_text(GTK_ENTRY(mail_user_agent_entry));
	if (strcmp(s, mail_user_agent.command))
		{
		g_free(mail_user_agent.command);
		mail_user_agent.command = g_strdup(s);
		}
	s = gtk_entry_get_text(GTK_ENTRY(mail_fetch_entry));
	if (strcmp(s, mail_fetch.command))
		{
		g_free(mail_fetch.command);
		mail_fetch.command = g_strdup(s);
		}
	s = gtk_entry_get_text(GTK_ENTRY(mail_notify_entry));
	if (strcmp(s, mail_notify))
		{
		g_free(mail_notify);
		mail_notify = g_strdup(s);
		}
	spin = GTK_SPIN_BUTTON(mail_fetch_spin_button);
	mail_fetch_timeout = gtk_spin_button_get_value_as_int(spin);

	spin = GTK_SPIN_BUTTON(mail_check_spin_button);
	mail_check_timeout = gtk_spin_button_get_value_as_int(spin);

	if (mailboxes_modified)
		{
		/* Destroy the mailbox_list and recreate it from the clist.
		*/
		while (mailbox_list)
			{
			mbox = (Mailbox *) mailbox_list->data;
			g_free(mbox);
			mailbox_list = g_list_remove(mailbox_list, mbox);
			}
		for (r = 0; r < GTK_CLIST(mbox_clist)->rows; ++r)
			{
			gtk_clist_get_text(GTK_CLIST(mbox_clist), r, 0, &s);
			add_mailbox(s);
			}
		}	
	}

static gchar	*mail_info_text	=
"      Mailboxes\n"
"Mailbox files to monitor can be mbox or maildir style mailboxes.\n\n"
"      Setup\n"
"If you enter a mail reading program (your mail user agent or MUA)\n"
"it can be run by clicking on the mail monitor panel.\n"
"For example you could enter:\n"
"    xfmail\n"
"or\n"
"    xterm -e mutt\n"
"If you enter a notify (sound) command, it will be run when new mail\n"
"is detected.  For example, you could enter:\n"
"    cat /usr/local/sounds/newmail.wav > /dev/dsp\n"
"or\n"
"    bplay /usr/local/sounds/newmail.wav\n"
"A remote mail fetch program can be run periodically to download\n"
"remote mail to mailboxes you are monitoring.  If the fetch program\n"
"is \"fetchmail -c\" and under the Options tab you check the box for\n"
"\"Fetch program will check but not download messages\", then the\n"
"count of remote messages can be reported and the messages will be\n"
"left on the remote server for your MUA to download.\n\n"
"      User Interaction\n"
" * Left click the mail panel to execute the mail reading program.\n"
" * Middle click the mail panel to toggle a mute mode which inhibits\n"
"    the sound notify program and optionally inhibits all mail checking.\n"
;

static gchar	*mbox_title[] = {"Mailbox files to monitor" };

void
create_mail_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*tabs;
	GtkWidget		*table;
	GtkWidget		*frame;
	GtkWidget		*vbox, *hbox;
	GtkWidget		*separator;
	GtkWidget		*label;
	GtkWidget		*button;
	GtkWidget		*scrolled;
	GtkWidget		*text;
	GList			*list;
	GSList			*group;
	GtkSpinButton	*spin;
	GtkAdjustment	*adj;
	gchar			*buf[2];
	gint			row;

	mailboxes_modified = FALSE;
	selected_row = -1;
	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

/* --Mailboxes tab */
	vbox = create_tab(tabs, "Mailboxes");
	mbox_path_entry = gtk_entry_new_with_max_length(255);
	gtk_box_pack_start(GTK_BOX(vbox), mbox_path_entry, FALSE, FALSE, 2);
	gtk_entry_set_text(GTK_ENTRY(mbox_path_entry), "");
	gtk_signal_connect (GTK_OBJECT (mbox_path_entry), "activate",
			(GtkSignalFunc) cb_enter_mailbox, NULL);			
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 5);
	button = gtk_button_new_with_label("Enter");
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			(GtkSignalFunc) cb_enter_mailbox, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 5);
	button = gtk_button_new_with_label("Delete");
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			(GtkSignalFunc) cb_delete_mailbox, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 5);

	separator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), separator, TRUE, TRUE, 0);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
	mbox_clist = gtk_clist_new_with_titles(1, mbox_title);
	gtk_signal_connect (GTK_OBJECT(mbox_clist), "select_row",
		(GtkSignalFunc) cb_mailbox_selected, NULL);
	gtk_signal_connect (GTK_OBJECT(mbox_clist), "unselect_row",
		(GtkSignalFunc) cb_mailbox_unselected, NULL);
	gtk_container_add (GTK_CONTAINER (scrolled), mbox_clist);
	for (list = mailbox_list; list; list = list->next)
		{
		buf[0] = ((Mailbox *) (list->data))->path;
		buf[1] = NULL;
		row = gtk_clist_append(GTK_CLIST(mbox_clist), buf);
		}

/* --Setup tab */
	vbox = create_tab(tabs, "Setup");

	enable_mailcheck_button = gtk_check_button_new_with_label(
				"Enable Mailcheck");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(enable_mailcheck_button),
				UC.enable_mailcheck);
	gtk_box_pack_start(GTK_BOX(vbox), enable_mailcheck_button, TRUE, TRUE, 0);

	separator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), separator, TRUE, TRUE, 0);

	table = gtk_table_new(3, 3, FALSE /*homogeneous*/);
	gtk_table_set_col_spacings(GTK_TABLE(table), 2);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 2);

	label = gtk_label_new("Mail reading program");
	/* Attach left right top bottom */
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	mail_user_agent_entry = gtk_entry_new_with_max_length(255);
	gtk_table_attach_defaults(GTK_TABLE(table), mail_user_agent_entry,
				1, 3, 0, 1);
	gtk_entry_set_text(GTK_ENTRY(mail_user_agent_entry),
				mail_user_agent.command);

	label = gtk_label_new("Notify (sound) program");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	mail_notify_entry = gtk_entry_new_with_max_length(255);
	gtk_table_attach_defaults(GTK_TABLE(table), mail_notify_entry, 1, 3, 1, 2);
	gtk_entry_set_text(GTK_ENTRY(mail_notify_entry), mail_notify);

	label = gtk_label_new("Remote fetch program");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	mail_fetch_entry = gtk_entry_new_with_max_length(255);
	gtk_table_attach_defaults(GTK_TABLE(table), mail_fetch_entry, 1, 3, 2, 3);
	gtk_entry_set_text(GTK_ENTRY(mail_fetch_entry), mail_fetch.command);

	table = gtk_table_new(3, 3, FALSE /*homogeneous*/);
	gtk_table_set_col_spacings(GTK_TABLE(table), 2);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 2);

	label = gtk_label_new("Execute fetch program every");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_RIGHT);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
	adj = (GtkAdjustment *) gtk_adjustment_new(
			mail_fetch_timeout, 1.0 /*lower*/, 60.0 /*upper*/,
			1.0 /*step*/, 5.0 /*page*/, 0.0);
	mail_fetch_spin_button = gtk_spin_button_new (adj, 0.5 /*climb rate*/, 0);
	gtk_widget_set_usize(mail_fetch_spin_button, 20, 0);
	spin = GTK_SPIN_BUTTON(mail_fetch_spin_button);
	gtk_spin_button_set_numeric(spin, TRUE);
	gtk_table_attach_defaults(GTK_TABLE(table), mail_fetch_spin_button,
				1, 2, 0, 1);
	label = gtk_label_new("minutes");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 0, 1);

	label = gtk_label_new("Check mailbox files every");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_RIGHT);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
	adj = (GtkAdjustment *) gtk_adjustment_new(
			mail_check_timeout, 2.0 /*lower*/, 100.0 /*upper*/,
			1.0 /*step*/, 5.0 /*page*/, 0.0);
	mail_check_spin_button = gtk_spin_button_new (adj, 0.5 /*climb rate*/, 0);
	gtk_widget_set_usize(mail_check_spin_button, 20, 0);
	spin = GTK_SPIN_BUTTON(mail_check_spin_button);
	gtk_spin_button_set_numeric(spin, TRUE);
	gtk_table_attach_defaults(GTK_TABLE(table), mail_check_spin_button,
				1, 2, 1, 2);
	label = gtk_label_new("seconds");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 1, 2);


/* --Options tab */
	vbox = create_tab(tabs, "Options");

	frame = gtk_frame_new ("Animation Select");
	gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 2);
	hbox = gtk_hbox_new (FALSE, 3);
	gtk_container_add(GTK_CONTAINER(frame), hbox);

	button = gtk_radio_button_new_with_label(NULL, "None");
	gtk_box_pack_start(GTK_BOX (hbox), button, TRUE, TRUE, 0);
	group = gtk_radio_button_group(GTK_RADIO_BUTTON (button));
	animation_select_button[0] = button;
	button = gtk_radio_button_new_with_label(group, "Envelope");
	gtk_box_pack_start(GTK_BOX (hbox), button, TRUE, TRUE, 0);
	group = gtk_radio_button_group(GTK_RADIO_BUTTON (button));
	animation_select_button[1] = button;
	button = gtk_radio_button_new_with_label(group, "Penguin");
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
	group = gtk_radio_button_group(GTK_RADIO_BUTTON (button));
	animation_select_button[2] = button;
	button = gtk_radio_button_new_with_label(group, "Both");
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
	animation_select_button[3] = button;
	button = animation_select_button[animation_select_mode];
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

	frame = gtk_frame_new ("Message Counting");
	gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 2);
	hbox = gtk_hbox_new (FALSE, 3);
	gtk_container_add(GTK_CONTAINER(frame), hbox);
	button = gtk_radio_button_new_with_label(NULL, "new/total");
	gtk_box_pack_start(GTK_BOX (hbox), button, TRUE, TRUE, 0);
	group = gtk_radio_button_group(GTK_RADIO_BUTTON (button));
	msg_count_button[0] = button;
	button = gtk_radio_button_new_with_label(group, "new");
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
	group = gtk_radio_button_group(GTK_RADIO_BUTTON (button));
	msg_count_button[1] = button;
	button = gtk_radio_button_new_with_label(group, "do not count");
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
	msg_count_button[2] = button;
	button = msg_count_button[msg_count_mode];
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

	check_only_button = gtk_check_button_new_with_label(
	"Fetch program will check but not download messages (fetchmail -c)");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_only_button),
				fetch_check_only_mode);
	gtk_box_pack_start(GTK_BOX(vbox), check_only_button, TRUE, TRUE, 0);
	reset_remote_button = gtk_check_button_new_with_label(
	"Reset remote message counts when mail reader is executed");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(reset_remote_button),
				reset_remote_mode);
	gtk_box_pack_start(GTK_BOX(vbox), reset_remote_button, TRUE, TRUE, 0);


	super_mute_button = gtk_check_button_new_with_label(
	"Mute mode inhibits all mail checking, not just notify (sound) program");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(super_mute_button),
				UC.super_mute_mailcheck);
	gtk_box_pack_start(GTK_BOX(vbox), super_mute_button, TRUE, TRUE, 0);

	mua_inhibit_button = gtk_check_button_new_with_label(
	"Inhibit all mail checking while the mail reader is running");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mua_inhibit_button),
				mua_inhibit_mode);
	gtk_box_pack_start(GTK_BOX(vbox), mua_inhibit_button, TRUE, TRUE, 0);
	
/* --Info tab */
	vbox = create_tab(tabs, "Info");
	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
	text = gtk_text_new(NULL, NULL);
	gtk_text_insert(GTK_TEXT(text), NULL, NULL, NULL, mail_info_text, -1);
	gtk_text_set_editable(GTK_TEXT(text), FALSE);
	gtk_container_add(GTK_CONTAINER(scrolled), text);
	}

