/*
 This file is part of MOST.

 Copyright (c) 1991, 1999 John E. Davis

 This program is free software; you can redistribute it and/or modify it
 under the terms of the GNU General Public License as published by the Free
 Software Foundation; either version 2 of the License, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 more details.

 You should have received a copy of the GNU General Public License along
 with this program; if not, write to the Free Software Foundation, Inc., 675
 Mass Ave, Cambridge, MA 02139, USA. 
*/
#include "config.h"

#include <stdio.h>
#include <string.h>

#include <slang.h>
#include "jdmacros.h"

#include "most.h"
#include "line.h"
#include "window.h"
#include "display.h"

int Most_Tab_Width = 8;

#define MAX_LINE_LEN 8196
int Most_Selective_Display = 0;
/* take 16 binary characters and put them in displayable form */
static void ascii_format_line (unsigned char *beg, unsigned char *end,
			       char *buf)
{
   unsigned char *b;
   char *s, *s1;
   unsigned char ch;
   int count;

   count = 0;
   b = beg;
   s = buf;

   while (b < end)
     {
	if (count == 4)
	  {
	     *s++ = ' ';
	     count = 0;
	  }
	count++;

	ch = *b++;

	if ((Most_V_Opt == 0)
	    || (ch & 0x80))
	  {
	     sprintf (s, "%02X", ch);
	     s += 2;
	     continue;
	  }

	if ((ch >= ' ') && (ch < 0x7F))
	  {
	     *s++ = ' ';
	     *s++ = (char) ch;
	     continue;
	  }

	*s++ = '^';
	if (ch < ' ') ch += '@';
	else ch = '?';
	*s++ = ch;
     }

   s1 = buf + (9 * 4) + 4;
   while (s < s1)
     *s++ = ' ';

   b = beg;
   while (b < end)
     {
        ch = *b++;
	if (((ch >= ' ') && (ch < 0x7F))
	    || (ch >= SLsmg_Display_Eight_Bit))
	  {
	     *s++ = ch;
	     continue;
	  }
	*s++ = '.';
     }
   *s = 0;
}

static void output_binary_formatted_line (void)
{
   unsigned char *beg, *end;
   char buf[256];

   beg = Most_Beg + Most_C_Offset;
   end = beg + 16;

   if (end > Most_Eob) end = Most_Eob;

   sprintf (buf, "0x%08X: ", Most_C_Offset);
   ascii_format_line (beg, end, buf + 12);
   SLsmg_write_string (buf);
   SLsmg_erase_eol ();
}

static int most_analyse_line(unsigned char *begg, unsigned char *endd, char *out, char *attributes)
{
   register unsigned char *beg = begg, *end = endd, ch;
   register int ii, i;
   int test, fold, ii_max, j, ok;
   char attr;

   test = 0;
   i = ii = 0;
   ii_max = 0;
   fold = 0;
   while ((beg < end) && ((ch = *beg) != '\n')) /*  && (ch != '\0')) */
     {
	beg++;
	if (ii > ii_max) fold = 0;  /* beyond previous high */
	attr = ' ';

	/*
	 *  set up bolding of line if ^M
	 */
	if (!Most_V_Opt && (ch == '\r'))                   /* ^M */
	  {
	     if (beg < end)
	       {
		  fold = 1;
		  if (ii > ii_max) ii_max = ii - 1; /* ^M contributes nil */
		  ii = 0;
	       }
	  }

	/*
	 * set up bolding or underlining of character if '\b' (^H)
	 */
	else if (!Most_V_Opt && (ch == '\b'))
	  {
	     test = 1;
	     if (ii != 0) ii--;
	     if (i != 0) i--;
	  }
	/*
	 * assign bolding or underlining attributes
	 */
	else if (test || fold)
	  {
	     test = 0;
	     if (ch == (unsigned char) out[ii])
	       attr = 'b';
	     else if (out[ii] == '_')
	       attr = 'u';

	     if (fold && ch == ' ')
	       ii++;
	     else
	       {
		  attributes[i++] = attr;
		  out[ii++] = ch;
	       }
	  }
	else if (!Most_T_Opt && (ch == '\t'))
	  {
	     j = Most_Tab_Width * (ii/Most_Tab_Width + 1) - ii;  /* Most_Tab_Width column tabs */
	     while (j--)
	       {
		  out[ii++] = ' ';
		  attributes[i++] = attr;
	       }
	  }
	else
	  {
	     out[ii++] = ch;
	     attributes[i++] = attr;
	     if (0 == (((ch >= ' ') && (ch < 0x7F))
		       || (ch >= SLsmg_Display_Eight_Bit)))
	       {
		  attributes [i++] = attr;
		  if (ch & 0x80)
		    attributes [i++] = attr;
	       }
	  }
     }
   if (fold) ii = ii_max + 1;
   if ((beg > end) && Most_Selective_Display && !Most_W_Opt)
     {
	ok = 1;
	if (*beg == '\n') beg++;
	while ((*beg <= ' ') && ok)
	  {
	     if (*beg != '\n') beg++;
	     if (beg >= Most_Eob) break;
	     if ((*beg == '\n') || (most_apparant_distance(beg) >= Most_Selective_Display))
	       {
		  ok = 0;
	       }
	  }
	if (!ok)
	  {
	     ok = 3;
	     while(ok--)
	       {
		  out[ii++] = '.';
		  attributes[i++] = ' ';
	       }
	  }

     }
   /* Most_Selective_Display */
   out[ii] = '\0';
   return(ii);
}

static void output_with_attr (unsigned char *out, unsigned char *attr)
{
   unsigned char at, ch, lat;
   unsigned char *p = out;

   if (Most_V_Opt) 
     {
	SLsmg_write_string ((char *) out);
	return;
     }

   lat = ' ';
   while ((ch = *p) != 0)
     {
	if (lat != *attr)
	  {
	     if (p != out)
	       {
		  SLsmg_write_nchars ((char *) out, (unsigned int) (p - out));
		  out = p;
	       }

	     at = *attr;
	     if (at == 'u')
	       {
		  most_tt_underline_video ();
	       }
	     else if (at == 'b')
	       {
		  most_tt_bold_video ();
	       }
	     else most_tt_normal_video ();
	     lat = at;
	  }
	p++;
	attr++;
     }

   SLsmg_write_nchars ((char *) out, (unsigned int) (p - out));
   if (lat != ' ') most_tt_normal_video ();
}

static void most_output(unsigned char *line, unsigned int len, unsigned char *attr, char unsigned d_char)
{
   unsigned int i, ii, count, max_col;
   unsigned char ch;
   unsigned char out[1024];

   /* expand to the window start */
   i = count = 0;
   while ((i < len) && ((ch = line[i]) != '\n'))
     {
	if (((ch >= ' ') && (ch < 0x7F))
	    || (ch >= SLsmg_Display_Eight_Bit))
	  count += 1;
	else if (ch & 0x80) count += 3;
	else count += 2;

	if (count >= (unsigned int)Most_Column) break;
	i++;
     }
   if (count < (unsigned int)Most_Column) 
     return;

   /* Now i is where we can start filling the left part of the screen */
   attr += i; count = 0; ii = 0;
   max_col = SLtt_Screen_Cols;

   if (max_col >= sizeof (out))
     max_col = sizeof(out) - 1;

   while ((i < len) && (ii < max_col) && ((ch = line[i]) != '\n'))
     {
	if (((ch >= ' ') && (ch < 0x7F))
	    || (ch >= SLsmg_Display_Eight_Bit))
	  out [ii++] = ch;
	else
	  {
	     if (ch & 0x80)
	       {
		  out[ii++] = '~';
		  ch &= 0x7F;
	       }
	     out[ii++] = '^';

	     if (ch == 0x7F) ch = '?';
	     else ch += '@';
	     out[ii++] = ch;
	  }
	i++;
	/* if (ii >= max_col) break; */
     }

   if ((ii > max_col) || ((ii == max_col) && (i < len))
       || ((d_char != '$') && (ii = max_col - 1)))
     {
	ii = max_col;
	out[ii - 1] = d_char;
     }
   out[ii] = '\0';

   output_with_attr (out, attr);
}

void most_display_line (void)
{
   unsigned char *beg, *end;
   unsigned len;
   int v = 0, t = 0;
   unsigned char the_line[MAX_LINE_LEN];
   unsigned char the_attr[MAX_LINE_LEN];
   unsigned char *line, *attr, ch;

   if (Most_B_Opt)
     {
	output_binary_formatted_line ();
	return;
     }

   line = the_line;
   attr = the_attr;

   if (most_extract_line(&beg, &end) && Most_W_Opt) ch = '\\';
   else ch = '$';

   len = end - beg;  /* MOST4  --- was + 1 */
   if (len == 0)
     {
	SLsmg_erase_eol ();
	return;
     }

   if (len > MAX_LINE_LEN)
     {
	/* This needs fixed for files with really big lines */
	v = Most_V_Opt;
	t = Most_T_Opt;
	Most_V_Opt = 1;
	Most_T_Opt = 1;
	most_output(beg, len, NULL, '$');
	SLsmg_erase_eol ();
	return;
     }

   len = most_analyse_line(beg, end, (char *) line, (char *) attr);
   most_output(line,len,attr, ch);
   SLsmg_erase_eol ();
}

/* given a position in a line, return apparant distance from bol
   expanding tabs, etc... up to pos */
int most_apparant_distance (unsigned char *pos)
{
   int i;
   unsigned char *save_pos, ch;
   unsigned int save_offset;

   save_offset = Most_C_Offset;
   save_pos = pos;
   Most_C_Offset = (unsigned int) (pos - Most_Beg);
   pos = most_beg_of_line();
   Most_C_Offset = save_offset;

   i = 0;
   while (pos < save_pos)
     {
	ch = *pos++;
	if (((ch >= ' ') && (ch < 0x7F))
	    || (ch >= SLsmg_Display_Eight_Bit))
	  {
	     i++;
	     continue;
	  }

	if (!Most_V_Opt && (ch == '\b'))
	  {
	     if (i > 0) i--;
	  }
	else if (!Most_V_Opt && (ch == '\015')) /* ^M */
	  {
	     if (i != 1) i = 0;
	  }
	else if (!Most_T_Opt && (ch == '\t'))
	  {
	     i = Most_Tab_Width * (i/Most_Tab_Width + 1);  /* Most_Tab_Width column tabs */
	  }
	 /* else Control char */
	else
	  {
	     if (ch & 0x80) i += 3;
	     else i += 2;
	  }
     }
   return i;
}
