//
// STIL class - Implementation file
//
// AUTHOR: LaLa
// Email : lala@interaccess.com
// (C) 1998 by LaLa
//

#ifndef _STIL
#define _STIL

#include <iostream.h>
#include <string.h>
#include <fstream.h>
#include <stdio.h>      // For sprintf() and scanf()
#include "stil.h"

#include <sidplay/compconf.h>  // for ios::bin

#if defined(HAVE_IOS_BIN)
#define STILopenFlags (ios::in | ios::bin | ios::nocreate)
#else
#define STILopenFlags (ios::in | ios::binary | ios::nocreate)
#endif

#define VERSION_NO 1.24

#define CERR_STIL_DEBUG if (STIL_DEBUG) cerr << "Line #" << __LINE__ << " STIL::"

// CONSTRUCTOR
STIL::STIL()
{
    sprintf(versionString, "STIL View v%4.2f, (C) 1998 by LaLa (lala@interaccess.com)\n", VERSION_NO);

    STILVersion = 0.0;
    memset((void *)baseDir, 0, sizeof(baseDir));
    baseDirLength = 0;
    memset((void *)entrybuf, 0, sizeof(entrybuf));
    memset((void *)globalbuf, 0, sizeof(globalbuf));
    memset((void *)bugbuf, 0, sizeof(bugbuf));
    memset((void *)resultEntry, 0, sizeof(resultEntry));
    memset((void *)resultBug, 0, sizeof(resultBug));

    for (int i=0; i<STIL_MAX_DIRS; i++) {
        stilDirs[i].dirName = NULL;
        stilDirs[i].position = 0;
        bugDirs[i].dirName = NULL;
        bugDirs[i].position = 0;
    }

    STIL_EOL = '\n';
    STIL_EOL2 = '\0';

    STIL_DEBUG = false;
    lastError = NO_STIL_ERROR;
}

// DECONSTRUCTOR
STIL::~STIL()
{
    for (int i=0; i<STIL_MAX_DIRS; i++) {
        if (stilDirs[i].dirName != NULL) {
            delete stilDirs[i].dirName;
        }
        if (bugDirs[i].dirName != NULL) {
            delete bugDirs[i].dirName;
        }
    }

    CERR_STIL_DEBUG << "Deconstructor called" << endl;
}

char *
STIL::getVersion()
{
    lastError = NO_STIL_ERROR;
    return versionString;
}

float
STIL::getVersionNo()
{
    lastError = NO_STIL_ERROR;
    return VERSION_NO;
}

float
STIL::getSTILVersionNo()
{
    lastError = NO_STIL_ERROR;
    return STILVersion;
}

bool
STIL::setBaseDir(const char *pathToHVSC)
{
    char *temp;
    char tempName[STIL_MAX_PATH_SIZE];

    lastError = NO_STIL_ERROR;

    CERR_STIL_DEBUG << "setBaseDir() called, pathToHVSC=" << pathToHVSC << endl;

    // Check length
    if ((strlen(pathToHVSC) > STIL_MAX_PATH_SIZE) || (strlen(pathToHVSC) < 1)) {
        CERR_STIL_DEBUG << "setBaseDir() has problem with the size of pathToHVSC" << endl;
        lastError = BASE_DIR_LENGTH;
        return false;
    }

    strcpy(baseDir, pathToHVSC);

    // Chop the trailing slash
    temp = baseDir+strlen(baseDir)-1;
    if (*temp == SLASH) {
        *temp = '\0';
    }
    baseDirLength = strlen(baseDir);

    // Attempt to open STIL

    // Create the full path+filename
    strcpy(tempName, baseDir);
    strcat(tempName, PATH_TO_STIL);
    convertSlashes(tempName);

    stilFile.open(tempName, STILopenFlags);

    if (stilFile.bad()) {
        stilFile.close();
        CERR_STIL_DEBUG << "setBaseDir() open failed for" << tempName << endl;
        lastError = STIL_OPEN;
        return false;
    }

    CERR_STIL_DEBUG << "setBaseDir(): open succeeded for" << tempName << endl;

    // Attempt to open BUGlist

    // Create the full path+filename
    strcpy(tempName, baseDir);
    strcat(tempName, PATH_TO_BUGLIST);
    convertSlashes(tempName);

    bugFile.open(tempName, STILopenFlags);

    if (bugFile.bad()) {

        // This is not a critical error - some earlier versions of HVSC may
        // not have a BUGlist.txt file at all.

        bugFile.close();
        CERR_STIL_DEBUG << "setBaseDir() open failed for" << tempName << endl;
        lastError = BUG_OPEN;
    }

    CERR_STIL_DEBUG << "setBaseDir(): open succeeded for" << tempName << endl;

    // Find out what the EOL really is
    if (determineEOL() != true) {
        stilFile.close();
        bugFile.close();
        CERR_STIL_DEBUG << "determinEOL() failed" << endl;
        lastError = NO_EOL;
        return false;
    }

    // These will populate the stilDirs and bugDirs arrays (or not :)

    if (getDirs(stilFile, stilDirs, true) != true) {
        stilFile.close();
        bugFile.close();
        CERR_STIL_DEBUG << "getDirs() failed for stilFile" << endl;
        lastError = NO_STIL_DIRS;
        return false;
    }

    if (bugFile.good()) {
        if (getDirs(bugFile, bugDirs, false) != true) {
            stilFile.close();
            bugFile.close();
            CERR_STIL_DEBUG << "getDirs() failed for bugFile" << endl;
            lastError = NO_BUG_DIRS;
            return false;
        }
    }

    stilFile.close();
    bugFile.close();

    CERR_STIL_DEBUG << "setBaseDir() succeeded" << endl;
    return true;
}

char *
STIL::getAbsEntry(const char *absPathToEntry, int tuneNo, int timestamp, STILField field)
{
    char tempDir[STIL_MAX_PATH_SIZE];

    lastError = NO_STIL_ERROR;

    CERR_STIL_DEBUG << "getAbsEntry() called, absPathToEntry=" << absPathToEntry << endl;

    // Determine if the baseDir is in the given pathname.

    if (MYSTRNCMP(absPathToEntry, baseDir, baseDirLength) != 0) {
        CERR_STIL_DEBUG << "getAbsEntry() failed: baseDir=" << baseDir << ", absPath=" << absPathToEntry << endl;
        lastError = WRONG_DIR;
        return NULL;
    }

    strcpy(tempDir, absPathToEntry+baseDirLength);
    convertToSlashes(tempDir);

    return (getEntry(tempDir, tuneNo, timestamp, field));
}

char *
STIL::getEntry(const char *relPathToEntry, int tuneNo, int timestamp, STILField field)
{
    lastError = NO_STIL_ERROR;

    if (STILVersion < 2.59) {

        // Older version of STIL is detected.

        tuneNo = 0;
        timestamp = 0;
        field = all;
    }

    CERR_STIL_DEBUG << "getEntry() called, relPath=" << relPathToEntry << ", rest=" << tuneNo << "," << timestamp << "," << field << endl;

    // Find out whether we have this entry in the buffer.

    if (MYSTRNCMP(entrybuf, relPathToEntry, strlen(relPathToEntry)) != 0) {

        CERR_STIL_DEBUG << "getEntry(): entry not in buffer" << endl;

        // No, we don't - pull it in.

        char tempName[STIL_MAX_PATH_SIZE];

        // Create the full path+filename
        strcpy(tempName, baseDir);
        strcat(tempName, PATH_TO_STIL);
        convertSlashes(tempName);

        stilFile.open(tempName, STILopenFlags);

        if (stilFile.bad()) {
            stilFile.close();
            CERR_STIL_DEBUG << "getEntry() open failed for stilFile" << endl;
            lastError = STIL_OPEN;
            return NULL;
        }

        CERR_STIL_DEBUG << "getEntry() open succeeded for stilFile" << endl;

        if (positionToEntry(relPathToEntry, stilFile, stilDirs) == false) {
            // Copy the entry's name to the buffer.
            strcpy(entrybuf, relPathToEntry);
            addNewline(entrybuf);
            CERR_STIL_DEBUG << "getEntry() posToEntry() failed" << endl;
            lastError = NOT_IN_STIL;
        }
        else {
            *entrybuf= '\0';
            readEntry(stilFile, entrybuf);
            CERR_STIL_DEBUG << "getEntry() entry read" << endl;
        }

        stilFile.close();
    }

    // Put the requested field into the result string.

    if (getField(resultEntry, entrybuf, tuneNo, timestamp, field) != true) {
        return NULL;
    }
    else {
        return resultEntry;
    }
}

char *
STIL::getAbsBug(const char *absPathToEntry, int tuneNo)
{
    lastError = NO_STIL_ERROR;

    CERR_STIL_DEBUG << "getAbsBug() called, absPathToEntry=" << absPathToEntry << endl;

    char tempDir[STIL_MAX_PATH_SIZE];

    // Determine if the baseDir is in the given pathname.

    if (MYSTRNCMP(absPathToEntry, baseDir, baseDirLength) != 0) {
        CERR_STIL_DEBUG << "getAbsBug() failed: baseDir=" << baseDir << ", absPath=" << absPathToEntry << endl;
        lastError = WRONG_DIR;
        return NULL;
    }

    strcpy(tempDir, absPathToEntry+baseDirLength);
    convertToSlashes(tempDir);

    return (getBug(tempDir, tuneNo));
}

char *
STIL::getBug(const char *relPathToEntry, int tuneNo)
{
    lastError = NO_STIL_ERROR;

    // Older version of STIL is detected.

    if (STILVersion < 2.59) {
        tuneNo = 0;
    }

    CERR_STIL_DEBUG << "getBug() called, relPath=" << relPathToEntry << ", rest=" << tuneNo << endl;

    // Find out whether we have this bug entry in the buffer.

    if (MYSTRNCMP(bugbuf, relPathToEntry, strlen(relPathToEntry)) != 0) {

        CERR_STIL_DEBUG << "getBug(): entry not in buffer" << endl;

        // No, we don't - pull it in.

        char tempName[STIL_MAX_PATH_SIZE];

        // Create the full path+filename
        strcpy(tempName, baseDir);
        strcat(tempName, PATH_TO_BUGLIST);
        convertSlashes(tempName);

        bugFile.open(tempName, STILopenFlags);

        if (bugFile.bad()) {
            bugFile.close();
            CERR_STIL_DEBUG << "getBug() open failed for bugFile" << endl;
            lastError = BUG_OPEN;
            return NULL;
        }

        CERR_STIL_DEBUG << "getBug() open succeeded for bugFile" << endl;

        if (positionToEntry(relPathToEntry, bugFile, bugDirs) == false) {
            // Copy the entry's name to the buffer.
            strcpy(bugbuf, relPathToEntry);
            addNewline(bugbuf);
            CERR_STIL_DEBUG << "getBug() posToEntry() failed" << endl;
            lastError = NOT_IN_BUG;
        }
        else {
            *bugbuf = '\0';
            readEntry(bugFile, bugbuf);
            CERR_STIL_DEBUG << "getBug() entry read" << endl;
        }

        bugFile.close();
    }

    // Put the requested field into the result string.

    if (getField(resultBug, bugbuf, tuneNo) != true) {
        return NULL;
    }
    else {
        return resultBug;
    }
}

char *
STIL::getAbsGlobalComment(const char *absPathToEntry)
{
    lastError = NO_STIL_ERROR;

    char tempDir[STIL_MAX_PATH_SIZE];

    CERR_STIL_DEBUG << "getAbsGC() called, absPathToEntry=" << absPathToEntry << endl;

    // Determine if the baseDir is in the given pathname.

    if (MYSTRNCMP(absPathToEntry, baseDir, baseDirLength) != 0) {
        CERR_STIL_DEBUG << "getAbsGC() failed: baseDir=" << baseDir << ", absPath=" << absPathToEntry << endl;
        lastError = WRONG_DIR;
        return NULL;
    }

    strcpy(tempDir, absPathToEntry+baseDirLength);
    convertToSlashes(tempDir);

    return (getGlobalComment(tempDir));
}

char *
STIL::getGlobalComment(const char *relPathToEntry)
{
    char dir[STIL_MAX_PATH_SIZE];
    size_t endOfPath;
    char *temp;
    char *lastSlash;

    lastError = NO_STIL_ERROR;

    CERR_STIL_DEBUG << "getGC() called, relPath=" << relPathToEntry << endl;

    // Save the dirpath.

    lastSlash = (char *)strrchr(relPathToEntry, '/');

    if (lastSlash == NULL) {
        lastError = WRONG_DIR;
        return NULL;
    }

    endOfPath = lastSlash-relPathToEntry+1;
    strncpy(dir, relPathToEntry, endOfPath);
    *(dir+endOfPath) = '\0';

    // Find out whether we have this global comment in the buffer.

    if (MYSTRNCMP(globalbuf, dir, endOfPath) != 0) {

        CERR_STIL_DEBUG << "getGC(): entry not in buffer" << endl;

        // No, we don't - pull it in.

        char tempName[STIL_MAX_PATH_SIZE];

        // Create the full path+filename
        strcpy(tempName, baseDir);
        strcat(tempName, PATH_TO_STIL);
        convertSlashes(tempName);

        stilFile.open(tempName, STILopenFlags);

        if (stilFile.bad()) {
            stilFile.close();
            CERR_STIL_DEBUG << "getGC() open failed for stilFile" << endl;
            lastError = STIL_OPEN;
            return NULL;
        }

        if (positionToEntry(dir, stilFile, stilDirs) == false) {
            // Copy the dirname to the buffer.
            strcpy(globalbuf, dir);
            addNewline(globalbuf);
            CERR_STIL_DEBUG << "getGC() posToEntry() failed" << endl;
            lastError = NOT_IN_STIL;
        }
        else {
            *globalbuf = '\0';
            readEntry(stilFile, globalbuf);
            CERR_STIL_DEBUG << "getGC() entry read" << endl;
        }

        stilFile.close();
    }

    CERR_STIL_DEBUG << "getGC() globalbuf=" << globalbuf << endl;
    CERR_STIL_DEBUG << "-=END=-" << endl;

    // Position pointer to the global comment field.

    temp = strchr(globalbuf, '\n');
    temp++;

    // Check whether this is a NULL entry or not.

    if (*temp == '\0') {
        return NULL;
    }
    else {
        return temp;
    }
}

//////// PRIVATE

bool
STIL::determineEOL()
{
    char line[STIL_MAX_LINE_SIZE+5];
    int i=0;

    CERR_STIL_DEBUG << "detEOL() called" << endl;

    if (stilFile.bad()) {
        CERR_STIL_DEBUG << "detEOL() open failed" << endl;
        return false;
    }

    stilFile.seekg(0);

    // Read in the first line from stilFile to determine what the
    // EOL character is (it can be different from OS to OS).

    stilFile.read(line, sizeof(line)-1);
    line[sizeof(line)-1] = '\0';

    CERR_STIL_DEBUG << "detEOL() line=" << line << endl;

    // Now find out what the EOL char is (or are).

    STIL_EOL = '\0';
    STIL_EOL2 = '\0';

    while (line[i] != '\0') {
        if ((line[i] == 0x0d) || (line[i] == 0x0a)) {
            if (STIL_EOL == '\0') {
                STIL_EOL = line[i];
            }
            else {
                if (line[i] != STIL_EOL) {
                    STIL_EOL2 = line[i];
                }
            }
        }
        i++;
    }

    if (STIL_EOL == '\0') {
        // Something is wrong - no EOL-like char was found.
        CERR_STIL_DEBUG << "detEOL() no EOL found" << endl;
        return false;
    }

    CERR_STIL_DEBUG << "detEOL() EOL1=" << hex << (int) STIL_EOL << " EOL2=" << hex << (int) STIL_EOL2 << dec << endl;

    return true;
}

bool
STIL::getDirs(ifstream& inFile, dirList *dirs, bool isSTILFile)
{
    char line[STIL_MAX_LINE_SIZE];
    int i=0;
    size_t j;
    bool newDir;

    if (isSTILFile) {
        newDir = false;
    }
    else {
        newDir = true;
    }

    CERR_STIL_DEBUG << "getDirs() called" << endl;

    inFile.seekg(0);

    while (inFile.good()) {
        getStilLine(inFile, line);

        // Try to extract STIL's version number if it's not done, yet.

        if (isSTILFile && (STILVersion == 0.0)) {
            if (strncmp(line, "#  STIL v", 9) == 0) {

                // Get the version number
                sscanf(line+9, "%f", &STILVersion);

                // Put it into the string, too.
                sprintf(line, "SID Tune Information List (STIL) v%4.2f\n", STILVersion);
                strcat(versionString, line);

                CERR_STIL_DEBUG << "getDirs() STILVersion=" << STILVersion << endl;

                continue;
            }
        }

        // Search for the start of a dir separator first.

        if (isSTILFile && !newDir && (MYSTRNCMP(line, "### ", 4) == 0)) {
            newDir = true;
            continue;
        }

        // Is this the start of an entry immediately following a dir separator?

        if (newDir && (*line == '/')) {

            // Get the directory only
            j = strrchr(line,'/')-line+1;

            if (!isSTILFile) {
                // Compare it to the last stored dirname
                if (i==0) {
                    newDir = true;
                }
                else if (MYSTRNCMP(dirs[i-1].dirName, line, j) != 0) {
                    newDir = true;
                }
                else {
                    newDir = false;
                }
            }

            // Store the info
            if (newDir) {
                dirs[i].dirName = new char [j+1];
                strncpy(dirs[i].dirName, line, j);
                *(dirs[i].dirName+j) = '\0';
                dirs[i].position = inFile.tellg()-strlen(line)-1;

                CERR_STIL_DEBUG << "getDirs() i=" << i << ", dirName=" << dirs[i].dirName << ", pos=" << dirs[i].position <<  endl;
                i++;
            }

            if (isSTILFile) {
                newDir = false;
            }
            else {
                newDir = true;
            }
        }
    }

    if (i == 0) {
        // No entries found - something is wrong
        CERR_STIL_DEBUG << "getDirs() no dirs found" << endl;
        return false;
    }

    CERR_STIL_DEBUG << "getDirs() successful" << endl;

    return true;
}

bool
STIL::positionToEntry(const char *entryStr, ifstream& inFile, dirList *dirs)
{
    size_t endOfPath;
    size_t entryStrLen;
    char line[STIL_MAX_LINE_SIZE];
    int temp;
    bool foundIt = false;
    bool globComm = false;
    int i=0;
    char *chrptr;

    CERR_STIL_DEBUG << "pos2Entry() called, entryStr=" << entryStr << endl;

    inFile.seekg(0);

    // Get the dirpath.

    chrptr = strrchr((char *)entryStr, '/');

    // If no slash was found, something is screwed up in the entryStr.

    if (chrptr == NULL) {
        return false;
    }

    endOfPath = chrptr-entryStr+1;

    // Determine whether a section-global comment is asked for.

    entryStrLen = strlen(entryStr);
    if (endOfPath == entryStrLen) {
        globComm = true;
    }

    // Find it in the table.

    while (dirs[i].dirName) {
        if (MYSTRNCMP(dirs[i].dirName, entryStr, endOfPath) == 0) {
            CERR_STIL_DEBUG << "pos2Entry() found dir, i=" << i << ", dirName=" << dirs[i].dirName << endl;
            break;
        }
        i++;
    }

    // Check if we went through the list.
    if (dirs[i].dirName == NULL) {
        // The directory was not found.
        CERR_STIL_DEBUG << "pos2Entry() did not find the dir" << endl;
        return false;
    }

    // Jump to the first entry of this section.
    inFile.seekg(dirs[i].position);

    // Now find the desired entry

    do {
        getStilLine(inFile, line);
        if (inFile.eof()) {
            break;
        }

        // Check if it is the start of an entry

        if (*line == '/') {

            if (MYSTRNCMP(dirs[i].dirName, line, endOfPath) != 0) {
                // We are outside the section - get out of the loop,
                // which will fail the search.
                break;
            }

            // Check whether we need to find a section-global comment or
            // a specific entry.

            if (globComm) {
                temp = MYSTRCMP(line, entryStr);
            }
            else {
                temp = MYSTRNCMP(line, entryStr, entryStrLen);
            }

            CERR_STIL_DEBUG << "pos2Entry() line=" << line << endl;

            if (temp == 0) {
                // Found it!
                foundIt = true;
            }

        }
    } while (!foundIt);

    if (foundIt) {
        // Reposition the file pointer back to the start of the entry.
        inFile.seekg(inFile.tellg()-strlen(line)-1);
        CERR_STIL_DEBUG << "pos2Entry() entry found" << endl;
        return true;
    }
    else {
        CERR_STIL_DEBUG << "pos2Entry() entry not found" << endl;
        return false;
    }
}

void
STIL::readEntry(ifstream& inFile, char *buffer)
{
    char line[STIL_MAX_LINE_SIZE];

    do {
        getStilLine(inFile, line);
        strcat(buffer, line);
        if (*line != '\0') {
            addNewline(buffer);
        }
    } while (*line != '\0');
}

bool
STIL::getField(char *result, char *buffer, int tuneNo, int timestamp, STILField field)
{
    char *start, *firstTuneNo;

    CERR_STIL_DEBUG << "getField() called, buffer=" << buffer << ", rest=" << tuneNo << "," << timestamp << "," << field << endl;

    // Clean out the result buffer first.
    *result = '\0';

    // Position pointer to the first char beyond the file designation.

    start = strchr(buffer, '\n');
    start++;

    // Check whether this is a NULL entry or not.

    if (*start == '\0') {
        CERR_STIL_DEBUG << "getField() null entry" << endl;
        return false;
    }

    // Is this a multitune entry?
    firstTuneNo = strstr(start, "(#");

    // This is a tune designation only if the previous char was
    // a newline (ie. if the "(#" is on the beginning of a line).
    if ((firstTuneNo != NULL) && (*(firstTuneNo-1) != '\n')) {
        firstTuneNo = NULL;
    }

    if (firstTuneNo == NULL) {

        //-------------------//
        // SINGLE TUNE ENTRY //
        //-------------------//

        CERR_STIL_DEBUG << "getField() single tune entry" << endl;

        // Was the complete entry asked for?

        if ((tuneNo == 0) || (field == all)) {

            // Yes. Simply copy the stuff in.

            strcpy(result, start);
            CERR_STIL_DEBUG << "getField() copied to resultbuf" << endl;
            return true;
        }

        // No. Return the specified field only.

        return ( getOneField(result, start, start+strlen(start), field) );
    }
    else {

        //-------------------//
        // MULTITUNE ENTRY
        //-------------------//

        CERR_STIL_DEBUG << "getField() multitune entry" << endl;

        // Was the complete entry asked for?

        if (tuneNo == 0) {

            switch (field) {

                case all:

                    // Yes. Simply copy the stuff in.

                    strcpy(result, start);
                    CERR_STIL_DEBUG << "getField() copied all to resultbuf" << endl;
                    return true;
                    break;

                case comment:

                    // Only the tune-global comment field was asked for.
                    // But if it does not exist, return false.

                    if (firstTuneNo == start) {
                        CERR_STIL_DEBUG << "getField() no firstTuneNo" << endl;
                        return false;
                    }
                    break;

                default:

                    // If a specific field other than a comment is
                    // asked for tuneNo=0, this is illegal.

                    CERR_STIL_DEBUG << "getField() illegal operation" << endl;
                    return false;
                    break;
            }
        }

        //
        // A specific tune and/or a specific field is asked for.
        //

        // If there is a tune-global comment and all fields were asked
        // for, or tuneNo=0 and the comment field was asked for, put
        // the tune-global comment into the result.

        if ((firstTuneNo != start) &&
            ((field == all) || (tuneNo == 0)) ) {

            strncat(result, start, (firstTuneNo-start)-1);
            addNewline(result);
            CERR_STIL_DEBUG << "getField() copied tuneglobal comment" << endl;
        }

        // If only the tune-global comment was asked for, we are done.

        if ((tuneNo == 0) && (field == comment)) {
            if (*result == '\0') {
                CERR_STIL_DEBUG << "getField() no tuneglobal comment" << endl;
                return false;
            }
            else {
                CERR_STIL_DEBUG << "getField() TG comment result=" << result << endl;
                return true;
            }
        }

        char *myTuneNo, *nextTuneNo;
        char tuneNoStr[8];

        // Search for the requested tune number.

        sprintf(tuneNoStr, "(#%d)", tuneNo);
        myTuneNo = strstr(start, tuneNoStr);

        if (myTuneNo != NULL) {

            // We found the requested tune number.
            // Set the pointer beyond it.
            myTuneNo = strchr(myTuneNo, '\n') + 1;

            // Where is the next one?

            nextTuneNo = strstr(myTuneNo, "(#");
            if (nextTuneNo == NULL) {
                // There is no next one - set pointer to end of entry.
                nextTuneNo = start+strlen(start);
            }

            if (field == all) {

                // Add the whole entry for the tune number into the result.

                strncat(result, myTuneNo, nextTuneNo-myTuneNo-1);
                addNewline(result);
                CERR_STIL_DEBUG << "getField() copied all to result" << endl;
            }
            else {

                // Get only the requested field.

                getOneField(result+strlen(result), myTuneNo, nextTuneNo, field);
            }
        }

        if (*result == '\0') {
            // No tune-global comment and the tune number had no
            // entry, either.
            CERR_STIL_DEBUG << "getField() nothing found" << endl;
            return false;
        }
        else {
            CERR_STIL_DEBUG << "getField() result=" << result << endl;
            return true;
        }
    }
}

bool
STIL::getOneField(char *result, char *start, char *end, STILField field)
{
    char *temp = NULL;

    // Sanity checking

    if ((end < start) || (*(end-1) != '\n')) {
        *result = '\0';
        return false;
    }

    switch (field) {

        case all:

            // Although technically speaking this is an illegal operation,
            // it's easy to implement.

            strncat(result, start, end-start);
            return true;
            break;

        case title:

            temp = strstr(start, _TITLE_STR);
            break;

        case artist:

            temp = strstr(start, _ARTIST_STR);
            break;

        case comment:

            temp = strstr(start, _COMMENT_STR);
            break;

        default:

            break;
    }

    // If the field was not found or it is not in between 'start'
    // and 'end', it is declared a failure.

    if ((temp == NULL) || (temp < start) || (temp > end)) {
        *result = '\0';
        return false;
    }

    // Search for the end of this field. This is done by finding
    // where the next field starts.

    char *nextTitle, *nextArtist, *nextComment, *nextField;

    nextTitle = strstr(temp+1, _TITLE_STR);
    nextArtist = strstr(temp+1, _ARTIST_STR);
    nextComment = strstr(temp+1, _COMMENT_STR);

    // If any of these fields is beyond 'end', they are ignored.

    if ((nextTitle != NULL) && (nextTitle >= end)) {
        nextTitle = NULL;
    }

    if ((nextArtist != NULL) && (nextArtist >= end)) {
        nextArtist = NULL;
    }

    if ((nextComment != NULL) && (nextComment >= end)) {
        nextComment = NULL;
    }

    // Now determine which one is the closest to our field - that one
    // will mark the end of the required field.

    nextField = nextTitle;

    if (nextField == NULL) {
        nextField = nextArtist;
    }
    else if ((nextArtist != NULL) && (nextArtist < nextField)) {
        nextField = nextArtist;
    }

    if (nextField == NULL) {
        nextField = nextComment;
    }
    else if ((nextComment != NULL) && (nextComment < nextField)) {
        nextField = nextComment;
    }

    if (nextField == NULL) {
        nextField = end;
    }

    // Now nextField points to the last+1 char that should be copied to
    // result. Do that.

    strncat(result, temp, nextField-temp);
    return true;
}

void
STIL::getStilLine(ifstream& infile, char *line)
{
    char temp;

    if (STIL_EOL2 != '\0') {

        // If there was a remaining EOL char from the previous read, eat it up.

        temp = infile.peek();

        if ((temp == 0x0d) || (temp == 0x0a)) {
            infile.get(temp);
        }
    }

    infile.getline(line, STIL_MAX_LINE_SIZE, STIL_EOL);
}

void
STIL::addNewline(char *str)
{
    char *tempptr;

    tempptr = str+strlen(str);
    strcpy(tempptr, "\n");
}

#endif // _STIL
