/*
 *  WSoundPrefs - WMSound Server Preferences Program
 *
 * Copyright (c) 1998, 1999 Pascal Hofstee
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "WSoundPrefs.h"
#include "SoundEvents.h"
#include "SoundPaths.h"
#include "SystemInfo.h"

#define LOAD_SOUNDSET_ICON_FILE "loadset"
#define SAVE_SOUNDSET_ICON_FILE "saveset"

static void loadConfigurations(WMScreen *scr, WMWindow *mainw);

char*	locateImage(char *name);
Bool	isFormatSupported(char *format);

proplist_t WMSoundDomain = NULL;

/* ------------------------------------------------------------------------- */

void
quit(WMWidget *w, void *data)
{
	if (WMSoundDomain)
		PLRelease(WMSoundDomain);

	exit(0);
}

void
save(WMWidget *w, void *data)
{
	Panel		*panel = WMGetHangedData(w);
	proplist_t	list;
	proplist_t	tmp;
	int		i;
	char		*p;
	
	list = PLMakeArrayFromElements(NULL, NULL);
	for (i = 0; i < WMGetListNumberOfRows(panel->sndL); i++) {
		p = WMGetListItem(panel->sndL, i)->text;
		tmp = PLMakeString(p);
		PLAppendArrayElement(list, tmp);
	}
	SetObjectForKey(list, "SoundPath");
	
	list = PLMakeArrayFromElements(NULL, NULL);
	for (i = 0; i < WMGetListNumberOfRows(panel->sndsetL); i++) {
		p = WMGetListItem(panel->sndsetL, i)->text;
		tmp = PLMakeString(p);
		PLAppendArrayElement(list, tmp);
	}
	SetObjectForKey(list, "SoundSetPath");

	PLSave(WMSoundDomain, YES);
}

void
Initialize(WMScreen *scr)
{
	Panel		*panel;
	WMTabViewItem	*tab;
	char		*path;
	char		text[80];
	WMPixmap	*icon;
	WMFont		*font;
	WMColor		*darkgray = WMDarkGrayColor(scr);
	RColor		color;
	SystemInfo      *sysinfo = getSystemInfo(scr);
	static WMTextFieldDelegate	sndfileTFDelegate = {
		NULL,
		NULL,
		changedSound,
		NULL,
		NULL,
		NULL
	};
	static WMTextFieldDelegate	devTFDelegate = {
		NULL,
		NULL,
		changedDevice,
		NULL,
		NULL,
		NULL
	};
	
	color.red = color.green = color.blue = 0xaa;

	panel = wmalloc(sizeof(Panel));
	memset(panel, 0, sizeof(Panel));
	
	
	panel->white = WMWhiteColor(scr);
	panel->black = WMBlackColor(scr);
	panel->red = WMCreateRGBColor(scr, 0xffff, 0, 0, True);
	panel->font = WMSystemFontOfSize(scr, 12);
	
	/* Main Window */
	panel->win = WMCreateWindow(scr, "wsoundprefs");
	WMResizeWidget(panel->win, 482, 268);
	WMSetWindowTitle(panel->win, _("WMSound Server Preferences"));
	WMSetWindowCloseAction(panel->win, quit, NULL);
	WMSetWindowMaxSize(panel->win, 482, 268);
	WMSetWindowMinSize(panel->win, 482, 268);
	WMSetWindowMiniwindowTitle(panel->win, _("WSoundPrefs"));
	
	panel->saveBtn = WMCreateCommandButton(panel->win);
	WMResizeWidget(panel->saveBtn, 75, 24);
	WMMoveWidget(panel->saveBtn, 316, 236);
	WMSetButtonText(panel->saveBtn, _("Save"));
	WMSetButtonAction(panel->saveBtn, save, NULL);
	WMHangData(panel->saveBtn, panel);
	
	panel->closeBtn = WMCreateCommandButton(panel->win);
	WMResizeWidget(panel->closeBtn, 75, 24);
	WMMoveWidget(panel->closeBtn, 399, 236);
	WMSetButtonText(panel->closeBtn, _("Close"));
	WMSetButtonAction(panel->closeBtn, quit, NULL);

	/* TabView */
	panel->tabView = WMCreateTabView(panel->win);
	WMMoveWidget(panel->tabView, 8, 8);
	WMResizeWidget(panel->tabView, 466, 220);
	
	/* Event Frame */
	panel->eventF = WMCreateFrame(panel->win);
	WMSetFrameRelief(panel->eventF, WRFlat);
	
	tab = WMCreateTabViewItemWithIdentifier(0);
	WMSetTabViewItemView(tab , WMWidgetView(panel->eventF));
	WMAddItemInTabView(panel->tabView, tab);
	WMSetTabViewItemLabel(tab, _("Sound Events"));
	
		/*********** Sound Event **********/
		panel->sndevntF = WMCreateFrame(panel->eventF);
		WMResizeWidget(panel->sndevntF, 212, 108);
		WMMoveWidget(panel->sndevntF, 8, 10);
		WMSetFrameTitle(panel->sndevntF, _("Sound Event"));
		
		panel->sndevntP = WMCreatePopUpButton(panel->sndevntF);
		WMResizeWidget(panel->sndevntP, 196, 20);
		WMMoveWidget(panel->sndevntP, 8, 16);
		WMAddPopUpButtonItem(panel->sndevntP, _("WMSound Startup"));
		WMAddPopUpButtonItem(panel->sndevntP, _("Application Startup"));
		WMAddPopUpButtonItem(panel->sndevntP, _("Application Exit"));
		WMAddPopUpButtonItem(panel->sndevntP, _("Maximize"));
		WMAddPopUpButtonItem(panel->sndevntP, _("UnMaximize"));
		WMAddPopUpButtonItem(panel->sndevntP, _("Iconify"));
		WMAddPopUpButtonItem(panel->sndevntP, _("DeIconify"));
		WMAddPopUpButtonItem(panel->sndevntP, _("Hide"));
		WMAddPopUpButtonItem(panel->sndevntP, _("UnHide"));
		WMAddPopUpButtonItem(panel->sndevntP, _("Shade"));
		WMAddPopUpButtonItem(panel->sndevntP, _("UnShade"));
		WMSetPopUpButtonAction(panel->sndevntP, changeSoundEvent, panel);
		WMSetPopUpButtonSelectedItem(panel->sndevntP, 0);
		
		panel->sndevntL = WMCreateLabel(panel->sndevntF);
		WMResizeWidget(panel->sndevntL, 196, 60);
		WMMoveWidget(panel->sndevntL, 8, 40);
		WMSetLabelTextAlignment(panel->sndevntL, WACenter);

		WMMapSubwidgets(panel->sndevntF);
		
		
		/*************** Sound File *****************/
		panel->sndfileF = WMCreateFrame(panel->eventF);
		WMResizeWidget(panel->sndfileF, 212, 70);
		WMMoveWidget(panel->sndfileF, 8, 122);
		WMSetFrameTitle(panel->sndfileF, _("Sound File"));
		
		panel->sndfileT = WMCreateTextField(panel->sndfileF);
		WMResizeWidget(panel->sndfileT, 196, 20);
		WMMoveWidget(panel->sndfileT, 8, 16);
		WMHangData(panel->sndfileT, panel);
		WMSetTextFieldDelegate(panel->sndfileT, &sndfileTFDelegate);

		panel->sndplayB = WMCreateCommandButton(panel->sndfileF);
		WMResizeWidget(panel->sndplayB, 60, 20);
		WMMoveWidget(panel->sndplayB, 8, 42);
		WMSetButtonText(panel->sndplayB, _("Play"));
		WMSetButtonAction(panel->sndplayB, (WMAction *)playClick, panel);
	
		panel->sndclearB = WMCreateCommandButton(panel->sndfileF);
		WMResizeWidget(panel->sndclearB, 60, 20);
		WMMoveWidget(panel->sndclearB, 76, 42);
		WMSetButtonText(panel->sndclearB, _("Clear"));
		WMSetButtonAction(panel->sndclearB, (WMAction *)clearClick, panel);
		
		panel->sndbrowseB = WMCreateCommandButton(panel->sndfileF);
		WMResizeWidget(panel->sndbrowseB, 60, 20);
		WMMoveWidget(panel->sndbrowseB, 144, 42);
		WMSetButtonText(panel->sndbrowseB, _("Browse"));
		WMSetButtonAction(panel->sndbrowseB, (WMAction *)browseClick, panel);
		
		WMMapSubwidgets(panel->sndfileF);
		
		/*************** Sound Set ******************/
		panel->sndsetF = WMCreateFrame(panel->eventF);
		WMResizeWidget(panel->sndsetF, 228, 72);
		WMMoveWidget(panel->sndsetF, 228, 10);
		WMSetFrameTitle(panel->sndsetF, _("Sound Set"));
		
		panel->sndsetloadB = WMCreateCommandButton(panel->sndsetF);
		WMResizeWidget(panel->sndsetloadB, 48, 48);
		WMMoveWidget(panel->sndsetloadB, 8, 16);
		WMSetButtonImagePosition(panel->sndsetloadB, WIPImageOnly);
		WMSetButtonAction(panel->sndsetloadB, (WMAction *)loadClick, panel);
		path = locateImage(LOAD_SOUNDSET_ICON_FILE);
		if (path) {
			icon = WMCreateBlendedPixmapFromFile(scr, path, &color);
			if (icon) {
				WMSetButtonImage(panel->sndsetloadB, icon);
				WMReleasePixmap(icon);
			} else {
				wwarning(_("could not load icon %s"), path);
			}
			free(path);
		}
		
		panel->sndsetsaveB = WMCreateCommandButton(panel->sndsetF);
		WMResizeWidget(panel->sndsetsaveB, 48, 48);
		WMMoveWidget(panel->sndsetsaveB, 56, 16);
		WMSetButtonImagePosition(panel->sndsetsaveB, WIPImageOnly);
		WMSetButtonAction(panel->sndsetsaveB, (WMAction *)saveClick, panel);
		path = locateImage(SAVE_SOUNDSET_ICON_FILE);
		if (path) {
			icon = WMCreateBlendedPixmapFromFile(scr, path, &color);
			if (icon) {
				WMSetButtonImage(panel->sndsetsaveB, icon);
				WMReleasePixmap(icon);
			} else {
				wwarning(_("could not load icon %s"), path);
			}
			free(path);
		}
		
		panel->sndsetsL = WMCreateLabel(panel->sndsetF);
		WMResizeWidget(panel->sndsetsL, 108, 48);
		WMMoveWidget(panel->sndsetsL, 112, 16);
		WMSetLabelTextAlignment(panel->sndsetsL, WALeft);
		WMSetLabelText(panel->sndsetsL, _("Load and save an entire soundset."));
		
		WMMapSubwidgets(panel->sndsetF);

		
		/********** sound device ***********/
		panel->devF = WMCreateFrame(panel->eventF);
		WMResizeWidget(panel->devF, 228, 106);
		WMMoveWidget(panel->devF, 228, 86);
		WMSetFrameTitle(panel->devF, _("Sound Device"));
		
		panel->devT = WMCreateTextField(panel->devF);
		WMResizeWidget(panel->devT, 212, 20);
		WMMoveWidget(panel->devT, 8, 16);
		WMHangData(panel->devT, panel);
		WMSetTextFieldDelegate(panel->devT, &devTFDelegate);
		
		panel->devL = WMCreateLabel(panel->devF);
		WMResizeWidget(panel->devL, 212, 54);
		WMMoveWidget(panel->devL, 8, 44);
		WMSetLabelTextAlignment(panel->devL, WACenter);
		WMSetLabelText(panel->devL, _("Specify the sound device that should be "
					"used to play the sounds through."));
		
		WMMapSubwidgets(panel->devF);
									    


	WMMapSubwidgets(panel->eventF);
	

       /* SoundPath Frame */
	panel->soundpathF = WMCreateFrame(panel->win);
	WMSetFrameRelief(panel->soundpathF, WRFlat);
	
	tab = WMCreateTabViewItemWithIdentifier(1);
	WMSetTabViewItemView(tab , WMWidgetView(panel->soundpathF));
	WMAddItemInTabView(panel->tabView, tab);
	WMSetTabViewItemLabel(tab, _("SoundPaths"));
	
		/* *********** Sound Path List **********/
		panel->sndL = WMCreateList(panel->soundpathF);
		WMResizeWidget(panel->sndL, 448, 158);
		WMMoveWidget(panel->sndL, 8, 10);
		WMSetListUserDrawProc(panel->sndL, paintItem);
		WMHangData(panel->sndL, panel);
		
		panel->sndaB = WMCreateCommandButton(panel->soundpathF);
		WMResizeWidget(panel->sndaB, 75, 20);
		WMMoveWidget(panel->sndaB, 298, 174);
		WMSetButtonText(panel->sndaB, _("Add"));
		WMSetButtonAction(panel->sndaB, browseForPath, panel);
		WMSetButtonImagePosition(panel->sndaB, WIPRight);

		panel->sndrB = WMCreateCommandButton(panel->soundpathF);
		WMResizeWidget(panel->sndrB, 75, 20);
		WMMoveWidget(panel->sndrB, 381, 174);
		WMSetButtonText(panel->sndrB, _("Remove"));
		WMSetButtonAction(panel->sndrB, removePath, panel);

	
	WMMapSubwidgets(panel->soundpathF);
	
	/* SoundSetPath Frame */
	panel->soundsetpathF = WMCreateFrame(panel->win);
	WMSetFrameRelief(panel->soundsetpathF, WRFlat);
	
	tab = WMCreateTabViewItemWithIdentifier(2);
	WMSetTabViewItemView(tab , WMWidgetView(panel->soundsetpathF));
	WMAddItemInTabView(panel->tabView, tab);
	WMSetTabViewItemLabel(tab, _("SoundSetPaths"));
	
		/* *********** Sound Path List **********/
		panel->sndsetL = WMCreateList(panel->soundsetpathF);
		WMResizeWidget(panel->sndsetL, 448, 158);
		WMMoveWidget(panel->sndsetL, 8, 10);
		WMSetListUserDrawProc(panel->sndsetL, paintItem);
		WMHangData(panel->sndsetL, panel);
		
		panel->sndsetaB = WMCreateCommandButton(panel->soundsetpathF);
		WMResizeWidget(panel->sndsetaB, 75, 20);
		WMMoveWidget(panel->sndsetaB, 298, 174);
		WMSetButtonText(panel->sndsetaB, _("Add"));
		WMSetButtonAction(panel->sndsetaB, browseForPath, panel);
		WMSetButtonImagePosition(panel->sndsetaB, WIPRight);

		panel->sndsetrB = WMCreateCommandButton(panel->soundsetpathF);
		WMResizeWidget(panel->sndsetrB, 75, 20);
		WMMoveWidget(panel->sndsetrB, 381, 174);
		WMSetButtonText(panel->sndsetrB, _("Remove"));
		WMSetButtonAction(panel->sndsetrB, removePath, panel);

	
	WMMapSubwidgets(panel->soundsetpathF);

	/* About Frame */
	panel->aboutF = WMCreateFrame(panel->win);
	WMSetFrameRelief(panel->aboutF, WRFlat);
	
	tab = WMCreateTabViewItemWithIdentifier(3);
	WMSetTabViewItemView(tab , WMWidgetView(panel->aboutF));
	WMAddItemInTabView(panel->tabView, tab);
	WMSetTabViewItemLabel(tab, _("About"));
	
		/* Icon */
		panel->iconL = WMCreateLabel(panel->aboutF);
		WMResizeWidget(panel->iconL, 72, 72);
		WMMoveWidget(panel->iconL, 82, 10);
		WMSetLabelImagePosition(panel->iconL, WIPImageOnly);
		WMSetLabelImage(panel->iconL, WMGetApplicationIconImage(scr));
		
		/* Title */
		panel->titleL = WMCreateLabel(panel->aboutF);
		WMResizeWidget(panel->titleL, 225, 40);
		WMMoveWidget(panel->titleL, 150, 32);
		WMSetLabelText(panel->titleL, _("WSoundPrefs"));
		WMSetLabelTextAlignment(panel->titleL, WACenter);
		font = WMCreateFont(scr, "-*-lucidabright-medium-r-normal-*-*-250-100-100-*-*-*-*");
		if (!font)
			font = WMBoldSystemFontOfSize(scr, 24);
		WMSetLabelFont(panel->titleL, font);
		WMReleaseFont(font);
		
		/* Authors */
		panel->authorsL = WMCreateLabel(panel->aboutF);
		WMResizeWidget(panel->authorsL, 448, 20);
		WMMoveWidget(panel->authorsL, 8, 88);
		WMSetLabelText(panel->authorsL, "Authors: Pascal Hofstee");
		WMSetLabelTextAlignment(panel->authorsL, WACenter);
		font = WMBoldSystemFontOfSize(scr, 12);
		WMSetLabelFont(panel->authorsL, font);
		WMReleaseFont(font);
		
		/* Copyright */
		panel->copyrightL = WMCreateLabel(panel->aboutF);
		WMResizeWidget(panel->copyrightL, 448, 20);
		WMMoveWidget(panel->copyrightL, 8, 178);
		WMSetLabelTextAlignment(panel->copyrightL, WACenter);
		font = WMCreateNormalFont(scr, "-*-helvetica-medium-r-normal-*-10-*-*-*-*-*-*-*");
		if (font)
			WMSetLabelText(panel->copyrightL, "Copyright \xa9 1999 The Az\xe5rg-r\xfbh");
		else {
			font = WMSystemFontOfSize(scr, 10);
			WMSetLabelText(panel->copyrightL, "Copyright 1999 The Azarg-ruh");
		}
		WMSetLabelFont(panel->copyrightL, font);
		WMReleaseFont(font);
		WMSetLabelTextColor(panel->copyrightL, darkgray);
		WMReleaseColor(darkgray);

		/* System Information */
		panel->infoFrm = WMCreateFrame(panel->aboutF);
		WMResizeWidget(panel->infoFrm, 448, 60);
		WMMoveWidget(panel->infoFrm, 10, 110);
		WMSetFrameRelief(panel->infoFrm, WRGroove);
		
		font = WMSystemFontOfSize(scr, 10);
		
		/* Version Info */
		panel->versionL = WMCreateLabel(panel->infoFrm);
		WMResizeWidget(panel->versionL, 216, 12);
		WMMoveWidget(panel->versionL, 6, 10);
		sprintf(text, "WSoundPrefs %s for WM %s", WVERSION, WMVERSION);
		WMSetLabelText(panel->versionL, text);
		WMSetLabelTextAlignment(panel->versionL, WALeft);
		WMSetLabelFont(panel->versionL, font);
		
		/* WINGs Version */
		panel->wingsL = WMCreateLabel(panel->infoFrm);
		WMResizeWidget(panel->wingsL, 216, 12);
		WMMoveWidget(panel->wingsL, 6, 24);
		sprintf(text, "WINGs Version %d", sysinfo->WINGsVersion);
		WMSetLabelText(panel->wingsL, text);
		WMSetLabelTextAlignment(panel->wingsL, WALeft);
		WMSetLabelFont(panel->wingsL, font);
		
		/* Window Server Version */
		panel->vendorL = WMCreateLabel(panel->infoFrm);
		WMResizeWidget(panel->vendorL, 136, 12);
		WMMoveWidget(panel->vendorL, 6, 38);
		sprintf(text, "Window Server %d", sysinfo->VendorRelease);
		WMSetLabelText(panel->vendorL, text);
		WMSetLabelTextAlignment(panel->vendorL, WALeft);
		WMSetLabelFont(panel->vendorL, font);
		
		/* System Version */
		panel->osL = WMCreateLabel(panel->infoFrm);
		WMResizeWidget(panel->osL, 216, 12);
		WMMoveWidget(panel->osL, 226,  10);
		sprintf(text, "%s %s", sysinfo->SystemName, sysinfo->SystemRelease);
		WMSetLabelText(panel->osL, text);
		WMSetLabelTextAlignment(panel->osL, WARight);
		WMSetLabelFont(panel->osL, font);
		
		/* Processor */
		panel->cpuL = WMCreateLabel(panel->infoFrm);
		WMResizeWidget(panel->cpuL, 216, 12);
		WMMoveWidget(panel->cpuL, 226,  24);
		sprintf(text, "Processor: %s", sysinfo->Processor);
		WMSetLabelText(panel->cpuL, text);
		WMSetLabelTextAlignment(panel->cpuL, WARight);
		WMSetLabelFont(panel->cpuL, font);
		
		/* User Info */
		panel->userL = WMCreateLabel(panel->infoFrm);
		WMResizeWidget(panel->userL, 272, 12);
		WMMoveWidget(panel->userL, 442 - WMWidgetWidth(panel->userL) ,  38);
		sprintf(text, "%s@%s", sysinfo->User, sysinfo->Host);
		WMSetLabelText(panel->userL, text);
		WMSetLabelTextAlignment(panel->userL, WARight);
		WMSetLabelFont(panel->userL, font); 
		
		destroySystemInfo(sysinfo);
		WMReleaseFont(font);
		
	WMMapSubwidgets(panel->aboutF);
	
	
	WMRealizeWidget(panel->win);
	WMMapSubwidgets(panel->win);
	WMMapWidget(panel->win);
	
	
	loadConfigurations(scr, panel->win);
	setSoundPathData(panel);
	setSoundSetPathData(panel);
	WMSetTextFieldText(panel->devT, GetStringForKey("Device"));
	changeSoundEvent(panel->sndevntP, panel);
}

static void
loadConfigurations(WMScreen *scr, WMWindow *mainw)
{
	proplist_t db;
	char *path;
	char mbuf[1024];
	
	path = wdefaultspathfordomain("WMSound");
	db = PLGetProplistWithPath(path);
	if (db) {
		if (!PLIsDictionary(db)) {
			PLRelease(db);
			db = NULL;
			sprintf(mbuf, _("WMSound domain (%s) is corrupted!"), path);
			WMRunAlertPanel(scr, mainw, _("Error"), mbuf, _("OK"), NULL, NULL);
		}
	} else {
		sprintf(mbuf, _("Could not load WMSound domain (%s) from defaults database."), path);
		WMRunAlertPanel(scr, mainw, _("Error"), mbuf, _("OK"), NULL, NULL);
	}
	free(path);
	
	if (!db) {
		db = PLMakeDictionaryFromEntries(NULL, NULL, NULL);
	}
	
	WMSoundDomain = db;
}


char*
locateImage(char *name) {
	char *path;
	char *tmp = wmalloc(strlen(name)+8);
	
	if (isFormatSupported("TIFF")) {
		sprintf(tmp, "%s.tiff", name);
		path = WMPathForResourceOfType(tmp, "tiff");
	} else {
		sprintf(tmp, "%s.xpm", name);
		path = WMPathForResourceOfType(tmp, "xpm");
	}
	free(tmp);
	
	return path;
}

Bool
isFormatSupported(char *format)
{
	char **list;
	int  i;
	
	list = RSupportedFileFormats();
	for (i=0; list[i]!=NULL; i++) {
		if (strcmp(list[i], format) == 0) {
			return True;
		}
	}
	
	return False;
}

