
/*
 * The Real SoundTracker - gtk+ Playlist widget (header)
 *
 * Copyright (C) 1999 Michael Krause
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <string.h>

#include "i18n.h"

#include "playlist.h"
#include "gui-subs.h"

enum {
    SIG_CURRENT_POSITION_CHANGED,
    SIG_RESTART_POSITION_CHANGED,
    SIG_SONG_LENGTH_CHANGED,
    SIG_ENTRY_CHANGED,
    LAST_SIGNAL
};

static guint playlist_signals[LAST_SIGNAL] = { 0 };

void
playlist_freeze (Playlist *p)
{
}

void
playlist_thaw (Playlist *p)
{
}

void
playlist_enable (Playlist *p,
		 gboolean enable)
{
    gtk_widget_set_sensitive(p->spin_songlength, enable);
    gtk_widget_set_sensitive(p->spin_songpat, enable);
    gtk_widget_set_sensitive(p->spin_restartpos, enable);
    gtk_widget_set_sensitive(p->ibutton, enable);
    gtk_widget_set_sensitive(p->dbutton, enable);
}

void
playlist_freeze_signals (Playlist *p)
{
    p->signals_disabled++;
}

void
playlist_thaw_signals (Playlist *p)
{
    g_assert(p->signals_disabled > 0);

    p->signals_disabled--;
}

void
playlist_set_length (Playlist *p, int length)
{
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(p->spin_songlength), length);
}

int
playlist_get_length (Playlist *p)
{
    return p->length;
}

void
playlist_set_nth_pattern (Playlist *p,
			  int pos,
			  int pat)
{
    g_assert(pos >= 0 && pos < p->length);
    g_assert(pat >= p->min_pattern && pat <= p->max_pattern);

    p->patterns[pos] = pat;

    if(pos == p->current_position) {
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(p->spin_songpat), pat);
    }
}

int
playlist_get_nth_pattern (Playlist *p,
			  int pos)
{
    g_assert(pos >= 0 && pos < p->length);

    return p->patterns[pos];
}

void
playlist_set_position (Playlist *p,
		       int pos)
{
    g_assert(pos >= 0 && pos < p->length);

    gtk_spin_button_set_value(GTK_SPIN_BUTTON(p->spin_songpos), pos);
}

int
playlist_get_position (Playlist *p)
{
    return p->current_position;
}

void
playlist_set_restart_position (Playlist *p,
			       int pos)
{
    g_assert(pos >= 0 && pos < p->length);

    gtk_spin_button_set_value(GTK_SPIN_BUTTON(p->spin_restartpos), pos);
}

int
playlist_get_restart_position (Playlist *p)
{
    return p->restart_position;
}

static void
playlist_songpos_changed (GtkSpinButton *spin,
			  Playlist *p)
{
    int newpos;

    g_assert(IS_PLAYLIST(p));

    newpos = gtk_spin_button_get_value_as_int(spin);

    if(newpos == p->current_position)
	return;

    p->current_position = newpos;

    gtk_spin_button_set_value(GTK_SPIN_BUTTON(p->spin_songpat), p->patterns[newpos]);

    if(!p->signals_disabled)
	gtk_signal_emit(GTK_OBJECT(p),
			playlist_signals[SIG_CURRENT_POSITION_CHANGED],
			newpos);
}

static void
playlist_songlength_changed (GtkSpinButton *spin,
			     Playlist *p)
{
    int i, newlen;

    g_assert(IS_PLAYLIST(p));

    newlen = gtk_spin_button_get_value_as_int(spin);
    
    gui_update_spin_adjustment(GTK_SPIN_BUTTON(p->spin_songpos), 0, newlen - 1);
    gui_update_spin_adjustment(GTK_SPIN_BUTTON(p->spin_restartpos), 0, newlen - 1);

    if(newlen == p->length)
	return;

    if(newlen > p->alloc_length) {
	p->patterns = g_renew(int, p->patterns, newlen);
	p->alloc_length = newlen;
    }

    if(newlen > p->length) {
	for(i = p->length; i < newlen; i++) {
	    p->patterns[i] = p->patterns[p->length - 1];
	}
    }

    p->length = newlen;

    if(!p->signals_disabled)
	gtk_signal_emit(GTK_OBJECT(p),
			playlist_signals[SIG_SONG_LENGTH_CHANGED],
			p->length);
}

static void
playlist_songpat_changed (GtkSpinButton *spin,
			  Playlist *p)
{
    int n = gtk_spin_button_get_value_as_int(spin);

    g_assert(IS_PLAYLIST(p));

    if(p->patterns[p->current_position] == n)
	return;

    p->patterns[p->current_position] = n;

    if(!p->signals_disabled)
	gtk_signal_emit(GTK_OBJECT(p),
			playlist_signals[SIG_ENTRY_CHANGED],
			p->current_position, n);
}

static void
playlist_restartpos_changed (GtkSpinButton *spin,
			     Playlist *p)
{
    int n = gtk_spin_button_get_value_as_int(spin);

    g_assert(IS_PLAYLIST(p));

    if(p->restart_position == n)
	return;

    p->restart_position = n;

    if(!p->signals_disabled)
	gtk_signal_emit(GTK_OBJECT(p),
			playlist_signals[SIG_RESTART_POSITION_CHANGED],
			n);
}

static void
playlist_insert_clicked (GtkWidget *w,
			 Playlist *p)
{
    int pos;

    g_assert(IS_PLAYLIST(p));

    if(p->length == p->max_length)
	return;

    p->length++;
    if(p->length > p->alloc_length) {
	p->patterns = g_renew(int, p->patterns, p->length);
	p->alloc_length = p->length;
    }

    pos = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(p->spin_songpos));
    memmove(&p->patterns[pos + 1], &p->patterns[pos], (p->length - pos - 1) * sizeof(int));

    gtk_spin_button_set_value(GTK_SPIN_BUTTON(p->spin_songlength), p->length);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(p->spin_songpat), p->patterns[pos]);
}

static void
playlist_delete_clicked (GtkWidget *w,
			 Playlist *p)
{
    int pos;

    g_assert(IS_PLAYLIST(p));

    if(p->length == 1)
	return;

    p->length--;

    pos = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(p->spin_songpos));
    memmove(&p->patterns[pos], &p->patterns[pos + 1], (p->length - pos) * sizeof(int));

    gtk_spin_button_set_value(GTK_SPIN_BUTTON(p->spin_songlength), p->length);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(p->spin_songpat), p->patterns[pos]);
}


GtkWidget *
playlist_new (void)
{
    Playlist *p;
    GtkWidget *vbox, *hbox, *thing;

    p = gtk_type_new(playlist_get_type());
    GTK_BOX(p)->spacing = 2;
    GTK_BOX(p)->homogeneous = FALSE;

    vbox = GTK_WIDGET(p);

    gui_put_labelled_spin_button(vbox, _("Song length"), 1, p->max_length, &p->spin_songlength, playlist_songlength_changed, p);
    gui_put_labelled_spin_button(vbox, _("Current pos"), 0, p->length - 1, &p->spin_songpos, playlist_songpos_changed, p);
    gui_put_labelled_spin_button(vbox, _("Pattern"), p->min_pattern, p->max_pattern, &p->spin_songpat, playlist_songpat_changed, p);
    gui_put_labelled_spin_button(vbox, _("Restart pos"), 0, p->length - 1, &p->spin_restartpos, playlist_restartpos_changed, p);

    hbox = gtk_hbox_new(TRUE, 4);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);
    gtk_widget_show(hbox);

    p->ibutton = thing = gtk_button_new_with_label(_("Insert"));
    gtk_box_pack_start(GTK_BOX(hbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(playlist_insert_clicked), p);
    
    p->dbutton = thing = gtk_button_new_with_label(_("Delete"));
    gtk_box_pack_start(GTK_BOX(hbox), thing, TRUE, TRUE, 0);
    gtk_widget_show(thing);
    gtk_signal_connect(GTK_OBJECT(thing), "clicked",
		       GTK_SIGNAL_FUNC(playlist_delete_clicked), p);

    return GTK_WIDGET(p);
}

/* --- gtk+ object initialization crap --- */

static void
my_1int_marshal (GtkObject *object,
		 GtkSignalFunc func,
		 gpointer func_data,
		 GtkArg *args)
{
    typedef void (*my_1int_marshal_func) (Playlist *, int);

    my_1int_marshal_func rfunc = (my_1int_marshal_func) func;
    (*rfunc) (PLAYLIST(object),
	      GTK_VALUE_INT(args[0]));
}
  
static void
my_2int_marshal (GtkObject *object,
		 GtkSignalFunc func,
		 gpointer func_data,
		 GtkArg *args)
{
    typedef void (*my_2int_marshal_func) (Playlist *, int, int);

    my_2int_marshal_func rfunc = (my_2int_marshal_func) func;
    (*rfunc) (PLAYLIST(object),
	      GTK_VALUE_INT(args[0]),
	      GTK_VALUE_INT(args[1]));
}
  
static void
playlist_class_init (PlaylistClass *class)
{
    GtkObjectClass *object_class;

    object_class = (GtkObjectClass*) class;

    playlist_signals[SIG_CURRENT_POSITION_CHANGED] = gtk_signal_new ("current_position_changed",
								     GTK_RUN_FIRST,
								     object_class->type,
								     GTK_SIGNAL_OFFSET(PlaylistClass, current_position_changed),
								     my_1int_marshal,
								     GTK_TYPE_NONE, 1,
								     GTK_TYPE_INT);
    playlist_signals[SIG_RESTART_POSITION_CHANGED] = gtk_signal_new ("restart_position_changed",
								     GTK_RUN_FIRST,
								     object_class->type,
								     GTK_SIGNAL_OFFSET(PlaylistClass, restart_position_changed),
								     my_1int_marshal,
								     GTK_TYPE_NONE, 1,
								     GTK_TYPE_INT);
    playlist_signals[SIG_SONG_LENGTH_CHANGED] = gtk_signal_new ("song_length_changed",
								GTK_RUN_FIRST,
								object_class->type,
								GTK_SIGNAL_OFFSET(PlaylistClass, song_length_changed),
								my_1int_marshal,
								GTK_TYPE_NONE, 1,
								GTK_TYPE_INT);
    playlist_signals[SIG_ENTRY_CHANGED] = gtk_signal_new ("entry_changed",
							  GTK_RUN_FIRST,
							  object_class->type,
							  GTK_SIGNAL_OFFSET(PlaylistClass, entry_changed),
							  my_2int_marshal,
							  GTK_TYPE_NONE, 2,
							  GTK_TYPE_INT,
							  GTK_TYPE_INT);

    gtk_object_class_add_signals(object_class, playlist_signals, LAST_SIGNAL);
    
    class->current_position_changed = NULL;
    class->restart_position_changed = NULL;
    class->song_length_changed = NULL;
    class->entry_changed = NULL;
}

static void
playlist_init (Playlist *p)
{
    // These presets should probably be configurable via the interface
    p->max_length = 256;
    p->min_pattern = 0;
    p->max_pattern = 255;

    // A reasonable default playlist
    p->length = 1;
    p->alloc_length = 1;
    p->patterns = g_new0(int, p->alloc_length);
    p->current_position = 0;
    p->restart_position = 0;
    p->signals_disabled = 0;
}

guint
playlist_get_type()
{
    static guint playlist_type = 0;
    
    if (!playlist_type) {
	GtkTypeInfo playlist_info =
	{
	    "Playlist",
	    sizeof(Playlist),
	    sizeof(PlaylistClass),
	    (GtkClassInitFunc) playlist_class_init,
	    (GtkObjectInitFunc) playlist_init,
	    (GtkArgSetFunc) NULL,
	    (GtkArgGetFunc) NULL,
	};
	
	playlist_type = gtk_type_unique (gtk_vbox_get_type (), &playlist_info);
    }

    return playlist_type;
}

