/* tie CLM module into Guile/Scheme */
/* see clm.html and extsnd.html for full documentation */

/* This module uses the float array support in vct.c -- it needs to be loaded with vct.o */

/* every generator that has embedded arrays handles these through an extra layer of
 * pointers; the problem here is that we allow the caller to access and set these directly,
 * (and don't want to copy data unnecessarily), so we can easily have many pointers
 * floating around to the same C memory; there's no way at this level to set up
 * reference counters, so in C, the various free_<gen> functions check that they
 * allocated the given memory (and all vct objects are allocated elsewhere),
 * before freeing an embedded array; then here, all such arrays are wrapped up 
 * as separate SCM objects and at every level only the bottom-most reference allows 
 * the free to go forward.
 */

#if defined(HAVE_CONFIG_H)
  #include "config.h"
#endif
#if defined(WITH_MUS_MODULE) && (!defined(HAVE_SNDLIB))
  #define HAVE_SNDLIB 1
#endif

#include <ctype.h>
#include <stddef.h>
#include <math.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#if (defined(NEXT) || (defined(HAVE_LIBC_H) && (!defined(HAVE_UNISTD_H))))
  #include <libc.h>
#else
  #ifndef _MSC_VER
    #include <unistd.h>
  #endif
#endif

#include <guile/gh.h>
#include "clm.h"
#include "vct.h"

#ifndef CALLOC
  #define CALLOC(a,b)  calloc(a,b)
  #define MALLOC(a,b)  malloc(a,b)
  #define FREE(a)      free(a)
  #define REALLOC(a,b) realloc(a,b)
#endif


void init_mus2scm_module(void);

#ifndef WITH_MUS_MODULE
static void mus_error2scm(int type, char *msg)
{
  scm_misc_error("mus_error",msg,SCM_EOL);
}
#endif


/* ---------------- keywords ---------------- */

#define SC_frequency        ":frequency"
#define SC_initial_phase    ":initial-phase"
#define SC_wave             ":wave"
#define SC_cosines          ":cosines"
#define SC_amplitude        ":amplitude"
#define SC_ratio            ":ratio"
#define SC_size             ":size"
#define SC_a0               ":a0"
#define SC_a1               ":a1"
#define SC_a2               ":a2"
#define SC_b1               ":b1"
#define SC_b2               ":b2"
#define SC_input            ":input"
#define SC_srate            ":srate"
#define SC_file             ":file"
#define SC_channel          ":channel"
#define SC_start            ":start"
#define SC_initial_contents ":initial-contents"
#define SC_initial_element  ":initial-element"
#define SC_scaler           ":scaler"
#define SC_feedforward      ":feedforward"
#define SC_feedback         ":feedback"
#define SC_max_size         ":max-size"
#define SC_radius           ":radius"
#define SC_gain             ":gain"
#define SC_partials         ":partials"
#define SC_r                ":r"
#define SC_a                ":a"
#define SC_n                ":n"
#define SC_fill_time        ":fill-time"
#define SC_order            ":order"
#define SC_x_coeffs         ":xcoeffs"
#define SC_y_coeffs         ":ycoeffs"
#define SC_envelope         ":envelope"
#define SC_base             ":base"
#define SC_duration         ":duration"
#define SC_offset           ":offset"
#define SC_end              ":end"
#define SC_direction        ":direction"
#define SC_degree           ":degree"
#define SC_distance         ":distance"
#define SC_reverb           ":reverb"
#define SC_output           ":output"
#define SC_fft_size         ":fft-size"
#define SC_expansion        ":expansion"
#define SC_length           ":length"
#define SC_hop              ":hop"
#define SC_ramp             ":ramp"
#define SC_jitter           ":jitter"
#define SC_type             ":type"
#define SC_format           ":format"
#define SC_comment          ":comment"
#define SC_channels         ":channels"
#define SC_filter           ":filter"
#define SC_revout           ":revout"
#define SC_width            ":width"

#define NUM_KEYWORDS 56
enum {C_frequency,C_initial_phase,C_wave,C_cosines,C_amplitude,
      C_r,C_ratio,C_size,C_a0,C_a1,C_a2,C_b1,C_b2,C_max_size,
      C_input,C_srate,C_file,C_channel,C_start,
      C_initial_contents,C_initial_element,C_scaler,C_feedforward,C_feedback,
      C_radius,C_gain,C_partials,C_fill_time,C_a,C_n,
      C_order,C_x_coeffs,C_y_coeffs,C_envelope,C_base,C_duration,C_offset,C_end,
      C_direction,C_degree,C_distance,C_reverb,C_output,C_fft_size,
      C_expansion,C_length,C_hop,C_ramp,C_jitter,
      C_type,C_format,C_comment,C_channels,C_filter,C_revout,C_width
};

static char *keywords[NUM_KEYWORDS] = {SC_frequency,SC_initial_phase,SC_wave,SC_cosines,SC_amplitude,
				       SC_r,SC_ratio,SC_size,SC_a0,SC_a1,SC_a2,SC_b1,SC_b2,SC_max_size,
				       SC_input,SC_srate,SC_file,SC_channel,SC_start,
				       SC_initial_contents,SC_initial_element,SC_scaler,SC_feedforward,SC_feedback,
				       SC_radius,SC_gain,SC_partials,SC_fill_time,SC_a,SC_n,
				       SC_order,SC_x_coeffs,SC_y_coeffs,SC_envelope,SC_base,SC_duration,SC_offset,SC_end,
				       SC_direction,SC_degree,SC_distance,SC_reverb,SC_output,SC_fft_size,
				       SC_expansion,SC_length,SC_hop,SC_ramp,SC_jitter,
				       SC_type,SC_format,SC_comment,SC_channels,SC_filter,SC_revout,SC_width
};
static SCM all_keys[NUM_KEYWORDS];
/* what about user-declared keywords? */

short keyword_tag = 0;
static SCM mark_keyword(SCM obj) {SCM_SETGC8MARK(obj); return(SCM_BOOL_F);}
static int c_keyword_p(SCM obj) {return((SCM_NIMP(obj)) && (SCM_TYP16(obj) == (SCM)keyword_tag));}
#define get_keyword(arg) ((int)gh_cdr(arg))
static scm_sizet free_keyword(SCM obj) {return(0);}
static int print_keyword(SCM obj, SCM port, scm_print_state *pstate) {scm_puts(keywords[get_keyword(obj)],port); return(1);}

#if (!HAVE_MAKE_SMOB_TYPE)
static scm_smobfuns keyword_smobfuns = {
  &mark_keyword,
  &free_keyword,
  &print_keyword,
  0};
#endif

static SCM make_keyword(int val)
{
  SCM new_keyword;
  SCM_NEWCELL(new_keyword);
  SCM_SETCAR(new_keyword,keyword_tag);
  SCM_SETCDR(new_keyword,(SCM)val);
  return(new_keyword);
}

static void init_keywords(void)
{
  int i;
#if HAVE_MAKE_SMOB_TYPE
  keyword_tag = scm_make_smob_type_mfpe("keyword",sizeof(SCM),mark_keyword,free_keyword,print_keyword,0);
#else
  keyword_tag = scm_newsmob(&keyword_smobfuns);
#endif
  for (i=0;i<NUM_KEYWORDS;i++) 
    {
      all_keys[i] = make_keyword(i);
      gh_define(keywords[i],all_keys[i]);
    }
}

static int decode_keywords(char *caller, int nkeys, SCM *keys, int nargs, SCM *args, int *orig)
{
  /* implement the "optional-key" notion in CLM */
  int arg_ctr = 0,key_start = 0,rtn_ctr = 0,i,key,keying = 0,key_found = 0;
  arg_ctr = 0;
  while ((arg_ctr < nargs) && (args[arg_ctr] != SCM_UNDEFINED))
    {
      if (!(c_keyword_p(args[arg_ctr])))
	{
	  if (keying) {fprintf(stderr,"unmatched value within keyword section?");}
	  /* type checking on the actual values has to be the caller's problem */
	  if (arg_ctr > nkeys) {fprintf(stderr,"%s (%d > %d) ran out of key space!",caller,arg_ctr,nkeys);}
	  keys[arg_ctr] = args[arg_ctr];
	  orig[arg_ctr] = arg_ctr;
	  arg_ctr++;
	  key_start = arg_ctr;
	  rtn_ctr++;
	}
      else
	{
	  if (arg_ctr == (nargs-1)) 
	    {
	      fprintf(stderr,"key without value? ");
	      break;
	    }
	  keying = 1;
	  key = get_keyword(args[arg_ctr]);
	  if (c_keyword_p(args[arg_ctr+1])) {fprintf(stderr,"two keys in a row?");}
	  if (key_start > nkeys) {fprintf(stderr,"%s has extra trailing args?",caller);}
	  key_found = 0;
	  for (i=key_start;i<nkeys;i++)
	    {
	      if ((c_keyword_p(keys[i])) && (get_keyword(keys[i]) == key))
		{
		  keys[i] = args[arg_ctr+1];
		  orig[i] = arg_ctr+1;
		  arg_ctr += 2;
		  rtn_ctr++;
		  key_found = 1;
		}
	    }
	  if (key_found == 0)
	    {
	      /* either there's a redundant keyword pair or a keyword that 'caller' doesn't recognize */
	      fprintf(stderr,"redundant or invalid key found");
	      arg_ctr += 2;
	    }
	}
    }
  return(rtn_ctr);
}

static float fkeyarg (SCM key, char *caller, int n, SCM val, float def)
{
  if (!(c_keyword_p(key)))
    {
      if (gh_number_p(key))
	return(gh_scm2double(key));
      else scm_wrong_type_arg(caller,n,val);
    }
  return(def);
}

static int ikeyarg (SCM key, char *caller, int n, SCM val, int def)
{
  if (!(c_keyword_p(key)))
    {
      if (gh_number_p(key))
	return(gh_scm2int(key));
      else scm_wrong_type_arg(caller,n,val);
    }
  return(def);
}


/* ---------------- AM and simple stuff ---------------- */

#define S_radians_hz             "radians->hz"
#define S_hz_radians             "hz->radians"
#define S_in_hz                  "in-hz"
#define S_degrees_radians        "degrees->radians"
#define S_radians_degrees        "radians->degrees"
#define S_db_linear              "db->linear"
#define S_linear_db              "linear->db"
#define S_ring_modulate          "ring-modulate"
#define S_amplitude_modulate     "amplitude-modulate"
#define S_contrast_enhancement   "contrast-enhancement"
#define S_set_srate              "mus-set-srate"
#define S_srate                  "mus-srate"
#define S_set_array_print_length "mus-set-array-print-length"
#define S_array_print_length     "mus-array_print_length"
#define S_dot_product            "dot-product"
#define S_clear_array            "clear-array"
#define S_polynomial             "polynomial"
#define S_multiply_arrays        "multiply-arrays"
#define S_make_fft_window        "make-fft-window"
#define S_mus_fft                "mus-fft"
#define S_spectrum               "spectrum"
#define S_convolution            "convolution"
#define S_rectangular2polar      "rectangular->polar"
#define S_array_interp           "array-interp"


static SCM g_radians2hz(SCM val) 
{
  SCM_ASSERT(gh_number_p(val),val,SCM_ARG1,S_radians_hz);
  return(gh_double2scm(mus_radians2hz(gh_scm2double(val))));
}

static SCM g_hz2radians(SCM val) 
{
  SCM_ASSERT(gh_number_p(val),val,SCM_ARG1,S_hz_radians); 
  return(gh_double2scm(mus_hz2radians(gh_scm2double(val))));
}

static SCM g_radians2degrees(SCM val) 
{
  SCM_ASSERT(gh_number_p(val),val,SCM_ARG1,S_radians_degrees); 
  return(gh_double2scm(mus_radians2degrees(gh_scm2double(val))));
}

static SCM g_degrees2radians(SCM val) 
{
  SCM_ASSERT(gh_number_p(val),val,SCM_ARG1,S_degrees_radians); 
  return(gh_double2scm(mus_degrees2radians(gh_scm2double(val))));
}

static SCM g_db2linear(SCM val) 
{
  SCM_ASSERT(gh_number_p(val),val,SCM_ARG1,S_db_linear);
  return(gh_double2scm(mus_db2linear(gh_scm2double(val))));
}

static SCM g_linear2db(SCM val) 
{
  SCM_ASSERT(gh_number_p(val),val,SCM_ARG1,S_linear_db);
  return(gh_double2scm(mus_linear2db(gh_scm2double(val))));
}

/* can't use a variable *srate* directly here because the set! side would not communicate the change to C */
static SCM g_srate(void) 
{
  return(gh_double2scm(mus_srate()));
}

static SCM g_set_srate(SCM val) 
{
  SCM_ASSERT(gh_number_p(val),val,SCM_ARG1,S_set_srate);
  return(gh_double2scm(mus_set_srate(gh_scm2double(val))));
}

static SCM g_array_print_length(void) 
{
  return(gh_int2scm(mus_array_print_length()));
}

static SCM g_set_array_print_length(SCM val) 
{
  SCM_ASSERT(gh_number_p(val),val,SCM_ARG1,S_set_array_print_length);
  return(gh_int2scm(mus_set_array_print_length(gh_scm2int(val))));
}

static SCM g_ring_modulate(SCM val1, SCM val2) 
{
  SCM_ASSERT(gh_number_p(val1),val1,SCM_ARG1,S_ring_modulate);
  SCM_ASSERT(gh_number_p(val2),val2,SCM_ARG2,S_ring_modulate);
  return(gh_double2scm(mus_ring_modulate(gh_scm2double(val1),gh_scm2double(val2))));
}

static SCM g_amplitude_modulate(SCM val1, SCM val2, SCM val3) 
{
  SCM_ASSERT(gh_number_p(val1),val1,SCM_ARG1,S_amplitude_modulate);
  SCM_ASSERT(gh_number_p(val2),val2,SCM_ARG2,S_amplitude_modulate);
  SCM_ASSERT(gh_number_p(val3),val3,SCM_ARG3,S_amplitude_modulate);
  return(gh_double2scm(mus_amplitude_modulate(gh_scm2double(val1),gh_scm2double(val2),gh_scm2double(val3))));
}

static SCM g_contrast_enhancement(SCM val1, SCM val2) 
{
  SCM_ASSERT(gh_number_p(val1),val1,SCM_ARG1,S_contrast_enhancement);
  SCM_ASSERT(gh_number_p(val2),val2,SCM_ARG2,S_contrast_enhancement);
  return(gh_double2scm(mus_contrast_enhancement(gh_scm2double(val1),gh_scm2double(val2))));
}

static SCM g_dot_product(SCM val1, SCM val2) 
{
  vct *v1,*v2;
  SCM_ASSERT(vct_p(val1),val1,SCM_ARG1,S_dot_product);
  SCM_ASSERT(vct_p(val2),val2,SCM_ARG2,S_dot_product);
  v1 = get_vct(val1);
  v2 = get_vct(val2);
  return(gh_double2scm(mus_dot_product(v1->data,v2->data,v1->length)));
}

static SCM g_fft_window_1(char *caller, int choice, SCM val1, SCM val2) 
{
  vct *v1,*v2;
  int len;
  SCM_ASSERT(vct_p(val1),val1,SCM_ARG1,caller);
  SCM_ASSERT(vct_p(val2),val2,SCM_ARG2,caller);
  v1 = get_vct(val1);
  v2 = get_vct(val2);
  len = v1->length;
  if (v2->length < len) len = v2->length;
  if (choice)
    mus_multiply_arrays(v1->data,v2->data,len);
  else mus_rectangular2polar(v1->data,v2->data,len);
  return(val1);
}

static SCM g_multiply_arrays(SCM val1, SCM val2) {return(g_fft_window_1(S_multiply_arrays,TRUE,val1,val2));}
static SCM g_rectangular2polar(SCM val1, SCM val2) {return(g_fft_window_1(S_rectangular2polar,FALSE,val1,val2));}

static SCM g_mus_fft(SCM url, SCM uim, SCM len, SCM usign)
{
  int sign,n;
  vct *v1,*v2;
  SCM_ASSERT((vct_p(url)),url,SCM_ARG1,S_mus_fft);
  SCM_ASSERT((vct_p(uim)),uim,SCM_ARG2,S_mus_fft);
  v1 = get_vct(url);
  v2 = get_vct(uim);
  if (gh_number_p(usign)) sign = gh_scm2int(usign); else sign = 1;
  if (gh_number_p(len)) n = gh_scm2int(len); else n = v1->length;
  mus_fft(v1->data,v2->data,n,sign);
  return(SCM_BOOL_T);
}

static SCM g_make_fft_window(SCM size, SCM type, SCM ubeta)
{
  float beta = 0.0;
  int n;
  SCM_ASSERT((gh_number_p(size)),size,SCM_ARG1,S_make_fft_window);
  SCM_ASSERT((gh_number_p(type)),type,SCM_ARG2,S_make_fft_window);
  if (gh_number_p(ubeta)) beta = gh_scm2double(ubeta);
  n = gh_scm2int(size);
  return(make_vct(n,mus_make_fft_window(n,gh_scm2int(type),beta)));
}

static SCM g_spectrum(SCM url, SCM uim, SCM uwin, SCM un, SCM utype)
{
  int n,type;
  vct *v1,*v2,*v3 = NULL;
  SCM_ASSERT((vct_p(url)),url,SCM_ARG1,S_spectrum);
  SCM_ASSERT((vct_p(uim)),uim,SCM_ARG2,S_spectrum);
  if (uwin != SCM_BOOL_F) SCM_ASSERT((vct_p(uwin)),uwin,SCM_ARG3,S_spectrum);
  v1 = get_vct(url);
  v2 = get_vct(uim);
  if (uwin != SCM_BOOL_F) v3 = get_vct(uwin);
  if (gh_number_p(un)) n = gh_scm2int(un); else n = v1->length;
  if (gh_number_p(utype)) type = gh_scm2int(utype); else type = 1; /* linear normalized */
  mus_spectrum(v1->data,v2->data,(v3) ? (v3->data) : NULL,n,type);
  return(SCM_BOOL_T);
}

static SCM g_convolution(SCM url1, SCM url2, SCM un)
{
  int n;
  vct *v1,*v2;
  SCM_ASSERT((vct_p(url1)),url1,SCM_ARG1,S_convolution);
  SCM_ASSERT((vct_p(url2)),url2,SCM_ARG2,S_convolution);
  v1 = get_vct(url1);
  v2 = get_vct(url2);
  if (gh_number_p(un)) n = gh_scm2int(un); else n = v1->length;
  mus_convolution(v1->data,v2->data,n);
  return(SCM_BOOL_T);
}

static SCM g_clear_array(SCM arr)
{
  vct *v;
  SCM_ASSERT(vct_p(arr),arr,SCM_ARG1,S_clear_array);
  v = get_vct(arr);
  mus_clear_array(v->data,v->length);
  return(SCM_BOOL_F);
}

static SCM g_polynomial(SCM arr, SCM x)
{
  vct *v;
  SCM_ASSERT(vct_p(arr),arr,SCM_ARG1,S_polynomial);
  SCM_ASSERT(gh_number_p(x),x,SCM_ARG2,S_polynomial);
  v = get_vct(arr);
  return(gh_double2scm(mus_polynomial(v->data,gh_scm2double(x),v->length)));
}

static SCM g_array_interp(SCM obj, SCM phase, SCM size) /* opt size */
{
  int len;
  vct *v;
  SCM_ASSERT(vct_p(obj),obj,SCM_ARG1,S_array_interp);
  SCM_ASSERT((gh_number_p(phase)),phase,SCM_ARG2,S_array_interp);
  v = get_vct(obj);
  if (gh_number_p(size)) len = gh_scm2int(size); else len = v->length;
  return(gh_double2scm(mus_array_interp(v->data,gh_scm2double(phase),len)));
}

#ifdef __cplusplus
  static SCM gh_new_procedure3_1 (char *proc_name,SCM (*fn)(...)) {return(gh_new_procedure(proc_name,fn,3,1,0));}
#else
  static SCM gh_new_procedure3_1 (char *proc_name,SCM (*fn)()) {return(gh_new_procedure(proc_name,fn,3,1,0));}
#endif

static void init_simple_stuff(void)
{
  gh_new_procedure1_0(S_radians_hz,g_radians2hz);
  gh_new_procedure1_0(S_hz_radians,g_hz2radians);
  gh_new_procedure1_0(S_in_hz,g_hz2radians);
  gh_new_procedure1_0(S_radians_degrees,g_radians2degrees);
  gh_new_procedure1_0(S_degrees_radians,g_degrees2radians);
  gh_new_procedure1_0(S_db_linear,g_db2linear);
  gh_new_procedure1_0(S_linear_db,g_linear2db);
  gh_new_procedure0_0(S_srate,g_srate);
  gh_new_procedure1_0(S_set_srate,g_set_srate);
  gh_new_procedure0_0(S_array_print_length,g_array_print_length);
  gh_new_procedure1_0(S_set_array_print_length,g_set_array_print_length);
  gh_new_procedure2_0(S_ring_modulate,g_ring_modulate);
  gh_new_procedure3_0(S_amplitude_modulate,g_amplitude_modulate);
  gh_new_procedure2_0(S_contrast_enhancement,g_contrast_enhancement);
  gh_new_procedure2_0(S_dot_product,g_dot_product);
  gh_new_procedure1_0(S_clear_array,g_clear_array);
  gh_new_procedure2_0(S_polynomial,g_polynomial);
  gh_new_procedure2_0(S_multiply_arrays,g_multiply_arrays);
  gh_new_procedure2_1(S_make_fft_window,g_make_fft_window);
  gh_new_procedure3_1(S_mus_fft,g_mus_fft);
  gh_new_procedure(S_spectrum,g_spectrum,3,2,0);
  gh_new_procedure2_1(S_convolution,g_convolution);
  gh_new_procedure2_0(S_rectangular2polar,g_rectangular2polar);
  gh_new_procedure(S_array_interp,g_array_interp,2,1,0);
}


/* ---------------- mus-scm struct ---------------- */

typedef struct {
  mus_any *gen;
  SCM *vcts; /* one for each accessible float array (wrapped up here in a vct object) */
  int nvcts;
} mus_scm;

static short mus_scm_tag = 0;
#define mus_get_any(arg) (((mus_scm *)gh_cdr(arg))->gen)
#define mus_get_scm(arg) ((mus_scm *)gh_cdr(arg))

#define GH_TYPE_OF(a) (SCM_TYP16(a))
static int mus_scm_p(SCM obj) {return((SCM_NIMP(obj)) && (GH_TYPE_OF(obj) == (SCM)mus_scm_tag));}

static SCM mark_mus_scm(SCM obj) 
{
  int i;
  mus_scm *ms;
  ms = mus_get_scm(obj);
  for (i=0;i<ms->nvcts;i++) if (ms->vcts[i]) scm_gc_mark(ms->vcts[i]);
  SCM_SETGC8MARK(obj); 
  return(SCM_BOOL_F);
}

static scm_sizet free_mus_scm(SCM obj) 
{
  mus_scm *ms;
  ms = mus_get_scm(obj);
  if (ms->nvcts != -1) mus_free(mus_get_any(obj));
  if (ms->vcts) FREE(ms->vcts);  
  FREE(ms);
  return(0);
}

static int print_mus_scm(SCM obj, SCM port, scm_print_state *pstate)
{
  char *buf;
  buf = mus_describe(mus_get_any(obj));
  scm_puts(buf,port);
  FREE(buf);
  return(1);
}

static SCM equalp_mus_scm(SCM obj1, SCM obj2) 
{
  return((mus_equalp(mus_get_any(obj1),mus_get_any(obj2))) ? SCM_BOOL_T : SCM_BOOL_F);
}

#if (!HAVE_MAKE_SMOB_TYPE)
static scm_smobfuns mus_scm_smobfuns = {
  &mark_mus_scm,
  &free_mus_scm,
  &print_mus_scm,
  &equalp_mus_scm};
#endif

static void init_mus_scm(void)
{
#if HAVE_MAKE_SMOB_TYPE
  mus_scm_tag = scm_make_smob_type_mfpe("mus",sizeof(mus_scm),mark_mus_scm,free_mus_scm,print_mus_scm,equalp_mus_scm);
#else
  mus_scm_tag = scm_newsmob(&mus_scm_smobfuns); 
#endif
}



/* ---------------- generic functions ---------------- */

#define S_phase          "mus-phase"
#define S_set_phase      "mus-set-phase"
#define S_frequency      "mus-frequency"
#define S_set_frequency  "mus-set-frequency"
#define S_length         "mus-length"
#define S_set_length     "mus-set-length"
#define S_data           "mus-data"
#define S_set_data       "mus-set-data"
#define S_scaler         "mus-scaler"
#define S_set_scaler     "mus-set-scaler"
#define S_inspect        "mus-inspect"
#define S_describe       "mus-describe"

static SCM g_inspect(SCM gen)
{
  SCM_ASSERT((mus_scm_p(gen)),gen,SCM_ARG1,S_inspect);
  return(gh_str02scm(mus_inspect(mus_get_any(gen))));
}

static SCM g_describe(SCM gen) 
{
  SCM_ASSERT((mus_scm_p(gen)),gen,SCM_ARG1,S_describe);
  return(gh_str02scm(mus_describe(mus_get_any(gen))));
}

static SCM g_phase(SCM gen) 
{
  SCM_ASSERT((mus_scm_p(gen)),gen,SCM_ARG1,S_phase);
  return(gh_double2scm(mus_phase(mus_get_any(gen))));
}

static SCM g_set_phase(SCM gen, SCM val) 
{
  SCM_ASSERT((mus_scm_p(gen)),gen,SCM_ARG1,S_set_phase);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG2,S_set_phase);
  return(gh_double2scm(mus_set_phase(mus_get_any(gen),gh_scm2double(val))));
}

static SCM g_scaler(SCM gen) 
{
  SCM_ASSERT((mus_scm_p(gen)),gen,SCM_ARG1,S_scaler);
  return(gh_double2scm(mus_scaler(mus_get_any(gen))));
}

static SCM g_set_scaler(SCM gen, SCM val) 
{
  SCM_ASSERT((mus_scm_p(gen)),gen,SCM_ARG1,S_set_scaler);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG2,S_set_scaler);
  return(gh_double2scm(mus_set_scaler(mus_get_any(gen),gh_scm2double(val))));
}

static SCM g_frequency(SCM gen) 
{
  SCM_ASSERT((mus_scm_p(gen)),gen,SCM_ARG1,S_frequency);
  return(gh_double2scm(mus_frequency(mus_get_any(gen))));
}

static SCM g_set_frequency(SCM gen, SCM val) 
{
  SCM_ASSERT((mus_scm_p(gen)),gen,SCM_ARG1,S_frequency);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG2,S_set_frequency);
  return(gh_double2scm(mus_set_frequency(mus_get_any(gen),gh_scm2double(val))));
}

static SCM g_length(SCM gen) 
{
  SCM_ASSERT((mus_scm_p(gen)),gen,SCM_ARG1,S_length);
  return(gh_int2scm(mus_length(mus_get_any(gen))));
}

static SCM g_set_length(SCM gen, SCM val) 
{
  SCM_ASSERT((mus_scm_p(gen)),gen,SCM_ARG1,S_length);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG2,S_set_length);
  return(gh_int2scm(mus_set_length(mus_get_any(gen),gh_scm2int(val))));
}

#define MUS_DATA_POSITION 0

static SCM g_data(SCM gen) 
{
  mus_scm *ms;
  SCM_ASSERT((mus_scm_p(gen)),gen,SCM_ARG1,S_data);
  ms = mus_get_scm(gen);
  if (ms->vcts)
    return(ms->vcts[MUS_DATA_POSITION]); 
  else return(SCM_BOOL_F);
}

static SCM g_set_data(SCM gen, SCM val) 
{
  mus_scm *ms;
  mus_any *ma;
  vct *v;
  SCM_ASSERT((mus_scm_p(gen)),gen,SCM_ARG1,S_set_data);
  SCM_ASSERT((vct_p(val)),val,SCM_ARG2,S_set_data);
  ms = mus_get_scm(gen);
  if (ms->vcts)
    {
      v = (vct *)gh_cdr(val);
      ma = ms->gen;
      mus_set_data(ma,v->data);  /* TO REMEMBER: if allocated, should have freed, and set to not allocated */
      ms->vcts[MUS_DATA_POSITION] = val;
      return(val);
    }
  return(SCM_BOOL_F);
}

static void init_generic_funcs(void)
{
  gh_new_procedure1_0(S_inspect,g_inspect);
  gh_new_procedure1_0(S_describe,g_describe);
  gh_new_procedure1_0(S_phase,g_phase);
  gh_new_procedure2_0(S_set_phase,g_set_phase);
  gh_new_procedure1_0(S_scaler,g_scaler);
  gh_new_procedure2_0(S_set_scaler,g_set_scaler);
  gh_new_procedure1_0(S_frequency,g_frequency);
  gh_new_procedure2_0(S_set_frequency,g_set_frequency);
  gh_new_procedure1_0(S_length,g_length);
  gh_new_procedure2_0(S_set_length,g_set_length);
  gh_new_procedure1_0(S_data,g_data);
  gh_new_procedure2_0(S_set_data,g_set_data);
}


/* ---------------- oscil ---------------- */

#define S_make_oscil "make-oscil"
#define S_oscil "oscil"
#define S_oscil_p "oscil?"

static SCM g_make_oscil(SCM arg1, SCM arg2, SCM arg3, SCM arg4)
{
  SCM new_osc;
  mus_scm *gn;
  int vals;
  SCM args[4],keys[2];
  int orig_arg[2] = {0,0};
  float freq = 440.0,phase = 0.0;
  keys[0] = all_keys[C_frequency];
  keys[1] = all_keys[C_initial_phase];
  args[0] = arg1; args[1] = arg2; args[2] = arg3; args[3] = arg4; 
  vals = decode_keywords(S_make_oscil,2,keys,4,args,orig_arg);
  if (vals > 0)
    {
      freq = fkeyarg(keys[0],S_make_oscil,orig_arg[0]+1,args[orig_arg[0]],freq);
      phase = fkeyarg(keys[1],S_make_oscil,orig_arg[1]+1,args[orig_arg[1]],phase);
    }
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->gen = mus_make_oscil(freq,phase);
  gn->nvcts = 0;
  SCM_NEWCELL(new_osc);
  SCM_SETCAR(new_osc,mus_scm_tag);
  SCM_SETCDR(new_osc,(SCM)gn);
  return(new_osc);
}

static SCM g_oscil(SCM os, SCM fm, SCM pm)
{
  float fm1 = 0.0,pm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(os)) && (mus_oscil_p(mus_get_any(os)))),os,SCM_ARG1,S_oscil);
  if (gh_number_p(fm)) fm1 = gh_scm2double(fm); else if (fm != SCM_UNDEFINED) scm_wrong_type_arg(S_oscil,2,fm);
  if (gh_number_p(pm)) pm1 = gh_scm2double(pm); else if (pm != SCM_UNDEFINED) scm_wrong_type_arg(S_oscil,3,pm);
  return(gh_double2scm(mus_oscil(mus_get_any(os),fm1,pm1)));
}

static SCM g_oscil_p(SCM os) {return(((mus_scm_p(os)) && (mus_oscil_p(mus_get_any(os)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static void init_oscil(void)
{
  gh_new_procedure1_0(S_oscil_p,g_oscil_p);
  gh_new_procedure(S_make_oscil,g_make_oscil,0,4,0);
  gh_new_procedure1_2(S_oscil,g_oscil);
}



/* ---------------- delay ---------------- */

#define S_make_delay "make-delay"
#define S_delay "delay"
#define S_delay_p "delay?"
#define S_tap "tap"
#define S_comb "comb"
#define S_make_comb "make-comb"
#define S_comb_p "comb?"
#define S_notch "notch"
#define S_make_notch "make-notch"
#define S_notch_p "notch?"
#define S_all_pass "all-pass"
#define S_make_all_pass "make-all-pass"
#define S_all_pass_p "all-pass?"
#define S_feedback "mus-feedback"
#define S_set_feedback "mus-set-feedback"
#define S_feedforward "mus-feedforward"
#define S_set_feedforward "mus-set-feedforward"

enum {G_DELAY,G_COMB,G_NOTCH,G_ALL_PASS};

static SCM g_make_delay_1(int choice, SCM arglist)
{
  SCM new_dly;
  mus_scm *gn;
  vct *v;
  char *caller,*errstr;
  SCM args[14],keys[7];
  int orig_arg[7] = {0,0,0,0,0,0,0};
  int vals,i,argn=0,len,arglist_len,keyn,max_size = -1;
  int size = 1;
  float *line = NULL;
  float scaler = 0.0, feedback = 0.0, feedforward = 0.0;
  SCM initial_contents = SCM_UNDEFINED;
  float initial_element = 0.0;
  
  switch (choice)
    {
    case G_DELAY:   caller = S_make_delay;                                       break;
    case G_COMB:    caller = S_make_comb;     keys[argn++] = all_keys[C_scaler]; break;
    case G_NOTCH:   caller = S_make_notch;    keys[argn++] = all_keys[C_scaler]; break;
    case G_ALL_PASS: caller = S_make_all_pass; keys[argn++] = all_keys[C_feedback]; keys[argn++] = all_keys[C_feedforward]; break;
    }
  keys[argn++] = all_keys[C_size];
  keys[argn++] = all_keys[C_initial_contents];
  keys[argn++] = all_keys[C_initial_element];
  keys[argn++] = all_keys[C_max_size];
  for (i=0;i<14;i++) args[i] = SCM_UNDEFINED;
  arglist_len = gh_length(arglist);
  for (i=0;i<arglist_len;i++) args[i] = gh_list_ref(arglist,gh_int2scm(i));
  vals = decode_keywords(caller,argn,keys,argn*2,args,orig_arg);
  if (vals > 0)
    {
      keyn = 0;
      switch (choice)
	{
	case G_DELAY: 
	  break;
	case G_COMB: case G_NOTCH:
	  scaler = fkeyarg(keys[keyn],caller,orig_arg[keyn]+1,args[orig_arg[keyn]],scaler);
	  keyn++;
	  break;
	case G_ALL_PASS:
	  feedback = fkeyarg(keys[keyn],caller,orig_arg[keyn]+1,args[orig_arg[keyn]],feedback);
	  keyn++;
	  feedforward = fkeyarg(keys[keyn],caller,orig_arg[keyn]+1,args[orig_arg[keyn]],feedforward);
	  keyn++;
	  break;
	}
      size = ikeyarg(keys[keyn],caller,orig_arg[keyn]+1,args[orig_arg[keyn]],size);
      keyn++;
      if (!(c_keyword_p(keys[keyn])))
	{
	  initial_contents = keys[keyn];
	  if (vct_p(initial_contents))
	    {
	      v = get_vct(initial_contents);
	      line = v->data;
	    }
	  else
	    {
	      if (gh_list_p(initial_contents))
		{
		  len = gh_length(initial_contents);
		  line = (float *)CALLOC(len,sizeof(float));
		  for (i=0;i<len;i++) line[i] = gh_scm2double(gh_list_ref(initial_contents,gh_int2scm(i)));
		}
	    }
	}
      keyn++;
      initial_element = fkeyarg(keys[keyn],caller,orig_arg[keyn]+1,args[orig_arg[keyn]],0.0);
      keyn++;
      max_size = ikeyarg(keys[keyn],caller,orig_arg[keyn]+1,args[orig_arg[keyn]],size);
    }
  if (max_size == -1) max_size = size;
  if (max_size <= 0)
    {
      errstr = (char *)CALLOC(64,sizeof(char));
      sprintf(errstr,"%s: delay line length is %d?",caller,max_size);
      scm_misc_error(caller,errstr,arglist);
      FREE(errstr);
    }
  if (line == NULL) line = (float *)CALLOC(max_size,sizeof(float));
  if (initial_element != 0.0) for (i=0;i<max_size;i++) line[i] = initial_element;
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  switch (choice)
    {
    case G_DELAY: gn->gen = mus_make_delay(size,line,max_size); break;
    case G_COMB: gn->gen = mus_make_comb(scaler,size,line,max_size); break;
    case G_NOTCH: gn->gen = mus_make_notch(scaler,size,line,max_size); break;
    case G_ALL_PASS: gn->gen = mus_make_all_pass(feedback,feedforward,size,line,max_size); break;
    }
  gn->nvcts = 1;
  gn->vcts = (SCM *)CALLOC(1,sizeof(SCM));
  SCM_NEWCELL(new_dly);
  SCM_SETCAR(new_dly,mus_scm_tag);
  SCM_SETCDR(new_dly,(SCM)gn);
  gn->vcts[MUS_DATA_POSITION] = make_vct(max_size,line);
  return(new_dly);
}

static SCM g_make_delay(SCM args) {return(g_make_delay_1(G_DELAY,args));}
static SCM g_make_comb(SCM args) {return(g_make_delay_1(G_COMB,args));}
static SCM g_make_notch(SCM args) {return(g_make_delay_1(G_NOTCH,args));}
static SCM g_make_all_pass(SCM args) {return(g_make_delay_1(G_ALL_PASS,args));}

static SCM g_delay(SCM obj, SCM input, SCM pm)
{
  float in1 = 0.0, pm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_delay_p(mus_get_any(obj)))),obj,SCM_ARG1,S_delay);
  if (gh_number_p(input)) in1 = gh_scm2double(input); else if (input != SCM_UNDEFINED) scm_wrong_type_arg(S_delay,2,input);
  if (gh_number_p(pm)) pm1 = gh_scm2double(pm); else if (pm != SCM_UNDEFINED) scm_wrong_type_arg(S_delay,3,pm);
  return(gh_double2scm(mus_delay(mus_get_any(obj),in1,pm1)));
}

static SCM g_notch(SCM obj, SCM input, SCM pm)
{
  float in1 = 0.0, pm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_notch_p(mus_get_any(obj)))),obj,SCM_ARG1,S_notch);
  if (gh_number_p(input)) in1 = gh_scm2double(input); else if (input != SCM_UNDEFINED) scm_wrong_type_arg(S_notch,2,input);
  if (gh_number_p(pm)) pm1 = gh_scm2double(pm); else if (pm != SCM_UNDEFINED) scm_wrong_type_arg(S_notch,3,pm);
  return(gh_double2scm(mus_notch(mus_get_any(obj),in1,pm1)));
}

static SCM g_comb(SCM obj, SCM input, SCM pm)
{
  float in1 = 0.0, pm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_comb_p(mus_get_any(obj)))),obj,SCM_ARG1,S_comb);
  if (gh_number_p(input)) in1 = gh_scm2double(input); else if (input != SCM_UNDEFINED) scm_wrong_type_arg(S_comb,2,input);
  if (gh_number_p(pm)) pm1 = gh_scm2double(pm); else if (pm != SCM_UNDEFINED) scm_wrong_type_arg(S_comb,3,pm);
  return(gh_double2scm(mus_comb(mus_get_any(obj),in1,pm1)));
}

static SCM g_all_pass(SCM obj, SCM input, SCM pm)
{
  float in1 = 0.0, pm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_all_pass_p(mus_get_any(obj)))),obj,SCM_ARG1,S_all_pass);
  if (gh_number_p(input)) in1 = gh_scm2double(input); else if (input != SCM_UNDEFINED) scm_wrong_type_arg(S_all_pass,2,input);
  if (gh_number_p(pm)) pm1 = gh_scm2double(pm); else if (pm != SCM_UNDEFINED) scm_wrong_type_arg(S_all_pass,3,pm);
  return(gh_double2scm(mus_all_pass(mus_get_any(obj),in1,pm1)));
}

static SCM g_tap(SCM obj, SCM loc)
{
  float dloc = 0.0;
  SCM_ASSERT((mus_scm_p(obj)),obj,SCM_ARG1,S_tap);
  if (gh_number_p(loc)) dloc = gh_scm2double(loc); else if (loc != SCM_UNDEFINED) scm_wrong_type_arg(S_tap,2,loc);
  return(gh_double2scm(mus_tap(mus_get_any(obj),dloc)));
}

static SCM g_delay_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_delay_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}
static SCM g_comb_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_comb_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}
static SCM g_notch_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_notch_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}
static SCM g_all_pass_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_all_pass_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_feedback(SCM obj)
{
  SCM_ASSERT((mus_scm_p(obj)),obj,SCM_ARG1,S_feedback);
  return(gh_double2scm(mus_feedback(mus_get_any(obj))));
}

static SCM g_set_feedback(SCM obj, SCM val)
{
  SCM_ASSERT((mus_scm_p(obj)),obj,SCM_ARG1,S_set_feedback);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG2,S_set_feedback);
  return(gh_double2scm(mus_set_feedback(mus_get_any(obj),gh_scm2double(val))));
}

static SCM g_feedforward(SCM obj)
{
  SCM_ASSERT((mus_scm_p(obj)),obj,SCM_ARG1,S_feedforward);
  return(gh_double2scm(mus_feedforward(mus_get_any(obj))));
}

static SCM g_set_feedforward(SCM obj, SCM val)
{
  SCM_ASSERT((mus_scm_p(obj)),obj,SCM_ARG1,S_set_feedforward);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG2,S_set_feedforward);
  return(gh_double2scm(mus_set_feedforward(mus_get_any(obj),gh_scm2double(val))));
}

static void init_dly(void)
{
  gh_eval_str("(define %delay delay)"); /* protect the original meaning */
  gh_new_procedure(S_make_delay,g_make_delay,0,0,1);
  gh_new_procedure(S_make_comb,g_make_comb,0,0,1);
  gh_new_procedure(S_make_notch,g_make_notch,0,0,1);
  gh_new_procedure(S_make_all_pass,g_make_all_pass,0,0,1);
  gh_new_procedure(S_delay,g_delay,1,2,0);
  gh_new_procedure(S_tap,g_tap,1,1,0);
  gh_new_procedure(S_notch,g_notch,1,2,0);
  gh_new_procedure(S_comb,g_comb,1,2,0);
  gh_new_procedure(S_all_pass,g_all_pass,1,2,0);
  gh_new_procedure(S_delay_p,g_delay_p,1,0,0);
  gh_new_procedure(S_notch_p,g_notch_p,1,0,0);
  gh_new_procedure(S_comb_p,g_comb_p,1,0,0);
  gh_new_procedure(S_all_pass_p,g_all_pass_p,1,0,0);
  gh_new_procedure(S_feedback,g_feedback,1,0,0);
  gh_new_procedure(S_set_feedback,g_set_feedback,2,0,0);
  gh_new_procedure(S_feedforward,g_feedforward,1,0,0);
  gh_new_procedure(S_set_feedforward,g_set_feedforward,2,0,0);
}


/* -------- sum-of-cosines -------- */

#define S_make_sum_of_cosines "make-sum-of-cosines"
#define S_sum_of_cosines "sum-of-cosines"
#define S_sum_of_cosines_p "sum-of-cosines?"
#define S_cosines "mus-cosines"

static SCM g_sum_of_cosines_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_sum_of_cosines_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_make_sum_of_cosines(SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6)
{
  SCM new_cosp;
  mus_scm *gn;
  SCM args[6],keys[3];
  int orig_arg[3] = {0,0,0};
  int vals;
  int cosines = 1;
  float freq = 440.0;
  float phase = 0.0;
  keys[0] = all_keys[C_cosines];
  keys[1] = all_keys[C_frequency];
  keys[2] = all_keys[C_initial_phase];
  args[0] = arg1; args[1] = arg2; args[2] = arg3; args[3] = arg4; args[4] = arg5; args[5] = arg6; 
  vals = decode_keywords(S_make_sum_of_cosines,3,keys,6,args,orig_arg);
  if (vals > 0)
    {
      cosines = ikeyarg(keys[0],S_make_sum_of_cosines,orig_arg[0]+1,args[orig_arg[0]],cosines);
      freq = fkeyarg(keys[1],S_make_sum_of_cosines,orig_arg[1]+1,args[orig_arg[1]],freq);
      phase = fkeyarg(keys[2],S_make_sum_of_cosines,orig_arg[2]+1,args[orig_arg[2]],phase);
    }
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->gen = mus_make_sum_of_cosines(cosines,freq,phase);
  gn->nvcts = 0;
  SCM_NEWCELL(new_cosp);
  SCM_SETCAR(new_cosp,mus_scm_tag);
  SCM_SETCDR(new_cosp,(SCM)gn);
  return(new_cosp);
}

static SCM g_sum_of_cosines(SCM obj, SCM fm)
{
  float fm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_sum_of_cosines_p(mus_get_any(obj)))),obj,SCM_ARG1,S_sum_of_cosines);
  if (gh_number_p(fm)) fm1 = gh_scm2double(fm); else if (fm != SCM_UNDEFINED) scm_wrong_type_arg(S_sum_of_cosines,2,fm);
  return(gh_double2scm(mus_sum_of_cosines(mus_get_any(obj),fm1)));
}

static SCM g_cosines(SCM obj)
{
  SCM_ASSERT((mus_scm_p(obj)),obj,SCM_ARG1,S_cosines);
  return(gh_double2scm(mus_cosines(mus_get_any(obj))));
}

static void init_cosp(void)
{
  gh_new_procedure(S_make_sum_of_cosines,g_make_sum_of_cosines,0,6,0);
  gh_new_procedure(S_sum_of_cosines,g_sum_of_cosines,1,1,0);
  gh_new_procedure(S_sum_of_cosines_p,g_sum_of_cosines_p,1,0,0);
  gh_new_procedure(S_cosines,g_cosines,1,0,0);
}



/* ---------------- rand, rand_interp ---------------- */

#define S_make_rand "make-rand"
#define S_rand "rand"
#define S_rand_p "rand?"
#define S_make_rand_interp "make-rand-interp"
#define S_rand_interp "rand-interp"
#define S_rand_interp_p "rand-interp?"
#define S_set_rand_seed "mus-set-rand-seed"
#define S_mus_random "mus-random"

static SCM g_make_noi(int rand_case, SCM arg1, SCM arg2, SCM arg3, SCM arg4)
{
  SCM new_noi;
  mus_scm *gn;
  SCM args[4],keys[2];
  int orig_arg[2] = {0,0};
  int vals;
  float freq = 440.0;
  float base = 1.0;
  keys[0] = all_keys[C_frequency];
  keys[1] = all_keys[C_amplitude];
  args[0] = arg1; args[1] = arg2; args[2] = arg3; args[3] = arg4;
  vals = decode_keywords(S_make_rand,2,keys,4,args,orig_arg);
  if (vals > 0)
    {
      freq = fkeyarg(keys[0],S_make_rand,orig_arg[0]+1,args[orig_arg[0]],freq);
      base = fkeyarg(keys[1],S_make_rand,orig_arg[1]+1,args[orig_arg[1]],base);
    }
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  if (rand_case)
    gn->gen = mus_make_rand(freq,base);
  else gn->gen = mus_make_rand_interp(freq,base);
  SCM_NEWCELL(new_noi);
  SCM_SETCAR(new_noi,mus_scm_tag);
  SCM_SETCDR(new_noi,(SCM)gn);
  return(new_noi);
}

static SCM g_make_rand_interp(SCM arg1, SCM arg2, SCM arg3, SCM arg4)
{
  return(g_make_noi(FALSE,arg1,arg2,arg3,arg4));
}

static SCM g_make_rand(SCM arg1, SCM arg2, SCM arg3, SCM arg4)
{
  return(g_make_noi(TRUE,arg1,arg2,arg3,arg4));
}

static SCM g_rand(SCM obj, SCM fm)
{
  float fm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_rand_p(mus_get_any(obj)))),obj,SCM_ARG1,S_rand);
  if (gh_number_p(fm)) fm1 = gh_scm2double(fm); else if (fm != SCM_UNDEFINED) scm_wrong_type_arg(S_rand,2,fm);
  return(gh_double2scm(mus_rand(mus_get_any(obj),fm1)));
}

static SCM g_rand_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_rand_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_rand_interp(SCM obj, SCM fm)
{
  float fm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_rand_interp_p(mus_get_any(obj)))),obj,SCM_ARG1,S_rand_interp);
  if (gh_number_p(fm)) fm1 = gh_scm2double(fm); else if (fm != SCM_UNDEFINED) scm_wrong_type_arg(S_rand_interp,2,fm);
  return(gh_double2scm(mus_rand_interp(mus_get_any(obj),fm1)));
}

static SCM g_rand_interp_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_rand_interp_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_mus_random(SCM a) 
{
  SCM_ASSERT((gh_number_p(a)),a,SCM_ARG1,S_mus_random);
  return(gh_double2scm(mus_random(gh_scm2double(a))));
}

static SCM g_set_rand_seed(SCM a) 
{
  SCM_ASSERT((gh_number_p(a)),a,SCM_ARG1,S_set_rand_seed);
  mus_set_rand_seed(gh_scm2int(a)); 
  return(a);
}

static void init_noi(void)
{
  gh_new_procedure(S_make_rand,g_make_rand,0,4,0);
  gh_new_procedure(S_make_rand_interp,g_make_rand_interp,0,4,0);
  gh_new_procedure(S_rand,g_rand,1,1,0);
  gh_new_procedure(S_rand_interp,g_rand_interp,1,1,0);
  gh_new_procedure(S_rand_p,g_rand_p,1,0,0);
  gh_new_procedure(S_rand_interp_p,g_rand_interp_p,1,0,0);
  gh_new_procedure1_0(S_mus_random,g_mus_random);
  gh_new_procedure1_0(S_set_rand_seed,g_set_rand_seed);
}



/* ---------------- table lookup ---------------- */

static int DEFAULT_TABLE_SIZE = 512;

#define S_table_lookup_p "table-lookup?"
#define S_make_table_lookup "make-table-lookup"
#define S_table_lookup "table-lookup"
#define S_partials2wave "partials->wave"
#define S_phasepartials2wave "phase-partials->wave"

static SCM g_table_lookup_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_table_lookup_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_partials2wave(SCM partials, SCM utable, SCM normalize)
{
  vct *f;
  SCM table;
  float *partial_data,*wave;
  int len,i;
  SCM_ASSERT(gh_list_p(partials),partials,SCM_ARG1,S_partials2wave);
  if (utable == SCM_UNDEFINED)
    {
      wave = (float *)CALLOC(DEFAULT_TABLE_SIZE,sizeof(float));
      table = make_vct(DEFAULT_TABLE_SIZE,wave);
    }
  else table = utable;
  f = get_vct(table);
  len = gh_length(partials);
  partial_data = (float *)CALLOC(len,sizeof(float));
  for (i=0;i<len;i++) partial_data[i] = gh_scm2double(gh_list_ref(partials,gh_int2scm(i)));
  mus_partials2wave(partial_data,len / 2,f->data,f->length,(normalize == SCM_BOOL_T));
  FREE(partial_data);
  return(table);
}

static SCM g_phasepartials2wave(SCM partials, SCM utable, SCM normalize)
{
  vct *f;
  SCM table;
  float *partial_data,*wave;
  int len,i;
  SCM_ASSERT(gh_list_p(partials),partials,SCM_ARG1,S_phasepartials2wave);
  if (utable == SCM_UNDEFINED)
    {
      wave = (float *)CALLOC(DEFAULT_TABLE_SIZE,sizeof(float));
      table = make_vct(DEFAULT_TABLE_SIZE,wave);
    }
  else table = utable;
  f = get_vct(table);
  len = gh_length(partials);
  partial_data = (float *)CALLOC(len,sizeof(float));
  for (i=0;i<len;i++) partial_data[i] = gh_scm2double(gh_list_ref(partials,gh_int2scm(i)));
  mus_phasepartials2wave(partial_data,len / 3,f->data,f->length,(normalize == SCM_BOOL_T));
  FREE(partial_data);
  return(table);
}

static SCM g_make_table_lookup (SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6)
{
  SCM new_tbl;
  mus_scm *gn;
  int vals,table_size = DEFAULT_TABLE_SIZE;
  SCM args[6],keys[3];
  int orig_arg[3] = {0,0,0};
  float freq = 440.0,phase = 0.0;
  float *table = NULL;
  SCM wave = SCM_UNDEFINED;
  vct *v;
  keys[0] = all_keys[C_frequency];
  keys[1] = all_keys[C_initial_phase];
  keys[2] = all_keys[C_wave];
  args[0] = arg1; args[1] = arg2; args[2] = arg3; args[3] = arg4; args[4] = arg5; args[5] = arg6;
  vals = decode_keywords(S_make_table_lookup,3,keys,6,args,orig_arg);
  if (vals > 0)
    {
      freq = fkeyarg(keys[0],S_make_table_lookup,orig_arg[0]+1,args[orig_arg[0]],freq);
      phase = fkeyarg(keys[1],S_make_table_lookup,orig_arg[1]+1,args[orig_arg[1]],phase);
      if (!(c_keyword_p(keys[2])))
	{
	  if (vct_p(keys[2]))
	    {
	      wave = keys[2];
	      v = get_vct(wave);
	      table = v->data;
	      table_size = v->length;
	    }
	  else scm_wrong_type_arg(S_make_table_lookup,orig_arg[2]+1,args[orig_arg[2]]);
	}
    }
  if (table == NULL) table = (float *)CALLOC(table_size,sizeof(float));
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->gen = mus_make_table_lookup(freq,phase,table,table_size);
  gn->vcts = (SCM *)CALLOC(1,sizeof(SCM));
  gn->nvcts = 1;
  SCM_NEWCELL(new_tbl);
  SCM_SETCAR(new_tbl,mus_scm_tag);
  SCM_SETCDR(new_tbl,(SCM)gn);
  if (wave == SCM_UNDEFINED) wave = make_vct(table_size,table);
  gn->vcts[0] = wave;
  return(new_tbl);
}

static SCM g_table_lookup (SCM obj, SCM fm) 
{
  float fm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_table_lookup_p(mus_get_any(obj)))),obj,SCM_ARG1,S_table_lookup);
  if (gh_number_p(fm)) fm1 = gh_scm2double(fm);  else if (fm != SCM_UNDEFINED) scm_wrong_type_arg(S_table_lookup,2,fm);
  return(gh_double2scm(mus_table_lookup(mus_get_any(obj),fm1)));
}

static void init_tbl(void)
{
  gh_new_procedure1_0(S_table_lookup_p,g_table_lookup_p);
  gh_new_procedure(S_make_table_lookup,g_make_table_lookup,0,6,0);
  gh_new_procedure(S_table_lookup,g_table_lookup,1,1,0);
  gh_new_procedure1_2(S_partials2wave,g_partials2wave);
  gh_new_procedure1_2(S_phasepartials2wave,g_phasepartials2wave);
}


/* ---------------- sawtooth et al ---------------- */

#define S_make_sawtooth_wave "make-sawtooth-wave"
#define S_sawtooth_wave "sawtooth-wave"
#define S_sawtooth_wave_p "sawtooth-wave?"
#define S_make_square_wave "make-square-wave"
#define S_square_wave "square-wave"
#define S_square_wave_p "square-wave?"
#define S_make_triangle_wave "make-triangle-wave"
#define S_triangle_wave "triangle-wave"
#define S_triangle_wave_p "triangle-wave?"
#define S_make_pulse_train "make-pulse-train"
#define S_pulse_train "pulse-train"
#define S_pulse_train_p "pulse-train?"

enum {G_SAWTOOTH_WAVE,G_SQUARE_WAVE,G_TRIANGLE_WAVE,G_PULSE_TRAIN};

static SCM g_make_sw(int type, float def_phase, SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6)
{
  SCM new_sw;
  mus_scm *gn;
  char *caller;
  SCM args[6],keys[3];
  int orig_arg[3] = {0,0,0};
  int vals;
  float freq = 440.0;
  float base = 1.0;
  float phase;
  phase = def_phase;
  switch (type)
    {
      case G_SAWTOOTH_WAVE: caller = S_make_sawtooth_wave; break;
      case G_SQUARE_WAVE: caller = S_make_square_wave; break;
      case G_TRIANGLE_WAVE: caller = S_make_triangle_wave; break;
      case G_PULSE_TRAIN: caller = S_make_pulse_train; break;
    }
  keys[0] = all_keys[C_frequency];
  keys[1] = all_keys[C_amplitude];
  keys[2] = all_keys[C_initial_phase];
  args[0] = arg1; args[1] = arg2; args[2] = arg3; args[3] = arg4; args[4] = arg5; args[5] = arg6; 
  vals = decode_keywords(caller,3,keys,6,args,orig_arg);
  if (vals > 0)
    {
      freq = fkeyarg(keys[0],caller,orig_arg[0]+1,args[orig_arg[0]],freq);
      base = fkeyarg(keys[1],caller,orig_arg[1]+1,args[orig_arg[1]],base);
      phase = fkeyarg(keys[2],caller,orig_arg[2]+1,args[orig_arg[2]],phase);
    }
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  switch (type)
    {
    case G_SAWTOOTH_WAVE: gn->gen = mus_make_sawtooth_wave(freq,base,phase); break;
    case G_SQUARE_WAVE: gn->gen = mus_make_square_wave(freq,base,phase); break;
    case G_TRIANGLE_WAVE: gn->gen = mus_make_triangle_wave(freq,base,phase); break;
    case G_PULSE_TRAIN: gn->gen = mus_make_pulse_train(freq,base,phase); break;
    }
  SCM_NEWCELL(new_sw);
  SCM_SETCAR(new_sw,mus_scm_tag);
  SCM_SETCDR(new_sw,(SCM)gn);
  return(new_sw);
}

static SCM g_make_sawtooth_wave(SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6) 
{
  return(g_make_sw(G_SAWTOOTH_WAVE,M_PI,arg1,arg2,arg3,arg4,arg5,arg6));
}

static SCM g_make_square_wave(SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6) 
{
  return(g_make_sw(G_SQUARE_WAVE,0.0,arg1,arg2,arg3,arg4,arg5,arg6));
}

static SCM g_make_triangle_wave(SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6) 
{
  return(g_make_sw(G_TRIANGLE_WAVE,0.0,arg1,arg2,arg3,arg4,arg5,arg6));
}

static SCM g_make_pulse_train(SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6) 
{
  return(g_make_sw(G_PULSE_TRAIN,2*M_PI,arg1,arg2,arg3,arg4,arg5,arg6));
}

static SCM g_sawtooth_wave(SCM obj, SCM fm) 
{
  float fm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_sawtooth_wave_p(mus_get_any(obj)))),obj,SCM_ARG1,S_sawtooth_wave);
  if (gh_number_p(fm)) fm1 = gh_scm2double(fm); else if (fm != SCM_UNDEFINED) scm_wrong_type_arg(S_sawtooth_wave,2,fm);
  return(gh_double2scm(mus_sawtooth_wave(mus_get_any(obj),fm1)));
}

static SCM g_square_wave(SCM obj, SCM fm) 
{
  float fm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_square_wave_p(mus_get_any(obj)))),obj,SCM_ARG1,S_square_wave);
  if (gh_number_p(fm)) fm1 = gh_scm2double(fm); else if (fm != SCM_UNDEFINED) scm_wrong_type_arg(S_square_wave,2,fm);
  return(gh_double2scm(mus_square_wave(mus_get_any(obj),fm1)));
}

static SCM g_triangle_wave(SCM obj, SCM fm) 
{
  float fm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_triangle_wave_p(mus_get_any(obj)))),obj,SCM_ARG1,S_triangle_wave);
  if (gh_number_p(fm)) fm1 = gh_scm2double(fm); else if (fm != SCM_UNDEFINED) scm_wrong_type_arg(S_triangle_wave,2,fm);
  return(gh_double2scm(mus_triangle_wave(mus_get_any(obj),fm1)));
}

static SCM g_pulse_train(SCM obj, SCM fm) 
{
  float fm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_pulse_train_p(mus_get_any(obj)))),obj,SCM_ARG1,S_pulse_train);
  if (gh_number_p(fm)) fm1 = gh_scm2double(fm); else if (fm != SCM_UNDEFINED) scm_wrong_type_arg(S_pulse_train,2,fm);
  return(gh_double2scm(mus_pulse_train(mus_get_any(obj),fm1)));
}

static SCM g_sawtooth_wave_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_sawtooth_wave_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}
static SCM g_square_wave_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_square_wave_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}
static SCM g_triangle_wave_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_triangle_wave_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}
static SCM g_pulse_train_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_pulse_train_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static void init_sw(void)
{
  gh_new_procedure(S_make_sawtooth_wave,g_make_sawtooth_wave,0,6,0);
  gh_new_procedure(S_sawtooth_wave,g_sawtooth_wave,1,1,0);
  gh_new_procedure(S_sawtooth_wave_p,g_sawtooth_wave_p,1,0,0);
  gh_new_procedure(S_make_triangle_wave,g_make_triangle_wave,0,6,0);
  gh_new_procedure(S_triangle_wave,g_triangle_wave,1,1,0);
  gh_new_procedure(S_triangle_wave_p,g_triangle_wave_p,1,0,0);
  gh_new_procedure(S_make_square_wave,g_make_square_wave,0,6,0);
  gh_new_procedure(S_square_wave,g_square_wave,1,1,0);
  gh_new_procedure(S_square_wave_p,g_square_wave_p,1,0,0);
  gh_new_procedure(S_make_pulse_train,g_make_pulse_train,0,6,0);
  gh_new_procedure(S_pulse_train,g_pulse_train,1,1,0);
  gh_new_procedure(S_pulse_train_p,g_pulse_train_p,1,0,0);
}


/* ---------------- asymmetric-fm ---------------- */

#define S_make_asymmetric_fm "make-asymmetric-fm"
#define S_asymmetric_fm "asymmetric-fm"
#define S_asymmetric_fm_p "asymmetric-fm?"

static SCM g_make_asymmetric_fm(SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6, SCM arg7, SCM arg8)
{
  SCM new_asyfm;
  mus_scm *gn;
  SCM args[8],keys[4];
  int orig_arg[4] = {0,0,0};
  int vals;
  float freq = 440.0;
  float phase = 0.0;
  float r = 1.0;
  float ratio = 1.0;
  keys[0] = all_keys[C_frequency];
  keys[1] = all_keys[C_initial_phase];
  keys[2] = all_keys[C_r];
  keys[3] = all_keys[C_ratio];
  args[0] = arg1; args[1] = arg2; args[2] = arg3; args[3] = arg4; args[4] = arg5; args[5] = arg6; args[6] = arg7; args[7] = arg8; 
  vals = decode_keywords(S_make_asymmetric_fm,4,keys,8,args,orig_arg);
  if (vals > 0)
    {
      freq = fkeyarg(keys[0],S_make_asymmetric_fm,orig_arg[0]+1,args[orig_arg[0]],freq);
      phase = fkeyarg(keys[1],S_make_asymmetric_fm,orig_arg[1]+1,args[orig_arg[1]],phase);
      r = fkeyarg(keys[2],S_make_asymmetric_fm,orig_arg[2]+1,args[orig_arg[2]],r);
      ratio = fkeyarg(keys[3],S_make_asymmetric_fm,orig_arg[3]+1,args[orig_arg[3]],ratio);
    }
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->gen = mus_make_asymmetric_fm(freq,phase,r,ratio);
  SCM_NEWCELL(new_asyfm);
  SCM_SETCAR(new_asyfm,mus_scm_tag);
  SCM_SETCDR(new_asyfm,(SCM)gn);
  return(new_asyfm);
}

static SCM g_asymmetric_fm(SCM obj, SCM index, SCM fm)
{
  float fm1 = 0.0, index1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_asymmetric_fm_p(mus_get_any(obj)))),obj,SCM_ARG1,S_asymmetric_fm);
  if (gh_number_p(fm)) fm1 = gh_scm2double(fm); else if (fm != SCM_UNDEFINED) scm_wrong_type_arg(S_asymmetric_fm,3,fm);
  if (gh_number_p(index)) index1 = gh_scm2double(index); else if (index != SCM_UNDEFINED) scm_wrong_type_arg(S_asymmetric_fm,2,index);
  return(gh_double2scm(mus_asymmetric_fm(mus_get_any(obj),index1,fm1)));
}

static SCM g_asymmetric_fm_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_asymmetric_fm_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static void init_asyfm(void)
{
  gh_new_procedure(S_make_asymmetric_fm,g_make_asymmetric_fm,0,8,0);
  gh_new_procedure(S_asymmetric_fm,g_asymmetric_fm,1,2,0);
  gh_new_procedure(S_asymmetric_fm_p,g_asymmetric_fm_p,1,0,0);
}



/* ---------------- simple filters ---------------- */

#define S_make_one_zero "make-one-zero"
#define S_one_zero      "one-zero"
#define S_one_zero_p    "one-zero?"
#define S_make_one_pole "make-one-pole"
#define S_one_pole      "one-pole"
#define S_one_pole_p    "one-pole?"
#define S_make_two_zero "make-two-zero"
#define S_two_zero      "two-zero"
#define S_two_zero_p    "two-zero?"
#define S_make_two_pole "make-two-pole"
#define S_two_pole      "two-pole"
#define S_two_pole_p    "two-pole?"
#define S_make_zpolar   "make-zpolar"
#define S_make_ppolar   "make-ppolar"
#define S_a0            "mus-a0"
#define S_a1            "mus-a1"
#define S_a2            "mus-a2"
#define S_b1            "mus-b1"
#define S_b2            "mus-b2"
#define S_set_a0        "mus-set-a0"
#define S_set_a1        "mus-set-a1"
#define S_set_a2        "mus-set-a2"
#define S_set_b1        "mus-set-b1"
#define S_set_b2        "mus-set-b2"

enum {G_ONE_POLE,G_ONE_ZERO,G_TWO_POLE,G_TWO_ZERO,G_ZPOLAR,G_PPOLAR};
static char *smpflts[6] = {S_make_one_pole,S_make_one_zero,S_make_two_pole,S_make_two_zero,S_make_zpolar,S_make_ppolar};

static SCM g_make_smpflt_1(int choice, SCM arg1, SCM arg2, SCM arg3, SCM arg4)
{
  SCM new_smpflt;
  mus_scm *gn;
  SCM args[4],keys[2];
  int orig_arg[2] = {0,0};
  int vals;
  float a0 = 0.0;
  float a1 = 0.0;
  switch (choice)
    {
    case G_ONE_ZERO: keys[0] = all_keys[C_a0]; keys[1] = all_keys[C_a1]; break;
    case G_ONE_POLE: keys[0] = all_keys[C_a0]; keys[1] = all_keys[C_b1]; break;
    default: keys[0] = all_keys[C_r]; keys[1] = all_keys[C_frequency]; break;
    }
  args[0] = arg1; args[1] = arg2; args[2] = arg3; args[3] = arg4;
  vals = decode_keywords(smpflts[choice],2,keys,4,args,orig_arg);
  if (vals > 0)
    {
      a0 = fkeyarg(keys[0],smpflts[choice],orig_arg[0]+1,args[orig_arg[0]],a0);
      a1 = fkeyarg(keys[1],smpflts[choice],orig_arg[1]+1,args[orig_arg[1]],a1);
    }
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  switch (choice)
    {
    case G_ONE_ZERO: gn->gen = mus_make_one_zero(a0,a1); break;
    case G_ONE_POLE: gn->gen = mus_make_one_pole(a0,a1); break;
    case G_ZPOLAR: gn->gen = mus_make_zpolar(a0,a1); break;
    case G_PPOLAR: gn->gen = mus_make_ppolar(a0,a1); break;
    }
  SCM_NEWCELL(new_smpflt);
  SCM_SETCAR(new_smpflt,mus_scm_tag);
  SCM_SETCDR(new_smpflt,(SCM)gn);
  return(new_smpflt);
}

static SCM g_make_one_zero(SCM arg1, SCM arg2, SCM arg3, SCM arg4) {return(g_make_smpflt_1(G_ONE_ZERO,arg1,arg2,arg3,arg4));}
static SCM g_make_one_pole(SCM arg1, SCM arg2, SCM arg3, SCM arg4) {return(g_make_smpflt_1(G_ONE_POLE,arg1,arg2,arg3,arg4));}
static SCM g_make_zpolar(SCM arg1, SCM arg2, SCM arg3, SCM arg4) {return(g_make_smpflt_1(G_ZPOLAR,arg1,arg2,arg3,arg4));}
static SCM g_make_ppolar(SCM arg1, SCM arg2, SCM arg3, SCM arg4) {return(g_make_smpflt_1(G_PPOLAR,arg1,arg2,arg3,arg4));}

static SCM g_make_smpflt_2(int choice, SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6)
{
  SCM new_smpflt;
  mus_scm *gn;
  SCM args[6],keys[3];
  int orig_arg[3] = {0,0,0};
  int vals;
  float a0 = 0.0;
  float a1 = 0.0;
  float a2 = 0.0;
  if (choice == G_TWO_ZERO)
    {
      keys[0] = all_keys[C_a0];
      keys[1] = all_keys[C_a1];
      keys[2] = all_keys[C_a2];
    }
  else
    {
      keys[0] = all_keys[C_a0];
      keys[1] = all_keys[C_b1];
      keys[2] = all_keys[C_b2];
    }
  args[0] = arg1; args[1] = arg2; args[2] = arg3; args[3] = arg4; args[4] = arg5; args[5] = arg6;
  vals = decode_keywords(smpflts[choice],3,keys,6,args,orig_arg);
  if (vals > 0)
    {
      a0 = fkeyarg(keys[0],smpflts[choice],orig_arg[0]+1,args[orig_arg[0]],a0);
      a1 = fkeyarg(keys[1],smpflts[choice],orig_arg[1]+1,args[orig_arg[1]],a1);
      a2 = fkeyarg(keys[2],smpflts[choice],orig_arg[2]+1,args[orig_arg[2]],a2);
    }
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  if (choice == G_TWO_ZERO)
    gn->gen = mus_make_two_zero(a0,a1,a2);
  else gn->gen = mus_make_two_pole(a0,a1,a2);
  SCM_NEWCELL(new_smpflt);
  SCM_SETCAR(new_smpflt,mus_scm_tag);
  SCM_SETCDR(new_smpflt,(SCM)gn);
  return(new_smpflt);
}

static SCM g_make_two_zero(SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6) 
{
  return(g_make_smpflt_2(G_TWO_ZERO,arg1,arg2,arg3,arg4,arg5,arg6));
}

static SCM g_make_two_pole(SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6) 
{
  return(g_make_smpflt_2(G_TWO_POLE,arg1,arg2,arg3,arg4,arg5,arg6));
}

static SCM g_one_zero(SCM obj, SCM fm)
{
  float fm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_one_zero_p(mus_get_any(obj)))),obj,SCM_ARG1,S_one_zero);
  if (gh_number_p(fm)) fm1 = gh_scm2double(fm); else if (fm != SCM_UNDEFINED) scm_wrong_type_arg(S_one_zero,2,fm);
  return(gh_double2scm(mus_one_zero(mus_get_any(obj),fm1)));
}

static SCM g_one_pole(SCM obj, SCM fm)
{
  float fm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_one_pole_p(mus_get_any(obj)))),obj,SCM_ARG1,S_one_pole);
  if (gh_number_p(fm)) fm1 = gh_scm2double(fm); else if (fm != SCM_UNDEFINED) scm_wrong_type_arg(S_one_pole,2,fm);
  return(gh_double2scm(mus_one_pole(mus_get_any(obj),fm1)));
}

static SCM g_two_zero(SCM obj, SCM fm)
{
  float fm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_two_zero_p(mus_get_any(obj)))),obj,SCM_ARG1,S_two_zero);
  if (gh_number_p(fm)) fm1 = gh_scm2double(fm); else if (fm != SCM_UNDEFINED) scm_wrong_type_arg(S_two_zero,2,fm);
  return(gh_double2scm(mus_two_zero(mus_get_any(obj),fm1)));
}

static SCM g_two_pole(SCM obj, SCM fm)
{
  float fm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_two_pole_p(mus_get_any(obj)))),obj,SCM_ARG1,S_two_pole);
  if (gh_number_p(fm)) fm1 = gh_scm2double(fm); else if (fm != SCM_UNDEFINED) scm_wrong_type_arg(S_two_pole,2,fm);
  return(gh_double2scm(mus_two_pole(mus_get_any(obj),fm1)));
}

static SCM g_one_zero_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_one_zero_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}
static SCM g_one_pole_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_one_pole_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}
static SCM g_two_zero_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_two_zero_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}
static SCM g_two_pole_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_two_pole_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_a0(SCM obj) {return(gh_double2scm(mus_a0(mus_get_any(obj))));}
static SCM g_a1(SCM obj) {return(gh_double2scm(mus_a1(mus_get_any(obj))));}
static SCM g_a2(SCM obj) {return(gh_double2scm(mus_a2(mus_get_any(obj))));}
static SCM g_b1(SCM obj) {return(gh_double2scm(mus_b1(mus_get_any(obj))));}
static SCM g_b2(SCM obj) {return(gh_double2scm(mus_b2(mus_get_any(obj))));}
static SCM g_set_a0(SCM obj, SCM val) {return(gh_double2scm(mus_set_a0(mus_get_any(obj),gh_scm2double(val))));}
static SCM g_set_a1(SCM obj, SCM val) {return(gh_double2scm(mus_set_a1(mus_get_any(obj),gh_scm2double(val))));}
static SCM g_set_a2(SCM obj, SCM val) {return(gh_double2scm(mus_set_a2(mus_get_any(obj),gh_scm2double(val))));}
static SCM g_set_b1(SCM obj, SCM val) {return(gh_double2scm(mus_set_b1(mus_get_any(obj),gh_scm2double(val))));}
static SCM g_set_b2(SCM obj, SCM val) {return(gh_double2scm(mus_set_b2(mus_get_any(obj),gh_scm2double(val))));}

static void init_smpflt(void)
{
  gh_new_procedure(S_make_one_zero,g_make_one_zero,0,4,0);
  gh_new_procedure(S_one_zero,g_one_zero,1,1,0);
  gh_new_procedure(S_one_zero_p,g_one_zero_p,1,0,0);
  gh_new_procedure(S_make_one_pole,g_make_one_pole,0,4,0);
  gh_new_procedure(S_one_pole,g_one_pole,1,1,0);
  gh_new_procedure(S_one_pole_p,g_one_pole_p,1,0,0);
  gh_new_procedure(S_make_two_zero,g_make_two_zero,0,6,0);
  gh_new_procedure(S_two_zero,g_two_zero,1,1,0);
  gh_new_procedure(S_two_zero_p,g_two_zero_p,1,0,0);
  gh_new_procedure(S_make_two_pole,g_make_two_pole,0,6,0);
  gh_new_procedure(S_two_pole,g_two_pole,1,1,0);
  gh_new_procedure(S_two_pole_p,g_two_pole_p,1,0,0);
  gh_new_procedure(S_make_zpolar,g_make_zpolar,0,4,0);
  gh_new_procedure(S_make_ppolar,g_make_ppolar,0,4,0);
  gh_new_procedure(S_a0,g_a0,1,0,0);
  gh_new_procedure(S_a1,g_a1,1,0,0);
  gh_new_procedure(S_a2,g_a2,1,0,0);
  gh_new_procedure(S_b1,g_b1,1,0,0);
  gh_new_procedure(S_b2,g_b2,1,0,0);
  gh_new_procedure(S_set_a0,g_set_a0,2,0,0);
  gh_new_procedure(S_set_a1,g_set_a1,2,0,0);
  gh_new_procedure(S_set_a2,g_set_a2,2,0,0);
  gh_new_procedure(S_set_b1,g_set_b1,2,0,0);
  gh_new_procedure(S_set_b2,g_set_b2,2,0,0);
}



/* ---------------- formant ---------------- */

#define S_make_formant "make-formant"
#define S_formant "formant"
#define S_formant_p "formant?"
#define S_formant_radius "mus-formant-radius"
#define S_set_formant_radius "mus-set-formant-radius"

static SCM g_make_formant(SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6)
{
  SCM new_osc;
  mus_scm *gn;
  int vals;
  SCM args[6],keys[3];
  int orig_arg[3] = {0,0,0};
  float freq = 0.0,radius = 0.0,gain=1.0;
  keys[0] = all_keys[C_radius];
  keys[1] = all_keys[C_frequency];
  keys[2] = all_keys[C_gain];
  args[0] = arg1; args[1] = arg2; args[2] = arg3; args[3] = arg4; args[4] = arg5; args[5] = arg6;
  vals = decode_keywords(S_make_formant,3,keys,6,args,orig_arg);
  if (vals > 0)
    {
      radius = fkeyarg(keys[0],S_make_formant,orig_arg[0]+1,args[orig_arg[0]],radius);
      freq = fkeyarg(keys[1],S_make_formant,orig_arg[1]+1,args[orig_arg[1]],freq);
      gain = fkeyarg(keys[2],S_make_formant,orig_arg[2]+1,args[orig_arg[2]],gain);
    }
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->gen = mus_make_formant(radius,freq,gain);
  gn->nvcts = 0;
  SCM_NEWCELL(new_osc);
  SCM_SETCAR(new_osc,mus_scm_tag);
  SCM_SETCDR(new_osc,(SCM)gn);
  return(new_osc);
}

static SCM g_formant(SCM gen, SCM input)
{
  float in1 = 0.0;
  SCM_ASSERT(((mus_scm_p(gen)) && (mus_formant_p(mus_get_any(gen)))),gen,SCM_ARG1,S_formant);
  if (gh_number_p(input)) in1 = gh_scm2double(input); else if (input != SCM_UNDEFINED) scm_wrong_type_arg(S_formant,2,input);
  return(gh_double2scm(mus_formant(mus_get_any(gen),in1)));
}

static SCM g_formant_p(SCM os) {return(((mus_scm_p(os)) && (mus_formant_p(mus_get_any(os)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_formant_radius (SCM gen)
{
  SCM_ASSERT(((mus_scm_p(gen)) && (mus_formant_p(mus_get_any(gen)))),gen,SCM_ARG1,S_formant_radius);
  return(gh_double2scm(mus_formant_radius(mus_get_any(gen))));
}

static SCM g_set_formant_radius (SCM gen, SCM val)
{
  SCM_ASSERT(((mus_scm_p(gen)) && (mus_formant_p(mus_get_any(gen)))),gen,SCM_ARG1,S_formant_radius);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG2,S_set_formant_radius);
  return(gh_double2scm(mus_set_formant_radius(mus_get_any(gen),gh_scm2double(val))));
}

static void init_formant(void)
{
  gh_new_procedure1_0(S_formant_p,g_formant_p);
  gh_new_procedure(S_make_formant,g_make_formant,0,6,0);
  gh_new_procedure1_1(S_formant,g_formant);
  gh_new_procedure1_0(S_formant_radius,g_formant_radius);
  gh_new_procedure2_0(S_set_formant_radius,g_set_formant_radius);
}



/* ---------------- frame ---------------- */

#define S_make_frame       "make-frame"
#define S_frame_p          "frame?"
#define S_frame_add        "frame+"
#define S_frame_multiply   "frame*"
#define S_frame_ref        "frame-ref"
#define S_set_frame_ref    "frame-set!"

static SCM g_make_frame(SCM arglist)
{
  /* make_empty_frame from first of arglist, then if more args, load vals */
  mus_scm *gn;
  mus_frame *fr;
  SCM new_frame,cararg;
  int size = 0,i,len;
  SCM_ASSERT((gh_list_p(arglist)),arglist,SCM_ARG1,S_make_frame);
  len = gh_length(arglist);
  if (len == 0) scm_misc_error(S_make_frame,"no arguments?",SCM_EOL);
  cararg = gh_list_ref(arglist,gh_int2scm(0));
  if (!(gh_number_p(cararg))) scm_wrong_type_arg(S_make_frame,1,cararg);
  size = gh_scm2int(cararg);
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->gen = (mus_any *)mus_make_empty_frame(size);
  if (len > 1)
    {
      fr = (mus_frame *)(gn->gen);
      for (i=1;i<len;i++)
	fr->vals[i-1] = gh_scm2double(gh_list_ref(arglist,gh_int2scm(i)));
    }
  SCM_NEWCELL(new_frame);
  SCM_SETCAR(new_frame,mus_scm_tag);
  SCM_SETCDR(new_frame,(SCM)gn);
  return(new_frame);
}

static SCM g_frame_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_frame_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

#define DONT_FREE_FRAME -1
#define FREE_FRAME 1

static SCM g_wrap_frame(mus_frame *val, int dealloc)
{
  mus_scm *gn;
  SCM new_frame;
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->gen = (mus_any *)val;
  gn->nvcts = dealloc;
  SCM_NEWCELL(new_frame);
  SCM_SETCAR(new_frame,mus_scm_tag);
  SCM_SETCDR(new_frame,(SCM)gn);
  return(new_frame);
}

static SCM g_frame_add(SCM uf1, SCM uf2, SCM ures) /* optional res */
{
  mus_frame *res = NULL;
  SCM_ASSERT(((mus_scm_p(uf1)) && (mus_frame_p(mus_get_any(uf1)))),uf1,SCM_ARG1,S_frame_add);
  SCM_ASSERT(((mus_scm_p(uf2)) && (mus_frame_p(mus_get_any(uf2)))),uf2,SCM_ARG2,S_frame_add);
  if ((mus_scm_p(ures)) && (mus_frame_p(mus_get_any(ures)))) res = (mus_frame *)mus_get_any(ures);
  return(g_wrap_frame(mus_frame_add((mus_frame *)mus_get_any(uf1),(mus_frame *)mus_get_any(uf2),res),(res) ? DONT_FREE_FRAME : FREE_FRAME));
}

static SCM g_frame_multiply(SCM uf1, SCM uf2, SCM ures) /* optional res */
{
  mus_frame *res = NULL;
  SCM_ASSERT(((mus_scm_p(uf1)) && (mus_frame_p(mus_get_any(uf1)))),uf1,SCM_ARG1,S_frame_multiply);
  SCM_ASSERT(((mus_scm_p(uf2)) && (mus_frame_p(mus_get_any(uf2)))),uf2,SCM_ARG2,S_frame_multiply);
  if ((mus_scm_p(ures)) && (mus_frame_p(mus_get_any(ures)))) res = (mus_frame *)mus_get_any(ures);
  return(g_wrap_frame(mus_frame_multiply((mus_frame *)mus_get_any(uf1),(mus_frame *)mus_get_any(uf2),res),(res) ? DONT_FREE_FRAME : FREE_FRAME));
}

static SCM g_frame_ref(SCM uf1, SCM uchan)
{
  SCM_ASSERT(((mus_scm_p(uf1)) && (mus_frame_p(mus_get_any(uf1)))),uf1,SCM_ARG1,S_frame_ref);
  SCM_ASSERT((gh_number_p(uchan)),uchan,SCM_ARG2,S_frame_ref);
  return(gh_double2scm(mus_frame_ref((mus_frame *)mus_get_any(uf1),gh_scm2int(uchan))));
}

static SCM g_set_frame_ref(SCM uf1, SCM uchan, SCM val)
{
  SCM_ASSERT(((mus_scm_p(uf1)) && (mus_frame_p(mus_get_any(uf1)))),uf1,SCM_ARG1,S_set_frame_ref);
  SCM_ASSERT((gh_number_p(uchan)),uchan,SCM_ARG2,S_set_frame_ref);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG3,S_set_frame_ref);
  return(gh_double2scm(mus_frame_set((mus_frame *)mus_get_any(uf1),gh_scm2int(uchan),gh_scm2double(val))));
}

static void init_frame(void)
{
  gh_new_procedure(S_make_frame,g_make_frame,0,0,1);
  gh_new_procedure(S_frame_p,g_frame_p,1,0,0);
  gh_new_procedure(S_frame_add,g_frame_add,2,1,0);
  gh_new_procedure(S_frame_multiply,g_frame_multiply,2,1,0);
  gh_new_procedure(S_frame_ref,g_frame_ref,2,0,0);
  gh_new_procedure(S_set_frame_ref,g_set_frame_ref,3,0,0);
}



/* ---------------- mixer ---------------- */

#define S_make_mixer       "make-mixer"
#define S_mixer_p          "mixer?"
#define S_mixer_multiply   "mixer*"
#define S_mixer_ref        "mixer-ref"
#define S_set_mixer_ref    "mixer-set!"

#define S_frame2sample     "frame->sample"
#define S_sample2frame     "sample->frame"
#define S_frame2frame      "frame->frame"

static SCM g_mixer_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_mixer_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_mixer_ref(SCM uf1, SCM in, SCM out)
{
  SCM_ASSERT(((mus_scm_p(uf1)) && (mus_mixer_p(mus_get_any(uf1)))),uf1,SCM_ARG1,S_mixer_ref);
  SCM_ASSERT((gh_number_p(in)),in,SCM_ARG2,S_mixer_ref);
  SCM_ASSERT((gh_number_p(out)),out,SCM_ARG3,S_mixer_ref);
  return(gh_double2scm(mus_mixer_ref((mus_mixer *)mus_get_any(uf1),gh_scm2int(in),gh_scm2int(out))));
}

static SCM g_set_mixer_ref(SCM uf1, SCM in, SCM out, SCM val)
{
  SCM_ASSERT(((mus_scm_p(uf1)) && (mus_mixer_p(mus_get_any(uf1)))),uf1,SCM_ARG1,S_set_mixer_ref);
  SCM_ASSERT((gh_number_p(in)),in,SCM_ARG2,S_set_mixer_ref);
  SCM_ASSERT((gh_number_p(out)),out,SCM_ARG2,S_set_mixer_ref);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG4,S_set_mixer_ref);
  return(gh_double2scm(mus_mixer_set((mus_mixer *)mus_get_any(uf1),gh_scm2int(in),gh_scm2int(out),gh_scm2double(val))));
}

#define DONT_FREE_MIXER -1
#define FREE_MIXER 1

static SCM g_wrap_mixer(mus_mixer *val, int dealloc)
{
  mus_scm *gn;
  SCM new_mixer;
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->gen = (mus_any *)val;
  gn->nvcts = dealloc;
  SCM_NEWCELL(new_mixer);
  SCM_SETCAR(new_mixer,mus_scm_tag);
  SCM_SETCDR(new_mixer,(SCM)gn);
  return(new_mixer);
}

static SCM g_mixer_multiply(SCM uf1, SCM uf2, SCM ures) /* optional res */
{
  mus_mixer *res = NULL;
  SCM_ASSERT(((mus_scm_p(uf1)) && (mus_mixer_p(mus_get_any(uf1)))),uf1,SCM_ARG1,S_mixer_multiply);
  SCM_ASSERT(((mus_scm_p(uf2)) && (mus_mixer_p(mus_get_any(uf2)))),uf2,SCM_ARG2,S_mixer_multiply);
  if ((mus_scm_p(ures)) && (mus_mixer_p(mus_get_any(ures)))) res = (mus_mixer *)mus_get_any(ures);
  return(g_wrap_mixer(mus_mixer_multiply((mus_mixer *)mus_get_any(uf1),(mus_mixer *)mus_get_any(uf2),res),(res) ? DONT_FREE_MIXER : FREE_MIXER));
}

static SCM g_frame2frame(SCM mx, SCM infr, SCM outfr) /* optional outfr */
{
  mus_frame *res = NULL;
  SCM_ASSERT(((mus_scm_p(mx)) && (mus_mixer_p(mus_get_any(mx)))),mx,SCM_ARG1,S_frame2frame);
  SCM_ASSERT(((mus_scm_p(infr)) && (mus_frame_p(mus_get_any(infr)))),infr,SCM_ARG2,S_frame2frame);
  if ((mus_scm_p(outfr)) && (mus_frame_p(mus_get_any(outfr)))) res = (mus_frame *)mus_get_any(outfr);
  return(g_wrap_frame(mus_frame2frame((mus_mixer *)mus_get_any(mx),(mus_frame *)mus_get_any(infr),res),(res) ? DONT_FREE_FRAME : FREE_FRAME));
}

static SCM g_frame2sample(SCM mx, SCM fr)
{
  SCM_ASSERT((mus_scm_p(mx)),mx,SCM_ARG1,S_frame2sample);
  SCM_ASSERT(((mus_scm_p(fr)) && (mus_frame_p(mus_get_any(fr)))),fr,SCM_ARG2,S_frame2sample);
  return(gh_double2scm(mus_frame2sample(mus_get_any(mx),(mus_frame *)mus_get_any(fr))));
}

static SCM g_sample2frame(SCM mx, SCM insp, SCM outfr) /* optional outfr */
{
  mus_frame *res = NULL;
  SCM_ASSERT((mus_scm_p(mx)),mx,SCM_ARG1,S_sample2frame);
  SCM_ASSERT((gh_number_p(insp)),insp,SCM_ARG2,S_sample2frame);
  if ((mus_scm_p(outfr)) && (mus_frame_p(mus_get_any(outfr)))) res = (mus_frame *)mus_get_any(outfr);
  return(g_wrap_frame(mus_sample2frame(mus_get_any(mx),gh_scm2double(insp),res),(res) ? DONT_FREE_FRAME : FREE_FRAME));
}

static SCM g_make_mixer(SCM arglist)
{
  /* make_empty_mixer from first of arglist, then if more args, load vals */
  mus_scm *gn;
  mus_mixer *fr;
  SCM new_mixer,cararg;
  int size = 0,i,j,k,len;
  SCM_ASSERT((gh_list_p(arglist)),arglist,SCM_ARG1,S_make_mixer);
  len = gh_length(arglist);
  if (len == 0) scm_misc_error(S_make_mixer,"no arguments?",SCM_EOL);
  cararg = gh_list_ref(arglist,gh_int2scm(0));
  if (!(gh_number_p(cararg))) scm_wrong_type_arg(S_make_mixer,1,cararg);
  size = gh_scm2int(cararg);
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->gen = (mus_any *)mus_make_empty_mixer(size);
  if (len > 1)
    {
      fr = (mus_mixer *)(gn->gen);
      j = 0;
      k = 0;
      for (i=1;i<len;i++)
	{
	  fr->vals[j][k] = gh_scm2double(gh_list_ref(arglist,gh_int2scm(i)));
	  k++;
	  if (k == size)
	    {
	      k = 0;
	      j++;
	    }
	}
    }
  SCM_NEWCELL(new_mixer);
  SCM_SETCAR(new_mixer,mus_scm_tag);
  SCM_SETCDR(new_mixer,(SCM)gn);
  return(new_mixer);
}

static void init_mixer(void)
{
  gh_new_procedure(S_make_mixer,g_make_mixer,0,0,1);
  gh_new_procedure(S_mixer_p,g_mixer_p,1,0,0);
  gh_new_procedure(S_mixer_multiply,g_mixer_multiply,2,1,0);
  gh_new_procedure(S_mixer_ref,g_mixer_ref,3,0,0);
  gh_new_procedure(S_set_mixer_ref,g_set_mixer_ref,4,0,0);
  gh_new_procedure(S_frame2sample,g_frame2sample,2,0,0);
  gh_new_procedure(S_frame2frame,g_frame2frame,2,1,0);
  gh_new_procedure(S_sample2frame,g_sample2frame,2,1,0);
}


/* ---------------- buffer ---------------- */

#define S_make_buffer    "make-buffer"
#define S_buffer_p       "buffer?"
#define S_buffer2sample  "buffer->sample"
#define S_sample2buffer  "sample->buffer"
#define S_buffer2frame   "buffer->frame"
#define S_frame2buffer   "frame->buffer"
#define S_buffer_empty_p "buffer-empty?"

static SCM g_buffer_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_buffer_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_make_buffer(SCM arg1, SCM arg2, SCM arg3, SCM arg4)
{
  SCM new_rblk;
  mus_scm *gn;
  SCM args[4],keys[2];
  int orig_arg[2] = {0,0};
  int vals;
  float *buf;
  int siz = DEFAULT_TABLE_SIZE;
  float filltime = 0.0;
  keys[0] = all_keys[C_size];
  keys[1] = all_keys[C_fill_time];
  args[0] = arg1; args[1] = arg2; args[2] = arg3; args[3] = arg4;
  vals = decode_keywords(S_make_buffer,3,keys,6,args,orig_arg);
  if (vals > 0)
    {
      siz = ikeyarg(keys[0],S_make_buffer,orig_arg[0]+1,args[orig_arg[0]],siz);
      filltime = fkeyarg(keys[1],S_make_buffer,orig_arg[1]+1,args[orig_arg[1]],0.0);
    }
  buf = (float *)CALLOC(siz,sizeof(float));
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->vcts = (SCM *)CALLOC(1,sizeof(SCM));
  gn->nvcts = 1;
  gn->gen = mus_make_buffer(buf,siz,filltime);
  SCM_NEWCELL(new_rblk);
  SCM_SETCAR(new_rblk,mus_scm_tag);
  SCM_SETCDR(new_rblk,(SCM)gn);
  gn->vcts[0] = make_vct(siz,buf);
  return(new_rblk);
}

static SCM g_buffer2sample(SCM obj)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_buffer_p(mus_get_any(obj)))),obj,SCM_ARG1,S_buffer2sample);
  return(gh_double2scm(mus_buffer2sample(mus_get_any(obj))));
}

static SCM g_buffer2frame(SCM obj, SCM fr)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_buffer_p(mus_get_any(obj)))),obj,SCM_ARG1,S_buffer2frame);
  SCM_ASSERT(((mus_scm_p(fr)) && (mus_frame_p(mus_get_any(fr)))),fr,SCM_ARG2,S_buffer2frame);
  return(g_wrap_frame((mus_frame *)mus_buffer2frame(mus_get_any(obj),mus_get_any(fr)),DONT_FREE_FRAME));
}

static SCM g_buffer_empty_p(SCM obj)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_buffer_p(mus_get_any(obj)))),obj,SCM_ARG1,S_buffer_empty_p);
  return(gh_int2scm(mus_buffer_empty_p(mus_get_any(obj))));
}

static SCM g_sample2buffer(SCM obj, SCM val)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_buffer_p(mus_get_any(obj)))),obj,SCM_ARG1,S_sample2buffer);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG1,S_sample2buffer);
  return(gh_double2scm(mus_sample2buffer(mus_get_any(obj),gh_scm2double(val))));
}

static SCM g_frame2buffer(SCM obj, SCM val)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_buffer_p(mus_get_any(obj)))),obj,SCM_ARG1,S_sample2buffer);
  SCM_ASSERT(((mus_scm_p(val)) && (mus_frame_p(mus_get_any(val)))),val,SCM_ARG2,S_sample2buffer);
  return(g_wrap_frame((mus_frame *)mus_frame2buffer(mus_get_any(val),mus_get_any(obj)),DONT_FREE_FRAME));
}

static void init_rblk(void)
{
  gh_new_procedure(S_make_buffer,g_make_buffer,0,4,0);
  gh_new_procedure(S_buffer_p,g_buffer_p,1,0,0);
  gh_new_procedure(S_buffer_empty_p,g_buffer_empty_p,1,0,0);
  gh_new_procedure(S_buffer2sample,g_buffer2sample,1,0,0);
  gh_new_procedure(S_buffer2frame,g_buffer2frame,1,0,0);
  gh_new_procedure(S_sample2buffer,g_sample2buffer,2,0,0);
  gh_new_procedure(S_frame2buffer,g_frame2buffer,2,0,0);
}


/* ---------------- wave-train ---------------- */

#define S_make_wave_train "make-wave-train"
#define S_wave_train "wave-train"
#define S_wave_train_p "wave-train?"

static SCM g_make_wave_train(SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6)
{
  SCM new_wt,gwave = SCM_UNDEFINED;
  mus_scm *gn;
  SCM args[6],keys[3];
  int orig_arg[3] = {0,0,0};
  int vals,wsize;
  vct *v;
  float freq = 440.0;
  float phase = 0.0;
  float *wave = NULL;
  wsize = DEFAULT_TABLE_SIZE;
  keys[0] = all_keys[C_frequency];
  keys[1] = all_keys[C_initial_phase];
  keys[2] = all_keys[C_wave];
  args[0] = arg1; args[1] = arg2; args[2] = arg3; args[3] = arg4; args[4] = arg5; args[5] = arg6; 
  vals = decode_keywords(S_make_wave_train,3,keys,6,args,orig_arg);
  if (vals > 0)
    {
      freq = fkeyarg(keys[0],S_make_wave_train,orig_arg[0]+1,args[orig_arg[0]],freq);
      phase = fkeyarg(keys[1],S_make_wave_train,orig_arg[1]+1,args[orig_arg[1]],phase);
      if (!(c_keyword_p(keys[2])))
        {
	  if (vct_p(keys[2]))
	    {
	      gwave = keys[2];
	      v = get_vct(gwave);
	      wave = v->data;
	      wsize = v->length;
	    }
          else scm_wrong_type_arg(S_make_wave_train,orig_arg[2]+1,args[orig_arg[2]]);
        }
    }
  if (wave == NULL) wave = (float *)CALLOC(wsize,sizeof(float));
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->gen = mus_make_wave_train(freq,phase,wave,wsize);
  gn->vcts = (SCM *)CALLOC(1,sizeof(SCM));
  gn->nvcts = 1;
  SCM_NEWCELL(new_wt);
  SCM_SETCAR(new_wt,mus_scm_tag);
  SCM_SETCDR(new_wt,(SCM)gn);
  if (gwave == SCM_UNDEFINED) gwave = make_vct(wsize,wave);
  gn->vcts[0] = gwave;
  return(new_wt);
}

static SCM g_wave_train(SCM obj, SCM fm)
{
  float fm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_wave_train_p(mus_get_any(obj)))),obj,SCM_ARG1,S_wave_train);
  if (gh_number_p(fm)) fm1 = gh_scm2double(fm); else if (fm != SCM_UNDEFINED) scm_wrong_type_arg(S_wave_train,2,fm);
  return(gh_double2scm(mus_wave_train(mus_get_any(obj),fm1)));
}

static SCM g_wave_train_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_wave_train_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static void init_wt(void)
{
  gh_new_procedure(S_make_wave_train,g_make_wave_train,0,6,0);
  gh_new_procedure(S_wave_train,g_wave_train,1,1,0);
  gh_new_procedure(S_wave_train_p,g_wave_train_p,1,0,0);
}


/* ---------------- waveshape ---------------- */

#define S_make_waveshape           "make-waveshape"
#define S_waveshape                "waveshape"
#define S_waveshape_p              "waveshape?"
#define S_partials2waveshape       "partials->waveshape"
#define S_partials2polynomial      "partials->polynomial"
#define S_phasepartials2waveshape  "phase-partials->waveshape"

static float *list2partials(SCM harms, int *npartials)
{
  int listlen,i,maxpartial,curpartial;
  float *partials;
  listlen = gh_length(harms);
  /* the list is '(partial-number partial-amp ... ) */
  maxpartial = gh_scm2int(gh_list_ref(harms,gh_int2scm(0)));
  for (i=2;i<listlen;i+=2)
    {
      curpartial = gh_scm2int(gh_list_ref(harms,gh_int2scm(i)));
      if (curpartial > maxpartial) maxpartial = curpartial;
    }
  partials = (float *)CALLOC(maxpartial+1,sizeof(float));
  (*npartials) = maxpartial+1;
  for (i=0;i<listlen;i+=2)
    {
      curpartial = gh_scm2int(gh_list_ref(harms,gh_int2scm(i)));
      partials[curpartial] = gh_scm2double(gh_list_ref(harms,gh_int2scm(i+1)));
    }
  return(partials);
}

static SCM g_make_waveshape(SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6, SCM arg7, SCM arg8)
{
  SCM new_wt,gwave = SCM_UNDEFINED;
  mus_scm *gn;
  SCM args[8],keys[4];
  int orig_arg[4] = {0,0,0,0};
  int vals,wsize,npartials = 0,partials_allocated = 0;
  vct *v;
  float freq = 440.0;
  float *wave = NULL,*partials = NULL;
  wsize = DEFAULT_TABLE_SIZE;
  keys[0] = all_keys[C_frequency];
  keys[1] = all_keys[C_partials];
  keys[2] = all_keys[C_size];
  keys[3] = all_keys[C_wave];
  args[0] = arg1; args[1] = arg2; args[2] = arg3; args[3] = arg4; args[4] = arg5; args[5] = arg6; args[6] = arg7; args[7] = arg8;
  vals = decode_keywords(S_make_waveshape,4,keys,8,args,orig_arg);
  if (vals > 0)
    {
      freq = fkeyarg(keys[0],S_make_waveshape,orig_arg[0]+1,args[orig_arg[0]],freq);
      if (!(c_keyword_p(keys[1])))
        {
	  if (gh_list_p(keys[1]))
	    {
	      partials = list2partials(keys[1],&npartials);
	      partials_allocated = 1;
	    }
          else scm_wrong_type_arg(S_make_waveshape,orig_arg[1]+1,args[orig_arg[1]]);
        }
      wsize = ikeyarg(keys[2],S_make_waveshape,orig_arg[2]+1,args[orig_arg[2]],wsize);
      if (!(c_keyword_p(keys[3])))
        {
	  if (vct_p(keys[3]))
	    {
	      gwave = keys[3];
	      v = get_vct(gwave);
	      wave = v->data;
	      wsize = v->length;
	    }
          else scm_wrong_type_arg(S_make_waveshape,orig_arg[3]+1,args[orig_arg[3]]);
        }
    }
  if (wave == NULL) 
    {
      if (partials == NULL) return(SCM_BOOL_F);
      wave = mus_partials2waveshape(npartials,partials,wsize,(float *)CALLOC(wsize,sizeof(float)));
    }
  if (partials_allocated) FREE(partials);
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->gen = mus_make_waveshape(freq,0.0,wave,wsize);
  gn->vcts = (SCM *)CALLOC(1,sizeof(SCM));
  gn->nvcts = 1;
  SCM_NEWCELL(new_wt);
  SCM_SETCAR(new_wt,mus_scm_tag);
  SCM_SETCDR(new_wt,(SCM)gn);
  if (gwave == SCM_UNDEFINED) gwave = make_vct(wsize,wave);
  gn->vcts[0] = gwave;
  return(new_wt);
}

static SCM g_waveshape(SCM obj, SCM index, SCM fm)
{
  float fm1 = 0.0, index1 = 1.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_waveshape_p(mus_get_any(obj)))),obj,SCM_ARG1,S_waveshape);
  if (gh_number_p(index)) index1 = gh_scm2double(index); else if (index != SCM_UNDEFINED) scm_wrong_type_arg(S_waveshape,2,index);
  if (gh_number_p(fm)) fm1 = gh_scm2double(fm); else if (fm != SCM_UNDEFINED) scm_wrong_type_arg(S_waveshape,3,fm);
  return(gh_double2scm(mus_waveshape(mus_get_any(obj),index1,fm1)));
}

static SCM g_waveshape_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_waveshape_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_partials2waveshape(SCM amps, SCM size)
{
  int npartials;
  float *partials,*wave;
  SCM gwave;
  partials = list2partials(amps,&npartials);
  wave = mus_partials2waveshape(npartials,partials,size,(float *)CALLOC(size,sizeof(float)));
  gwave = make_vct(size,wave);
  FREE(partials);
  return(gwave);
}

static SCM g_partials2polynomial(SCM amps, SCM ukind)
{
  int npartials,kind;
  float *partials,*wave;
  if (gh_number_p(ukind))
    kind = gh_scm2int(ukind);
  else kind = 1;
  partials = list2partials(amps,&npartials);
  wave = mus_partials2polynomial(npartials,partials,kind);
  return(make_vct(npartials,wave));
}

static void init_ws(void)
{
  gh_new_procedure(S_make_waveshape,g_make_waveshape,0,8,0);
  gh_new_procedure(S_waveshape,g_waveshape,1,2,0);
  gh_new_procedure(S_waveshape_p,g_waveshape_p,1,0,0);
  gh_new_procedure(S_partials2waveshape,g_partials2waveshape,1,1,0);
  gh_new_procedure(S_partials2polynomial,g_partials2polynomial,1,1,0);
}


/* ---------------- sine-summation ---------------- */

#define S_make_sine_summation "make-sine-summation"
#define S_sine_summation      "sine-summation"
#define S_sine_summation_p    "sine-summation?"

static SCM g_sine_summation_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_sine_summation_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_sine_summation(SCM obj, SCM fm)
{
  float fm1 = 0.0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_sine_summation_p(mus_get_any(obj)))),obj,SCM_ARG1,S_sine_summation);
  if (gh_number_p(fm)) fm1 = gh_scm2double(fm); else if (fm != SCM_UNDEFINED) scm_wrong_type_arg(S_sine_summation,2,fm);
  return(gh_double2scm(mus_sine_summation(mus_get_any(obj),fm1)));
}

static SCM g_make_sine_summation(SCM arglist)
{
  SCM new_sss;
  mus_scm *gn;
  SCM args[10],keys[5];
  int orig_arg[5] = {0,0,0,0,0};
  int vals,i,arglist_len;
  float freq = 440.0,phase = 0.0,a=.5,ratio=1.0;
  int n=1;
  keys[0] = all_keys[C_frequency];
  keys[1] = all_keys[C_initial_phase];
  keys[2] = all_keys[C_n];
  keys[3] = all_keys[C_a];
  keys[4] = all_keys[C_ratio];
  for (i=0;i<10;i++) args[i] = SCM_UNDEFINED;
  arglist_len = gh_length(arglist);
  for (i=0;i<arglist_len;i++) args[i] = gh_list_ref(arglist,gh_int2scm(i));
  vals = decode_keywords(S_make_sine_summation,5,keys,10,args,orig_arg);
  if (vals > 0)
    {
      freq = fkeyarg(keys[0],S_make_sine_summation,orig_arg[0]+1,args[orig_arg[0]],freq);
      phase = fkeyarg(keys[1],S_make_sine_summation,orig_arg[1]+1,args[orig_arg[1]],phase);
      n = ikeyarg(keys[2],S_make_sine_summation,orig_arg[2]+1,args[orig_arg[2]],n);
      a = fkeyarg(keys[3],S_make_sine_summation,orig_arg[3]+1,args[orig_arg[3]],a);
      ratio = fkeyarg(keys[4],S_make_sine_summation,orig_arg[4]+1,args[orig_arg[4]],ratio);
    }
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->gen = mus_make_sine_summation(freq,phase,n,a,ratio);
  SCM_NEWCELL(new_sss);
  SCM_SETCAR(new_sss,mus_scm_tag);
  SCM_SETCDR(new_sss,(SCM)gn);
  return(new_sss);
}

static void init_sss(void)
{
  gh_new_procedure(S_make_sine_summation,g_make_sine_summation,0,0,1);
  gh_new_procedure(S_sine_summation,g_sine_summation,1,1,0);
  gh_new_procedure(S_sine_summation_p,g_sine_summation_p,1,0,0);
}



/* ----------------  filter ---------------- */

#define S_filter          "filter"
#define S_filter_p        "filter?"
#define S_make_filter     "make-filter"
#define S_fir_filter      "fir-filter"
#define S_fir_filter_p    "fir-filter?"
#define S_make_fir_filter "make-fir-filter"
#define S_iir_filter      "iir-filter"
#define S_iir_filter_p    "iir-filter?"
#define S_make_iir_filter "make-iir-filter"
#define S_mus_xcoeffs     "mus-xcoeffs"
#define S_mus_ycoeffs     "mus-ycoeffs"
#define S_mus_order       "mus-order"

static SCM g_filter_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_filter_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}
static SCM g_fir_filter_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_fir_filter_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}
static SCM g_iir_filter_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_iir_filter_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_filter(SCM obj, SCM input)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_filter_p(mus_get_any(obj)))),obj,SCM_ARG1,S_filter);
  SCM_ASSERT((gh_number_p(input)),input,SCM_ARG2,S_filter);
  return(gh_double2scm(mus_filter(mus_get_any(obj),gh_scm2double(input))));
}

static SCM g_fir_filter(SCM obj, SCM input)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_fir_filter_p(mus_get_any(obj)))),obj,SCM_ARG1,S_fir_filter);
  SCM_ASSERT((gh_number_p(input)),input,SCM_ARG2,S_fir_filter);
  return(gh_double2scm(mus_fir_filter(mus_get_any(obj),gh_scm2double(input))));
}

static SCM g_iir_filter(SCM obj, SCM input)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_iir_filter_p(mus_get_any(obj)))),obj,SCM_ARG1,S_iir_filter);
  SCM_ASSERT((gh_number_p(input)),input,SCM_ARG2,S_iir_filter);
  return(gh_double2scm(mus_iir_filter(mus_get_any(obj),gh_scm2double(input))));
}

enum {G_FILTER,G_FIR_FILTER,G_IIR_FILTER};

static SCM g_make_filter_1(int choice, SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6)
{

  SCM new_flt,xwave,ywave;
  mus_scm *gn;
  SCM args[6],keys[3];
  int orig_arg[3] = {0,0,0};
  vct *x = NULL,*y = NULL;
  int nkeys,vals,order;
  char *caller;
  if (choice == G_FILTER) caller = S_filter; else if (choice == G_FIR_FILTER) caller = S_fir_filter; else caller = S_iir_filter;
  keys[0] = all_keys[C_order];
  if (choice == G_IIR_FILTER)
    keys[1] = all_keys[C_y_coeffs];
  else keys[1] = all_keys[C_x_coeffs];
  if (choice == G_FILTER)
    {
      keys[2] = all_keys[C_y_coeffs];
      nkeys = 3;
    }
  else nkeys = 2;
  args[0] = arg1; args[1] = arg2; args[2] = arg3; args[3] = arg4; args[4] = arg5; args[5] = arg6;
  vals = decode_keywords(caller,nkeys,keys,nkeys*2,args,orig_arg);
  if (vals > 0)
    {
      order = ikeyarg(keys[0],caller,orig_arg[0]+1,args[orig_arg[0]],0);
      if (!(c_keyword_p(keys[1])))
        {
	  if (vct_p(keys[1]))
	    {
	      xwave = keys[1];
	      x = get_vct(xwave);
	    }
          else scm_wrong_type_arg(caller,orig_arg[1]+1,args[orig_arg[1]]);
        }
      if (nkeys > 2)
	if (!(c_keyword_p(keys[2])))
	  {
	    if (vct_p(keys[2]))
	      {
		ywave = keys[2];
		y = get_vct(ywave);
	      }
	    else scm_wrong_type_arg(caller,orig_arg[2]+1,args[orig_arg[2]]);
	  }
    }
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  switch (choice)
    {
    case G_FILTER: gn->gen = mus_make_filter(order,x->data,y->data,NULL); break;
    case G_FIR_FILTER: gn->gen = mus_make_fir_filter(order,x->data,NULL); break;
    case G_IIR_FILTER: gn->gen = mus_make_iir_filter(order,x->data,NULL); break;
    }
  gn->vcts = (SCM *)CALLOC(nkeys-1,sizeof(SCM));
  gn->vcts[0] = xwave;
  if (nkeys > 2) gn->vcts[1] = ywave;
  gn->nvcts = nkeys-1;
  SCM_NEWCELL(new_flt);
  SCM_SETCAR(new_flt,mus_scm_tag);
  SCM_SETCDR(new_flt,(SCM)gn);
  return(new_flt);
}

static SCM g_make_filter(SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6)
{
  return(g_make_filter_1(G_FILTER,arg1,arg2,arg3,arg4,arg5,arg6));
}

static SCM g_make_fir_filter(SCM arg1, SCM arg2, SCM arg3, SCM arg4)
{
  return(g_make_filter_1(G_FIR_FILTER,arg1,arg2,arg3,arg4,SCM_UNDEFINED,SCM_UNDEFINED));
}

static SCM g_make_iir_filter(SCM arg1, SCM arg2, SCM arg3, SCM arg4)
{
  return(g_make_filter_1(G_IIR_FILTER,arg1,arg2,arg3,arg4,SCM_UNDEFINED,SCM_UNDEFINED));
}

static SCM g_mus_order(SCM obj)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_filter_p(mus_get_any(obj)))),obj,SCM_ARG1,S_mus_order);
  return(gh_int2scm(mus_order(mus_get_any(obj))));
}

static SCM g_mus_xcoeffs(SCM gen) 
{
  mus_scm *ms;
  SCM_ASSERT((mus_scm_p(gen)),gen,SCM_ARG1,S_data);
  ms = mus_get_scm(gen);
  if (ms->vcts)
    return(ms->vcts[0]); 
  else return(SCM_BOOL_F);
}

static SCM g_mus_ycoeffs(SCM gen) 
{
  mus_scm *ms;
  SCM_ASSERT((mus_scm_p(gen)),gen,SCM_ARG1,S_data);
  ms = mus_get_scm(gen);
  if (ms->vcts)
    {
      if (mus_iir_filter_p(mus_get_any(gen)))  
	return(ms->vcts[0]);
      else return(ms->vcts[1]); 
    }
  else return(SCM_BOOL_F);
}

static void init_flt(void)
{
  gh_new_procedure(S_make_filter,g_make_filter,0,6,0);
  gh_new_procedure(S_filter,g_filter,2,0,0);
  gh_new_procedure(S_filter_p,g_filter_p,1,0,0);
  gh_new_procedure(S_make_fir_filter,g_make_fir_filter,0,4,0);
  gh_new_procedure(S_fir_filter,g_fir_filter,2,0,0);
  gh_new_procedure(S_fir_filter_p,g_fir_filter_p,1,0,0);
  gh_new_procedure(S_make_iir_filter,g_make_iir_filter,0,4,0);
  gh_new_procedure(S_iir_filter,g_iir_filter,2,0,0);
  gh_new_procedure(S_iir_filter_p,g_iir_filter_p,1,0,0);
  gh_new_procedure(S_mus_order,g_mus_order,1,0,0);
  gh_new_procedure(S_mus_xcoeffs,g_mus_xcoeffs,1,0,0);
  gh_new_procedure(S_mus_ycoeffs,g_mus_ycoeffs,1,0,0);
}



/* ---------------- env ---------------- */

#define S_env_p       "env?"
#define S_env         "env"
#define S_make_env    "make-env"
#define S_restart_env "restart-env"
#define S_env_interp  "env-interp"

static SCM g_env_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_env_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_env(SCM obj) 
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_env_p(mus_get_any(obj)))),obj,SCM_ARG1,S_env);
  return(gh_double2scm(mus_env(mus_get_any(obj))));
}

static SCM g_restart_env(SCM obj) 
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_env_p(mus_get_any(obj)))),obj,SCM_ARG1,S_restart_env);
  mus_restart_env(mus_get_any(obj));
  return(SCM_BOOL_F);
}

static SCM g_make_env(SCM arglist)
{
  SCM new_e;
  mus_scm *gn;
  SCM args[14],keys[7];
  int orig_arg[7] = {0,0,0,0,0,0,0};
  int vals,i,len,arglist_len;
  float base,scaler,offset,duration;
  int start,end,npts;
  float *brkpts = NULL,*odata;
  keys[0] = all_keys[C_envelope];
  keys[1] = all_keys[C_scaler];
  keys[2] = all_keys[C_duration];
  keys[3] = all_keys[C_offset];
  keys[4] = all_keys[C_base];
  keys[5] = all_keys[C_end];
  keys[6] = all_keys[C_start];
  for (i=0;i<14;i++) args[i] = SCM_UNDEFINED;
  arglist_len = gh_length(arglist);
  for (i=0;i<arglist_len;i++) args[i] = gh_list_ref(arglist,gh_int2scm(i));
  vals = decode_keywords(S_make_env,7,keys,14,args,orig_arg);
  if (vals > 0)
    {
      /* env data is a list */
      if (!(c_keyword_p(keys[0])))
        {
	  if (gh_list_p(keys[0]))
	    {
	      len = gh_length(keys[0]);
	      npts = len/2;
	      brkpts = (float *)CALLOC(len,sizeof(float));
	      odata = (float *)CALLOC(len,sizeof(float));
	      for (i=0;i<len;i++)
		{
		  brkpts[i] = gh_scm2double(gh_list_ref(keys[0],gh_int2scm(i)));
		  odata[i] = brkpts[i];
		}
	    }
          else scm_wrong_type_arg(S_make_env,orig_arg[0]+1,args[orig_arg[0]]);
        }
      scaler = fkeyarg(keys[1],S_make_env,orig_arg[1]+1,args[orig_arg[1]],1.0);
      duration = fkeyarg(keys[2],S_make_env,orig_arg[2]+1,args[orig_arg[2]],0.0);
      offset = fkeyarg(keys[3],S_make_env,orig_arg[3]+1,args[orig_arg[3]],0.0);
      base = fkeyarg(keys[4],S_make_env,orig_arg[4]+1,args[orig_arg[4]],1.0);
      end = ikeyarg(keys[5],S_make_env,orig_arg[5]+1,args[orig_arg[5]],0);
      start = ikeyarg(keys[6],S_make_env,orig_arg[6]+1,args[orig_arg[6]],0);
    }
  if (brkpts == NULL) scm_misc_error(S_make_env,"no envelope?",SCM_EOL);
  /* odata = vct->data in this context [vcts[0]] */
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->gen = mus_make_env(brkpts,npts,scaler,offset,base,duration,start,end,odata);
  gn->nvcts = 1;
  gn->vcts = (SCM *)CALLOC(1,sizeof(SCM));
  FREE(brkpts);
  SCM_NEWCELL(new_e);
  SCM_SETCAR(new_e,mus_scm_tag);
  SCM_SETCDR(new_e,(SCM)gn);
  gn->vcts[MUS_DATA_POSITION] = make_vct(len,odata);
  return(new_e);
}

static SCM g_env_interp(SCM x, SCM env)
{
  SCM_ASSERT((gh_number_p(x)),x,SCM_ARG1,S_env_interp);
  SCM_ASSERT(((mus_scm_p(env)) && (mus_env_p(mus_get_any(env)))),env,SCM_ARG2,S_env_interp);
  return(gh_double2scm(mus_env_interp(gh_scm2double(x),mus_get_any(env))));
}

static void init_env(void)
{
  gh_new_procedure(S_env_p,g_env_p,1,0,0);
  gh_new_procedure(S_env,g_env,1,0,0);
  gh_new_procedure(S_restart_env,g_restart_env,1,0,0);
  gh_new_procedure(S_make_env,g_make_env,0,0,1);
  gh_new_procedure(S_env_interp,g_env_interp,2,0,0);
}


/* ---------------- io ---------------- */

#define S_file2sample      "file->sample"
#define S_file2sample_p    "file->sample?"
#define S_make_file2sample "make-file->sample"
#define S_sample2file      "sample->file"
#define S_sample2file_p    "sample->file?"
#define S_make_sample2file "make-sample->file"
#define S_file2frame       "file->frame"
#define S_file2frame_p     "file->frame?"
#define S_make_file2frame  "make-file->frame"
#define S_frame2file       "frame->file"
#define S_frame2file_p     "frame->file?"
#define S_make_frame2file  "make-frame->file"
#define S_input_p          "mus-input?"
#define S_output_p         "mus-output?"
#define S_in_any           "in-any"
#define S_out_any          "out-any"
#define S_ina              "ina"
#define S_inb              "inb"
#define S_outa             "outa"
#define S_outb             "outb"
#define S_outc             "outc"
#define S_outd             "outd"
#define S_make_file_input  "make-file-input"
#define S_make_file_output "make-file-output"
#define S_file2array       "file->array"
#define S_array2file       "array->file"

static SCM g_input_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_input_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}
static SCM g_output_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_output_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}
static SCM g_file2sample_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_file2sample_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}
static SCM g_file2frame_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_file2frame_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}
static SCM g_sample2file_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_sample2file_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}
static SCM g_frame2file_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_frame2file_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_in_any_1(char *caller, SCM frame, SCM chan, SCM inp)
{
  mus_input *input = NULL;
  SCM_ASSERT((gh_number_p(frame)),frame,SCM_ARG1,caller);
  SCM_ASSERT((gh_number_p(chan)),chan,SCM_ARG2,caller);
  if (inp != SCM_UNDEFINED) input = (mus_input *)mus_get_any(inp);
  return(gh_double2scm(mus_in_any(gh_scm2int(frame),gh_scm2int(chan),input)));
}

static SCM g_in_any(SCM frame, SCM chan, SCM inp) {return(g_in_any_1(S_in_any,frame,chan,inp));}
static SCM g_ina(SCM frame, SCM inp) {return(g_in_any_1(S_ina,frame,gh_int2scm(0),inp));}
static SCM g_inb(SCM frame, SCM inp) {return(g_in_any_1(S_inb,frame,gh_int2scm(1),inp));}

static SCM g_out_any_1(char *caller, SCM frame, SCM chan, SCM val, SCM outp)
{
  mus_output *output = NULL;
  SCM_ASSERT((gh_number_p(frame)),frame,SCM_ARG1,caller);
  SCM_ASSERT((gh_number_p(chan)),chan,SCM_ARG2,caller);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG3,caller);
  if (outp != SCM_UNDEFINED) output = (mus_output *)mus_get_any(outp);
  return(gh_double2scm(mus_out_any(gh_scm2int(frame),gh_scm2int(chan),gh_scm2double(val),output)));
}

static SCM g_out_any(SCM frame, SCM chan, SCM val, SCM outp) {return(g_out_any_1(S_out_any,frame,chan,val,outp));}
static SCM g_outa(SCM frame, SCM val) {return(g_out_any_1(S_outa,frame,gh_int2scm(0),val,SCM_UNDEFINED));}
static SCM g_outb(SCM frame, SCM val) {return(g_out_any_1(S_outb,frame,gh_int2scm(1),val,SCM_UNDEFINED));}
static SCM g_outc(SCM frame, SCM val) {return(g_out_any_1(S_outc,frame,gh_int2scm(2),val,SCM_UNDEFINED));}
static SCM g_outd(SCM frame, SCM val) {return(g_out_any_1(S_outd,frame,gh_int2scm(3),val,SCM_UNDEFINED));}

static SCM g_make_file2sample(SCM name)
{
  char *filename;
  SCM new_e;
  mus_scm *gn;
  SCM_ASSERT((gh_string_p(name)),name,SCM_ARG1,S_make_file2sample);
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  filename = gh_scm2newstr(name,NULL);
  gn->gen = mus_make_file2sample(filename);
  FREE(filename);
  gn->nvcts = 0;
  SCM_NEWCELL(new_e);
  SCM_SETCAR(new_e,mus_scm_tag);
  SCM_SETCDR(new_e,(SCM)gn);
  return(new_e);
}

static SCM g_file2sample(SCM obj, SCM samp, SCM chan)
{
  int channel = 0;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_input_p(mus_get_any(obj)))),obj,SCM_ARG1,S_file2sample);
  SCM_ASSERT((gh_number_p(samp)),samp,SCM_ARG2,S_file2sample);
  if (chan != SCM_UNDEFINED) 
    {
      SCM_ASSERT((gh_number_p(chan)),chan,SCM_ARG3,S_file2sample);
      channel = gh_scm2int(chan);
    }
  return(gh_double2scm(mus_file2sample(mus_get_any(obj),gh_scm2int(samp),channel)));
}

static SCM g_make_sample2file(SCM name, SCM chans)
{
  char *filename;
  SCM new_e;
  mus_scm *gn;
  SCM_ASSERT((gh_string_p(name)),name,SCM_ARG1,S_make_sample2file);
  SCM_ASSERT((gh_number_p(chans)),chans,SCM_ARG2,S_make_sample2file);
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  filename = gh_scm2newstr(name,NULL);
  gn->gen = mus_make_sample2file(filename,gh_scm2int(chans));
  FREE(filename);
  gn->nvcts = 0;
  SCM_NEWCELL(new_e);
  SCM_SETCAR(new_e,mus_scm_tag);
  SCM_SETCDR(new_e,(SCM)gn);
  return(new_e);
}

static SCM g_sample2file(SCM obj, SCM samp, SCM chan, SCM val)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_output_p(mus_get_any(obj)))),obj,SCM_ARG1,S_sample2file);
  SCM_ASSERT((gh_number_p(samp)),samp,SCM_ARG2,S_sample2file);
  SCM_ASSERT((gh_number_p(chan)),chan,SCM_ARG3,S_sample2file);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG4,S_sample2file);
  return(gh_double2scm(mus_sample2file(mus_get_any(obj),gh_scm2int(samp),gh_scm2int(chan),gh_scm2double(val))));
}

static SCM g_make_file2frame(SCM name)
{
  char *filename;
  SCM new_e;
  mus_scm *gn;
  SCM_ASSERT((gh_string_p(name)),name,SCM_ARG1,S_make_file2frame);
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  filename = gh_scm2newstr(name,NULL);
  gn->gen = mus_make_file2frame(filename);
  FREE(filename);
  gn->nvcts = 0;
  SCM_NEWCELL(new_e);
  SCM_SETCAR(new_e,mus_scm_tag);
  SCM_SETCDR(new_e,(SCM)gn);
  return(new_e);
}

static SCM g_file2frame(SCM obj, SCM samp, SCM outfr)
{
  mus_frame *res = NULL;
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_input_p(mus_get_any(obj)))),obj,SCM_ARG1,S_file2frame);
  SCM_ASSERT((gh_number_p(samp)),samp,SCM_ARG2,S_file2frame);
  if ((mus_scm_p(outfr)) && (mus_frame_p(mus_get_any(outfr)))) res = (mus_frame *)mus_get_any(outfr);
  return(g_wrap_frame(mus_file2frame(mus_get_any(obj),gh_scm2int(samp),res),(res) ? DONT_FREE_FRAME : FREE_FRAME));
}

static SCM g_make_frame2file(SCM name, SCM chans)
{
  char *filename;
  SCM new_e;
  mus_scm *gn;
  SCM_ASSERT((gh_string_p(name)),name,SCM_ARG1,S_make_frame2file);
  SCM_ASSERT((gh_number_p(chans)),chans,SCM_ARG2,S_make_frame2file);
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  filename = gh_scm2newstr(name,NULL);
  gn->gen = mus_make_frame2file(filename,gh_scm2int(chans));
  FREE(filename);
  gn->nvcts = 0;
  SCM_NEWCELL(new_e);
  SCM_SETCAR(new_e,mus_scm_tag);
  SCM_SETCDR(new_e,(SCM)gn);
  return(new_e);
}

static SCM g_frame2file(SCM obj, SCM samp, SCM val)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_output_p(mus_get_any(obj)))),obj,SCM_ARG1,S_frame2file);
  SCM_ASSERT((gh_number_p(samp)),samp,SCM_ARG2,S_frame2file);
  SCM_ASSERT(((mus_scm_p(val)) && (mus_frame_p(mus_get_any(val)))),val,SCM_ARG3,S_frame2file);
  return(g_wrap_frame(mus_frame2file(mus_get_any(obj),gh_scm2int(samp),(mus_frame *)mus_get_any(val)),DONT_FREE_FRAME));
}

static SCM g_make_file_input(SCM arglist)
{
  SCM new_e;
  mus_scm *gn;
  SCM args[6],keys[3];
  int orig_arg[3] = {0,0,0};
  int vals,i,arglist_len;
  int start = 0, chan = 0;
  char *filename = NULL;
  keys[0] = all_keys[C_file];
  keys[1] = all_keys[C_start];
  keys[2] = all_keys[C_channel];
  for (i=0;i<6;i++) args[i] = SCM_UNDEFINED;
  arglist_len = gh_length(arglist);
  for (i=0;i<arglist_len;i++) args[i] = gh_list_ref(arglist,gh_int2scm(i));
  vals = decode_keywords(S_make_file_input,3,keys,6,args,orig_arg);
  if (vals > 0)
    {
      if (!(c_keyword_p(keys[0]))) 
	{
	  if (gh_string_p(keys[0]))
	    filename = gh_scm2newstr(keys[0],NULL);
	  else scm_wrong_type_arg(S_make_file_input,orig_arg[0]+1,args[orig_arg[0]]);
	}
      start = ikeyarg(keys[1],S_make_file_input,orig_arg[1]+1,args[orig_arg[1]],start);
      chan = ikeyarg(keys[2],S_make_file_input,orig_arg[2]+1,args[orig_arg[2]],chan);
    }
  if (filename == NULL) scm_misc_error(S_make_file_input,"no filename?",SCM_EOL);
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->gen = mus_make_file_input(filename,start,chan);
  if (filename) FREE(filename);
  gn->nvcts = 0;
  SCM_NEWCELL(new_e);
  SCM_SETCAR(new_e,mus_scm_tag);
  SCM_SETCDR(new_e,(SCM)gn);
  return(new_e);
}

static SCM g_make_file_output(SCM arglist)
{
  SCM new_e;
  mus_scm *gn;
  SCM args[12],keys[6];
  int orig_arg[6] = {0,0,0,0,0,0};
  int vals,i,arglist_len;
  int srate = 44100, chans = 1, format = SNDLIB_16_LINEAR, type = NeXT_sound_file;
  char *comment = NULL;
  char *filename = NULL;
  keys[0] = all_keys[C_file];
  keys[1] = all_keys[C_srate];
  keys[2] = all_keys[C_channels];
  keys[3] = all_keys[C_format];
  keys[4] = all_keys[C_type];
  keys[5] = all_keys[C_comment];
  for (i=0;i<12;i++) args[i] = SCM_UNDEFINED;
  arglist_len = gh_length(arglist);
  for (i=0;i<arglist_len;i++) args[i] = gh_list_ref(arglist,gh_int2scm(i));
  vals = decode_keywords(S_make_file_output,6,keys,12,args,orig_arg);
  if (vals > 0)
    {
      if (!(c_keyword_p(keys[0]))) 
	{
	  if (gh_string_p(keys[0]))
	    filename = gh_scm2newstr(keys[0],NULL);
	  else scm_wrong_type_arg(S_make_file_output,orig_arg[0]+1,args[orig_arg[0]]);
	}
      srate = ikeyarg(keys[1],S_make_file_output,orig_arg[1]+1,args[orig_arg[1]],srate);
      chans = ikeyarg(keys[2],S_make_file_output,orig_arg[2]+1,args[orig_arg[2]],chans);
      format = ikeyarg(keys[3],S_make_file_output,orig_arg[3]+1,args[orig_arg[3]],format);
      type = ikeyarg(keys[4],S_make_file_output,orig_arg[4]+1,args[orig_arg[4]],type);
      if (!(c_keyword_p(keys[5]))) 
	{
	  if (gh_string_p(keys[5]))
	    comment = gh_scm2newstr(keys[5],NULL);
	  else scm_wrong_type_arg(S_make_file_output,orig_arg[5]+1,args[orig_arg[5]]);
	}
    }
  if (filename == NULL) scm_misc_error(S_make_file_output,"no filename?",SCM_EOL);
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->gen = mus_make_file_output(filename,srate,chans,format,type,comment);
  if (filename) FREE(filename);
  if (comment) FREE(comment);
  gn->nvcts = 0;
  SCM_NEWCELL(new_e);
  SCM_SETCAR(new_e,mus_scm_tag);
  SCM_SETCDR(new_e,(SCM)gn);
  return(new_e);
}

static SCM g_array2file(SCM filename, SCM data, SCM len, SCM srate, SCM channels)
{
  int olen;
  char *name = NULL;
  vct *v;
  SCM_ASSERT((gh_string_p(filename)),filename,SCM_ARG1,S_array2file);
  SCM_ASSERT((vct_p(data)),data,SCM_ARG2,S_array2file);
  SCM_ASSERT((gh_number_p(len)),len,SCM_ARG3,S_array2file);
  SCM_ASSERT((gh_number_p(srate)),srate,SCM_ARG4,S_array2file);
  SCM_ASSERT((gh_number_p(channels)),channels,SCM_ARG5,S_array2file);
  name = gh_scm2newstr(filename,NULL);
  v = get_vct(data);
  olen = mus_fltarray2file(name,v->data,gh_scm2int(len),gh_scm2int(srate),gh_scm2int(channels));
  if (name) FREE(name);
  return(gh_int2scm(olen));
}

static SCM g_file2array(SCM filename, SCM chan, SCM start, SCM samples, SCM data)
{
  int err;
  char *name = NULL;
  vct *v;
  SCM_ASSERT((gh_string_p(filename)),filename,SCM_ARG1,S_file2array);
  SCM_ASSERT((gh_number_p(chan)),chan,SCM_ARG2,S_file2array);
  SCM_ASSERT((gh_number_p(start)),start,SCM_ARG3,S_file2array);
  SCM_ASSERT((gh_number_p(samples)),samples,SCM_ARG4,S_file2array);
  SCM_ASSERT((vct_p(data)),data,SCM_ARG5,S_file2array);
  name = gh_scm2newstr(filename,NULL);
  v = get_vct(data);
  err = mus_file2fltarray(name,gh_scm2int(chan),gh_scm2int(start),gh_scm2int(samples),v->data);
  if (name) FREE(name);
  return(gh_int2scm(err));
}

static void init_io(void)
{
  gh_new_procedure(S_file2sample_p,g_file2sample_p,1,0,0);
  gh_new_procedure(S_make_file2sample,g_make_file2sample,1,0,0);
  gh_new_procedure(S_file2sample,g_file2sample,2,1,0);
  gh_new_procedure(S_file2frame_p,g_file2frame_p,1,0,0);
  gh_new_procedure(S_make_file2frame,g_make_file2frame,1,0,0);
  gh_new_procedure(S_file2frame,g_file2frame,2,0,0);
  gh_new_procedure(S_sample2file_p,g_sample2file_p,1,0,0);
  gh_new_procedure(S_make_sample2file,g_make_sample2file,2,0,0);
  gh_new_procedure(S_sample2file,g_sample2file,4,0,0);
  gh_new_procedure(S_frame2file_p,g_frame2file_p,1,0,0);
  gh_new_procedure(S_frame2file,g_frame2file,3,0,0);
  gh_new_procedure(S_make_frame2file,g_make_frame2file,2,0,0);
  gh_new_procedure(S_input_p,g_input_p,1,0,0);
  gh_new_procedure(S_output_p,g_output_p,1,0,0);
  gh_new_procedure(S_in_any,g_in_any,2,1,0);
  gh_new_procedure(S_ina,g_ina,1,1,0);  
  gh_new_procedure(S_inb,g_inb,1,1,0);
  gh_new_procedure(S_out_any,g_out_any,3,1,0);
  gh_new_procedure(S_outa,g_outa,2,0,0);
  gh_new_procedure(S_outb,g_outb,2,0,0);
  gh_new_procedure(S_outc,g_outc,2,0,0);
  gh_new_procedure(S_outd,g_outd,2,0,0);
  gh_new_procedure(S_make_file_input,g_make_file_input,0,0,1);
  gh_new_procedure(S_make_file_output,g_make_file_output,0,0,1);
  gh_new_procedure(S_array2file,g_array2file,5,0,0);
  gh_new_procedure(S_file2array,g_file2array,5,0,0);
}


/* ---------------- readin ---------------- */

#define S_readin        "readin"
#define S_readin_p      "readin?"
#define S_make_readin   "make-readin"
#define S_increment     "mus-increment"
#define S_set_increment "mus-set-increment"
#define S_location      "mus-location"
#define S_set_location  "mus-set-location"
#define S_channel       "mus-channel"

static SCM g_readin_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_readin_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_readin(SCM obj)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_readin_p(mus_get_any(obj)))),obj,SCM_ARG1,S_readin);
  return(gh_double2scm(mus_readin(mus_get_any(obj))));
}

static SCM g_make_readin(SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6, SCM arg7, SCM arg8)
{
  /* optkey file channel start direction */
  SCM new_rd;
  mus_scm *gn;
  char *file = NULL;
  SCM args[8],keys[4];
  int orig_arg[4] = {0,0,0,0};
  int vals;
  int channel = 0, start = 0, direction = 1;
  keys[0] = all_keys[C_file];
  keys[1] = all_keys[C_channel];
  keys[2] = all_keys[C_start];
  keys[3] = all_keys[C_direction];
  args[0] = arg1; args[1] = arg2; args[2] = arg3; args[3] = arg4; args[4] = arg5; args[5] = arg6; args[6] = arg7; args[7] = arg8; 
  vals = decode_keywords(S_make_readin,4,keys,8,args,orig_arg);
  if (vals > 0)
    {
      if (!(c_keyword_p(keys[0])))
        {
	  if (gh_string_p(keys[0]))
	    file = gh_scm2newstr(keys[0],NULL);
	  else scm_wrong_type_arg(S_make_readin,orig_arg[0]+1,args[orig_arg[0]]);
	}
      channel = ikeyarg(keys[1],S_make_readin,orig_arg[1]+1,args[orig_arg[1]],channel);
      start = ikeyarg(keys[2],S_make_readin,orig_arg[2]+1,args[orig_arg[2]],start);
      direction = ikeyarg(keys[3],S_make_readin,orig_arg[3]+1,args[orig_arg[3]],direction);
    }
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->gen = mus_make_readin(file,channel,start,direction);
  SCM_NEWCELL(new_rd);
  SCM_SETCAR(new_rd,mus_scm_tag);
  SCM_SETCDR(new_rd,(SCM)gn);
  if (file) FREE(file); /* copied by mus_make_readin, allocated by gh_scm2newstr */
  return(new_rd);
}

static SCM g_increment(SCM obj)
{
  SCM_ASSERT((mus_scm_p(obj)),obj,SCM_ARG1,S_increment);
  return(gh_double2scm(mus_increment(mus_get_any(obj))));
}

static SCM g_set_increment(SCM obj, SCM val)
{
  SCM_ASSERT((mus_scm_p(obj)),obj,SCM_ARG1,S_set_increment);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG2,S_set_increment);
  return(gh_double2scm(mus_set_increment(mus_get_any(obj),gh_scm2double(val))));
}

static SCM g_location(SCM obj)
{
  SCM_ASSERT((mus_scm_p(obj)),obj,SCM_ARG1,S_location);
  return(gh_int2scm(mus_location((mus_input *)mus_get_any(obj))));
}

static SCM g_set_location(SCM obj, SCM val)
{
  SCM_ASSERT((mus_scm_p(obj)),obj,SCM_ARG1,S_set_location);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG2,S_set_location);
  return(gh_int2scm(mus_set_location((mus_input *)mus_get_any(obj),gh_scm2int(val))));
}

static SCM g_channel(SCM obj)
{
  SCM_ASSERT((mus_scm_p(obj)),obj,SCM_ARG1,S_channel);
  return(gh_int2scm(mus_channel((mus_input *)mus_get_any(obj))));
}

static void init_rdin(void)
{
  gh_new_procedure(S_readin_p,g_readin_p,1,0,0);
  gh_new_procedure(S_readin,g_readin,1,0,0);
  gh_new_procedure(S_make_readin,g_make_readin,0,8,0);
  gh_new_procedure(S_location,g_location,1,0,0);
  gh_new_procedure(S_set_location,g_set_location,2,0,0);
  gh_new_procedure(S_increment,g_increment,1,0,0);
  gh_new_procedure(S_set_increment,g_set_increment,2,0,0);
  gh_new_procedure(S_channel,g_channel,1,0,0);
}


/* ---------------- locsig ---------------- */

#define S_locsig_p          "locsig?"
#define S_locsig            "locsig"
#define S_make_locsig       "make-locsig"
#define S_channels          "mus-channels"
#define S_locsig_ref        "locsig-ref"
#define S_locsig_set        "locsig-set!"
#define S_locsig_reverb_ref "locsig-reverb-ref"
#define S_locsig_reverb_set "locsig-reverb-set!"

static SCM g_locsig_ref(SCM obj, SCM chan)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_locsig_p(mus_get_any(obj)))),obj,SCM_ARG1,S_locsig_ref);
  SCM_ASSERT((gh_number_p(chan)),chan,SCM_ARG2,S_locsig_ref);
  return(gh_double2scm(mus_locsig_ref(mus_get_any(obj),gh_scm2int(chan))));
}

static SCM g_locsig_set(SCM obj, SCM chan, SCM val)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_locsig_p(mus_get_any(obj)))),obj,SCM_ARG1,S_locsig_set);
  SCM_ASSERT((gh_number_p(chan)),chan,SCM_ARG2,S_locsig_set);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG3,S_locsig_set);
  return(gh_double2scm(mus_locsig_set(mus_get_any(obj),gh_scm2int(chan),gh_scm2double(val))));
}

static SCM g_locsig_reverb_ref(SCM obj, SCM chan)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_locsig_p(mus_get_any(obj)))),obj,SCM_ARG1,S_locsig_reverb_ref);
  SCM_ASSERT((gh_number_p(chan)),chan,SCM_ARG2,S_locsig_reverb_ref);
  return(gh_double2scm(mus_locsig_reverb_ref(mus_get_any(obj),gh_scm2int(chan))));
}

static SCM g_locsig_reverb_set(SCM obj, SCM chan, SCM val)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_locsig_p(mus_get_any(obj)))),obj,SCM_ARG1,S_locsig_reverb_set);
  SCM_ASSERT((gh_number_p(chan)),chan,SCM_ARG2,S_locsig_reverb_set);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG3,S_locsig_reverb_set);
  return(gh_double2scm(mus_locsig_reverb_set(mus_get_any(obj),gh_scm2int(chan),gh_scm2double(val))));
}

static SCM g_locsig_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_locsig_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_locsig(SCM obj, SCM loc, SCM val)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_locsig_p(mus_get_any(obj)))),obj,SCM_ARG1,S_locsig);
  SCM_ASSERT((gh_number_p(loc)),loc,SCM_ARG2,S_locsig);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG3,S_locsig);
  return(g_wrap_frame(mus_locsig(mus_get_any(obj),gh_scm2int(loc),gh_scm2double(val)),DONT_FREE_FRAME));
}

static SCM g_make_locsig(SCM arglist)
{
  SCM new_loc,out_obj = SCM_UNDEFINED,rev_obj = SCM_UNDEFINED;
  mus_scm *gn;
  mus_output *outp = NULL, *revp = NULL;
  SCM args[12],keys[6];
  int orig_arg[6] = {0,0,0,0,0,0};
  int vals,i,arglist_len,vlen = 0,out_chans = 1;
  float degree = 0.0, distance = 1.0, reverb = 0.0;
  keys[0] = all_keys[C_degree];
  keys[1] = all_keys[C_distance];
  keys[2] = all_keys[C_reverb];
  keys[3] = all_keys[C_output];  
  keys[4] = all_keys[C_revout];
  keys[5] = all_keys[C_channels];
  for (i=0;i<12;i++) args[i] = SCM_UNDEFINED;
  arglist_len = gh_length(arglist);
  for (i=0;i<arglist_len;i++) args[i] = gh_list_ref(arglist,gh_int2scm(i));
  vals = decode_keywords(S_make_locsig,6,keys,12,args,orig_arg);
  if (vals > 0)
    {
      degree = fkeyarg(keys[0],S_make_locsig,orig_arg[0]+1,args[orig_arg[0]],degree);
      distance = fkeyarg(keys[1],S_make_locsig,orig_arg[1]+1,args[orig_arg[1]],distance);
      reverb = fkeyarg(keys[2],S_make_locsig,orig_arg[2]+1,args[orig_arg[2]],reverb);
      if (!(c_keyword_p(keys[3]))) 
	{
	  if ((mus_scm_p(keys[3])) && (mus_output_p(mus_get_any(keys[3]))))
	    {
	      out_obj = keys[3];
	      vlen++;
	      outp = (mus_output *)mus_get_any(keys[3]);
	      out_chans = mus_channels((mus_any *)outp);
	    }
	  else scm_wrong_type_arg(S_make_locsig,orig_arg[3]+1,args[orig_arg[3]]);
	}
      if (!(c_keyword_p(keys[4]))) 
	{
	  if ((mus_scm_p(keys[4])) && (mus_output_p(mus_get_any(keys[4]))))
	    {
	      rev_obj = keys[4];
	      vlen++;
	      revp = (mus_output *)mus_get_any(keys[4]);
	    }
	  else scm_wrong_type_arg(S_make_locsig,orig_arg[4]+1,args[orig_arg[4]]);
	}
      out_chans = ikeyarg(keys[5],S_make_locsig,orig_arg[5]+1,args[orig_arg[5]],out_chans);
    }
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->gen = mus_make_locsig(degree,distance,reverb,out_chans,outp,revp);
  if (vlen > 0)
    {
      gn->vcts = (SCM *)CALLOC(vlen,sizeof(SCM));
      i = 0;
      if (out_obj != SCM_UNDEFINED) gn->vcts[i++] = out_obj;
      if (rev_obj != SCM_UNDEFINED) gn->vcts[i] = rev_obj;
      gn->nvcts = vlen;
    }
  SCM_NEWCELL(new_loc);
  SCM_SETCAR(new_loc,mus_scm_tag);
  SCM_SETCDR(new_loc,(SCM)gn);
  return(new_loc);
}

static SCM g_channels(SCM obj)
{
  SCM_ASSERT((mus_scm_p(obj)),obj,SCM_ARG1,S_channels);
  return(gh_int2scm(mus_channels(mus_get_any(obj))));
}

static void init_locs(void)
{
  gh_new_procedure(S_locsig_p,g_locsig_p,1,0,0);
  gh_new_procedure(S_locsig,g_locsig,3,0,0);
  gh_new_procedure(S_make_locsig,g_make_locsig,0,0,1);
  gh_new_procedure(S_channels,g_channels,1,0,0);
  gh_new_procedure(S_locsig_ref,g_locsig_ref,2,0,0);
  gh_new_procedure(S_locsig_reverb_ref,g_locsig_reverb_ref,2,0,0);
  gh_new_procedure(S_locsig_set,g_locsig_set,3,0,0);
  gh_new_procedure(S_locsig_reverb_set,g_locsig_reverb_set,3,0,0);
}


/* ---------------- src ---------------- */

static float funcall_reader (void *ptr, int direction)
{
  /* if this is called, it's a callback from C, where ptr is a mus_scm object whose vcts[0]
   * field is an SCM procedure to be called, the result being returned back to C.  In the
   * Scheme world, it's a procedure of one arg, the current read direction
   */
  mus_scm *gn = (mus_scm *)ptr;
  if ((gn) && (gn->vcts) && (gn->vcts[0]) && (gh_procedure_p(gn->vcts[0])))
    /* the gh_procedure_p call can be confused by 0 -> segfault! */
    return(gh_scm2double(gh_call1(gn->vcts[0],gh_int2scm(direction))));
  else return(0.0);
}

#define S_src       "src"
#define S_src_p     "src?"
#define S_make_src  "make-src"

static SCM g_src_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_src_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_src(SCM obj, SCM pm, SCM func) 
{
  float pm1 = 0.0;
  mus_scm *gn = mus_get_scm(obj);
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_src_p(mus_get_any(obj)))),obj,SCM_ARG1,S_src);
  if (gh_number_p(pm)) pm1 = gh_scm2double(pm); else if (pm != SCM_UNDEFINED) scm_wrong_type_arg(S_src,2,pm);
  if (gh_procedure_p(func)) gn->vcts[0] = func;
  return(gh_double2scm(mus_src(mus_get_any(obj),pm1,0)));
}

static SCM g_make_src(SCM arg1, SCM arg2, SCM arg3, SCM arg4, SCM arg5, SCM arg6)
{
  SCM new_src,in_obj = SCM_UNDEFINED;
  mus_scm *gn;
  int vals,wid = 0;
  SCM args[6],keys[3];
  int orig_arg[3] = {0,0,0};
  float srate = 1.0;
  keys[0] = all_keys[C_input];
  keys[1] = all_keys[C_srate];
  keys[2] = all_keys[C_width];
  args[0] = arg1; args[1] = arg2; args[2] = arg3; args[3] = arg4; args[4] = arg5; args[5] = arg6;
  vals = decode_keywords(S_make_src,3,keys,6,args,orig_arg);
  if (vals > 0)
    {
      if (!(c_keyword_p(keys[0]))) 
	{
	  if (gh_procedure_p(keys[0]))
	    in_obj = keys[0];
	  else scm_wrong_type_arg(S_make_src,orig_arg[0]+1,args[orig_arg[0]]);
	}
      srate = fkeyarg(keys[1],S_make_src,orig_arg[1]+1,args[orig_arg[1]],srate);
      wid = ikeyarg(keys[2],S_make_src,orig_arg[2]+1,args[orig_arg[2]],wid);
    }
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->nvcts = 1;
  gn->vcts = (SCM *)CALLOC(1,sizeof(SCM));
  if (in_obj != SCM_UNDEFINED) gn->vcts[0] = in_obj;
  gn->gen = mus_make_src(funcall_reader,srate,wid,gn);
  SCM_NEWCELL(new_src);
  SCM_SETCAR(new_src,mus_scm_tag);
  SCM_SETCDR(new_src,(SCM)gn);
  return(new_src);
}

static void init_sr(void)
{
  gh_new_procedure(S_src_p,g_src_p,1,0,0);
  gh_new_procedure(S_src,g_src,1,2,0);
  gh_new_procedure(S_make_src,g_make_src,0,6,0);
}


/* ---------------- granulate ---------------- */

#define S_granulate_p    "granulate?"
#define S_granulate      "granulate"
#define S_make_granulate "make-granulate"
#define S_ramp           "mus-ramp"
#define S_set_ramp       "mus-set-ramp"
#define S_hop            "mus-hop"
#define S_set_hop        "mus-set-hop"

static SCM g_granulate_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_granulate_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_granulate(SCM obj, SCM func) 
{
  mus_scm *gn = mus_get_scm(obj);
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_granulate_p(mus_get_any(obj)))),obj,SCM_ARG1,S_granulate);
  if (gh_procedure_p(func)) gn->vcts[0] = func;
  return(gh_double2scm(mus_granulate(mus_get_any(obj),0)));
}

static SCM g_ramp(SCM obj)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_granulate_p(mus_get_any(obj)))),obj,SCM_ARG1,S_ramp);
  return(gh_double2scm(mus_ramp(mus_get_any(obj))));
}

static SCM g_set_ramp(SCM obj, SCM val)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_granulate_p(mus_get_any(obj)))),obj,SCM_ARG1,S_set_ramp);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG2,S_set_ramp);
  return(gh_double2scm(mus_set_ramp(mus_get_any(obj),gh_scm2double(val))));
}

static SCM g_hop(SCM obj)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_granulate_p(mus_get_any(obj)))),obj,SCM_ARG1,S_hop);
  return(gh_int2scm(mus_hop(mus_get_any(obj))));
}

static SCM g_set_hop(SCM obj, SCM val)
{
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_granulate_p(mus_get_any(obj)))),obj,SCM_ARG1,S_set_hop);
  SCM_ASSERT((gh_number_p(val)),val,SCM_ARG2,S_set_hop);
  return(gh_int2scm(mus_set_hop(mus_get_any(obj),gh_scm2int(val))));
}

static SCM g_make_granulate(SCM arglist)
{
  SCM new_e,in_obj = SCM_UNDEFINED;
  mus_scm *gn;
  SCM args[16],keys[8];
  int orig_arg[8] = {0,0,0,0,0,0,0,0};
  int vals,i,arglist_len,maxsize = 0;
  float expansion = 1.0, segment_length = .15, segment_scaler = .6, ramp_time = .4, output_hop = .05;
  float jitter = 1.0;
  keys[0] = all_keys[C_input];
  keys[1] = all_keys[C_expansion];
  keys[2] = all_keys[C_length];
  keys[3] = all_keys[C_scaler];
  keys[4] = all_keys[C_hop];
  keys[5] = all_keys[C_ramp];
  keys[6] = all_keys[C_jitter];
  keys[7] = all_keys[C_max_size];
  for (i=0;i<16;i++) args[i] = SCM_UNDEFINED;
  arglist_len = gh_length(arglist);
  for (i=0;i<arglist_len;i++) args[i] = gh_list_ref(arglist,gh_int2scm(i));
  vals = decode_keywords(S_make_granulate,8,keys,16,args,orig_arg);
  if (vals > 0)
    {
      if (!(c_keyword_p(keys[0]))) 
	{
	  if (gh_procedure_p(keys[0]))
	    in_obj = keys[0];
	  else scm_wrong_type_arg(S_make_granulate,orig_arg[0]+1,args[orig_arg[0]]);
	}
      expansion = fkeyarg(keys[1],S_make_granulate,orig_arg[1]+1,args[orig_arg[1]],expansion);
      segment_length = fkeyarg(keys[2],S_make_granulate,orig_arg[2]+1,args[orig_arg[2]],segment_length);
      segment_scaler = fkeyarg(keys[3],S_make_granulate,orig_arg[3]+1,args[orig_arg[3]],segment_scaler);
      output_hop = fkeyarg(keys[4],S_make_granulate,orig_arg[4]+1,args[orig_arg[4]],output_hop);
      ramp_time = fkeyarg(keys[5],S_make_granulate,orig_arg[5]+1,args[orig_arg[5]],ramp_time);
      jitter = fkeyarg(keys[6],S_make_granulate,orig_arg[6]+1,args[orig_arg[6]],jitter);
      maxsize = ikeyarg(keys[7],S_make_granulate,orig_arg[7]+1,args[orig_arg[7]],maxsize);
    }
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->vcts = (SCM *)CALLOC(1,sizeof(SCM));
  if (in_obj != SCM_UNDEFINED) gn->vcts[0] = in_obj;
  gn->nvcts = 1;
  gn->gen = mus_make_granulate(funcall_reader,expansion,segment_length,segment_scaler,output_hop,ramp_time,jitter,maxsize,gn);
  SCM_NEWCELL(new_e);
  SCM_SETCAR(new_e,mus_scm_tag);
  SCM_SETCDR(new_e,(SCM)gn);
  return(new_e);
}

static void init_spd(void)
{
  gh_new_procedure(S_granulate_p,g_granulate_p,1,0,0);
  gh_new_procedure(S_granulate,g_granulate,1,1,0);
  gh_new_procedure(S_ramp,g_ramp,1,0,0);
  gh_new_procedure(S_set_ramp,g_set_ramp,2,0,0);
  gh_new_procedure(S_hop,g_hop,1,0,0);
  gh_new_procedure(S_set_hop,g_set_hop,2,0,0);
  gh_new_procedure(S_make_granulate,g_make_granulate,0,0,1);
}



/* ---------------- convolve ---------------- */

#define S_convolve_p    "convolve?"
#define S_convolve      "convolve"
#define S_make_convolve "make-convolve"

static SCM g_convolve_p(SCM obj) {return(((mus_scm_p(obj)) && (mus_convolve_p(mus_get_any(obj)))) ? SCM_BOOL_T : SCM_BOOL_F);}

static SCM g_convolve(SCM obj, SCM func) 
{
  mus_scm *gn = mus_get_scm(obj);
  SCM_ASSERT(((mus_scm_p(obj)) && (mus_convolve_p(mus_get_any(obj)))),obj,SCM_ARG1,S_convolve);
  if (gh_procedure_p(func)) gn->vcts[0] = func;
  return(gh_double2scm(mus_convolve(mus_get_any(obj),0)));
}

/* filter-size? */

static SCM g_make_convolve(SCM arglist)
{
  SCM new_e;
  mus_scm *gn;
  SCM args[6],keys[3];
  int orig_arg[3] = {0,0,0};
  int vals,i,arglist_len,fftlen;
  vct *filter;
  SCM filt = SCM_UNDEFINED,in_obj = SCM_UNDEFINED;
  int fft_size = 0;
  keys[0] = all_keys[C_input];
  keys[1] = all_keys[C_filter];
  keys[2] = all_keys[C_fft_size];
  for (i=0;i<6;i++) args[i] = SCM_UNDEFINED;
  arglist_len = gh_length(arglist);
  for (i=0;i<arglist_len;i++) args[i] = gh_list_ref(arglist,gh_int2scm(i));
  vals = decode_keywords(S_make_convolve,3,keys,6,args,orig_arg);
  if (vals > 0)
    {
      if (!(c_keyword_p(keys[0]))) 
	{
	  if (gh_procedure_p(keys[0]))
	    in_obj = keys[0];
	  else scm_wrong_type_arg(S_make_convolve,orig_arg[0]+1,args[orig_arg[0]]);
	}
      if (!(c_keyword_p(keys[1]))) 
	{
	  if (vct_p(keys[1]))
	    {
	      filt = keys[1];
	      filter = get_vct(filt);
	    }
          else scm_wrong_type_arg(S_make_convolve,orig_arg[1]+1,args[orig_arg[1]]);
	}
      fft_size = ikeyarg(keys[2],S_make_convolve,orig_arg[2]+1,args[orig_arg[2]],fft_size);
    }
  fftlen = pow(2.0,1 + (int)ceil(log((float)(filter->length))/log(2.0)));
  if (fft_size < fftlen) fft_size = fftlen;
  gn = (mus_scm *)CALLOC(1,sizeof(mus_scm));
  gn->nvcts = 2;
  gn->vcts = (SCM *)CALLOC(2,sizeof(SCM));
  if (in_obj != SCM_UNDEFINED) gn->vcts[0] = in_obj;
  gn->vcts[1] = filt;
  gn->gen = mus_make_convolve(funcall_reader,filter->data,fft_size,filter->length,gn);
  SCM_NEWCELL(new_e);
  SCM_SETCAR(new_e,mus_scm_tag);
  SCM_SETCDR(new_e,(SCM)gn);
  return(new_e);
}

static void init_conv(void)
{
  gh_new_procedure(S_convolve_p,g_convolve_p,1,0,0);
  gh_new_procedure(S_convolve,g_convolve,1,1,0);
  gh_new_procedure(S_make_convolve,g_make_convolve,0,0,1);
}


/* ---------------- mix ---------------- */

#define S_mus_mix "mus-mix"

static SCM g_mus_mix(SCM out, SCM in, SCM ost, SCM olen, SCM ist, SCM mx, SCM envs)
{
  mus_mixer *mx1 = NULL;
  mus_any ***envs1 = NULL;
  char *outfile = NULL,*infile = NULL;
  int in_len,out_len,i,j,ostart = 0,istart = 0,osamps = 0;
  SCM_ASSERT((gh_string_p(out)),out,SCM_ARG1,S_mus_mix);
  SCM_ASSERT((gh_string_p(in)),in,SCM_ARG2,S_mus_mix);
  SCM_ASSERT((ost == SCM_UNDEFINED) || (gh_number_p(ost)),ost,SCM_ARG3,S_mus_mix);
  SCM_ASSERT((olen == SCM_UNDEFINED) || (gh_number_p(olen)),olen,SCM_ARG4,S_mus_mix);
  SCM_ASSERT(((ist == SCM_UNDEFINED) || gh_number_p(ist)),ist,SCM_ARG5,S_mus_mix);
  SCM_ASSERT((mx == SCM_UNDEFINED) || ((mus_scm_p(mx)) && (mus_mixer_p(mus_get_any(mx)))),mx,SCM_ARG6,S_mus_mix);
  SCM_ASSERT((envs == SCM_UNDEFINED) || (gh_vector_p(envs)),envs,SCM_ARG7,S_mus_mix);
  if (ost != SCM_UNDEFINED) ostart = gh_scm2int(ost);
  if (ist != SCM_UNDEFINED) istart = gh_scm2int(ist);
  if (mx != SCM_UNDEFINED) mx1 = (mus_mixer *)mus_get_any(mx);
  if (envs != SCM_UNDEFINED)
    {
      /* pack into a C-style array of arrays of env pointers */
      in_len = gh_vector_length(envs);
      out_len = gh_vector_length(gh_vector_ref(envs,gh_int2scm(0)));
      envs1 = (mus_any ***)CALLOC(in_len,sizeof(mus_any **));
      for (i=0;i<in_len;i++)
	{
	  envs1[i] = (mus_any **)CALLOC(out_len,sizeof(mus_any *));
	  for (j=0;j<out_len;j++) 
	    envs1[i][j] = mus_get_any(gh_vector_ref(gh_vector_ref(envs,gh_int2scm(i)),gh_int2scm(j)));
	}
    }
  outfile = gh_scm2newstr(out,NULL);
  infile = gh_scm2newstr(in,NULL);
  if (olen != SCM_UNDEFINED) osamps = gh_scm2int(olen); else osamps = sound_frames(infile);
  mus_mix(outfile,infile,ostart,osamps,istart,mx1,envs1);
  if (outfile) FREE(outfile);
  if (infile) FREE(infile);
  if (envs1) 
    {
      for (i=0;i<in_len;i++) if (envs1[i]) FREE(envs1[i]);
      FREE(envs1);
    }
  return(SCM_BOOL_T);
}

void init_mus2scm_module(void)
{
  init_mus_module();
  init_mus_scm();
#ifndef WITH_MUS_MODULE
  /* if we're being loaded into Snd, don't override its choice of mus error handlers */
  mus_set_error_handler(mus_error2scm);
#endif
  init_keywords();
  init_simple_stuff();
  init_generic_funcs();
  init_oscil();
  init_dly();
  init_noi();
  init_cosp();
  init_tbl();
  init_sw();
  init_asyfm();
  init_smpflt();
  init_wt();
  init_rblk();
  init_frame();
  init_mixer();
  init_formant();
  init_ws();
  init_sss();
  init_flt();
  init_env();
  init_locs();
  init_io();
  init_rdin();
  init_spd();
  init_sr();
  init_conv();
  init_spd();

  gh_new_procedure(S_mus_mix,g_mus_mix,2,5,0);

  /* this next code implements (setf (mus-frequency gen) val) constructs */
  /* it is from the guile mailing list, written by Maciej Stachowiak <mstachow@mit.edu> */
  gh_eval_str("(defmacro setf (place value) (if (pair? place) `((setter ,(car place)) ,@(cdr place) ,value) `(set! place value)))\n\
               (define (setter proc) (procedure-property proc 'setter))\n\
               (set-procedure-property! setter 'setter (lambda (proc setter) (set-procedure-property! proc 'setter setter)))\n\
\n\
               (setf (setter " S_phase ") " S_set_phase ")\n\
               (setf (setter " S_data ") " S_set_data ")\n\
               (setf (setter " S_length ") " S_set_length ")\n\
               (setf (setter " S_frequency ") " S_set_frequency ")\n\
               (setf (setter " S_scaler ") " S_set_scaler ")\n\
               (setf (setter " S_a0 ") " S_set_a0 ")\n\
               (setf (setter " S_a1 ") " S_set_a1 ")\n\
               (setf (setter " S_a2 ") " S_set_a2 ")\n\
               (setf (setter " S_b1 ") " S_set_b1 ")\n\
               (setf (setter " S_b2 ") " S_set_b2 ")\n\
               (setf (setter " S_formant_radius ") " S_set_formant_radius ")\n\
               (setf (setter " S_feedback ") " S_set_feedback ")\n\
               (setf (setter " S_feedforward ") " S_set_feedforward ")\n\
               (setf (setter " S_location ") " S_set_location ")\n\
               (setf (setter " S_increment ") " S_set_increment ")\n\
               (setf (setter " S_ramp ") " S_set_ramp ")\n\
               (setf (setter " S_hop ") " S_set_hop ")\n\
               (setf (setter " S_frame_ref ") " S_set_frame_ref ")\n\
               (setf (setter " S_mixer_ref ") " S_set_mixer_ref ")\n\
               (setf (setter " S_locsig_ref ") " S_locsig_set ")\n\
               (setf (setter " S_locsig_reverb_ref ") " S_locsig_reverb_set ")\n\
\n\
");

}


/* (definstrument a (arg ...) (let (... (e (make-env))) (... (run (loop ... (env e) (outa ...))))))
 *   return closure of outer let and run as appliable func?
 *   could all CL-compatibility stuff be local to definstrument?  (i.e. first for car)
 *
 *  (let (...) (let (beg end) (init envs) (loop...)))
 *
 *  (with-sound (...) ...) -> current chans, create if none?
 *  sampling_rate from :srate
 *  how to handle user's keywords in definstrument?
 *  with-sound would need fluid-let as outer
 *
 * here is something close to with-sound body handling:
 * (defmacro with-fluids (bindings . body)
 *   `(with-fluids* (list ,@(map car bindings)) (list ,@(map cadr bindings))
 *      (lambda () ,@body)))
 * 
 * (defmacro with-sound (bindings . body)
 *   `(with-sound-internal (fluid-let ,bindings . ,body)))
 */

#define NUM_CLM_NAMES 276
static char *clm_names[NUM_CLM_NAMES] = {
  ":a",":a0",":a1",":a2",":amplitude",":b1",":b2",":base",":channel",":channels",":comment",":cosines",
  ":degree",":direction",":distance",":duration",":end",":envelope",":expansion",
  ":feedback",":feedforward",":fft-size",":file",":fill-time",":filter",":format",":frequency",
  ":gain",":hop",":initial-contents",":initial-element",":initial-phase",":input",":jitter",":length",":max-size",
  ":n",":offset",":order",":output",":partials",":r",":radius",":ramp",":ratio",":reverb",":revout",
  ":scaler",":size",":srate",":start",":type",":wave",":xcoeffs",":ycoeffs",
  "all-pass","all-pass?","amplitude-modulate","array->file","array-interp","asymmetric-fm","asymmetric-fm?",
  "buffer->frame","buffer->sample","buffer-empty?","buffer?",
  "clear-array","comb","comb?","contrast-enhancement","convolution","convolve","convolve?",
  "db->linear","degrees->radians","delay","delay?","dot-product",
  "env","env-interp","env?",
  "file->array","file->frame","file->frame?","file->sample","file->sample?","filter","filter?","fir-filter","fir-filter?",
  "formant","formant?","frame*","frame+","frame->buffer","frame->file","frame->file?","frame->frame","frame->sample","frame-ref",
  "frame-set!","frame?",
  "granulate","granulate?",
  "hz->radians",
  "iir-filter","iir-filter?","in-any","in-hz","ina","inb",
  "linear->db","locsig","locsig-ref","locsig-reverb-ref","locsig-reverb-set!","locsig-set!","locsig?",
  "make-all-pass","make-asymmetric-fm","make-buffer","make-comb","make-convolve","make-delay","make-env",
  "make-fft-window","make-file->frame","make-file->sample","make-file-input","make-file-output","make-filter",
  "make-fir-filter","make-formant","make-frame","make-frame->file","make-granulate","make-iir-filter","make-locsig",
  "make-mixer","make-notch","make-one-pole","make-one-zero","make-oscil","make-ppolar","make-pulse-train","make-rand",
  "make-rand-interp","make-readin","make-sample->file","make-sawtooth-wave","make-sine-summation","make-square-wave",
  "make-src","make-sum-of-cosines","make-table-lookup","make-triangle-wave","make-two-pole","make-two-zero",
  "make-wave-train","make-waveshape","make-zpolar",
  "mixer*","mixer-ref","mixer-set!","mixer?","multiply-arrays",
  "mus-a0","mus-a1","mus-a2","mus-array-print-length",
  "mus-b1","mus-b2","mus-channel","mus-channels","mus-cosines","mus-data",
  "mus-feedback","mus-feedforward","mus-fft","mus-formant-radius","mus-frequency","mus-hop",
  "mus-increment","mus-input?","mus-length","mus-location","mus-mix",
  "mus-order","mus-output?","mus-phase","mus-ramp","mus-random","mus-scaler",
  "mus-set-a0","mus-set-a1","mus-set-a2","mus-set-array-print-length",
  "mus-set-b1","mus-set-b2","mus-set-data","mus-set-feedback","mus-set-feedforward","mus-set-formant-radius",
  "mus-set-frequency","mus-set-hop","mus-set-increment","mus-set-length","mus-set-location",
  "mus-set-phase","mus-set-ramp","mus-set-rand-seed","mus-set-scaler",
  "mus-set-srate","mus-srate","mus-xcoeffs","mus-ycoeffs",
  "notch","notch?",
  "one-pole","one-pole?","one-zero","one-zero?","oscil","oscil?","out-any","outa","outb","outc","outd",
  "partials->polynomial","partials->wave","partials->waveshape","phase-partials->wave","phase-partials->waveshape",
  "polynomial","pulse-train","pulse-train?",
  "radians->degrees","radians->hz","rand","rand-interp","rand-interp?","rand?","readin",
  "readin?","rectangular->polar","restart-env","ring-modulate",
  "sample->buffer","sample->file","sample->file?","sample->frame","sawtooth-wave","sawtooth-wave?",
  "sine-summation","sine-summation?","spectrum","square-wave","square-wave?","src","src?",
  "sum-of-cosines","sum-of-cosines?",
  "table-lookup","table-lookup?","tap","triangle-wave","triangle-wave?","two-pole","two-pole?",
  "two-zero","two-zero?",
  "wave-train","wave-train?","waveshape","waveshape?"
};

int mus_num_commands(void);
char **mus_commands(void);

int mus_num_commands(void) {return(NUM_CLM_NAMES);}
char **mus_commands(void) {return(clm_names);}

char *get_CLM_help(void);

static char CLM_help[] = 
"  all-pass            (gen input pm)       all-pass filter\n\
  all-pass?           (gen)                #t if gen is all-pass filter\n\
  amplitude-modulate  (carrier in1 in2)    amplitude modulation\n\
  array-interp        (arr x)              interpolated array lookup\n\
  array->file         (filename vct len srate channels)\n\
  asymmetric-fm       (gen index fm)       asymmetric-fm generator\n\
  asymmetric-fm?      (gen)                #t if gen is asymmetric-fm generator\n\
  buffer->frame       (gen frame           buffer generator returning frame\n\
  buffer->sample      (gen)                buffer generator returning sample\n\
  buffer-empty?       (gen)                #t if buffer has no data\n\
  buffer?             (gen)                #t if gen is buffer generator\n\
  clear-array         (arr)                set all elements of arr to 0.0\n\
  comb                (gen input pm)       comb filter\n\
  comb?               (gen)                #t if gen is comb filter\n\
  contrast-enhancement(input (index 1.0))  a kind of phase modulation\n\
  convolution         (sig1 sig2 n)        convolve sig1 with sig2 (size n), returning new sig1\n\
  convolve            (gen input-function) convolve generator\n\
  convolve?           (gen)                #t if gen is convolve generator\n\
  db->linear          (db)                 translate dB value to linear\n\
  degrees->radians    (deg)                translate degrees to radians\n\
  delay               (gen input pm)       delay line\n\
  delay?              (gen)                #t if gen is delay line\n\
  dot-product         (sig1 sig2)          return dot-product of sig1 with sig2\n\
  env                 (gen)                envelope generator\n\
  env-interp          (x env (base 1.0))   return value of env at x\n\
  env?                (gen)                #t if gen is env (from make-env)\n\
  mus-fft             (rl im n sign)       fft of rl and im (sign = -1 for ifft), result in rl\n\
  file->array         (filename chan start len vct)\n\
  file->frame         (gen loc frame)      return frame from file at loc\n\
  file->frame?        (gen)                #t if gen is file->frame generator\n\
  file->sample        (gen loc chan)       return sample from file at loc\n\
  file->sample?       (gen)                #t if gen is file->sample generator\n\
  filter              (gen input)          filter\n\
  filter?             (gen)                #t if gen is filter\n\
  fir-filter          (gen input)          FIR filter\n\
  fir-filter?         (gen)                #t if gen is fir filter\n\
  formant             (gen input)          formant generator\n\
  formant?            (gen)                #t if gen is formant generator\n\
  frame*              (fr1 fr2 outfr)      element-wise multiply\n\
  frame+              (fr1 fr2 outfr)      element-wise add\n\
  frame->buffer       (buf frame)          add frame to buffer\n\
  frame->file         (gen loc frame)      write (add) frame to file at loc\n\
  frame->file?        (gen)                #t if gen is frame->file generator\n\
  frame->frame        (mixer frame outfr)  pass frame through mixer\n\
  frame-ref           (frame chan)         return frame[chan]\n\
  frame->sample       (frmix frame)        pass frame through frame or mixer to produce sample\n\
  frame-set!          (frame chan val)     frame[chan]=val\n\
  frame?              (gen)                #t if gen is frame object\n\
  granulate           (gen input-function) granular synthesis generator\n\
  granulate?          (gen)                #t if gen is granulate generator\n\
  hz->radians         (freq)               translate freq to radians/sample\n\
  iir-filter          (gen input)          IIR filter\n\
  iir-filter?         (gen)                #t if gen is iir-filter\n\
  in-any              (loc chan stream)    return sample in stream at loc and chan\n\
  in-hz               (freq)               translate freq to radians/sample\n\
  ina                 (loc stream)         return sample in stream at loc, chan 0\n\
  inb                 (loc stream)         return sample in stream at loc, chan 1\n\
  linear->db          (val)                translate linear val to dB\n\
  locsig              (gen loc input)      place input in output channels at loc\n\
  locsig-ref          (gen chan)           locsig-scaler[chan]\n\
  locsig-reverb-ref   (gen chan)           locsig-reverb-scaler[chan]\n\
  locsig-set!         (gen chan val)       locsig-scaler[chan] = val\n\
  locsig-reverb-set!  (gen chan val)       locsig-reverb-scaler[chan] = val\n\
  locsig?             (gen)                #t if gen is locsig generator\n\
  ;; all the make function arguments are optional-key args\n\
  make-all-pass       (feedback feedforward size max-size initial-contents initial-element)\n\
  make-asymmetric-fm  (frequency initial-phase r ratio)\n\
  make-buffer         (size fill-time)\n\
  make-comb           (scaler size max-size initial-contents initial-element)\n\
  make-convolve       (input filter fft-size)\n\
  make-delay          (size initial-contents initial-element max-size)\n\
  make-env            (envelope scaler duration offset base end start)\n\
  make-fft-window     (type size)\n\
  make-file->frame    (name)\n\
  make-file->sample   (name)\n\
  make-filter         (order xcoeffs ycoeffs)\n\
  make-fir-filter     (order xcoeffs)\n\
  make-formant        (radius frequency gain)\n\
  make-frame          (chans &rest vals)\n\
  make-frame->file    (name chans)\n\
  make-granulate      (input expansion length scaler hop ramp jitter max-size)\n\
  make-iir-filter     (order ycoeffs)\n\
  make-locsig         (degree distance reverb output revout channels)\n\
  make-mixer          (chans &rest vals)\n\
  make-notch          (scaler size max-size initial-contents initial-element)\n\
  make-one-pole       (a0 b1)\n\
  make-one-zero       (a0 a1)\n\
  make-oscil          (frequency initial-phase)\n\
  make-ppolar         (radius frequency)\n\
  make-pulse-train    (frequency amplitude initial-phase)\n\
  make-rand           (frequency amplitude)\n\
  make-rand-interp    (frequency amplitude)\n\
  make-readin         (file channel start)\n\
  make-sample->file   (name chans)\n\
  make-sawtooth-wave  (frequency amplitude initial-phase)\n\
  make-sine-summation (frequency initial-phase n a ratio)\n\
  make-square-wave    (frequency amplitude initial-phase)\n\
  make-src            (input srate width)\n\
  make-sum-of-cosines (frequency initial-phase cosines)\n\
  make-table-lookup   (frequency initial-phase wave)\n\
  make-triangle-wave  (frequency amplitude initial-phase)\n\
  make-two-pole       (a0 b1 b2)\n\
  make-two-zero       (a0 a1 a2)\n\
  make-wave-train     (frequency initial-phase wave)\n\
  make-waveshape      (frequency partials)\n\
  make-zpolar         (radius frequency)\n\
  mixer*              (mix1 mix2 outmx)    matrix multiply of mix1 and mix2\n\
  mixer-ref           (mix in out)         mix-scaler[in,out]\n\
  mixer-set!          (mix in out val)     mix-scaler[in,out] = val\n\
  mixer?              (gen)                #t if gen is mixer object\n\
  multiply-arrays     (arr1 arr2)          arr1[i] *= arr2[i]\n\
  ;; the \"mus-\" functions are generic functions, to set use mus-set-var as in mus-set-frequency\n\
  mus-a0              (gen)                a0 field (simple filters)\n\
  mus-a1              (gen)                a1 field (simple filters)\n\
  mus-a2              (gen)                a2 field (simple filters)\n\
  mus-array-print-length ()                how many array elements to print in mus_describe\n\
  mus-b1              (gen)                b1 field (simple filters)\n\
  mus-b2              (gen)                b2 field (simple filters)\n\
  mus-channel         (gen)                channel of gen\n\
  mus-channels        (gen)                channels of gen\n\
  mus-cosines         (gen)                cosines of sum-of-cosines gen\n\
  mus-data            (gen)                data array of gen\n\
  mus-feedback        (gen)                feedback term of gen (simple filters)\n\
  mus-feedforward     (gen)                feedforward term of gen (all-pass)\n\
  mus-formant-radius  (gen)                formant radius\n\
  mus-frequency       (gen)                frequency of gen (Hz)\n\
  mus-hop             (gen)                hop amount of gen (granulate)\n\
  mus-increment       (gen)                increment of gen (src, readin, granulate)\n\
  mus-input?          (gen)                #t if gen is input source\n\
  mus-length          (gen)                length of gen\n\
  mus-location        (gen)                location (read point) of gen\n\
  mus-mix             (outfile infile outloc frames inloc mixer envs)\n\
  mus-order           (gen)                order of gen (filters)\n\
  mus-output?         (gen)                #t if gen is output generator\n\
  mus-phase           (gen)                phase of gen (radians)\n\
  mus-ramp            (gen)                ramp time of gen (granulate)\n\
  mus-random          (val)                random numbers bewteen -val and val\n\
  mus-scaler          (gen)                scaler of gen\n\
  mus-set-rand-seed   (val)                set random number generator seed to val\n\
  mus-set-srate       (val)                set sampling rate to val\n\
  mus-srate           ()                   current sampling rate\n\
  mus-xcoeffs         (gen)                feedforward (FIR) coeffs of filter\n\
  mus-ycoeffs         (gen)                feedback (IIR) coeefs of filter\n\
  notch               (gen input pm)       notch filter\n\
  notch?              (gen)                #t if gen is notch filter\n\
  one-pole            (gen input)          one-pole filter\n\
  one-pole?           (gen)                #t if gen is one-pole filter\n\
  one-zero            (gen input)          one-zero filter\n\
  one-zero?           (gen)                #t if gen is one-zero filter\n\
  oscil               (gen fm pm)          sine wave generator\n\
  oscil?              (gen)                #t if gen is oscil generator\n\
  out-any             (loc samp chan stream) write (add) samp to stream at loc in channel chan\n\
  outa                (loc samp stream)    write (add) samp to stream at loc in chan 0\n\
  outb                (loc samp stream)    write (add) samp to stream at loc in chan 1\n\
  outc                (loc samp stream)    write (add) samp to stream at loc in chan 2\n\
  outd                (loc samp stream)    write (add) samp to stream at loc in chan 3\n\
  partials->polynomial(partials kind)      create waveshaping polynomial from partials\n\
  partials->wave      (synth-data table norm) load table from synth-data\n\
  partials->waveshape (partials norm size) create waveshaping table from partials\n\
  phase-partials->wave(synth-data table norm) load table from synth-data\n\
  phase-partials->waveshape(partials phases size) create waveshaping table from partials\n\
  polynomial          (coeffs x)           evaluate polynomial at x\n\
  pulse-train         (gen fm)             pulse-train generator\n\
  pulse-train?        (gen)                #t if gen is pulse-train generator\n\
  radians->degrees    (rads)               convert radians to degrees\n\
  radians->hz         (rads)               convert radians/sample to Hz\n\
  rand                (gen fm)             random number generator\n\
  rand-interp         (gen fm)             interpolating random number generator\n\
  rand-interp?        (gen)                #t if gen is interpolating random number generator\n\
  rand?               (gen)                #t if gen is random number generator\n\
  readin              (gen)                read one value from associated input stream\n\
  readin?             (gen)                #t if gen is readin generator\n\
  rectangular->polar  (rl im)              translate from rectangular to polar coordinates\n\
  restart-env         (env)                return to start of env\n\
  ring-modulate       (sig1 sig2)          sig1 * sig2 (element-wise)\n\
  sample->buffer      (buf samp)           store samp in buffer\n\
  sample->file        (gen loc chan val)   store val in file at loc in channel chan\n\
  sample->file?       (gen)                #t if gen is sample->file generator\n\
  sample->frame       (frmix samp outfr)   convert samp to frame\n\
  sawtooth-wave       (gen fm)             sawtooth-wave generator\n\
  sawtooth-wave?      (gen)                #t if gen is sawtooth-wave generator\n\
  sine-summation      (gen fm)             sine-summation generator\n\
  sine-summation?     (gen)                #t if gen is sine-summation generator\n\
  spectrum            (rl im win type)     produce spectrum of data in rl\n\
  square-wave         (gen fm)             square-wave generator\n\
  square-wave?        (gen)                #t if gen is square-wave generator\n\
  src                 (gen fm input-function) sample rate converter\n\
  src?                (gen)                #t if gen is sample-rate converter\n\
  sum-of-cosines      (gen fm)             sum-of-cosines (pulse-train) generator\n\
  sum-of-cosines?     (gen)                #t if gen is sum-of-cosines generator\n\
  table-lookup        (gen fm)             table-lookup generator\n\
  table-lookup?       (gen)                #t if gen is table-lookup generator\n\
  tap                 (gen pm)             delay line tap\n\
  triangle-wave       (gen fm)             triangle-wave generator\n\
  triangle-wave?      (gen)                #t if gen is triangle-wave generator\n\
  two-pole            (gen input)          two-pole filter\n\
  two-pole?           (gen)                #t if gen is two-pole filter\n\
  two-zero            (gen input)          two-zero filter\n\
  two-zero?           (gen)                #t if gen is two-zero filter\n\
  wave-train          (gen fm)             wave-train generator\n\
  wave-train?         (gen)                #t if gen is wave-train generator\n\
  waveshape           (gen index fm)       waveshaping generator\n\
  waveshape?          (gen)                #t if gen is waveshape generator\n\
";

char *get_CLM_help(void)
{
  return(CLM_help);
}
