#!/location/of/perl
#
# Transfer Synchronized calendar to PalmPilot.
# Original script by Dwayne Forsyth.
# Albert Lee, acllee@megsinet.net, 19981124
#
# New features:
# -All Synchronize events are marked with ascii-167 (I don't know what
#  it's called, but it looks like a fancy S).  If you delete the
#  character, the event is ignored during subsequent executions of the script.
# -The datebook events are compared and only those which actually need to be
#  deleted are.  The entire datebook is never deleted.
# -Substrings are used for comparisons, so if you have a Synchronize event like
#  "Meeting", and you change it on the pilot to "Meeting Room 1", it won't be
#  deleted, nor will a duplicate entry be added on a subsequent execution.
# -Daily reminders and ToDo's are supported as notes.  Substrings apply here
#  as well.
#
require "getopts.pl";

use lib "/location/of/pilot-link/Perl/lib/perl5/site_perl";
use PDA::Pilot;

# Globals
my(%sync_events, %pilot_events, %add_events, @delete_events);
my($DBNAME) = "DatebookDB";
my($SPECIAL_CHR) = chr(167);



#
# S U B R O U T I N E S
#



sub dump_events {
	
	my($events) = @_;
	my($a);
	
	print "\nSTART OF EVENT DUMP\n\n";
    
    printf "Number of events: %d\n\n", $#$events + 1;

	for ($a=0; $a<=$#$events; $a++) {
       print "START " . @$events[$a]->{begin_month}  . " " .
                        @$events[$a]->{begin_day}    . " " .
                        @$events[$a]->{begin_year}   . " " .
                        @$events[$a]->{begin_hour}   . " " .
                        @$events[$a]->{begin_minute} . " " .
             "END "   . @$events[$a]->{end_month}    . " " .
                        @$events[$a]->{end_day}      . " " .
                        @$events[$a]->{end_year}     . " " .
                        @$events[$a]->{end_hour}     . " " .
                        @$events[$a]->{end_minute}   . " " .
             "ID"     . @$events[$a]->{id}           . " " .
             "DESC "  . @$events[$a]->{description}  . " " .
			 "NOTE "  . @$events[$a]->{note}         . "\n";
	}

	print "\nEND OF EVENT DUMP\n\n";
}



sub load_pilot_date_book {
    my($dlp, $events) = @_;
 	my($rec, $i, $j, $db, $event);
 
    $db  = $dlp->open($DBNAME);
	
	if (!$db) {
		$db  = $dlp->create($DBNAME, 'date', 'DATA', 0, 0);
	}
	
    $i = 0;
    while ($rec = $db->getRecord($i++)) {
		if ($rec->{description} =~ /$SPECIAL_CHR/) {
			if ($rec->{'archived'} == 1) {
				next;
			}
            $event = {};
		 	$event->{begin_month}   = $rec->{'begin'}[4];
			$event->{begin_day}     = $rec->{'begin'}[3];
			$event->{begin_year}    = $rec->{'begin'}[5];
			$event->{begin_hour}    = $rec->{'begin'}[2];
			$event->{begin_minute}  = $rec->{'begin'}[1];
			$event->{end_month}     = $rec->{'end'}[4];
    		$event->{end_day}       = $rec->{'end'}[3];
    		$event->{end_year}      = $rec->{'end'}[5];
    		$event->{end_hour}      = $rec->{'end'}[2];
    		$event->{end_minute}    = $rec->{'end'}[1];
			$event->{description}   = $rec->{'description'};
            $event->{id}            = $rec->{'id'};
			$event->{note}          = $rec->{'note'};
			$event->{to_be_deleted} = 1;
			$event->{event}         = $rec->{'event'};
            push @$events, $event;
		}
	}
	
	$db->close();

	undef $db; # Close database
}



sub load_sync {
    my($data, $days) = @_;

	open(SYNC, "synchronize -day -range $days|");
    @$data = <SYNC>;
    close (SYNC);
}



sub convert_to_24_hours {
    my ($hour_12, $am_or_pm) = @_;
    my $hour_24;

    if (($hour_12 == 12) and ($am_or_pm eq "am"))
    {
        # It's midnight, so it's zero o'clock in 24-hour time
        $hour_24 = 0;
    }
    elsif (($hour_12 != 12) and ($am_or_pm eq "pm"))
    {
        # It's somewhere between 1pm and 11pm, so add 12 to get 24-hour time
        $hour_24 = $hour_12 + 12;
    }
    else
    {
        # It's somewhere between 1am and 11am, so 24-hour time is the same
        $hour_24 = $hour_12;
    }

    return $hour_24;
}



sub parse_sync {
    my($data_in, $events) = @_;
    my($year, $month, $day, $a, $b, $i);

    $i = 0;
    while ($i < @$data_in) {
        $a = @$data_in[$i];
        if ($a =~ /#For \w+ \d+, (\d+)/) {
            $year = $1;
        } else {
            if ($a =~ /(\d+)\/(\d+)/) {
                $month = $1;
                $day = $2;
            }
            if ($a =~ /(\d\d):(\d\d)([ap]m)-(\d\d):(\d\d)([ap]m)\t(.*)/) {
                $event = {};
                $event->{begin_year}   = $year - 1900;
                $event->{begin_month}  = $month - 1;
                $event->{begin_day}    = $day;
                $event->{begin_hour}   = convert_to_24_hours($1, $3);
                $event->{begin_minute} = $2;
                $event->{end_year}     = $year - 1900;
                $event->{end_month}    = $month - 1;
                $event->{end_day}      = $day;
                $event->{end_hour}     = convert_to_24_hours($4, $6);
                $event->{end_minute}   = $5;
                $event->{description}  = $SPECIAL_CHR . $7;
				$event->{event}        = 0;
                push @$events, $event;
            }
            elsif ($a =~ /^Reminders:\s+(.*)$/) {
				$reminder = $1;
                while (defined ($b = @$data_in[$i+1])) {
                    if ($b =~ /^\t\t(.*)$/) {
						$reminder = $reminder. "\n" . $1;
                        $i++
                    }
                    else {
                        last;
                    }
                }
				if ($reminder !~ /^\s*$/) {
                	$event = {};
                	$event->{begin_year}   = $year - 1900;
                	$event->{begin_month}  = $month - 1;
                	$event->{begin_day}    = $day;
                	$event->{begin_hour}   = 0;
                	$event->{begin_minute} = 0;
                	$event->{description}  = $SPECIAL_CHR . "Reminder";
					$event->{note}         = $reminder;
					$event->{event}        = 1;
                	push @$events, $event;
				}				
            }
            elsif ($a =~ /^ToDos:\s+(.*)$/) {
				$todo = $1;
                while (defined ($b = @$data_in[$i+1])) {
                    if ($b =~ /^\t\t(.*)$/) {
						$todo = $todo . "\n" . $1;
                        $i++
                    }
                    else {
                        last;
                    }
                }
				if ($todo !~ /^\s*$/) {
				 	$event = {};
                 	$event->{begin_year}   = $year - 1900;
                 	$event->{begin_month}  = $month - 1;
                 	$event->{begin_day}    = $day;
                 	$event->{begin_hour}   = 0;
                 	$event->{begin_minute} = 0;
                 	$event->{description}  = $SPECIAL_CHR . "To Do";
				 	$event->{note}  	   = $todo;
					$event->{event}        = 1;
                 	push @$events, $event;
				}		
            }
        }
        $i++;
    }
}



sub store_pilot_date_book {
    my($dlp, $add_events, $delete_events) = @_;
    my($a, $db, $rec);
   
	$db  = $dlp->open("$DBNAME");
       
	for ($a=0; $a<=$#$add_events; $a++) {
        $rec = $db->newRecord();
        $rec->{'begin'}[0]    = 0;
        $rec->{'begin'}[1]    = @$add_events[$a]->{begin_minute};
        $rec->{'begin'}[2]    = @$add_events[$a]->{begin_hour};
        $rec->{'begin'}[3]    = @$add_events[$a]->{begin_day};
        $rec->{'begin'}[4]    = @$add_events[$a]->{begin_month};
        $rec->{'begin'}[5]    = @$add_events[$a]->{begin_year};
        $rec->{'end'}[0] 	  = 0;
        $rec->{'end'}[1] 	  = @$add_events[$a]->{end_minute};
        $rec->{'end'}[2] 	  = @$add_events[$a]->{end_hour};
        $rec->{'end'}[3] 	  = @$add_events[$a]->{end_day};
        $rec->{'end'}[4] 	  = @$add_events[$a]->{end_month};
        $rec->{'end'}[5] 	  = @$add_events[$a]->{end_year};
        $rec->{'description'} = @$add_events[$a]->{description};
        $rec->{'secret'} 	  = 0;
        $rec->{'deleted'} 	  = 0;
        $rec->{'modified'}    = 0;
        $rec->{'busy'}  	  = 0;
        $rec->{'archived'}    = 0;
		$rec->{'event'}       = @$add_events[$a]->{event};
		if (@$add_events[$a]->{note} ne "") {
			$rec->{'note'} = @$add_events[$a]->{note};		
		}
        $db->setRecord($rec);
    }
	
	for ($a=0; $a<=$#$delete_events; $a++) {
        $rec = $db->newRecord();
		$rec->{'id'} = @$delete_events[$a];
        $rec->{'deleted'} = 1;
        $db->setRecord($rec);
    }

    # Close database
    $db->close();
	undef $db; 
}



sub compare_events {
	my($pilot_events, $sync_events, $add_events, $delete_events) = @_;
    my($a, $b, $found, $event);
    
    for ($a=0; $a<=$#$sync_events; $a++) {
		$found = 0;
        for ($b=0; $b<=$#$pilot_events; $b++) {
			if ((@$sync_events[$a]->{begin_year} ==
                 @$pilot_events[$b]->{begin_year}) &&
                (@$sync_events[$a]->{begin_month} ==
                 @$pilot_events[$b]->{begin_month}) &&
                (@$sync_events[$a]->{begin_day} ==
                 @$pilot_events[$b]->{begin_day}) &&
                (@$sync_events[$a]->{begin_hour} ==
                 @$pilot_events[$b]->{begin_hour}) &&
                (@$sync_events[$a]->{begin_minute} ==
                 @$pilot_events[$b]->{begin_minute}) &&
			    (@$sync_events[$a]->{end_year} ==
                 @$pilot_events[$b]->{end_year}) &&
                (@$sync_events[$a]->{end_month} ==
                 @$pilot_events[$b]->{end_month}) &&
                (@$sync_events[$a]->{end_day} ==
                 @$pilot_events[$b]->{end_day}) &&
                (@$sync_events[$a]->{end_hour} ==
                 @$pilot_events[$b]->{end_hour}) &&
                (@$sync_events[$a]->{end_minute} ==
                 @$pilot_events[$b]->{end_minute}) && 
                (index(@$pilot_events[$b]->{description},
                 @$sync_events[$a]->{description}) >= 0) &&
                (index(@$pilot_events[$b]->{note},
                 @$sync_events[$a]->{note}) >= 0)) {                 

                @$pilot_events[$b]->{to_be_deleted} = 0;
				$found = 1;
				last;
			}
		}
		if (!$found) {		
            push @$add_events, @$sync_events[$a];
		}
	}
	
	for ($b=0; $b<=$#$pilot_events; $b++) {
		if (@$pilot_events[$b]->{to_be_deleted}) {
			push @$delete_events, @$pilot_events[$b]->{id};
		}
	}
}
	


#
# M A I N
#

print "\n";
print "**************************************************\n";
print "* PSYNC -- Version 1.2 -- Last modified 11/23/98 *\n";
print "**************************************************\n";
print "\n";

&Getopts("d:p:");

# Number of days to extract from Synchronize
if (!defined $opt_d) {
	$opt_d = 15;
}

if (!defined $opt_p) {
	print "Usage: Psync [-d <number of days>] -p <port>\n\n";
	print "Enter the serial port device name.\n\n";
    exit(1);
}

print "Reading Synchronize database (this may take some time)...\n";
&load_sync(\@data, $opt_d);

print "\nPress the HotSync button.\n";

$socket = PDA::Pilot::openPort($opt_p);
$dlp = PDA::Pilot::accept($socket);
$dlp->getStatus();

# Get events from the Pilot.
print "Reading from PalmIII...\n";
&load_pilot_date_book($dlp, \@pilot_events);
#&dump_events(\@pilot_events);

# Get events from the Synchronize database.
print "Parsing Synchronize database...\n";
&parse_sync(\@data, \@sync_events);
#&dump_events(\@sync_events);

# Compare Pilot and Synchronize events.
print "Synchronizing...\n";
&compare_events(\@pilot_events, \@sync_events, \@add_events,
\@delete_events);
#&dump_events(\@add_events);

# Update Pilot.
print "Updating PalmIII...\n";
&store_pilot_date_book($dlp, \@add_events, \@delete_events);

# Close connection
undef $dlp;

print "Done.\n\n";

