/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <stdio.h>
#include <sys/stat.h>
#include <errno.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-util.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-dialog-util.h>

#if defined(__FreeBSD__) || defined(__svr4__)
#include <unistd.h>
#include <netinet/in.h>
#include <sys/socket.h>
#endif

#ifdef AF_NEEDS_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#include <arpa/inet.h>
#include <netdb.h>
#include <pwd.h>

#include "debug.h"
#include "globals.h"
#include "util.h"
#include "prefs.h"


/*---[ wrap_long_lines ]----------------------------------------------
 * take a really long line and put newlines into the appropriate
 * places so the string is never wider than maxlen
 *--------------------------------------------------------------------*/
gchar*
wrap_long_lines (gchar *string, gint maxlen)
{
	gint i=maxlen, j=0;
	gint length = strlen (string) - maxlen;

	while (j <= length) {
		while ((*(string + j + i) != ' ') &&
		       (*(string + j + i) != '\n'))
			i--;
		*(string + j + i) = '\n';
		j += i;
		i = maxlen;
	}
	
	return string;
}


void
pan_error_dialog (const gchar *format, ...)
{
	va_list args;
	gchar *str = NULL;

	g_return_if_fail (format != NULL);

	va_start (args, format);
	str = g_strdup_vprintf (format, args);
	va_end (args);

	pan_lock();
	gnome_error_dialog (str);
	pan_unlock();

	g_free (str);
}



/*---[ not_implemented ]----------------------------------------------
 * ok dialog popup to notify user that we are so lazy we haven't
 * added their feature yet.
 *--------------------------------------------------------------------*/
void
not_implemented (void)
{
	gnome_ok_dialog (_("Not yet implemented."));
}


static pthread_t has_lock_thr;

void
pan_lock (void)
{
	pthread_t thr;

	thr = pthread_self();
	if (thr != Pan.main_t)
	{
		if (thr == has_lock_thr)
			debug(DEBUG_LOCK,"thread %lu attempted double lock!", thr);
		gdk_threads_enter();
		has_lock_thr = thr;
		debug(DEBUG_LOCK,"thread %lu entered gdk_threads", thr);
	} else
		debug(DEBUG_LOCK,"mainthread %lu attempted lock", thr);
}

void
pan_unlock (void)
{
	pthread_t thr;

	thr = pthread_self();
	if (thr != Pan.main_t)
	{
		has_lock_thr = (pthread_t)0;
		gdk_threads_leave();
		debug(DEBUG_LOCK,"thread %lu left gdk_threads", thr);
	} else
		debug(DEBUG_LOCK,"mainthread %lu attempted unlock", thr);
}


/*--------------------------------------------------------------------
 *
 *--------------------------------------------------------------------*/
void
directory_check (const gchar *pathname)
{
	struct stat st;
	gchar *dir;
	gint len, i;

	len = strlen (pathname);

	i = 1;
	while (i <= len) {
		if (pathname[i] == '/' || pathname[i] == '\0') {
			dir = g_strndup (pathname, i);
			if (stat (dir, &st) != 0) {
				int ret;
				ret = mkdir (dir, S_IRUSR | S_IWUSR | S_IXUSR);
				if (ret == -1) {
					pan_error_dialog (_("Couldn't create directory: "),
							  pathname, strerror(errno));
					return;
				}
			}
			g_free (dir);
		}
		i++;
	}
}


int lower_bound (
   const void* key,
   const void* base,
   size_t n,
   size_t size,
   int (*compare)(const void *, const void *),
   gboolean* exact_match )
{
   int low = 0;
   int high = n-1;

   while ( low <= high )
   {
      const int mid = (low+high)/2;
      const int comp = (*compare)(key,(const void*)(((char*)(base))+(mid*size)));
      if ( comp > 0 ) low = mid + 1;
      else if ( comp < 0 ) high = mid - 1;
      else {
         if ( exact_match!=NULL )
            *exact_match = TRUE;
         return mid;
      }
   }

   if ( exact_match!=NULL )
      *exact_match = FALSE;
   return low;
}



GArray*
read_file (const char* filename)
{
	GArray* array = NULL;
	FILE* fp;

	/* get length of file... */
	size_t len = get_filesize(filename);
	if ( !len )
		return NULL;

	/* open the file... */
	fp = fopen ( filename, "r" );
	if ( !fp ) {
		pan_error_dialog ( _("Unable to open file: %s\n%s"), filename, strerror(errno) );
		return NULL;
	}

	/* read the file... */
	array = g_array_new ( TRUE, FALSE, 1 );
	g_array_set_size ( array, len );
	array->len = fread ( array->data, 1, len, fp );
	fclose ( fp );
	return array;
}


/*--------------------------------------------------------------------
 * 
 *--------------------------------------------------------------------*/
void
commatize_ulong (gulong num,
		 gchar* setme)
{
	char buf[16], *src=buf;
	int len;
	int i;

	sprintf ( buf, "%ld", num );
	len = strlen ( buf );
	i = len % 3;
	for ( ;; ) {
		while ( *src && i-- )
			*setme++ = *src++;
		if ( !*src ) {
			*setme = '\0';
			return;
		}
		else if ( src!=buf )
		   *setme++ = ',';
		i = 3;
	}

	pan_warn_if_reached ();
}

/*---[ get_filesize ]-------------------------------------------------------
 * Use stat() to get and then return the size of the file.  Much faster
 * than opening the file and using ftell().
 *--------------------------------------------------------------------------*/
extern size_t
get_filesize (const gchar* filename)
{
	struct stat buf;
	if ( !stat(filename, &buf) )
		return buf.st_size;

	return 0;
}

gchar*
pan_append (gchar* old, const gchar* appendme, char delimeter)
{
	gchar *pch = NULL;
	const gchar *prepend = old ? old : "";

	if (delimeter)
		pch = g_strdup_printf ("%s%c%s", prepend, delimeter, appendme);
	else
		pch = g_strdup_printf ("%s%s", prepend, appendme);
	g_free (old);
	return pch;
}


/**
***  substitute
***
***  substitute should be safe with overlapping elements of text,
***  although I have not tested it with elaborately cruel cases.
***
***  examples:
***  speech = substitute (old_speech, "that damn lawyer", "that lawyer" );
***
**/
gchar*
pan_substitute (const gchar *original, const gchar* search, const gchar* replace)
{
   size_t slen;		/* length of search */
   size_t rlen;		/* length of replace */
   size_t tlen;		/* length of target (predicted) */
   int i;
   const char *o;
   const char *pchar;
   char *t;
   gchar *out = NULL;

   g_return_val_if_fail (original!=NULL, NULL);
   g_return_val_if_fail (*original!='\0', NULL);
   g_return_val_if_fail (search!=NULL, NULL);
   g_return_val_if_fail (*search!='\0', NULL);
   g_return_val_if_fail (replace!=NULL, NULL);

   slen = strlen (search);
   rlen = strlen (replace);

   /* calculate the length */

   i = 0;
   tlen = 0;
   pchar = original;
   while ((pchar = strstr (pchar, search))) {
      i++;
      pchar += slen;
   }
   tlen = strlen(original) + i*(rlen - slen);
   out = g_malloc (tlen + 1);

   /**
   ***  Make the substitution.
   **/

   o = original;
   t = out;
   while ((pchar = strstr (o, search))) {
      (void) memcpy (t, o, (size_t)(pchar-o));
      t += pchar-o;
      (void) memcpy (t, replace, (size_t)rlen);
      t += rlen;
      o = pchar + slen;
   }
   (void) strcpy ( t, o );

   g_assert (strlen(out) == tlen);

   return out;
}

gchar *
add_re (const gchar *subject)
{
	if (!g_strncasecmp ("Re: ", subject, 4)) {
		return g_strdup (subject);
	} else {
		return g_strconcat ( "Re: ", subject, NULL );
	}
}

gchar *
pan_make_temp (FILE **f)
{
	gchar *template = g_strdup_printf ("/%s/pan%uXXXXXX", temp_dir, (unsigned int)getpid());
	int fd = mkstemp (template);
	
	if (fd == -1) {
		return NULL;
	}
	*f = fdopen (fd, "w+");
	return template;
}

void
pan_clean_temp_dir (void)
{
	gchar *dirpath = g_strdup_printf ("/%s", temp_dir); 
	gchar *prefix = g_strdup_printf ("pan%u", (unsigned int)getpid ());
	gint prefix_l = strlen (prefix);
	DIR *dir_p = opendir(dirpath);
	struct dirent *dirent_p;

	while((dirent_p = readdir(dir_p))) {
		if (!strncmp (prefix, dirent_p->d_name, prefix_l)) {
			gchar *filename = g_strdup_printf ("/%s/%s", temp_dir, dirent_p->d_name);
			unlink (filename);
			g_free (filename);
		}	
	}
	closedir (dir_p);
	g_free (prefix);
	g_free (dirpath);
}

gboolean file_exists (const gchar* filename)
{
	return g_file_exists (filename);
}

gchar*
get_host_name (void)
{
	gchar *ptr;
	gchar hostname[256] = { '\0' };

	if ((gethostname(hostname, sizeof(hostname))) == -1) {
		if ((ptr = getenv("HOST")) != NULL)
			strncpy(hostname, ptr, MAXHOSTNAMELEN);
		else if ((ptr = getenv("HOSTNAME")) != NULL)
			strncpy(hostname, ptr, MAXHOSTNAMELEN);
		else
			hostname[0] = '\0';
	}
	hostname[255] = '\0';
	ptr = strtok (hostname, ".");

	return g_strdup (ptr);
}

/* Returns the fully qualified domain name */
gchar*
get_fqdn (const gchar *host)
{
	char name[512];
	char line[1025];
	char *cp, *domain;
	char fqdn[1024];
	struct hostent *hp;
	struct in_addr in;
	FILE *inf;

	*fqdn = '\0';
	domain = NULL;

	name[MAXHOSTNAMELEN] = '\0';
	if (host) {
		if (strchr(host, '.'))
			return g_strdup(host);
		strncpy(name, host, MAXHOSTNAMELEN);
	}
	else {
		if (gethostname(name, MAXHOSTNAMELEN))
			return NULL;
	}

	if ('0' <= *name && *name <= '9') {
		in.s_addr = inet_addr(name);
		if ((hp = gethostbyaddr((char *) &in.s_addr, 4, AF_INET)))
			in.s_addr = (*hp->h_addr);
		return g_strdup((hp && strchr(hp->h_name, '.') ? hp->h_name : (gchar *) inet_ntoa(in)));
	}
	if ((hp = gethostbyname(name)) && !strchr(hp->h_name, '.'))
		if ((hp = gethostbyaddr(hp->h_addr, hp->h_length, hp->h_addrtype)))
			in.s_addr = (*hp->h_addr);

	sprintf(fqdn, "%s", hp ? strchr(hp->h_name, '.')
		? hp->h_name : (gchar *) inet_ntoa(in)
		: "");
	if (!*fqdn || (fqdn[strlen(fqdn) - 1] <= '9')) {
		*fqdn = '\0';
		inf = fopen("/etc/resolv.conf", "r");
		if (inf) {
			while (fgets(line, 1024, inf)) {
				line[1024] = '\0';
				g_strchug(line);
				g_strchomp(line);
				if (strncmp(line, "domain ", 7) == 0) {
					domain = line + 7;
					break;
				}
				if (strncmp(line, "search ", 7) == 0) {
					domain = line + 7;
					cp = strchr(domain, ' ');
					if (cp)
						*cp = '\0';
					break;
				}
			}
			if (domain)
				sprintf(fqdn, "%s.%s", name, domain);
			fclose(inf);
		}
	}
	return g_strdup (fqdn);
}

void
array_shrink (gpointer array, int index, size_t element_size, int element_qty)
{
	g_return_if_fail (array!=NULL);
	g_return_if_fail (element_size!=0);
	g_return_if_fail (index>=0);
	g_return_if_fail (index<=element_qty);

	if (index != element_qty-1)
	{
		g_memmove (
			((gchar*)array) + element_size * index,
			((gchar*)array) + element_size * (index + 1),
			element_size * (element_qty - index -1));
	}
}

