/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include "globals.h"
#include "grouplist.h"
#include "gui.h"
#include "prefs.h"
#include "util.h"

static void server_selected_cb (GtkWidget *, server_data *);
static void server_list_save (void);

static GtkWidget *server_menu = NULL;


GtkWidget *
server_menu_create (void)
{
	server_menu = gtk_option_menu_new ();
	return (server_menu);
}

void
server_shutdown (void)
{
	GSList *l;
	server_data *last;
	for (l = Pan.serverlist; l; l = l->next) {
		server_data *sdata = l->data;
		pan_db_close (sdata->db);
	}
	last = grouplist_get_current_server ();
	if (last && last->name) {
		gnome_config_set_string ("/Pan/State/Server", last->name);
		gnome_config_sync ();
	}
}

void
server_menu_update (void)
{
	GtkWidget *menu = gtk_menu_new ();
	int sel_index = -1;

	gtk_option_menu_remove_menu (GTK_OPTION_MENU (server_menu));

	if (Pan.serverlist == NULL)
	{
		gtk_menu_append (
			GTK_MENU (menu),
			gtk_menu_item_new_with_label (
				_("No servers defined")));
	}
	else
	{
		server_data *active = grouplist_get_current_server ();
		GSList *l;
		int i;
		for (i=0, l=Pan.serverlist; l; ++i, l=l->next)
		{
			server_data *sdata = l->data;

			GtkWidget *check = gtk_menu_item_new_with_label (
				sdata->name);

			gtk_signal_connect (GTK_OBJECT (check), "activate", 
				(GtkSignalFunc)server_selected_cb,
				sdata);

			gtk_menu_append (GTK_MENU (menu), check);

			/* open the server's db... */
			if (sdata->db == NULL)
			{
	       		gchar *path = g_strdup_printf ("/%s/%s.db", data_dir, sdata->name);
				sdata->db = pan_db_open (path);
	       		g_free (path);
			}

			/* is this the already active server? */
			if (sdata == active)
				sel_index = i;
		}

		if (sel_index!=-1)
			gtk_menu_set_active (GTK_MENU(menu), sel_index);
		if (active!=NULL)
			grouplist_server_selected (active);
	}

	gtk_option_menu_set_menu (GTK_OPTION_MENU (server_menu), menu);
	gtk_widget_show_all (GTK_WIDGET(server_menu));

	server_list_save ();
}

server_data*
server_fetch (const gchar *name)
{
	server_data *sdata = NULL;
	gboolean has_section;

	gchar* buf = g_strdup_printf ( "/Pan/%s/", name );
	gnome_config_push_prefix ( buf );
	has_section = gnome_config_has_section ( buf );
	g_free ( buf );

	if ( has_section ) {
		sdata = (server_data *) g_malloc0(sizeof(server_data));
		if ((sdata->address = gnome_config_get_string ("Address")) == NULL) {
			g_free (sdata);
			sdata = NULL;
		} else {
			gchar *path = g_strdup_printf ("/%s/%s/", data_dir, name);
			
			/* Let's make sure the server's config directory exists */
			directory_check (path);
			g_free (path);

			sdata->port = gnome_config_get_int ("Port=119");
			sdata->name = g_strdup(name);
			sdata->gen_msgid = gnome_config_get_int ("Generate_Msgid=0");
			sdata->need_auth = gnome_config_get_int ("Auth_Required=0");
			sdata->username = gnome_config_get_string ("Username");
			sdata->password = gnome_config_get_string ("Password");
			sdata->max_connections = gnome_config_get_int ("Max_Connections=2");
			sdata->grouplist = NULL;
		}
	}
	gnome_config_pop_prefix ();

	return sdata;
}


void
server_free (server_data *sdata)
{
	if (sdata) {
		g_free (sdata->name);
		g_free (sdata->address);
		g_free (sdata->username);
		g_free (sdata->password);
		g_free (sdata);
	}
}


void
server_init (void)
{
	gchar *last = NULL;
	GSList *l = NULL;

	/* This gets called before gtk_main and
	 * at the end of guru_finish */
	Pan.serverlist = server_list_load ();
	server_menu_update ();

	/* try hard to select a server.. if there's one in the ini file, use
	 * it; otherwise, go with the first one in the list (if there is one)
	 */
	last = gnome_config_get_string ("/Pan/State/Server");
	if (!last && Pan.serverlist)
		last = g_strdup (((server_data*)Pan.serverlist->data)->name);
	if (last!=NULL)
	{
		for (l=Pan.serverlist; l; l=l->next)
		{
			server_data *sdata = (server_data*) l->data;
			if (!strcmp(sdata->name, last))
			{
				gtk_option_menu_set_history (
					GTK_OPTION_MENU(server_menu), 
					g_slist_position (Pan.serverlist,l));
				grouplist_server_selected (sdata);
				break;
			}
		}
		g_free (last);
	}
}

#if 0
void
server_list_free (GSList *sl)
{
	GSList *l;

	for (l=sl; l; l = l->next) {
		server_free(l->data);
	}
	g_slist_free(sl);
}
#endif

static void
server_list_save (void)
{
	server_data *sdata = NULL;
	GSList *l = NULL;
	int i = 0;
	char *p = NULL;
	char **sd = NULL;

	/* Out with the old */
	if ((p = gnome_config_get_string ("/Pan/Servers/Names"))) {
		sd = g_strsplit (p, ",", -1);
		for (i = 0; sd[i]; i++) {
			gchar* buf = g_strdup_printf ( "/Pan/%s", sd[i] );
			gnome_config_clean_section (buf);
			g_free ( buf );
		}
		g_strfreev (sd);
		g_free (p);
	}
	gnome_config_sync ();

	/* In with the new */
	p = NULL;
	for (l = Pan.serverlist; l; l = l->next) {
		
		gchar* buf;
		sdata = (server_data *) l->data;

		buf = g_strdup_printf ( "/Pan/%s/Address", sdata->name );
		gnome_config_set_string (buf, sdata->address);
		g_free ( buf );

		buf = g_strdup_printf ( "/Pan/%s/Port", sdata->name );
		gnome_config_set_int (buf, sdata->port);
		g_free ( buf );

		buf = g_strdup_printf ( "/Pan/%s/Auth_Required", sdata->name);
		gnome_config_set_int (buf, sdata->need_auth);
		g_free ( buf );

		buf = g_strdup_printf ( "/Pan/%s/Generate_Msgid", sdata->name );
		gnome_config_set_int (buf, sdata->gen_msgid);
		g_free ( buf );

		buf = g_strdup_printf ( "/Pan/%s/Username", sdata->name );
		gnome_config_set_string (buf, sdata->username);
		g_free ( buf );

		buf = g_strdup_printf ( "/Pan/%s/Password", sdata->name );
		gnome_config_set_string (buf, sdata->password);
		g_free ( buf );

		buf = g_strdup_printf ( "/Pan/%s/Max_Connections", sdata->name );
		gnome_config_set_int (buf, sdata->max_connections);
		g_free ( buf );

		if (p) {
			p = g_realloc (p, strlen(p) + strlen(sdata->name) + 2);
			p = strcat (p, ",");
			p = strcat (p, sdata->name);
		} else
			p = g_strdup (sdata->name);
	}

	gnome_config_set_string ("/Pan/Servers/Names", p);
	g_free (p);

	gnome_config_sync ();
}

/**
 * server_list_load:
 *
 * Load a list of servers from the user's ~/.gnome/Pan file
 *
 * Return value: GSList of server_data's for each server found
 **/
GSList *
server_list_load (void)
{
	gchar *p;
	gchar **sd;
	gint i;
	GSList *l = NULL;

	if ((p = gnome_config_get_string ("/Pan/Servers/Names")) == NULL)
		return NULL;
    
	sd = g_strsplit (p, ",", 0);
	g_free (p);
	
	if (sd == NULL)
		return NULL;

	for (i = 0; sd[i]; i++) {
		server_data *sdata = server_fetch (sd[i]);
		if (sdata)
			l = g_slist_append (l, sdata);
	}
	g_strfreev (sd);
	return l;
}

static void
server_selected_cb (GtkWidget *w, server_data *sdata)
{
	grouplist_server_selected (sdata);
}
