/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>
#include <gnome.h>

#include "acache.h"
#include "article.h"
#include "articlelist.h"
#include "debug.h"
#include "druid.h"
#include "globals.h"
#include "gui.h"
#include "log.h"
#include "pan.h"
#include "prefs.h"
#include "queue.h"
#include "text.h"
#include "util.h"

static gint save_session (GnomeClient *client,
			  gint phase,
			  GnomeSaveStyle save_style,
			  gint is_shutdown,
			  GnomeInteractStyle interact_style,
			  gint is_fast,
			  gpointer client_data);
static void session_die (GnomeClient *client, gpointer client_data);


const gchar* sockread_err_msg = NULL;
const gchar* sockwrite_err_msg = NULL;

PanApp Pan;

static gchar     *geometry = NULL;
static gint       debug_acache = 0;
static gint       debug_socket_input = 0;
static gint       debug_socket_output = 0;
static gint       debug_queue = 0;
static gint       debug_decode = 0;
static gint       debug_gui_locks = 0;
static gint       debug_db = 0;
static gint       debug_pan_object = 0;
static gint       debug_everything = 0;
static gint       mute = 0;

poptContext      pctx;

struct poptOption options[] = {
	{ "geometry", '\0', POPT_ARG_STRING, &geometry, 0,
	  N_("Specify the geometry of the window."),
	  N_("GEOMETRY") },

	{ "mute", '\0', 0, &mute, 0,
	  N_("Sent outgoing messages to stdout instead of smtp/nntp servers."), NULL },
	{ "debug-acache", '\0', 0, &debug_acache, 0,
	  N_("Turn on article cache debugging messages."), NULL },
	{ "debug-db", '\0', 0, &debug_db, 0,
	  N_("Turn on database debugging messages."), NULL },
	{ "debug-socket-input", '\0', 0, &debug_socket_input, 0,
	  N_("Turn on socket input debugging messages."), NULL },
	{ "debug-socket-output", '\0', 0, &debug_socket_output, 0,
	  N_("Turn on socket output debugging messages."), NULL },
	{ "debug-queue", '\0', 0, &debug_queue, 0,
	  N_("Turn on task scheduler debugging messages."), NULL },
	{ "debug-decode", '\0', 0, &debug_decode, 0,
	  N_("Turn on attachment decoding debugging messages."), NULL },
	{ "debug-gui-locks", '\0', 0, &debug_gui_locks, 0,
	  N_("Turn on gui thread lock debugging messages."), NULL },
	{ "debug-pan-object", '\0', 0, &debug_pan_object, 0,
	  N_("Turn on pan object system debugging messages."), NULL },
	{ "debug", '\0', 0, &debug_everything, 0,
	  N_("Turn on ALL debugging messages."), NULL },

	{ NULL, '\0', 0, NULL, 0 }
};


static int
pan_post_gtkmain_init (gpointer null)
{
	server_init ();
	return 0;
}

/**
 * pan_init:
 *
 * For first time users it is called after the druid finishes.  For
 * second time users it is called right before gtk_main()
 **/

void
pan_init (void)
{
	log_init ();
	queue_init ();
	acache_init ();

	log_add (_("Pan Started"));
	gtk_idle_add (pan_post_gtkmain_init, NULL);
}

/*---[ main ]---------------------------------------------------------
 * _the_ main function!
 *--------------------------------------------------------------------*/
int
main (int argc, char *argv[])
{
	GnomeClient *client;
	gulong debug_flags = 0;

	bindtextdomain(PACKAGE, GNOMELOCALEDIR);
	textdomain(PACKAGE);

	sockread_err_msg = _("Error reading from socket.");
	sockwrite_err_msg = _("Error writing to socket.");

	g_thread_init (NULL);

	gnome_init_with_popt_table ("Pan", VERSION, argc, argv, options, 0, &pctx);
	poptFreeContext (pctx);

	debug_flags = (debug_acache ? DEBUG_ACACHE : 0)
		| (debug_socket_input ? DEBUG_SOCKET_INPUT : 0)
		| (debug_socket_output ? DEBUG_SOCKET_OUTPUT : 0)
		| (debug_queue ? DEBUG_QUEUE : 0)
		| (debug_decode ? DEBUG_DECODE : 0)
		| (debug_gui_locks ? DEBUG_LOCK : 0)
		| (debug_pan_object ? DEBUG_PAN_OBJECT : 0)
		| (debug_db ? DEBUG_DB : 0)
		| (debug_everything ? ~0 : 0);
	set_debug_level (debug_flags);

	prefs_init ();

	/* Connect up to the session manager */
	client = gnome_master_client ();
	gtk_signal_connect (GTK_OBJECT (client), "save_yourself",
			    GTK_SIGNAL_FUNC (save_session), argv[0]);
	gtk_signal_connect (GTK_OBJECT (client), "die",
			    GTK_SIGNAL_FUNC (session_die), NULL);

	/* initialize variables in the global app structure */
	Pan.main_t = pthread_self ();
	Pan.mute = mute;
	if (Pan.mute)
		g_message ("Running in mute mode: messages will be sent to "
		           "stdout instead of smtp/nntp server");

	article_init ();
	gui_construct (geometry);

	/* Check if a Preferences file exists. */
	if (!gnome_config_has_section ("/Pan/User")) {
//		gtk_widget_show_all (Pan.window);
		druid_spawn ();
	}
	else {
		pan_init ();
		gtk_widget_show_all (Pan.window);
	}

	gtk_main ();

	return 0;
}

/**
 * pan_shutdown:
 *
 * Shutdown Pan, closing all TCP connections and saving all preferences
 **/
void
pan_shutdown (void)
{
	//FIXME tcp_close ();

	articlelist_set_current_group ( NULL, NULL );

	/*
	 * if (Pan.grouplist)
		grouplist_save (Pan.grouplist);
	*/

	if (current_article)
		article_free(current_article);

	g_free (message_body_font);
	g_free (articlelist_font);
	g_free (grouplist_font);

	server_shutdown ();

	if (gnome_config_get_bool("/Pan/Cache/FlushOnExit"))
		acache_expire_all ();

	gtk_main_quit ();
	pan_clean_temp_dir ();
}


/*** Session management **/

/**
 * save_session
 *
 * Saves the current session for later revival
 **/
static gint
save_session (GnomeClient *client, gint phase, GnomeSaveStyle save_style,
              gint is_shutdown, GnomeInteractStyle interact_style,
              gint is_fast, gpointer client_data)
{
	gchar **argv;
	guint argc;

	argv = g_malloc0(sizeof (gchar *)*4);
	argc = 1;

	argv[0] = client_data;
	/* add any addtional state info here. */
	gnome_client_set_clone_command (client, argc, argv);
	gnome_client_set_restart_command (client, argc, argv);
	return TRUE;
}

/**
 * session_die:
 *
 * Gracefullly end the session
 **/
static void
session_die (GnomeClient *client, gpointer client_data)
{
	pan_shutdown ();
}



